/*
 * Streams interface to ethernet. 
 */

#if	defined(SYSV) && defined(RFS)
#include "../h/types.h"
#include "../h/param.h"
#include "../h/user.h"
#include "../sysv/sys/sysmacros.h"
#include "../sysv/sys/stream.h"
#include "../sysv/sys/stropts.h"
#include "../sysv/sys/errno.h"
#include "../sysv/sys/tihdr.h"
#include "../sysv/sys/tiuser.h"
#include "../sysv/sys/rfs_enet.h"
#include "../sysv/sys/debug.h"

#include "mbuf.h"
#include "socket.h"
#include "../net/if.h"
#include "../netinet/in.h"
#include "../netinet/if_ether.h"
#ifdef GWS
#include "../is68kdev/gpreg.h"
#endif

#ifdef RN_DEBUG
/*
 * These defines come from subr_prf.c.  They are needed int RfsNetDebug()
 * which calls prf() directly. 
 */
extern int      tocons;
#define TOCONS 0x1
#define TOLOG 0x4
#define TOGWS 0x8


int             rndebug;
#endif

static unsigned char Req_to_event[] = {
				        /* T_CONN_REQ */ TE_CONN_REQ,
				        /* T_CONN_RES */ TE_CONN_RES,
				        /* T_DISCON_REQ */ TE_DISCON_REQ,
				        /* T_DATA_REQ */ TE_DATA_REQ,
				        /* T_EXDATA_REQ */ TE_EXDATA_REQ,
				        /* T_INFO_REQ */ -1,
				        /* T_BIND_REQ */ TE_BIND_REQ,
				        /* T_UNBIND_REQ */ TE_UNBIND_REQ,
				        /* T_UNITDATA_REQ */ TE_UNITDATA_REQ,
				        /* T_OPTMGMT_REQ */ TE_OPTMGMT_REQ,
				        /* T_ORDREL_REQ */ TE_ORDREL_REQ
};

/*
 * Maximum packet size will be 1502. MAX_ETHERNET_BUFFER - sizeof
 * (ether_header) 1518		16 Me thinks these are the correct numbers. 
 */
static struct module_info minfo = {
				   0, "rfsnet", 0, 1502, 512, 128
};

 /* static */ int 
RfsNetopen (), RfsNetclose (), RfsNetwput (), RfsNetwsrv ();
 /* static */ int RfsNetrsrv ();

static struct qinit rinit = {
	       NULL, RfsNetrsrv, RfsNetopen, RfsNetclose, NULL, &minfo, NULL
};

static struct qinit winit = {
		      RfsNetwput, RfsNetwsrv, NULL, NULL, NULL, &minfo, NULL
};

struct streamtab RfsNetinfo = {&rinit, &winit, NULL, NULL};

extern struct rfsnet rfs_net[];
extern struct rfs_net_match seq_matches[];
extern int      rfsnet_cnt;
extern unsigned char ti_statetbl[TE_NOEVENTS][TS_NOSTATES];

struct ifqueue  rfsintrq;
void 
RfsNetError (), RfsNetSetOk (), RfsNetTimer ();
struct mbuf    *mblk_to_mbuf ();
mblk_t         *mbuf_to_mblk ();

/* ARGSUSED */
int
RfsNetopen (q, dev, flag, sflag)
	queue_t        *q;
{
	struct rfsnet  *rn;

	/*
	 * If CLONEOPEN pick a minor device number to use. Otherwise, check
	 * the minor device range. 
	 */
	if (sflag == CLONEOPEN) {
		for (dev = 0; dev < rfsnet_cnt; dev++)
			if (!(rfs_net[dev].lup_flags & RFSNET_INUSE))
				break;
	}
	else
		dev = minor (dev);

	if (dev >= rfsnet_cnt)
		return OPENFAIL;

	if (rfs_net[dev].lup_flags & RFSNET_INUSE)	/* already open */
		return (dev);

	rn = &rfs_net[dev];
	rn->lup_seq = 1;
	rn->lup_rseq = 0;
	rn->lup_retry = 1;
	rn->lup_flags = RFSNET_INUSE;
	rn->lup_state = TS_UNBND;
	rn->lup_aqlen = rn->lup_qlen = 0;
	rn->lup_baddr[NETADDR] = rn->lup_baddr[PORTADDR] = 0;
	WR (q)->q_ptr = (char *) rn;
	q->q_ptr = (char *) rn;
	rn->qptr = WR (q);

	/*
	 * Return minor device 
	 */
	RfsNetDebug (rn, RN_G, "Opened %d\n", rn - rfs_net);

	/*
	 * This should be placed in a initialization routine. 
	 */
	rfsintrq.ifq_maxlen = IFQ_MAXLEN;
	return dev;
}


/*
 * Write put proc 
 */

int
RfsNetwput (q, mp)
	queue_t        *q;
	register mblk_t *mp;
{
	register struct rfsnet *rn;

	rn = (struct rfsnet *) q->q_ptr;

	switch (mp->b_datap->db_type) {
	case M_IOCTL:
		mp->b_datap->db_type = M_IOCNAK;
		((struct iocblk *) mp->b_rptr)->ioc_error = EINVAL;
		qreply (q, mp);
		break;

	case M_FLUSH:
		/*
		 * Canonical flush processing 
		 */
		if (*mp->b_rptr & FLUSHW)
			flushq (q, FLUSHDATA);
		if (*mp->b_rptr & FLUSHR) {
			flushq (RD (q), FLUSHDATA);
			*mp->b_rptr & = ~FLUSHW;
			qreply (q, mp);
		}
		else
			freemsg (mp);
		break;

	case M_PCPROTO:{
			register union T_primitives *ackp;
			register mblk_t *tmp;

			if ((tmp = allocb (sizeof (struct T_info_ack), BPRI_MED)) == NULL) {
				mp->b_datap->db_type = M_ERROR;
				mp->b_rptr = mp->b_wptr = mp->b_datap->db_base;
				*mp->b_wptr++ = EPROTO;
				qreply (q, mp);
			}
			else if (((union T_primitives *) mp->b_rptr)->type ==
				 T_INFO_REQ && (mp->b_wptr - mp->b_rptr) ==
				 sizeof (struct T_info_req)) {
				ackp = (union T_primitives *) tmp->b_rptr;
				ackp->info_ack.PRIM_type = T_INFO_ACK;
				ackp->info_ack.TSDU_size = RFSNET_MAXTSDUSIZE;
				ackp->info_ack.ETSDU_size = RFSNET_MAXETSDUSIZE;
				ackp->info_ack.CDATA_size = RFSNET_MAXCDATASIZE;
				ackp->info_ack.DDATA_size = RFSNET_MAXDDATASIZE;
				ackp->info_ack.ADDR_size = RFSNET_MAXADDRSIZE;
				ackp->info_ack.OPT_size = RFSNET_MAXOPTSIZE;
				ackp->info_ack.TIDU_size = RFSNET_MAXTIDUSIZE;
				ackp->info_ack.SERV_type = RFSNET_SERVTYPE;
				ackp->info_ack.CURRENT_state = rn->lup_state;
				tmp->b_wptr += sizeof (struct T_info_ack);
				tmp->b_datap->db_type = M_PCPROTO;
			}
			else {
				putq (q, mp);
				freemsg (tmp);
				break;
			}
			if (tmp != NULL)
				qreply (q, tmp);
			freemsg (mp);
			break;
		}

	default:
		putq (q, mp);
	}
}


int
RfsNetwsrv (q)
	queue_t        *q;
{
	register mblk_t *mp, *tmp;
	register struct rfsnet *rn;
	register union T_primitives *pptr, *ackp;
	int             event;

	rn = (struct rfsnet *) q->q_ptr;
	ASSERT (rn->lup_flags & RFSNET_INUSE);

	tmp = NULL;
	while ((mp = getq (q)) != NULL) {
		if (IS_LOCKED (rn)) {
			putbq (q, mp);
			RfsNetDebug (rn, RN_O,
			      "RfsNetwsrv: locked queue, returning ....\n");
			return (0);	/* call back some other time */
		}

		event = -1;
		switch (mp->b_datap->db_type) {
		default:
			RfsNetDebug (rn, RN_S | RN_C, "RfsNetwsrv: Default to (%d.%d)\n",
				     rn->lup_daddr[PORTADDR], rn->lup_seq);
	passon:
			if (rn->lup_flags & RFSNET_CONNECTED) {
				/*
				 * For now state changes are not being
				 * checked 
				 */
				/*
				 * XXX Need some form of flow control.  When
				 * the packet gets to the other side with a
				 * full queue it would have to be dropped
				 * without any notification.  This side would
				 * already have freed the message block. 
				 */

				if (rfs_to_net (rn, mp)) {
					/*
					 * Some how the net has been locked
					 * between the time that we checked
					 * it and we tried to send, so
					 * requeue the message and return. 
					 */
					putbq (q, mp);
					return (0);
				}
				else
					continue;	/* Don't break */
			}
			else {
		garbage:
				mp->b_datap->db_type = M_ERROR;
				mp->b_rptr = mp->b_wptr = mp->b_datap->db_base;
				*mp->b_wptr++ = EPROTO;
				qreply (q, mp);
				if (tmp != NULL)
					freemsg (tmp);
				break;
			}

		case M_DATA:
			/* Send it as a data indication */
			RfsNetDebug (rn, RN_C | RN_S, "RfsNetwsrv: M_DATA to (%d.%d)\n",
				     rn->lup_daddr[PORTADDR], rn->lup_seq);
			if ((tmp = allocb (sizeof (struct T_data_ind), BPRI_MED)) == NULL) {
				mp->b_datap->db_type = M_ERROR;
				mp->b_rptr = mp->b_wptr = mp->b_datap->db_base;
				*mp->b_wptr++ = EAGAIN;
				qreply (q, mp);
				goto out1;
			}
			pptr = (union T_primitives *) tmp->b_rptr;
			pptr->data_ind.PRIM_type = T_DATA_IND;
			pptr->data_ind.MORE_flag = 0;	/* For now */
			tmp->b_wptr = tmp->b_rptr + sizeof (struct T_data_ind);
			tmp->b_datap->db_type = M_PROTO;
			linkb (tmp, mp);
			mp = tmp;
			tmp = NULL;
			goto passon;

		case M_PCPROTO:
			/* Ideally these messages should not come here! */
		case M_PROTO:{
				register        len, port;
				register struct rfs_addr *pa;
				int             error, s;

				pptr = (union T_primitives *) mp->b_rptr;
				if ((tmp = allocb (MAX_REPLYSIZ, BPRI_MED)) == NULL) {
					mp->b_datap->db_type = M_ERROR;
					mp->b_rptr = mp->b_wptr = mp->b_datap->db_base;
					*mp->b_wptr++ = EAGAIN;
					qreply (q, mp);
					goto out1;
				}

				if ((unsigned) pptr->type >= T_CONN_IND) {
					RfsNetError (mp, tmp, TNOTSUPPORT, 0);
					qreply (q, tmp);
					freemsg (mp);
					break;
				}
				RfsNetDebug (rn, RN_S | RN_C, "RfsNetwsrv: type 0x%x event %d\n",
				      pptr->type, Req_to_event[pptr->type]);
				s = RfsNetLock (rn);
				if (error = RfsNetState (rn, Req_to_event[pptr->type])) {
					RfsNetUnlock (rn, s);
					RfsNetError (mp, tmp, hiword (error), loword (error));
					qreply (q, tmp);
					freemsg (mp);
					break;
				}

				switch (pptr->type) {
				default:
					RfsNetUnlock (rn, s);
					RfsNetDebug (rn, RN_WARN | RN_S | RN_C,
					      "RfsNetwsrv: garbage type\n");
					goto garbage;

				case T_BIND_REQ:
					ackp = (union T_primitives *) tmp->b_rptr;
					len = pptr->bind_req.ADDR_length;
					if (len)
						pa = (struct rfs_addr *)
						    (mp->b_rptr + pptr->bind_req.ADDR_offset);

					if ((mp->b_wptr - mp->b_rptr) < sizeof (struct T_bind_req) ||
					    (len &&
					     ((mp->b_wptr - mp->b_rptr) !=
					      (len + pptr->bind_req.ADDR_offset)))) {
						RfsNetError (mp, tmp, TBADDATA, 0);
						goto bind_err;
					}
					if (len < 0 || len > RFSNET_MAXADDRSIZE) {
						RfsNetError (mp, tmp, TBADADDR, 0);
						goto bind_err;
					}
					/*
					 * If port number is supplied check
					 * to make sure that no one else is
					 * bound to it. Else alloc the next
					 * free one. 
					 */
					if (len == sizeof (struct rfs_addr) && pa->port != 0) {
						if (RfsNetInUse (pa->port) == -1) {
							RfsNetError (mp, tmp, TNOADDR, 0);
							goto bind_err;
						}
						else
							port = pa->port;
					}
					else {
						if ((port = allocport ()) == -1) {
							RfsNetError (mp, tmp, TNOADDR, 0);
							goto bind_err;
						}
					}
					if (rn->lup_flags & RFSNET_BOUND) {
						RfsNetError (mp, tmp, TOUTSTATE, 0);
						goto bind_err;
					}
					RfsNetDebug (rn, RN_C | RN_S,
						     "\tT_BIND_REQ: len %d port %d addr 0x%x\n",
					     len, port, len ? pa->addr : 0);
					rn->lup_flags |= RFSNET_BOUND;
					rn->lup_baddr[PORTADDR] = port;
					rn->lup_baddr[NETADDR] = len ? pa->addr : 0;
					rn->lup_aqlen = pptr->bind_req.CONIND_number;
					ackp->bind_ack.PRIM_type = T_BIND_ACK;
					ackp->bind_ack.ADDR_length = sizeof (struct rfs_addr);
					ackp->bind_ack.ADDR_offset = sizeof (struct T_bind_ack);
					ackp->bind_ack.CONIND_number = rn->lup_aqlen;
					tmp->b_wptr += sizeof (struct T_bind_ack);
					((struct rfs_addr *) tmp->b_wptr)->addr = len ? pa->addr : 0;
					((struct rfs_addr *) tmp->b_wptr)->port = port;
					tmp->b_wptr += sizeof (struct rfs_addr);
					tmp->b_datap->db_type = M_PROTO;
					event = TE_BIND_ACK;
			bind_err:
					if (event == -1)
						event = TE_ERROR_ACK;
					break;

				case T_UNBIND_REQ:
					if ((mp->b_wptr - mp->b_rptr) !=
					    sizeof (struct T_unbind_req))
						RfsNetError (mp, tmp, TBADDATA, 0);
					else if (!(rn->lup_flags & RFSNET_BOUND))
						RfsNetError (mp, tmp, TOUTSTATE, 0);
					else {
						/*
						 * Some more checks would be
						 * needed before unbinding 
						 */
						rn->lup_flags &= ~RFSNET_BOUND;
						rn->lup_baddr[PORTADDR] = 0;
						rn->lup_baddr[NETADDR] = 0;
						RfsNetSetOk (mp, tmp);
						event = TE_OK_ACK1;
					}
					if (event == -1)
						event = TE_ERROR_ACK;
					break;

				case T_CONN_REQ:
					ackp = (union T_primitives *) tmp->b_rptr;
					len = pptr->conn_req.DEST_length;
					if (len != sizeof (struct rfs_addr)) {
						RfsNetError (mp, tmp, TBADADDR, 0);
						goto conn_err;
					}
					pa = (struct rfs_addr *) (mp->b_rptr +
						pptr->bind_req.ADDR_offset);

					if (pa->port == 0 || pa->addr == 0) {
						RfsNetError (mp, tmp, TBADADDR, 0);
						goto conn_err;
					}
					if (pptr->conn_req.OPT_length ||
					    pptr->conn_req.OPT_offset) {
						RfsNetError (mp, tmp, TNOTSUPPORT, 0);
						goto conn_err;
					}
					if ((mp->b_wptr - mp->b_rptr) < sizeof (struct T_conn_req) ||
					    (mp->b_wptr - mp->b_rptr) !=
					(len + pptr->conn_req.DEST_offset)) {
						RfsNetError (mp, tmp, TBADDATA, 0);
						goto conn_err;
					}
					/*
					 * XXX Figure out an error ACK from
					 * the other side when nobody is
					 * bound to the port that we're
					 * trying to talk to. 
					 */

					if (!(rn->lup_flags & RFSNET_BOUND)) {
						RfsNetError (mp, tmp, TOUTSTATE, 0);
						goto conn_err;
					}
					rn->lup_daddr[NETADDR] = pa->addr;
					rn->lup_daddr[PORTADDR] = pa->port;
					error = RfsNetConnInd (rn, mp);
					if (error)
						RfsNetError (mp, tmp, hiword (error), loword (error));
					else {
						RfsNetSetOk (mp, tmp);
						event = TE_OK_ACK1;
					}
			conn_err:
					if (event == -1)
						event = TE_ERROR_ACK;
					break;

				case T_CONN_RES:
					ackp = (union T_primitives *) tmp->b_rptr;
					if (pptr->conn_res.OPT_length || pptr->conn_res.OPT_offset)
						RfsNetError (mp, tmp, TNOTSUPPORT, 0);
					else if ((mp->b_wptr - mp->b_rptr) !=
						 sizeof (struct T_conn_res))
						RfsNetError (mp, tmp, TBADDATA, 0);
					else if (error = RfsNetConnect (rn, mp, &event))
						RfsNetError (mp, tmp, hiword (error),
							     loword (error));
					else {
						rn->lup_qlen--;
						RfsNetSetOk (mp, tmp);
					}
					if (event == -1)
						event = TE_ERROR_ACK;
					break;

				case T_DATA_REQ:
				case T_EXDATA_REQ:
					{
						int             rtn;

						ASSERT (rn->lup_flags & RFSNET_CONNECTED);
						RfsNetDebug (rn, RN_C | RN_S, "%s\n",
							     pptr->type == T_DATA_REQ ?
							     "T_DATA_REQ" : "T_EXDATA_REQ");
						pptr->type = (pptr->type == T_DATA_REQ) ?
						  T_DATA_IND : T_EXDATA_IND;
						rtn = rfs_to_net (rn, mp);
						/*
						 * Ideally the destination
						 * state need to be changed.
						 * But we are avoiding it
						 * because WE KNOW that a
						 * state change does not
						 * occur in data indications 
						 */
						freemsg (tmp);
						RfsNetUnlock (rn, s);
						if (rtn) {
							/*
							 * The network became
							 * locked between the
							 * time we last
							 * checked and the
							 * call to
							 * rfs_to_net().  Put
							 * the message back
							 * on the queue and
							 * return. 
							 */
							putbq (q, mp);
							return (0);
						}
						else
							continue;	/* Don't break */
					}

				case T_DISCON_REQ:
					RfsNetDebug (rn, RN_S | RN_C, "\tT_DISCON_REQ:\n");

					if ((mp->b_wptr - mp->b_rptr) != sizeof (struct T_discon_req))
						RfsNetError (mp, tmp, TBADDATA, 0);
					else if (error = RfsNetDisconnect (rn, mp, &event))
						RfsNetError (mp, tmp, hiword (error), loword (error));

					if (event == -1) {
						event = TE_ERROR_ACK;
					}
					else {
						RfsNetSetOk (mp, tmp);
					}
					break;

				case T_UNITDATA_REQ:
				case T_OPTMGMT_REQ:
				case T_ORDREL_REQ:
					/*
					 * The following messages should
					 * never occur! 
					 */
				case T_CONN_IND:
				case T_CONN_CON:
				case T_DISCON_IND:
				case T_DATA_IND:
				case T_EXDATA_IND:
				case T_INFO_ACK:
				case T_BIND_ACK:
				case T_ERROR_ACK:
				case T_OK_ACK:
				case T_UNITDATA_IND:
				case T_UDERROR_IND:
				case T_OPTMGMT_ACK:
				case T_ORDREL_IND:

					RfsNetError (mp, tmp, TNOTSUPPORT, 0);
					break;
				}	/* End switch */
				if (RfsNetState (rn, event))
					panic ("RfsNetwsrv: can not change state\n");
				RfsNetUnlock (rn, s);
				qreply (q, tmp);
				freemsg (mp);
				break;
			}		/* end case M_PROTO */
		}			/* end switch */
	}
	if (rn->lup_flags & RFSNET_CLOSING) {
		rn->lup_flags &= ~RFSNET_CLOSING;
		wakeup (&rn->lup_flags);
	}
out1:
	if (mp)
		freemsg (mp);
out:
	return;
}


int
RfsNetrsrv (q)
	queue_t        *q;
{
	printf ("rfs_net: Read server called\n");
}

int
RfsNetclose (q)
	queue_t        *q;
{
	register struct rfsnet *rn;
	int             event, s;

	rn = (struct rfsnet *) q->q_ptr;
	ASSERT (rn->lup_flags & RFSNET_INUSE);
	RfsNetDebug (rn, RN_G, "Closed %d\n", rn - rfs_net);
	s = splstr ();
	/*
	 * Wait for the stream to be flushed.  RfsNetDisconnect can't be
	 * called with a packet in progress. 
	 */
	while (q->q_first || IS_LOCKED (rn)) {
		RfsNetDebug (rn, RN_G, "RfsNetclose: flushing queue\n");
		rn->lup_flags |= RFSNET_CLOSING;
		sleep (&rn->lup_flags, RFSNET_SLP);
	}
	splx (s);

	/*
	 * If we are connected to another stream, break the linkage. The
	 * disconnection is not well developed. 
	 */

	if (rn->lup_flags & RFSNET_CONNECTED) {
		ASSERT (rn->lup_flags & RFSNET_BOUND);
		s = RfsNetLock (rn);
		if (RfsNetState (rn, TE_DISCON_REQ))
			printf ("RfsNetclose: RfsNetState error for T_DISCON_IND\n");
		else if (RfsNetDisconnect (rn, NULL, &event))
			printf ("RfsNetclose: RfsNetState error for RfsNetDisconnect\n");
		else if (RfsNetState (rn, TE_OK_ACK1))
			printf ("RfsNetclose: RfsNetState error for T_DISCON_ACK1\n");
		else if (RfsNetState (rn, TE_UNBIND_REQ))
			printf ("RfsNetclose: RfsNetState error for T_UNBND_REQ\n");
		else {
			rn->lup_flags &= ~(RFSNET_BOUND | RFSNET_CONNECTED);
			rn->lup_baddr[PORTADDR] = 0;
			rn->lup_baddr[NETADDR] = 0;
			RfsNetState (rn, TE_OK_ACK1);
		}
		RfsNetUnlock (rn, s);
	}
	rn->qptr = NULL;
	rn->lup_flags = 0;
}


/*
 * Sets an error reply 
 */
void
RfsNetError (mp, tmp, tlierror, unixerror)
	mblk_t         *mp, *tmp;
{
	register union T_primitives *pptr;
	register struct T_error_ack *ackp;

	pptr = (union T_primitives *) mp->b_rptr;
	ackp = (struct T_error_ack *) tmp->b_rptr;
	ackp->PRIM_type = T_ERROR_ACK;
	ackp->ERROR_prim = pptr->type;
	ackp->TLI_error = tlierror;
	ackp->UNIX_error = unixerror;
	tmp->b_wptr = tmp->b_rptr + sizeof (struct T_error_ack);
	tmp->b_datap->db_type = M_PCPROTO;
	RfsNetDebug (0, RN_WARN, "RfsNetError: type 0x%x\n", pptr->type);
}


/*
 * Sets an ok reply 
 */
void
RfsNetSetOk (mp, tmp)
	mblk_t         *mp, *tmp;
{
	register union T_primitives *pptr;
	register struct T_ok_ack *ackp;

	pptr = (union T_primitives *) mp->b_rptr;
	ackp = (struct T_ok_ack *) tmp->b_rptr;
	ackp->PRIM_type = T_OK_ACK;
	ackp->CORRECT_prim = pptr->type;
	tmp->b_wptr = tmp->b_rptr + sizeof (struct T_ok_ack);
	tmp->b_datap->db_type = M_PCPROTO;
}

/*
 * Used to check if an address is already in use while binding If the address
 * is in use -1 is returned, else the address. 
 */
 /* static */ int
RfsNetInUse (port)
	register unsigned port;
{
	register struct rfsnet *rn;

	for (rn = &rfs_net[0]; rn < &rfs_net[rfsnet_cnt]; rn++)
		if ((rn->lup_flags & RFSNET_INUSE) &&
		    (rn->lup_baddr[PORTADDR] == port))
			return (-1);
	return (port);
}


/*
 * Allocate an used address for binding. The fields in the message are
 * directly updated. For now there is a bound for the search that is made. It
 * may not be there forever!! 
 */
 /* static */ int
allocport ()
{
	register unsigned addr;
	register struct rfsnet *rn;
	register        inuse;

	for (addr = RFSNET_MINPORTNO; addr <= RFSNET_MAXPORTNO; addr++) {
		inuse = 0;
		for (rn = &rfs_net[0]; rn < &rfs_net[rfsnet_cnt]; rn++)
			if ((rn->lup_flags & RFSNET_INUSE) &&
			    (rn->lup_baddr[PORTADDR] == addr)) {
				inuse++;
				break;
			}
		if (!inuse)
			return (addr);
	}
	return (-1);
}


/*
 * Send a connect indication on the destination message que 
 */
 /* static */ int
RfsNetConnInd (rn, mp)
	struct rfsnet  *rn;
	mblk_t         *mp;
{
	register mblk_t *tmp;
	register struct T_conn_ind *msgp;
	int             retval;

	if ((tmp = allocb (sizeof (*msgp) + RFSNET_MAXADDRSIZE, BPRI_MED)) == NULL)
		return (makerr (TSYSERR, EAGAIN));
	msgp = (struct T_conn_ind *) tmp->b_rptr;
	msgp->PRIM_type = T_CONN_IND;
	msgp->SRC_length = sizeof (struct rfs_addr);
	msgp->SRC_offset = sizeof (*msgp);
	msgp->OPT_length = 0;		/* Not supported */
	msgp->OPT_offset = 0;
	msgp->SEQ_number = (int) (rn - rfs_net);
	tmp->b_wptr = tmp->b_rptr + sizeof (*msgp);

	/*
	 * If no current address was giving then grab
	 * rn->lup_ac->ac_ipaddr.s_addr 
	 */
	if (rn->lup_baddr[NETADDR] == 0)
		rn->lup_baddr[NETADDR] = rn->lup_ac->ac_ipaddr.s_addr;

	RfsNetDebug (rn, RN_C, "RfsNetConnIn: saddr %x.%x daddr %x.%x\n",
		     rn->lup_baddr[NETADDR], rn->lup_baddr[PORTADDR],
		     rn->lup_daddr[NETADDR], rn->lup_daddr[PORTADDR]);
	((struct rfs_addr *) tmp->b_wptr)->addr = rn->lup_baddr[NETADDR];
	((struct rfs_addr *) tmp->b_wptr)->port = rn->lup_baddr[PORTADDR];
	tmp->b_wptr += sizeof (struct rfs_addr);
	tmp->b_datap->db_type = M_PROTO;

	if (rfs_to_net (rn, tmp))
		panic ("RfsNetConnInd");
	else
		return (0);
err:
	freemsg (tmp);
	return (retval);
}


/*
 * RfsNetConnect : connects the two streams so that data transfer can
 * continue. If connection succeeds a message is sent to the connected
 * channel. It is upto the caller to update the qlen. 
 */
 /* static */ int
RfsNetConnect (msg, mp, rtneventp)
	struct rfsnet  *msg;		/* Channel in which msg arrived	 */
	mblk_t         *mp;		/* Acceptance message		 */
	uint           *rtneventp;	/* Event to be considered for msg */
{
	register struct T_conn_con *conptr;
	register struct T_conn_res *resptr;
	register struct rfsnet *rn;	/* Channel of acceptance */
	register struct rfs_net_match *seq;
	mblk_t         *tmp;
	queue_t        *accq;		/* Q for rn */
	int             s, rtn;

	resptr = (struct T_conn_res *) mp->b_rptr;
	accq = resptr->QUEUE_ptr;
	rn = (struct rfsnet *) accq->q_ptr;
	if (rn < &rfs_net[0] || rn >= &rfs_net[rfsnet_cnt])
		return (makerr (TBADDATA, 0));
	if (WR (accq) != rn->qptr)
		return (makerr (TBADDATA, 0));
	if (!(rn->lup_flags & RFSNET_INUSE) ||
	    !(rn->lup_flags & RFSNET_BOUND) ||
	    (rn->lup_flags & RFSNET_CONNECTED))
		return (makerr (TBADDATA, 0));
	if (msg == rn && msg->lup_qlen > 1)
		return (makerr (TBADF, 0));
	if ((tmp = allocb (sizeof (*conptr) + RFSNET_MAXADDRSIZE, BPRI_MED)) == NULL)
		return (makerr (TSYSERR, EAGAIN));

	for (seq = seq_matches; seq < &seq_matches[rfsnet_cnt]; seq++)
		if (seq->SEQ_number == resptr->SEQ_number)
			break;
	if (seq == &seq_matches[rfsnet_cnt])
		return (makerr (TSYSERR, EAGAIN));

	rn->lup_daddr[NETADDR] = seq->addr;
	rn->lup_daddr[PORTADDR] = seq->port;
	seq->addr = 0;			/* clear the slot */

	s = RfsNetLock (rn);
	if (rn == msg)
		*rtneventp = TE_OK_ACK2;
	else {
		if (rtn = RfsNetState (rn, TE_PASS_CONN)) {
			freemsg (tmp);
			RfsNetUnlock (rn, s);
			return (rtn);
		}
		*rtneventp = (msg->lup_qlen == 1) ? TE_OK_ACK3 : TE_OK_ACK4;
	}
	conptr = (struct T_conn_con *) tmp->b_rptr;
	conptr->PRIM_type = T_CONN_CON;
	conptr->RES_length = sizeof (struct rfs_addr);	/* HARD-WIRED */
	conptr->RES_offset = sizeof (*conptr);
	conptr->OPT_length = 0;		/* Not supported */
	conptr->OPT_offset = 0;
	tmp->b_wptr = tmp->b_rptr + sizeof (*conptr);
	((struct rfs_addr *) tmp->b_wptr)->port = rn->lup_baddr[PORTADDR];
	((struct rfs_addr *) tmp->b_wptr)->addr = rn->lup_baddr[NETADDR];
	tmp->b_wptr += sizeof (struct rfs_addr);
	tmp->b_datap->db_type = M_PROTO;

	/*
	 * Once we have connect the new circuit reset the controlling
	 * circuit's incomming sequence number so that it can accept new
	 * T_CONN_IND requests. 
	 */
	msg->lup_rseq = 0;

	if (rfs_to_net (rn, tmp))
		panic ("RfsNetConnect");
	rn->lup_flags |= RFSNET_CONNECTED;

	RfsNetUnlock (rn, s);
	return (0);
}

/*
 * RfsNetState: Changes the state of the provider. If a valid transition
 * return 0, else return an integer with t_errno in upper two bytes and errno
 * in lower two bytes. All kinds of out the order behaviour needs to be taken
 * care of in this routine! 
 */
RfsNetState (rn, event)
	register struct rfsnet *rn;
	register unsigned event;
{
	register uint   newstate;
	register int    rtn;

	if (!(rn->lup_flags & RFSNET_CHST))
		panic ("RfsNet: bad change of state\n");

	newstate = ti_statetbl[event][rn->lup_state];
	RfsNetDebug (rn, RN_NOISE, "(%d.%d) state %d event %d newstate %d\n",
		     rn->lup_baddr[PORTADDR], rn->lup_seq, rn->lup_state, event, newstate);

	if (event >= TE_NOEVENTS || rn->lup_state >= TS_NOSTATES)
		rtn = makerr (TBADDATA, 0);
	else if (newstate >= TS_NOSTATES) {
		RfsNetDebug (rn, RN_WARN, "RfsNetState: (%x.%x) from %d to %d\n",
			     rn->lup_baddr[NETADDR], rn->lup_baddr[PORTADDR],
			     rn->lup_state, event);
#ifdef RN_DEBUG
		if (rndebug & RN_WARN) {
			register int    i;

			printf ("Valid events for your state %d: ", rn->lup_state);
			for (i = 0; i < TE_NOEVENTS; i++)
				if (ti_statetbl[i][rn->lup_state] < TS_NOSTATES)
					printf ("%d ", i);
			printf ("\n");
		}
#endif
		/*
		 * All exceptions!! 
		 */
		if ((event == TE_DATA_REQ || event == TE_EXDATA_REQ) &&
		rn->lup_state != TS_DATA_XFER && rn->lup_state != TS_IDLE) {
			rtn = makerr (TSYSERR, EPROTO);
		}
		else
			rtn = makerr (TOUTSTATE, 0);
	}
	else {
		rn->lup_state = newstate;
		rtn = 0;
	}

	return (rtn);
}

/*
 * RfsNetDisconnect: Processing related to disconnect request. This request
 * may come after/while data-xfer is going on or it may be a response to
 * connect indication. Acknowledgement event would be different depending on
 * this. 
 */
int
RfsNetDisconnect (rn, mp, rtneventp)
	register struct rfsnet *rn;	/* Channel in which msg arrived */
	mblk_t         *mp;		/* Acceptance message */
	uint           *rtneventp;	/* Event to be considered for rn */
{
	register struct T_discon_ind *dindptr;
	register struct T_discon_req *dreqptr;
	register uint   rtnevent;
	mblk_t         *tmp;
	int             s;

	if (mp)
		dreqptr = (struct T_discon_req *) mp->b_rptr;
	else
		dreqptr = NULL;
	rtnevent = TE_ERROR_ACK;	/* default */
	if ((tmp = allocb (sizeof (*dindptr), BPRI_MED)) == NULL)
		return (makerr (TSYSERR, EAGAIN));

	switch (rn->lup_state) {
	default:
	case TS_WACK_DREQ6:
		/*
		 * Waiting T_CONN_CON 
		 */
	case TS_WACK_DREQ10:
	case TS_WACK_DREQ11:
		freemsg (tmp);
		/*
		 * Don't know yet, what to do 
		 */
		RfsNetDebug (rn, RN_G, "RfsNetDisconnect: what should I do with state %d\n",
			     rn->lup_state);
		return;

	case TS_WACK_DREQ7:
		/*
		 * In response to conn_ind 
		 */
		if (!dreqptr)
			return (makerr (TOUTSTATE, 0));
		if (dreqptr->SEQ_number < 0 || dreqptr->SEQ_number > rfsnet_cnt)
			return (makerr (TBADDATA, 0));
		rtnevent = (rn->lup_qlen == 1) ? TE_OK_ACK3 : TE_OK_ACK4;
		break;

	case TS_WACK_DREQ9:
		rtnevent = TE_OK_ACK1;
		break;
	}

	dindptr = (struct T_discon_ind *) tmp->b_rptr;
	dindptr->PRIM_type = T_DISCON_IND;
	dindptr->DISCON_reason = 0;	/* Not supported */
	dindptr->SEQ_number = (rn - rfs_net);
	tmp->b_wptr = tmp->b_rptr + sizeof (*dindptr);
	tmp->b_datap->db_type = M_PROTO;
	if (rfs_to_net (rn, tmp))
		panic ("RfsNetDisconnect");
	rn->lup_flags &= ~RFSNET_CONNECTED;
	*rtneventp = rtnevent;
	return (0);
}


/*------------------------------------------------------------------------------
 * Ethernet interface routines.
 * These routines should be placed in a seperate file creating a generic
 * streams to ethernet interface.  A few things still need to be cleaned up
 * though.  The main one being the selection of an ethernet interface.
 * Currently there is a *hack* in if_ex.c which calls RfsNetAttach giving
 * the drivers ifnet pointer.
 *------------------------------------------------------------------------------
 */

/*
 * Take a streams mblk and transfer it to mbuf's then passing it onto the
 * output interface. 
 */
rfs_to_net (rn, mblk0)
	register struct rfsnet *rn;
	register mblk_t *mblk0;
{
	register struct mbuf *top;
	struct sockaddr dest;
	register int    s;

	LOCK_OUTBOUND (rn);
	top = mblk_to_mbuf (rn->lup_daddr[PORTADDR], mblk0, rn->lup_seq);
	if (top != (struct mbuf *) - 1) {
		/*
		 * Setup up the dest. address 
		 */
		dest.sa_family = AF_RFS;
		RfsNetDebug (rn, RN_O, "rfs_to_net: (%x.%x)\n",
			     rn->lup_daddr[NETADDR],
			     (mtod (top, struct rfs_net_pkt *))->port);
		((struct sockaddr_in *) & dest)->sin_addr.s_addr =
		    rn->lup_daddr[NETADDR];
		(*rn->lup_ifp->if_output) (rn->lup_ifp, top, &dest);

		rn->lup_m = mblk0;
		timeout (RfsNetTimer, rn, (rn->lup_retry * HZ) / 2);
		return (0);
	}
	else {
		UNLOCK_OUTBOUND (rn);
		return (2);
	}
}

netrfsintr ()
{
	register struct rfsnet *rn;
	register struct mbuf *m0;
	register mblk_t *mblk;
	register union T_primitives *pptr;
	register struct rfs_net_pkt *r;
	struct ifnet   *ifp;
	struct sockaddr dest;
	int             port, s, seq;

	RfsNetDebug (0, RN_I, "netrfsintr: called\n");
loop:
	s = splimp ();
	IF_DEQUEUEIF (&rfsintrq, m0, ifp);
	if (m0 == 0) {
		RfsNetDebug (0, RN_I, "netrfsintr: finished\n");
		splx (s);
		return;
	}

	if (m0->m_len < sizeof (struct rfs_net_pkt)) {
		RfsNetDebug (0, RN_I | RN_ERR,
			     "netrfsintr: Bad incoming packet, no header\n");
		m_freem (m0);
		goto loop;
	}

	/*
	 * mbuf-to_mblk will free m0. 
	 */
	port = seq = 0;
	mblk = mbuf_to_mblk (&seq, &port, m0);

	RfsNetDebug (0, RN_I, "netrfsintr: mblk = 0x%x\n", mblk);
	if (mblk == (mblk_t *) - 1) {
		RfsNetDebug (0, RN_I | RN_WARN,
			 "netrfsintr: Can't convert mbuf to mblk (%d.%d)\n",
			     port, seq);
		goto loop;
	}

	/*
	 * Find destination queue from the port number. 
	 */
	for (rn = rfs_net; rn < &rfs_net[rfsnet_cnt]; rn++)
		if ((rn->lup_flags & RFSNET_INUSE) &&
		    rn->lup_baddr[PORTADDR] == port)
			break;
	if (rn == &rfs_net[rfsnet_cnt]) {
		RfsNetDebug (0, RN_I | RN_WARN,
		      "netrfsintr: Incoming packet with bad port (%d.%d)\n",
			     port, seq);
		goto bad;
	}

	if (mblk == 0) {		/* Incomming packet was a ack */
		if (seq == rn->lup_seq) {
			RfsNetDebug (rn, RN_ACK, "netrfsintr: got ack (%d.%d)\n",
				     rn->lup_baddr[PORTADDR], seq);
			untimeout (RfsNetTimer, rn);
			freemsg (rn->lup_m);
			rn->lup_m = 0;
			rn->lup_seq++;
			rn->lup_retry = 1;
			UNLOCK_OUTBOUND (rn);
		}
		else
			RfsNetDebug (rn, RN_I | RN_WARN,
			 "netrfsintr: ERROR: act is %d should be (%d.%d)\n",
				 seq, rn->lup_baddr[PORTADDR], rn->lup_seq);
	}
	else {
		/*
		 * We may get a new port number after RfsNetPeek has been
		 * called. 
		 */
		if (RfsNetPeek (rn, mblk)) {
			RfsNetDebug (rn, RN_ERR, "RfsNet: Bad incomming packet\n");
			goto bad;
		}

		/*
		 * send ack back to other side. 
		 */
		port = rn->lup_daddr[PORTADDR];
		RfsNetDebug (rn, RN_ACK, "netrfsintr: sending ack  (%d.%d)\n",
			     port, seq);
		bzero (&dest, sizeof (dest));
		MGET (m0, M_DONTWAIT, MT_DATA);
		if (m0 == 0)
			goto bad;
		r = mtod (m0, struct rfs_net_pkt *);
		r->len = RNACK;
		r->port = port;
		r->seq = seq;
		m0->m_len = sizeof (struct rfs_net_pkt);
		dest.sa_family = AF_RFS;
		((struct sockaddr_in *) & dest)->sin_addr.s_addr =
		    rn->lup_daddr[NETADDR];
		(*rn->lup_ifp->if_output) (rn->lup_ifp, m0, &dest);

		/*
		 * Tallyho If we've already seen the packet then don't send
		 * it up stream. 
		 */
		if (seq <= rn->lup_rseq) {
			RfsNetDebug (rn, RN_I, "netrfsintr: packets already been sent upstream\n");
			goto bad;
		}
		else {
			rn->lup_rseq = seq;
			putnext (RD (rn->qptr), mblk);
		}
	}

	goto loop;

bad:
	freemsg (mblk);
	goto loop;
}

RfsAttach (ifp, ac)
	register struct ifnet *ifp;
	register struct arpcom *ac;
{
	register struct rfsnet *rn;

	for (rn = rfs_net; rn < &rfs_net[rfsnet_cnt]; rn++) {
		rn->lup_ifp = ifp;
		rn->lup_ac = ac;
	}
}

/*
 * Sometimes we need to take a look at the incomming packets so that we can
 * change our state as necessary. 
 */
RfsNetPeek (rn, mblk)
	register struct rfsnet *rn;
	register mblk_t *mblk;
{
	register union T_primitives *pptr;
	register struct rfs_net_match *m;
	register int    rtn = 0;

	pptr = (union T_primitives *) mblk->b_rptr;
	switch (pptr->type) {
	default:
		RfsNetDebug (rn, RN_NOISE, "RfsNetPeek: Default case type = %d\n",
			     pptr->type);
		break;

		/*
		 * Check message type to see if it's T_CONN_CON which means
		 * that the client is getting a new port number to talk with. 
		 */
	case T_CONN_CON:
		RfsNetDebug (rn, RN_I, "RfsNetPeek:\tOld port %d, \n",
			     rn->lup_daddr[PORTADDR]);
		rn->lup_flags |= RFSNET_CONNECTED;
		rn->lup_daddr[PORTADDR] =
		    ((struct rfs_addr *) (mblk->b_rptr +
					  sizeof (struct T_conn_con)))->port;
		if (rn->lup_flags & RFSNET_CHST)
			rtn = 1;
		else {
			rn->lup_flags |= RFSNET_CHST;
			if (RfsNetState (rn, TE_CONN_CON))
				rtn = 1;
			rn->lup_flags &= ~RFSNET_CHST;
		}
		RfsNetDebug (rn, RN_I, "\t\t\tNew port %d\n",
			     rn->lup_daddr[PORTADDR]);
		break;

		/*
		 * Connection Indication is a message from client to server.
		 * This packet contains the port and net address of the
		 * client which needs to be saved. Change the dest net and
		 * port address so that an ack can be sent to the correct
		 * location. 
		 */
	case T_CONN_IND:
		for (m = seq_matches; m < &seq_matches[rfsnet_cnt]; m++)
			if (m->addr == 0) {	/* empty */
				m->port = rn->lup_daddr[PORTADDR] =
				    ((struct rfs_addr *) (mblk->b_rptr +
					 sizeof (struct T_conn_ind)))->port;
				m->addr = rn->lup_daddr[NETADDR] =
				    ((struct rfs_addr *) (mblk->b_rptr +
					 sizeof (struct T_conn_ind)))->addr;
				m->SEQ_number =
				    ((struct T_conn_ind *) mblk->b_rptr)->SEQ_number;
				break;
			}
		if (m == &seq_matches[rfsnet_cnt]) {
			RfsNetDebug (rn, RN_I, "RfsNetPeek: Can't store sequence %d\n",
			  ((struct T_conn_ind *) mblk->b_rptr)->SEQ_number);
			rtn = 1;
			break;
		}

		if (rn->lup_flags & RFSNET_CHST)
			rtn = 1;
		else {
			rn->lup_flags |= RFSNET_CHST;
			if (RfsNetState (rn, TE_CONN_IND))
				rtn = 1;
			rn->lup_flags &= ~RFSNET_CHST;
		}
		RfsNetDebug (rn, RN_I, "RfsNetPeek: T_CONN_IND from (%x.%x) SEQ %d\n",
			     m->addr, m->port, m->SEQ_number);
		break;

	case T_DISCON_IND:
		/*
		 * Always let the ack happen if we're not locked. 
		 */
		RfsNetDebug (rn, RN_I, "RfsNetPeek: T_DISCON_IND\n");
		rn->lup_flags &= ~RFSNET_CONNECTED;
		if (rn->lup_flags & RFSNET_CHST)
			rtn = 1;
		else {
			rn->lup_flags |= RFSNET_CHST;
			switch (rn->lup_state) {
			case TS_WRES_CIND:
				RfsNetState (rn, rn->lup_qlen == 1 ? TE_DISCON_IND2 : TE_DISCON_IND3);
				break;

			default:
				RfsNetState (rn, TE_DISCON_IND1);
				break;
			}
			rn->lup_flags &= ~RFSNET_CHST;
		}
	}
	return (rtn);
}

/*
 * During state changes lock the driver.  This must be done at splstr() to
 * prevent the streams scheduler from sneaking in and causing mayhem. 
 */
RfsNetLock (rn)
	register struct rfsnet *rn;
{
	while (rn->lup_flags & RFSNET_CHST)
		sleep (rn, RFSNET_SLP);
	rn->lup_flags |= RFSNET_CHST;
	return (splstr ());
}


RfsNetUnlock (rn, s)
	register struct rfsnet *rn;
	register int    s;
{
	rn->lup_flags &= ~RFSNET_CHST;
	wakeup (rn);
	splx (s);
}

RfsNetDebug (rn, mask, fmt, x1)
	register struct rfsnet *rn;
	unsigned        mask;
	char           *fmt;
{
	if (rndebug & mask) {
		/*
		 * The third argument should be the TOCONS but, the only
		 * place that it is defined is in sys/subr_prf.c 
		 */
		if (rn != 0)
			printf ("[%d]: ", rn - rfs_net);
		else
			printf ("[x]: ");

		prf (fmt, &x1, 
#ifdef	GWS
		    ((gptype & GPEXISTS) ? TOGWS : TOCONS) | tocons | TOLOG,
#else	GWS
		    TOCONS | tocons | TOLOG,
#endif	GWS
		     (struct tty *) 0);
	}
}


/*
 * mblk_to_mbuf is called by rfs_to_net.  Convert mblk's to an mbuf chain.
 */
struct mbuf    *
mblk_to_mbuf (port, mblk0, seq)
	int             port, seq;
	mblk_t         *mblk0;
{
	register struct mbuf **mp, *top = 0, *m;
	register mblk_t *mblk;
	register unsigned char *p;
	register int    len, n;
	register struct rfs_net_pkt *r;

	if (mblk0 == 0)
		return ((struct mbuf *) - 1);

	for (mblk = mblk0; mblk; mblk = mblk->b_cont) {
		MGET (m, M_DONTWAIT, MT_DATA);
		if (m == 0)
			return ((struct mbuf *) - 1);

		if (top == 0)
			top = m;
		else
			*mp = m;

		len = mblk->b_wptr - mblk->b_rptr;
		p = mblk->b_rptr;
		m->m_len = sizeof (struct rfs_net_pkt);
		r = mtod (m, struct rfs_net_pkt *);
		r->port = port;
		r->len = len;
		r->type = mblk->b_datap->db_type;
		r->seq = seq;
		RfsNetDebug (0, RN_O, "mblk_to_mbuf: db_type %d, len %d, seq %d\n",
			     r->type, r->len, r->seq);

#ifdef RN_DEBUG
		if ((len == 1024) && (rndebug & RN_NOISE)) {
			printf ("mblk_to_mbuf: Packet = :");
			for (n = 0; n < 16; n++)
				printf ("%x:", p[n]);
			printf ("\n");
		}
#endif

		mp = &m->m_next;
		while (len) {
			MGET (m, M_DONTWAIT, MT_DATA);
			if (m == 0)
				goto bad;
			n = MIN (len, MLEN);
			bcopy (p, mtod (m, unsigned char *), n);
			m->m_len = n;
			len -= n;
			p += n;
			*mp = m;
			mp = &m->m_next;
		}
	}
	MGET (m, M_DONTWAIT, MT_DATA);
	if (m == 0)
		goto bad;
	*mp = m;
	m->m_len = sizeof (struct rfs_net_pkt);
	r = mtod (m, struct rfs_net_pkt *);
	r->len = RNLAST;
	r->seq = seq;
	r->port = port;

	return (top);
bad:
	m_freem (top);
	return ((struct mbuf *) - 1);
}


mblk_t         *
mbuf_to_mblk (seq, port, m0)
	int            *seq, *port;
	register struct mbuf *m0;
{
	register int    n;
	register mblk_t *tblk = 0, *mblk;
	register struct mbuf *m;
	register struct rfs_net_pkt *r;

	while (m0) {
		if ((m0->m_len < sizeof (struct rfs_net_pkt)) &&
		    (m0 = m_pullup (m0, sizeof (struct rfs_net_pkt))) == 0) {
			RfsNetDebug (0, RN_ERR, "mbuf_to_mblk: not enough data\n");
			goto bad;
		}

		r = mtod (m0, struct rfs_net_pkt *);

		if (*seq == 0)
			*seq = r->seq;
		if (*port == 0)
			*port = r->port;

		if (r->len == RNLAST || r->len == RNACK) {
			if (r->len == RNLAST && tblk == 0) {
				RfsNetDebug (0, RN_ERR,
				    "mbuf_to_mblk: First packet is last\n");
				goto bad;
			}
			RfsNetDebug (0, RN_I, "mbuf_to_mblk: %s to (%d.%d)\n",
			    r->len == RNLAST ? "Data" : "Ack", *port, *seq);
			break;
		}

		m0->m_off += sizeof (struct rfs_net_pkt);
		m0->m_len -= sizeof (struct rfs_net_pkt);

		mblk = allocb (r->len, BPRI_MED);
		if (mblk == (mblk_t *) 0) {
			RfsNetDebug (0, RN_ERR,
			  "mbuf_to_mblk: (%d.%d) no mblk's(%d) available\n",
				     port, seq, r->len);
			goto bad;
		}

#ifdef RN_DEBUG
		if ((r->len == 1024) && (rndebug & RN_NOISE)) {
			register char  *pp = mtod (m0, char *);

			printf ("mbuf_to_mblk: Packet = :");
			for (n = 0; n < 16; n++)
				printf ("%x:", *pp++);
			printf ("\n");
		}
#endif

		RfsNetDebug (0, RN_I, "mbuf_to_mblk: r->len = %d\n", r->len);
		while (r->len) {
			n = MIN (r->len, m0->m_len);
			bcopy (mtod (m0, char *), mblk->b_wptr, n);
			m0->m_len -= n;
			m0->m_off += n;
			mblk->b_wptr += n;
			r->len -= n;
			if (m0->m_len == 0) {
				MFREE (m0, m);
				m0 = m;
				if (m0 == 0) {
					RfsNetDebug (0, RN_ERR,
					 "mbuf_to_mblk: Not enough data\n");
					goto bad;
				}
			}
		}
		mblk->b_datap->db_type = r->type;
		if (tblk)
			linkb (tblk, mblk);
		else
			tblk = mblk;
	}

#ifdef RN_DEBUG
	/*
	 * At this point m0 should be the last mbuf on this chain.  We have
	 * free all of the others. 
	 */
	if (m0->m_next)
		RfsNetDebug (0, RN_ERR, "mbuf_to_mblk: m0 is not end of chain.\n");
#endif
	m_freem (m0);
	return (tblk);


bad:
	if (m0)
		m_freem (m0);
	if (tblk)
		freemsg (tblk);
	return ((mblk_t *) - 1);
}

BreakPoint (s)
{
	printf ("BreakPoint @ %s\n", s);
}


static int      Once;

void
RfsNetTimer (rn)
	register struct rfsnet *rn;
{
	register mblk_t *tmp;
	register int    s;

	if (!rn || !(rn->lup_flags & RFSNET_INUSE) || !rn->qptr) {
		if (rn->lup_m) {
			freemsg (rn->lup_m);
			rn->lup_m = 0;
		}
		RfsNetDebug (rn, RN_ERR, "RfsNetTimer: invalid rn 0x%x\n", rn);
		return;
	}

	RfsNetDebug (rn, RN_ACK | RN_WARN,
		     "RfsNetTimer: Lost packet (%d,%d) mblk 0x%x\n",
		     rn->lup_daddr[PORTADDR], rn->lup_seq, rn->lup_m);

	if (rn->lup_retry++ > RN_RETRIES) {
		/*
		 * When we fail to contact the other side use the current
		 * mblk to send a disconnect message upstairs. 
		 */
		RfsNetDebug (rn, RN_ACK | RN_ERR,
			     "RfsNetTimer: dropping packet (%d.%d)\n",
			     rn->lup_daddr[PORTADDR], rn->lup_seq);

		tmp = rn->lup_m;
		if (!tmp || !tmp->b_datap || !tmp->b_rptr) {
			RfsNetDebug (rn, RN_ERR,
				     "RfsNetTimer: bogus mblk(0x%x)->b_datap(0x%x)\n",
				     tmp, tmp ? tmp->b_datap : 0);
			freemsg (tmp);
		}
		else {
			tmp->b_datap->db_type = M_PROTO;
			((union T_primitives *) tmp->b_rptr)->type = T_DISCON_IND;
			putnext (RD (rn->qptr), tmp);
		}
		rn->lup_retry = 1;
		rn->lup_m = 0;
		UNLOCK_OUTBOUND (rn);
	}
	else {
		/*
		 * Don't do UNLOCK because that will cause an unnecessary
		 * wakeup. we simply don't this call to block. 
		 */
		s = splimp ();
		rn->lup_flags &= ~RFSNET_INPROGRES;
		switch (rfs_to_net (rn, rn->lup_m)) {
		case 2:
			timeout (RfsNetTimer, rn, (rn->lup_retry * HZ) / 2);
			rn->lup_flags |= RFSNET_INPROGRES;
			break;

		case 1:
			panic ("RfsNetTimer");
			break;

		default:
			break;
		}
		splx (s);
	}
}
#endif	defined(SYSV) && defined(RFS)
