#ifdef	SYSV
#include "sxt.h"
#if NSXT > 0
/*	Copyright (c) 1984 AT&T	*/
/*	  All Rights Reserved  	*/

/*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF AT&T	*/
/*	The copyright notice above does not evidence any   	*/
/*	actual or intended publication of such source code.	*/

/*
#ident	"@(#)kern-port:io/sxt.c	10.8"
/*
 * SXT --  Driver for shell layers
 */

#include "../h/types.h"
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/user.h"
#include "../h/conf.h"
#include "../h/errno.h"
#include "../h/proc.h"
#include "../h/tty.h"
#include "../h/file.h"
#include "../h/uio.h"
#include "../h/kernel.h"
#include "../sysv/sys/termio.h"
#include "../sysv/sys/ttold.h"
#include "../sysv/io/sxtreg.h"

#define	SXT_CNT	6

#define	ISPEED	B9600				/* for 4.3 tty handling */
#define	IFLAGS	(EVENP|ODDP|ECHO|CRMOD)
 
/*  A real terminal is associated with a number of virtual tty's.
 *  The real terminal remains the communicator to the low level
 *  driver,  while the virtual terminals and their queues are
 *  used by the upper level (ie., ttread, ttwrite).
 *
 *  real tty (tp)   
 *		linesw == SXT
 *		proc   == real driver proc routine
 *  virtual tty (vtp)
 *		linesw == original value of real tty
 *		proc   == SXTVTPROC
 */

/* Maximum size of a link structure */
#define LINKSIZE (sizeof(struct Link) + sizeof(struct Channel)*(MAXPCHAN-1))
#define LINKHEAD (sizeof(struct Link))

Link_p		linkTable[MAXLINKS];

#	if	SXTRACE == 1
char	tracing;		/* print osm msgs for tracing */
#	endif


char	sxtbusy[MAXLINKS], *sxtbuf ;

int	sxt_cnt = SXT_CNT;
char	sxt_buf[SXT_CNT*LINKSIZE] ;

Link_p	sxtalloc();

int sxtnullproc();
int sxtvtproc();



sxtopen(dev, flag)
{
	register Link_p lp;
	register chan;
	register struct tty *vtp;
	register bit;
	register int	oldpri;
	register struct proc *pp = u.u_procp;

	chan = CHAN(dev);

#if	SXTRACE == 1
	if (tracing)
		printf("!sxtopen: link %d, chan %d\n", LINK(dev), chan);
#endif

	/*  linkTable:  0 - unused, 1 - master slot reserved, 
	*			   >1 - master operational */
	if ((lp = linkTable[LINK(dev)]) == (Link_p)1) {
		return EBUSY;
	}

	if (lp == NULL) {
		if (chan != 0)
			/* Master on chan 0 must be opened first */
			return EINVAL;
		else
			linkTable[LINK(dev)] = (Link_p)1;
		return 0;		/* nothing else to do here */
	}

	if (chan == 0) {
		/* master already set up on this link	*/
		return EBUSY;
	}

	if (chan >= lp->nchans) {
		/* channel number out of range	*/
		return ENXIO;
	}

	bit = (1 << chan);
	if (lp->open & bit) {
		if ((flag & FEXCL) || (lp->xopen & bit)) {
			return ENXIO;
		}
	} else {
		/* open new channel */
		lp->open |= bit;

		if (lp->chans[chan].tty.svt_proc == sxtnullproc ||
				lp->chans[chan].tty.svt_proc == sxtvtproc)
			sv_ttyflush(&lp->chans[chan].tty, (FREAD | FWRITE));

#ifdef u3b
		kclear((char *) &lp->chans[chan], sizeof (struct Channel));
#else
		bzero((char *) &lp->chans[chan], sizeof (struct Channel));
#endif
		sxtlinit(&lp->chans[chan], chan, dev, lp->old,
			lp->line);
		if (flag & FEXCL)
			lp->xopen |= bit;
	}

	vtp = &lp->chans[chan].tty;

	/* do the pertinent part of ttopen() here */ 
	oldpri = spltty();
	sxt_setuniverse(vtp, lp->line);
	if (pp->p_pgrp == 0) {
		u.u_ttyp = vtp;
		u.u_ttyd = dev;
		if (vtp->t_pgrp == 0)
			vtp->t_pgrp = pp->p_pid;
		pp->p_pgrp = vtp->t_pgrp;
	} else {
		if (pp->p_universe == UNIVERSE_SYSV)
			if ((pp->p_pid == pp->p_pgrp) && (vtp->t_pgrp == 0)) {
	   			if (u.u_ttyp == NULL)
					u.u_ttyp = vtp;
				vtp->t_pgrp = pp->p_pgrp;
			}
		else
			if (vtp->t_pgrp == 0 && u.u_ttyp == vtp)
				vtp->t_pgrp = pp->p_pgrp;
	}
	vtp->svt_state |= ISOPEN;
	map_state(vtp, TO43);
	splx(oldpri);
	return 0;
}


sxtclose(dev, flag)
{
	register Link_p lp;
	register chan;
	register struct tty *vtp;
	int sps;
	int i;
	int error;

#if SXTRACE == 1
	if (tracing)
		printf("!sxtclose: link %d, chan %d\n", LINK(dev), CHAN(dev));
#endif

	if ((lp = linkTable[LINK(dev)]) == (Link_p)1) {
		/* no work to be done - master slot was only reserved,
		 * not used.					     
		 */
		linkTable[LINK(dev)] = NULL;
		return 0;
	}

	chan = CHAN(dev);
	vtp = &lp->chans[chan].tty;

#ifndef u3b
	vtp->svt_state &= ~(BUSY|TIMEOUT);
	map_state(vtp, TO43);
#endif

	sxt_setuniverse(vtp, lp->line);
	if (u.u_procp->p_universe == UNIVERSE_BSD)
		error = (*linesw[vtp->t_line].l_close)(vtp);
	else
		error = (*sv_linesw[vtp->svt_line].l_close)(vtp);

	vtp->t_pgrp = 0;		/* To keep out /dev/tty */
	vtp->svt_state &= ~CARR_ON;
	map_state(vtp, TO43);

	chan = 1 << chan;
	lp->xopen &= ~chan;
	lp->open &= ~chan;

	sps = spltty();
	if (chan == 1)			/* e.g. channel 0 */
	{
		lp->controllingtty = 0;
		lp->line->svt_line = lp->old;
		lp->line = vtp;		/* release real tty */
		vtp->svt_proc = sxtnullproc;

		for (i = 1; i < lp->nchans; ++i)
		{
			vtp = &lp->chans[i].tty;

			vtp->t_pgrp = 0;
			if (vtp->svt_proc == sxtnullproc ||
					vtp->svt_proc == sxtvtproc)
				sv_ttyflush(vtp, (FREAD | FWRITE));
		}
	}

	if (lp->open == 0) 
	{
		/* no other opens on this set of virt ttys */

		linkTable[LINK(dev)] = NULL;
		sxtfree(lp);
	}
	splx(sps);
	return error;
}


sxtioctl(dev, cmd, arg, mode)
{
	register Link_p lp;
	register struct tty *tp, *vtp;
	register sxtld;
	struct termio cb;
	struct sgttyb tb;
	struct sxtblock sb;
	int flag;
	int n;
	int sps;
	int i;
	char c_line;
	extern sxtin();
	register int error = 0;

#if	SXTRACE == 1
	if (tracing)
		printf("!sxtioctl: cmd %x, link %d, chan %d\n", cmd,
				LINK(dev), CHAN(dev));
#endif

	lp = linkTable[LINK(dev)];

	vtp = &lp->chans[CHAN(dev)].tty;
	sxt_setuniverse(vtp, lp->line);

	if (u.u_procp->p_universe == UNIVERSE_BSD) {
		if ((error = (*linesw[vtp->t_line].l_ioctl)
						(vtp, cmd, arg, mode)) >= 0)
			return error;
		ttioctl(vtp, cmd, arg, mode);
		switch (cmd) {

		    case TIOCSETP:		/* device specific commands */
		    case TIOCSETN:
		    case TIOCLSET:
		    case TIOCLBIS:
		    case TIOCLBIC:
		    case TIOCSBRK:
		    case TIOCCBRK:
		    case TIOCSDTR:
		    case TIOCCDTR:
			if (CHAN(dev) == lp->controllingtty)
				error = (*cdevsw[major(lp->dev)].d_ioctl)
						(lp->dev, cmd, arg, mode);
			break;

		    default:
			error = 0;
			break;
		}
		return error;
	}

	if (lp == (Link_p)1 && cmd != SXTIOCLINK) {
		/* only a link command would be valid here */
		return ENXIO;
	}

	switch (cmd) {

	case SXTIOCLINK:
		if (arg > MAXPCHAN || arg < 1) {
			error = EINVAL;
			break;
		}
		if ((tp = cdevsw[major(u.u_ttyd)].d_ttys) == NULL) {
			error = ENOTTY;
			break;
		}

		tp += minor(u.u_ttyd);	/* Real 'tty' line */

		/* find  sxt line discipline entry number in linesw */
		for (sxtld = 0; sxtld < sv_linecnt; sxtld++)
			if (sv_linesw[sxtld].l_input == sxtin)
				break;
		if (sxtld == sv_linecnt) {
			error = ENXIO;	/* SXT not in linesw */
			break;
		}
		if (lp == (Link_p)0) {
			error = EBADF; 	/* file not opened */
			break;
		}
		if (lp != (Link_p)1) {
			error = EBUSY;	/* Pre-empted! */
			break;
		}

		if ((lp = sxtalloc(arg)) == NULL) {
			error = ENOMEM;	/* No memory, try again */
			break;
		}

		sv_ttyflush(tp, FREAD|FWRITE);
		lp->dev = u.u_ttyd;	/* save major/minor dev #s	*/
		lp->controllingtty = 0;	/* channel 0			*/
		lp->lwchan = 0;		/* last channel to write	*/
		lp->wpending = 0;	/* write pending bits/chan	*/
		lp->wrcnt = 0;		/* number of writes on last channel written */
		lp->line = tp;
		lp->old = tp->svt_line;	/* Remember old line discipline */
		lp->nchans = arg;

		lp->chanmask = 0xFF;
		for (i = lp->nchans; i < MAXPCHAN; ++i)
			lp->chanmask >>= 1;

		lp->open = lp->xopen = 1;	/* Open channel 0	*/
		sxtlinit(&lp->chans[0], 0, dev, lp->old, tp);

		sps = spltty();
		linkTable[LINK(dev)] = lp;	/* Now visible		*/
		tp->svt_line = sxtld;		/* Stack new one	*/
		tp->svt_link = LINK(dev);	/* Back ptr to link struct */
		vtp = &lp->chans[0].tty;

		/* do the pertinent part of ttopen() here		*/ 
		if ((u.u_procp->p_pid == u.u_procp->p_pgrp)
		   			&& (vtp->t_pgrp == 0)) {
		   	if (u.u_ttyp == NULL)
				u.u_ttyp = vtp;
			vtp->t_pgrp = u.u_procp->p_pgrp;
		}
		vtp->svt_state |= ISOPEN;
		map_state(vtp, TO43);
		splx(sps);
		break;

	case SXTIOCSWTCH:
		/* ***  make new vtty top dog -
		 *	download new vtty characteristics and wake up
		 */
		if (lp == (Link_p) 1 || lp == (Link_p) 0) {
			error = EINVAL;
			break;
		}
		if (!(1<<arg & lp->open)) {
			error = EINVAL;
			break;
		}
		sps = spltty();
		if (CHAN(dev) != 0) {
			/* only controller can switch windows */
			error = EPERM;
			splx(sps);
			break;
		}

#if	SXTRACE == 1
		if (tracing)
			printf("!sxtioctl: switch arg=%d, control=%d\n",
				arg,lp->controllingtty);
#endif
		if (lp->controllingtty != arg) {
				
			lp->controllingtty = arg;
			if (arg != 0) {
				/*  download valid portions of tty struct*/
				ttywait(&lp->chans[0].tty) ;
				tp = lp->line;
				vtp = &lp->chans[arg].tty;
				/*  change flags */
				tp->svt_iflag = vtp->svt_iflag;
				tp->svt_oflag = vtp->svt_oflag;
				tp->svt_cflag = vtp->svt_cflag;
				tp->svt_lflag = vtp->svt_lflag;
				bcopy(vtp->svt_cc, tp->svt_cc, NCC);
				map_flags(tp, TO43);
	
				/*  do download to have new values take effect */
				(*tp->svt_proc)(tp, T_PARM);
			}
		}
		splx(sps);
		wakeup((caddr_t) &lp->chans[arg]);
		break;

	case SXTIOCWF:
		/* wait til chan arg is in foreground and
		 * then return
		 */
		if (lp == (Link_p) 1 || lp == (Link_p) 0) {
			error = EINVAL;
			break;
		}
		if (!(1<<arg & lp->open)) {
			error = EINVAL;
			break;
		}
		if (lp->controllingtty == arg)
			/* nothing to be done */
			break;

		sps = spltty();
		while (lp->controllingtty != arg) {
			sleep((caddr_t) &lp->chans[arg], TTOPRI);
		}
		splx(sps);

		break;

	case SXTIOCBLK:
		/*
		 *  set SV_LOBLK in indicated window
		 */
		
		if (lp == (Link_p) 1 || lp == (Link_p) 0) {
			error = EINVAL;
			break;
		}
		if (!(1<<arg & lp->open)) {
			error = EINVAL;
			break;
		}
		vtp = &lp->chans[arg].tty;
		vtp->svt_cflag |= SV_LOBLK;
		break;

	case SXTIOCUBLK:
		/*
		 *  unset SV_LOBLK in indicated window
		 */
		
		if (lp == (Link_p) 1 || lp == (Link_p) 0) {
			error = EINVAL;
			break;
		}
		if (!(1<<arg & lp->open) || (arg == 0)) {
			error = EINVAL;
			break;
		}
		vtp = &lp->chans[arg].tty;
		vtp->svt_cflag &= ~SV_LOBLK;
		wakeup((caddr_t) &lp->chans[arg]);
		break;

	case SXTIOCSTAT:
		/*
		 *  return bit map of blocked channels to user
		 */

		if (lp == (Link_p) 1 || lp == (Link_p) 0) {
			error = EINVAL;
			break;
		}
		
		sb.input = lp->iblocked;
		sb.output = lp->oblocked;
		if (copyout(&sb, arg, sizeof sb)) {
			error = EFAULT;
			break;
		}
		break;

#if SXTRACE == 1
	case SXTIOCTRACE:
		tracing = 1;
		break;

	case SXTIOCNOTRACE:
		tracing = 0;
		break;
#endif

	case TIOCEXCL:
		lp->xopen |= (1<<CHAN(dev));
		break;

	case TIOCNXCL:
		lp->xopen &= ~(1<<CHAN(dev));
		break;

	case TCGETA:
	case TIOCGETP:
		sv_ttiocom(&lp->chans[CHAN(dev)].tty, cmd, arg);
		break;

	case TIOCSETP:
		sv_ttiocom(&lp->chans[CHAN(dev)].tty, cmd, arg);
		if (CHAN(dev) == lp->controllingtty) {
			/* TIOCSETP real tty without flush or ttywait */
			tp = lp->line;
			/* next section lifted from tty.c */

			if (copyin(arg, &tb, sizeof(tb))) {
				error = EFAULT;
				break;
			}
			tp->svt_iflag = 0;
			tp->svt_oflag = 0;
			tp->svt_lflag = 0;
			tp->svt_cflag = (tb.sg_ispeed&SV_CBAUD)|SV_CREAD;
			if ((tb.sg_ispeed&SV_CBAUD)==B110)
				tp->svt_cflag |= SV_CSTOPB;
			tp->svt_cc[VERASE] = tb.sg_erase;
			tp->svt_cc[VKILL] = tb.sg_kill;
			flag = tb.sg_flags;
			if (flag&O_HUPCL)
				tp->svt_cflag |= SV_HUPCL;
			if (flag&O_XTABS)
				tp->svt_oflag |= SV_TAB3;
			else if (flag&O_TBDELAY)
				tp->svt_oflag |= SV_TAB1;
			if (flag&O_LCASE) {
				tp->svt_iflag |= SV_IUCLC;
				tp->svt_oflag |= SV_OLCUC;
				tp->svt_lflag |= SV_XCASE;
			}
			if (flag&O_ECHO)
				tp->svt_lflag |= SV_ECHO;
			if (!(flag&O_NOAL))
				tp->svt_lflag |= SV_ECHOK;
			if (flag&O_CRMOD) {
				tp->svt_iflag |= SV_ICRNL;
				tp->svt_oflag |= SV_ONLCR;
				if (flag&O_CR1)
					tp->svt_oflag |= SV_CR1;
				if (flag&O_CR2)
					tp->svt_oflag |= SV_ONOCR|SV_CR2;
			} else {
				tp->svt_oflag |= SV_ONLRET;
				if (flag&O_NL1)
					tp->svt_oflag |= SV_CR1;
				if (flag&O_NL2)
					tp->svt_oflag |= SV_CR2;
			}
			if (flag&O_RAW) {
				tp->svt_cc[VTIME] = 1;
				tp->svt_cc[VMIN] = 6;
				tp->svt_iflag &= ~(SV_ICRNL|SV_IUCLC);
				tp->svt_cflag |= SV_CS8;
			} else {
				tp->svt_cc[VEOF] = SV_CEOF;
				tp->svt_cc[VEOL] = 0;
				tp->svt_iflag |=
#ifdef	OLD
				SV_BRKINT|SV_IGNPAR|SV_ISTRIP|SV_IXON|SV_IXANY;
#else	OLD
				SV_BRKINT|SV_ISTRIP|SV_IXON|SV_IXANY;
#endif	OLD
				tp->svt_oflag |= SV_OPOST;
				tp->svt_cflag |= SV_CS7|SV_PARENB;
				tp->svt_lflag |= SV_ICANON|SV_ISIG;
			}
			tp->svt_iflag |= SV_INPCK;
			if (flag&O_ODDP)
				if (flag&O_EVENP)
					tp->svt_iflag &= ~SV_INPCK;
				else
					tp->svt_cflag |= SV_PARODD;
			if (flag&O_VTDELAY)
				tp->svt_oflag |= SV_FFDLY;
			if (flag&O_BSDELAY)
				tp->svt_oflag |= SV_BSDLY;
	
			map_flags(tp, TO43);
	
			/* download tty change */
			(*tp->svt_proc)(tp, T_PARM);
		}
		break;

	case TCSETA:
	case TCSETAW:
	case TCSETAF:
		sv_ttiocom(&lp->chans[CHAN(dev)].tty, cmd, arg);
		if (CHAN(dev) == lp->controllingtty) {
			/* must perform action on real tty */
			/* insure that line disps remain correct */
			if (copyin(arg, &cb, sizeof cb)) {
				error = EFAULT;
				break;
			}
			c_line = cb.c_line;
			cb.c_line = lp->line->svt_line;
			if (copyout(&cb, arg, sizeof cb)) {
				error = EFAULT;
				break;
			}
			(*cdevsw[major(lp->dev)].d_ioctl)
				(lp->dev, cmd, arg, mode);
			/*  now restore user buffer */
			cb.c_line = c_line;
			if (copyout(&cb, arg, sizeof cb)) {
				error = EFAULT;
				break;
			}
		}
		break;

	case TCSBRK:
		sv_ttiocom(&lp->chans[CHAN(dev)].tty, cmd, arg);
		if (CHAN(dev) == lp->controllingtty && arg == 0)
			(*lp->line->svt_proc)(lp->line, T_BREAK);
		break;

	case TCXONC:
	case TCFLSH:
		sv_ttiocom(&lp->chans[CHAN(dev)].tty, cmd, arg);
		break;


	default:
		sv_ttiocom(&lp->chans[CHAN(dev)].tty, cmd, arg);
		if (CHAN(dev) == lp->controllingtty)
			/* must perform action on real tty */
			(*cdevsw[major(lp->dev)].d_ioctl)
				(lp->dev, cmd, arg, mode);
		break;
	}
	return error;
}

int nodev();

sxtlinit(chp, channo, dev, ld, tp)
register Ch_p chp;
register struct tty * tp;	/* real tty structure */
{

#if	SXTRACE == 1
		if (tracing)
			printf("!sxtlinit: channo %d\n", channo);
#endif
	sv_ttinit(&chp->tty);

	chp->tty.t_oproc = nodev;
	ttychars(&chp->tty);
	chp->tty.t_ispeed = chp->tty.t_ospeed = ISPEED;
	chp->tty.t_flags = IFLAGS;
	chp->tty.t_line = 0;			/* default line disp */

	chp->tty.svt_line = ld;			/* old line discipline */
	chp->tty.svt_proc = sxtvtproc;
	chp->tty.svt_link = LINK(dev);
	chp->tty.svt_state |=  (tp->svt_state
				& ~(OASLP|IASLP|BUSY|ISOPEN));
	map_state(&chp->tty, TO43);
	u.u_ttyd = dev;				/* for /dev/tty */
}


sxtread(dev, uio)
dev_t		dev;
struct uio	*uio;
{
	register Link_p lp;
	register struct tty *vtp;
	register int channo;
	int sps;
	register int error = 0;


#if	SXTRACE == 1
	if (tracing)
		printf("!sxtread: link %d, chan %d\n", LINK(dev),
				CHAN(dev));
#endif

	channo = CHAN(dev);
	if ((lp = linkTable[LINK(dev)]) == (Link_p)1)
		error = ENXIO;
	else if (lp == (Link_p)0)
		error = EBADF;	/* link not opened */
	else if (!(lp->open & (1<<channo)))
		error = EBADF;
	else {
		vtp = &lp->chans[channo].tty;

		sps = spltty();
		sxt_setuniverse(vtp, lp->line);
		while (lp->controllingtty != channo) {
			lp->iblocked |= (1 << channo);
			sleep((caddr_t) &lp->chans[channo], TTOPRI);
		}
		lp->iblocked &= ~(1 << channo);
		splx(sps);

		if (u.u_procp->p_universe == UNIVERSE_SYSV)
			return (*sv_linesw[vtp->svt_line].l_read)(vtp, uio);
		else
			return (*linesw[vtp->t_line].l_read)(vtp, uio);
	}
	return error;
}


sxtwrite(dev, uio)
dev_t		dev;
struct uio	*uio;
{
	register Link_p	lp;
	register struct tty *vtp;
	register int channo;
	int sps;
	register int error;
	
#if	SXTRACE == 1
	if (tracing)
		printf("!sxtwrite: link %d, chan %d\n", LINK(dev),
				CHAN(dev));
#endif

	channo = CHAN(dev);
	if ((lp = linkTable[LINK(dev)]) == (Link_p)1)
		error = ENXIO;
	else if (lp == (Link_p)0)
		error = EBADF;	/* link not opened */
	else if (!(lp->open & (1<<channo)))
		error = EBADF;
	else {
		channo = CHAN(dev);
		vtp = &lp->chans[channo].tty;

		sps = spltty();
		sxt_setuniverse(vtp, lp->line);
		while ((vtp->svt_cflag & SV_LOBLK) &&
		    (lp->controllingtty != channo)) {
			lp->oblocked |= (1 << channo);
			sleep((caddr_t) &lp->chans[channo], TTOPRI);
		}
		lp->oblocked &= ~(1 << channo);
		splx(sps);

		if (u.u_procp->p_universe == UNIVERSE_SYSV)
			return (*sv_linesw[vtp->svt_line].l_write)(vtp, uio);
		else {
			error =  (*linesw[vtp->t_line].l_write)(vtp, uio);
			(*vtp->svt_proc)(vtp, T_OUTPUT);
		}
	}
	return error;
}


#ifdef	unneeded
sxtrwrite(rtp)
register struct tty *rtp;
{
	register struct tty *vtp;
	register index;
	register Link_p	lp;

	/* write issued to a real tty device.  Look for the real tty's group
 	* of virtual ttys and do the write to the controllingtty of that 
 	* group.  Hash this???
 	*/
 	for (index = 0; index < MAXLINKS; index++) {
		if ((lp=linkTable[index]) != (Link_p)0 && lp != (Link_p)1) {
			if (lp->line == rtp)
				break;
		}
	}
	if (index == MAXLINKS)	/* no match */
		return;		/* drop output on floor */
	/* write to controlling tty 	    */
	vtp = &lp->chans[lp->controllingtty].tty;
	(*sv_linesw[vtp->svt_line].l_write)(vtp, uio);
}
#endif	unneeded

sxtvtproc(vtp, cmd)
register struct tty *vtp;
{
	register Link_p	lp;
	register cnt;

	/* 
	 *     called with a virtual tty.
	 */

#if	SXTRACE == 1
	if (tracing)
		printf("!sxtvtproc: cmd %d \n", cmd);
#endif

	lp = linkTable[vtp->svt_link];

	switch (cmd) {

	    default:
		return;

	    case T_TIME:
		vtp->svt_state &= ~TIMEOUT;
		map_state(vtp, TO43);
		(*lp->line->svt_proc)(lp->line, T_TIME);
		break;

	    case T_BREAK:
		vtp->svt_state |= TIMEOUT;
		map_state(vtp, TO43);
		(*lp->line->svt_proc)(lp->line, T_BREAK);

	    case T_WFLUSH:
		if (lp->controllingtty == (vtp - &lp->chans[0].tty))
			(*lp->line->svt_proc)(lp->line, T_WFLUSH);
		break;

	    case T_RESUME:
		vtp->svt_state &= ~TTSTOP;
		map_state(vtp, TO43);
		cnt = vtp - &lp->chans[0].tty;
		lp->wpending |= 1<< cnt;
		(*lp->line->svt_proc)(lp->line, T_RESUME);
		break;

	    case T_OUTPUT:
		cnt = vtp - &lp->chans[0].tty;
		lp->wpending |= 1<< cnt;
		(*lp->line->svt_proc)(lp->line, T_OUTPUT);  /* real proc */
		break;

	    case T_SUSPEND:
		vtp->svt_state |= TTSTOP;
		map_state(vtp, TO43);
		(*lp->line->svt_proc)(lp->line, T_SUSPEND);
		break;

	    case T_RFLUSH:
		if (lp->controllingtty == (vtp - &lp->chans[0].tty))
			(*lp->line->svt_proc)(lp->line, T_RFLUSH);
		break;

#if vax || u3b5 || u3b2 || m68k || is68k
	    case T_SWTCH:
		/* change control to channel 0 */
		lp->controllingtty = 0;
		wakeup ((caddr_t) &lp->chans[0]);
		break;
#endif
	    case T_PARM:
		(*lp->line->svt_proc)(lp->line, T_SUSPEND);
		break;
	}
}


sxtnullproc(vtp, cmd)
register struct tty *vtp;
{
	register Link_p lp;
	unsigned char	tmp;
	int cnt = 0;
	/* 
	 *     called with a virtual tty.
	 */

#if	SXTRACE == 1
	if (tracing)
		printf("!sxtnullproc: cmd %d \n", cmd);
#endif

	if (cmd == T_OUTPUT)
	{
		lp = linkTable[vtp->svt_link];

		tmp = lp->wpending;
		while (tmp >>= 1)
			cnt++;

		sv_ttyflush(&lp->chans[cnt].tty, FWRITE);
		lp->wpending = 0;
	}
}


/*
 * real tty output routine
 * multiplexing done here!
 */
sxtout(tp)
struct tty *tp;
{
	register Link_p lp;
	register cnt;
	register struct tty *vtp;
	unsigned char tmp;
	int sps;
	int c;

/*
 * the next two lines ensures that
 * this guy copies from the virtual tty clist to the real tty clist only
 * when there is no output pending on the real tty.  This makes the
 * 'catq' call a simple pointer copy, rather than a laborious clist
 * copy that hogs the system badly
 */
	if (tp->t_outq.c_cc != 0)
		return 1;

	lp = linkTable[tp->svt_link];

#if	SXTRACE == 1
	if (tracing)
		printf("!sxtout:  link %d, chan %d\n", tp->svt_link,
				lp->lwchan);
#endif

	sps = spltty();
	if (lp->wpending & lp->lwchan) {

		cnt = 0;
		tmp = lp->lwchan;
		while (tmp >>= 1)
			cnt++;

		vtp = &lp->chans[cnt].tty;

		catq(&vtp->t_outq, &tp->t_outq);
		lp->wpending &= ~(lp->lwchan);
		goto out;
	}

	/* try to schedule next channel in round-robin list of wpending 
	   and dont give up until ALL possibilities are exhausted */

	for (cnt = 0; cnt < lp->nchans; ++cnt) {
			/* find next channel which had write pending */
			/* when done, lwchan specifies a candidate for reschedule */
		lp->lwchan = (lp->lwchan << 1) & lp->chanmask;
		if (lp->lwchan == 0)
			lp->lwchan = 1;
		if (lp->wpending & lp->lwchan)
			break;
	}

	if (cnt < lp->nchans) {		/* channel in bounds */
		cnt = 0;
		tmp = lp->lwchan;
		while (tmp >>= 1)
			cnt++;

		vtp = &lp->chans[cnt].tty;

		catq(&vtp->t_outq, &tp->t_outq);

		lp->wpending &= ~(lp->lwchan);
	}
out:
	splx(sps);
	for (cnt = 0; cnt < lp->nchans; ++cnt) {
		vtp = &lp->chans[cnt].tty;
		if (vtp->svt_state&OASLP && vtp->t_outq.c_cc <= 
					ttlowat[vtp->svt_cflag&SV_CBAUD]) {
			    			/* wake up any sleepers */
			vtp->svt_state &= ~OASLP; 
			map_state(vtp, TO43);
			lp->wpending |= 1<< cnt;
			wakeup((caddr_t)&vtp->t_outq);
		}
	}
	return 1;		/* always return 1 to flush output */
}




/*
 * real tty input routine
 * returns data to controlling tty
 */
sxtin(c, tp, code)
register int c;
register struct tty *tp;
{
	register struct tty *vtty;
	register Link_p	lp;
	register n;

#if	SXTRACE == 1
	if (tracing)
		printf("!sxtin:  link %d, code %d\n", tp->svt_link, code);
#endif

	lp = linkTable[tp->svt_link];
	n = lp->controllingtty;
	vtty = &lp->chans[n].tty;

	switch (code) {

#ifdef u3b
	case L_SWITCH:

		/* change control to channel 0 */
		/* first flush input queue     */
		if (!vtty->svt_lflag & NOFLSH)
			sv_ttyflush(vtty, FREAD);
		if (n != 0)
			lp->controllingtty = 0;
		wakeup ((caddr_t) &lp->chans[0]);
		break;

	case L_INTR:
	case L_QUIT:
#endif
	case L_BREAK:

		(*sv_linesw[vtty->svt_line].l_input)(c, vtty, code);
		break;
		
	case L_BUF:

		if (vtty->t_universe == UNIVERSE_BSD)
			(*linesw[vtty->t_line].l_rint)(c, vtty);
		else
			(*sv_linesw[vtty->svt_line].l_input)(c, vtty, code);
		break;

	default:
		panic("sxtin: invalid code");
	}
}


sxtfree(lp)
Link_p lp;
{
	int i;

#if	SXTRACE == 1
	if (tracing)
		printf("!sxtfree\n");
#endif


	i = ((char *)lp - sxtbuf)/LINKSIZE;
	sxtbusy[i] = 0;
}


Link_p
sxtalloc(arg)
{
	register i;
	Link_p lp;

#if	SXTRACE == 1
	if (tracing)
		printf("!sxtalloc\n");
#endif

#if vax || u3b2 || is68k
	if (sxtbuf == NULL)
		sxtinit();
#endif

	if (sxtbuf != NULL) {
		for (i=0; i < sxt_cnt; i++)
			if (!sxtbusy[i]) {
				lp = (Link_p) (sxtbuf + (i*LINKSIZE));
#ifdef u3b
				kclear((char *)lp, LINKHEAD);
#else
				bzero((char *)lp, LINKHEAD);
#endif
				sxtbusy[i] = 1;
				return(lp);
			}
	}
	return(NULL);
}

sxt_setuniverse(vtp, tp)
register struct tty *vtp;		/* virtual tty */
register struct tty *tp;		/* real tty */
{
	vtp->t_universe = u.u_procp->p_universe;
	if (tp == NULL)
		return;
	tp->t_universe = UNIVERSE_SYSV;
}

/***************************** 3b5 ******************************/
#if u3b5

sxtinit()
{
	extern char sxt_buf[];

	if ((sxtbuf = sxt_buf) == NULL)
		printf("sxt cannot allocate link buffers\n");
}
#endif
/**************************** 3b20 & 3b2 *******************************/
#if u3b || u3b2

sxtinit()
{

	if ((sxtbuf = (char *)kseg(btoc(sxt_cnt*LINKSIZE))) == NULL)
		printf("sxt cannot allocate link buffers\n");
}

#endif

/**************************** VAX only *******************************/
#ifdef vax

sxtinit()
{

	if ((sxtbuf = (char *)sptalloc(btoc(sxt_cnt*LINKSIZE),
				PG_V|PG_KW, 0)) == NULL)
		printf("sxt cannot allocate link buffers\n");
}
#endif

/**************************** is68k only *******************************/
#ifdef is68k

sxtinit()
{
/* 
 * use static alloc for now, should really use kernel malloc capability
 */
	sxtbuf = sxt_buf;
}
#endif

#endif
#endif	SYSV
