#ifdef	SYSV

#include "types.h"
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "map.h"
#include "user.h"
#include "proc.h"
#include "text.h"
#include "buf.h"
#include "seg.h"
#include "vm.h"
#include "cmap.h"
#include "file.h"
#include "uio.h"
#include "exec.h"
#include "vfs.h"
#include "vnode.h"
#include "../sysv/sys/region.h"

/*
 * Allocate pregion, regions, and swap space for child's private regions
 * Called by fork and vfork.
 */
preg_t *
preg_fork(p, q, isvfork)
struct proc *p, *q;
int isvfork;
{
	register preg_t *tprp, *tprq;
	register preg_t *prp = p->p_region, *prq;
	register reg_t *rp, *rq;

	if (prp == NULL) 
		prq = NULL;
	else if ((tprq = prq = preg_alloc()) != NULL) {
		tprq->p_procp = q;
		tprp = prp;
		while (tprp->p_type != PT_UNUSED) {
			if (tprp->p_type == PT_LIBDAT) {
				rp = tprp->p_pp;
				if ((rq = reg_alloc()) == NULL) {
					preg_freeswap(prq);
					preg_freeregs(prq);
					preg_free(prq);
					prq = NULL;
					break;
				} else {
					if (!isvfork) {
						if (!vsexpand(rp->r_pgsz,
						   	     &rq->r_dmap, 0)) {
							preg_freeswap(prq);
							preg_freeregs(prq);
							preg_free(prq);
							prq = NULL;
							break;
						}
						rq->r_pgsz = rp->r_pgsz;
					}
				}
				tprq->p_pp = rq;
			}
			tprq->p_type = tprp->p_type;
			tprq->p_regva = tprp->p_regva;
			tprq->p_flags = tprp->p_flags;
			tprq->p_procp = q;
			tprp++;
			tprq++;
		}
	}
	return(prq);
}

/*
 * Initialize the shared region pointers. Called by fork and vfork.
 */
preg_initshreg(prp, prq, isvfork)
register preg_t *prp, *prq;
int isvfork;
{
	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if ((prp->p_type==PT_LIBTXT)||(prp->p_type==PT_SHMEM))
				prq->p_ps = isvfork ? NULL : prp->p_ps;
			prp++;
			prq++;
		}
	}
}

/* 
 * Exchange shared region pointers. Used by vfork
 */
preg_exchgsh(p, q)
register struct proc *p, *q;
{
	register preg_t *prp = p->p_region;
	register preg_t *prq = q->p_region;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if ((prp->p_type==PT_LIBTXT)||(prp->p_type==PT_SHMEM)){
				prq->p_ps = prp->p_ps;
				preg_xrepl(prp->p_ps, prq->p_ps);
				prp->p_ps = 0;
			}
			prp++;
			prq++;
		}	
	}	
}

/* 
 * Exchange swap map for the private regions. Used by vfork
 */
preg_exchgswp(p, q)
register struct proc *p, *q;
{
	register preg_t *prp = p->p_region;
	register preg_t *prq = q->p_region;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type == PT_LIBDAT) {
				prq->p_pp->r_dmap = prp->p_pp->r_dmap;
				prp->p_pp->r_dmap = zdmap;
			}
			prp++;
			prq++;
		}
	}
}

/*
 * Duplicate the pregion by attaching shared regions and replicate private 
 * regions. Used by fork.
 */
preg_dup(p, q)
register struct proc *p, *q;
{
	register preg_t *prp = p->p_region;
	register preg_t *prq = q->p_region;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type == PT_LIBDAT) {
				vmdup(q, prq->p_pte, btop(prq->p_regva),
					prq->p_pp->r_pgsz, CLIBDAT);
			} else {
				prq->p_ps->x_count++;
				preg_xlink(prq, (struct sysv_exdata *)0);
			}
			prp++;
			prq++;
		}
	}
}

/*
 * Release memory for the regions. Called by vrelvm on behave of exit or execl.
 */
preg_freemem(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type == PT_LIBDAT) {
				p->p_rssize -= vmemfree(prp->p_pte,
							prp->p_pp->r_pgsz);
			} else {
				preg_xfree(prp);
			}
			prp++;
		}
	}
}

/*
 * Free swap space for the private regions. Called by vrelvm on behave of 
 * exit or execl.
 */
preg_freeswap(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type == PT_LIBDAT)
				vsexpand((size_t)0, &prp->p_pp->r_dmap, 1);
			prp++;
		}
	}
}

/*
 * Set page tables for the process regions. Used in swapin, vfork, and fork.
 */
preg_vgetpt(p)
register struct proc *p;
{
	register size, i, a;
	register preg_t *prp, *prq;

	if (prp = p->p_region) {
		while (prp->p_type != PT_UNUSED) {
			prp->p_pte = &p->p_p0br[(int)btoc(prp->p_regva)];
			prp++;
		}
	}
}

/*
 * Pass the process region page tables of process p to process q.
 * Used during vfork().  P and q are not symmetric;
 * p is the giver and q the receiver.
 */
preg_vpasspt(p, q)
	register struct proc *p, *q;
{
	register preg_t *prp, *prq;
	register struct pte *pte;

	if (prp = p->p_region) {
		prq = q->p_region;
		while (prp->p_type != PT_UNUSED) {
			pte = prp->p_pte;
			prp->p_pte = prq->p_pte;
			prq->p_pte = pte;
			prp++;
			prq++;
		}
	}
}

/*
 * Initialize text portion of page table. Called by preg_xlink.
 */
preg_vinitpt(prp, ep)
register preg_t *prp;
struct sysv_exdata *ep;
{
	struct proc *p = prp->p_procp;
	register struct text *xp;
	register preg_t *prq;
	register struct pte *pte;
	register int i;
	struct pte proto;

	xp = prp->p_ps;
	if (xp == 0)
		return;
	pte = prp->p_pte;
	/*
	 * If there is another instance of same text in core
	 * then just copy page tables from other process.
	 */
	if (prq = xp->x_preg) {
		bcopy((caddr_t)prq->p_pte, (caddr_t)pte,
		    (unsigned) (sizeof(struct pte) * xp->x_size));
		goto checkprot;
	}
	/*
	 * Initialize text page tables, zfod if we are loading
	 * the text now; unless the process is demand loaded,
	 * this will suffice as the text will henceforth either be
	 * read from a file or demand paged in.
	 */
	if (prp->p_flags & PF_RDONLY)
		*(int *)&proto = PG_URKR;
	else
		*(int *)&proto = PG_UW;

	if (xp->x_flag & XLOAD) {
		proto.pg_fod = 1;
		((struct fpte *)&proto)->pg_fileno = PG_FZERO;
	}

	if ((xp->x_flag & XPAGV) == 0) {
		for (i = 0; i < xp->x_size; i++)
			*pte++ = proto;
		goto done;
	}

	/*
	 * Text is demand loaded.  If process is not loaded (i.e. being
	 * swapped in) then retrieve page tables from swap area.  Otherwise
	 * this is the first time and we must initialize the page tables
	 * from the blocks in the file system.
	 */
	if (xp->x_flag & XLOAD) {
		if (prp->p_type == PT_LIBTXT)
			vinifod((struct fpte *)pte, PG_FTEXT, xp->x_vptr,
			    (long)(clrnd(btop(ep->a_toffset))), xp->x_size);
	} else
		swap(prp->p_procp, xp->x_ptdaddr, (caddr_t)pte,
		    xp->x_size * sizeof (struct pte), B_READ,
		    B_PAGET, swapdev_vp, 0);
checkprot:
	if (prp->p_flags & PF_RDONLY) {
		if ((*(int *)pte & PG_PROT) != PG_URKR) {
			for (i = 0; i < xp->x_size; i++, pte++)
				*(int *)pte = (*(int *)pte & ~PG_PROT)|PG_URKR;
		}
	} else {
		if ((*(int *)pte & PG_PROT) != PG_UW) {
			for (i = 0; i < xp->x_size; i++, pte++)
				*(int *)pte = (*(int *)pte & ~PG_PROT) | PG_UW;
		}
	}
done:
	/*
	 * In the case where we are overlaying ourself with new page
	 * table entries, old user-space translations should be flushed.
	 */
	if (prp->p_procp == u.u_procp)
		newptes(prp->p_pte, btop(prp->p_regva), xp->x_size);
	else
		prp->p_procp->p_flag |= SPTECHG;
}

/*
 * Update the page tables of all processes linked
 * to a shared region, by distributing
 * dpte to the page.
 *
 * Note that invalidation in the translation buffer for
 * the current process is the responsibility of the caller.
 */
preg_distpte(prp, page, dpte)
	register preg_t *prp;
	register unsigned page;
	register struct pte *dpte;
{
	register struct pte *pte;
	register preg_t *prq;
	register int i;

	for (prq = prp->p_ps->x_preg; prq; prq = prq->p_xlink) {
		pte = &prq->p_pte[page];
		prq->p_procp->p_flag |= SPTECHG;
		if (pte != dpte) {
			for_CLSIZE(i)
				pte[i] = dpte[i];
		}
	}
}

/*
 * Swap a segment of process region virtual memory to disk,
 * by locating the contiguous dirty pte's
 * and calling vschunk with each chunk.
 * We ignore swap errors here because swap()
 * will panic on an error when writing to disk.
 */
preg_vsswap(prp)
	register preg_t *prp;
{
	register int size = 0;
	register struct cmap *c;
	register struct pte *pte;
	int type;
	register int vsbase, vscount;
	struct dmap *dmp;

	pte = prp->p_pte;
	vsbase = 0;
	if (prp->p_type == PT_LIBDAT)
		vscount = prp->p_pp->r_pgsz;
	else
		vscount = prp->p_ps->x_size;

	if (vscount % CLSIZE)
		panic("vsswap");
	for (;;) {
		if (vscount == 0 || !dirtycl(pte)) {
			if (size) {
				preg_vschunk(prp, vsbase, size);
				vsbase += size;
				size = 0;
			}
			if (vscount == 0)
				return;
			vsbase += CLSIZE;
			if (pte->pg_fod == 0 && pte->pg_pfnum)
				if (prp->p_type == PT_LIBDAT)
					prp->p_procp->p_rssize -=
						vmemfree(pte, CLSIZE);
				else
					prp->p_ps->x_rssize -= 
						vmemfree(pte, CLSIZE);
		} else {
			size += CLSIZE;
			c = &cmap[pgtocm(pte->pg_pfnum)];
			MLOCK(c);
			MUNLOCK(c);
		}
		vscount -= CLSIZE;
		pte += CLSIZE;
	}
}

preg_vschunk(prp, base, size)
	register preg_t *prp;
	register int base, size;
{
	register struct pte *pte;
	struct dblock db;
	unsigned v;

	base = ctod(base);
	size = ctod(size);
	while (size > 0) {
		pte = &prp->p_pte[dtoc(base)];
		v = (unsigned)prp->p_regva + (unsigned)ptob(dtoc(base));
		if (prp->p_type == PT_LIBDAT) {
			vstodb(base, size, &prp->p_pp->r_dmap, &db, 0);
			(void)swap(prp->p_procp, db.db_base,
			    v, (int)dtob(db.db_size),
			    B_WRITE, 0, swapdev_vp, 0);
			prp->p_procp->p_rssize -=
				 vmemfree(pte, (int)dtoc(db.db_size));
		} else {
			db.db_size = dmtext - base % dmtext;
			if (db.db_size > size)
				db.db_size = size;
			(void)swap(prp->p_procp,
			    prp->p_ps->x_daddr[base/dmtext] + base%dmtext,
			    v, (int)dtob(db.db_size),
			    B_WRITE, 0, swapdev_vp, 0);
			prp->p_ps->x_rssize -=
			    vmemfree(pte, (int)dtoc(db.db_size));
		}
		base += db.db_size;
		size -= db.db_size;
	}
}

/*
 * Convert a virtual page number in a process region.
 * to its corresponding disk block number.
 * Used in pagein/pageout to initiate single page transfers.
 */
swblk_t
preg_vtod(p, v, db)
register struct proc *p;
unsigned v;
struct dblock *db;
{
	register u_int page;
	register preg_t *prp;

	prp = preg_vtopreg(p, v);
	page = v - btop(prp->p_regva);
	if (prp->p_type == PT_LIBDAT)
		vstodb(ctod(page), ctod(1), &prp->p_pp->r_dmap, db, 0);
	else
		db->db_base = prp->p_ps->x_daddr[ctod(page)/dmtext] 
			+ (ctod(page) % dmtext);
}

/*
 * Add a region to those sharing a region by
 * getting the page tables and then linking to x_preg.
 */
preg_xlink(prp, ep)
	register preg_t *prp;
	struct sysv_exdata *ep;
{
	register struct text *xp = prp->p_ps;

	if (xp == 0)
		return;
	preg_vinitpt(prp, ep);
	prp->p_xlink = xp->x_preg;
	xp->x_preg = prp;
	xp->x_ccount++;
}

/*
 * Unlink a shared region from an incore linked list.
 */
preg_xunlink(prp)
register preg_t	*prp;
{
	register struct text *xp = prp->p_ps;
	register preg_t	*qrp;

	if (xp == 0)
		return;
	if (xp->x_preg == prp) {
		xp->x_preg = prp->p_xlink;
		prp->p_xlink = 0;
		return;
	}
	for (qrp = xp->x_preg; qrp->p_xlink; qrp = qrp->p_xlink)
		if (qrp->p_xlink == prp) {
			qrp->p_xlink = prp->p_xlink;
			prp->p_xlink = 0;
			return;
		}
	panic("lost shared region");
}

/*
 * Move shared memory from pregion prp to pregion prq.
 */
preg_xmove(prp, prq)
register preg_t	*prp, *prq;
{
	register struct text *xp = prp->p_ps;
	register preg_t	*qrp;

	if ((xp == 0) || (prp->p_type == PT_LIBDAT))
		return;
	if (xp->x_preg == prp) {
		xp->x_preg = prq;
		return;
	}
	for (qrp = xp->x_preg; qrp->p_xlink; qrp = qrp->p_xlink)
		if (qrp->p_xlink == prp) {
			qrp->p_xlink = prq;
			return;
		}
	panic("lost shared region");
}

/*
 * Replace p by q in a shared region incore linked list.
 * Used by vfork(), internally.
 */
preg_xrepl(prp, qrp)
	register preg_t *prp, *qrp;
{
	register struct text *xp = qrp->p_ps;

	if (xp == 0)
		return;
	preg_xunlink(prp);
	qrp->p_xlink = xp->x_preg;
	xp->x_preg = qrp;
}

/*
 * Find an existing text structure for a shared region.
 * If failed, allocate a new one.
 */
struct text *
preg_xalloc(vp)
	register struct vnode *vp;
{
	register struct text *xp;

	xstats.alloc++;
	if (vp != NULL) {
		
		/* Must be for shared library */
		while ((xp = vp->v_text) != NULL) {
			if (xp->x_flag&XLOCK) {
				/*
				 * Wait for text to be unlocked,
				 * then start over (may have changed state).
				 */
				xwait(xp);
				continue;
			}
			X_LOCK(xp);
			if (xp->x_back) {
				xstats.alloc_cachehit++;
				ALLOC(xp);
				xp->x_flag &= ~XUNUSED;
				xcache--;
			} else
				xstats.alloc_inuse++;
			XUNLOCK(xp);
			return(xp);
		}
	}
	xp = xhead;
	if (xp == NULL) {
		return(NULL);
	}
	ALLOC(xp);
	if (xp->x_vptr) {
		xstats.alloc_cacheflush++;
		if (xp->x_flag & XUNUSED)
			xstats.alloc_unused++;
		xuntext(xp);
		xcache--;
	}
	xp->x_count = 0;
	xp->x_ccount = 0;
	xp->x_rssize = 0;
	xp->x_flag = XREGION;
	return(xp);
}

/*
 * Free the resources associated with a shared memory region.
 */
preg_freeshmem(xp, noswap)
register struct text *xp;
int noswap;
{
	if (!noswap) {
		while (xp->x_poip)
			sleep((caddr_t)&xp->x_poip, PSWP+1);
		xp->x_flag &= ~XLOCK;
		vsxfree(xp, (long)xp->x_size);
	}
	FREE_AT_HEAD(xp);
}

/*
 * relinquish use of the shared region
 * of a process.
 */
preg_xfree(prp)
register preg_t *prp;
{
	register struct text *xp;
	register struct vnode *vp;
	struct vattr vattr;

	if ((xp = prp->p_ps) == NULL)
		return;
	xstats.free++;
	X_LOCK(xp);
	if (prp->p_type == PT_SHMEM) {
		if (--xp->x_count==0 && (xp->x_flag & XNOFREE)==0) {
			xp->x_rssize -= vmemfree(prp->p_pte, (int)xp->x_size);
			if (xp->x_rssize != 0)
				panic("xfree rssize");
			preg_freeshmem(xp, 0);
		} else
			preg_xccdec(prp);
	} else {
		vp = xp->x_vptr;

		if (--xp->x_count == 0 && (
#ifdef	TRFS
		   (xp->x_flag & XTRFS) ||
#endif	TRFS
		   (VOP_GETATTR(vp, &vattr, u.u_cred) != 0 || 
	     	   	(vattr.va_mode & VSVTX) == 0))) {

			if (xcache >= maxtextcache || 
			    (vp && (vp->v_flag & VTEXTMOD)) ||
#ifdef	TRFS
		    	    (xp->x_flag & XTRFS) ||
#endif	TRFS
#if defined(SYSV) && defined(RFS)
		    	    (vp && (vp->v_flag & VRFSNODE)) ||
#endif
		    	    vattr.va_nlink == 0) {		/* XXX */

				xp->x_rssize -= vmemfree(prp->p_pte, 
							(int)xp->x_size);
				if (xp->x_rssize != 0)
					panic("xfree rssize");
				while (xp->x_poip)
					sleep((caddr_t)&xp->x_poip, PSWP+1);
				xp->x_flag &= ~XLOCK;
				xuntext(xp);
				FREE_AT_HEAD(xp);
			} else {
				if (xp->x_flag & XWRIT) {
					xstats.free_cacheswap++;
					xp->x_flag |= XUNUSED;
				}
				xcache++;
				xstats.free_cache++;
				preg_xccdec(prp);
				FREE_AT_TAIL(xp);
			}
		} else {
			preg_xccdec(prp);
			xstats.free_inuse++;
		}
	}
	preg_xunlink(prp);
	XUNLOCK(xp);
}

/*
 * Decrement the in-core usage count of a shared region
 * which must be locked.  When the count drops to zero,
 * free the core space.
 */
preg_xccdec(prp)
register preg_t *prp;
{
	register struct text *xp = prp->p_ps;

	if (--xp->x_ccount == 0) {
		if ((prp->p_type == PT_SHMEM) && prp->p_ps->x_noswapcnt)
			return;
		if ((xp->x_flag & XWRIT) || (prp->p_type == PT_SHMEM)) {
			preg_vsswap(prp);
			if (xp->x_flag & XPAGV)
				swap(prp->p_procp, xp->x_ptdaddr,
				    (caddr_t)prp->p_pte,
				    xp->x_size * sizeof (struct pte),
				    B_WRITE, B_PAGET, swapdev_vp, 0);
			xp->x_flag &= ~XWRIT;
		} else
			xp->x_rssize -= vmemfree(prp->p_pte, (int)xp->x_size);
		if (xp->x_rssize != 0)
			panic("text rssize");
	}
}

preg_xattach(prp, vp, ep, pagi, bp, rpid)
register preg_t *prp;
struct vnode *vp;
register struct sysv_exdata *ep;
unsigned *bp;
unsigned long rpid;
{
	register struct text *xp = prp->p_ps;

	xp->x_count++;
	if (xp->x_count != 1) {
		X_LOCK(xp);
		preg_xlink(prp, (struct sysv_exdata *)0);
	} else if (prp->p_type == PT_SHMEM) {
		if (pagi) {
			xp->x_flag |= (XLOAD|XLOCK|XPAGV);
			if (vsxalloc(xp) == NULL) {
				xp->x_flag &= ~(XLOCK | XLOAD);
				return(0);
			}
			preg_xlink(prp, (struct sysv_exdata *)0);
			xp->x_flag |= XWRIT;
			xp->x_flag &= ~XLOAD;
		} else {
			X_LOCK(xp);
			preg_xlink(prp, (struct sysv_exdata *)0);
		}
	} else {
		/* Shared library text */
		xp->x_flag = XLOAD|XLOCK|XREGION;
#ifdef	TRFS
		if (bp)
			xp->x_flag |= XTRFS;
#endif	TRFS
		if (pagi)
			xp->x_flag |= XPAGV;
		xp->x_size = clrnd(btoc(ep->a_tsize));
		if (vsxalloc(xp) == NULL) {
			xp->x_flag &= ~(XLOCK | XLOAD);
			return(0);
		}
		xp->x_vptr = vp;
		if (vp) {
			vp->v_flag |= VTEXT;
			vp->v_text = xp;
			VN_HOLD(vp);
		}
		preg_xlink(prp, ep);
		if (pagi == 0) {
			preg_settprot(prp, RW);
			u.u_procp->p_flag |= SKEEP;
#ifdef is68k
#ifdef	TRFS
			if (bp)
				(void) ClPload(rpid, bp, ep->a_txtorg,
					(int)ep->a_tsize, ep->a_toffset,
					UIOSEG_USER);
			else
#endif	TRFS
			(void) vn_rdwr(UIO_READ, vp,
				ep->a_txtorg, (int)ep->a_tsize,
				ep->a_toffset,
				UIOSEG_USER, IO_UNIT, (int *)0);
#endif is68k
			u.u_procp->p_flag &= ~SKEEP;
		}
		preg_settprot(prp, RO);
		xp->x_flag |= XWRIT;
		xp->x_flag &= ~XLOAD;
	}
	XUNLOCK(xp);
	return(1);
}

/*
 * Detach a process from the in-core region.
 * External interface to preg_xccdec, used when swapping out a process.
 */
preg_xdetach(prp)
	register preg_t *prp;
{
	register struct text *xp = prp->p_ps;

	if (xp && xp->x_ccount != 0) {
		X_LOCK(xp);
		preg_xccdec(prp);
		preg_xunlink(prp);
		XUNLOCK(xp);
	}
}

/*
 * Check if any of the shared region has x_poip set.
 */
preg_poip(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type != PT_LIBDAT)
				if (prp->p_ps->x_poip)
					return(1);
			prp++;
		}
	}
	return(0);
}

/*
 * Check if any of the shared region is locked.
 */
preg_chklocks(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type != PT_LIBDAT)
				if (prp->p_ps->x_flag & XLOCK)
					return(1);
			prp++;
		}
	}
	return(0);
}

/*
 * Calculate the sum of swrss of all shared regions.
 */
preg_swrss(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;
	register int swrss = 0;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type != PT_LIBDAT)
				swrss += prp->p_ps->x_swrss;
			prp++;
		}
	}
	return(swrss);
}

/*
 * Calculate the sum of swap priority of all shared regions.
 */
preg_swpri(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;
	register int swprpi = 0;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type != PT_LIBDAT)
				swprpi += prp->p_ps->x_rssize
					  / prp->p_ps->x_ccount;
			prp++;
		}
	}
	return(swprpi);
}

/*
 * Lock all shared regions.
 */
preg_locksh(p)
register struct proc *p;
{
	register preg_t *prp, *oprp = p->p_region;

	if (oprp != NULL) {
loop:
		prp = oprp;
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type != PT_LIBDAT)
				if (prp->p_ps->x_flag & XLOCK) {
					xlock(prp->p_ps);
					xunlock(prp->p_ps);
					goto loop;
				}
			prp++;
		}
		prp = oprp;
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type != PT_LIBDAT)
				xlock(prp->p_ps);
			prp++;
		}
	}
}

/*
 * Unlock all shared regions.
 */
preg_unlocksh(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;

	if (prp != NULL)
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type != PT_LIBDAT)
				xunlock(prp->p_ps);
			prp++;
		}
}

/*
 * Swap out all regions.
 */
preg_swapout(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;

	if (prp != NULL)
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type == PT_LIBDAT) {
				preg_vsswap(prp);
				swap(p, prp->p_pp->r_ptdaddr, 
					prp->p_pte,
					prp->p_pp->r_pgsz * sizeof (struct pte),
					B_WRITE, B_PAGET, swapdev_vp, 0);
			} else {
				if (prp->p_ps->x_ccount == 1)
					prp->p_ps->x_swrss = prp->p_ps->x_rssize;
				preg_xdetach(prp);
			}
			prp++;
		}
}

/*
 * Swap in all regions.
 */
preg_swapin(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;

	if (prp != NULL)
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type == PT_LIBDAT) {
				swap(p, prp->p_pp->r_ptdaddr, 
					prp->p_pte,
					prp->p_pp->r_pgsz * sizeof (struct pte),
					B_READ, B_PAGET, swapdev_vp, 0);
			} else {
				preg_xlink(prp, (struct sysv_exdata *)0);
				xunlock(prp->p_ps);
			}
			prp++;
		}
}

/*
 * Read in and set up memory for executed file.
 */
preg_getxfile(ep)
	register struct sysv_exdata *ep;
{
	register struct vnode *vp = ep->vp;
	register struct proc *p = u.u_procp;
	register reg_t *rp = NULL;
	register struct text *xp = NULL;
	register preg_t *prp;
	register size_t ts, ds;
	register int pagi;

	/*
	 * Check to make sure nobody is modifying the text right now
	 */
	if ((vp->v_flag & VTEXTMOD) != 0) {
		u.u_error  = ETXTBSY;
		return(-1);
	}
	if ((ep->a_tsize != 0) && ((vp->v_flag & VTEXT) == 0) &&
	    (vp->v_count != 1)) {
		register struct file *fp;

		for (fp = file; fp < fileNFILE; fp++) {
			if (fp->f_type == DTYPE_VNODE &&
			    fp->f_count > 0 &&
			    (struct vnode *)fp->f_data == vp &&
			    (fp->f_flag&FWRITE)) {
				u.u_error = ETXTBSY;
				return(-1);
			}
		}
	}

	/*
	 * Determine whether to demand load the image
	 */
	if (vp->v_vfsp->vfs_bsize < CLBYTES)
		pagi = 0;
	else
		pagi = SPAGV;

	/*
 	 * Allocate region for data, and text for text.
	 */
	ts = clrnd(btoc(ep->a_tsize));
	ds = clrnd(btoc((ep->a_datorg & PGOFSET)+ep->a_dsize + ep->a_bsize));
	if ((rp = reg_alloc()) == NULL)
		goto bad;
	if (preg_attachreg(&u, rp, ep->a_datorg & ~SOFFMASK,
			PT_LIBDAT, 0, ds) == NULL)
		goto bad;
	rp->r_pgsz = ds;
	rp->r_vptr = vp;
	if (!vsexpand(rp->r_pgsz, &rp->r_dmap, 0)) {
		u.u_error = ENOMEM;
		goto bad;
	}
	xp = preg_xalloc(vp);
	if ((prp = preg_attachreg(&u, xp, ep->a_txtorg & ~SOFFMASK,
					PT_LIBTXT, 1, ts)) == NULL)
		goto bad;

	if (pagi == 0)
		u.u_error = vn_rdwr(UIO_READ, vp, ep->a_datorg, ep->a_dsize, 
			ep->a_doffset, UIOSEG_USER, IO_UNIT, (int *)0);

	preg_xattach(prp, vp, ep, pagi, (unsigned *)0, 0);
	if (u.u_error)
		goto bad;
	if (pagi && prp->p_ps) {
		vinifod((struct fpte *)preg_vtopte(p, btop(ep->a_datorg)),
		    PG_FTEXT, prp->p_ps->x_vptr,
		    (long)(clrnd(btop(ep->a_doffset))),
		    (int) (btoc((ep->a_datorg & PGOFSET) + ep->a_dsize)));

		/* Bss overlaps with data, zero the bss  */
		/* portion of the page , since the       */
		/* file may not be zero padded for bss.  */
		if (ep->a_bssorg & CLOFSET)
			u_bzero(ep->a_bssorg, 
				CLBYTES - (ep->a_bssorg & CLOFSET));
	}

	if (u.u_error) {
bad:
		swkill(p, "preg_getxfile: Can't exec shared library");
		return(-1);
	}
	return(0);
}

/*
 * Determine the size of p0 region which consists of
 * data, text, and shared regions.
 */
preg_p0size(p, vaddr)
register struct proc *p;
caddr_t vaddr;
{
	register preg_t *prp = p->p_region;
	caddr_t t;

	vaddr = (caddr_t)ctob((int)vaddr);
	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type == PT_LIBDAT)
				t = (caddr_t)(prp->p_regva +
						ctob(prp->p_pp->r_pgsz));
			else
				t = (caddr_t)(prp->p_regva +
						ctob(prp->p_ps->x_size));
			if (t > vaddr)
				vaddr = t;
			prp++;
		}
	}
	return(btoc((int)vaddr));
}

/*
 * Change protection of a region.
 */
preg_settprot(prp, tprot)
register preg_t *prp;
long tprot;
{
	register struct text *xp;
	register struct pte *pte;
	register int i;

	xp = prp->p_ps;
	if (xp == 0)
		return;
	pte = prp->p_pte;

	for (i = 0; i < xp->x_size; i++) {
		*(int *)pte &= ~PG_PROT;
		*(int *)pte |= tprot;
		pte++;
	}
	newptes(prp->p_pte, btop(prp->p_regva), xp->x_size);
}

#ifdef	TRFS
trfs_preg_getxfile(rpid, bp, ep)
	unsigned long rpid;
	unsigned *bp;
	register struct sysv_exdata *ep;
{
	register struct proc *p = u.u_procp;
	register reg_t *rp = NULL;
	register struct text *xp = NULL;
	register preg_t *prp;
	register size_t ts, ds;

	/*
 	 * Allocate region for data, and text for text.
	 */
	ts = clrnd(btoc(ep->a_tsize));
	ds = clrnd(btoc((ep->a_datorg & PGOFSET) + ep->a_dsize + ep->a_bsize));
	if ((rp = reg_alloc()) == NULL)
		goto bad;
	if (preg_attachreg(&u, rp, ep->a_datorg & ~SOFFMASK,
			PT_LIBDAT, 0, ds) == NULL)
		goto bad;
	rp->r_pgsz = ds;
	rp->r_vptr = (struct vnode *)0;
	if (!vsexpand(rp->r_pgsz, &rp->r_dmap, 0)) {
		u.u_error = ENOMEM;
		goto bad;
	}
	xp = preg_xalloc((struct vnode *) 0);
	if ((prp = preg_attachreg(&u, xp, ep->a_txtorg & ~SOFFMASK,
					PT_LIBTXT, 1, ts)) == NULL)
		goto bad;

	u.u_error = ClPload(rpid, bp, ep->a_datorg, ep->a_dsize,
				ep->a_doffset, UIOSEG_USER);
	preg_xattach(prp, (struct vnode *)0,  ep, 0, bp, rpid);
	if (u.u_error) {
bad:
		swkill(p, "trfs_preg_getxfile: Can't exec shared library");
		return(-1);
	}
	return(0);
}
#endif	TRFS
#endif	SYSV
