/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)kern_sig.c	7.1 (Berkeley) 6/5/86
 */



#include "../machine/reg.h"
#include "../machine/pte.h"
#include "../machine/psl.h"

#include "param.h"
#include "types.h"
#include "systm.h"
#include "user.h"
#include "vfs.h"
#include "vnode.h"
#include "proc.h"
#include "timeb.h"
#include "times.h"
#include "buf.h"
#include "text.h"
#include "seg.h"
#include "vm.h"
#include "acct.h"
#include "uio.h"
#include "kernel.h"
#include "wait.h"
#ifdef	SYSV
#include "../sysv/sys/signal.h"
#ifdef	RFS
#include "../sysv/sys/message.h"
#endif	RFS
#endif	SYSV

#ifdef vax
#include "../vax/mtpr.h"
#endif vax

#define	cantmask	(sigmask(SIGKILL)|sigmask(SIGCONT)|sigmask(SIGSTOP))
#define	stopsigmask	(sigmask(SIGSTOP)|sigmask(SIGTSTP)| \
			sigmask(SIGTTIN)|sigmask(SIGTTOU))

/*
 * Generalized interface signal handler.
 */
sigvec()
{
	register struct a {
		int	signo;
		struct	sigvec *nsv;
		struct	sigvec *osv;
	} *uap = (struct a  *)u.u_ap;
	struct sigvec vec;
	register struct sigvec *sv;
	register int sig;
	int bit;

	sig = uap->signo;
	if (sig <= 0 || sig >= NSIG || sig == SIGKILL || sig == SIGSTOP) {
		u.u_error = EINVAL;
		return;
	}
	sv = &vec;
	if (uap->osv) {
		sv->sv_handler = u.u_signal[sig];
		sv->sv_mask = u.u_sigmask[sig];
		bit = sigmask(sig);
		sv->sv_flags = 0;
		if ((u.u_sigonstack & bit) != 0)
			sv->sv_flags |= SV_ONSTACK;
		if ((u.u_sigintr & bit) != 0)
			sv->sv_flags |= SV_INTERRUPT;
		u.u_error =
		    copyout((caddr_t)sv, (caddr_t)uap->osv, sizeof (vec));
		if (u.u_error)
			return;
	}
	if (uap->nsv) {
		u.u_error =
		    copyin((caddr_t)uap->nsv, (caddr_t)sv, sizeof (vec));
		if (u.u_error)
			return;
		if (sig == SIGCONT && sv->sv_handler == SIG_IGN) {
			u.u_error = EINVAL;
			return;
		}
		setsigvec(sig, sv);
	}
}

setsigvec(sig, sv)
	int sig;
	register struct sigvec *sv;
{
	register struct proc *p;
	register int bit;

	bit = sigmask(sig);
	p = u.u_procp;
	/*
	 * Change setting atomically.
	 */
	(void) splhigh();
	u.u_signal[sig] = sv->sv_handler;
	u.u_sigmask[sig] = sv->sv_mask &~ cantmask;
	if (sv->sv_flags & SV_INTERRUPT)
		u.u_sigintr |= bit;
	else
		u.u_sigintr &= ~bit;
	if (sv->sv_flags & SV_ONSTACK)
		u.u_sigonstack |= bit;
	else
		u.u_sigonstack &= ~bit;
	if (sv->sv_handler == SIG_IGN) {
		p->p_sig &= ~bit;		/* never to be seen again */
		p->p_sigignore |= bit;
		p->p_sigcatch &= ~bit;
	} else {
		p->p_sigignore &= ~bit;
		if (sv->sv_handler == SIG_DFL)
			p->p_sigcatch &= ~bit;
		else
			p->p_sigcatch |= bit;
	}
	(void) spl0();
}

sigblock()
{
	struct a {
		int	mask;
	} *uap = (struct a *)u.u_ap;
	register struct proc *p = u.u_procp;

	(void) splhigh();
	u.u_r.r_val1 = p->p_sigmask;
	p->p_sigmask |= uap->mask &~ cantmask;
	(void) spl0();
}

sigsetmask()
{
	struct a {
		int	mask;
	} *uap = (struct a *)u.u_ap;
	register struct proc *p = u.u_procp;

	(void) splhigh();
	u.u_r.r_val1 = p->p_sigmask;
	p->p_sigmask = uap->mask &~ cantmask;
	(void) spl0();
}

sigpause()
{
	struct a {
		int	mask;
	} *uap = (struct a *)u.u_ap;
	register struct proc *p = u.u_procp;

	/*
	 * When returning from sigpause, we want
	 * the old mask to be restored after the
	 * signal handler has finished.  Thus, we
	 * save it here and mark the proc structure
	 * to indicate this (should be in u.).
	 */
	u.u_oldmask = p->p_sigmask;
	p->p_flag |= SOMASK;
	p->p_sigmask = uap->mask &~ cantmask;
	for (;;)
		sleep((caddr_t)&u, PSLEP);
	/*NOTREACHED*/
}
#undef cantmask

sigstack()
{
	register struct a {
		struct	sigstack *nss;
		struct	sigstack *oss;
	} *uap = (struct a *)u.u_ap;
	struct sigstack ss;

	if (uap->oss) {
		u.u_error = copyout((caddr_t)&u.u_sigstack, (caddr_t)uap->oss, 
		    sizeof (struct sigstack));
		if (u.u_error)
			return;
	}
	if (uap->nss) {
		u.u_error =
		    copyin((caddr_t)uap->nss, (caddr_t)&ss, sizeof (ss));
		if (u.u_error == 0)
			u.u_sigstack = ss;
	}
}

kill()
{
	register struct a {
		int	pid;
		int	signo;
	} *uap = (struct a *)u.u_ap;
	register struct proc *p;

	if (uap->signo < 0 || uap->signo > NSIG) {
		u.u_error = EINVAL;
		return;
	}
	if (uap->pid > 0) {
		/* kill single process */
		p = pfind(uap->pid);
		if (p == 0) {
			u.u_error = ESRCH;
			return;
		}
#ifdef	SYSV
		if ((p->p_universe == UNIVERSE_BSD &&
			u.u_uid && u.u_uid != p->p_uid) ||
		    (p->p_universe == UNIVERSE_SYSV &&
		    	(!(u.u_uid == 0 || u.u_uid == p->p_uid ||
		        u.u_ruid == p->p_uid || u.u_uid == p->p_suid ||
		        u.u_ruid == p->p_suid		))))
#else	SYSV
		if (u.u_uid && u.u_uid != p->p_uid)
#endif	SYSV
			u.u_error = EPERM;
		else if (uap->signo) {
#ifdef	SYSV
		    if ((u.u_procp->p_universe != p->p_universe &&
			uap->signo > SIGTERM) || 
			(p->p_universe == UNIVERSE_SYSV &&
			uap->signo >= NSIG_SV))
					/*
					 * This is needed because NSIG_SV is
					 * less than NSIG
					 */
			    u.u_error = EINVAL;
		    else
#endif	SYSV
			psignal(p, uap->signo);
		}
		return;
	}
	switch (uap->pid) {
	case -1:		/* broadcast signal */
		u.u_error = killpg1(uap->signo, 0, 1);
		break;
	case 0:			/* signal own process group */
		u.u_error = killpg1(uap->signo, 0, 0);
		break;
	default:		/* negative explicit process group */
		u.u_error = killpg1(uap->signo, -uap->pid, 0);
		break;
	}
	return;
}

killpg()
{
	register struct a {
		int	pgrp;
		int	signo;
	} *uap = (struct a *)u.u_ap;

	if (uap->signo < 0 || uap->signo > NSIG) {
		u.u_error = EINVAL;
		return;
	}
	u.u_error = killpg1(uap->signo, uap->pgrp, 0);
}

/* KILL CODE SHOULDNT KNOW ABOUT PROCESS INTERNALS !?! */

killpg1(signo, pgrp, all)
	int signo, pgrp, all;
{
	register struct proc *p;
	int f, error = 0;

	if (!all && pgrp == 0) {
		/*
		 * Zero process id means send to my process group.
		 */
		pgrp = u.u_procp->p_pgrp;
		if (pgrp == 0)
			return (ESRCH);
	}
	for (f = 0, p = allproc; p != NULL; p = p->p_nxt) {
		if ((p->p_pgrp != pgrp && !all) || p->p_ppid == 0 ||
		    (p->p_flag&SSYS) || (all && p == u.u_procp))
			continue;
#ifdef	SYSV
		if (p->p_universe == UNIVERSE_SYSV) {
		    if (!( u.u_uid == 0 || u.u_uid == p->p_uid || 
		        u.u_ruid == p->p_uid || u.u_uid == p->p_suid ||
		        u.u_ruid == p->p_suid		))
			    continue;
		} else 
#endif	SYSV
		if (u.u_uid != 0 && u.u_uid != p->p_uid &&
		    (signo != SIGCONT || !inferior(p))) {
			if (!all)
				error = EPERM;
			continue;
		}
		f++;
		if (signo)
#ifdef	SYSV
		    if (u.u_procp->p_universe != p->p_universe && 
			signo > SIGTERM)
			    error = EINVAL;
		    else
#endif	SYSV
			psignal(p, signo);
	}
	return (error ? error : (f == 0 ? ESRCH : 0));
}

/*
 * Send the specified signal to
 * all processes with 'pgrp' as
 * process group.
 */
gsignal(pgrp, sig)
	register int pgrp;
{
	register struct proc *p;

	if (pgrp == 0)
		return;
	for (p = allproc; p != NULL; p = p->p_nxt)
		if (p->p_pgrp == pgrp)
			psignal(p, sig);
}

/*
 * Send the specified signal to
 * the specified process.
 */
psignal(p, sig)
	register struct proc *p;
	register int sig;
{
	register int s;
	register int (*action)();
	int mask;

#ifdef	SYSV
	/*
	 * If the signal does not make sense for the universe that the
	 * process is currently in, return silently
	 */
	if(p->p_universe == UNIVERSE_BSD) {
#endif	SYSV
		if ((unsigned)sig >= NSIG)
			return;
#ifdef	SYSV
	} else if ((unsigned)sig >= NSIG_SV)
		return;
#endif	SYSV
	mask = sigmask(sig);

	/*
	 * If proc is traced, always give parent a chance.
	 */
	if (p->p_flag & STRC)
		action = SIG_DFL;
	else {
		/*
		 * If the signal is being ignored,
		 * then we forget about it immediately.
		 */
		if (p->p_sigignore & mask)
#ifdef	SYSV
			/* Can't always return! */
			if(p->p_universe == UNIVERSE_SYSV && sig == SIGCLD_SV) {
				action = SIG_IGN;
				goto next;
			} else
#endif	SYSV
			return;
		if (p->p_sigmask & mask)
			action = SIG_HOLD;
		else if (p->p_sigcatch & mask)
			action = SIG_CATCH;
		else
			action = SIG_DFL;
	}
next:
	if (sig) {
		p->p_sig |= mask;
#ifdef	SYSV
		if(p->p_universe == UNIVERSE_BSD)
#endif	SYSV
		switch (sig) {

		case SIGTERM:
			if ((p->p_flag&STRC) || action != SIG_DFL)
				break;
			/* fall into ... */

		case SIGKILL:
			if (p->p_nice > NZERO)
				p->p_nice = NZERO;
			break;

		case SIGCONT:
			p->p_sig &= ~stopsigmask;
			break;

		case SIGSTOP:
		case SIGTSTP:
		case SIGTTIN:
		case SIGTTOU:
			p->p_sig &= ~sigmask(SIGCONT);
			break;
		}
	}
#undef stops
	/*
	 * Defer further processing for signals which are held.
	 */
	if (action == SIG_HOLD)
		return;
	s = splhigh();
	switch (p->p_stat) {

	case SSLEEP:
		/*
		 * If process is sleeping at negative priority
		 * we can't interrupt the sleep... the signal will
		 * be noticed when the process returns through
		 * trap() or syscall().
		 */
		if (p->p_pri <= PZERO)
			goto out;
		/*
		 * Process is sleeping and traced... make it runnable
		 * so it can discover the signal in issig() and stop
		 * for the parent.
		 */
		if (p->p_flag&STRC)
			goto run;
#ifdef	SYSV
		if(p->p_universe == UNIVERSE_BSD)
#endif	SYSV
		switch (sig) {

		case SIGSTOP:
		case SIGTSTP:
		case SIGTTIN:
		case SIGTTOU:
			/*
			 * These are the signals which by default
			 * stop a process.
			 */
			if (action != SIG_DFL)
				goto run;
			/*
			 * Don't clog system with children of init
			 * stopped from the keyboard.
			 */
			if (sig != SIGSTOP && p->p_pptr == &proc[1]) {
				psignal(p, SIGKILL);
				p->p_sig &= ~mask;
				splx(s);
				return;
			}
			/*
			 * If a child in vfork(), stopping could
			 * cause deadlock.
			 */
			if (p->p_flag&SVFORK)
				goto out;
			p->p_sig &= ~mask;
			p->p_cursig = sig;
#ifdef	SYSV
			/* In System V SIGCLD is sent only for death! */
			if(p->p_pptr->p_universe == UNIVERSE_BSD)
#endif	SYSV
			psignal(p->p_pptr, SIGCHLD);
			stop(p);
			goto out;

		case SIGIO:
		case SIGURG:
		case SIGCHLD:
#ifdef GWS
		case SIGREF:		/* ISI: GWS: */
		case SIGADJ:
#else  GWS
		case SIGWINCH:
#endif GWS
			/*
			 * These signals are special in that they
			 * don't get propogated... if the process
			 * isn't interested, forget it.
			 */
			if (action != SIG_DFL)
				goto run;
			p->p_sig &= ~mask;		/* take it away */
			goto out;

		default:
			/*
			 * All other signals cause the process to run
			 */
			goto run;
		}
#ifdef	SYSV
		/* The proc is in UNIVERSE_SYSV */
		else if(sig == SIGCLD_SV || sig == SIGPWR /* || sig == SIGREF
					|| sig == SIGADJ */) {
			if(action != SIG_DFL) goto run;
			p->p_sig &= ~mask;
			goto out;
		} else
			goto run;
#endif	SYSV
		/*NOTREACHED*/

	case SSTOP:
		/*
		 * If traced process is already stopped,
		 * then no further action is necessary.
		 */
		if (p->p_flag&STRC)
			goto out;
#ifdef	SYSV
		if(p->p_universe == UNIVERSE_BSD)
#endif	SYSV
		switch (sig) {

		case SIGKILL:
			/*
			 * Kill signal always sets processes running.
			 */
			goto run;

		case SIGCONT:
			/*
			 * If the process catches SIGCONT, let it handle
			 * the signal itself.  If it isn't waiting on
			 * an event, then it goes back to run state.
			 * Otherwise, process goes back to sleep state.
			 */
			if (action != SIG_DFL || p->p_wchan == 0)
				goto run;
			p->p_stat = SSLEEP;
			goto out;

		case SIGSTOP:
		case SIGTSTP:
		case SIGTTIN:
		case SIGTTOU:
			/*
			 * Already stopped, don't need to stop again.
			 * (If we did the shell could get confused.)
			 */
			p->p_sig &= ~mask;		/* take it away */
			goto out;

		default:
			/*
			 * If process is sleeping interruptibly, then
			 * unstick it so that when it is continued
			 * it can look at the signal.
			 * But don't setrun the process as its not to
			 * be unstopped by the signal alone.
			 */
			if (p->p_wchan && p->p_pri > PZERO)
				unsleep(p);
			goto out;
		}
#ifdef	SYSV
		else if(sig == SIGKILL) 
			goto run;
		else if(p->p_wchan && p->p_pri > PZERO) {
			unsleep(p);
			goto out;
		}
#endif	SYSV
		/*NOTREACHED*/

	default:
		/*
		 * SRUN, SIDL, SZOMB do nothing with the signal,
		 * other than kicking ourselves if we are running.
		 * It will either never be noticed, or noticed very soon.
		 */
		if (p == u.u_procp && !noproc)
			aston();
		goto out;
	}
	/*NOTREACHED*/
run:
	/*
	 * Raise priority to at least PUSER.
	 */
#ifdef	is68k
	if (p->p_pri > PUSER)
		if ((p != u.u_procp || noproc) && p->p_stat == SRUN &&
		    (p->p_flag & SLOAD)) {
			remrq(p);
			p->p_pri = PUSER;
			setrq(p);
		} else
			p->p_pri = PUSER;
#else	is68k
	if (p->p_pri > PUSER)
		p->p_pri = PUSER;
#endif	is68k
	setrun(p);
out:
	splx(s);
}

/*
 * Returns true if the current
 * process has a signal to process.
 * The signal to process is put in p_cursig.
 * This is asked at least once each time a process enters the
 * system (though this can usually be done without actually
 * calling issig by checking the pending signal masks.)
 * A signal does not do anything
 * directly to a process; it sets
 * a flag that asks the process to
 * do something to itself.
 */
issig()
{
	register struct proc *p;
	register int sig;
	int sigbits, mask;

	p = u.u_procp;
	for (;;) {
		sigbits = p->p_sig &~ p->p_sigmask;
#ifdef	SYSV
		/* There exists a special condition in UNIVERSE_SYSV in which
		 * the control comes here even if SIGCLD is ignored. The action
		 * here should be to free all ZOMBIE children.
		 */
		if(p->p_universe == UNIVERSE_SYSV) {
			mask = sigmask(SIGCLD_SV);
			if((sigbits & mask) && (p->p_sigignore & mask)) {
				wait1(SYSV_WSPL, 0); /* free all zombies */
				p->p_sig &= ~mask;
				continue;
			}
		}
#endif	SYSV
		if ((p->p_flag&STRC) == 0)
			sigbits &= ~p->p_sigignore;
		if (p->p_flag&SVFORK)
			sigbits &= ~stopsigmask;
		if (sigbits == 0)
			break;
		sig = ffs((long)sigbits);
		mask = sigmask(sig);
		p->p_sig &= ~mask;		/* take the signal! */
		p->p_cursig = sig;
		if (p->p_flag&STRC && (p->p_flag&SVFORK) == 0) {
			/*
			 * If traced, always stop, and stay
			 * stopped until released by the parent.
			 */
#ifdef	SYSV
			/*
			 * In System V SIGCLD is sent only for death!
			 */
			if (p->p_pptr->p_universe == UNIVERSE_BSD)
#endif	SYSV
			psignal(p->p_pptr, SIGCHLD);
			do {
				stop(p);
				swtch();
			} while (!procxmt() && p->p_flag&STRC);

			/*
			 * If the traced bit got turned off,
			 * then put the signal taken above back into p_sig
			 * and go back up to the top to rescan signals.
			 * This ensures that p_sig* and u_signal are consistent.
			 */
			if ((p->p_flag&STRC) == 0) {
				p->p_sig |= mask;
				continue;
			}

			/*
			 * If parent wants us to take the signal,
			 * then it will leave it in p->p_cursig;
			 * otherwise we just look for signals again.
			 */
			sig = p->p_cursig;
			if (sig == 0)
				continue;

			/*
			 * If signal is being masked put it back
			 * into p_sig and look for other signals.
			 */
			mask = sigmask(sig);
			if (p->p_sigmask & mask) {
				p->p_sig |= mask;
				continue;
			}
		}
		switch ((int)u.u_signal[sig]) {

		case SIG_DFL:
			/*
			 * Don't take default actions on system processes.
			 */
			if (p->p_ppid == 0)
				break;
#ifdef	SYSV
			if(p->p_universe == UNIVERSE_BSD)
#endif	SYSV
			switch (sig) {

			case SIGTSTP:
			case SIGTTIN:
			case SIGTTOU:
				/*
				 * Children of init aren't allowed to stop
				 * on signals from the keyboard.
				 */
				if (p->p_pptr == &proc[1]) {
					psignal(p, SIGKILL);
					continue;
				}
				/* fall into ... */

			case SIGSTOP:
				if (p->p_flag&STRC)
					continue;
#ifdef	SYSV
				/* In System V SIGCLD is sent only for death! */
				if (p->p_pptr->p_universe == UNIVERSE_BSD)
#endif	SYSV
				psignal(p->p_pptr, SIGCHLD);
				stop(p);
				swtch();
				continue;

			case SIGCONT:
			case SIGCHLD:
			case SIGURG:
			case SIGIO:
#ifdef GWS
			case SIGREF:			/* ISI: GWS: */
			case SIGADJ:
#else  GWS
			case SIGWINCH:
#endif GWS
				/*
				 * These signals are normally not
				 * sent if the action is the default.
				 */
				continue;		/* == ignore */

			default:
				goto send;
			}
#ifdef	SYSV
			else if (sig == SIGCLD_SV || sig == SIGPWR /* ||
					sig == SIGREF || sig == SIGADJ */)
				continue;
			else 
				goto send;
#endif	SYSV
			/*NOTREACHED*/

		case (int)SIG_HOLD:
		case (int)SIG_IGN:
			/*
			 * Masking above should prevent us
			 * ever trying to take action on a held
			 * or ignored signal, unless process is traced.
			 */
			if ((p->p_flag&STRC) == 0)
				printf("issig\n");
			continue;

		default:
			/*
			 * This signal has an action, let
			 * psig process it.
			 */
			goto send;
		}
		/*NOTREACHED*/
	}
	/*
	 * Didn't find a signal to send.
	 */
	p->p_cursig = 0;
	return (0);

send:
	/*
	 * Let psig process the signal.
	 */
	return (sig);
}

/*
 * Put the argument process into the stopped
 * state and notify the parent via wakeup.
 * Signals are handled elsewhere.
 */
stop(p)
	register struct proc *p;
{

	p->p_stat = SSTOP;
	p->p_flag &= ~SWTED;
#ifdef	is68k			/* ISI: CDB: */
	if ((p->p_flag&STRC) && (p->p_dptr != 0))
		wakeup((caddr_t)p->p_dptr);
	else
		wakeup((caddr_t)p->p_pptr);
	if ((p->p_flag&STRC) == 0)
#ifdef	SYSV
		if (p->p_pptr->p_universe == UNIVERSE_BSD)
#endif	SYSV
		psignal(p->p_pptr, SIGCHLD);
#else	is68k
	wakeup((caddr_t)p->p_pptr);
#endif	is68k
}

/*
 * Perform the action specified by
 * the current signal.
 * The usual sequence is:
 *	if (issig())
 *		psig();
 * The signal bit has already been cleared by issig,
 * and the current signal number stored in p->p_cursig.
 */
psig()
{
	register struct proc *p = u.u_procp;
	register int sig = p->p_cursig;
	int mask = sigmask(sig), returnmask;
	register int (*action)();

	if (sig == 0)
		panic("psig");
	action = u.u_signal[sig];
	if (action != SIG_DFL) {
		if (action == SIG_IGN || (p->p_sigmask & mask))
			panic("psig action");
		u.u_error = 0;
		/*
		 * Set the new mask value and also defer further
		 * occurences of this signal (unless we're simulating
		 * the old signal facilities). 
		 *
		 * Special case: user has done a sigpause.  Here the
		 * current mask is not of interest, but rather the
		 * mask from before the sigpause is what we want restored
		 * after the signal processing is completed.
		 */
		(void) splhigh();
		if (p->p_flag & SOUSIG) {
			if (sig != SIGILL && sig != SIGTRAP) {
				u.u_signal[sig] = SIG_DFL;
				p->p_sigcatch &= ~mask;
			}
			mask = 0;
		}
#ifdef	SYSV
		if (p->p_universe == UNIVERSE_SYSV) {
			returnmask = p->p_sigmask;
			if(p->p_chold & mask)
				/* Sigset, so block further signals */
				p->p_sigmask |= mask;
			else if(sig != SIGILL && sig != SIGTRAP &&
							sig != SIGPWR) {
				u.u_signal[sig] = SIG_DFL;
				p->p_sigcatch &= ~mask;
			}
		 } else {
#endif	SYSV
			if (p->p_flag & SOMASK) {
				returnmask = u.u_oldmask;
				p->p_flag &= ~SOMASK;
			} else
				returnmask = p->p_sigmask;
			p->p_sigmask |= u.u_sigmask[sig] | mask;
#ifdef	SYSV
		}
#endif	SYSV
		(void) spl0();
		u.u_ru.ru_nsignals++;
#ifdef	SYSV
		if (p->p_universe == UNIVERSE_SYSV)
			sysv_sendsig(action, sig, returnmask);
		else
#endif	SYSV
			sendsig(action, sig, returnmask);
		p->p_cursig = 0;
		return;
	}
	u.u_acflag |= AXSIG;
	switch (sig) {

	case SIGILL:
	case SIGIOT:
	case SIGBUS:
	case SIGQUIT:
	case SIGTRAP:
	case SIGEMT:
	case SIGFPE:
	case SIGSEGV:
	case SIGSYS:
#ifdef	SYSV
		if (p->p_universe == UNIVERSE_BSD)
#endif	SYSV
		u.u_arg[0] = sig;
#ifdef	TRFS
		if (ClCore())
#else	TRFS
		if (core())
#endif	TRFS
			sig += 0200;
	}
#ifdef	TRFS
	ClExit(sig);
#else	TRFS
	exit(sig);
#endif	TRFS
}

/*
 * Create a core image on the file "core"
 * If you are looking for protection glitches,
 * there are probably a wealth of them here
 * when this occurs to a suid command.
 *
 * It writes UPAGES block of the
 * user.h area followed by the entire
 * data+stack segments.
 */
core()
{
	struct vnode *vp;
	struct vattr vattr;
	int len;
	extern char *strncpy();
	int offset = 0, fd;
#if	defined(SYSV) && defined(RFS)
	unsigned char tsyscall;
#endif	defined(SYSV) && defined(RFS)

	if (u.u_uid != u.u_ruid || 
#ifdef	SYSV
	       ((u.u_procp->p_universe == UNIVERSE_BSD) && u.u_gid != u.u_rgid))
#else	SYSV
	       u.u_gid != u.u_rgid)
#endif	SYSV
		return (0);
#ifdef	SYSV
	if (ctob(UPAGES+u.u_dsize+u.u_ssize) >=
	    u.u_rlimit[RLIMIT_CORE].rlim_cur)
		if (u.u_procp->p_universe == UNIVERSE_BSD)
			return (0);
		else {
			u.u_dsize = 0;
			if (ctob(UPAGES+u.u_ssize) >=
			     u.u_rlimit[RLIMIT_CORE].rlim_cur)
			     u.u_ssize = 0;
		}
#else	SYSV
	if (ctob(UPAGES+u.u_dsize+u.u_ssize) >=
	    u.u_rlimit[RLIMIT_CORE].rlim_cur)
		return (0);
#endif	SYSV
	u.u_error = 0;
#ifdef	TRFS
	u.u_tok = 0;
#endif	TRFS

	vattr_null(&vattr);
	vattr.va_type = VREG;
#ifdef	SYSV
	vattr.va_mode = (u.u_procp->p_universe == UNIVERSE_BSD) ? 0644 : 0666;
#ifdef	RFS
	u.u_procp->p_systemcall = DUCOREDUMP;
#endif	RFS
#else	SYSV
	vattr.va_mode = 0644;
#endif	SYSV
	u.u_error =
	    vn_create("core", UIOSEG_KERNEL, &vattr, NONEXCL, VWRITE, &vp);
	if (u.u_error)
		return (0);
	if (vattr.va_nlink != 1) {
		u.u_error = EFAULT;
		goto out;
	}
#ifndef	NOMMAP
	for (fd = 0; fd < NOFILE; fd++) 
		if (u.u_ofile[fd] && (u.u_pofile[fd] & UF_MAPPED))
			munmapfd(fd);
#endif	NOMMAP
	vattr_null(&vattr);
	vattr.va_size = 0;
	VOP_SETATTR(vp, &vattr, u.u_cred);
	u.u_acflag |= ACORE;

	u.u_error = vn_rdwr(UIO_WRITE, vp, (caddr_t)&u, ctob(UPAGES), 0, 
		UIOSEG_KERNEL, IO_UNIT, (int *)0);
	offset += ctob(UPAGES);
	if (u.u_error == 0)
		u.u_error = vn_rdwr(UIO_WRITE, vp,
		    (caddr_t)ctob(dptov(u.u_procp, 0)), ctob(u.u_dsize),
		    offset, UIOSEG_USER, IO_UNIT, (int *)0);
	offset += ctob(u.u_dsize);
	if (u.u_error == 0)
		u.u_error = vn_rdwr(UIO_WRITE, vp,
		    (caddr_t)ctob(sptov(u.u_procp, u.u_ssize - 1)),
		    ctob(u.u_ssize), offset, UIOSEG_USER, IO_UNIT, (int *)0);
	offset += ctob(u.u_ssize);

out:	VN_RELE(vp);
	return (u.u_error == 0);
}
