/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)kern_descrip.c	7.1 (Berkeley) 6/5/86
 */

#include "param.h"
#include "systm.h"
#include "user.h"
#include "kernel.h"
#include "vnode.h"
#include "vfs.h"
#include "proc.h"
#include "file.h"
#include "socket.h"
#include "socketvar.h"
#include "stat.h"

#include "ioctl.h"
#include "../specfs/snode.h"

/*
 * Descriptor management.
 */

/*
 * TODO:
 *	eliminate u.u_error side effects
 */

/*
 * System calls on descriptors.
 */
getdtablesize()
{

	u.u_r.r_val1 = NOFILE;
}

getdopt()
{

}

setdopt()
{

}

dup(uap)
	register struct a {
		int	i;
	} *uap;
{
	struct file *fp;
	int j;

	if (uap->i &~ 077) { uap->i &= 077; dup2(uap); return; }	/* XXX */

	GETF(fp, uap->i);
	j = ufalloc(0);
	if (j < 0)
		return;
	dupit(j, fp, u.u_pofile[uap->i] &~ UF_EXCLOSE);
}

dup2(uap)
	register struct a {
		int	i, j;
	} *uap;
{
	register struct file *fp, *fp2;

	GETF(fp, uap->i);
	if (uap->j < 0 || uap->j >= NOFILE) {
		u.u_error = EBADF;
		return;
	}
	u.u_r.r_val1 = uap->j;
	if (uap->i == uap->j)
		return;
	if (u.u_ofile[uap->j]) {
		/* Release all System-V style record locks, if any */
#if defined(SYSV) && defined(RFS)
	    	if ((fp->f_type != DTYPE_VNODE) ||
		    !(((struct vnode *)fp->f_data)->v_flag & VRFSNODE))
#endif
		(void) vno_lockrelease(u.u_ofile[uap->j]);	/* errors? */

		if (u.u_pofile[uap->j] & UF_MAPPED)
			munmapfd(uap->j);
		fp2 = u.u_ofile[uap->j];
		u.u_ofile[uap->j] = 0;
		closef(fp2, 1);
		if (u.u_error)
			return;
	}
	dupit(uap->j, fp, u.u_pofile[uap->i] &~ UF_EXCLOSE);
}

dupit(fd, fp, flags)
	int fd;
	register struct file *fp;
	register int flags;
{

	u.u_ofile[fd] = fp;
	u.u_pofile[fd] = flags &~ UF_EXCLOSE;
	fp->f_count++;
	if (fd > u.u_lastfile)
		u.u_lastfile = fd;
}

/*
 * The file control system call.
 */
fcntl(uap)
	register struct a {
		int	fdes;
		int	cmd;
		int	arg;
	} *uap;
{
	register struct file *fp;
	register i;
	register char *pop;
	struct flock ld;
	register int oldwhence;
	int cmd = uap->cmd;

	GETF(fp, uap->fdes);
	pop = &u.u_pofile[uap->fdes];

	switch(cmd) {

	case F_DUPFD:
		i = uap->arg;
		if (i < 0 || i >= NOFILE) {
			u.u_error = EINVAL;
			return;
		}
		if ((i = ufalloc(i)) < 0)
			return;
		dupit(i, fp, *pop &~ UF_EXCLOSE);
		break;

	case F_GETFD:
		u.u_r.r_val1 = *pop & 1;
		break;

	case F_SETFD:
		*pop = (*pop &~ 1) | (uap->arg & 1);
		break;

	case F_GETFL:
		u.u_r.r_val1 = fp->f_flag+FOPEN;
		break;

	case F_SETFL:
		fp->f_flag &= FCNTLCANT;
		fp->f_flag |= (uap->arg-FOPEN) &~ FCNTLCANT;
		u.u_error = fset(fp, FNDELAY, fp->f_flag & FNDELAY);
		if (u.u_error)
			break;
		u.u_error = fset(fp, FASYNC, fp->f_flag & FASYNC);
		if (u.u_error)
			(void) fset(fp, FNDELAY, 0);
		break;

	case F_GETOWN:
		u.u_error = fgetown(fp, &u.u_r.r_val1);
		break;

	case F_SETOWN:
		u.u_error = fsetown(fp, uap->arg);
		break;

		/* System-V Record-locking (lockf() maps to fcntl()) */
	case F_GETLK:
	case F_SETLK:
	case F_SETLKW:
		/* get flock structure from user-land */
		if (copyin((caddr_t) uap->arg, (caddr_t) &ld, sizeof (ld))) 
			return;
#ifdef	SYSV
		if (u.u_procp->p_universe == UNIVERSE_SYSV) {
			register int tmp;

			tmp = ld.l_pid;
			ld.l_pid = ld.l_xxx;
			ld.l_xxx = tmp;
		}
#endif	SYSV

		oldwhence = ld.l_whence;	/* save to renormalize later */
		if (u.u_error = reclock(fp, cmd, &ld, uap->fdes, 0))
			return;

		/* if F_GETLK, return flock structure to user-land */
		if (cmd == F_GETLK) {
			/* per SVID, change only 'l_type' field if unlocked */
			if (ld.l_type == F_UNLCK) {
				if (copyout((caddr_t) &ld.l_type,
				    (caddr_t)&((struct flock*)uap->arg)->l_type,
				    sizeof (ld.l_type))) {
					return;
				}
			} else {
				if (u.u_error = rewhence(&ld, fp, oldwhence))
					return;
#ifdef	SYSV
				if (u.u_procp->p_universe == UNIVERSE_SYSV) {
					register int tmp;

					tmp = ld.l_pid;
					ld.l_pid = ld.l_xxx;
					ld.l_xxx = tmp;
				}
#endif	SYSV
				if (copyout((caddr_t) &ld, (caddr_t) uap->arg,
				    sizeof (ld))) {
					return;
				}
			}
		}
		break;

	default:
		u.u_error = EINVAL;
	}
}

fset(fp, bit, value)
	struct file *fp;
	int bit, value;
{
	if (value)
		fp->f_flag |= bit;
	else
		fp->f_flag &= ~bit;

	return (fioctl(fp, (int)(bit == FNDELAY ? FIONBIO : FIOASYNC),
	    (caddr_t)&value));
}

fgetown(fp, valuep)
	struct file *fp;
	int *valuep;
{
	int error;

	switch (fp->f_type) {

	case DTYPE_SOCKET:
		*valuep = ((struct socket *)fp->f_data)->so_pgrp;
		return (0);

	default:
		error = fioctl(fp, (int)TIOCGPGRP, (caddr_t)valuep);
		*valuep = -*valuep;
		return (error);
	}
}

fsetown(fp, value)
	struct file *fp;
	int value;
{

	if (fp->f_type == DTYPE_SOCKET) {
		((struct socket *)fp->f_data)->so_pgrp = value;
		return (0);
	}
	if (value > 0) {
		struct proc *p = pfind(value);
		if (p == 0)
			return (EINVAL);
		value = p->p_pgrp;
	} else
		value = -value;
	return (fioctl(fp, (int)TIOCSPGRP, (caddr_t)&value));
}

fioctl(fp, cmd, value)
	struct file *fp;
	int cmd;
	caddr_t value;
{

	return ((*fp->f_ops->fo_ioctl)(fp, cmd, value));
}

close(uap)
	struct a {
		int	i;
	} *uap;
{
	register int i = uap->i;
	register struct file *fp;
	register u_char *pf;

	GETF(fp, i);
	if (fp == 0)
		return;
	
	/* Release all System-V style record locks, if any */
#if defined(SYSV) && defined(RFS)
	if ((fp->f_type != DTYPE_VNODE) ||
	    !(((struct vnode *)fp->f_data)->v_flag & VRFSNODE))
#endif
	(void) vno_lockrelease(fp);	/* WHAT IF error returned? */

	pf = (u_char *)&u.u_pofile[i];
	if (*pf & UF_MAPPED)
		munmapfd(i);
	u.u_ofile[i] = NULL;
	while (u.u_lastfile >= 0 && u.u_ofile[u.u_lastfile] == NULL)
		u.u_lastfile--;
	*pf = 0;
	closef(fp, 1);
	/* WHAT IF u.u_error ? */
}

fstat(uap)
	register struct a {
		int	fdes;
		struct	stat *sb;
	} *uap;
{
	register struct file *fp;
	struct stat ub;

	GETF(fp, uap->fdes);
	switch (fp->f_type) {

	case DTYPE_VNODE:
		u.u_error = vno_stat((struct vnode *)fp->f_data, &ub);
		break;

	case DTYPE_SOCKET:
		u.u_error = soo_stat((struct socket *)fp->f_data, &ub);
		break;

	default:
		panic("fstat");
		/*NOTREACHED*/
	}
	if (u.u_error == 0)
		u.u_error = copyout((caddr_t)&ub, (caddr_t)uap->sb,
		    sizeof (ub));
}

/*
 * Allocate a user file descriptor.
 */
ufalloc(i)
	register int i;
{

	for (; i < NOFILE; i++)
		if (u.u_ofile[i] == NULL) {
			u.u_r.r_val1 = i;
			u.u_pofile[i] = 0;
			if (i > u.u_lastfile)
				u.u_lastfile = i;
			return (i);
		}
	u.u_error = EMFILE;
	return (-1);
}

ufavail()
{
	register int i, avail = 0;

	for (i = 0; i < NOFILE; i++)
		if (u.u_ofile[i] == NULL)
			avail++;
	return (avail);
}

struct	file *lastf;
/*
 * Allocate a user file descriptor
 * and a file structure.
 * Initialize the descriptor
 * to point at the file structure.
 */
struct file *
falloc()
{
	register struct file *fp;
	register i;

	i = ufalloc(0);
	if (i < 0)
		return (NULL);
	if (lastf == 0)
		lastf = file;
	for (fp = lastf; fp < fileNFILE; fp++)
		if (fp->f_count == 0)
			goto slot;
	for (fp = file; fp < lastf; fp++)
		if (fp->f_count == 0)
			goto slot;
	tablefull("file");
	u.u_error = ENFILE;
	return (NULL);
slot:
	u.u_ofile[i] = fp;
	fp->f_count = 1;
	fp->f_data = 0;
	fp->f_offset = 0;
	fp->f_offset_sysv = 0;	/* SYSV directory emulation */
	fp->f_offset_buf = 0;	/* SYSV directory emulation */
#ifdef	TRFS
	fp->f_rmid = 0;
#endif	TRFS
	crhold(u.u_cred);
	fp->f_cred = u.u_cred;
	lastf = fp + 1;
	return (fp);
}

/*
 * Convert a user supplied file descriptor into a pointer
 * to a file structure.  Only task is to check range of the descriptor.
 * Critical paths should use the GETF macro.
 */
struct file *
getf(f)
	register int f;
{
	register struct file *fp;

	if ((unsigned)f >= NOFILE || (fp = u.u_ofile[f]) == NULL) {
		u.u_error = EBADF;
		return (NULL);
	}
	return (fp);
}

int rfs_closef_lock;
#define RFSLOCKED 01
#define RFSWANT   02
#define CLOSE_LOCK() \
{ \
	while (rfs_closef_lock & RFSLOCKED) { \
	    rfs_closef_lock |= RFSWANT; \
	    sleep (&rfs_closef_lock, PINOD); \
	} \
	rfs_closef_lock |= RFSLOCKED; \
}

#define CLOSE_UNLOCK() \
{ \
    rfs_closef_lock &= ~RFSLOCKED; \
    if (rfs_closef_lock & RFSWANT) { \
	rfs_closef_lock &= ~RFSWANT; \
	wakeup(&rfs_closef_lock); \
    } \
}
	
/*
 * Internal form of close.
 * Decrement reference count on file structure.
 * If last reference not going away, but no more
 * references except in message queues, run a
 * garbage collect.  This would better be done by
 * forcing a gc() to happen sometime soon, rather
 * than running one each time.
 */
closef(fp, opened)
	register struct file *fp;
{

	if (fp == NULL)
		return;

#if defined(SYSV) && defined(RFS)
    	if ((fp->f_type == DTYPE_VNODE) &&
	    (((struct vnode *)fp->f_data)->v_flag & VRFSNODE)) {
		extern rtenable, rtindex;
		/*
		 * v_data should be casted to rfsnode.  At this point it
		 * seems impossible or very close to it.  One of the header
		 * files for System V declares an external routine dequeue.
		 * One of the BSD headers files defines dequeue to be some
		 * macro which makes the compiler barf.
		 */
		rfslock (((struct vnode *)fp->f_data)->v_data);
		rtenable = 1;
		rtindex = 0;
		rfs_trace (1, 0);
		Rfs_close (fp);
		rfs_trace (6, 0);
		rfsunlock (((struct vnode *)fp->f_data)->v_data);

		rtenable = 0;
		if (fp->f_count-- > 1)
		    return;
		VN_RELE((struct vnode *)fp->f_data);
	} else
#endif
	{
	    if (fp->f_count > 1) {
#ifdef SYSV
		if ((fp->f_type == DTYPE_VNODE) &&
		    (((struct vnode *)fp->f_data)->v_type == VCHR)) {
			register struct snode *sp;

			if ((sp = VTOS((struct vnode *)fp->f_data)) &&
			    (sp->s_sptr))
				strclean(sp);
		}
#endif	SYSV
		fp->f_count--;
		if (fp->f_count == fp->f_msgcount)
			unp_gc();
#ifdef	TRFS
		/*
		 * If this is the last close for this process, then close on 
		 * the cousin.  NOTE: you must zero u.u_ofile before calling 
		 * closef!  CTH
		 */
		if (fp->f_type == DTYPE_TRFS && !(u.u_procp->p_flag & SWEXIT)) {
			register int i, ref;

			for (i = 0, ref = 0; i <= u.u_lastfile; i++)
				if (u.u_ofile[i] == fp) {
					ref = 1;
					break;
				}
			if (ref == 0 && (i = RmidToRpid(fp->f_rmid)))
				SdClose(i, fp->f_rfdes);
		}
#endif	TRFS
		return;
	    }
	    if (fp->f_count < 1)
		panic("closef: count < 1");
	    if (opened)
		(*fp->f_ops->fo_close)(fp);
	}
	crfree(fp->f_cred);
	fp->f_count = 0;
}

/*
 * Normalize SystemV-style record locks
 */
rewhence(ld, fp, newwhence)
	struct flock *ld;
	struct file *fp;
	int newwhence;
{
	struct vattr va;
	register int error;

	/* if reference to end-of-file, must get current attributes */
	if ((ld->l_whence == 2) || (newwhence == 2)) {
		if (error = VOP_GETATTR((struct vnode *)fp->f_data, &va,
		    u.u_cred))
			return(error);
	}

	/* normalize to start of file */
	switch (ld->l_whence) {
	case 0:
		break;
	case 1:
		ld->l_start += fp->f_offset;
		break;
	case 2:
		ld->l_start += va.va_size;
		break;
	default:
		return(EINVAL);
	}

	/* renormalize to given start point */
	switch (ld->l_whence = newwhence) {
	case 1:
		ld->l_start -= fp->f_offset;
		break;
	case 2:
		ld->l_start -= va.va_size;
		break;
	}
	return(0);
}

/*
 * Apply an advisory lock on a file descriptor.
 */
flock(uap)
	register struct a {
		int	fd;
		int	how;
	} *uap;
{
	register struct file *fp;

	GETF(fp, uap->fd);
	if (fp->f_type != DTYPE_VNODE) {
		u.u_error = EOPNOTSUPP;
		return;
	}
	if (uap->how & LOCK_UN) {
		vno_bsd_unlock(fp, FSHLOCK|FEXLOCK);
		return;
	}
	if ((uap->how & (LOCK_SH | LOCK_EX)) == 0)
		return;					/* error? */
	if (uap->how & LOCK_EX)
		uap->how &= ~LOCK_SH;
	u.u_error = vno_bsd_lock(fp, uap->how);
}


reclock(fp, cmd, ld, fdes, lf)
	register struct file *fp;
	struct flock *ld;
	int cmd, fdes, lf;
{
	register char *pop;
	int error;

	pop = &u.u_pofile[fdes];
	/* First off, allow only vnodes here */
	if (fp->f_type != DTYPE_VNODE)
		return(EBADF);

	/* *** NOTE ***
	 * The SVID does not say what to return on file access errors!
	 * Here, EBADF is returned, which is compatible with S5R3
	 * and is less confusing than EACCES
	 */
	/* check access permissions */
	if (cmd != F_GETLK && cmd != F_GETLKW) {
		switch (ld->l_type) {
		case F_RDLCK:
			if (!(fp->f_flag & FREAD))
				return(EBADF);
			break;

		case F_WRLCK:
			if (!lf && !(fp->f_flag & FWRITE))
				return(EBADF);
			break;

		case F_UNLCK:
			break;

		default:
			return(EINVAL);
		}
	}
	
	/* convert offset to start of file */
	if (error = rewhence(ld, fp, 0))
		return(error);

	/* convert negative lengths to positive */
	if (ld->l_len < 0) {
		ld->l_start += ld->l_len;		/* adjust start point */
		ld->l_len = -(ld->l_len);		/* absolute value */
	}
 
	/* check for validity */
	if (ld->l_start < 0)
		return(EINVAL);

	if ((cmd != F_GETLK) && (cmd != F_GETLKW) && (ld->l_type != F_UNLCK)) {
		/* If any locking is attempted, mark file locked
		 * to force unlock on close.
		 * Also, since the SVID specifies that the FIRST
		 * close releases all locks, mark process to
		 * reduce the search overhead in vno_lockrelease().
		 */
		*pop |= UF_FDLOCK;
		u.u_procp->p_flag |= SLKDONE;
	}

	/*
	 * Dispatch out to vnode layer to do the actual locking.
	 * Then, translate error codes for SVID compatibility
	 */
	if ((error = VOP_LOCKCTL((struct vnode *)fp->f_data,
	    ld, cmd, fp->f_cred, lf)) == EWOULDBLOCK)
		return(EACCES);
	else
		return(error);		/* some other error code */
}
