/*
 *  $Header: npc_vnodeops.c,v 1.2 89/04/07 12:38:37 root Exp $
 */
#include "param.h"
#include "vfs.h"
#include "../netinet/in.h"	/* needed for mount.h */
#include "socketvar.h"
#undef NFS
#include "mount.h"
#include "errno.h"
#include "file.h"
#include "user.h"
#include "proc.h"
#include "stat.h"
#include "vnode.h"
#include "nbios.h"
#include "npcfs.h"
#include "smb.h"
#include "sendbuf.h"
#include "dir.h"
int hutchdebug;
char* kmem_alloc();
int npc_open();
int npc_close();
int npc_rdwr();
int npc_ioctl();
int npc_getattr();
int npc_setattr();
int npc_access();
int npc_inactive();
int npc_badop();
int npc_lookup();
int npc_create();
int npc_fsync();
int npc_remove();
int npc_link();
int npc_rename();
int npc_mkdir();
int npc_rmdir();
int npc_symlink();
int npc_readdir();
int npc_lockctl();
int npc_select();
#ifdef SYSV
int s5getdents();
#endif SYSV


struct vnodeops	npc_vnodeops = {
	npc_open,		  
	npc_close,
	npc_rdwr,	
	npc_ioctl,
	npc_select,
	npc_getattr,	
	npc_setattr,
	npc_access,		 
	npc_lookup,
	npc_create,
	npc_remove,
	npc_link,
	npc_rename,
	npc_mkdir,
	npc_rmdir,
	npc_readdir,
	npc_symlink,
	npc_badop,		    /* readlink */
	npc_fsync,
	npc_inactive,
	npc_badop,		    /* bmap */
	npc_badop,		    /* strategy */
	npc_badop,		    /* badop */
	npc_badop,		    /* badop */
	npc_lockctl,
	npc_badop,		    /* invalop */
#ifdef	SYSV
	s5getdents,		/* ISI: SYSV */
#else	SYSV
	npc_badop,
#endif	SYSV
};

extern struct timeval time;


npc_open(vpp, flag, cred)
	struct vnode **vpp;
	int flag;
	struct ucred *cred;
{
    struct sendbuf	*sb;
    struct vnode	*vp = *vpp;
    struct pcmountinfo  *pmp = vftomi(vp->v_vfsp);
    struct pcnode	*pcp = vtopc(vp);
    char		*cp;
    int			havestuff = 0, havepath = 0;
    int			conjunct, rerror=0, smb_omode,pidx;
    unsigned int	trash;
    unsigned char	rcls;
    unsigned short	err,fid,access;
    time_t		timex;

#define		NPC_OMASK		(FREAD|FWRITE|FEXCLO)

    CKHEALTH(pmp);
    
    if ( vp->v_type == VDIR ) {
	pcp->pc_opencnt++;
	return 0;
    }

    /*----------------------------------------------------------------+
    |  form a conjunction of omode and flag.  If that differs from    |
    |  omode, we have some work to do.  If WRITE currently in effect  |
    |  and EXCLO needed, we require special two-step sequence of open |
    |  and close to get where we want.  We finally do the SMBopen we  |
    |  really want, and if another open is in effect, we close it. The|
    |  idea is to never make the current open vulnerable.             |
    +--------------------------------------------------------------- */

    conjunct = ( pcp->pc_omode ) | ( flag & NPC_OMASK );
    if ( conjunct != pcp->pc_omode ) {
	
	/*----------------------------------------------------+
	|  npc_create tells us to leave things along and just |
	|  bump up opencnt by setting fid and omode but leaving
	|  opencnt zero.  This is in the case where npc_create|
	|  created the file readonly but we have to write it  |
	|  so the create's fid must stand		      |
	+----------------------------------------------------*/
	if ( pcp->pc_opencnt == 0 && pcp->pc_omode != 0 ) {
	    pcp->pc_opencnt = 1;
	    /* can we safely disregard FEXCLO ? */
	    goto out;
	}

	cp = kmem_alloc(MAXPATHLEN);
        npc_buildpath(vp,cp);
	havepath = 1;
	
	if ( (pcp->pc_omode & FWRITE) &&
	     (conjunct & FEXCLO) ) {		/*  special handling */
	    
	    /* probably need phony pid here, if stays open for any reason */
	    rerror = doSMB1(buildSMBopen, havestuff, pmp, havestuff, &sb,
			  PHONY_PID,0x40,0,cp);  /*  readonly, deny nothing */
#ifdef DEBUG
	    printf("npc_open: doSMBopen special handling %d\n",rerror);
#endif DEBUG
	    if (rerror) {
		goto out;
	    }
	    havestuff = 1;
	    extractSMBopen(sb->buf,&rcls,&err,&fid,&trash,&trash,&trash,
			   &access);
	    if ( rerror = map_SMBopen_err(rcls,err) ) {
#ifdef DEBUG
	    printf("npc_open: special handling rcls %d err %d\n",rcls,err);
#endif DEGUG
		goto out;
	    }

	    /*---------------------------------------------------------+
	    |  close the current open.  Any FWRITE being closed is not |
	    |  needed by any process here -- vfs layer has seen to that|
	    |  If the close fails, go ahead and install the replacement|
	    |  open but error out.				       |
	    +---------------------------------------------------------*/

#ifdef DEBUG
	    printf("npc_open: about to redo_locks, special case\n");
#endif DEBUG
	    rerror = redo_locks(vp,pcp->pc_smbfid,fid,havestuff);

	    timex = time.tv_sec;
	    Greenwich_to_local(&timex);
	    /* best to use phony pid here */
	    rerror = doSMB1(buildSMBclose,
				(rerror)? HAVENOSOCK : HAVESOCK,
				pmp,havestuff,&sb,
			  	pcp->pc_opid,pcp->pc_smbfid,timex);

	    pcp->pc_smbfid = fid;
	    pcp->pc_omode = FREAD;
	    pcp->pc_opid = PHONY_PID;
	
	    if ( rerror ) {
		printf("npc_open: doSMB close in special case %d\n",rerror);
		if ( pmp->m_sock->ps_state == PS_RECV ) {
		    printf("***** WARNING: NPC file failure to close *****\n");
		}
		goto out;
	    }
	    conjunct = (pcp->pc_omode) | (flag & NPC_OMASK);
	}

	/*------------------------------------------------------------+
	|  now more conventional action.  This is usually the first   |
	|  SMB traffic, but follows the special handling above.  If   |
	|  the open fails here, the new one is already installed as   |
	|  the standing open so we can just get out.                  |
	+------------------------------------------------------------*/ 

	if ( conjunct & FEXCLO ) {
	    smb_omode = 0x20;	      /* deny write */
	    pidx = u.u_procp->p_pid;  /* essential for denywrite case */
	    if ( conjunct & FWRITE ) {
		smb_omode |= 0x02;	/* write AND READ access because
					 * FEXCLO & FWRITE alone is a state
					 * we can't protect in a replacement
					 * action so we won't permit it
					 */
		conjunct |= FREAD;	/* may already be there; make pc_omode
					 * reflect the hack mentioned above,
					 * or it won't accomplish a thing 
					 */
 	    } /* else just read, the zero that is already there */
	} else {
	    smb_omode = 0x40;	   /* deny nothing */
	    pidx = PHONY_PID;	   /* necessary for no deny case */
	    if ( (conjunct & FWRITE) && (conjunct & FREAD) ) {
		smb_omode |= 0x02;	/* read and write access */
	    } else if ( conjunct & FWRITE ) {
		smb_omode |= 0x01;	/* write access */
	    } /* else just read, the zero that is already there */
	}
#ifdef DEBUG
	printf("npc_open: omode %o flag %o smb_omode 0x%x\n",
				pcp->pc_omode,flag,smb_omode);
#endif DEBUG

	rerror = doSMB1(buildSMBopen, havestuff, pmp, havestuff, &sb,
			     pidx,smb_omode, 0, cp);
#ifdef DEBUG
	printf("npc_open: conventional action doSMBopen %d\n",rerror);
#endif DEBUG
        if ( rerror ) {
	    goto out;
	}
	havestuff = 1;
	
	extractSMBopen(sb->buf,&rcls,&err,&fid,&trash,&trash,&trash,&access);
	if ( rerror = map_SMBopen_err(rcls,err) ) {
#ifdef DEBUG
	printf("npc_open: conventional action rcls %d err %d\n",rcls,err);
#endif DEBUG
	    goto out;
	}
	/*------------------------------------------------------+
	|  process cant send SMBexit if an exclo in effect under|
	|  its name when process exits.  If the same process    |
	|  closes the exclo before it exits ( no one else has   |
	|  exclo in effect at the time), it will turn off the   |
	|  flag we're about to set here.  otherwise, there is a |
	|  hole in our scheme to SMBexit using pc_opid at a     |
	|  date. (see comments in npc_close)			|
	+------------------------------------------------------*/
	if ( pidx != PHONY_PID ) {
	    u.u_procp->p_xflag = PX_NPCUSE | 
		( (u.u_procp->p_xflag & PX_EXCLO) + 1 );
#ifdef DEBUG
	    printf("npc_open: xflag %x\n",u.u_procp->p_xflag);
#endif DEBUG
	}

	/*------------------------------------------------------+
	|  notice here that the file has to be open already for |
	|  us to do anything.  In the simple case of opening a  |
	|  file that isn't open already, only the "conventional |
	|  action" above takes place				|
	+------------------------------------------------------*/
	if ( pcp->pc_opencnt ) {

#ifdef DEBUG
	    printf("npc_open: about to redo_locks, conv case\n");
#endif DEBUG
	    rerror = redo_locks(vp,pcp->pc_smbfid,fid,havestuff);
	
	    timex = time.tv_sec;
	    Greenwich_to_local(&timex);
	    rerror = doSMB1(buildSMBclose,
				(rerror)? HAVENOSOCK : HAVESOCK,
				pmp,havestuff,&sb,
			   	pcp->pc_opid,pcp->pc_smbfid,timex);
	    if ( rerror ) {
		printf("npc_open: doSMB close ordinary case %d\n",rerror);
		if ( pmp->m_sock->ps_state == PS_RECV ) {
		    printf("***** WARNING: NPC file failure to close *****\n");
		}
	    }
	}

	pcp->pc_smbfid = fid;
	pcp->pc_omode = conjunct;
	pcp->pc_opid = pidx;
    }

    /*-----------------------------------------------------+
    | do this always even if no SMB traffic -- unless that |
    | close up there had an error                          |
    +-----------------------------------------------------*/

    if ( rerror == 0 ) {
        pcp->pc_opencnt++;
        if ( flag & FEXCLO ) {
	    pcp->pc_xocnt++;
    	}
    }

out:
    if ( havestuff ) {
	CKIN_VSOCK(vp);
	CKIN_BUF(sb);
    }
    if ( havepath ) {
	kmem_free(cp,MAXPATHLEN );
    }

    return(npc_doerror(rerror, pmp));
}


int map_SMBopen_err(rcls,err)
unsigned char	rcls;
unsigned short	err;
{
    switch (rcls) {
        case ERRDOS :
	    switch (err) {
	        case ERRbadfile:
		    return ENOENT;
		case ERRnofids:
		    return EMFILE;
		case ERRnoaccess:
		    return EACCES;
		case ERRbadshare:
		    return EACCES;
		default:
		    return EIO;
	    }
	case ERRSRV:
	    switch (err) {
	        case ERRaccess:
		    return EACCES;
		default:
		    return EIO;
	    }
	case ERRHRD:
	    switch (err) {
	        case ERRbadshare:
		    return EACCES;
		default:
		    return EIO;
	    }
	case SUCCESS:
	    return 0;
	default:
	    return EIO;
    }		
}
	    

npc_close(vp, flag, cred)
struct vnode *vp;
int flag;
struct ucred *cred;
{
    struct sendbuf 		*sb;
    struct pcmountinfo 		*pmp = vftomi(vp->v_vfsp);
    struct pcnode  		*pcp = vtopc(vp);
    int             		fid,access,trash,error = 0;
    int				havestuff = 0;
    char           		*cp;
    unsigned char   		rcls;
    unsigned short  		err;
    time_t          		timex;

    CKHEALTH(pmp);
    
    /*---------------------------------------------------+
    |  check out the sock here to single-thread this code|
    +---------------------------------------------------*/
    if ( error = npc_ckout_sock(pmp->m_sock) ) {
	printf("npc_close: cannot checkout the socket %d\n",error);
	return error;
    }

    if (--(pcp->pc_opencnt) == 0) {
        if (vp->v_type == VREG) {
	    timex = time.tv_sec;
	    Greenwich_to_local(&timex);
	    /* may be necessary later to decide which pid to use */
	    if(error = doSMB1(buildSMBclose, HAVESOCK, pmp, HAVENOBUF, &sb,
			     pcp->pc_opid,vtopc(vp)->pc_smbfid, timex)) {
		printf("npc_close: failed doSMB, errno:%d\n", error);
		pcp->pc_opencnt++;
		goto out;
   	    }
	    havestuff = 1;
	    if ( flag & FEXCLO ) {
		pcp->pc_xocnt--;
		if ( u.u_procp->p_pid == pcp->pc_opid ) {
		    u.u_procp->p_xflag = PX_NPCUSE |
			( (u.u_procp->p_xflag & PX_EXCLO) - 1 ) ;
#ifdef DEBUG
		    printf("npc_close: pid %d xflag %x (all case)\n",
				u.u_procp->p_pid,u.u_procp->p_xflag);
#endif DEBUG
		}
	    }
 	    pcp->pc_omode = 0;	/* precautionary */
	}
    } else {

	if (flag & FEXCLO) {
	    if (--(pcp->pc_xocnt) == 0) {

		cp = kmem_alloc(MAXPATHLEN);
		npc_buildpath(vp, cp);

		/* must use phony pid so later exit wont wreck things */
		error = doSMB1(buildSMBopen, HAVESOCK, pmp, havestuff, &sb,
			      PHONY_PID,0x40,0,cp); /* readonly, deny nothing */
		kmem_free(cp, MAXPATHLEN);
		if (error) {
		    pcp->pc_opencnt++;	   /* close never took place */
		    pcp->pc_xocnt++;	   /* ditto */
		    goto out;
		}
		havestuff = 1;
		extractSMBopen(sb->buf, &rcls, &err, &fid,
			       &trash, &trash, &trash, &access);
		if (error = map_SMBopen_err(rcls, err)) {
		    pcp->pc_opencnt++;
		    pcp->pc_xocnt++;
		    goto out;
		}

#ifdef DEBUG
		printf("npc_close: about to redo locks\n");
#endif DEBUG
		error = redo_locks(vp,pcp->pc_smbfid,fid,havestuff);

		timex = time.tv_sec;
		Greenwich_to_local(&timex);
		/* real pid probably ok here, but watch out */
		error = doSMB1(buildSMBclose, 
				(error)? HAVENOSOCK : HAVESOCK, 
				pmp, havestuff, &sb,
			        pcp->pc_opid, pcp->pc_smbfid, timex);
		if (error) {
		    havestuff = 0;
		    pcp->pc_opencnt++;
		    pcp->pc_xocnt++;
		    printf("npc_close: doSMB close FEXCLO case %d\n", error);
		}else{
		    havestuff = 1;
		}
		/* go ahead and replace with the new open in case the pc
		   registered the close for us */
		pcp->pc_smbfid = fid;
		pcp->pc_omode = FREAD;
		/*------------------------------------------------------+
		|  before pc_opid gets reassigned, turn off EXCLO flag  |
		|  in proc structure.  this allows process to SMBexit.  |
		|  The flag may not be turned on in this pid if xocnt   |
		|  was greater than one at some point, and the original |
		|  opener, reflected in opid, closed early.  The right  |
		|  way to do this is to see if pid == opid, and if not, |
		|  search thru the proc table to match opid. if found,  |
		|  decrement exclo flag. if not found, cant npc_exit    |
		+------------------------------------------------------*/
		if ( u.u_procp->p_pid == pcp->pc_opid ) {
		    u.u_procp->p_xflag = PX_NPCUSE |
			( (u.u_procp->p_xflag & PX_EXCLO) - 1 );
#ifdef DEBUG
		    printf("npc_close: pid %d xflag %x (xo case)\n",
				u.u_procp->p_pid,u.u_procp->p_xflag);
#endif DEBUG
		}
		pcp->pc_opid = PHONY_PID;
  	    }
        }
    }
out:
    if (havestuff) {
	CKIN_BUF(sb);
    }
    CKIN_VSOCK(vp);

    return(npc_doerror(error, pmp));
}


npc_rdwr(vp, uiop, rw, ioflag, cred)
struct vnode 			*vp;
struct uio 			*uiop;
enum uio_rw 			rw;
int 				ioflag;
struct ucred 			*cred;

{
    struct sendbuf	    *sb;
    struct pcmountinfo	    *pmp;
    struct pcnode	    *pcp;
    int			    ii;
    int			    len;
    int			    error;
    unsigned short	    count,count2;
    unsigned char	    rcls,*cp;
    unsigned short	    err;
    int			    offset,total_cnt,lim;
    int			    havesock = 0;
    int			    havebuf = 0;

    
#define		WBOFF	    48
    pcp = vtopc(vp);
    pmp = vftomi(vp->v_vfsp);
    
    CKHEALTH(pmp);
    
    lim = MIN( SENDBUFSIZ,pmp->m_maxpkt);
    if ( vp->v_type == VREG ) {

	if ( rw == UIO_READ ) {
#ifdef DEBUG
/* printf("npc_rdwr: fresh read call, maxpkt %x\n",pmp->m_maxpkt); */
#endif DEBUG
	    while ( uiop->uio_resid ) {
#ifdef DEBUG
/* printf("npc_rdwr: read uiop->uio_resid %d\n",uiop->uio_resid); */
#endif DEBUG
		if(error = doSMB(buildSMBread, havesock, pmp, havebuf, &sb,
			      pcp->pc_smbfid,
			      ii = MIN(uiop->uio_resid,
				       /* pmp->m_maxpkt - MIN_SMBPKT - 13),*/
				       lim - MIN_SMBPKT - 13),
				       uiop->uio_offset,
				       uiop->uio_resid)) {
#ifdef DEBUG
		    printf("npc_rdwr, failed doSMB, errno:%d\n", error);
#endif
		    goto out;
		}
		havesock = havebuf = 1;
		extractSMBread(sb->buf,&rcls,&err,&count,&cp);
#ifdef DEBUG
/* hex(sb->buf,63); */
printf("npc_rdwr: SMBread rcls %d err %d\n",rcls,err);
#endif DEBUG
		if ( rcls != SUCCESS ) {
		    CKIN_BUF(sb);
		    CKIN_VSOCK(vp);
		    error = EACCES;
		    goto out;
		}
#ifdef DEBUG
/* printf("rdwr/read: about to uiomove %d bytes\n",count); */
#endif DEBUG
		uiomove(cp,count,UIO_READ,uiop);  /*  fix first arg */
		if ( count < ii ) {
		    CKIN_BUF(sb);
		    CKIN_VSOCK(vp);
		    return 0;
		}
	    }
	    if(havebuf)CKIN_BUF(sb);
	    if(havesock)CKIN_VSOCK(vp);
	    return 0;
	}else{    /* rw is UIO_WRITE */
	    if ( ioflag & IO_APPEND ) {
		if(error = doSMB(buildSMBseek, havesock, pmp, havebuf, &sb,
				 pcp->pc_smbfid,2,0)) {
#ifdef DEBUG
		    printf("npc_rdwr, failed doSMB, errno:%d\n", error);
#endif
		    goto out;
		}
		havesock = havebuf = 1;
		extractSMBseek(sb->buf,&rcls,&err,&offset);
#ifdef DEBUG
printf("npc_rdwr: SMBseek rcls %d err %d offset %d\n",rcls,err,offset);
#endif DEBUG
		if ( rcls != SUCCESS ) {
		    CKIN_BUF(sb);
		    CKIN_VSOCK(vp);
		    error = EACCES;
		    goto out;
		}
		uiop->uio_offset = offset;
	    }
	    if(!havebuf) {
		if(error = ckout_buf(&sb)) {
		    if(havesock) CKIN_VSOCK(vp);
		    goto out;
		}
		havebuf++;
	    }
	    if(setjmp(&u.u_qsave)) {
#ifdef DEBUG
		printf("did longjmp in npc_rdwr\n");
#endif DEBUG
		CKIN_BUF(sb);
		if(havesock)CKIN_VSOCK(vp);
		error = EINTR;
		goto out;
	    }
	    if(!havesock) {
		if(error = npc_ckout_sock(vftomi(vp->v_vfsp)->m_sock)) {
		    CKIN_BUF(sb);
		    goto out;
		}
		havesock++;
	    }
	    total_cnt = uiop->uio_resid;
	    while ( uiop->uio_resid ) {
		len = buildSMBwrite(sb->buf,
				  pmp->m_tid,
				  PSEUDO_PID,
				  pcp->pc_smbfid,
				  count = 
			 	   MIN( uiop->uio_resid, lim - MIN_SMBPKT - 13),
				  uiop->uio_offset,
				  uiop->uio_resid,
				  (char*)0);
		
		uiomove(sb->buf+WBOFF,count,UIO_WRITE,uiop);  
		if(error = nb_send(pmp->m_sock,sb->buf,len)) {
		    CKIN_BUF(sb);
		    CKIN_VSOCK(vp);
		    goto out;
		}
		if ( error = nb_recv(pmp->m_sock,sb) ) {
		    CKIN_BUF(sb);
		    CKIN_VSOCK(vp);
		    goto out;
		}
		extractSMBwrite(sb->buf,&rcls,&err,&count2);
#ifdef DEBUG
printf("npc_rdwr: SMBwrite rcls %d err %d\n",rcls,err);
#endif DEBUG
		if ( rcls != SUCCESS ) {
		    CKIN_BUF(sb);
		    CKIN_VSOCK(vp);
#define	Errnoresource		89
		    if ( rcls && (err == Errnoresource) ) {
			uiop->uio_resid += count;
			if ( uiop->uio_resid == total_cnt ) {
			    return ENOSPC;
			}else{
			    return 0;
			}
		    }
		    error = EACCES;
		    goto out;
		}
		/*--------------------------------------------------------+
		|  SMB literature says if SMBwrite ever says less written |
		|  than asked, then pc disk is full.  We will always tell |
		|  user the amount they have written, if that amount is   |
		|  not zero; if zero, we return ENOSPC			  |
		|  We want resid to reflect actual amount written.  Do we |
		|  also want to correct uio_offset too?			  |
		+--------------------------------------------------------*/
		if ( count2 != count ) {
		    uiop->uio_resid += count - count2;
		    CKIN_BUF(sb);
		    CKIN_VSOCK(vp);
		    if ( uiop->uio_resid == total_cnt ) {
			return ENOSPC;
		    }
		    return 0;
		}
	    }
	    CKIN_BUF(sb);
	    CKIN_VSOCK(vp);
	    return 0;
	}
    }else if ( vp->v_type == VDIR ) {
#ifdef DEBUG
printf("npc_rdwr: thinks vp is VDIR\n");
#endif DEBUG
	if ( rw == UIO_READ ) {
	    error = npc_readdir_int(vp,uiop,cred);
	    goto out;
	}else{
	    error = EISDIR;
	    goto out;
	}
    }
    error = EIO;	/* wonder how we got here */
out:
    return(npc_doerror(error, pmp));
#undef	    WBOFF
}

npc_ioctl(vp, com, data, flag, cred)
	struct vnode *vp;
	int com;
	caddr_t data;
	int flag;
	struct ucred *cred;
{
	return(EOPNOTSUPP);
}


npc_select(vp, which, cred)
	struct vnode *vp;
	int which;
	struct ucred *cred;
{
	CKHEALTH(vftomi(vp->v_vfsp));
	
	return(1);
}


npc_getattr(vp, vap, cred)
	struct vnode *vp;
	register struct vattr *vap;
	struct ucred *cred;
{
    struct pcmountinfo	*pmp = vftomi(vp->v_vfsp);
    struct sendbuf	*sb;
    char		*cp;
    unsigned char	*dirdata;
    unsigned int	ltime,fsize;
    int			error;
    unsigned char	rcls;
    unsigned short	err,attr,count;
    unsigned short	bsize,trash,timefields, datefields;
    unsigned long	convert_time_and_date();

    CKHEALTH(pmp);
    
    vap->va_uid = -5;  /* totally arbitrary */
    vap->va_gid = -5;  /* ditto */
    vap->va_fsid = vp->v_vfsp->vfs_fsid.val[1];
    vap->va_nodeid = vtopc(vp)->pc_nodeid;
    vap->va_nlink = 1;
    vap->va_rdev = -1;

    if ( pmp->m_blksize ) {
        vap->va_blocksize = pmp->m_blksize;
    }else{
        if(error = doSMB(buildSMBdskattr, HAVENOSOCK,pmp,HAVENOBUF,&sb)) {
	    goto out;
        }   
        extractSMBdskattr(sb->buf,&rcls,&err,&trash,&trash,&bsize,&trash);
        CKIN_BUF(sb);
        CKIN_VSOCK(vp);
        if( rcls != SUCCESS ) {
#ifdef DEBUG
printf("npc_getattr: SMBdskattr rcls %d err %d\n",rcls,err);
#endif DEBUG
	    error = EIO;
	    goto out;
        }
    
	if ( bsize ) {
            vap->va_blocksize = pmp->m_blksize = bsize;
	}else{
	    vap->va_blocksize = pmp->m_blksize = 512;	/* default */
#ifdef DEBUG
	    printf("npc_getattr: SMBdskattr gave blksiz of zero\n");
#endif DEBUG
	}
    }
    
    if (  (vap->va_type = vp->v_type) == VDIR ) {


	vap->va_mode = S_IFDIR | 0777;

	cp = kmem_alloc(MAXPATHLEN);
	npc_buildpath(vp,cp);
#ifdef notdef
printf("npc_getattr: path for SMBsearch ==%s==\n",cp);
#endif DEBUG
	error = doSMB(buildSMBsearch,HAVENOSOCK,pmp,HAVENOBUF,&sb,
		      1,FA_DIR,cp,0,(char*)0);
	kmem_free(cp,MAXPATHLEN);
	if ( error ) {
	    goto out;
	}
	CKIN_VSOCK(vp);
	extractSMBsearch(sb->buf,&rcls,&err,&count,&dirdata);
	CKIN_BUF(sb);
	if ( rcls != SUCCESS ) {
#ifdef DEBUG
printf("npc_getattr: SMBsearch rcls %d err %d count %d\n",rcls,err,count);
#endif DEBUG
	    error = EIO;
	    goto out;
	}
	if ( count != 1 ) {

	    vap->va_size = 0;
	    vap->va_blocks = 0;

	    vap->va_ctime.tv_sec = 
	    vap->va_atime.tv_sec = 
	    vap->va_mtime.tv_sec = time.tv_sec;  /* this may not be sound */
	    
        } else {

	    vap->va_size = dirdata[DIR_INFO_SIZE_H +1] << 24;
	    vap->va_size |= dirdata[DIR_INFO_SIZE_H] << 16;
	    vap->va_size |= dirdata[DIR_INFO_SIZE_L +1] << 8;
	    vap->va_size |= dirdata[DIR_INFO_SIZE_L];


	    if ( pmp->m_blksize ) {
	        vap->va_blocks = ( vap->va_size % pmp->m_blksize )
			? vap->va_size / pmp->m_blksize + 1
			: vap->va_size / pmp->m_blksize;
	    }else{
		vap->va_blocks = -1;
#ifdef DEBUG
	    	printf("npc_getattr: pmp->m_blksize is zero\n");
#endif DEBUG
	    }

	    timefields = dirdata[DIR_INFO_TIME +1] << 8;
	    timefields |= dirdata[DIR_INFO_TIME];
	    datefields = dirdata[DIR_INFO_DATE +1] << 8;
	    datefields |= dirdata[DIR_INFO_DATE];

	    vap->va_ctime.tv_sec =
	    vap->va_atime.tv_sec =
	    vap->va_mtime.tv_sec = convert_time_and_date( timefields,
							  datefields );
	}
    } else {                        /*  VREG */

	cp = kmem_alloc(MAXPATHLEN);
	npc_buildpath(vp,cp);
#ifdef NOTDEF
printf("npc_getattr: SMBgetatr path ==%s==\n",cp);
#endif NOTDEF
	error = doSMB(buildSMBgetatr, HAVENOSOCK, pmp, HAVENOBUF, &sb, cp);
	kmem_free(cp, MAXPATHLEN);
	if(error) {
	    goto out;
	}
	CKIN_VSOCK(vp);
	extractSMBgetatr(sb->buf,&rcls,&err,&attr,&ltime,&fsize);
	CKIN_BUF(sb);
	if ( rcls != SUCCESS ){
#ifdef DEBUG
	    printf("npc_getattr: SMBgetatr rcls %d err %d\n",rcls,err);
#endif DEBUG
	    error = EIO;
	    goto out;
	}

	vap->va_mode = (attr & 0x0001) ? 0444 : 0666;
        vap->va_mode |= S_IFREG;

	vap->va_size = fsize;

	if ( pmp->m_blksize ) {
            vap->va_blocks = ( vap->va_size % pmp->m_blksize )
			? vap->va_size / pmp->m_blksize + 1
			: vap->va_size / pmp->m_blksize;
	} else {
	    vap->va_blocks = -1;	/* might want some save default */
#ifdef DEBUG
	    printf("npc_getattr: pmp->m_blocksize is zero\n");
#endif DEBUG
	}

	local_to_Greenwich(&ltime);
	vap->va_atime.tv_sec = vap->va_mtime.tv_sec = 
			       vap->va_ctime.tv_sec = ltime;
    }
    return 0;
out:
    return(npc_doerror(error, pmp));
}


npc_setattr(vp,vap,cred)
struct vnode		*vp;
struct vattr		*vap;
struct ucred		*cred;

{
    struct sendbuf	*sb;
    struct pcnode	*pcp = vtopc(vp);
    struct pcmountinfo	*pmp = vftomi(vp->v_vfsp);
    unsigned char	rcls;
    char		*cp;
    unsigned short	err,attr,fid,access;
    unsigned int	offset,time1,size;
    int			closeflag;
    time_t		timex;
    int			serror = 0;
    int			havestuff = 0;

    CKHEALTH(pmp);
    
    if (( vap->va_type != -1 ) ||
	( vap->va_uid != -1 ) ||
	( vap->va_gid != -1 ) ||
	( vap->va_fsid != -1 ) ||
	( vap->va_nodeid != -1 ) ||
	( vap->va_nlink != -1 ) ||
	( vap->va_blocksize != -1 ) ||
	( vap->va_ctime.tv_sec != -1 ) ||
	( vap->va_rdev != -1 ) ||
	( vap->va_blocks != -1 )    ) {
	serror = EINVAL;
	goto out;
    }

    if ( vp->v_type == VDIR ) {
	if (( vap->va_mode != 0xffff ) ||
	    ( vap->va_size != 0xffffffff ) ||
	    ( vap->va_atime.tv_sec != -1 ) ||
	    ( vap->va_mtime.tv_sec != -1 )    ) {
	    serror = EINVAL;
	    goto out;
	}
    }

    /*-------------------------------------------------------------+
    |  the Unix syscalls chmod, chown, truncate, utimes, and their |
    |  variants all come to this routine.  We aren't accepting     |
    |  chown.  Although the calls currently set JUST their target  |
    |  attrs, it is best to assume in this routine that several    |
    |  attributes might be set simultaneously.  Chmod and utimes   |
    |  both generate SMBsetatr.  If the call is going out for one  |
    |  of these and not the other, the SMB can be rigged not to    |
    |  change the time but cannot be rigged to not change the attr.|
    |  So to keep from changing the attr, we do SMBgetatr and use  |
    |  the attr from that.                                         |
    +------------------------------------------------------------ */

    if ( ( vap->va_mode != 0xffff ) || ( vap->va_atime.tv_sec != -1 ) ||
	 ( vap->va_mtime.tv_sec != -1 ) ) {

	cp = kmem_alloc(MAXPATHLEN);
	npc_buildpath(vp,cp);

	if ( vap->va_mode == 0xffff ) {
	    serror = doSMB(buildSMBgetatr, havestuff, pmp, havestuff, &sb, cp);
	    if (serror) {
		kmem_free(cp,MAXPATHLEN);
		goto out;
	    }
	    havestuff++;
	    extractSMBgetatr(sb->buf,&rcls,&err,&attr,&time1,&size);
	    if ( rcls != SUCCESS ) {
#ifdef DEBUG
printf("npc_setattr: SMBgetatr rcls %d err %d attr %x\n",rcls,err,attr);
#endif DEBUG
		serror = EACCES;
clean1:		CKIN_BUF(sb);
		CKIN_VSOCK(vp);
		kmem_free(cp,MAXPATHLEN);
		goto out;
	    }
	} else {
	    switch ( vap->va_mode & 0200 ) {
	        case 0:
		    attr = FA_RO;
		    break;
		default:
		    attr = 0;
	    }
	}

	time1 = 0;
	if ( vap->va_atime.tv_sec != -1 ) {
	    time1 = vap->va_atime.tv_sec;
	    Greenwich_to_local(&time1);
#ifdef DEBUG
printf("npc_setattr: va_atime %x\n",time1);
#endif DEBUG
	}
	if ( vap->va_mtime.tv_sec != -1 ) {
	    time1 = vap->va_mtime.tv_sec;
	    Greenwich_to_local(&time1);
#ifdef DEBUG
printf("npc_setattr: va_mtime %x\n",time1);
#endif DEBUG
	}

	serror = doSMB(buildSMBsetatr, havestuff, pmp, havestuff, &sb,
		      attr, time1, cp);
	kmem_free(cp,MAXPATHLEN);
	if (serror) {
	    goto out;
	}
	extractSMBsetatr(sb->buf,&rcls,&err);
	CKIN_VSOCK(vp);
	CKIN_BUF(sb);
	havestuff = 0;
	switch (rcls) {
	    case SUCCESS: break;
	    case ERRDOS:
#ifdef DEBUG
printf("npc_setattr: SMBsetatr rcls %d err %d\n",rcls,err);
#endif DEBUG
		switch ( err ) {
		    case ERRbadfunc:
			serror = EINVAL;
			goto out;
		    case ERRbadpath:
			serror = ENOENT;
			goto out;
		    case ERRnoaccess:
		    default:
			serror = EACCES;
			goto out;
		}
		break;
	    default:
#ifdef DEBUG
printf("npc_setattr: SMBsetatr rcls %d err %d\n",rcls,err);
#endif DEBUG
		serror = EIO;
		goto out;
	}
    }

    /*----------------------------------------------------------+
    |  The truncate syscall is expressed t/ va_size; an SMBwrite|
    |  with offset of the requested size and length of zero will|
    |  do that.  We will open the file first; dont forget close |
    +--------------------------------------------------------- */

    if ( vap->va_size != 0xffffffff ) {

        cp = kmem_alloc(MAXPATHLEN);
        npc_buildpath(vp,cp);

        serror = doSMB(buildSMBopen,havestuff,pmp,havestuff,&sb,0x41,0,cp);

        kmem_free(cp,MAXPATHLEN);
        if (serror ) {
            goto out;
        }
        havestuff++;
        extractSMBopen(sb->buf,&rcls,&err,&fid,&attr,&time1,&size,&access);
        if ( rcls != SUCCESS ) {
#ifdef DEBUG
printf("npc_setattr: SMBopen rcls %d err %d\n",rcls,err);
#endif DEBUG
	    CKIN_VSOCK(vp);
	    CKIN_BUF(sb);
	    serror = EACCES;
	    goto out;
        }

	/*-----------------------------------------------------------+
	|  don't do the write unless requested size is smaller than  |
	|  the current file size.  Note this strategy has a racey    |
	|  problem.           					     |
	+-----------------------------------------------------------*/
	serror = doSMB(buildSMBseek,havestuff,pmp,havestuff,&sb,fid,2,0);
	if ( serror ) {
	    goto getout;
	}
	extractSMBseek(sb->buf,&rcls,&err,&offset);
	if ( rcls != SUCCESS ) {
#ifdef DEBUG
printf("npc_setattr: SMBseek rcls %d err %d offset %d\n",rcls,err,offset);
#endif DEBUG
	    CKIN_BUF(sb);
	    CKIN_VSOCK(vp);
	    serror = EACCES;
	    goto getout;
	}

	if ( vap->va_size < offset ) {

	    serror = doSMB(buildSMBwrite, havestuff, pmp, havestuff, &sb, 
		      fid,0,vap->va_size,0,(char*)0 );
	    if( serror ) {
	        goto getout;
	    }
	
	    extractSMBwrite(sb->buf,&rcls,&err,&size);
	    if ( rcls != SUCCESS ) {
#ifdef DEBUG
printf("npc_setattr: SMBwrite rcls %d err %d count %d\n",rcls,err,size);
#endif DEBUG
	        serror = EACCES;
	    }
	}

getout:
	/* make the timestamp go th Greenwich_to_local */
	timex = time.tv_sec;
	Greenwich_to_local(&timex);
		    
	serror = doSMB(buildSMBclose,havestuff,pmp,havestuff,
			   &sb,fid,timex);
    }

    if(havestuff) {
	CKIN_VSOCK(vp);
	CKIN_BUF(sb);
    }
out:
    return(npc_doerror(serror, pmp));
}


int
npc_access(vp, mode, cred)
struct vnode	*vp;
int		mode;
struct ucred	*cred;
{
    struct pcmountinfo	*mi;
    struct pcaccess	*acp;
    int			error;

    mi = vftomi(vp->v_vfsp);

    CKHEALTH(mi);
    
#ifdef NOTDEF
    if((mi->m_flags & NPCMNT_ACTIVE) == 0) {
	    /* socket has been closed */
	    error = EACESS;
	    goto out;
    }
#endif NOTDEF
    if(mi->m_flags & NPCMNT_BADFS) {
#ifdef DEBUG
printf("npc_access: EIO because bad fs\n");
#endif
	error = EIO;
	goto out;
    }
    
    if((mode & FWRITE) && (vp->v_vfsp->vfs_flag & VFS_RDONLY)) {
#ifdef DEBUG
printf("npc_access: EACCES because readonly fs\n");
#endif
	error = EACCES;
	goto out;
    }
    
    if (mi->m_passwd[0] == '\0') {
	/* this is the easy one, it is public */
#ifdef DEBUG
/* printf("npc_access: return ok becasue no passwd\n"); */
#endif
	return(0);
    } else {
	/* look for uid that matches */
	for (acp=mi->m_alist; acp != NULL; acp = acp->pa_next) {
	    if(acp->pa_uid == u.u_cred->cr_uid) {
		/* ok -- pay dirt */
#ifdef DEBUG
printf("npc_access: return ok becasue found uid\n");
#endif
		return(0);
	    }
	}
    }
#ifdef DEBUG
printf("npc_access: return EACCES by default\n");
#endif
    error = EACCES;
out:
    return(npc_doerror(error, mi));
}


npc_lookup(dvp, nm, vpp, cred)
struct vnode	*dvp;
char		*nm;
struct vnode	**vpp;
struct ucred	*cred;
{
    int	error = 0;

    CKHEALTH(vftomi(dvp->v_vfsp));
    
    if(dvp->v_flag & VROOT) {
	/* TODO check VEXEC and cred */
	if(error = npc_access(dvp, VEXEC, cred)) {
	    goto out;
	}
    }
    if(strcmp(nm, "..") == 0) {
	/* vfs_lookup prevents this from being VROOT */
	*vpp = vtopc(dvp)->pc_parent;
    } else if(strcmp(nm, ".") == 0) {
	*vpp = dvp;
    } else {
	*vpp = npc_dirlookup(dvp, nm);
    }

    if (*vpp) {
	VN_HOLD(*vpp);
    } else {
	/*
	 * nm is either not there or not in cache for some reason,
	 * go to pc and make sure
	 */
#ifdef DEBUG
	printf("npc_lookup: failed dirlookup, do findpccomp, nm:%s:",nm);
	printf(" dvp->name:%s:\n",vtopc(dvp)->pc_name);
#endif DEBUG
	*vpp = npc_findpccomp(dvp,nm);
	if ( *vpp == (struct vnode *)0 ) {
	    if(u.u_error) {
		error = u.u_error;
	    } else {
		error = ENOENT;
	    }
	}
    }
out:
    return(npc_doerror(error, vftomi(dvp->v_vfsp)));
}


npc_create(dvp,nm,vap,exclusive,mode,vpp,cred)
struct vnode		*dvp;
char			*nm;
struct vattr		*vap;
enum vcexcl		exclusive;
int			mode;
struct vnode		**vpp;
struct ucred		*cred;

{
    struct sendbuf	    *sb;
    unsigned short	    attr,err,fid;
    unsigned char	    *cp,*cx,rcls;
    int			    i,trash,fsize;
    struct pcmountinfo	    *pmp = vftomi(dvp->v_vfsp);
    struct pcnode	    *pcp;
    time_t		    timex;
    int			    error;

    CKHEALTH(pmp);
    
    if ( vap->va_type != VREG ) {
#ifdef DEBUG
printf("npc_create: EPERM because vap->va_type isn't VREG %o\n",vap->va_type);
#endif DEBUG
	error = EPERM;
	goto out;
    }
    if(npc_checkdosname(nm) < 0) {
	error = EINVAL;
	goto out;
    }

    *vpp = (struct vnode *)0;
    
    error = npc_lookup(dvp, nm, vpp, cred);

    if( error == 0 ) {
	if ( exclusive == EXCL ) {
	    VN_RELE( *vpp )
	    error = EEXIST;
	    goto out;
  	}
    } else if ( error != ENOENT ) {
	goto out;
    }
    
    /*---------------------------------------------------------+
    |  case one -- vpp already points at an active vnode, so we|
    |  determine we already have an existing file.  The only   |
    |  reason to do anything is that FTRUNC was specified, re- |
    |  quiring us to zero the file using SMBcreate.  Remember  |
    |  to close it if the create succeeds                      |
    +-------------------------------------------------------- */
    /* !!!!!!!!!!!! I think a hold needs to be put on vnode
     * !!!!!!!!!!!! I am checking with chris */
    if ( *vpp ) {
	if ( vap->va_size == 0 ) {
	    cp = (unsigned char *)kmem_alloc(MAXPATHLEN);
	    cx = (unsigned char *)npc_buildpath( *vpp, cp );

	    /*-------------------------------------------------+
            | The Unix manual specifies that vm_mode is not to |
	    | be used in this case (stays same).  So we have to|
	    | do getatr and use the current attr in the create |
	    +-------------------------------------------------*/
	    error = doSMB(buildSMBgetatr,HAVENOSOCK,pmp,HAVENOBUF,&sb,cp);
	    if ( error ) {
		VN_RELE(*vpp);
		kmem_free(cp,MAXPATHLEN);
		goto out;
	    }
	    extractSMBgetatr(sb->buf,&rcls,&err,&attr,&trash,&fsize);
#ifdef DEBUG
printf("npc_create: SMBgetatr on path %s rcls %d err %d\n",cp,rcls,err);
#endif DEBUG
	    if ( rcls != SUCCESS ) {
		VN_RELE(*vpp);
		CKIN_VSOCK(dvp);
		CKIN_BUF(sb);
		kmem_free(cp,MAXPATHLEN);
		error = EACCES;
		goto out;
	    }

	    if ( fsize == 0 ) {      /* no need to trunc */
		CKIN_VSOCK(dvp);
		CKIN_BUF(sb);
		kmem_free(cp,MAXPATHLEN);
		return 0;
	    }
	    
	    error = doSMB(buildSMBcreate,HAVESOCK,pmp,HAVEBUF,&sb,attr,0,cp);
	    kmem_free(cp, MAXPATHLEN);
	    if (error) {
		VN_RELE(*vpp);
		goto out;
	    }
	    extractSMBcreate(sb->buf,&rcls,&err,&fid);
	    if ( rcls == SUCCESS ) {
#ifdef DEBUG
printf("npc_create: SMBcreate to truncate succeeds\n");
#endif DEBUG
		/* make the time stamp go th Greenwich_to_local */
	 	timex = time.tv_sec;
		Greenwich_to_local(&timex);
		if(error = doSMB(buildSMBclose, HAVESOCK, pmp, HAVEBUF,
					&sb,fid, timex)) {
		    printf("npc_create: failed doSMB, errno:%d\n", error);
		} else {
		    CKIN_VSOCK(dvp);
		    CKIN_BUF(sb);
		}
	    } else {
#ifdef DEBUG
printf("npc_create: SMBcreate to truncate fails, rcls %d err %d\n",rcls,err);
#endif DEBUG
		CKIN_VSOCK(dvp);
		CKIN_BUF(sb);
		error = EACCES;
	    }
	    if(error) {
		VN_RELE(*vpp);
	    }
#ifdef DEBUG
	    printf("end of create, vp->count:%x\n", (*vpp)->v_count);
#endif
	} else {        /* va_size isnot zero */
#ifdef DEBUG
	    printf("va_size is not zero\n");
#endif
	    error = 0;
	}
	/* so if success, hold has been put on vnode */
	goto out;
    } else {
	/*--------------------------------------------------------+
	|  case two -- no vpp, therefore MAYBE no already existing|
	|  file.  This time we're going to do SMBmknew, which fails
	|  ERRfilexists if the file already exists.  That's no big|
	|  deal to us unless exclusive is set.  If mknew succeeds |
	|  we'll want to close it, and then allocate a vnode and  |
	|  fill it in                                             |
	+------------------------------------------------------- */

	cp = (unsigned char *)kmem_alloc(MAXPATHLEN );
	cx = (unsigned char *)npc_buildpath(dvp,cp);
	*cx++ = 0x5c;
	bcopy(nm,cx,strlen(nm)+1);
#ifdef DEBUG
printf("npc_create: SMBmknew path ==%s==\n",cp);
#endif DEBUG
	/* use phony pid in case remains open.  match with close below */
	error = doSMB1(buildSMBmknew, HAVENOSOCK, pmp, HAVENOBUF, &sb,
	   		PHONY_PID,(vap->va_mode&0x0080) ? 0 : FA_RO, 0, cp);
	kmem_free(cp, MAXPATHLEN);
	if(error) {
	    goto out;
	}
	extractSMBmknew(sb->buf,&rcls,&err,&fid);
	if ( rcls == SUCCESS ) {
#ifdef DEBUG
printf("npc_create: mknew succeeds\n");
#endif DEBUG
	    /*----------------------------------------------------+
	    |  If the file was just created RO, we must install   |
	    |  this fid in the pcnode, NOT close the file, and    |
	    |  leave opencnt zero as a sign to npc_open not to do |
	    |  any open.  This probably will work ok.		  |
	    +----------------------------------------------------*/
	    if ( vap->va_mode & 0x0080 ) {
	        /* make the time stamp go th Greenwich_to_local */
	        timex = time.tv_sec;
	        Greenwich_to_local(&timex);
	        if(error = doSMB1(buildSMBclose, HAVESOCK, pmp, HAVEBUF, &sb,
			     PHONY_PID,fid, timex)) {
#ifdef DEBUG
		    printf("npc_create: fail doSMB2, errno:%d\n", error);
#endif
		    goto out;
	        }
	    }
	    CKIN_VSOCK(dvp);
	    CKIN_BUF(sb);
makevn:
	    *vpp = npc_getvnode(dvp->v_vfsp);
	    (*vpp)->v_type = VREG;
	    pcp = vtopc(*vpp);
	    pcp->pc_parent = dvp;
	    VN_HOLD(dvp);
	    bcopy(nm,pcp->pc_name,MAXDOSNAME+1);
	    /*--------------------------------------------------+
	    |  again, if the file was created readonly, install |
	    |  fid in pcnode, and set omode to readwrite.  Open |
	    |  question: what about FEXCLO?			|
	    +--------------------------------------------------*/
	    if ( !(vap->va_mode & 0x0080) ) {
		pcp->pc_smbfid = fid;
		pcp->pc_omode = FREAD ;
		pcp->pc_opid = PHONY_PID;
	    }
	    npc_enterinhash(*vpp);
	    return 0;
	} else {
#ifdef DEBUG
printf("npc_create: mknew fails, rcls %d err %d\n",rcls,err);
#endif DEBUG
	    CKIN_VSOCK(dvp);
	    CKIN_BUF(sb);
	    switch (rcls) {
	    case ERRDOS:
		switch (err) {
		case ERRfilexists:
		    if ( exclusive == EXCL ){
#ifdef DEBUG
printf("npc_create: mknew fails on EXCL and we do EEXIST\n");
#endif DEBUG
			error = EEXIST;
		    }else{
#ifdef DEBUG
printf("npc_create: mknew fails on ERRfilexists, but we aren't EXCL\n");
#endif DEBUG
			goto makevn;
		    }
		    break;
		case ERRbadpath:
		    error = ENOTDIR;
		    break;
		case ERRnofids:
		    error = EMFILE;
		    break;
		case ERRnoaccess:
		default:
		    error = EACCES;
		    break;
		}
		break;
	    default:
		error = EACCES;
	    }/* end switch */
	    goto out;
	}
    }
out:
    return(npc_doerror(error, pmp));
}


npc_remove(dvp,nm,cred)
struct vnode	*dvp;
char		*nm;
struct ucred	*cred;

{
    struct pcmountinfo	    *pmp = vftomi(dvp->v_vfsp);
    struct sendbuf	    *sb;
    int			    error;
    unsigned char	    *cp,rcls;
    unsigned short	    err;
    struct vnode	    *vp;

    CKHEALTH(pmp);
    
    error = npc_lookup(dvp, nm, &vp, cred);
    if(error) {
#ifdef DEBUG
	printf("npc_remove: could not find file, error:%d\n", error);
#endif DEBUG
	goto out;
    }
    if(vtopc(vp)->pc_opencnt) {
#ifdef DEBUG
	printf("npc_remove: EACCES because opencnt not zero\n");
#endif DEBUG
	VN_RELE(vp);
	error = EACCES;
	goto out;
    }

#ifdef NOTDEF  /*--------------------------------------------------*/
    vp = npc_findv(dvp,nm);	
    if ( vp ) {
	if ( vtopc(vp)->pc_opencnt ) {
	    error = EACCES;
	    goto out;
	}
    }
#endif NOTDEF /*---------------------------------------------------*/

    cp = (unsigned char *)kmem_alloc(MAXPATHLEN);
    npc_buildpath(vp,cp);

#ifdef DEBUG
printf("npc_remove: SMBunlink path ==%s==\n",cp);
#endif DEBUG
    error = doSMB(buildSMBunlink, HAVENOSOCK, pmp, HAVENOBUF, &sb, 0, cp);
    kmem_free(cp, MAXPATHLEN);
    if(error) {
	VN_RELE(vp);
	goto out;
    }
    extractSMBunlink(sb->buf,&rcls,&err);
    CKIN_VSOCK(vp);
    CKIN_BUF(sb);
    VN_RELE(vp);

#ifdef DEBUG
printf("npc_remove: SMBunlink rcls %d err %d\n",rcls,err);
#endif DEBUG
    if ( rcls == SUCCESS ) {
	return 0;
    }else{
	error = EACCES;	    /*  this might be fleshed out if needed */
out:
	return(npc_doerror(error, pmp));
    }
}


npc_link(vp,tdvp,tnm,cred)
struct	vnode	*vp;
struct	vnode	*tdvp;
char		*tnm;
struct  ucred	*cred;

{
    /*-------------------------------------------------------------+
    |  It is legitimate to get called, but it is not legitimate to |
    |  try to do anything.                                         |
    +------------------------------------------------------------ */

    return ( EINVAL );
}


npc_rename(sdvp,snm,tdvp,tnm,cred)
struct vnode	    *sdvp;	    /* parent dir for source "from" */
char		    *snm;	    /* name of "from" file */
struct vnode	    *tdvp;	    /* parent dir for target "to" */
char		    *tnm;	    /* name of target "to" file */
struct ucred	    *cred;

{
    struct pcmountinfo	    *tpmp = vftomi(tdvp->v_vfsp);
    unsigned short	    err;
    unsigned char	    rcls,*cp1,*cx,*cp2;
    int			    error;
    struct vnode	    *vp;
    struct sendbuf	    *sb;

    CKHEALTH(tpmp);

    if(npc_checkdosname(snm) < 0 || npc_checkdosname(tnm) < 0) {
	error = EINVAL;
	goto out;
    }
    cp1 = (unsigned char *)kmem_alloc(MAXPATHLEN);
    cx = (unsigned char *)npc_buildpath(sdvp,cp1);
    *cx++ = 0x5c;
    bcopy(snm,cx,strlen(snm)+1);

    cp2 = (unsigned char *)kmem_alloc(MAXPATHLEN);
    cx = (unsigned char *)npc_buildpath(tdvp,cp2);
    *cx++ = 0x5c;
    bcopy(tnm,cx,strlen(tnm)+1);

#ifdef DEBUG
printf("npc_rename: SMBmv spath ==%s== tpath ==%s==\n,",cp1,cp2);
#endif DEBUG
    error = doSMB(buildSMBmv, HAVENOSOCK, tpmp,HAVENOBUF, &sb, 0, cp1, cp2);
    kmem_free(cp1,MAXPATHLEN);
    kmem_free(cp2,MAXPATHLEN);
    if ( error ) {
	goto out;
    }
    CKIN_VSOCK(tdvp);
    extractSMBmv(sb->buf,&rcls,&err);
    CKIN_BUF(sb);
#ifdef DEBUG
printf("npc_rename: SMBmv rcls %d err %d\n",rcls,err);
#endif DEBUG
    switch ( rcls ) {
	case ERRDOS:
	    switch ( err ) {
	    case ERRbadfile:  error = ENOENT; break;
	    case ERRbadshare: error = EBUSY; break;
	    case ERRfilexists: error = EEXIST; break;
	    default:  error = EACCES; break;
	    }
	    break;
	case ERRSRV:  error = EACCES; break;
	case ERRHRD:
	    switch ( err ) {
		case ERRbadshare: error = EBUSY; break;
		default: error = EIO; break;
	    }
	    break;
	case SUCCESS:
	    vp = npc_findv(sdvp,snm);
	    if ( vp ) {                /* it was found in hash */
		npc_removefromhash(vp);
		bcopy(tnm,vtopc(vp)->pc_name,strlen(tnm)+1);
		npc_enterinhash(vp);
	    }
	    /* else case: wasnot in hash, but no big deal */
	    error = 0;
	    break;
	default:  error = EIO; break;
    }
out:
    return(npc_doerror(error, tpmp));
}

	    
npc_mkdir(dvp,nm,vap,vpp,cred)
struct vnode	    *dvp;
char		    *nm;
struct vattr	    *vap;
struct vnode	    **vpp;
struct ucred	    *cred;

{
    struct sendbuf	*sb;
    struct pcmountinfo	*pmp = vftomi(dvp->v_vfsp);
    struct pcnode	*pcp;
    unsigned char	rcls;
    unsigned short	err;
    int			error;
    char		*cp,*cx;

    CKHEALTH(pmp);
    
    if(npc_checkdosname(nm) < 0) {
	error = EINVAL;
	goto out;
    }
    cp = kmem_alloc(MAXPATHLEN);
    cx = npc_buildpath(dvp,cp);
    *cx++ = 0x5c;
    bcopy(nm,cx,strlen(nm)+1);
#ifdef DEBUG
printf("npc mkdir: SMBmkdir path ==%s==\n",cp);
#endif DEBUG
    error = doSMB(buildSMBmkdir, HAVENOSOCK, pmp, HAVENOBUF, &sb, cp);
    kmem_free(cp, MAXPATHLEN);
    if ( error ) {
	goto out;
    }
    CKIN_VSOCK(dvp);
    extractSMBmkdir(sb->buf,&rcls,&err);
    CKIN_BUF(sb);
#ifdef DEBUG
printf("npc_mkdir: SMBmkdir rcls %d err %d\n",rcls,err);
#endif DEBUG
    if ( rcls == SUCCESS ) {
	*vpp = npc_getvnode ( dvp->v_vfsp);
	(*vpp)->v_type = VDIR;
	pcp = vtopc(*vpp);
	pcp->pc_parent = dvp;
	VN_HOLD(dvp);
	bcopy(nm,pcp->pc_name,strlen(nm)+1);
	npc_enterinhash(*vpp);
	return 0;
    }else if ( rcls == ERRDOS ) {
	if ( err == ERRfilexists ) {
	    error = EEXIST;
	} else {
	    error = EACCES;
        }
    } else {
	error = EACCES;
    }
out:
    return(npc_doerror(error, pmp));
}



npc_rmdir(dvp,nm,cred)
struct vnode	*dvp;
char		*nm;
struct ucred	*cred;

{
    struct sendbuf	*sb;
    struct pcmountinfo  *pmp = vftomi(dvp->v_vfsp);
    struct vnode	*vp;
    char		*cp,*cx;
    int			error;
    unsigned char 	rcls;
    unsigned short	err;

    CKHEALTH(pmp);

    if(npc_checkdosname(nm) < 0) {
	error = EINVAL;
	goto out;
    }
    vp = npc_findv(dvp,nm);
    if ( vp ) {
        if ( vtopc(vp)->pc_opencnt ) {
#ifdef DEBUG
	    printf("npc_rmdir: opencnt non-zero\n");
#endif DEBUG
	    error = EBUSY;
	    goto out;
	}
    }
    cp = kmem_alloc(MAXPATHLEN);
    cx = npc_buildpath(dvp,cp);
    *cx++ = 0x5c;
    strcpy(cx, nm);
    error = doSMB(buildSMBrmdir, HAVENOSOCK, pmp, HAVENOBUF, &sb, cp);
    kmem_free(cp, MAXPATHLEN);
    if ( error ) {
	goto out;
    }
    CKIN_VSOCK(dvp);
    extractSMBrmdir(sb->buf,&rcls,&err);
    CKIN_BUF(sb);
#ifdef DEBUG
printf("npc_rmdir: SMBrmdir rcls %d err %d\n",rcls,err);
#endif DEBUG
    
    switch ( rcls ) {
        case SUCCESS:  return 0;
	case ERRSRV:   error = EACCES; break;
	case ERRDOS:
	    switch ( err ) {
	        case ERRbadpath:  error = ENOTDIR; break;
		case ERRnoaccess:  error = ENOTEMPTY; break;
		case ERRremcd:  error = EBUSY; break;
		default: error = EIO; break;
	    }
	    break;
	default: error = EIO; break;
    }
 out:
    return(npc_doerror(error, pmp));
}
	    
/*
 *  go to the pc, get the entries in a directory and put them
 *  in unix form
 */
#define	WILDFILE    "\\????????.???"
/* #define MAXENT  ((SENDBUFSIZ-(SMB_WCT+10)) / DIR_INFO_SIZ) */

#define MAXENT	    (MIN(((SENDBUFSIZ-(SMB_WCT+10)) / DIR_INFO_SIZ) ,\
		         ((mi->m_maxpkt -(SMB_WCT+10)) / DIR_INFO_SIZ)))
npc_readdir(vp, uiop, cred)
struct vnode	*vp;
struct uio	*uiop;
struct ucred	*cred;
{
    CKHEALTH(vftomi(vp->v_vfsp));
    
    return(npc_doerror(npc_readdir_int(vp, uiop, cred), vftomi(vp->v_vfsp)));
}

npc_readdir_int(vp, uiop, cred)
struct vnode	*vp;
struct uio	*uiop;
struct ucred	*cred;
{
    struct sendbuf	*sb;
    off_t		offset;		/* virtual offset into unix directory
					 * data */
    char		*obuf;		/* buffer for user */
    char		*optr;		/* offset into optr */
    char		*path;		/* full path for the current vnode */
    int			resid;
    off_t		start;		/* virtual start pt for this call */
    int			fileno;		/* virtual file number */
    int			index;		/* index into dirdata entries */
    int			len;
    char		*dirdata;	/* ptr to dirents in smb buffer */
    unsigned char	rcls;
    unsigned short	err;
    unsigned short	count;
    int			error;
    struct pcmountinfo	*mi = vftomi(vp->v_vfsp);
    int			rsave;
    int			dirflg;
    int			dis_siz;
    char		*dis_addr;
    int			havestuff = 0;
    
    start = uiop->uio_offset;
    path = (char *)kmem_alloc(MAXPATHLEN);
    bcopy(WILDFILE, npc_buildpath(vp, path), sizeof(WILDFILE));
#ifdef notdef
    printf("npc_readdir: SMBsearch path:%s: start %d resid:%d\n", 
					path, start, uiop->uio_resid);
#endif DEBUG
    dirflg = FA_DIR;
    dis_siz = 0;
    dis_addr = (char *)0;
    
    resid = uiop->uio_resid;
    obuf = optr = (char *)kmem_alloc(rsave=resid);
    
    for(offset = 0, fileno = MAXPCNODE;;) {

	error = doSMB(buildSMBsearch, havestuff, mi, havestuff, &sb,
		      MAXENT, dirflg, path, dis_siz, dis_addr);
	if(path != 0) {
	    /* path is a convenient marker for first time thru */
	    kmem_free(path,MAXPATHLEN);
	    path = (char *)0;
	    dirflg = 0;
	    dis_siz = DIR_INFO_STATSIZ;
	    havestuff++;
	}
	if(error) {
	    kmem_free(obuf, rsave);
	    return(error);
	}
	
	extractSMBsearch(sb->buf, &rcls, &err, &count, &dirdata);
	if(rcls != SUCCESS || count == 0) {
#ifdef DEBUG
printf("npc_readdir: rcls %d err %d count %d\n",rcls,err,count);
#endif DEBUG
	    CKIN_VSOCK(vp);
	    CKIN_BUF(sb);
	    kmem_free(obuf, rsave);
	    return(ENOENT);
	}

	index = 0;
	
	if(offset < start) {
#ifdef NOTDEF
	    printf("rdir: throw away data from pc while offset:%x<start:%x    count:%x  %x  %x\n", offset, start, count, sb->buf, dirdata);
#endif DEBUG
	    while(index < count) {
		
		if(offset >= start) {
		    goto copydata;
		}
		/* pretend we are building dirents */
		{
		    register char   *cp =
				    &dirdata[index*DIR_INFO_SIZ+DIR_INFO_PATH];
		    for(len = 0; *cp != ' ' && len < MAXDOSNAME; cp++, len++ );
		}
		offset += (sizeof(struct direct) - (MAXNAMLEN + 1) +
		          ((len + 1 + 3) & ~3));
		index++;
		fileno++;
	    }
	} else {
copydata:	
	    /* we are at the data now, start filling user buffer until done */
	    while (index < count) {
		char	*cx;
		char	*cp = &dirdata[index*DIR_INFO_SIZ+DIR_INFO_PATH];
		int	reclen;
		/* name from pc is blank padded, kill the blanks
		 * and comput len */
		for( cx = cp, len = 0; *cx != ' ' && len < MAXDOSNAME; cx++, len++ );
		reclen = sizeof(struct direct) - (MAXNAMLEN + 1) +
			    ((len + 1 + 3) &~ 3);

		if(resid - reclen < 0) {
		    goto done;
		}
		((struct direct *)optr)->d_namlen = len;		
		((struct direct *)optr)->d_reclen = reclen;
		bcopy(cp, ((struct direct *)optr)->d_name, len);
		npc_tolower(((struct direct *)optr)->d_name, len);
		((struct direct *)optr)->d_name[len] = '\0';
		{
		    struct vnode    *tvp;
		    if(tvp = npc_findv(vp, ((struct direct *)optr)->d_name)) {
			((struct direct *)optr)->d_fileno =
					    vtopc(tvp)->pc_nodeid;
		    } else {
			((struct direct *)optr)->d_fileno = fileno++;
		    }
		}
		optr += reclen;
		offset += reclen;
		resid -= reclen;
		index++;
	    }
	}

	if (count < MAXENT) {
	    /* thats all folks */
	    break;
	}
	
	/* need more data from pc */
	dis_addr = &dirdata[(count-1)*DIR_INFO_SIZ+DIR_INFO_STAT];
    }
	
done:
    CKIN_VSOCK(vp);
    CKIN_BUF(sb);
    error = uiomove((caddr_t)obuf, uiop->uio_resid-resid, UIO_READ, uiop);
    kmem_free(obuf, rsave);
    uiop->uio_offset = offset;
#ifdef notdef
printf("npc_readdir: resid at end: %d\n",uiop->uio_resid);
#endif DEBUG
    return(error);
}

npc_symlink(dvp,lnm,vap,tnm,cred)
struct vnode	    *dvp;
char		    *lnm;
struct vattr	    *vap;
char		    *tnm;
struct ucred	    *cred;

{
    /*--------------------------------------------------------------+
    |  It is legitimate to get called but not to try and do anything|
    +------------------------------------------------------------- */

    return ( EPERM );
}


/*
 *	I N S E R T :
 *	(npc_readlink)
 */


npc_fsync(vp,cred)
struct vnode	*vp;
struct ucred	*cred;

{
    struct sendbuf	*sb;
    struct pcmountinfo  *pmp = vftomi(vp->v_vfsp);
    unsigned char	rcls;
    unsigned short	err;
    int			error;

    CKHEALTH(pmp);
    
    error = doSMB(buildSMBflush, HAVENOSOCK, pmp, HAVENOBUF, &sb,
		  vtopc(vp)->pc_smbfid);
    if(error) {
#ifdef DEBUG
	printf("npc_fsync: failed doSMB, errno:%d\n", error);
#endif
	goto out;
    }
    extractSMBflush(sb->buf,&rcls,&err);
#ifdef DEBUG
printf("npc_fsync: SMBflush rcls %d err %d\n",rcls,err);
#endif DEBUG
    CKIN_BUF(sb);
    CKIN_VSOCK(vp);
    if (rcls != SUCCESS) {
	error = EIO;
out:	return(npc_doerror(error, pmp));
    }
    return 0;
}

	   


/*
 *	return vp to pool
 *	when rootvp and badfs, delete vfs
 */
int
npc_inactive(vp, cred)
struct vnode	*vp;
struct ucred	*cred;
{
    register struct pcmountinfo *mi = vftomi(vp->v_vfsp);

    if ( vp->v_flag & VROOT) {
	struct pcaccess		    *ap;
	struct pcaccess		    *tap;
	struct vfs		    **vfspp;

#ifdef DEBUG
	if(mi->m_vcnt != 1) {
	    printf("error in rootvp inactivate\n");
	}
#endif DEBUG
	
	if(mi->m_sock) {
#ifdef DEBUG
	    if(mi->m_sock->ps_refcnt == 0) {
		printf("error in sock refcnt\n");
	    }
#endif
	    /* CKOUT_SOCK(mi->m_sock); */
	    {
		struct pcsock	*psp = mi->m_sock;
#ifdef DEBUG
		if(psp->ps_busy) {
		    printf("had to sleep for socket in inactive\n");
		}
#endif DEBUG
		while (psp->ps_busy) {
		    if(sleep((caddr_t)psp, (PZERO+1 | PCATCH))) {
		    /* yukko, what to do now, skip the disconnect I guess.
		     * still messy, what if someone was realy using sock?
		     */
			--mi->m_sock->ps_refcnt;
			goto skip;
		    }
		}
		psp->ps_busy++;
	    }
	    if(mi->m_sock->ps_refcnt > 1) {
		if(npc_treedisconnect(mi)) {
		    /* socket is now checked in */
#ifdef DEBUG
		    printf("error in tree disconnect\n");
#endif
		    --mi->m_sock->ps_refcnt;
		    goto skip;
		} else {
		    CKIN_SOCK(mi->m_sock);
		}
	    }
	    if(--(mi->m_sock->ps_refcnt) == 0) {
		if(mi->m_sock->ps_sock) {
		    soclose(mi->m_sock->ps_sock);
		}
		/* dispose of any pids attached to sock struct */
		{ 
		    struct npcpid	*pidp, **pidpp;
		    pidp = mi->m_sock->ps_pidp;
		    pidpp = &(mi->m_sock->ps_pidp);
#ifdef DEBUG
		    if ( pidp ) {
			printf("npc_inactive: stray pids ");
		    }
#endif DEBUG
		    while ( pidp ) {
#ifdef DEBUG
			printf("%d ",pidp->pid);
#endif DEBUG
			*pidpp = pidp->next;
			kmem_free(pidp,sizeof(struct npcpid));
			pidp = *pidpp;
		    }
#ifdef DEBUG
		    printf("\n");
#endif DEBUG
		}
		kmem_free((caddr_t)mi->m_sock, sizeof(struct pcsock));
		mi->m_sock = NULL;
	    }
	}
skip:
	ap = mi->m_alist;
	while (ap) {
	    tap = ap;
	    ap = ap->pa_next;
	    kmem_free((caddr_t)tap, sizeof(struct pcaccess));
	}
	if(hutchdebug) printf("releasing\n");
	mi->m_alist = 0;

	for(vfspp = &rootvfs; *vfspp != NULL; vfspp = &((*vfspp)->vfs_next)) {
	    if(*vfspp == mi->m_vfsp) {
		*vfspp = (*vfspp)->vfs_next;
		break;
	    }
	}
	kmem_free(mi->m_vfsp, sizeof(struct vfs));
	/* m_rootvp is release in unmount */
	/* kind of strange */
	kmem_free(mi, sizeof(struct pcmountinfo));
    } else {
#ifdef DEBUG
	if(mi->m_vcnt <= 1) {
	    printf("error in m_vcnt\n");
	}
#endif DEBUG
	mi->m_vcnt--;
    }
    if((vp->v_flag & VROOT) == 0) {
	npc_removefromhash(vp);
	VN_RELE(vtopc(vp)->pc_parent);
    }
    npc_relnodeid(vtopc(vp)->pc_nodeid);
    if(pcnodecnt > NORMPCNODE) {
	kmem_free((caddr_t)vtopc(vp), sizeof(struct pcnode));
    } else {
	vtopc(vp)->pc_next = pcnodefreel;
	pcnodefreel = vtopc(vp);
    }
    pcnodecnt--;
}


npc_lockctl(vp,ld,cmd,cred,lf)
struct vnode	    *vp;
struct flock	    *ld;
int		    cmd;
struct ucred	    *cred;
int		    lf;

{
    struct sendbuf	*sb;
    struct pcnode	*pcp = vtopc(vp);
    struct pcmountinfo  *pmp = vftomi(vp->v_vfsp);
    unsigned char	rcls;
    unsigned short	err;
    int			error;

    CKHEALTH(pmp);

    if ( cmd == F_GETLK ) {
	error = EINVAL;
	goto out;
    }
    if ( cmd == F_SETLKW ) {
	error = EINVAL;
	goto out;
    }

    if ( ld->l_type == F_UNLCK ) {

	if ( ld->l_start == 0 && ld->l_len == 0 ) {
	    error = unlock_all(vp);
	} else {
	    error = find_and_unlock(vp,ld);
	}
    }else{
	/* F_RDLCK will be treated identically to F_WRLCK */

	if ( error = lock_collision(vp,ld) ) {
	    goto out;
	}
	
	error = doSMB(buildSMBlock, HAVENOSOCK, pmp, HAVENOBUF, &sb,
			  pcp->pc_smbfid,
			  ld->l_len,      /* corresponds to SMB byte count */
			  ld->l_start,    /* corresponds to SMB offset */
			  u.u_procp->p_pid);    /*  notice, not PSEUDO_PID */
	if (error) {
	    goto out;
	}
	extractSMBlock(sb->buf,&rcls,&err);
	CKIN_VSOCK(vp);
	CKIN_BUF(sb);
#ifdef DEBUG
printf("npc_lockctl: SMBlock rcls %d err %d\n",rcls,err);
#endif DEBUG
	switch ( rcls ) {
	    case SUCCESS:
	        add_lock(vp,ld);
	        return 0;
	    case ERRDOS: error = EACCES; break;
            default: error = EIO; break;
	}
    }
out:
    return(npc_doerror(error, pmp));
}
    

/*
 *	this is where we die because we forgot to implement something...
 *	(what do you mean we, turkey)
 */
int
npc_badop()
{
    panic("npc_badop");
}
npc_noop()
{
    return(0);
}
