/*
 * Token ring address resolution protocol.
 */

#include "param.h"
#include "systm.h"
#include "mbuf.h"
#include "socket.h"
#include "time.h"
#include "kernel.h"
#include "errno.h"
#include "ioctl.h"
#include "syslog.h"

#include "../net/if.h"
#include "../netinet/in.h"
#include "../netinet/in_systm.h"
#include "../netinet/ip.h"
#include "../netinet/if_ether.h"
#include "../netinet/if_token.h"
#ifdef	APPLETALK
#include "../netddp/atalk.h"
#include "../netddp/katalk.h"
#define	ATF_ISAT	0x40		/* AppleTalk seg. (change from 4.2) */
#endif	APPLETALK


#define	ARPTAB_NB	19		/* number of buckets */
#define	ARPTAB_SIZE	(ARPTAB_BSIZ * ARPTAB_NB)

struct	trarptab trarptab[ARPTAB_SIZE];
int	trarptab_size = ARPTAB_SIZE;	/* for arp command */

#ifdef	ARPTAB_HASH
#undef	ARPTAB_HASH
#endif
#define	ARPTAB_HASH(a) ((short)((((a) >> 16) ^ (a)) & 0x7fff) % ARPTAB_NB)

#define	TRARPTAB_LOOK(at,addr) { \
	register n; \
	at = &trarptab[ARPTAB_HASH(addr) * ARPTAB_BSIZ]; \
	for (n = 0 ; n < ARPTAB_BSIZ ; n++,at++) \
		if (at->at_iaddr.s_addr == addr) \
			break; \
	if (n >= ARPTAB_BSIZ) \
		at = 0; }

int	trarpdebug = 0;

/* timer values */
#define	ARPT_AGE	(60*1)	/* aging timer, 1 min. */
#define	ARPT_KILLC	20	/* kill completed entry in 20 mins. */
#define	ARPT_KILLI	3	/* kill incomplete entry in 3 minutes */

u_char tokenbroadcastaddr[6] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
extern struct ifnet loif;

/*
** LLC and SNAP headers are odd length, C will kindly give us the
**	exta byte for the headers, which we don't need.
*/

struct trarp_packet {			/* Complete packet */
	struct token_header ap_th;
	u_char		    ap_hdrbuf[8];	/* Holds llc+snap headers */
	struct token_arp    ap_ta;
};

/*
 * Timeout routine.  Age arp_tab entries once a minute.
 */
trarptimer()
{
	register struct trarptab *at;
	register i;

	timeout(trarptimer, (caddr_t)0, ARPT_AGE * hz);
	at = &trarptab[0];
	for (i = 0; i < ARPTAB_SIZE; i++, at++) {
		if (at->at_flags == 0 || (at->at_flags & ATF_PERM))
			continue;
		if (++at->at_timer < ((at->at_flags&ATF_COM) ?
		    ARPT_KILLC : ARPT_KILLI))
			continue;
		/* timer has expired, clear entry */
		trarptfree(at);
	}
}

/*
 * Broadcast an ARP packet, asking who has addr on interface ac.
 */
trarpwhohas(ac, addr)
register struct trarpcom *ac;
struct in_addr *addr;
{
	register struct mbuf *m;
	register struct token_header *th;
	register struct token_arp *ta;
	register struct trarp_packet *ap;
	struct sockaddr sa;
	struct llc_header *llc;
	struct snap *snap;

	if ((m = m_get(M_DONTWAIT, MT_DATA)) == NULL)
		return (1);
	m->m_len = sizeof (struct trarp_packet);
	ap = mtod(m, struct trarp_packet *);
	bzero((caddr_t)ap, sizeof (struct trarp_packet));

	th = &ap->ap_th;
	bcopy(tokenbroadcastaddr, th->token_dhost, sizeof (tokenbroadcastaddr));
	th->token_frame = FC_STD;
	th->token_access = 0;
	th->token_shost[0] = 0;		/* no routing info. */

	llc = (struct llc_header *)ap->ap_hdrbuf;
	llc->llc_dsap = llc->llc_ssap = SAP_IP;
	llc->llc_cntl = LLC_U;

	snap = (struct snap *)&ap->ap_hdrbuf[3];
	snap->snap_spare[0]= snap->snap_spare[1]= snap->snap_spare[2]= 0;
	*(u_short *)&snap->snap_type[0] = ETHERTYPE_ARP;

	ta = &ap->ap_ta;
	ta->trarp_hrd = htons(TRARPHRD_ETHER);
	ta->trarp_pro = htons(ETHERTYPE_IP);
	ta->trarp_hln = sizeof (ta->trarp_sha);	/* hardware address length */
	ta->trarp_pln = sizeof (ta->trarp_spa);	/* protocol address length */
	ta->trarp_op = htons(ARPOP_REQUEST);
	bcopy(ac->ac_traddr, ta->trarp_sha, sizeof(ta->trarp_sha));
	bcopy((caddr_t)&ac->ac_ipaddr, (caddr_t)ta->trarp_spa,
		sizeof(ta->trarp_spa));
	bcopy((caddr_t)addr, (caddr_t)ta->trarp_tpa, sizeof(ta->trarp_tpa));
	sa.sa_family = AF_UNSPEC;
	if (trarpdebug)
	    printf("TRARP: sending REQUEST\n");
	return ((*ac->ac_if.if_output)(&ac->ac_if, m, &sa));
}

extern int	useloopback;	/* should we use the loopback? */
/*
 * Resolve an IP address into an token ring address.  If success, 
 * desttr is filled in and 1 is returned.  If there is no entry
 * in arptab, set one up and broadcast a request 
 * for the IP address;  return 0.  Hold onto this mbuf and 
 * resend it once the address is finally resolved.
 *
 * We do some (conservative) locking here at splimp, since
 * arptab is also altered from input interrupt service (ecintr/ilintr
 * calls arpinput when TOKEN_ARPTYPE packets come in).
 */
trarpresolve(ac, m, destip, desttr)
	register struct trarpcom *ac;
	struct mbuf *m;
	register struct in_addr *destip;
	register u_char *desttr;
{
	register struct trarptab *at;
	register struct ifnet *ifp;
	register int i;
	struct sockaddr_in sin;
	int s, lna;

	if (in_broadcast(*destip)) {
		bcopy(tokenbroadcastaddr, desttr, sizeof(tokenbroadcastaddr));
		return (1);
	}
	lna = in_lnaof(*destip);
	ifp = &ac->ac_if;
	/* if for us, then use software loopback driver */
	if (destip->s_addr == ac->ac_ipaddr.s_addr) {
	    if(useloopback) {
		sin.sin_family = AF_INET;
		sin.sin_addr = *destip;
		return (looutput(&loif, m, (struct sockaddr *)&sin));
	    } else {
		bcopy(ac->ac_traddr, desttr, sizeof(ac->ac_traddr));
		return(1);
	    }
	}
	s = splimp();
	TRARPTAB_LOOK(at, destip->s_addr);
	if (at == 0) {			/* not found */
		if (ifp->if_flags & IFF_NOARP) {
		    bcopy(ac->ac_traddr, desttr, sizeof(ac->ac_traddr));
		    desttr[3] = (lna >> 16) & 0x7f;
		    desttr[4] = (lna >> 8) & 0xff;
		    desttr[5] = lna & 0x7f;
		    splx(s);
		    return (1);
		} else {
		    at = trarptnew(destip);
		    at->at_hold = m;
		    trarpwhohas(ac, destip);
		    splx(s);
		    return (0);
		}
	}
	at->at_timer = 0;		/* restart the timer */
	if (at->at_flags & ATF_COM) {	/* entry IS complete */
		bcopy(at->at_traddr, desttr, sizeof(at->at_traddr));
		splx(s);
		return (1);
	}
	/*
	 * There is an arptab entry, but no token address
	 * response yet.  Replace the held mbuf with this
	 * latest one.
	 */
	if (at->at_hold)
		m_freem(at->at_hold);
	at->at_hold = m;
	trarpwhohas(ac, destip);		/* ask again */
	splx(s);
	return (0);
}

/*
 * Called from ecintr/ilintr when token packet type TOKEN_ARP
 * is received.  Algorithm is that given in RFC 826.
 * In addition, a sanity check is performed on the sender
 * protocol address, to catch impersonators.
 */
trarpinput(ac, m)
	register struct trarpcom *ac;
	struct mbuf *m;
{
	register struct arphdr *ar;

	if (trarpdebug)
	    printf("trarpinput called\n");

	if (ac->ac_if.if_flags & IFF_NOARP)
		goto out;
	IF_ADJ(m);
	if (m->m_len < sizeof(struct arphdr))
		goto out;
	
	ar = mtod(m, struct arphdr *);
	if (ntohs(ar->ar_hrd) != TRARPHRD_ETHER)
		goto out;
	if (m->m_len < sizeof(struct arphdr) + 2*ar->ar_hln + 2*ar->ar_pln)
		goto out;

	switch (ntohs(ar->ar_pro))
	{
	   case ETHERTYPE_IP:
		in_trarpinput(ac, m);
		return;

	   default:
		break;
	}
out:
	if (trarpdebug)
	    printf("trarpinput: went to out.\n");
	m_freem(m);
}


/*
 * ARP for Internet protocols on 10 Mb/s Ethernet.
 * Algorithm is that given in RFC 826.
 * In addition, a sanity check is performed on the sender
 * protocol address, to catch impersonators.
 */
in_trarpinput(ac, m)
	register struct trarpcom *ac;
	struct mbuf *m;
{
	register struct token_arp *ta;
	register struct trarp_packet *ap;
	register struct trarptab *at;  /* same as "merge" flag */
	struct token_header *th;
	struct mbuf *m0 = m;
	struct sockaddr_in sin;
	struct sockaddr sa;
	struct in_addr isaddr, itaddr, myaddr;
	register int proto, op, s;
	struct llc_header *llc;
	struct snap *snap;

	myaddr.s_addr = ac->ac_ipaddr.s_addr;
	ta = mtod(m, struct token_arp *);
	proto = ntohs(ta->trarp_pro);
	op = ntohs(ta->trarp_op);
	isaddr.s_addr = ((struct in_addr *)ta->trarp_spa)->s_addr;
	itaddr.s_addr = ((struct in_addr *)ta->trarp_tpa)->s_addr;
#ifdef	APPLETALK
	/* high bytes of dest == 0's?  If so, use our AppleTalk address */
	if (bcmp(&itaddr, "\000\000\000", 3) == 0) {
		struct ifnet *ifp = &ac->ac_if;
		extern struct a_addr *ddp_getaddr();
		struct a_addr *appleaddr;

		if (((u_char *)&itaddr.s_addr)[3] == 0)
			goto out;
		myaddr = itaddr;
		appleaddr = ddp_getaddr(ifp);
		((u_char *)&myaddr)[3] = appleaddr->at_Node;
	}
#endif	APPLETALK
	if (!bcmp((caddr_t)ta->trarp_sha, (caddr_t)ac->ac_traddr,
						  sizeof (ta->trarp_sha)))
		goto out;	/* it's from me, ignore it. */
	if (!bcmp((caddr_t)ta->trarp_sha, (caddr_t)tokenbroadcastaddr,
						  sizeof (ta->trarp_sha))) {
		log(LOG_ERR,
		    "arp: source token addr is broadcast for IP addr %x!\n",
						    ntohl(isaddr.s_addr));
		if (trarpdebug)
		    printf("arp: source token addr is broadcast for IP addr %x!\n",
						    ntohl(isaddr.s_addr));
		goto out;
	}
	if (isaddr.s_addr == myaddr.s_addr) {
		log(LOG_ERR, 
		    "arp: duplicate IP address %x sent from token address %e\n",
		    ntohl(isaddr.s_addr), ta->trarp_sha);
		if (trarpdebug)
		    printf("arp: dup IP addr %x sent from token addr %e\n",
					    ntohl(isaddr.s_addr),ta->trarp_sha);
		itaddr = myaddr;
		if (op == ARPOP_REQUEST)
			goto reply;
		goto out;
	}
	s = splimp();
	TRARPTAB_LOOK(at, isaddr.s_addr);
	if (at) {
		bcopy((caddr_t)ta->trarp_sha, (caddr_t)at->at_traddr,
						    sizeof(ta->trarp_sha));
		at->at_flags |= ATF_COM;
		if (at->at_hold) {
#ifdef	APPLETALK
			if (at->at_flags & ATF_ISAT)
				sin.sin_family = AF_APPLETALK;
			else
#endif	APPLETALK
			sin.sin_family = AF_INET;
			sin.sin_addr = isaddr;
			(*ac->ac_if.if_output)(&ac->ac_if, 
			    at->at_hold, (struct sockaddr *)&sin);
			at->at_hold = 0;
		}
	}
	if (at == 0 && itaddr.s_addr == myaddr.s_addr) {
		/* ensure we have a table entry */
		at = trarptnew(&isaddr);
		if (trarpdebug)
		    printf("Add entry:\n\tInet 0x%x\n\tToken 0x%x.%x\n", 
						isaddr.s_addr, 
						*(u_long *)ta->trarp_sha,
						*(u_char *)&ta->trarp_sha[4]);
		bcopy((caddr_t)ta->trarp_sha, (caddr_t)at->at_traddr,
		    sizeof(ta->trarp_sha));
		at->at_flags |= ATF_COM;
	}
	splx(s);

reply:
	if (op != ARPOP_REQUEST)
		goto out;

	if (itaddr.s_addr == myaddr.s_addr) {
		/* I am the target */
		bcopy((caddr_t)ta->trarp_sha, (caddr_t)ta->trarp_tha,
		    sizeof(ta->trarp_sha));
		bcopy((caddr_t)ac->ac_traddr, (caddr_t)ta->trarp_sha,
		    sizeof(ta->trarp_sha));
	} else {
		TRARPTAB_LOOK(at, itaddr.s_addr);
		if (at == NULL || (at->at_flags & ATF_PUBL) == 0)
			goto out;
		bcopy((caddr_t)ta->trarp_sha, (caddr_t)ta->trarp_tha,
		    sizeof(ta->trarp_sha));
		bcopy((caddr_t)at->at_traddr, (caddr_t)ta->trarp_sha,
		    sizeof(ta->trarp_sha));
	}

	bcopy((caddr_t)ta->trarp_spa, (caddr_t)ta->trarp_tpa,
	    sizeof(ta->trarp_spa));
	bcopy((caddr_t)&itaddr, (caddr_t)ta->trarp_spa,
	    sizeof(ta->trarp_spa));
	ta->trarp_op = htons(ARPOP_REPLY); 
	ta->trarp_pro = htons(ETHERTYPE_IP);

	/*
         * Add local net header.  If no space in first mbuf,
         * allocate another.
         */

	m = m_get(M_DONTWAIT, MT_HEADER);
	if (m == 0) {
		m_freem(m0);
		return;
	}
	m->m_next = m0;
	m->m_len = sizeof(struct token_header)+ LLC_SIZE + SNAP_SIZE;

	th = mtod(m, struct token_header *);
	bcopy((caddr_t)ta->trarp_tha, (caddr_t)th->token_dhost,
	    sizeof(th->token_dhost));
	th->token_frame = FC_STD;
	th->token_access = 0;
	th->token_shost[0] = 0;  /* No routing info field */

	llc = (struct llc_header *)(th + 1);
	llc->llc_dsap = llc->llc_ssap = SAP_IP;
	llc->llc_cntl = LLC_U;

	snap = (struct snap *)((int)llc + LLC_SIZE);
	snap->snap_spare[0]= snap->snap_spare[1]= snap->snap_spare[2]= 0;
	*(u_short *)&snap->snap_type[0] = ETHERTYPE_ARP;

	sa.sa_family = AF_UNSPEC;
	(*ac->ac_if.if_output)(&ac->ac_if, m, &sa);
	return;

out:
	if (trarpdebug)
	    printf("in_trarpinput: went to out.\n");
	m_freem(m);
	return;
}


/*
 * Free an arptab entry.
 */
trarptfree(at)
	register struct trarptab *at;
{
	int s = splimp();

	if (at->at_hold)
		m_freem(at->at_hold);
	at->at_hold = 0;
	at->at_timer = at->at_flags = 0;
	at->at_iaddr.s_addr = 0;
	splx(s);
}


/*
 * Enter a new address in arptab, pushing out the oldest entry 
 * from the bucket if there is no room.
 * This always succeeds since no bucket can be completely filled
 * with permanent entries (except from arpioctl when testing whether
 * another permanent entry).
 */
struct trarptab *
trarptnew(addr)
	struct in_addr *addr;
{
	register n;
	int oldest = 0;
	register struct trarptab *at, *ato = NULL;
	static int first = 1;

	if (first) {
		first = 0;
		timeout(trarptimer, (caddr_t)0, hz);
	}
	at = &trarptab[ARPTAB_HASH(addr->s_addr) * ARPTAB_BSIZ];
	for (n = 0 ; n < ARPTAB_BSIZ ; n++,at++) {
		if (at->at_flags == 0)
			goto out;	 /* found an empty entry */
		if (at->at_flags & ATF_PERM)
			continue;
		if (at->at_timer > oldest) {
			oldest = at->at_timer;
			ato = at;
		}
	}
	if (ato == NULL)
		return(NULL);
	at = ato;
	trarptfree(at);
out:
	at->at_iaddr = *addr;
	at->at_flags = ATF_INUSE;
	return (at);
}

trarpioctl(cmd, data)
	int cmd;
	caddr_t data;
{
	register struct arpreq *ar = (struct arpreq *)data;
	register struct trarptab *at;
	register struct sockaddr_in *sin;
	int s;

	if (ar->arp_pa.sa_family != AF_INET ||
	    ar->arp_ha.sa_family != AF_UNSPEC)
		return (EAFNOSUPPORT);
	sin = (struct sockaddr_in *)&ar->arp_pa;
	if (trarpdebug)
		printf("trarpioctl: looking for 0x%x\n", sin->sin_addr.s_addr);
	s = splimp();
	TRARPTAB_LOOK(at, sin->sin_addr.s_addr);
	if (at == NULL) {		/* not found */
		if (cmd != SIOCSARP) {
			splx(s);
			return (ENXIO);
		}
		if (ifa_ifwithnet(&ar->arp_pa) == NULL) {
			splx(s);
			return (ENETUNREACH);
		}
	}
	switch (cmd) {

	case SIOCSARP:		/* set entry */
		if (at == NULL) {
			at = trarptnew(&sin->sin_addr);
			if (ar->arp_flags & ATF_PERM) {
			/* never make all entries in a bucket permanent */
				register struct trarptab *tat;
				
				/* try to re-allocate */
				tat = trarptnew(&sin->sin_addr);
				if (tat == NULL) {
					trarptfree(at);
					splx(s);
					return (EADDRNOTAVAIL);
				}
				trarptfree(tat);
			}
		}
		bcopy(ar->arp_ha.sa_data, at->at_traddr, sizeof(at->at_traddr));
		at->at_flags = ATF_COM | ATF_INUSE |
			(ar->arp_flags & (ATF_PERM|ATF_PUBL));
		at->at_timer = 0;
		break;

	case SIOCDARP:		/* delete entry */
		trarptfree(at);
		break;

	case SIOCGARP:		/* get entry */
		bcopy(at->at_traddr, ar->arp_ha.sa_data, sizeof(at->at_traddr));
		ar->arp_flags = at->at_flags;
		break;
	}
	splx(s);
	return (0);
}


#ifdef	RARP
/*
 * REVerse Address Resolution Protocol (revarp) is used by client
 * to find out its IP address when all it knows is its Token address.
 */
int trrevarp = 1;
int trrevarpdebug = 0;
static struct in_addr myaddr;

trrevarp_myaddr(ifp)
	register struct ifnet *ifp;
{
	register struct sockaddr_in *sin;
	struct sockaddr temp;
	int s;

	/*
	 * We need to give the interface a temporary address just
	 * so it gets initialized. Hopefully, the address won't get used.
	 */
	sin = (struct sockaddr_in *)&temp;
	sin->sin_family = AF_INET;
	sin->sin_port = 0;
	sin->sin_addr = if_makeaddr(INADDR_ANY, 123/*machineid*/ & 0xFFFFFF);
	if (if_setaddr(ifp, (struct sockaddr *)sin))
		printf("revarp: can't set temp inet addr\n");
	if (revarp) {
		myaddr.s_addr = 0;
		revarp_start(ifp);
		s = splimp();
		while (myaddr.s_addr == 0)
			(void) sleep((caddr_t)&myaddr, PZERO-1);
		(void) splx(s);
		sin->sin_addr = myaddr;
		if (if_setaddr(ifp, (struct sockaddr *)sin))
			printf("revarp: can't set perm inet addr\n");
	}
}

trrevarp_start(ifp)
	register struct ifnet *ifp;
{
	register struct mbuf *m;
	register struct token_arp *ta;
	register struct trarp_packet *ap;
	register struct token_header *th;
	static int retries = 0;
	u_char	mytoken;
	struct sockaddr sa;

	if (myaddr.s_addr != 0) {
		if (retries >= 2)
			printf("Found Internet address %x\n", myaddr.s_addr);
		retries = 0;
		return;
	}
	(void) localtokenaddr(u_char *)NULL, &mytoken);
	if (++retries == 2)
	    printf("revarp: Requesting Internet address for %e\n", &mytoken);

	if ((m = m_get(M_DONTWAIT, MT_DATA)) == NULL)
		panic("revarp: no mbufs");
	m->m_len = sizeof (struct trarp_packet);
	ap = mtod(m, struct trarp_packet *);
	bzero((caddr_t)ap, sizeof (struct trarp_packet));

	sa.sa_family = AF_UNSPEC;

	th = &ap->ap_th;
	th->token_dhost = tokenbroadcastaddr;
	th->token_frame = FC_STD;
	th->token_access = 0;
	th->token_shost[0] = 0;

	llc = (struct llc_header *)ap->ap_hdrbuf;
	llc->llc_dsap = llc->llc_ssap = SAP_IP;
	llc->llc_cntl = LLC_U;

	snap = (struct snap *)&ap->ap_hdrbuf[3];
	snap->snap_spare[0]= snap->snap_spare[1]= snap->snap_spare[2]= 0;
	*(u_short *)&snap->snap_type[0] = TOKEN_ARPTYPE;

	ta = &ap->ap_ta;
	ta->arp_hrd = htons(TRARPHRD_ETHER);
	ta->arp_pro = htons(TOKEN_IPTYPE);
	ta->arp_hln = sizeof trarp_sha(ta);	/* hardware address length */
	ta->arp_pln = sizeof trarp_spa(ta);	/* protocol address length */
	ta->arp_op = htons(TRREVARP_REQUEST);
	arp_sha(ea) = mytoken;
	arp_tha(ea) = mytoken;
	(*ifp->if_output)(ifp, m, &sa);
	timeout(revarp_start, (caddr_t)ifp, 3*hz);
}

/*
 * Reverse-ARP input
 * Server side answers requests based on the arp table
 */
trrevarpinput(ac, m)
	register struct trarpcom *ac;
	struct mbuf *m;
{
	register struct token_arp *ta;
	register struct arptab *at = 0;
	register struct token_header *th;
	u_char mytoken;
	struct sockaddr sa;
	struct mbuf *m0 = m;

	ta = mtod(m, struct token_arp *);
	if (m->m_len < sizeof(struct token_arp))
		goto out;
	if (ac->ac_if.if_flags & IFF_NOARP)
		goto out;
	if (ntohs(ta->arp_pro) != TOKEN_IPTYPE)
		goto out;
	if (ntohs(ta->arp_op) == TRREVARP_REPLY) {
		(void) localtokenaddr((u_char *)NULL, &mytoken);
		if (bcmp((caddr_t)&arp_tha(ta), (caddr_t)&mytoken, 6) == 0) {
			myaddr = arp_tpa(ta);
			wakeup((caddr_t)&myaddr);
		}
	} else if (trrevarp && ntohs(ta->arp_op) == TRREVARP_REQUEST) {
		if (trrevarpdebug)
			printf("tr revarp request from %e\n", &(arp_tha(ta)));
		/*  see if we know the IP addr of the requestor  */
		for (at = trarptab ; at < &trarptab[TRARPTAB_SIZE] ; at++) {
			if (at->at_flags & ATF_PERM &&
			    !bcmp((caddr_t)&at->at_traddr,
			    (caddr_t)&arp_tha(ta), 6))
				break;
		}
		if (at == &trarptab[TRARPTAB_SIZE])
			at = 0;
		if (at) {
			/* found a match, send it back */

			arp_tpa(ta) = at->at_iaddr;
			arp_spa(ta) =
			  ((struct sockaddr_in *)&ac->ac_if.if_addr)->sin_addr;
			(void) localtokenaddr((u_char *)NULL, &mytoken);
			arp_sha(ta) = mytoken;
			ta->arp_op = htons(TRREVARP_REPLY);

			/*
			 * Add local net header.  If no space in first mbuf,
			 * allocate another.
			 */

			m = m_get(M_DONTWAIT, MT_HEADER);
			if (m == 0) {
				m_freem(m0);
				return;
			}
			m->m_next = m0;
			m->m_len = sizeof(struct token_header) + LLC_SIZE
								+ SNAP_SIZE;

			th = mtod(m, struct token_header *);
			bcopy((caddr_t)ta->arp_sha, (caddr_t)th->token_dhost,
			    sizeof(th->token_dhost));
			th->token_frame = FC_STD;
			th->token_access = 0;
			th->token_shost[0] = 0;

			llc = (struct llc_header *)(th + 1);
			llc->llc_dsap = llc->llc_ssap = SAP_IP;
			llc->llc_cntl = LLC_U;

			snap = (struct snap *)((int)llc + LLC_SIZE);
			snap->snap_spare[0] = 0;
			snap->snap_spare[1] = 0;
			snap->snap_spare[2] = 0;
			*(u_short *)&snap->snap_type[0] = TOKEN_REVARPTYPE;

			sa.sa_family = AF_UNSPEC;

			if (trrevarpdebug)
				printf("tr_revarp reply to %X\n", arp_tpa(ta));
			(*ac->ac_if.if_output)(&ac->ac_if, m, &sa);
			return;
		}
	}
out:
	m_freem(m);
	return;
}

localtokenaddr(hint, result)
	u_char *hint, *result;
{
	static int found = 0;
	static u_char addr;

	if (!found) {
		found = 1;
		if (hint == NULL)
			return (0);
		addr = *hint;
		printf("Tokennet address = %e\n", &addr);
	}
	if (result != NULL)
		*result = addr;
	return (1);
}
#endif	RARP
