/*
 * hex - translate object files into ASCII formats suitable for Motorola 
 *		S-record downloading
 *
 * 	Usage:
 *	    hex [-b] [-f] [-l] [-n#] [-r] [-s0] [-s2] [-s3] [-ns8] [+saddr] file
 *	where:
 *		b	generate records for bss segment.
 *		f	just ship ifile as data, not as an executable file
 *		l	output "Loading at" message
 *		n#	number of characters to output
 *		r	output a \r instead of a \n after each record
 *		s0	output a leading s0 record
 *		s2	S2 records only
 *		s3	S3 records only
 *		ns8	do not output a trailing s8 record
 *		saddr	starting load address (in hex). files starting address
 *			is used if saddr is not present
 */

#include <stdio.h>
#include <a.out.h>
#include <sys/types.h>
#include <sys/stat.h>

#define MAXOUT	1000
#define NOUT	32

char *ifile;
struct exec exec;
char tbuf[MAXOUT+10];
int nout = NOUT;
int bflg;
int lflg;
int fflg;
int rflg;
int s0flg;
int s2flg;
int s3flg;
int ns8flg;
int csum;
FILE *inp;

char hex[] = {
'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
};

#define FATAL 1	
#define NONFATAL 0
#define A16	((unsigned) 0x0000ffff)
#define A24	((unsigned) 0x00ffffff)

/* 
 * main - process args, open files
 */
main(argc, argv)
char **argv;
{
	register i;
	int saddr = -1;
	char _obuf[BUFSIZ];

	setbuf(stdout, _obuf);

	ifile = NULL;
	if (argc == 1)
bad:
	    diagnostic(FATAL, "usage:hex [-f] [-l] [-n#] [-r] [-s0] [-s2] [-ns8] [+saddr] ifile", NULL);
	while (argc > 1) {
		if (strcmp("-f", argv[1]) == 0) {
			fflg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-l", argv[1]) == 0) {
			lflg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-r", argv[1]) == 0) {
			rflg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-s0", argv[1]) == 0) {
			s0flg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-s2", argv[1]) == 0) {
			s2flg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-s3", argv[1]) == 0) {
			s3flg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-ns8", argv[1]) == 0) {
			ns8flg++;
			--argc; ++argv;
			continue;
		}
		if (strncmp("-n", argv[1], 2) == 0) {
			nout = atoi(&argv[1][2]);
			if (nout >= MAXOUT)
				diagnostic(FATAL, "hex:n parameter must be less than %d", MAXOUT, NULL);
			--argc; ++argv;
			continue;
		}
		if (*argv[1] == '+') {
			sscanf(argv[1] , "+%lx" , &saddr);
			--argc; ++argv;
			continue;
		}
		if (ifile != NULL)
			goto bad;
		ifile = argv[1];
		--argc;
		++argv;
	}

	if (ifile == NULL)
		goto bad;

	if ((inp = fopen(ifile, "r")) == NULL) 		
		diagnostic(FATAL, "Can't open ", ifile, NULL);

	if (!fflg) {
		seekfile(0);
		if (getbytes(&exec, sizeof(struct exec)) != sizeof(struct exec))
			diagnostic(FATAL, "Can't fetch file header", NULL);

		if (N_BADMAG(exec))
			diagnostic(FATAL, "Bad file type", NULL);
	}

	if (saddr == -1)
		if (fflg)
			saddr = 0;
		else
			saddr = exec.a_entry;
	if (lflg)
		printf("Load starting at 0x%lx\n", saddr);

	if (s0flg)
		sends0(ifile);
	if (fflg) {
		senddata(saddr, siz());
	} else {
		seekfile(N_TXTOFF(exec));
		senddata(saddr, exec.a_text);

		seekfile(N_TXTOFF(exec) + exec.a_text);
		senddata(saddr + exec.a_text, exec.a_data);

		if (bflg) {
		    seekfile(N_TXTOFF(exec) + exec.a_text + exec.a_data);	
		    senddata(saddr + exec.a_text + exec.a_data, exec.a_bss);
		}
	}
	if (ns8flg == 0)
		sends8(saddr);
	exit(0);
}

siz()
{
	struct stat sb;

	if (fstat(fileno(inp), &sb) < 0)
		return(0);
	return(sb.st_size);
}

diagnostic(fatal, args)
char *args;
{
	register char **a;

	for (a = &args; *a != NULL; a++)
		printf("%s", *a);
	printf("\n");
	if (fatal)
		exit(0);
}


seekfile(loc)
{
	fseek(inp, loc, 0);
}

getbytes(addr, ct)
register char *addr;
register int ct;
{
	register int n;

	n = 0;
	while (ct-- > 0) {
		*addr = getc(inp);
		if (feof(inp) || ferror(inp))
			break;
		addr++;
		n++;
	}
	return(n);
}

sends0(fname)
register char *fname;
{
	csum = 0;
	printf("S0");
	sendbyte(strlen(fname)+1);
	while (*fname != 0)
		sendbyte(*fname++);
	sendbyte((255-csum) & 0xff);
	sendend();
}

sends8(addr)
{
	csum = 0;
	if (s3flg || addr > A24) {
		printf("S7");
		sendbyte(4+1);
		send32(addr);
	} else if (s2flg || addr > A16) {
		printf("S8");
		sendbyte(3+1);
		send24(addr);
	} else {
		printf("S9");
		sendbyte(2+1);
		send16((int) addr);
	}
	sendbyte((255-csum) & 0xff);
	sendend();
}

senddata(addr, ct)
register int addr, ct;
{
	register int i, n;

	for (;ct > 0; ct -= n) { 
		n = (ct > nout) ? nout : ct;
		if ((n = getbytes(tbuf, n)) == 0)
			break;
		csum = 0;
		if (s3flg || addr > A24) {
			printf("S3");
			sendbyte(4+n+1);
			send32(addr);
		} else if (s2flg || addr > A16) {
			printf("S2");
			sendbyte(3+n+1);
			send24(addr);
		} else {
			printf("S1");	
			sendbyte(2+n+1);
			send16((int) addr);
		}
		for (i = 0; i < n; i++)
			sendbyte(tbuf[i]);
		sendbyte((255-csum) & 0xff);
		sendend();
		addr += n;
	}
}

sendend()
{
	if (rflg)
		printf("\r");
	else
		printf("\n");
}

sendbyte(c)
register int c;
{
	csum += c;
	putchar(hex[(c>>4)&0xf]);
	putchar(hex[c&0xf]);
}


send16(x)
register int x;
{
	sendbyte(x >> 8);
	sendbyte(x);
}


send24(x)
register int x;
{
	sendbyte(x >> 16);
	sendbyte(x >> 8);
	sendbyte(x);
}

send32(x)
register int x;
{
	sendbyte(x >> 24);
	sendbyte(x >> 16);
	sendbyte(x >> 8);
	sendbyte(x);
}
