/*
 * Standalone driver for the Interphase VME_SMD disk controller.
 */
#include "saio.h"
#include "sais68k.h"
#include "../dev/smreg.h"

#define	SMADDR0	  	((struct smdevice *)&vme_short[0x0000])
#define	SMADDR1	  	((struct smdevice *)&vme_short[0x0200])
#define	SMADDR2	  	((struct smdevice *)&vme_short[0x0400])
#define	SMADDR3	  	((struct smdevice *)&vme_short[0x0600])
#define	SMADDR4	  	((struct smdevice *)&vme_short[0x0800])

#define	SM_CN		5			/* number of controllers */
#define	SM_DR		4			/* drives per controller */
#define	UTOCN(x)	((x) >> 2)
#define	UTODR(x)	((x) & 0x3)
#define	SM_MAXCNT	0x1F000			/* must be multiple of nbsec */
#define	SM_RETRYCNT	4
#ifdef	QX
#	define	SM_CPUMEMT	(MEMT_16BIT)
#	define	SM_ADDRMOD	(ADRM_STD_N_D)
#else	QX
#	ifdef	M68030
#		define	SM_CPUMEMT	(MEMT_32BIT)
#		define	SM_ADDRMOD	(ADRM_STD_N_D)
#	else
#		ifdef	M68020
#			define	SM_CPUMEMT	((((*BSR) & BSR_MEMWIDTH) == BSR_16BITMEM) \
				? MEMT_16BIT : MEMT_32BIT)
#			ifdef	M68025
#				define	SM_ADDRMOD	(ADRM_EXT_N_D)
#			else	M68025
#				define	SM_ADDRMOD	(ADRM_STD_N_D)
#			endif	M68025
#		else	M68020
#			define	SM_CPUMEMT	(MEMT_16BIT)
#			define	SM_ADDRMOD	(ADRM_STD_N_D)
#		endif	M68020
#	endif	M68030
#endif	QX
#define	SMD_VECT	0x40

struct smdevice *smstd[SM_CN] = { SMADDR0, SMADDR1, SMADDR2, SMADDR3, SMADDR4};

extern int	topmem;
int forbb, xbad144, xbad144_soft, xbad144_hadsoft, smstrat();
struct smuib smuib_dflt = { 0,10,0,0,100,0,512,16,32,1,3,1024,5,0,1,0xFF};
union smuib_io {
	struct smuib	sm_uib;
	char		sm_uib_pad[512];
}	smuib_tmp;

struct	iob	csmbuf;
unsigned long	sm_qbaddr;
char smrupt[SM_CN];

smopen(io)
	register struct iob *io;
{

	register int		cn = UTOCN(io->i_unit); 
	register int 		dr = UTODR(io->i_unit);
	register struct smdevice *smaddr;
	register struct sm_iopb	*iopb;
	struct smuib		*uib = &smuib_tmp.sm_uib;

	if (dr >= SM_DR || cn >= SM_CN) {
		printf("sm: bad unit\n");
		goto err;
	}
	smaddr = smstd[cn];
	if (!probe(&smaddr->sm_cstatus,&smaddr->sm_cstatus)) {
		printf("sm%d: controller %d not present\n", io->i_unit, cn);
		goto err;
	}
	smaddr->sm_cstatus |= CS_BDCLR;
	smaddr->sm_cstatus &= ~CS_BDCLR;
	if (smbusy(smaddr)) {
		printf("sm%d: fail diagnostic 0\n", io->i_unit);
		goto err;
	}
	if ((smaddr->sm_cstatus & CS_BOK) == 0) {
		printf("sm%d: fail diagnostic 1\n", io->i_unit);
		goto err;
	}
	if (smdrdy(smaddr, (dr&0x2)>>1)) {
		printf("sm%d: drive not ready\n", io->i_unit);
		goto err;
	}

	if (sm_qbaddr == 0) {
		csmbuf.i_ma = (long) &smuib_tmp.sm_uib;
		csmbuf.i_cc = 512;
		sm_qbaddr = VDMA_STD(qbsetup(&csmbuf, 1),vbnum);
	}

	smaddr->sm_cstatus &= ~CS_SLED;
	iopb = &smaddr->sm_iopb[0];
	iopb->iopb_mem_type = MEMT_16BIT;
	iopb->iopb_adr_mod = SM_ADDRMOD;
	iopb->iopb_n_vector = iopb->iopb_e_vector = SMD_VECT;
	iopb->iopb_dma_burst = DMA_BURST;
	iopb->iopb_skew_offset = 0;

	iopb->iopb_cmd = CMD_RECALB;
	iopb->iopb_coptions = ((dr&0x2) << 6);
	iopb->iopb_error_code = iopb->iopb_status = 0;
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR) ) {
		printf("sm%d: error RECALB\n", io->i_unit);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		goto err;
	}
	smaddr->sm_cstatus &= ~CS_DONE;

	bcopy(&smuib_dflt, uib, sizeof(struct smuib));
	iopb->iopb_cmd = CMD_INIT;
	iopb->iopb_coptions = ((dr&0x2) << 6);
	iopb->iopb_error_code = iopb->iopb_status = 0;
	iopb->iopb_buf_addrh = hiword(sm_qbaddr);
	iopb->iopb_buf_addrl = loword(sm_qbaddr);
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR) ) {
		printf("sm%d: error INITD\n", io->i_unit);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		goto err;
	}
	smaddr->sm_cstatus &= ~CS_DONE;

	iopb->iopb_cmd = CMD_READ;
	iopb->iopb_coptions = ((dr&0x2) << 6) | COP_ECC_EN;
	iopb->iopb_dsk_addr.log.logh = UIB_DSK_ADDR >> 16;
	iopb->iopb_dsk_addr.log.logl = UIB_DSK_ADDR & 0xFFFF;
	iopb->iopb_blk_count = 1;
	iopb->iopb_buf_addrh = hiword(sm_qbaddr);
	iopb->iopb_buf_addrl = loword(sm_qbaddr);
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR) ) {
		printf("sm%d: error reading UIB %x %x\n", io->i_unit,
		    smaddr->sm_DSTATUS((dr&0x2)>>1), iopb->iopb_error_code);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		goto err;
	}
	smaddr->sm_cstatus &= ~CS_DONE;
	if (uib->uib_bytes_sec == 0) {
		printf("sm%d: not formatted\n", io->i_unit);
		goto err;
	}
	if (uib->uib_ext == UIB_MAGIC)
		uib->uib_ext = 0;
	iopb->iopb_cmd = CMD_INIT;
	iopb->iopb_coptions = (dr&0x2) << 6;
	if (xbad144) {
		uib->uib_retries = 1;
		uib->uib_attrib &= ~(UIB_AT_CE | UIB_AT_RSK);
	}
	iopb->iopb_buf_addrh = hiword(sm_qbaddr);
	iopb->iopb_buf_addrl = loword(sm_qbaddr);
	iopb->iopb_mem_type = MEMT_16BIT;
	iopb->iopb_adr_mod = SM_ADDRMOD;
	iopb->iopb_dma_burst = DMA_BURST;
	iopb->iopb_skew_offset = 0;
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR) ) {
		printf("sm%d: error INIT\n", io->i_unit);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		goto err;
	}
	smaddr->sm_cstatus &= ~CS_DONE;
	if (uib->uib_vol_head[dr&1].num_heads == 0) {
		printf("sm%d: null volume\n", io->i_unit);
		goto err;
	}
	io->i_st.nsect = uib->uib_sec_trk;
	io->i_st.ntpc = uib->uib_vol_head[dr&1].num_heads;
	io->i_st.nspc = io->i_st.nsect * io->i_st.ntpc;
	io->i_st.nbsec = 512;
	io->i_st.ncyl = uib->uib_num_cyl - 1;	/* reserve 1 cylinder for UIB */
	io->i_st.off = NULL;
	printf("(%dx%dx%d)\n",io->i_st.nsect,io->i_st.ntpc,io->i_st.ncyl); /**/
	if ((io->i_boff = diskpart(io)) < 0) {
		printf("sm%d: bad offset\n", io->i_unit);
		goto err;
	}
	return (0);
   err:	return (-1);
}

smstrategy(io, func)
	register struct iob *io;
{
	register int 			bn = io->i_bn;
	register int 			dr = UTODR(io->i_unit);

	switch (func) {
	  case READ:
		func = CMD_READ; 
		break;
	  case WRITE:
		func = CMD_WRITE; 
		break;
	  default:
		printf("sm%d: func %x\n", io->i_unit, func);
		return (-1);
	}
	if (forbb) {
		if (smstrat(io,bn,io->i_cc,func) == -1)
			return (-1);
		io->i_ma += io->i_cc;
	} else if (badstrat(io,bn,io->i_cc,func,smstrat,CMD_READ) == -1)
		return (-1);
	return (io->i_cc);
}

smstrat(io,bn,cc,func)
	register struct iob *io;
	register int bn;
	int func;
{
	register int 			cn = UTOCN(io->i_unit);
	register int 			dr = UTODR(io->i_unit);
	register struct smdevice 	*smaddr = smstd[cn];
	register struct sm_iopb		*iopb = &smaddr->sm_iopb[0];
	static int 			errcnt = 0;
	int				i;
	unsigned long		qbinfo;

	while (1) {
		iopb->iopb_cmd = func;
			iopb->iopb_coptions = ((dr&0x3) << 6) | COP_LOG_TRAN;
		if (!xbad144)
			iopb->iopb_coptions |= COP_ECC_EN;
		if (smrupt[UTOCN(io->i_unit)])
			iopb->iopb_coptions |= COP_INT_EN;
		i = bn + (io->i_st.nsect * io->i_st.ntpc);
		iopb->iopb_dsk_addr.log.logh = hiword(i);
		iopb->iopb_dsk_addr.log.logl = loword(i);
		iopb->iopb_blk_count = (cc + 511) >> 9;
		qbinfo = VDMA_STD(qbsetup(io, 0),vbnum);
		iopb->iopb_buf_addrh = hiword(qbinfo);
		iopb->iopb_buf_addrl = loword(qbinfo);
		if ((SM_CPUMEMT == MEMT_32BIT) && ((loword(io->i_ma)&0x3) == 0))
			iopb->iopb_mem_type = MEMT_32BIT;
		else
			iopb->iopb_mem_type = MEMT_16BIT;
		iopb->iopb_adr_mod = SM_ADDRMOD;
		iopb->iopb_dma_burst = DMA_BURST;
		iopb->iopb_skew_offset = 0;
		smaddr->sm_cstatus &= ~CS_DONE;
		smaddr->sm_cstatus |= CS_GO;
		if (smrupt[UTOCN(io->i_unit)]) {
			return(0);
		}
		if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR)) {
			i = (iopb->iopb_dsk_addr.log.logh << 16) |
			    iopb->iopb_dsk_addr.log.logl;
			printf("sm%d: sn%d HARD %s %x\n",
				io->i_unit, i - io->i_st.nspc,
				(func == CMD_READ) ? "READ" : "WRITE", 
				iopb->iopb_error_code);
			smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
			if (!xbad144 && ++errcnt < SM_RETRYCNT)
				continue;
			errcnt = 0;
			return (-1);
		} else if (iopb->iopb_status == STA_COMP_EX) {
			i = (iopb->iopb_dsk_addr.log.logh << 16) |
			    iopb->iopb_dsk_addr.log.logl;
			printf("sm%d: sn%d SOFT %s %b\n",
				io->i_unit, i - io->i_st.nspc,
				(func == CMD_READ) ? "READ" : "WRITE", 
				iopb->iopb_error_code, ERR_COR_BITS);
			if (xbad144_soft) {
				xbad144_hadsoft = 1;
				return (-1);
			}
		}
		smaddr->sm_cstatus &= ~CS_DONE;
		return (0);
	}
}

smioctl(io)
	register struct iob *io;
{
	smrupt[UTOCN(io->i_unit)] = 1;
}

smbusy(smaddr)
	register struct smdevice *smaddr;
{ 	
	register int l = 50000;

	DELAY(400);
	while (smaddr->sm_cstatus & CS_BUSY)  {
		if (l-- == 0)
			return (-1);
		DELAY(400);
	}
	return (0);
}

smwait(smaddr)
	register struct smdevice *smaddr;
{ 	
	register int l = 50000;

	DELAY(400);
	while ((smaddr->sm_cstatus & CS_DONE) == 0)
		if (l-- == 0) {
			printf("sm: timeout\n");
			return (-1);
		} else
			DELAY(400);
	return (0);
}

smdrdy(smaddr, dr)
	register struct smdevice *smaddr;
{ 	
	register int l = 50000;

	DELAY(400);
	while ((smaddr->sm_DSTATUS(dr)&DS_DREADY)==0)
		if (l-- == 0)
			return (-1);
		else
			DELAY(400);
	return (0);
}

#ifdef	DISK_FORMAT
static char buf[100];
	
smformat()
{
	register struct smdevice *smaddr;
	register struct sm_iopb	*iopb;
	register struct smuib	*uib = &smuib_tmp.sm_uib;
	register char *p;
	register int i, j, n, t;
	int nheads, ncyl, fc;
	char test;
	u_short drive = 0x0;
	int magic = 0;

	printf("\nFormatting SM disk\n");
loop:	printf("\nFunctions available:\n");
	printf("  a) Format Disk\n");
	printf("  b) Read UIB\n");
	printf("  c) Modify UIB\n\n");
	printf("Enter letter of desired function ('H' for help): ");
	stripwhite(gets(buf));
	test = buf[0];
	if (test == 'H')
		goto loop;
	if (test != 'a' && test != 'A' && test != 'b' && test != 'B' &&
	    test != 'c' && test != 'C')
		goto loop;
cntlr:  printf("Controller number (0-%d) ? ",SM_CN-1);
	stripwhite(gets(buf));
	i = getnum(buf);
	if (i < 0 || i >= SM_CN) {
		printf("illegal controller number\n");
		goto cntlr;
	}
	smaddr = smstd[i];
	if (!probe(&smaddr->sm_cstatus,&smaddr->sm_cstatus)) {
		printf("controller %d at 0x%x not present\n", i, smaddr);
		goto cntlr;
	}
	if ((smaddr->sm_cstatus & CS_BOK) == 0) {
		printf("CONTROLLERS SELF DIAGNOSTICS FAILED\n");
		goto cntlr;
	}

	if (sm_qbaddr == 0) {
		csmbuf.i_ma = (long) &smuib_tmp.sm_uib;
		csmbuf.i_cc = 512;
		sm_qbaddr = VDMA_STD(qbsetup(&csmbuf, 1),vbnum);
	}

	smaddr->sm_cstatus &= ~CS_SLED;
	iopb = &smaddr->sm_iopb[0];

drv:    printf("Drive number (0-%d) ? ",SM_DR-1);
	stripwhite(gets(buf));
	drive = getnum(buf);
	if (drive < 0 || drive >= SM_DR) {
		printf("illegal drive number\n");
		goto drv;
	}
	drive = (drive&1) << 7;
	if (test == 'a' || test == 'A') {
		bzero(uib, sizeof (struct smuib));
		smaskuib(uib, 0);
		if (uib->uib_ext == UIB_MAGIC) {
			magic = uib->uib_ext;
			uib->uib_ext = 0;
		} else
			magic = 0;
		iopb->iopb_cmd = CMD_INIT;
		iopb->iopb_coptions = drive | COP_LOG_TRAN | COP_ECC_EN;
		iopb->iopb_buf_addrh = ((int)sm_qbaddr) >> 16;
		iopb->iopb_buf_addrl = ((int)sm_qbaddr) & 0xFFFF;
		iopb->iopb_mem_type = MEMT_16BIT;
		iopb->iopb_adr_mod = SM_ADDRMOD;
		iopb->iopb_dma_burst = DMA_BURST;
		iopb->iopb_skew_offset = 0;
		smaddr->sm_cstatus |= CS_GO;
		smwait(smaddr);
		if (smaddr->sm_cstatus & CS_ERR) {
			printf("Would not init\n");
			goto err;
		}
		smaddr->sm_cstatus &= ~CS_DONE;
		
		if (magic)
			uib->uib_ext = magic;
		if (uib->uib_vol_head[0].num_heads) {
		  printf(  "Cylinder Volume 0\n");
		  for ( i = 0 ; i < uib->uib_num_cyl; i++) {
		    printf("\r%d",i);
		    for (j = uib->uib_vol_head[0].start_head ; 
			j < uib->uib_vol_head[0].start_head + 
			uib->uib_vol_head[0].num_heads; j++) {
			iopb->iopb_cmd = CMD_FMT_TRK;
			iopb->iopb_coptions = drive | COP_ECC_EN;
			iopb->iopb_dsk_addr.log.logh = 0;
			iopb->iopb_dsk_addr.log.logl = 0;
			iopb->iopb_blk_count = 0;
			iopb->iopb_dsk_addr.phys.cyl = i;
			iopb->iopb_dsk_addr.phys.hd = j;
			t = (iopb->iopb_dsk_addr.log.logh << 16) |
			    iopb->iopb_dsk_addr.log.logl;
			iopb->iopb_buf_addrh =  t >> 16;
			iopb->iopb_buf_addrl =  t & 0xFFFF;
			iopb->iopb_skew_offset = 0;
			smaddr->sm_cstatus |= CS_GO;
			if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR))
				goto err;
			smaddr->sm_cstatus &= ~CS_DONE;
		    }
		  }
		}
		if (uib->uib_vol_head[1].num_heads) {
		  printf(  "\nCylinder Volume 1\n");
		  for ( i = 0 ; i < uib->uib_num_cyl; i++) {
		    printf("\r%d",i);
		    for (j = uib->uib_vol_head[1].start_head ; 
			j < uib->uib_vol_head[1].start_head + 
			uib->uib_vol_head[1].num_heads; j++) {
			iopb->iopb_cmd = CMD_FMT_TRK;
			iopb->iopb_coptions = drive | COP_VOLUME_N | COP_ECC_EN;
			iopb->iopb_dsk_addr.log.logh = 0;
			iopb->iopb_dsk_addr.log.logl = 0;
			iopb->iopb_blk_count = 0;
			iopb->iopb_dsk_addr.phys.cyl = i;
			iopb->iopb_dsk_addr.phys.hd = j;
			t = (iopb->iopb_dsk_addr.log.logh << 16) |
			    iopb->iopb_dsk_addr.log.logl;
			iopb->iopb_buf_addrh =  t >> 16;
			iopb->iopb_buf_addrl =  t & 0xFFFF;
			iopb->iopb_skew_offset = 0;
			smaddr->sm_cstatus |= CS_GO;
			if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR))
				goto err;
			smaddr->sm_cstatus &= ~CS_DONE;
		    }
		  }
		}
		if (smaddr->sm_cstatus & CS_ERR) {
err:			printf("\nERROR ");
			printf("cstatus=%b dstat=%b %b iopb: stat %x ec %x\n",
				smaddr->sm_cstatus, CS_BITS,
				smaddr->sm_dstatus[0], DS_BITS,
				smaddr->sm_dstatus[1], DS_BITS,
				iopb->iopb_status, iopb->iopb_error_code);
			smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		} else {
			iopb->iopb_cmd = CMD_WRITE;
			iopb->iopb_coptions = drive | COP_ECC_EN;
			iopb->iopb_dsk_addr.log.logh = UIB_DSK_ADDR >> 16;
			iopb->iopb_dsk_addr.log.logl = UIB_DSK_ADDR & 0xFFFF;
			iopb->iopb_blk_count = 1;
			iopb->iopb_buf_addrh = ((int)sm_qbaddr) >> 16;
			iopb->iopb_buf_addrl = ((int)sm_qbaddr) & 0xFFFF;
			iopb->iopb_mem_type = MEMT_16BIT;
			iopb->iopb_adr_mod = SM_ADDRMOD;
			iopb->iopb_dma_burst = DMA_BURST;
			iopb->iopb_skew_offset = 0;
			smaddr->sm_cstatus |= CS_GO;
			if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR))
				goto err;
			smaddr->sm_cstatus &= ~CS_DONE;
			printf("\nSUCCESSFUL FORMAT\n");
		}
	} else if (test == 'b' || test == 'B') {
		iopb->iopb_cmd = CMD_READ;
		iopb->iopb_coptions = drive | COP_ECC_EN;
		iopb->iopb_dsk_addr.log.logh = UIB_DSK_ADDR >> 16;
		iopb->iopb_dsk_addr.log.logl = UIB_DSK_ADDR & 0xFFFF;
		iopb->iopb_buf_addrh = ((int)sm_qbaddr) >> 16;
		iopb->iopb_buf_addrl = ((int)sm_qbaddr) & 0xFFFF;
		iopb->iopb_blk_count = 1;
		iopb->iopb_mem_type = MEMT_16BIT;
		iopb->iopb_adr_mod = SM_ADDRMOD;
		iopb->iopb_dma_burst = DMA_BURST;
		iopb->iopb_skew_offset = 0;
		smaddr->sm_cstatus |= CS_GO;
		if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR))
			goto err;
		smaddr->sm_cstatus &= ~CS_DONE;

print_uib:	if (uib->uib_bytes_sec == 0) {
			printf("BAD UIB\n");
			goto loop;
		}
		printf(" [0].start_head  %d\n", uib->uib_vol_head[0].start_head);
		printf(" [0].num_heads   %d\n", uib->uib_vol_head[0].num_heads);
		printf(" [1].start_head  %d\n", uib->uib_vol_head[1].start_head);
		printf(" [1].num_heads   %d\n", uib->uib_vol_head[1].num_heads);
		printf(" sec_trk         %d\n", uib->uib_sec_trk);
		printf(" skew_factor     %d\n", uib->uib_skew_factor);
		printf(" bytes_sec       %d\n", uib->uib_bytes_sec);
		printf(" gap1_words      %d\n", uib->uib_gap1_words);
		printf(" gap2_words      %d\n", uib->uib_gap2_words);
		printf(" sct_int         %d\n", uib->uib_sct_int);
		printf(" retries         %d\n", uib->uib_retries);
		printf(" num_cyl         %d\n", uib->uib_num_cyl);
		printf(" attrib          %b\n", uib->uib_attrib, UIB_AT_BITS);
		if (uib->uib_ext == UIB_MAGIC)
		    printf(" ext             MAGIC\n");
		else
		    printf(" ext             %b\n", uib->uib_ext, UIB_EXT_BITS);
		printf(" d0_level        %d\n", uib->uib_d0_level);
		printf(" d0_vector       %d\n", uib->uib_d0_vector);
	} else if (test == 'c' || test == 'C') {
		iopb->iopb_cmd = CMD_READ;
		iopb->iopb_coptions = drive | COP_ECC_EN;
		iopb->iopb_dsk_addr.log.logh = UIB_DSK_ADDR >> 16;
		iopb->iopb_dsk_addr.log.logl = UIB_DSK_ADDR & 0xFFFF;
		iopb->iopb_buf_addrh = ((int)sm_qbaddr) >> 16;
		iopb->iopb_buf_addrl = ((int)sm_qbaddr) & 0xFFFF;
		iopb->iopb_blk_count = 1;
		iopb->iopb_mem_type = MEMT_16BIT;
		iopb->iopb_adr_mod = SM_ADDRMOD;
		iopb->iopb_dma_burst = DMA_BURST;
		iopb->iopb_skew_offset = 0;
		smaddr->sm_cstatus |= CS_GO;
		if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR))
			goto err;
		smaddr->sm_cstatus &= ~CS_DONE;

		if (uib->uib_bytes_sec == 0) {
			printf("BAD UIB\n");
			goto loop;
		}
		smaskuib(uib, 1);

		if (uib->uib_ext == UIB_MAGIC) {
			magic = uib->uib_ext;
			uib->uib_ext = 0;
		} else
			magic = 0;
		iopb->iopb_cmd = CMD_INIT;
		iopb->iopb_coptions = drive | COP_LOG_TRAN | COP_ECC_EN;
		iopb->iopb_buf_addrh = ((int)sm_qbaddr) >> 16;
		iopb->iopb_buf_addrl = ((int)sm_qbaddr) & 0xFFFF;
		iopb->iopb_mem_type = MEMT_16BIT;
		iopb->iopb_adr_mod = SM_ADDRMOD;
		iopb->iopb_dma_burst = DMA_BURST;
		iopb->iopb_skew_offset = 0;
		smaddr->sm_cstatus |= CS_GO;
		smwait(smaddr);
		if (smaddr->sm_cstatus & CS_ERR) {
			printf("Would not init\n");
			goto err;
		}
		smaddr->sm_cstatus &= ~CS_DONE;
		
		if (magic)
			uib->uib_ext = magic;
		iopb->iopb_cmd = CMD_WRITE;
		iopb->iopb_coptions = drive | COP_ECC_EN;
		iopb->iopb_dsk_addr.log.logh = UIB_DSK_ADDR >> 16;
		iopb->iopb_dsk_addr.log.logl = UIB_DSK_ADDR & 0xFFFF;
		iopb->iopb_blk_count = 1;
		iopb->iopb_buf_addrh = ((int)sm_qbaddr) >> 16;
		iopb->iopb_buf_addrl = ((int)sm_qbaddr) & 0xFFFF;
		iopb->iopb_mem_type = MEMT_16BIT;
		iopb->iopb_adr_mod = SM_ADDRMOD;
		iopb->iopb_dma_burst = DMA_BURST;
		iopb->iopb_skew_offset = 0;
		smaddr->sm_cstatus |= CS_GO;
		if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR))
			goto err;
		smaddr->sm_cstatus &= ~CS_DONE;
		goto print_uib;
	}
	goto loop;
}

smaskuib(uib, ask)
	register struct smuib	*uib;
{
	printf("Start head volume 0 ?          ");
	if (ask)
		printf("%8d: ", uib->uib_vol_head[0].start_head);
    	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_vol_head[0].start_head = getnum(buf);

	printf("Number of heads volume 0 ?     ");
	if (ask)
		printf("%8d: ",uib->uib_vol_head[0].num_heads);
    	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_vol_head[0].num_heads = getnum(buf);

	printf("Start head volume 1 ?          ");
	if (ask)
		printf("%8d: ",uib->uib_vol_head[1].start_head);
    	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_vol_head[1].start_head = getnum(buf);

	printf("Number of heads volume 1 ?     ");
	if (ask)
		printf("%8d: ",uib->uib_vol_head[1].num_heads);
    	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_vol_head[1].num_heads = getnum(buf);

	printf("Number of cylinders ?          ");
	if (ask)
		printf("%8d: ",uib->uib_num_cyl);
    	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_num_cyl = getnum(buf);

	printf("Number of sectors per track ?  ");
	if (ask)
		printf("%8d: ",uib->uib_sec_trk);
	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_sec_trk = getnum(buf);

	printf("Amount of cylinder skew ?      ");
	if (ask)
		printf("%8d: ",uib->uib_skew_factor);
	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_skew_factor = getnum(buf);

	printf("Size of gap 1 ?                ");
	if (ask)
		printf("%8d: ",uib->uib_gap1_words);
	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_gap1_words = getnum(buf);

	printf("Size of gap 2 ?                ");
	if (ask)
		printf("%8d: ",uib->uib_gap2_words);
	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_gap2_words = getnum(buf);

	printf("Sector interleave factor ?     ");
	if (ask)
		printf("%8d: ",uib->uib_sct_int);
	stripwhite(gets(buf));
	if (buf[0])
		uib->uib_sct_int = getnum(buf);

	printf("Insert UIB magic number ?      ");
	if (ask)
		printf("%s: ",uib->uib_ext == UIB_MAGIC ? "     yes" : "      no");
	stripwhite(gets(buf));
	if (buf[0] == 'y' || buf[0] == 'Y')
		uib->uib_ext = UIB_MAGIC;
	else {
	    uib->uib_ext = 0;
	    printf("SMD-E interface (y/n)?         ");
	    if (ask)
		printf("%s: ",uib->uib_ext & UIB_EXT_SMD_E ? "     yes" : "      no");
	    stripwhite(gets(buf));
	    if (buf[0] == 'y' || buf[0] == 'Y')
		    uib->uib_ext |= UIB_EXT_SMD_E;
	    else
		    uib->uib_ext &= ~UIB_EXT_SMD_E;
	}

	/* HARD CODED VALUES */
	uib->uib_bytes_sec = 512;
	uib->uib_retries = 10;
	uib->uib_attrib = UIB_AT_CE | UIB_AT_RSK | UIB_AT_INH;
}
#endif	DISK_FORMAT
