/*
 *   Integrated Solutions SIO I/O Routines
 */

#include "../h/param.h"
#include "../machine/board.h"
#include "../dev/sioreg.h"

int	putlocal(), wputlocal(), putremote(), wputremote();
int	getloc(), getlocal(), wgetlocal(), getrem(), getremote(), wgetremote();
int	getchar(),  putchar();

int (*putch)() = putchar;
int (*getch)() = getchar;
int putslow = 0;
int autoboot = 0; 
extern	int	(* user_exit)();

#define CTLQ 	 0x11
#define CTLS 	 0x13
static char lochar = 0, rmchar = 0;

#ifdef	BSR_KEY_BOARD
static int iws_keyboard = 1;
static int iws_keyboard_ready = 0;
#endif	BSR_KEY_BOARD

#define SPLOCAL ((struct siochan *)OCTART_ADDR)
#define SPREMOTE ((struct siochan *)(OCTART_ADDR+0x08))

getchar ()
{
	register int c;

	c = wgetlocal();
	if (c == '\r')
		wputlocal('\n');
	putchar(c);
	return c;
}

putchar (c)
register int c;
{
	if (autoboot)
		return;
	wputlocal(c);
	if (c == '\n')
		wputlocal('\r');
}

/* wait for and return a character from the remote */
wgetremote()
{
	register int c;

	while ((c = getremote()) == 0) ;
	return(c);
}

/* wait for and return a character from the local */
wgetlocal()
{
	register int c;

	while ((c = getlocal()) == 0) ;
		return(c);
}

/* wait for local to be ready and output a charater */
wputlocal(c)
{
	register int ic, buf[2];

#ifdef	BSR_KEY_BOARD
	if (iws_keyboard && !iws_keyboard_ready) {
		if (((getbsr()&BSR_KEY_BOARD) == BSR_KEY_IWS))
			iws_keyboard_init();
		else
			iws_keyboard = 0;
	}
#endif	BSR_KEY_BOARD
	if ((ic = getloc()) != 0) {
		if (ic == CTLS)
			while (getloc() != CTLQ) ;
		else if (ic != CTLQ) {
			lochar = ic;
			if((ic == '' || ic == '') && user_exit){
				buf[0] = ic;
				lochar = 0;
				buf[1] = '\0';
				(* user_exit)(buf);
			}
		}
	}
	while (putlocal(c) == 0) ;
}

/* wait for remote to be ready and output a charater */
wputremote(c)
{
	register int ic;

	if ((ic = getrem()) != 0) {
		if (ic == CTLS)
			while (getrem() != CTLQ) ;
		else if (ic != CTLQ)
			rmchar = ic;
	}

	while (putremote(c) == 0) ;
}

/*
 * getremote()
 *	get a character from the remote port
 *	do not wait! return 0 if there is none
 */
getremote()
{
	register char c;

	if (rmchar) {
		c = rmchar;
		rmchar = 0;
		return(c);
	} else
		return(getrem());
}

/*
 * getlocal()
 *	get a character from the local port
 *	do not wait! return 0 if there is none
 */
getlocal()
{
	register char c;
	
	if (lochar) {
		c = lochar;
		lochar = 0;
		return(c);
	} else
		return(getloc());
}

getloc(x)
{
	register int c;

#ifdef	BSR_KEY_BOARD
	if (iws_keyboard && !iws_keyboard_ready) {
		if (((getbsr()&BSR_KEY_BOARD) == BSR_KEY_IWS)) {
			iws_keyboard_init();
			return 0;
		} else
			iws_keyboard = 0;
	}
#endif	BSR_KEY_BOARD
	if ((afcrb(&SPLOCAL->chan_sr_csr) & SR_RXRDY) == 0)
		return (0);
	c = afcrb(&SPLOCAL->chan_rhr_thr);
#ifdef	BSR_KEY_BOARD
	if (iws_keyboard)
		c = iws_keyboard_map(c);
#endif	BSR_KEY_BOARD
	return (c & 0x7F);
}

/*
 * putlocal()
 *	put a character to the local port
 *	do not wait! return 0 if it is not ready
 */
putlocal(c)
char c;
{
	putslow = 0;
	if (putloc(c) == 0)
		return 0;
	g_putc(c);		/* graphics putc */
	return 1;
}

putloc(c)
char c;
{
	if ((afcrb(&SPLOCAL->chan_sr_csr) & SR_TXRDY) == 0)
		return 0;
	afcwb(&SPLOCAL->chan_rhr_thr, c);
	return 1;
}

getrem()
{
	if ((afcrb(&SPREMOTE->chan_sr_csr) & SR_RXRDY) == 0)
		return (0);
	return(afcrb(&SPREMOTE->chan_rhr_thr) & 0x7F);
}

/*
 * putremote()
 *	put a character to the remote port
 *	do not wait! return 0 if it is not ready
 */
putremote(c)
char c;
{
	register int i;

	if ((afcrb(&SPREMOTE->chan_sr_csr) & SR_TXRDY) == 0)
		return (0);
	afcwb(&SPREMOTE->chan_rhr_thr, c);
	if (putslow)
		DELAY(2000);
	return(1);
}

#ifdef	BSR_KEY_BOARD
/* IWS keyboard mapping for standalone code.  */

#define     BADCODE     0	/* ignore this character */
#define	    LSHIFTDN	0x68	/* left shift down */
#define	    LSHIFTUP	0xe8	/* left shift up */
#define	    RSHIFTDN	0x37	/* right shift down */
#define	    RSHIFTUP	0xb7	/* right shift up */
#define	    MSHIFTDN	0x76	/* mouse shift down */
#define	    MSHIFTUP	0xf6	/* mouse shift up */
#define	    LCTRLDN     0x2b	/* left control down */
#define	    LCTRLUP	0xab	/* left control up */
#define	    RCTRLDN	0x45	/* right control down */
#define	    RCTRLUP	0xc5	/* right control up */
#define	    MCTRLDN	0x77	/* mouse control down */
#define	    MCTRLUP	0xf7	/* mouse control up */
#define	    SQUEAKDN	0x75	/* squeak button down */
#define	    SQUEAKUP	0xf5	/* squeak button up */

static char table [] = { /* input translation table */
    BADCODE,			/* 00, No key, null map */
    BADCODE,			/* 01, Select Key */
    BADCODE,			/* 02, Numeric pad '.' key */
    BADCODE,			/* 03, Mark key */
    BADCODE,			/* 04, Center arrow key, null map */
    ' ',			/* 05, Space bar */
    '0',			/* 06, '0' key */
    'p',			/* 07, 'p' key */
    '//',			/* 08, '/' key */
    BADCODE,			/* 09, Soft function key 6 */
    ';',			/* 0a, ';' key */
    BADCODE,			/* 0b, Numeric pad '/' key */
    BADCODE,			/* 0c, Numeric pad '*' key */
    BADCODE,			/* 0d, Calc key */
    BADCODE,			/* 0e, Numeric pad 2nd key */
    BADCODE,			/* 0f, Numeric pad '7' key */
    BADCODE,			/* 10, Numeric pad '0' key */
    BADCODE,			/* 11, Numeric pad '4' key */
    BADCODE,			/* 12, Numeric pad '1' key */
    BADCODE,			/* 13, Numeric pad '3' key */
    BADCODE,			/* 14, Numeric pad '2' key */
    0x7f,			/* 15, Backspace key */
    BADCODE,			/* 16, Enter key */
    BADCODE,			/* 17, Up arrow key */
    BADCODE,			/* 18, Right arrow key */
    BADCODE,			/* 19, Down arrow key */
    BADCODE,			/* 1a, Numeric pad '6' key */
    BADCODE,			/* 1b, Numeric pad '5' key */
    BADCODE,			/* 1c, Prev key */
    BADCODE,			/* 1d, Numeric pad '+' key */
    BADCODE,			/* 1e, Delete key */
    BADCODE,			/* 1f, Goto key */
    BADCODE,			/* 20, Open key */
    BADCODE,			/* 21, Insert key */
    BADCODE,			/* 22, Numeric pad '9' key */
    BADCODE,			/* 23, Numeric pad '8' key */
    BADCODE,			/* 24, Next key */
    BADCODE,			/* 25, Numeric pad '-' key */
    BADCODE,			/* 26, Undo key */
    BADCODE,			/* 27, Run key */
    BADCODE,			/* 28, Close key */
    BADCODE,			/* 29, View key */
    's',			/* 2a, 's' key */
    BADCODE,			/* 2b, left control key */
    '4',			/* 2c, '4' key */
    'x',			/* 2d, 'x' key */
    'r',			/* 2e, 'r' key */
    'v',			/* 2f, 'v' key */
    BADCODE,			/* 30, Left arrow key */
    'f',			/* 31, 'f' key */
    'z',			/* 32, 'z' key */
    '`',			/* 33, '`' key right */
    '=',			/* 34, '=' key */
    'a',			/* 35, 'a' key */
    ']',			/* 36, ']' key */
    BADCODE,			/* 37, Shift right key */
    BADCODE,			/* 38, Soft function key 8 */
    '\n',			/* 39, Return key */
    '2',			/* 3a, '2' key */
    '1',			/* 3b, '1' key */
    BADCODE,		    	/* 3c, Soft function key 2 */
    'w',			/* 3d, 'w' key */
    '3',			/* 3e, '3' key */
    'q',			/* 3f, 'q' key */
    'e',			/* 40, 'e' key */
    'c',			/* 41, 'c' key */
    BADCODE,			/* 42, Soft function key 3 */
    'd',			/* 43, 'd' key */
    BADCODE,			/* 44, Soft function key 1 */
    BADCODE,			/* 45, Ctrl key right */
    '-',			/* 46, '-' key */
    '\t',			/* 47, Tab key */
    '[',			/* 48, '[' key */
    BADCODE,			/* 49, Soft function key 7 */
    '\'',			/* 4a, ''' key */
    BADCODE,			/* 4b, Again key */
    BADCODE,			/* 4c, Stop key */
    '6',			/* 4d, '6' key */
    0x1b,			/* 4e, Escape key */
    'y',			/* 4f, 'y' key */
    'n',			/* 50, 'n' key */
    'h',			/* 51, 'h' key */
    BADCODE,			/* 52, Find key */
    BADCODE,			/* 53, Aide key */
    '9',			/* 54, '9' key */
    BADCODE,			/* 55, Window key */ 
    'o',			/* 56, 'o' key */
    '.',			/* 57, '.' key */
    'l',			/* 58, 'l' key */
    BADCODE,			/* 59, Copy key */
    BADCODE,			/* 5a, Keys key */
    '7',			/* 5b, '7' key */
    BADCODE,			/* 5c, Device key */
    'u',			/* 5d, 'u' key */
    'm',			/* 5e, 'm' key */
    BADCODE,			/* 5f, Help key */
    'j',			/* 60, 'j' key */
    BADCODE,			/* 61, Move key */
    BADCODE,			/* 62, Script key */
    '8',			/* 63, '8' key */
    'i',			/* 64, 'i' key */
    ',',			/* 65, ',' key */
    BADCODE,			/* 66, Soft function key 5 */
    'k',			/* 67, 'k' key */
    BADCODE,			/* 68, Shift left key */
    '\\',		        /* 69, '\' key left */
    '5',			/* 6a, '5' key */
    BADCODE,			/* 6b, Caps lock key */
    't',			/* 6c, 't' key */
    'b',			/* 6d, 'b' key */
    BADCODE,    		/* 6e, Soft function key 4 */
    'g',			/* 6f, 'g' key */
    BADCODE,			/* 70, Mouse button down */
    BADCODE,			/* 71, Mouse shift down, no map */
    BADCODE,			/* 72, Mouse ctrl down, no map */
    BADCODE,			/* 73, No mapping for now */
    BADCODE,			/* 74, No mapping for now */
    BADCODE,			/* 75, No mapping for now */
    BADCODE,			/* 76, No mapping for now */
    BADCODE,			/* 77, No mapping for now */
    BADCODE,			/* 78, No mapping for now */
    BADCODE,			/* 79, No mapping for now */
    BADCODE,			/* 7a, No mapping for now */
    BADCODE,			/* 7b, No mapping for now */
    BADCODE,			/* 7c, No mapping for now */
    BADCODE,			/* 7d, No mapping for now */
    BADCODE,			/* 7e, No mapping for now */
    BADCODE,			/* 7f, No mapping for now */

    BADCODE,	/* Begin shift plane - 00, No key, null map */
    BADCODE,			/* 01, Select Key */
    BADCODE,			/* 02, Numeric pad '.' key */
    BADCODE,			/* 03, Mark key */
    BADCODE,			/* 04, Center arrow key, null map */
    ' ',			/* 05, Space bar */
    ')',			/* 06, ')' key */
    'P',			/* 07, 'p' key */
    '?',			/* 08, '/' key */
    BADCODE,			/* 09, Soft function key 6 */
    ':',			/* 0a, ';' key */
    BADCODE,			/* 0b, Numeric pad '/' key */
    BADCODE,			/* 0c, Numeric pad '*' key */
    BADCODE,			/* 0d, Calc key */
    BADCODE,			/* 0e, Numeric pad 2nd key */
    BADCODE,			/* 0f, Numeric pad '7' key */
    BADCODE,			/* 10, Numeric pad '0' key */
    BADCODE,			/* 11, Numeric pad '4' key */
    BADCODE,			/* 12, Numeric pad '1' key */
    BADCODE,			/* 13, Numeric pad '3' key */
    BADCODE,			/* 14, Numeric pad '2' key */
    0x7f,			/* 15, Backspace key */
    BADCODE,			/* 16, Enter key */
    BADCODE,			/* 17, Up arrow key */
    BADCODE,			/* 18, Right arrow key */
    BADCODE,			/* 19, Down arrow key */
    BADCODE,			/* 1a, Numeric pad '6' key */
    BADCODE,			/* 1b, Numeric pad '5' key */
    BADCODE,			/* 1c, Prev key */
    BADCODE,			/* 1d, Numeric pad '+' key */
    BADCODE,			/* 1e, Delete key */
    BADCODE,			/* 1f, Goto key */
    BADCODE,			/* 20, Open key */
    BADCODE,			/* 21, Insert key */
    BADCODE,			/* 22, Numeric pad '9' key */
    BADCODE,			/* 23, Numeric pad '8' key */
    BADCODE,			/* 24, Next key */
    BADCODE,			/* 25, Numeric pad '-' key */
    BADCODE,			/* 26, Undo key */
    BADCODE,			/* 27, Run key */
    BADCODE,			/* 28, Close key */
    BADCODE,			/* 29, View key */
    'S',			/* 2a, 's' key */
    BADCODE,			/* 2b, left control key */
    '$',			/* 2c, '4' key */
    'X',			/* 2d, 'x' key */
    'R',			/* 2e, 'r' key */
    'V',			/* 2f, 'v' key */
    BADCODE,			/* 30, Left arrow key */
    'F',			/* 31, 'f' key */
    'Z',			/* 32, 'z' key */
    '~',			/* 33, '~' key right */
    '+',			/* 34, '=' key */
    'A',			/* 35, 'a' key */
    '}',			/* 36, ']' key */
    BADCODE,			/* 37, Shift right key */
    BADCODE,			/* 38, Soft function key 8 */
    '\n',			/* 39, Return key */
    '@',			/* 3a, '2' key */
    '!',			/* 3b, '1' key */
    BADCODE,		    	/* 3c, Soft function key 2 */
    'W',			/* 3d, 'w' key */
    '#',			/* 3e, '3' key */
    'Q',			/* 3f, 'q' key */
    'E',			/* 40, 'e' key */
    'C',			/* 41, 'c' key */
    BADCODE,			/* 42, Soft function key 3 */
    'D',			/* 43, 'd' key */
    BADCODE,			/* 44, Soft function key 1 */
    BADCODE,			/* 45, Ctrl key right */
    '_',			/* 46, '-' key */
    '\t',			/* 47, Tab key */
    '{',			/* 48, '[' key */
    BADCODE,			/* 49, Soft function key 7 */
    '"',			/* 4a, ''' key */
    BADCODE,			/* 4b, Again key */
    0x03,			/* 4c, Stop key */
    '^',			/* 4d, '6' key */
    0x1b,			/* 4e, Escape key */
    'Y',			/* 4f, 'y' key */
    'N',			/* 50, 'n' key */
    'H',			/* 51, 'h' key */
    BADCODE,			/* 52, Find key */
    BADCODE,			/* 53, Aide key */
    '(',			/* 54, '9' key */
    BADCODE,			/* 55, Window key */ 
    'O',			/* 56, 'o' key */
    '>',			/* 57, '.' key */
    'L',			/* 58, 'l' key */
    BADCODE,			/* 59, Copy key */
    BADCODE,			/* 5a, Keys key */
    '&',			/* 5b, '7' key */
    BADCODE,			/* 5c, Device key */
    'U',			/* 5d, 'u' key */
    'M',			/* 5e, 'm' key */
    BADCODE,			/* 5f, Help key */
    'J',			/* 60, 'j' key */
    BADCODE,			/* 61, Move key */
    BADCODE,			/* 62, Script key */
    '*',			/* 63, '8' key */
    'I',			/* 64, 'i' key */
    '<',			/* 65, ',' key */
    BADCODE,			/* 66, Soft function key 5 */
    'K',			/* 67, 'k' key */
    BADCODE,			/* 68, Shift left key */
    '|',	       		/* 69, '`' key left */
    '%',			/* 6a, '5' key */
    BADCODE,			/* 6b, Caps lock key */
    'T',			/* 6c, 't' key */
    'B',			/* 6d, 'b' key */
    BADCODE,			/* 6e, Soft function key 4 */
    'G',			/* 6f, 'g' key */
    BADCODE,			/* 70, Mouse button down */
    BADCODE,			/* 71, Mouse shift down, no map */
    BADCODE,			/* 72, Mouse ctrl down, no map */
    BADCODE,			/* 73, No mapping for now */
    BADCODE,			/* 74, No mapping for now */
    BADCODE,			/* 75, No mapping for now */
    BADCODE,			/* 76, No mapping for now */
    BADCODE,			/* 77, No mapping for now */
    BADCODE,			/* 78, No mapping for now */
    BADCODE,			/* 79, No mapping for now */
    BADCODE,			/* 7a, No mapping for now */
    BADCODE,			/* 7b, No mapping for now */
    BADCODE,			/* 7c, No mapping for now */
    BADCODE,			/* 7d, No mapping for now */
    BADCODE,			/* 7e, No mapping for now */
    BADCODE,			/* 7f, No mapping for now */
};

/* identify keyboard. */
iws_keyboard_init()
{
	register int i;
	register int c;
	static int iws_disconnected = 0;

	/* flush input */
	while ((afcrb(&SPLOCAL->chan_sr_csr) & SR_RXRDY) != 0)
		c = afcrb(&SPLOCAL->chan_rhr_thr);

	/* output identify request */
	while (putloc(0xff) == 0)
		;

	/* wait for identify reply */
	for (i = 400000; i ; i--)
		if (afcrb(&SPLOCAL->chan_sr_csr) & SR_RXRDY)
			break;
	if (i == 0) {
		printf("\nIWS keyboard disconnected");
		iws_disconnected = 1;
		return;
	}
	c = afcrb(&SPLOCAL->chan_rhr_thr);

	/* check reply code */
	if (c != 0xfe) {
		printf("IWS keyboard error 0x%x\n", c);
		return;
	}

	if (iws_disconnected) {
		iws_disconnected = 0;
		printf("\nIWS keyboard initialized\n\n: ");
	}

	/* turn off all led's */
	while (putloc(0xff) == 0)
		;
	while (putloc(0x00) == 0)
		;

	iws_keyboard_ready = 1;
	return;
}

/* translate keyboard code to ascii. */
iws_keyboard_map(c)
{
	static char shift = 0;		/* shift state */
	static char ctrl = 0;		/* ctrl state */
	int retchar = 0;		/* ascii char to return */

	if (c == 0xfc) {
		printf("IWS keyboard parity error\n");
		return 0;
	}

	switch(c) {			/* switch to correct action */
	    case LSHIFTDN:		/* left  shift key pressed */
	    case RSHIFTDN:		/* right shift key pressed */
		shift++;		/* increment shift count */
		break;

	    case LSHIFTUP:		/* left  shift key released */
	    case RSHIFTUP:		/* right shift key released */
		shift--;		/* decrement shift count */
		break;

	    case LCTRLDN:		/* left  control key pressed */
	    case RCTRLDN:		/* right control key pressed */
		ctrl++;			/* increment ctrl count */
		break;

	    case LCTRLUP:		/* left  control key released */
	    case RCTRLUP:		/* right control key released */
		ctrl--;			/* decrement ctrl count */
		break;

	    case MCTRLDN:		/* mouse control key pressed */
	    case MCTRLUP:		/* mouse control key released */
	    case SQUEAKUP:		/* mouse button gone up       */
	    case SQUEAKDN:		/* mouse button gone down     */
		break;			/* ignore */

	    default:
		if (c & 0x80)		/* throw away upstrokes */
			break;
		if (shift)		/* look in shift table if set */
			c += 128;
		retchar = table[c];
		if (ctrl)		/* convert to control character */
			retchar &= 0x1f;
		break;
	}
	return(retchar);	
}
#endif	BSR_KEY_BOARD
