/*
 * Standalone driver for the Ciprico Rimfire 3x00 disk controllers
 */
#include "saio.h"
#include "sais68k.h"
#include "../dev/rfreg.h"

#define	RFADDR0	  	((struct rf_device *)&vme_short[0x1000])
#define	RFADDR1	  	((struct rf_device *)&vme_short[0x1200])

#define	NRF		2			/* number of controllers */
#define	Nrf		8			/* volumes per controller */
#define	UTOCN(x)	((x) >> 3)
#define	UTODR(x)	((x) & 7)
#define	DRTODR(x)	((x) >> 1)
#define	DRTOVOL(x)	((x) & 1)
#define RF_MAXCNT	MAX_RF_BSIZE

#ifdef	QX
#define	RF_XFER_WID	(0)
#define	RF_ADDRMOD	(PB_ADDRMOD_STD)
#else	QX
#ifdef	M68020
#define	RF_XFER_WID	((((*BSR)&BSR_MEMWIDTH)==BSR_16BITMEM)?0:ABP_CTLAM_WID)
#ifdef	M68025
#define	RF_ADDRMOD	(PB_ADDRMOD_EXT)
#else	M68025
#define	RF_ADDRMOD	(PB_ADDRMOD_STD)
#endif	M68025
#else	M68020
#define	RF_XFER_WID	(0)
#define	RF_ADDRMOD	(PB_ADDRMOD_STD)
#endif	M68020
#endif	QX
#define	GETFIELD(s, v)			v = getfield(s, v, 10, 0, 0, 0)
#define	GETFIELD_MM(s, v, min, max)	v = getfield(s, v, 10, min, max, 0)
#define	GETBITS(s, bit, v)		v = getbits(s, bit, v);

struct rfdevice	*rfstd[NRF] = {RFADDR0, RFADDR1};
int		rf_bsybit[NRF];
int		rf_ctype[NRF];
int		rf_cyl_offset[NRF][Nrf];
int		rf_diskformat = 0;
int		rf_debug = 0;
int		forbb, xbad144, xbad144_soft, rfstrat();
extern int	topmem;

rfopen(io)
	register struct iob *io;
{
	register int	cn = UTOCN(io->i_unit); 
	register int 	dr = UTODR(io->i_unit);
	struct rf_uib	rf_uib;

	if (dr >= Nrf || cn >= NRF) {
	    printf("rf: bad unit\n");
	    return (-1);
	}
	if (rfinit(cn, dr) || rfuib(cn, dr, &rf_uib.uib, PB_COMMAND_READ))
		return (-1);
	io->i_st.nsect	= rf_uib.uib.uib_confd_nspt;
	io->i_st.ntpc	= rf_uib.uib.uib_confd_nhead;
	io->i_st.nspc	= io->i_st.nsect * io->i_st.ntpc;
	io->i_st.nbsec	= rf_uib.uib.uib_confd_nbps;
	io->i_st.ncyl	= rf_uib.uib.uib_confd_ncyl - rf_uib.uib.uib_cyl_offset;
	io->i_st.off	= NULL;
	if ((io->i_boff = diskpart(io)) < 0) {
	    printf("rf: bad offset\n");
	    return (-1);
	}
	rf_cyl_offset[cn][dr] = rf_uib.uib.uib_cyl_offset * io->i_st.nspc;
	return (0);
}

rfinit(cn, dr)
{
	struct	rf_xpb			xpb;
	register struct rf_pb		*pb = &xpb.xpb_pb;
	struct rf_uib			rf_uib;
	register struct uib		*uib = &rf_uib.uib;
	int				shead, nspt, ncyl;

	if (rfreset(cn))
	    return (-1);
	XPB_SET(&xpb, PB_COMMAND_GOPT, PB_UNIT_NODISK, PB_CONTROL_IOCG_0, 0);
	pb->gopt_throttle		= GOPT_THROTTLE;
	if (rfexec("GOPT", &xpb, cn, 1))
	    return (-1);

	if (rf_ctype[cn] == 3200 && DRTOVOL(dr) == 1) {
	    XPB_SET(&xpb, PB_COMMAND_SETCONF, PB_UNIT(dr&~1), PB_CONTROL_IOCG_0, 0);
	    pb->pb.pb_confd.confd_nbps	= 512;
	    pb->pb.pb_confd.confd_ncyl	= 1;
	    pb->pb.pb_confd.confd_nspt	= 1;
	    pb->pb.pb_confd.confd_nhead	= 1;
	    pb->pb.pb_confd.confd_shead	= 0;
	    pb->pb.pb_confd.confd_flags	= CONFD_FLAGS_SSP;
	    if (rfexec("SETCONF", &xpb, cn, 1))
		return (-1);

	    if (rfuib(cn, dr&~1, uib, PB_COMMAND_READ))
		return (-1);
	    if ((shead = uib->uib_confd_shead1) == 0) {
		printf("rf: no such volume\n");
		return (-1);
	    }
	    nspt = uib->uib_confd_nspt;
	} else {
	    shead = 0;
	    nspt = 1;
	}
	XPB_SET(&xpb, PB_COMMAND_SETCONF, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb.pb_confd.confd_nbps	= 512;
	pb->pb.pb_confd.confd_ncyl	= 1;
	pb->pb.pb_confd.confd_nspt	= nspt;
	pb->pb.pb_confd.confd_nhead	= 1;
	pb->pb.pb_confd.confd_shead	= shead;
	if (rf_ctype[cn] == 3200)
	    pb->pb.pb_confd.confd_flags	= CONFD_FLAGS_SSP;
	if (rfexec("SETCONF1", &xpb, cn, 1))
	    return (-1);

	if (rf_ctype[cn] == 3400) {
	    XPB_SET(&xpb, PB_COMMAND_START, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	    if (rfexec("START", &xpb, cn, 0))
		return (-1);
	}
	return (0);
}

rfuib(cn, dr, uib, func)
	struct uib	*uib;
{
	struct	rf_xpb			xpb;
	register struct rf_pb		*pb = &xpb.xpb_pb;
	int				sum;

	XPB_SET(&xpb, func, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb_addrmod			= RF_ADDRMOD;
	pb->pb.pb_std.std_daddr		= 0;
	pb->pb.pb_std.std_vmeaddr	= paddr(uib);
	pb->pb.pb_std.std_blkcnt	= 1;
	if (func == PB_COMMAND_WRITE) {
	    uib->uib_magic = RF_UIB_MAGIC;
	    uib->uib_cksum = 0;
	    uib->uib_cksum = rfchksum(uib);
	    if (rfexec("UIB WRITE", &xpb, cn, 1))
		return (-1);
	} else {
	    if (rfexec("UIB READ", &xpb, cn, 1))
		return (-1);
	    if (uib->uib_magic != RF_UIB_MAGIC) {
		printf("rf: drive not formatted\n");
		return (-1);
	    }
	    sum = uib->uib_cksum;
	    uib->uib_cksum = 0;
	    if (rfchksum(uib) != sum) {
		uib->uib_cksum = sum;
		printf("rf: drive not formatted (checksum)\n");
		return (-1);
	    }
	    uib->uib_cksum = sum;
	}
	return (rfconfig(cn, dr, uib));
}

rfchksum(uib)
	struct uib	*uib;
{
	register int		i;
	register u_short	*wp = (u_short *)uib;
	register u_short	cksum = 0;

	for (i = 0; i < sizeof(struct uib); i += sizeof(u_short))
	    cksum ^= *wp++;
	return (cksum);
}

rfconfig(cn, dr, uib)
	struct uib	*uib;
{
	struct	rf_xpb			xpb;
	register struct rf_pb		*pb = &xpb.xpb_pb;

	XPB_SET(&xpb, PB_COMMAND_SETCONF, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb.pb_confd.confd_nbps	= uib->uib_confd_nbps;
	pb->pb.pb_confd.confd_ncyl	= uib->uib_confd_ncyl;
	pb->pb.pb_confd.confd_nspares	= uib->uib_confd_nspares;
	pb->pb.pb_confd.confd_nspt	= uib->uib_confd_nspt;
	pb->pb.pb_confd.confd_nhead	= uib->uib_confd_nhead;
	pb->pb.pb_confd.confd_shead	= uib->uib_confd_shead;
	pb->pb.pb_confd.confd_flags	= uib->uib_confd_flags;
	if (rfexec("SETCONF2", &xpb, cn, 1))
	    return (-1);

	XPB_SET(&xpb, PB_COMMAND_SETDPARAM, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb.pb_dparam.dparam_intlv	= uib->uib_dparam_intlv;
	pb->pb.pb_dparam.dparam_hdskew	= uib->uib_dparam_hdskew;
	pb->pb.pb_dparam.dparam_cyskew	= uib->uib_dparam_cyskew;
	pb->pb.pb_dparam.dparam_hgskew	= uib->uib_dparam_hgskew;
	pb->pb.pb_dparam.dparam_rcvry	= uib->uib_dparam_rcvry;
	pb->pb.pb_dparam.dparam_gap1	= uib->uib_dparam_gap1;
	pb->pb.pb_dparam.dparam_gap2	= uib->uib_dparam_gap2;
	if (rfexec("SETDPARAM", &xpb, cn, 1))
	    return (-1);

	XPB_SET(&xpb, PB_COMMAND_SETIOCG, PB_UNIT_NODISK, PB_UNIT(dr), 0);
	if (xbad144 || rf_diskformat) {
	    pb->pb.pb_iocg.iocg_rcvry	= (IOCG_RCVRY_IGE | IOCG_RCVRY_COR);
	} else {
	    pb->pb.pb_iocg.iocg_rcvry	= uib->uib_iocg_rcvry;
	    pb->pb.pb_iocg.iocg_rahead	= uib->uib_iocg_rahead;
	    pb->pb.pb_iocg.iocg_cache	= uib->uib_iocg_cache;
	    pb->pb.pb_iocg.iocg_ndretry	= uib->uib_iocg_ndretry;
	    pb->pb.pb_iocg.iocg_dretry	= uib->uib_iocg_dretry;
	}
	if (rfexec("SETIOCG", &xpb, cn, 1))
	    return (-1);
	return (0);
}

rfstrategy(io, func)
	register struct iob *io;
{
	register int			bleft, bpart;
	register int 			bn = io->i_bn;
	register int 			dr = UTODR(io->i_unit);

	switch (func) {
	  case READ:
	    func = PB_COMMAND_READ; 
	    break;
	  case WRITE:
	    func = PB_COMMAND_WRITE; 
	    break;
	  default:
	    printf("rf%d: bad func\n", io->i_unit);
	    return (-1);
	}
	bleft = io->i_cc;
	do {
	    bpart = (bleft < RF_MAXCNT) ? bleft : RF_MAXCNT;
	    if (forbb) {
		if (rfstrat(io, bn, bpart, func) == -1)
		    return (-1);
		io->i_ma += bpart;
	    } else if (badstrat(io, bn, bpart, func, 
					rfstrat, PB_COMMAND_READ) == -1)
		    return (-1);
	    bleft -= bpart;
	    bn += bpart / io->i_st.nbsec;
	} while (bleft > 0);
	return (io->i_cc);
}

rfstrat(io, bn, bpart, func)
	register struct iob *io;
{
	register int 			cn = UTOCN(io->i_unit);
	register int 			dr = UTODR(io->i_unit);
	int	 			errcnt = 0;
	struct	rf_xpb			xpb;
	register struct rf_pb		*pb = &xpb.xpb_pb;

	XPB_SET(&xpb, func, PB_UNIT(dr), PB_UNIT(dr), 0);
	while (1) {
	    pb->pb_addrmod		= RF_ADDRMOD;
	    pb->pb.pb_std.std_daddr	= bn + rf_cyl_offset[cn][dr];
	    pb->pb.pb_std.std_vmeaddr	= io->i_ma;
	    pb->pb.pb_std.std_blkcnt	= (bpart + 511) >> 9;
	    if (rfexec(func==PB_COMMAND_READ ? "READ" : "WRITE", &xpb, cn, 1)) {
		if (!xbad144 && ++errcnt < 5)
			continue;
		return (-1);
	    }
	    return (0);
	}
}

rfexec(s, xpb, cn, timeout)
	char		 	*s;
	register struct rf_xpb	*xpb;
{
	register struct rfdevice 	*rfaddr = rfstd[cn];
	register struct sb		*sb = &xpb->xpb_sb.sb;
	int				timer = 50000, *p;

	if (rf_debug) {
	    p = (int *)xpb;
	    printf("%+10sPB:%8x %8x %8x %8x %8x\n",
		s,*p,*(p+1),*(p+2),*(p+3),*(p+4));
	}
	/* wait for previous command to clear */
	while ((rfaddr->rf_status & STATUS_BSY) != rf_bsybit[cn])
	    if (--timer)
		DELAY(400);
	    else {
		printf("rf: timeout\n");
		return (-1);
	    }
	rf_bsybit[cn] ^= STATUS_BSY;

	/* Write the address buffer port and signal a type-0 command */
	rfaddr->rf_abp_ctlam = ABP_CTLAM_SET | RF_XFER_WID | RF_ADDRMOD;
	rfaddr->rf_abp_pb_msb = paddr(xpb) >> 16;
	rfaddr->rf_abp_pb_lsb = paddr(xpb);
	rfaddr->rf_atn = ATN_TYPE0;

	timer = 0;
	while ((sb->sb_flag & SB_FLAG_CC) == 0)
	    if (timeout && (++timer == 50000)) {
		printf("rf: timeout\n");
		return (-1);
	    } else
		DELAY(400);
	if (rf_debug) {
	    p = (int *)sb;
	    printf("          SB:%8x %8x %8x\n", *p, *(p+1), *(p+2));
	}
	if (sb->sb_flag & SB_FLAG_ERR) {
	    printf("rf: error %x on %s: sn %d: flags %b: dstatus %b\n",
		sb->sb_error, s, sb->sb_daddr, sb->sb_flag, SB_FLAG_BITS, 
		sb->sb_dstatus, (rf_ctype[cn] == 3200) ? SB_DSTATUS_BITS_3200 :
		SB_DSTATUS_BITS_3400);
	    return (-1);
	} else
	    return (0);
}

rfreset(cn)
{ 	
	register struct rfdevice 	*rfaddr = rfstd[cn];
	int				foo;

	if (!probe(&rfaddr->rf_status, &foo)) {
	    printf("rf: controller %d not present\n", cn);
	    return (-1);
	}
	rfaddr->rf_reset = RF_RESET;
	rf_bsybit[cn] = 0;
	DELAY(500000);
	if ((rfaddr->rf_status & STATUS_RDY) == 0) {
	    printf("rf: not ready %x\n", rfaddr->rf_status);
	    return (1);
	}
	if (rfaddr->rf_status & STATUS_EMASK) {
	    printf("rf: diagnostics failed %b\n", 
		1<<((rfaddr->rf_status&STATUS_EMASK)>>STATUS_ESHFT), 
		STATUS_EBITS);
	    return (-1);
	}

	if ((rfaddr->rf_status&STATUS_BMASK) == STATUS_B3200)
	    rf_ctype[cn] = 3200;
	else
	    rf_ctype[cn] = 3400;
	return (0);
}

#ifdef	DISK_FORMAT

#define	RM_SETSIZE	100000
struct rm_set {
	long	rm_bits[RM_SETSIZE/32];
}			rm_set;
#define	RM_SET(n)	(rm->rm_bits[(n)/32] |=  (1 << ((n) % 32)))
#define	RM_CLR(n)	(rm->rm_bits[(n)/32] &= ~(1 << ((n) % 32)))
#define	RM_ISSET(n)	(rm->rm_bits[(n)/32] &   (1 << ((n) % 32)))
#define	RM_ISCLR(n)	(RM_ISSET(n) == 0)
#define RM_ZERO()	bzero((char *)rm, sizeof(*rm))

static char		buf[100];
struct rf_trkid		trkid[300];

rfformat()
{
	char			test;
	int			unformatted;
	struct rf_uib		rf_uib, drive_uib;
	register int		cn = 0, dr = 0, vol = 0;
	int			nlaps = 1;

	printf("\nFormatting RF disk\n");
	rf_diskformat++;
	bzero(&rf_uib, sizeof(struct rf_uib));
	strcpy(rf_uib.uib.uib_label, "UNKNOWN");
	rf_uib.uib.uib_confd_nbps		= 512;
	rf_uib.uib.uib_dparam_intlv		= 1;
	rf_uib.uib.uib_dparam_hdskew		= 3;
	rf_uib.uib.uib_dparam_cyskew		= 15;
	rf_uib.uib.uib_iocg_cache		= 
		IOCG_CACHE_SEA|IOCG_CACHE_CRD|IOCG_CACHE_XHD|IOCG_CACHE_SRD;
	rf_uib.uib.uib_iocg_rahead		= 255;
	rf_uib.uib.uib_iocg_dretry		= 11;
	rf_uib.uib.uib_iocg_ndretry		= 3;
	rf_uib.uib.uib_dparam_rcvry		= 
		DPARAM_RCVRY_TAG|DPARAM_RCVRY_DBV;
	rf_uib.uib.uib_cyl_offset		= 1;
loop:	printf("\nFunctions available:\n");
	printf("  a) Format Disk\n  b) Verify Disk\n");
	printf("  c) Read UIB\n  d) Tune UIB\n  e) Toggle Debug Mode");
	printf("\nEnter letter of desired function ('H' for help): ");
	stripwhite(gets(buf));
	test = buf[0];
	if (test == 'H')
	    goto loop;
	if (test == 'e' || test == 'E') {
	    rf_debug ^= 1;
	    goto loop;
	}
	if (test != 'a' && test != 'A' && 
	    test != 'b' && test != 'B' &&
	    test != 'c' && test != 'C' &&
	    test != 'd' && test != 'D')
	    goto loop;

	dr = DRTODR(dr);
	GETFIELD_MM("Controller number", cn, 0, NRF-1);
	GETFIELD_MM("Drive number", dr, 0, Nrf-1);
	GETFIELD_MM("Volume number", vol, 0, 1);
	dr = (dr << 1) |  vol;
	rfinit(cn, dr);

	if (test == 'a' || test == 'A') {
	    if (rfdoformat(cn, dr, &rf_uib.uib))
		printf("\nFORMAT FAILED\n");
	    else
		printf("\nSUCCESSFUL FORMAT\n");
	} else if (test == 'b' || test == 'B') {
	    if (rfdoverify(cn, dr, &rf_uib.uib))
		printf("\nVERIFY FAILED\n");
	    else
		printf("\nSUCCESSFUL VERIFY\n");
	} else if (test == 'c' || test == 'C')
	    rfprintuib(cn, dr, &rf_uib.uib);
	else if (test == 'd' || test == 'D')
	    rftuneuib(cn, dr, &rf_uib.uib);
	goto loop;
}

rfdoformat(cn, dr, uib)
	struct uib	*uib;
{
	struct	rf_xpb		xpb;
	register struct rf_pb	*pb = &xpb.xpb_pb;
	struct rm_set		*rm = &rm_set;
	register int		nspd, nspc;
	register int		sec;

	rfenteruib(cn, dr, uib);
	if (!getbits("*** Ok to format", 1, 0))
	    return (-1);
	printf("FORMATTING ...\n");
	if (rfconfig(cn, dr, uib))
	    return (-1);
	nspc = uib->uib_confd_nspt * uib->uib_confd_nhead;
	nspd = nspc * uib->uib_confd_ncyl;
	printf("   Cylinder: ");
	for (sec = 0 ; sec < nspd; sec += nspc) {
	    XPB_SET(&xpb, PB_COMMAND_FORMAT, PB_UNIT(dr), PB_UNIT(dr), 0);
	    pb->pb.pb_fmtv.fmtv_sector = sec;
	    pb->pb.pb_fmtv.fmtv_nsector = nspc;
	    printf("%8d\b\b\b\b\b\b\b\b", sec/nspc);
	    if (rfexec("FORMAT", &xpb, cn, 1))
		return (-1);
	}
	printf("\r                          \r");
	RM_ZERO();
	if (rfuib(cn, dr, uib, PB_COMMAND_WRITE) ||
	    rfrmmap(cn, dr, uib, PB_COMMAND_WRITE))
	    return (-1);
	return (0);
}

rfrmmap(cn, dr, uib, func)
	struct uib	*uib;
{
	struct	rf_xpb		xpb;
	register struct rf_pb	*pb = &xpb.xpb_pb;
	struct rm_set		*rm = &rm_set;
	int			n;

	n = ((((uib->uib_cyl_offset*uib->uib_confd_nspt*uib->uib_confd_nhead) +
	    7) / 8) + (uib->uib_confd_nbps - 1)) / uib->uib_confd_nbps;
	XPB_SET(&xpb, func, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb_addrmod			= RF_ADDRMOD;
	pb->pb.pb_std.std_daddr		= 1;
	pb->pb.pb_std.std_vmeaddr	= paddr(rm);
	pb->pb.pb_std.std_blkcnt	= n;
	if (func == PB_COMMAND_WRITE)
	    while (n >= 0) {
		RM_SET(n);
		n--;
	    }
	if (rfexec("RWMAP", &xpb, cn, 1))
		return (-1);
	return (0);
}

rfdoverify(cn, dr, uib, nlaps)
	struct uib	*uib;
{
	struct	rf_xpb		xpb;
	register struct rf_pb	*pb = &xpb.xpb_pb;
	register struct sb	*sb = &xpb.xpb_sb.sb;
	register int		nspt, ntpc;
	register int		sec, badsec, i;
	register int		fsec, rsec;
	int			lastsec, lastrsec;
	struct rm_set		*rm = &rm_set;
	int			lap, nlaps = 1;
	int			mapsoft = 1;

	if (rfuib(cn, dr, uib, PB_COMMAND_READ) ||
	    rfrmmap(cn, dr, uib, PB_COMMAND_READ))
	    return (-1);
	nspt = uib->uib_confd_nspt;
	ntpc = uib->uib_confd_nhead;
	lastsec = nspt * ntpc * uib->uib_confd_ncyl;
	lastrsec = nspt * ntpc * uib->uib_cyl_offset;
	if (lastrsec > RM_SETSIZE) {
		printf("RM_SETSIZE too small\n");
		return (-1);
	}

	GETFIELD("Number of laps for verify", 	nlaps);
	GETFIELD("Map soft errors", 		mapsoft);
	printf("VERIFYING REMAP AREA ... (0-%d)\n", lastrsec-1);
	for (lap = 1; lap <= nlaps; lap++) {
	    printf("Lap #%d\n",  lap);
	    for (sec = 1; sec < lastrsec; ) {
		XPB_SET(&xpb, PB_COMMAND_VERIFY, PB_UNIT(dr), PB_UNIT(dr), 0);
		pb->pb.pb_fmtv.fmtv_sector = sec;
		pb->pb.pb_fmtv.fmtv_nsector = (lastrsec - sec);
		if (rfexec("VERIFY", &xpb, cn, 0)) {
		    badsec = sb->sb_daddr;
		    if (SB_ERROR_ID(sb->sb_error)) {
			printf("  Track %d bad\n", badsec);
			fsec = (badsec / nspt) * nspt;
			for (i = 0; i < nspt; i++)
			    RM_SET(fsec+i);
			sec = fsec + nspt;
		    } else {
			printf("  Sector %d bad\n", badsec);
			RM_SET(badsec);
			sec = badsec + 1;
		    }
		} else
		    sec = lastrsec;
	    }
	}

	if (rfrmmap(cn, dr, uib, PB_COMMAND_WRITE) ||
	    rfdefects(cn, dr, uib, lastrsec)) {
	    (void) rfrmmap(cn, dr, uib, PB_COMMAND_WRITE);
	    return (-1);
	}

	printf("VERIFYING DATA AREA ... (%d-%d)\n", lastrsec, lastsec);
	for (lap = 1; lap <= nlaps; lap++) {
	    printf("Lap #%d\n", lap);
	    for (sec = lastrsec; sec < lastsec; ) {
		XPB_SET(&xpb, PB_COMMAND_VERIFY, PB_UNIT(dr), PB_UNIT(dr), 0);
		pb->pb.pb_fmtv.fmtv_sector = sec;
		pb->pb.pb_fmtv.fmtv_nsector = (lastsec - sec);
		if (rfexec("VERIFY", &xpb, cn, 0)) {
		    badsec = sb->sb_daddr;
		    if (!mapsoft && SB_ERROR_SOFT(sb->sb_error))
			sec = badsec;
		    else if ((SB_ERROR_ID(sb->sb_error) || 
			(sec = rfsmapsec(cn, dr, badsec, lastrsec, uib)) < 0) &&
			(sec = rfmaptrk(cn, dr, badsec, lastrsec, uib)) < 0) {
			    (void) rfrmmap(cn, dr, uib, PB_COMMAND_WRITE);
			    return (-1);
		    }
		} else
		    sec = lastsec;
	   }
	}
	if (rfrmmap(cn, dr, uib, PB_COMMAND_WRITE))
	    return (-1);
	return (0);
}

rfdefects(cn, dr, uib, lastrsec)
	struct uib	*uib;
{
	struct rm_set	*rm = &rm_set;
	register int	i = 0, j, k;
	register char	*p, *p0;
	int		sec, n, file, fpos = 0, x[4];

file:	printf("    %+40s: ", "Defect File (return for manual)");
	stripwhite(gets(buf));
	if (buf[0]) {
	    if ((file = open(buf, 0)) < 0)
		goto file;
	} else
	    file = -1;

enter:	if (file == -1)
	    printf("Enter: Cylinder/Head/Offset/Len (return when done)\n");
	for (;;i++) {
	    if (file == -1) {
		printf("    %3d : ",i);
		stripwhite(gets(buf));
		if (buf[0] == 0)
		    return (0);
	    } else {
		k = read(file, buf, 40);
		if (k <= 0) {
		    close(file);
		    return (0);
		}
		for (p = buf; *p && *p != '\n'; p++)
		    ;
		if (*p == '\n')
		    *p = 0;
		fpos += (strlen(buf) + 1);
		lseek(file, fpos, 0);
		stripwhite(buf);
	    }
	    for (j = 0, p = buf; j < 4; j++) {
		p0 = p;
		while (*p)
		    if (*p < '0' || *p > '9')
			break;
		    else
			p++;
		if (*p == 0 && j != 3)
		    goto enter;
		*p++ = 0;
		x[j] = atoi(p0);
	    }
	    printf("          %6d.%3d.%7d.%3d ==> ", x[0], x[1], x[2], x[3]);
	    switch (n = rfcompsec(cn, dr, uib, x[0], x[1], x[2], x[3], &sec)) {
		case 1:
		    if (sec < lastrsec) {
			printf(" %d, sector removed from remap\n", sec);
			RM_SET(sec);
			break;
		    } else if (rfsmapsec(cn, dr, sec, lastrsec, uib) < 0)
			return (-1);
		    break;

		case 0:
		    printf("ignored\n");
		    break;

		default:
		    if (n < 0)
			return (-1);
		    if (sec < lastrsec) {
			printf(" %d, track removed from remap\n", sec);
			for (k = 0; k < uib->uib_confd_nspt; k++)
			    RM_SET(sec+k);
			break;
		    } else if (rfmaptrk(cn, dr, sec, lastrsec, uib) < 0)
			return (-1);
		    break;
	    }
	}
}

/* 
 * This routine computes the first sector and number of sectors to map for
 * a manfactureer specified defect. If the defect occurs in a "bad" place, the
 * entire track must be mapped. Return:
 *	0 	"harmless" error
 *	1	slip/map single sector at "first".
 *	nspt	map the track starting at "first".
 */
#define CRITICAL	(26)		/* header bytes that cant be trash */

rfcompsec(cn, dr, uib, cyl, hd, off, len, first)
	struct uib	*uib;
	int		*first;
{
	struct	rf_xpb			xpb;
	register struct rf_pb		*pb = &xpb.xpb_pb;
        int				sec, trksec;	
        struct rf_trkid			*t;

	if ((cyl >= uib->uib_confd_ncyl) ||
	    (hd >= uib->uib_confd_nhead) ||
	    (off >= (uib->uib_pnbs * uib->uib_confd_nspt))) {
	    printf("not within geometry of disk ...");
	    return (0);
	}
	trksec = uib->uib_confd_nspt * (uib->uib_confd_nhead * cyl + hd);
	if (trksec > (uib->uib_confd_nspt * uib->uib_confd_nhead *
	    uib->uib_confd_ncyl)) {
	    printf("beyond end of disk ...");
	    return (0);
	}
	sec = off / uib->uib_pnbs;
	off %= uib->uib_pnbs;
	t = &trkid[sec];

	/* spare the track error not in data region */
	if ((off <= (CRITICAL  + uib->uib_dparam_gap1)) || 
	    ((off+len) >= uib->uib_pnbs) || (off == 0)){
	    *first = trksec;
	    return (uib->uib_confd_nspt);
	}

	XPB_SET(&xpb, PB_COMMAND_READTID, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb_addrmod			= RF_ADDRMOD;
	pb->pb.pb_std.std_daddr		= trksec;
	pb->pb.pb_std.std_vmeaddr	= paddr(trkid);
	if (rfexec("READTID", &xpb, cn, 1))
	    return (-1);

	if (rf_debug)
	    printf("TRKID: sec %x flag %x alt %x   ", 
		    t->trkid_sec, t->trkid_flag, t->trkid_alt);
	switch (t->trkid_sec) {
	    case TRKID_SEC_SPARE:
		printf("error in spare sector, map track ...");
		*first = trksec;
		return (uib->uib_confd_nspt);

	    case TRKID_SEC_SHORT:
		printf("short sector ...");
		return (0);

	    case TRKID_SEC_SLIP:
		printf("sector already sliped ...");
		return (0);

	    default:
		switch (t->trkid_flag) {
		    case TRKID_FLAG_NORMAL:
	    		*first = trksec + t->trkid_sec;
	    		return (1);

		    case TRKID_FLAG_DEF_SEC:
			printf("sector already mapped to %d ...", t->trkid_alt);
			return (0);

		    case TRKID_FLAG_DEF_TRK:
			printf("track already mapped ...");
			return (0);

		    default:
			printf("defects in a sector already used ");
			printf("for a remap are not supported, reformat!!\n");
			return (-1);
		}
	}
}

rfmaptrk(cn, dr, sec, lastrsec, uib)
	struct uib	*uib;
{
	struct rm_set		*rm = &rm_set;
	struct	rf_xpb		xpb;
	register struct rf_pb	*pb = &xpb.xpb_pb;
	register int		fsec, rsec, i;

	fsec = (sec / uib->uib_confd_nspt) * uib->uib_confd_nspt;
	for (rsec = 0 ; rsec < lastrsec; rsec += uib->uib_confd_nspt) {
	    for (i = 0; i < uib->uib_confd_nspt; i++)
		if (RM_ISSET(rsec+i))
		    break;
	    if (i == uib->uib_confd_nspt)
		    break;
	}
	if (rsec >= lastrsec) {
	    printf("Need more cylinders for uib/remap!\n");
	    return (-1);
	}
	XPB_SET(&xpb, PB_COMMAND_MAPTRACK, PB_UNIT(dr), PB_UNIT(dr), 0);
	pb->pb.pb_smap.smap_baditem = fsec;
	pb->pb.pb_smap.smap_alternate = rsec;
	if (rfexec("MAPTRACK", &xpb, cn, 1)) {
	    printf("Failed maptrack\n");
	    return (-1);
	}
	printf("  map track (%d-%d) to (%d-%d)\n", 
	    fsec, fsec + uib->uib_confd_nspt - 1, 
	    rsec, rsec + uib->uib_confd_nspt - 1);
	for (i = 0; i < uib->uib_confd_nspt; i++)
	    RM_SET(rsec+i);
	return (fsec + uib->uib_confd_nspt);
}

rfsmapsec(cn, dr, sec, lastrsec, uib)
	struct uib	*uib;
{
	struct rm_set		*rm = &rm_set;
	struct	rf_xpb		xpb;
	register struct rf_pb	*pb = &xpb.xpb_pb;
	register struct sb	*sb = &xpb.xpb_sb.sb;
	register int		rsec, i;

	if (uib->uib_confd_nspares) {
	    XPB_SET(&xpb, PB_COMMAND_SLIPSEC, PB_UNIT(dr), PB_UNIT(dr), 0);
	    pb->pb.pb_smap.smap_baditem = sec;
	    pb->pb.pb_smap.smap_alternate = 0;
	    if (rfexec("SLIPSEC", &xpb, cn, 1) == 0) {
		printf("  slip %d\n", sec);
		return (sec + 1);
	    }
	    if (SB_ERROR_ID(sb->sb_error))
		return (rfmaptrk(cn, dr, sec, lastrsec, uib));
	    printf("  *** Cant slip %d, try to map\n", sec);
	}
	for (rsec = 0 ; rsec < lastrsec; rsec++)
	    if (RM_ISCLR(rsec))
		break;
	if (rsec >= lastrsec) {
	    printf("Need more cylinders for uib/remap!\n");
	    return (-1);
	}
	XPB_SET(&xpb, PB_COMMAND_MAPSEC, PB_UNIT(dr), PB_UNIT(dr), 0);
	pb->pb.pb_smap.smap_baditem = sec;
	pb->pb.pb_smap.smap_alternate = rsec;
	RM_SET(rsec);
	if (rfexec("MAPSEC", &xpb, cn, 1) && SB_ERROR_ID(sb->sb_error))
		return (rfmaptrk(cn, dr, sec, lastrsec, uib));
	printf("  map %d to %d\n", sec, rsec);
	return (sec + 1);
}

rfenteruib(cn, dr, uib)
	struct uib	*uib;
{
	register int len, i;

	printf("    %+40s: %s", "Label", uib->uib_label);
	for (len = i = strlen(uib->uib_label); i ; i--)
	    printf("\b");
	stripwhite(gets(buf));
	if (buf[0])
	    strcpy(uib->uib_label, buf);
	if (strlen(uib->uib_label) < len)
	    printf("    %+39s*: %s\n", "", uib->uib_label);
	GETFIELD("Number of cylinders", 	uib->uib_confd_ncyl);
	GETFIELD("Number of spare sectors",	uib->uib_confd_nspares);
	GETFIELD("Number of sectors per track",	uib->uib_confd_nspt);
	GETFIELD("Number of heads",		uib->uib_confd_nhead);
	if (rf_ctype[cn] == 3200) {
	    if (DRTOVOL(dr))
		GETFIELD("Start head",		uib->uib_confd_shead);
	    else
		GETFIELD("Start head VOL1",	uib->uib_confd_shead1);
	    GETBITS("Short sector",	CONFD_FLAGS_SSP, uib->uib_confd_flags);
	    if (uib->uib_confd_ncyl > 1024)
		GETBITS("SMD-E interface",	CONFD_FLAGS_EAD, uib->uib_confd_flags);
	}
	GETFIELD("Interleave factor",	 	uib->uib_dparam_intlv);
	GETFIELD("Amount of head skew", 	uib->uib_dparam_hdskew);
	GETFIELD("Amount of cylinder skew", 	uib->uib_dparam_cyskew);
	if (rf_ctype[cn] == 3400)
	    GETFIELD("Amount of head group skew",uib->uib_dparam_hgskew);
	else {
	    uib->uib_dparam_rcvry =
	    	getfield("Data recovery", uib->uib_dparam_rcvry, 0x10, 0, 0, 0);
	    GETFIELD("Gap1 (ID preamble)",	uib->uib_dparam_gap1);
	    GETFIELD("Gap2 (DATA preamble)",	uib->uib_dparam_gap2);
	}
	GETBITS("Search cache",		IOCG_CACHE_SEA, uib->uib_iocg_cache);
	GETBITS("Cache read data",	IOCG_CACHE_CRD, uib->uib_iocg_cache);
	GETBITS("Read ahead priority",	IOCG_CACHE_RAP, uib->uib_iocg_cache);
	GETBITS("Cache write data",	IOCG_CACHE_CWT, uib->uib_iocg_cache);
	GETBITS("Sort reads",		IOCG_CACHE_SRD, uib->uib_iocg_cache);
	GETBITS("Sort writes",		IOCG_CACHE_SWT, uib->uib_iocg_cache);
	GETBITS("Inhibit zero latency",	IOCG_CACHE_ZLI, uib->uib_iocg_cache);
	GETBITS("Read ahead across trk",IOCG_CACHE_XHD, uib->uib_iocg_cache);
	GETBITS("Read ahead across cyl",IOCG_CACHE_XCY, uib->uib_iocg_cache);
	GETFIELD("Sector read ahead",		uib->uib_iocg_rahead);
	GETBITS("Ignore ECC",		
					IOCG_RCVRY_IGE, uib->uib_iocg_rcvry);
	GETBITS("Disable error correction",	
					IOCG_RCVRY_COR, uib->uib_iocg_rcvry);
	GETBITS("Apply track/data separator",	
					IOCG_RCVRY_EXT, uib->uib_iocg_rcvry);
	GETFIELD("Data retry count",		uib->uib_iocg_dretry);
	GETFIELD("Non-data retry count",	uib->uib_iocg_ndretry);
	GETFIELD_MM("Cylinders for uib/defects",
		uib->uib_cyl_offset, 1, 100);
	GETFIELD("Physical bytes per sector",	uib->uib_pnbs);
	printf("\n");
}

rfprintuib(cn, dr, uib)
	struct uib	*uib;
{
	if (rfuib(cn, dr, uib, PB_COMMAND_READ))
	    return;
	printf("\nUIB: controller %d, drive %d, volume %d\n", 
		cn, DRTODR(dr), DRTOVOL(dr));
#define	PX(S, V)	printf("    %+40s: 0x%x\n", S, V)
#define	PD(S, V)	printf("    %+40s: %d\n", S, V)
#define	PS(S, V)	printf("    %+40s: %s\n", S, V)
#define	PB(S, V, B)	printf("    %+40s: %b\n", S, V, B)
	PX("Check sum",			uib->uib_cksum);
	PX("Magic number",		uib->uib_magic);
	PS("Label",			uib->uib_label);
	PD("Bytes per sector",		uib->uib_confd_nbps);
	PD("Cylinders",			uib->uib_confd_ncyl);
	PD("Spare sectors",		uib->uib_confd_nspares);
	PD("Sectors per track",		uib->uib_confd_nspt);
	PD("Number of heads",		uib->uib_confd_nhead);
	if (rf_ctype[cn] == 3200) {
	    if (DRTOVOL(dr))
		PD("Start head",	uib->uib_confd_shead);
	    else
		PD("Start head VOL1",	uib->uib_confd_shead1);
	    PB("Configuration flags",	uib->uib_confd_flags, CONFD_FLAGS_BITS);
	}
	PD("Interleave factor",		uib->uib_dparam_intlv);
	PD("Head skew",			uib->uib_dparam_hdskew);
	PD("Cylinder skew",		uib->uib_dparam_cyskew);
	if (rf_ctype[cn] == 3400)
	    PD("Head group skew",	uib->uib_dparam_hgskew);
	else {
	    PD("Data recovery",		uib->uib_dparam_rcvry);
	    PD("Gap1",			uib->uib_dparam_gap1);
	    PD("Gap2",			uib->uib_dparam_gap2);
	}
	PB("Cache",			uib->uib_iocg_cache, IOCG_CACHE_BITS);
	PD("Readahead",			uib->uib_iocg_rahead);
	PB("Recovery",			uib->uib_iocg_rcvry, IOCG_RCVRY_BITS);
	PD("Data retry count",		uib->uib_iocg_dretry);
	PD("Non-data retry count",	uib->uib_iocg_ndretry);
	PD("Cylinders for defects",	uib->uib_cyl_offset);
	PD("Physical bytes per sector",	uib->uib_pnbs);
	printf("Hit return to continue...");
	gets(buf);
}

rftuneuib(cn, dr, uib)
	struct uib	*uib;
{
	if (rfuib(cn, dr, uib, PB_COMMAND_READ))
	    return;
	if (rf_ctype[cn] == 3200) {
	    GETBITS("Short sector",	CONFD_FLAGS_SSP, uib->uib_confd_flags);
	    if (uib->uib_confd_ncyl > 1024)
		GETBITS("SMD-E interface",	CONFD_FLAGS_EAD, uib->uib_confd_flags);
	    uib->uib_dparam_rcvry =
	    	getfield("Data recovery", uib->uib_dparam_rcvry, 0x10, 0, 0, 0);
	}
	GETBITS("Search cache",		IOCG_CACHE_SEA, uib->uib_iocg_cache);
	GETBITS("Cache read data",	IOCG_CACHE_CRD, uib->uib_iocg_cache);
	GETBITS("Read ahead priority",	IOCG_CACHE_RAP, uib->uib_iocg_cache);
	GETBITS("Cache write data",	IOCG_CACHE_CWT, uib->uib_iocg_cache);
	GETBITS("Sort reads",		IOCG_CACHE_SRD, uib->uib_iocg_cache);
	GETBITS("Sort writes",		IOCG_CACHE_SWT, uib->uib_iocg_cache);
	GETBITS("Inhibit zero latency",	IOCG_CACHE_ZLI, uib->uib_iocg_cache);
	GETBITS("Read ahead across trk",IOCG_CACHE_XHD, uib->uib_iocg_cache);
	GETBITS("Read ahead across cyl",IOCG_CACHE_XCY, uib->uib_iocg_cache);
	GETFIELD("Sector read ahead",		uib->uib_iocg_rahead);
	GETBITS("Ignore ECC",		
					IOCG_RCVRY_IGE, uib->uib_iocg_rcvry);
	GETBITS("Disable error correction",	
					IOCG_RCVRY_COR, uib->uib_iocg_rcvry);
	GETBITS("Apply track/data separator",	
					IOCG_RCVRY_EXT, uib->uib_iocg_rcvry);
	GETFIELD("Data retry count",		uib->uib_iocg_dretry);
	GETFIELD("Non-data retry count",	uib->uib_iocg_ndretry);
	GETFIELD("Physical bytes per sector",	uib->uib_pnbs);
	if (rfuib(cn, dr, uib, PB_COMMAND_WRITE))
	    return;
}
#endif	DISK_FORMAT
