#include <sys/types.h>
#include "openchip.h"
#include "qxoutport.h"
#include "video.h"
#include "console.h"
#include "imsg170.h"


/* parameters for the graphics monitor */

#define	N_C_SYNC_ON_V_AND_H	/* -ve composite sync on vert + horiz */

#define	PRATE		32	/* 32MHz pixel rate */

#define	X_PIXELS	736
#define	Y_PIXELS	512

#define	HTIME		(1000/32)	/* uSecs (32kHz line rate) */
#define	HSYNC_0		2	/* uSecs */
#define	HSYNC_1		3	/* uSecs */

#define	VSYNC_0		10	/* lines */
#define	VSYNC_1		4	/* lines */
#define	VSYNC_2		30	/* lines */

#undef	DEFPLANES
#define	DEFPLANES		1

int             planes = DEFPLANES;	/* Number of bitplanes in use */

mon_info        smonitor = {
       PRATE,				/* video pixel rate */
#ifdef	C_SYNC_ON_GREEN
        {
	{ACTIVE_LOW, DISABLED},		/* VSYNC */
	{ACTIVE_LOW, DISABLED},		/* HSYNC */
	{ACTIVE_HIGH, COMP_SYNC}	/* PSYNC */
	},
#endif	C_SYNC_ON_GREEN
#ifdef	N_C_SYNC_ON_V_AND_H
        {
	{ACTIVE_LOW, COMP_SYNC},	/* VSYNC */
	{ACTIVE_LOW, COMP_SYNC},	/* HSYNC */
	{ACTIVE_LOW, DISABLED}		/* PSYNC */
	},
#endif	N_C_SYNC_ON_V_AND_H
#ifdef	N_SYNC_ON_V_AND_H
        {ACTIVE_LOW, VERT_SYNC},	/* VSYNC */
        {ACTIVE_LOW, HORIZ_SYNC},	/* HSYNC */
        {ACTIVE_LOW, DISABLED}		/* PSYNC */
# endif	N_SYNC_ON_V_AND_H
#ifdef	INTERLACED
        1,				/* interlaced picture */
#else
        0,				/* non-interlaced picture */
#endif
        (HTIME * PRATE) / 32,		/* total scan line width */
        X_PIXELS / 32,			/* off to horizontal blank */
        (X_PIXELS + HSYNC_0 * PRATE) / 32,	/* off to horizontal 
						 * sync start */
        (X_PIXELS + (HSYNC_0 + HSYNC_1) * PRATE) / 32,	/* to h sync end */
        Y_PIXELS + VSYNC_0 + VSYNC_1 + VSYNC_2,	/* scan lines per frame */
        Y_PIXELS,			/* offset to vertical blank */
        Y_PIXELS + VSYNC_0,		/* offset to vertical sync start */
        Y_PIXELS + VSYNC_0 + VSYNC_1	/* offset to vertical sync end */
};

/* end of graphics monitor parameters */

#define CLEAR   2
#define INVERT  3

int             conon = 0;
form            screenform;	/* Current screen form */
point           pen;		/* The pen position */

word            zerobits[16];	/* Zero bit map */

/******* Start of standalone font declaration ********/
/*	 color, 1 plane, 8 pixels per byte	     */

static word strike [] =
{
   0x0000, 0x0008, 0x0010, 0x0018, 0x0020, 0x0028, 0x0030, 0x0038, 
   0x0040, 0x0048, 0x0050, 0x0058, 0x0060, 0x0068, 0x0070, 0x0078, 
   0x0080, 0x0088, 0x0090, 0x0098, 0x00a0, 0x00a8, 0x00b0, 0x00b8, 
   0x00c0, 0x00c8, 0x00d0, 0x00d8, 0x00e0, 0x00e8, 0x00f0, 0x00f8, 
   0x0100, 0x0108, 0x0110, 0x0118, 0x0120, 0x0128, 0x0130, 0x0138, 
   0x0140, 0x0148, 0x0150, 0x0158, 0x0160, 0x0168, 0x0170, 0x0178, 
   0x0180, 0x0188, 0x0190, 0x0198, 0x01a0, 0x01a8, 0x01b0, 0x01b8, 
   0x01c0, 0x01c8, 0x01d0, 0x01d8, 0x01e0, 0x01e8, 0x01f0, 0x01f8, 
   0x0200, 0x0208, 0x0210, 0x0218, 0x0220, 0x0228, 0x0230, 0x0238, 
   0x0240, 0x0248, 0x0250, 0x0258, 0x0260, 0x0268, 0x0270, 0x0278, 
   0x0280, 0x0288, 0x0290, 0x0298, 0x02a0, 0x02a8, 0x02b0, 0x02b8, 
   0x02c0, 0x02c8, 0x02d0, 0x02d8, 0x02e0, 0x02e8, 0x02f0, 0x02f8, 
   0x02f8
};

static word ow [] =
{
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 
   0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008, 0x00ff, 
};

static byte glyphs [] =
{
   0x00, 0x18, 0x6c, 0x36, 0x0c, 0x60, 0x38, 0x0c, 
   0x0c, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x3c, 0x18, 0x3c, 0x7c, 0x0c, 0x7e, 0x1c, 0x7e, 
   0x3c, 0x3c, 0x00, 0x00, 0x0c, 0x00, 0x30, 0x3c, 
   0x3c, 0x18, 0x7c, 0x3e, 0x7c, 0x7e, 0x7e, 0x3e, 
   0x66, 0x3c, 0x0c, 0x66, 0x60, 0x63, 0x66, 0x3c, 
   0x7c, 0x3c, 0x7c, 0x3e, 0x7e, 0x66, 0xc3, 0x63, 
   0xc3, 0xc3, 0x7e, 0x7c, 0x00, 0x3e, 0x18, 0x00, 
   0x30, 0x00, 0x60, 0x00, 0x06, 0x00, 0x1c, 0x00, 
   0x60, 0x18, 0x18, 0x60, 0x30, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x31, 0x3e, 
   0x00, 0x18, 0x6c, 0x36, 0x3f, 0x66, 0x6c, 0x18, 
   0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x06, 
   0x66, 0x38, 0x66, 0x06, 0x18, 0x60, 0x30, 0x06, 
   0x66, 0x66, 0x00, 0x00, 0x18, 0x00, 0x18, 0x66, 
   0x66, 0x3c, 0x66, 0x60, 0x66, 0x60, 0x60, 0x60, 
   0x66, 0x18, 0x0c, 0x6c, 0x60, 0x77, 0x66, 0x66, 
   0x66, 0x66, 0x66, 0x60, 0x18, 0x66, 0xc3, 0x63, 
   0x66, 0xc3, 0x06, 0x60, 0x60, 0x06, 0x3c, 0x00, 
   0x18, 0x00, 0x60, 0x00, 0x06, 0x00, 0x30, 0x00, 
   0x60, 0x00, 0x00, 0x60, 0x30, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x6b, 0x60, 
   0x00, 0x18, 0x6c, 0x7f, 0x68, 0x0c, 0x6c, 0x30, 
   0x30, 0x0c, 0x7e, 0x18, 0x00, 0x00, 0x00, 0x0c, 
   0x6e, 0x18, 0x06, 0x06, 0x30, 0x7c, 0x60, 0x0c, 
   0x66, 0x66, 0x18, 0x18, 0x30, 0x7e, 0x0c, 0x0c, 
   0x6e, 0x66, 0x66, 0x60, 0x66, 0x60, 0x60, 0x60, 
   0x66, 0x18, 0x0c, 0x78, 0x60, 0x7f, 0x76, 0x66, 
   0x66, 0x66, 0x66, 0x30, 0x18, 0x66, 0xc3, 0x6b, 
   0x3c, 0x66, 0x0c, 0x60, 0x30, 0x06, 0x66, 0x00, 
   0x0c, 0x3e, 0x7c, 0x3c, 0x3e, 0x3c, 0x30, 0x3e, 
   0x7c, 0x18, 0x18, 0x66, 0x30, 0xfe, 0x7c, 0x3c, 
   0x7c, 0x3e, 0x7c, 0x3e, 0x7c, 0x66, 0xc3, 0x63, 
   0x66, 0x63, 0x7e, 0x18, 0x18, 0x18, 0x46, 0x60, 
   0x00, 0x18, 0x00, 0x36, 0x3e, 0x18, 0x38, 0x00, 
   0x30, 0x0c, 0x3c, 0x7e, 0x00, 0x7e, 0x00, 0x18, 
   0x7e, 0x18, 0x0c, 0x3c, 0x6c, 0x06, 0x7c, 0x18, 
   0x3c, 0x3e, 0x18, 0x18, 0x60, 0x00, 0x06, 0x18, 
   0x6a, 0x66, 0x7c, 0x60, 0x66, 0x7c, 0x7e, 0x6e, 
   0x7e, 0x18, 0x0c, 0x70, 0x60, 0x6b, 0x7e, 0x66, 
   0x7c, 0x66, 0x7c, 0x18, 0x18, 0x66, 0xc3, 0x6b, 
   0x18, 0x3c, 0x18, 0x60, 0x18, 0x06, 0x42, 0x00, 
   0x00, 0x66, 0x66, 0x60, 0x66, 0x66, 0x7c, 0x66, 
   0x66, 0x18, 0x18, 0x6c, 0x30, 0xdb, 0x66, 0x66, 
   0x66, 0x66, 0x66, 0x60, 0x30, 0x66, 0xc3, 0x6b, 
   0x3c, 0x36, 0x0c, 0x70, 0x00, 0x0e, 0x00, 0x60, 
   0x00, 0x18, 0x00, 0x7f, 0x0b, 0x30, 0x6d, 0x00, 
   0x30, 0x0c, 0x7e, 0x18, 0x00, 0x00, 0x00, 0x30, 
   0x76, 0x18, 0x18, 0x06, 0x7e, 0x06, 0x66, 0x30, 
   0x66, 0x06, 0x00, 0x00, 0x30, 0x7e, 0x0c, 0x18, 
   0x6e, 0x7e, 0x66, 0x60, 0x66, 0x60, 0x60, 0x66, 
   0x66, 0x18, 0x0c, 0x78, 0x60, 0x6b, 0x6e, 0x66, 
   0x60, 0x6a, 0x6c, 0x0c, 0x18, 0x66, 0x66, 0x7f, 
   0x3c, 0x18, 0x30, 0x60, 0x0c, 0x06, 0x00, 0x00, 
   0x00, 0x66, 0x66, 0x60, 0x66, 0x7e, 0x30, 0x66, 
   0x66, 0x18, 0x18, 0x78, 0x30, 0xdb, 0x66, 0x66, 
   0x66, 0x66, 0x60, 0x3c, 0x30, 0x66, 0x66, 0x6b, 
   0x18, 0x1c, 0x18, 0x18, 0x18, 0x18, 0x00, 0x60, 
   0x00, 0x00, 0x00, 0x36, 0x7e, 0x66, 0x66, 0x00, 
   0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x60, 
   0x66, 0x18, 0x30, 0x06, 0x0c, 0x06, 0x66, 0x30, 
   0x66, 0x0c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x00, 
   0x60, 0x66, 0x66, 0x60, 0x66, 0x60, 0x60, 0x66, 
   0x66, 0x18, 0x0c, 0x6c, 0x60, 0x63, 0x66, 0x66, 
   0x60, 0x6c, 0x66, 0x06, 0x18, 0x66, 0x3c, 0x77, 
   0x66, 0x18, 0x60, 0x60, 0x06, 0x06, 0x00, 0x00, 
   0x00, 0x66, 0x66, 0x60, 0x66, 0x60, 0x30, 0x3e, 
   0x66, 0x18, 0x18, 0x6c, 0x30, 0xdb, 0x66, 0x66, 
   0x7c, 0x3e, 0x60, 0x06, 0x30, 0x66, 0x3c, 0x7f, 
   0x3c, 0x18, 0x30, 0x18, 0x18, 0x18, 0x00, 0x3e, 
   0x00, 0x18, 0x00, 0x36, 0x18, 0x06, 0x3b, 0x00, 
   0x0c, 0x30, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00, 
   0x3c, 0x18, 0x7e, 0x7c, 0x0c, 0x7c, 0x3c, 0x30, 
   0x3c, 0x38, 0x18, 0x18, 0x0c, 0x00, 0x30, 0x18, 
   0x3c, 0x66, 0x7c, 0x3e, 0x7c, 0x7e, 0x60, 0x3e, 
   0x66, 0x3c, 0x38, 0x66, 0x7e, 0x63, 0x66, 0x3c, 
   0x60, 0x36, 0x66, 0x7c, 0x18, 0x3c, 0x18, 0x63, 
   0xc3, 0x18, 0x7e, 0x7c, 0x00, 0x3e, 0x00, 0x00, 
   0x00, 0x3e, 0x7c, 0x3c, 0x3e, 0x3c, 0x30, 0x06, 
   0x66, 0x18, 0x18, 0x66, 0x1c, 0xdb, 0x66, 0x3c, 
   0x60, 0x06, 0x60, 0x7c, 0x30, 0x3e, 0x18, 0x36, 
   0x66, 0x30, 0x7e, 0x0c, 0x18, 0x30, 0x00, 0x18, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 
   0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x60, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
   0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 
};

static addrptr strptr = strike;
static addrptr owptr = ow;
static addrptr glyphp = glyphs;

font sysfont = {
    FIXED, 32, 127, 7, 1, 8, 8,
    {&glyphp, 0, 0, 760, 8, 48, 0, 1, 0},
    &strptr,
    &owptr,
};

/******* End of font declaration ********/

int             cursorlevel = 1;
int             cursorphase = 1;

HideCursor()
{
    /*
     * must increment cursorlevel BEFORE inverting it so that the recursive
     * calls within InvCursor() work correctly 
     */
    if (cursorlevel++ == 0 && cursorphase == 1)
	InvCursor();
}

ShowCursor()
{
    /*
     * must decrement cursorlevel AFTER inverting it so that the recursive
     * calls within InvCursor() work correctly 
     */
    if (cursorlevel == 1 && cursorphase == 1)
	InvCursor();

    if (cursorlevel)
	cursorlevel--;
}

/*
 * Invert the cursor.                                                         
 */
InvCursor()
{
    rectangle       rect;

    rect.x = pen.x;
    rect.y = pen.y - sysfont.ascent;
    rect.width = sysfont.maxwidth;
    rect.height = sysfont.glyphs.height;
    RectOp(INVERT, &rect);
}

/*
 * Clear the screen.                                                          
 */
cls()
{
    RectOp(CLEAR, &screenform.x);
    MoveCTo(0, 0);
}

/*
 * BITBLT 
 *
 * interface to the BitBlT routine.                                     
 */
bitblt(cps)
CopyStruct     *cps;
{
    HideCursor();
    blit(cps);
    ShowCursor();
}

/*
 * Initialize the console variables.                                          
 */
coninit()
{
    extern mon_info smonitor;
    register int fg, col;

    if (InitKey()) {
	conon = 0;
	return (-1);
    } else
	conon = 1;

    /* initialize the OpenChip to produce a display */
    vid_mode(&smonitor);

    pal_default ();

    col_set (0,0,0,0,40);

    cls();
    MoveCTo(0, 60);
    ShowCursor();

    return;
}

/*
 * Perform one of two graphical ops on a rectangle...                     
 */
RectOp(op, rect)
register        op;
register rectangle *rect;
{
    register CopyStruct cps;

    cps.destform = (form *) & screenform;
    cps.sourceform = 0;
    cps.destrect = cps.srcrect = rect;
    cps.cliprect = (rectangle *) & screenform.x;
    cps.cindex = 0;

    HideCursor();		/* for efficiency when framing rectangles */

    if (op == CLEAR) {
	cps.htoneform = &zerobits[0];
	cps.mode = M_COPY;
    } 
    else {		/* op == INVERT */
	cps.htoneform = 0;
	cps.mode = M_NDST;
    }

    bitblt(&cps);
    ShowCursor();  /**/
}

/*
 * ScrollUp only for values of dv >= 0, no horizontal scrolling 
 */
ScrollUp(dv)
register        dv;
{
    rectangle       destrect;
    CopyStruct      copyparms;
    register CopyStruct *cps = &copyparms;
    register rectangle *drect = &destrect;
    register rectangle *rect;

    rect = (rectangle *) & screenform.x;

    cps->sourceform = cps->destform = &screenform;
    cps->htoneform = 0;
    cps->srcrect = cps->cliprect = rect;
    cps->destrect = drect;
    cps->mode = M_COPY;

    drect->width = rect->width;
    drect->height = rect->height;
    drect->x = rect->x;
    drect->y = rect->y - dv;

    /* Move the data */

    bitblt(cps);

    /* Now clear the remaining parts */

    cps->sourceform = 0;
    cps->htoneform = &zerobits[0];
    drect->x = rect->x;

    /* assume dv >= 0 */
    drect->y = rect->y + rect->height - dv;
    drect->height = dv;
    drect->width = rect->width;

    bitblt(cps);
}

#define STRIKE(f,o)	((*(f)->strike)[o])
#define OW(f,o)		((*(f)->ow)[o])

/*
 * Draw a character                                                           
 */
DChar(value)
register byte   value;
{
    rectangle       destrect, srcrect;
    CopyStruct      cstruct;
    register font  *fptr = &sysfont;

    if (value < fptr->firstchar || value > fptr->lastchar)
	return;

    value -= fptr->firstchar;

    /*
     * Set up destrect as the rectangle in the layer where the char is to go 
     */

    destrect.x = pen.x + ((OW(fptr, value) >> 8) & 0xff);
    destrect.y = pen.y - fptr->ascent;

    srcrect.x = STRIKE(fptr, value);
    srcrect.y = 0;
    srcrect.width = destrect.width = STRIKE(fptr, value + 1) - STRIKE(fptr, value);
    srcrect.height = destrect.height = fptr->glyphs.height;

    cstruct.sourceform = &fptr->glyphs;
    cstruct.destform = &screenform;
    cstruct.htoneform = 0;
    cstruct.srcrect = &srcrect;
    cstruct.destrect = &destrect;
    cstruct.cliprect = (rectangle *) & screenform.x;
    cstruct.cindex = 0;
    cstruct.mode = M_COPY;

    bitblt(&cstruct);
}

/*
 * g_putc 
 *
 * character put routine for the console driver. 
 */

con_putc(value)
char            value;
{
    if (!conon)
	return 0;

    HideCursor();		/* since we may be changing its position */

    switch (value) {
    case CURLEFT: CLeft(); break;
    case CR: pen.x = 0; break;
    case CURDOWN: CDown(); break;
    case LF: CDown(); break;
    case CURRIGHT: CRight(); break;
    case BACKSPACE: CLeft(); DChar(' '); break;
    default: DChar(value); CRight(); break;
    }

    ShowCursor();

    return (value);
}


/*
 * Absolute text cursor move                                                  
 */
MoveCTo(x, y)
{
    pen.x = x * sysfont.maxwidth;
    pen.y = y * sysfont.leading + sysfont.ascent;
}

/*
 * CLEFT     CRIGHT    CDOWN 
 */
CLeft()
{
    pen.x -= sysfont.maxwidth;
    if (pen.x < 0)
	pen.x = screenform.width - sysfont.maxwidth;
}

CRight()
{
    pen.x += sysfont.maxwidth;
    if (pen.x >= screenform.width) {
	pen.x = 0;
	CDown();
    }
}

CDown()
{
    pen.y += sysfont.leading;

    /*
     * if the next character will appear partly or completely below the edit
     * area, then we must scroll up a line. 
     */

    if (pen.y > screenform.height - sysfont.descent) {
	ScrollUp(sysfont.leading, 0);
	pen.y -= sysfont.leading;
    }
}


static          curkey = 0;	/* Current matrix code               */
static          kstat = 0;	/* special key status                */

short           m2a[] =		/* matrix position to ascii table    */
{
 -1, 27, '1', '2', '3', '4', '5', '6',
 '7', '8', '9', '0', '-', '=', DELK, TABK,
 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i',
 'o', 'p', '[', ']', CRK, CTRLK, 'a', 's',
 'd', 'f', 'g', 'h', 'j', 'k', 'l', ';',
 '\'', '`', SHFTK, '\\', 'z', 'x', 'c', 'v',
 'b', 'n', 'm', ',', '.', '/', SHFTK, HELPK,
 CAPSDOWN, ' ', CAPSUP, F1, F2, F3, F4, F5,
 F6, F7, F8, F9, F10, -1, DK, '7',
 '8', '9', '-', '4', '5', '6', '+', '1',
 '2', '3', '0', '.', CRK, '*', '#', CUPK,
 CUDK, CULK, CURK, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ
};

byte            m2ua[] =	/* matrix position to upper case ascii table */
{
 -1, 27, '!', '@', '#', '$', '%', '^',
 '&', '*', '(', ')', '_', '+', DELK, TABK,
 'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I',
 'O', 'P', '{', '}', CRK, CTRLK, 'A', 'S',
 'D', 'F', 'G', 'H', 'J', 'K', 'L', ':',
 '"', '~', SHFTK, '|', 'Z', 'X', 'C', 'V',
 'B', 'N', 'M', '<', '>', '?', SHFTK, HELPK,
 CAPSDOWN, ' ', CAPSUP, F1, F2, F3, F4, F5,
 F6, F7, F8, F9, F10, -1, DK, '7',
 '8', '9', '-', '4', '5', '6', '+', '1',
 '2', '3', '0', '.', CRK, '*', '#', CUPK,
 CUDK, CULK, CURK, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ
};

byte            a2c[] =		/* Ascii to control code table */
{
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
 -1, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, -1, -1, -1, -1, -1
};

/*
 * Initialize keyboard variables.                                             
 */
InitKey()
{
    unsigned        c;
    int             count;

    curkey = 0;			/* No current key   */
    kstat = 0;			/* Status of special keys */

    Q_OP_PORT2 = OP2_SET_KEYRESET;	/* reset keyboard */
    for (count = 50000; count > 0; count--);	/* DELAY */
    Q_OP_PORT2 = OP2_RES_KEYRESET;	/* reenable keyboard */

    c = OPENCHIP->op_kbdata;
    OPENCHIP->op_kbdstat = OP_KB_OVFL;	/* clear overflow bit in case */

    count = 0;
    while ((c = GetScan()) == 0x100)
	if (count++ == 10000)
	    break;;

    if (c == 0x100)
	return (-1);
    else
	return (0);
}

/* 
 * get a character from the keyboard if it is ready
 */
con_getc()
{
    register        key;
    register unsigned short modifier;
    register char   ascii;

    for (key = GetScan(); key < 0x100; key = GetScan()) {

	/* run key through kbintr filter */
	modifier = m2a[key & 0x7f];

	/* Is this the diamond key? Just trash it */
	if (modifier == DK)
	    goto kexit;

	/* Is this the shift key? */
	if (modifier == SHFTK || modifier == CTRLK) {
	    if (key & 0x80)
		kstat &= ~modifier;	/* modifier gone up   */
	    else
		kstat |= modifier;	/* "       "  down */
	    goto kexit;
	}
	if (modifier == CAPSUP) {	/* Caps lock key changed */
	    if (key & 0x80)	/* Caps lock key gone up */
		kstat &= modifier;	/* Caps lock off  */
	    goto kexit;
	}
	if (modifier == CAPSDOWN) {	/* Caps lock key changed */
	    if (key & 0x80)	/* Caps lock key gone up */
		kstat |= modifier;	/* Caps lock on  */
	    goto kexit;
	}
	if (key == curkey)	/* Means 2 down or up codes in a row for same
				 * key */
	    goto kexit;		/* something is wrong, so forget it */

	/* key-up or key-down code received         */
	if (key == (curkey | 0x80))	/* current key has gone up */
	    curkey = 0;		/* and zero curkey */

	if (key < 128) {	/* new key has gone down so.. */
	    curkey = key;

	    if (kstat & SHFTK)
		ascii = m2ua[curkey];	/* get upper case ascii */
	    else
		ascii = m2a[curkey];	/* Get ascii code from matrix address */

	    if (ascii < 0)
		return;		/* There is no corresponding code */

	    if (ascii < 256) {	/* Standard extended ascii code   */
		if (kstat & CTRLK)
		    ascii = a2c[ascii];	/* Get ctrl code */
		if (ascii < 0)
		    return;	/* Invalid ctrl code */
		if (kstat & CAPSLK && (ascii >= 'a' && ascii <= 'z'))
		    ascii &= 0xdf;	/* Upper case alphabetics */
		return (ascii);
	    }
	}
kexit:
    }

    return (0);
}

/*
 * get a key-up, key-down code from the keyboard
 */
GetScan()
{
    register int    ch = 0x100;
    register byte   stat, dummy;

    if ((stat = OPENCHIP->op_kbdstat) & OP_KB_RDY)
	if (stat & OP_KB_FRM) {
	    dummy = OPENCHIP->op_kbdata;
	} else
	    ch = OPENCHIP->op_kbdata;

    if (stat & OP_KB_OVFL)
	OPENCHIP->op_kbdstat = stat & (OP_KB_OVFL | OP_KB_INEA);

    return (ch);
}


/* -------Forward declarations----------------------- */

raster_line    *AddScreen();
vid_sync       *AddSection();
vid_sync       *AddSync();
raster_line    *AddFlags();
byte            SyncBits();

/* -------Constants and macros----------------------- */

/* Constants to represent the states of the sync signals */

#define HSYNC_ON        -1
#define HSYNC_OFF       0
#define VSYNC_ON        -1
#define VSYNC_OFF       0

/* Mask for the address part of a video list data pointer */
#define	VDATA_SHIFT	24
#define ADDR_MASK       (~((~0L) << VDATA_SHIFT))

/* maximum sync repeat count length. For efficiency in OpenChip, use */
/* multiples of 32 when the count will not fit into a single byte */
#define	MAXSYNCLEN	(256 - 32)

raster_line    *video_list;	/* -> the video list in memory */
raster_line   **line_addr;	/* table of pointers to raster line info */
mon_info       *monptr;		/* -> to monitor def being used */

/* space for the video memory. This is allocated n*128 bytes per raster line */
/* of display, up to a maximum of 512 lines. The array is declared 128 bytes */
/* bigger than this so we can ensure that it is 128 byte aligned, in order to */
/* make sure that the OpenChip burst mode RAM access will be ok. The pointer */
/* vidmem is set to the 128 byte aligned video memory by routine vid_mode */

#define	MAXPLANES	4
#define MAXRASTERS	512
#define MAXRASTSIZE	RASTSIZE(736,MAXPLANES)

#define	RASTSIZE(w,p)	((((w)+15)/16*sizeof(word)*(p)+127L)&~127L)

/*
byte _vidmem[MAXRASTERS * MAXRASTSIZE + 128];
*/
byte _vidmem[MAXRASTERS * RASTSIZE(736,1) + 128];

addrptr         vidmem;
raster_line    *rastptrs[MAXRASTERS];
raster_line _vidlist[600];	/* should be plenty big enough */

/* Code to create a video list */

/*
 * MAKEVIDEOLIST
 *
 * Generate the video list used by the OpenChip. This routine uses the      
 * monitor information structure set up by GetMonInfo(), and will work
 * for interlaced or non-interlaced displays with any kind of sync
 * which it is possible to generate with the OpenChip.       
 * The screen is divided into four areas vertically:                        
 *       1. Displayed scan lines.                                           
 *       2. Scan lines which are blanked before the vertical sync pulse.    
 *       3. Scan lines during the vertical sync pulse.                      
 *       4. Scan lines which are blanked after the vertical sync pulse.     
 */
MakeVideoList(video_data, data_delta)
int             data_delta;	/* The offset from one line of video data to
				 * the next (in long words) */
lword          *video_data;	/* -> the first long word of video data */
{
    register raster_line *vl = video_list;

    if (monptr->interlaced) {
	vl = AddScreen(video_data, 2 * data_delta, 1, vl);
	vl = AddScreen(video_data + data_delta, 2 * data_delta, 2, vl);
    } else
	vl = AddScreen(video_data, data_delta, 0, vl);
}

/*
 * ADDSCREEN
 *
 * Generate one screen of the video list. If the display is interlaced, it  
 * will comprise two of these screens, one displaying the even raster
 * lines, the other the odd raster lines. This will be generated by two
 * calls to this routine. The value returned by this routine is a
 * pointer to the next free entry in the video list.
 */
static raster_line *
AddScreen(dataptr, data_delta, interlaced, vl)
register lword *dataptr;	/* -> the first raster line to be displayed */
register int    data_delta;	/* dataptr increment between each line */
register int    interlaced;	/* 0 for non-interlaced, 1 for the first
				 * screen of an interlaced picture and 2 for
				 * the second screen. */
register raster_line *vl;	/* -> the video list */
{
    raster_line     tline;
    register raster_line *line = &tline;
    register vid_sync *vsptr;	/* -> sync patterns during vertical sync */
    register int    i;
    register int    line_delta;
    register int    line_num;
    register int    vis_width;

    /* Calculate the total width of the visible part of a raster line */
    vis_width = monptr->h_blank * planes;

    /* Calculate the initial line number */
    line_num = (interlaced == 2) ? 1 : 0;

    /* If this is an interlaced picture, skip every other line */
    line_delta = (interlaced != 0) ? 2 : 1;

    /*
     * The video list entries for the scan lines which contain visible data
     * are all the same except for the data pointer. To set this part of the
     * table up, the constant parts of the entry are set up in a structure,
     * which is modified and appended to the end of the video list for each
     * visible line (except the last - see below). 
     */

    /* Initialize the constant parts of the entry for a visible scan line. */
    /*
     * front_porch and back_porch are adjusted for delay of 1 long word time
     * over the BLANK and data signals 
     */

    line->front_porch.vs_count =
	((monptr->h_sync_start - monptr->h_blank) * planes - 1);
    line->front_porch.vs_flags =
	OP_VID_BLANK | SyncBits(HSYNC_OFF, VSYNC_OFF);

    line->h_sync.vs_count =
	((monptr->h_sync_end - monptr->h_sync_start) * planes);
    line->h_sync.vs_flags =
	OP_VID_BLANK | SyncBits(HSYNC_ON, VSYNC_OFF);

    line->back_porch.vs_count =
	((monptr->line_width - monptr->h_sync_end) * planes + 1);
    line->back_porch.vs_flags =
	OP_VID_BLANK | SyncBits(HSYNC_OFF, VSYNC_OFF);

    line->flags.vs_count = 0;	/* repeat count of 0, no action flags */
    line->flags.vs_flags = 0;

    /*
     * Insert the video list entries for all of the visible scan lines except
     * the last one 
     */
    for (i = line_num + line_delta; i < monptr->v_blank; i += line_delta) {
	/* Calculate the address and length of the data */
	line->data = (lword *) (((unsigned long) dataptr & ADDR_MASK) | (vis_width << VDATA_SHIFT));

	/* Point to the data for the next line */
	dataptr += data_delta;

	/* Save the address of this raster line descriptor */
	line_addr[line_num] = vl;
	line_num += line_delta;

	/* Copy this descriptor into the video list */
	*vl++ = *line;
    }

    /*
     * the entry for the last scan line is slightly different because we
     * don't want the zero word to terminate the list of sync patterns since
     * the sync pattern list will be extended to contain the three vertical
     * blanking sections. 
     */
    line->data = (lword *) (((unsigned long) dataptr & ADDR_MASK)
			    | (vis_width << VDATA_SHIFT));
    /* re-correct for line with vertical sync not BLANK */
    line->back_porch.vs_count -= 1;
    line_addr[line_num] = vl;
    *vl = *line;

    /*
     * The rest of the video list is divided into three sections which are
     * almost the same. The regular horizontal sync signal continues through
     * all three sections. The first and last sections do not have the
     * vertical sync asserted, the middle section does. There is some minor
     * messing about to get the vertical sync offset by half a scan line for
     * an interlaced picture, but each of the three sections is basically the
     * same, so they are dealt with in turn by the routine AddSection(). 
     */

    /*
     * Add the entries for the part of vertical blanking before the vertical
     * sync pulse. The use of (interlaced == 2) here ensures that the
     * vertical blanking signal is delayed by half a scan line before the
     * start of the first screen of an interlaced picture so that the first
     * scan takes place 1/2 a line above the second 
     */

    vsptr = AddSection(&vl->flags, VSYNC_OFF,
	   monptr->v_sync_start - monptr->v_blank, 0,
	   (monptr->scan_lines - monptr->v_sync_start) == 0);

    /* Add the entries for the vertical sync part of the frame. */
    vsptr = AddSection(vsptr, VSYNC_ON,
	   monptr->v_sync_end - monptr->v_sync_start, interlaced == 2,
	   (monptr->scan_lines - monptr->v_sync_end) == 0);

    /*
     * Add the entries for the part of the frame after the vertical sync
     * pulse. 
     */
    vsptr = AddSection(vsptr, VSYNC_OFF,
	   monptr->scan_lines - monptr->v_sync_end, interlaced == 2, 1);

    /* the end of the screen so next line will be a raster */
    /*
     * if half way through an interlaced picture, we need a null sync word
     * here to get the OpenChip to treat the next entry as a pointer to video
     * data. 
     */
    return (AddFlags(vsptr, (interlaced == 1) ? 0 : OP_VIDL_RSTRT));
}

/*
 * ADDSECTION
 *
 * Add a section (as described above) of the screen to the video list. The  
 * first line of the section will be half started if we are doing the
 * first screen of an interlaced display. The vertical sync pulse will
 * have been delayed by half a raster line to cause the interlacing.
 * The value returned by this routine is the new video list
 * pointer.
 */
static vid_sync *
AddSection(vsptr, v, nlines, half_line, data_next)
register vid_sync *vsptr;	/* -> into the video list */
short           v;		/* vertical sync on/off */
short           nlines;		/* number of scan lines in this section */
short           half_line;	/* TRUE if half line of -ve vertical sync */
short           data_next;	/* TRUE if the next scan line contains data */
{
    register int    i;
    register int    length;
    register int    on_hsync, off_hsync;

    if (nlines <= 0)
	return;

    /* precompute the two sync patterns required */
    off_hsync = SyncBits(HSYNC_OFF, v) | OP_VID_BLANK;
    on_hsync = SyncBits(HSYNC_ON, v) | OP_VID_BLANK;

    if (half_line) {
	/* make the first half of the scan line have inverted vertical sync */
	length = monptr->line_width / 2;
	vsptr = AddSync(vsptr, planes * length,
	    SyncBits(HSYNC_OFF, v ^ (VSYNC_ON ^ VSYNC_OFF)) | OP_VID_BLANK);
	length = monptr->h_sync_start - length;
    } else
	length = monptr->h_sync_start;

    vsptr = AddSync(vsptr, planes * length, off_hsync);

    for (i = nlines - 1; i > 0; i--) {
	vsptr = AddSync(vsptr,
	    planes * (monptr->h_sync_end - monptr->h_sync_start), on_hsync);
	vsptr = AddSync(vsptr, planes * 
	    (monptr->line_width - (monptr->h_sync_end - monptr->h_sync_start)),
	    off_hsync);
    }

    vsptr = AddSync(vsptr,
	planes * (monptr->h_sync_end - monptr->h_sync_start), on_hsync);

    /*
     * add an extra long word time if the next line contains data to correct
     * for sync propogation delay 
     */
    return (AddSync(vsptr, planes * 
	(monptr->line_width - monptr->h_sync_end) + (data_next ? 1 : 0),
	off_hsync));
}

/*
 * ADDFLAGS
 *
 * Add a word of flag information into the video list. The updated video    
 * list pointer is returned as the result of this function. Each word of    
 * flag information is composed of a 0 and a flag byte. This flag byte is   
 * passed as f to this routine.						   
 */
static raster_line *
AddFlags(vsptr, f)
register vid_sync *vsptr;	/* current video list pointer */
byte            f;		/* flags for the word */
{
    vsptr->vs_count = 0;	/* mark as a flag entry */
    vsptr->vs_flags = f;	/* save the flags value */

    vsptr = NEXTVIDSYNC(vsptr);

    /* next entry will be a raster address so ensure it is aligned */

    return (RASTERALIGN(vsptr));
}

/*
 * ADDSYNC
 *
 * Add a word of sync information into the video list. The updated video    
 * list pointer is returned as the result of this function. Each word of    
 * sync information is composed of a count and a flag byte. These are
 * passed separately as n and s to this routine.
 * AddSync is only ever called after a previous sync pattern has been
 * added to the video list in AddSection() or AddScreen() so we can
 * optimize the size and speed of stepping through it by amalgamating
 * with the previous entry wherever possible!
 */
static vid_sync *
AddSync(vsptr, n, f)
register vid_sync *vsptr;	/* current video list pointer */
register int    n;		/* repeat count for the sync word */
register byte   f;		/* flags for the sync word */
{
    register vid_sync *pvsptr = PREVVIDSYNC(vsptr);

    if (pvsptr->vs_flags == f) {/* if the last entry had the same flags then */
	vsptr = pvsptr;		/* step back and amalgamate them */
	n += vsptr->vs_count;
    }
    while (n > 255) {
	/* insert the repeat count as multiples of 32 for efficiency */
	vsptr->vs_count = MAXSYNCLEN;
	n -= MAXSYNCLEN;
	/* and insert the sync flag bits */
	vsptr->vs_flags = f;
	vsptr = NEXTVIDSYNC(vsptr);
    }
    if (n > 0) {
	/* insert the repeat count */
	vsptr->vs_count = n;
	/* and insert the sync flag bits */
	vsptr->vs_flags = f;
	vsptr = NEXTVIDSYNC(vsptr);
    }
    return vsptr;
}

/*
 * SYNCBITS
 *
 * Calculate the sync bit pattern required given the current horizontal and 
 * vertical sync states. h and v should be -1 if the signal is active and   
 * 0 otherwise. sync points to the sync[] array in the mon_info
 * structure.
 */
static byte 
SyncBits(h, v)
register short  h, v;
{
    /* masks for each sync bit */
    static word     sync_masks[] = {OP_VID_VSYNC, OP_VID_HSYNC, OP_VID_PSYNC};

    int             i;
    register word   syncbits = 0;
    register word   addbits;
    register sync_info *sync = &monptr->sync[0];

    for (i = 0; i < NSYNCS; i++) {
	switch (sync->type) {
	case DISABLED:
	    addbits = sync->polarity;
	    break;

	case VERT_SYNC:
	    addbits = sync->polarity ^ v;
	    break;

	case HORIZ_SYNC:
	    addbits = sync->polarity ^ h;
	    break;

	case COMP_SYNC:
	    addbits = sync->polarity ^ (h ^ v);
	    break;
	}

	syncbits |= (addbits & sync_masks[i]);
	sync++;
    }

    return syncbits;
}

/*
 * VID_MODE
 *
 * interface routine to set up a new video mode, given:                 
 * a pointer to a mon_info definition describing the monitor and
 * display dimensions to use.
 */
vid_mode(moninfptr)
register mon_info *moninfptr;
{
    register short int i;

    /* Set up a 128 byte aligned video memory pointer in vidmem */
    vidmem = (addrptr) ((((unsigned long) _vidmem) + 127L) & ~127L);

    /* Set up the screen form */
    screenform.bits = &vidmem;
    screenform.x = screenform.y = 0;
    screenform.width = moninfptr->h_blank * 32;
    screenform.height = moninfptr->v_blank;
    screenform.rowwords = RASTSIZE(screenform.width, planes) / (sizeof(word) * planes);
    screenform.planes = planes;
    screenform.format = 0;
    screenform.formlist = 0;

    /* Create the video list */
    monptr = moninfptr;
    video_list = (raster_line *) (((unsigned long) _vidlist + 3L) & ~3L);
    line_addr = &rastptrs[0];
    MakeVideoList(vidmem, screenform.rowwords * sizeof(word) * planes / sizeof(long));

    /* do not mask any of the logical pixel information before lookup */
    PALETTE->ip_pmask = 0xff;

    /* Enable the palette change bit in the flags word used by PalChange() */
    line_addr[screenform.height - 3]->flags.vs_flags |= OP_VIDL_PCHNG;

    /* Actually start up the OpenChip */
    /* A complex task, not to be undertaken lightly... */

    Q_OP_PORT1 = OP1_SET_VIDSTOP;	/* Assert VIDSTOP to stop the video */
    for (i = 67 / 3 + 1; i > 0; i--);	/* delay at least 67 clock periods */

    /* Set the correct video mode (VIDSEL0 and VIDSEL1). This depends on */
    /* the number of bit planes */
    /* PLANES VIDSEL0 VIDSEL1 */
    /* 1      0      0     */
    /* 2      0      1     */
    /* 4      1      0     */

    if (planes == 2)
	Q_OP_PORT1 = OP1_SET_VIDSEL0;
    else
	Q_OP_PORT1 = OP1_RES_VIDSEL0;

    if (planes == 4)
	Q_OP_PORT1 = OP1_SET_VIDSEL1;
    else
	Q_OP_PORT1 = OP1_RES_VIDSEL1;

    /* Now reset and then reenable the video fifos */
    Q_OP_PORT1 = OP1_RES_RESFIFO;	/* reset fifo */
    Q_OP_PORT1 = OP1_SET_RESFIFO;	/* enable fifo */

    /* Write a sync pattern to the openchip with DATA bit set to zero */
    OPENCHIP->op_vidsyn = 0;

    /* Write 63 to the op_vidblk register to initiate a fifo fill */
    OPENCHIP->op_vidblk = 63;

    /* wait at least 67 clock cycles for this to complete */
    for (i = 67 / 3 + 1; i > 0; i--);

    /* Write an optimum value to the op_vidcnt register. */
    /*
     * This is: 
     *
     * 1 + 32 - (((planes * prate * 37) / crate) / 32) 
     *
     * where prate is the pixel rate in MHz, and crate is the CPU clock speed in
     * MHz. The latter is extracted from the OpenChip op_sysclk register; the
     * former is a constant in the EPROM. 
     */
    OPENCHIP->op_vidcnt = 2 + 32 - (((planes * moninfptr->prate * 37 * 1000000) / OPENCHIP->op_sysclk) / 32);

    /* sync pattern whilst displaying data */
    OPENCHIP->op_viewsyn = OP_VID_DATA |
	(moninfptr->sync[0].polarity == ACTIVE_LOW ? OP_VID_VSYNC : 0) |
	(moninfptr->sync[1].polarity == ACTIVE_LOW ? OP_VID_HSYNC : 0) |
	(moninfptr->sync[2].polarity == ACTIVE_LOW ? OP_VID_PSYNC : 0);

    /* Set the op_vidstart to be at the start of the video list */
    /* this enables the OpenChip FIFO fill task */
    OPENCHIP->op_vidstart = (long *) video_list;

    /* and finally reset VIDSTOP */
    Q_OP_PORT1 = OP1_RES_VIDSTOP;	/* deassert VIDSTOP */
    /* hopefully we have a display now ! */
}

vid_stop()
{
    int             i;

    Q_OP_PORT1 = OP1_SET_VIDSTOP;	/* Assert VIDSTOP to stop the video */
    for (i = 67 / 3 + 1; i > 0; i--);	/* delay at least 67 clock periods */
}

vid_start()
{
    vid_mode(&smonitor);
}

byte redgun[16]	  = { 0, 0,32,32, 0, 0,32,32, 0, 0,63,63, 0, 0,63,63 };
byte greengun[16] = { 0, 0, 0, 0,32,32,32,32, 0, 0, 0, 0,63,63,63,63 };
byte bluegun[16]  = { 0,32, 0,32, 0,32, 0,32, 0,63, 0,63, 0,63, 0,63 };

pal_default ()
{
   register int col;
   register int t;
   register int ncolsm1;

   /* do not mask any of the logical pixel information before lookup */
   PALETTE->ip_pmask = 0xff;

   /* ncolsm1 stands for no. of colours minus 1 */
   ncolsm1 = (1 << planes) - 1;

   for (col = 0; col <= ncolsm1; col++) {
      t = (col * 15) / ncolsm1;
      pal_col(col, redgun[t], greengun[t], bluegun[t]);
   }

   Q_OP_PORT1 = OP1_RES_PALSEL0;
   Q_OP_PORT1 = OP1_RES_PALSEL1;
}

pal_col (c, r, g, b)
int c;
unsigned char r, g, b;
{
   register int pal, dith;
   register int shift, mask, colmask;
   static dithmask[] = { 0, 0x3, 0x1, 0, 0x0 };

   for (dith = 0; dith < 4; dith++)
   {
      shift = ((dith ^ 3) & dithmask[planes]) * planes;
      mask = ~((~0) << planes);
      colmask = (mask & c) << shift;
      mask <<= shift;
      for (pal = 0; pal < 4; pal++)
         pal_set (pal, dith, mask, colmask, r, g, b);
   }
}

col_set (pal, c, r, g, b)
int pal;
int c;
unsigned char r, g, b;
{
   register int dith;
   register int shift, mask, colmask;
   static dithmask[] = { 0, 0x3, 0x1, 0, 0x0 };

   for (dith = 0; dith < 4; dith++)
   {
      shift = ((dith ^ 3) & dithmask[planes]) * planes;
      mask = ~((~0) << planes);
      colmask = (mask & c) << shift;
      mask <<= shift;
      pal_set (pal, dith, mask, colmask, r, g, b);
   }
}

pal_set (pal, dith, mask, colmask, r, g, b)
register int pal, dith, mask, colmask;
unsigned char r, g, b;
{
   register int col;

   for (col = 0; col < 16; col++)
   {
      if ((col & mask) == colmask)
      {
	 PALETTE->ip_paddr = (pal << 6) | (dith << 4) | col;
	 PALETTE->ip_cval  = r;
	 PALETTE->ip_cval  = g;
	 PALETTE->ip_cval  = b;
      }
   }
}
