#define	PIXELS
#undef	FONTS_LONGWORD
#define ASSEMBLY
#include	"../dev/gpreg.h"
/******************************************************************************
* ClearRaster(raster, x, y, w, h, color)
*
* Clears the rectangle of "raster" specified by "x", "y", "w", and "h" to
* "color".
*
* C Interface:
*
*	ClearRaster(raster, x, y, w, h, color)
*	RASTER *raster;
*	int x, y, w, h;
*	COLOR color;
*
* Assembly Interface:
*
*	+---------------------------------+
*	| raster pointer                  |  0 (L)
*	+---------------------------------+
*	|////////////////| x              |  6 (W)
*	+---------------------------------+
*	|////////////////| y              | 10 (W)
*	+---------------------------------+
*	|////////////////| w              | 14 (W)
*	+---------------------------------+
*	|////////////////| h              | 18 (W)
*	+---------------------------------+
*	|////////////////| color          | 22 (W)
*	+---------------------------------+
*
******************************************************************************/
	.globl	_ClearRaster
_ClearRaster:
	link	a6,#0			/* establish frame pointer */
	moveml	#<d2-d7,a2-a5>,sp@-	/* save registers */
	movl	a6@(0+8),a0		/* get raster pointer */
	movw	a0@,a1			/* get raster width */
	movl	a0@(2),a3		/* get raster address */
	movw	a6@(6+8),d2		/* get X coordinate */
	movw	a6@(10+8),d3		/* get Y coordinate */
	movw	a6@(14+8),d4		/* get width */
	movw	a6@(18+8),d5		/* get height */
	movw	a6@(22+8),d7		/* get color */

	/* Setup registers */
	movl	#0,a2			/* indicate no source */
	jsr	lsetup			/* do the setup */
	blt	5f			/* return if nothing to do */

	/* Clear according to specified color */
	tstw	d7			/* test color */
	bne	4f			/* branch if white */

	/* Clear rectangle to black */
	notl	d2			/* complement left mask for clearing */
	notl	d3			/* complement right mask too */

    1:	andl	d2,a3@+			/* clear left partial long */
	movw	d4,d6			/* initialize word counter */
	blt	3f
    2:	clrl	a3@+			/* clear a middle long */
	dbf	d6,2b			/* loop until entire middle cleared */
    3:	andl	d3,a3@			/* clear right partial long */
	addw	a1,a3			/* bump pointer to next row */
	dbf	d5,1b			/* loop until all rows cleared */
	bra	5f			/* return */

	/* Clear rectangle to white */
    4:	moveq	#-1,d0			/* get all ones for setting */
    1:	orl	d2,a3@+			/* set left partial long */
	movw	d4,d6			/* initialize word counter */
	blt	3f
    2:	movl	d0,a3@+			/* set a middle long */
	dbf	d6,2b			/* loop until entire middle set */
    3:	orl	d3,a3@			/* set right partial long */
	addw	a1,a3			/* bump pointer to next row */
	dbf	d5,1b			/* loop until all rows cleared */

	/* Clean up stack and return */
    5:	moveml	sp@+,#<d2-d7,a2-a5>	/* restore registers */
	unlk	a6			/* remove stack frame */
	rts				/* return */

/******************************************************************************
* LSETUP
*
* This routine initializes the registers to the values needed by the raster
* operation routines. On entry the registers should contain:
*
*	D0 (W):		source rectangle X coordinate
*	D1 (W):		source rectangle Y coordinate
*	D2 (W):		destination rectangle X coordinate
*	D3 (W):		destination rectangle Y coordinate
*	D4 (W):		rectangle width
*	D5 (W):		rectangle height
*
*	A0 (W):		source raster width
*	A1 (W):		destination raster width
*	A2 (W):		source raster address (0 if no source)
*	A3 (W):		destination raster address
*
* On exit, the registers contain:
*
*	D0 (B):		?
*	D1 (B):		alignment rotation value (0 and 80)
*	D2 (W):		initial mask
*	D3 (W):		final mask
*	D4 (W):		adjusted width in longs
*	D5 (W):		adjusted height in lines
*
*	A0 (W):		source vertical increment
*	A1 (W):		destination vertical increment
*	A2 (L):		source address
*	A3 (L):		destination address
*	A4 (W):		horizontal increment
*
* Also, on exit, the N condition code is set if the height of the rectangle
* is zero.
*
* This routine uses all registers except D7, A5, and A6.  Callers with values
* to preserve should use these registers.
*
******************************************************************************/
lsetup:
/*
 * Compensate for non-positive width or height by updating the X or Y
 * coordinates.
 */
	tstw	d4			/* test the width */
	bgt	1f			/* skip adjust if already positive */
	addw	d4,d0			/* compute X before source left edge */
	addqw	#1,d0			/* X of source left edge */
	addw	d4,d2			/* compute X before destination left edge */
	addqw	#1,d2			/* X of destination left edge */
	negw	d4			/* make width positive */
    1:	tstw	d5			/* test the height */
	bgt	3f			/* skip adjust if already positive */
	addw	d5,d1			/* compute Y before source top */
	addqw	#1,d1			/* Y of source top */
	addw	d5,d3			/* compute Y before destination top */
	addqw	#1,d1			/* Y of destination top */
	negw	d5			/* make height positive */
/*
 * Compute source address and store in A2.  This is the address of the word
 * containing the first source bit.
 */
    3:	movw	a0,d6			/* get source raster width */
	mulu	d6,d1			/* compute byte offset of row */
	addl	d1,a2			/* address of row */
	movw	d0,d6			/* get source X coordinate */
	asrw	#4,d6			/* compute word offset */
	aslw	#1,d6			/* byte offset */
	addw	d6,a2			/* address of word */
/*
 * Compute destination address and store in A3.  This is the address of the
 * word containing the first destination bit.
 */
	movw	a1,d6			/* get destination raster width */
	mulu	d6,d3			/* compute byte offset of row */
	addl	d3,a3			/* address of row */
	movw	d2,d6			/* get destination X coordinate */
	asrw	#4,d6			/* compute word offset */
	aslw	#1,d6			/* byte offset */
	addw	d6,a3			/* address of word */
/*
 * Copy the destination X coordinate to D0 and adjust it so that the first bit
 * lies in the high word of a long.  This is used in the following mask and
 * width computations to ensure that the left mask contains as many bits as
 * possible.  The modulo 16 address computations above require this.
 */
	movw	d2,d0			/* copy destination x to d0 */
	bclr	#4,d0			/* force to high word */
/*
 * Compute the left mask and store in D2.  This mask has one bit for each bit
 * in the left partial words.
 */
	movw	d0,d6			/* get source x */
	andw	#0x1F,d6		/* take modulo 32 (size of long) */
	moveq	#-1,d2			/* get all ones */
	lsrl	d6,d2			/* shift in the proper number of zeros */
/*
 * Compute the right mask and store in D3.  This mask has one bit for each bit
 * in the right partial words.  A side effect of this computation is that D4
 * contains the X coordinate of the last source bit on the first row.
 */
	addw	d0,d4			/* add source x to width */
	subqw	#1,d4			/* adjust to get x of last bit */
	movw	d4,d6			/* copy it */
	notw	d6			/* reverse bit numbering scheme */
	andw	#0x1F,d6		/* take modulo 32 */
	moveq	#-1,d3			/* get all ones */
	lsll	d6,d3			/* shift in the proper number of zeros */
/*
 * Compute the width in longs of the middle section and store in D4.  This is
 * the difference between the first and last long offsets, minus one.
 */
	asrw	#5,d0			/* compute long offset of first bit */
	asrw	#5,d4			/* compute long offset of last bit */
	subw	d0,d4			/* take their difference */
	subqw	#1,d4			/* and adjust */
/*
 * Adjust for the case where the entire row lies within a single long.  This
 * is indicated by a negative width.  In this case, only the left mask is used
 * and the right mask is zeroed.
 */
	bge	4f			/* branch if width is non-negative */
	andl	d3,d2			/* combine masks if width negative */
	clrl	d3			/* clear left mask */
	clrw	d4			/* and zero width */
/*
 * Compute the source and destination vertical increments and store in A0 
 * and A1.  These are the amounts by which the pointers have to be incremented
 * at the end of one row to get to the beginning of the next.  Also store
 * 4 as the horizontal increment in A4.
 */
    4:	movw	d4,d6			/* get width of middle section */
	aslw	#2,d6			/* convert to a byte width */
	subw	d6,a0			/* subtract middle from source width */
	subqw	#4,a0			/* subtract left side */
	subw	d6,a1			/* subtract middle from destination */
	subqw	#4,a1			/* subtract left side */
	movw	#4,a4			/* initialize horizontal increment */
/*
 * Finally, adjust the width and height for the dbf instruction used by the
 * raster routines.  Also set the N bit if the height is zero.
 */
	subqw	#1,d4			/* adjust width */
	subqw	#1,d5			/* adjust height (and set N) */
	rts				/* return */

/******************************************************************************
* CopyRaster(sraster, sx, sy, draster, dx, dy, w, h)
*
* Copies the rectangle of width "w" and height "h" from location ("sx", "sy")
* of "sraster" to location "("dx", "dy") of "draster".
*
* C Interface:
*
*	CopyRaster(sraster, sx, sy, draster, dx, dy, w, h)
*	RASTER *sraster, *draster;
*	int sx, sy, dx, dy, w, h;
*
* Assembly Interface:
*
*	+---------------------------------+
*	| sraster                         |  0 (L)
*	+---------------------------------+
*	|////////////////| sx             |  6 (W)
*	+---------------------------------+
*	|////////////////| sy             | 10 (W)
*	+---------------------------------+
*	| draster                         | 12 (L)
*	+---------------------------------+
*	|////////////////| dx             | 18 (W)
*	+---------------------------------+
*	|////////////////| dy             | 22 (W)
*	+---------------------------------+
*	|////////////////| w              | 26 (W)
*	+---------------------------------+
*	|////////////////| h              | 30 (W)
*	+---------------------------------+
*
******************************************************************************/
	.globl	_CopyRaster
_CopyRaster:
	link	a6,#0			/* establish frame pointer */
	moveml	#<d2-d7,a2-a5>,sp@-	/* save registers */
	movl	a6@(0+8),a0		/* get source raster pointer */
	movl	a0@(2),a2		/* get raster address */
	movw	a0@,a0			/* get raster width */
	movw	a6@(6+8),d0		/* get source X coordinate */
	movw	a6@(10+8),d1		/* get source Y coordinate */
	movl	a6@(12+8),a1		/* get destination raster pointer */
	movl	a1@(2),a3		/* get raster address */
	movw	a1@,a1			/* get raster width */
	movw	a6@(18+8),d2		/* get destination X coordinate */
	movw	a6@(22+8),d3		/* get destination Y coordinate */
	movw	a6@(26+8),d4		/* get width */
	movw	a6@(30+8),d5		/* get height */

	/* Setup registers */
	jsr	setup			/* do the setup */
	blt	9f			/* return if nothing to do */

	/* Copy according to direction and alignment */
	movw	d1,d0			/* get rotation */
	andw	#0xf,d0			/* test rotation mod 16 */
	beq	align			/* branch if aligned */
	cmpl	#0,a4			/* test inc direction for unaligned */
	bgt	uninc			/* branch if incrementing */
	bra	undec			/* branch if decrementing */

align:	cmpl	#0,a4			/* test inc direction for unaligned */
	bgt	alinc			/* branch if incrementing */
	bra	aldec			/* branch if decrementing */

	/* Copy unaligned, incrementing addresses */
uninc:
	tstw	d1			/* check for pre-load case */
	blt	incp1			/* yep, negative shift */
	andw	#0xf,d1			/* take shift mod 16 */
	addql	#2,a0			/* bump source vertical increment */
	addql	#2,a2			/* bump source address */
    1:	movw	a2@,d6			/* read left source word */
	lsrl	d1,d6			/* align with destination */
	andw	d2,d6			/* keep valid bits */
	notw	d2			/* reverse left mask */
	movw	a3@,d7			/* read destination bits */
	andw	d2,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@+			/* write destination back */
	notw	d2			/* restore left mask */
	movw	d4,d0			/* initialize word counter */
	blt	3f			/* skip loop if zero */
    2:	movl	a2@,d6			/* read a source word */
	lsrl	d1,d6			/* align with destination */
	movw	d6,a3@+			/* write it out */
	addql	#2,a2			/* point to next word */
	dbf	d0,2b			/* loop until row finished */
    3:	cmpw	#0x0,d3			/* check if no final mask */
	jeq	7f			/* loop around if true */
	movl	a2@,d6			/* read left source word */
	lsrl	d1,d6			/* align with destination */
	andw	d3,d6			/* keep valid bits */
	notw	d3			/* reverse right mask */
	movw	a3@,d7			/* read destination bits */
	andw	d3,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
	notw	d3			/* restore right mask */
    7:	addw	a0,a2			/* bump source address to next row */
	addw	a1,a3			/* bump destination address to next row */
	dbf	d5,1b			/* loop until block finished */
	bra	9f			/* return */

incp1:	andw	#0xf,d1			/* get a positive shift count */
    1:	movl	a2@,d6			/* read left source word */
	lsrl	d1,d6			/* align with destination */
	andw	d2,d6			/* keep valid bits */
	notw	d2			/* reverse left mask */
	movw	a3@,d7			/* read destination bits */
	andw	d2,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@+			/* write destination back */
	addql	#2,a2			/* bump source address */
	notw	d2			/* restore left mask */
	movw	d4,d0			/* initialize word counter */
	blt	3f			/* skip loop if zero */
    2:	movl	a2@,d6			/* read a source word */
	lsrl	d1,d6			/* align with destination */
	movw	d6,a3@+			/* write it out */
	addql	#2,a2			/* point to next word */
	dbf	d0,2b			/* loop until row finished */
    3:	cmpw	#0x0,d3			/* check if no final mask */
	jeq	7f			/* loop around if true */
	movl	a2@,d6			/* read left source word */
	lsrl	d1,d6			/* align with destination */
	andw	d3,d6			/* keep valid bits */
	notw	d3			/* reverse right mask */
	movw	a3@,d7			/* read destination bits */
	andw	d3,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
	notw	d3			/* restore right mask */
    7:	addw	a0,a2			/* bump source address to next row */
	addw	a1,a3			/* bump destination address to next row */
	dbf	d5,1b			/* loop until block finished */
	bra	9f			/* return */

	/* Copy unaligned, decrementing addresses */
undec:	tstw	d1			/* check for pre-load case */
	blt	decp1			/* yep, negative shift */
	andw	#0xf,d1			/* take shift mod 16 */
	subql	#2,a0			/* adjust src vertical incr */
    1:	movl	a2@,d6			/* read right source word */
	lsrl	d1,d6			/* align with destination */
	andw	d2,d6			/* keep valid bits */
	notw	d2			/* reverse left mask */
	movw	a3@,d7			/* read destination bits */
	andw	d2,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
	notw	d2			/* restore left mask */
	movw	d4,d0			/* initialize word counter */
	blt	3f			/* skip loop if zero */
    2:	subql	#2,a2			/* point to next word */
	movl	a2@,d6			/* read a source word */
	lsrl	d1,d6			/* align with destination */
	movw	d6,a3@-			/* write it out */
	dbf	d0,2b			/* loop until row finished */
    3:	cmpw	#0x0,d3			/* check if no final mask */
	jeq	7f			/* loop around if true */
	movw	a2@,d6			/* read right source word */
	lsrl	d1,d6			/* align with destination */
	andw	d3,d6			/* keep valid bits */
	notw	d3			/* reverse right mask */
	movw	a3@-,d7			/* read destination bits */
	andw	d3,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
	notw	d3			/* restore right mask */
    7:	addw	a0,a2			/* bump source address to next row */
	addw	a1,a3			/* bump destination address to next row */
	dbf	d5,1b			/* loop until block finished */
	bra	9f			/* return */

decp1:	andw	#0xf,d1			/* convert shift to positive */
    1:	movl	a2@,d6			/* read right source word */
	lsrl	d1,d6			/* align with destination */
	andw	d2,d6			/* keep valid bits */
	notw	d2			/* reverse left mask */
	movw	a3@,d7			/* read destination bits */
	andw	d2,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
    	subql	#2,a2			/* bump source address */
	notw	d2			/* restore left mask */
	movw	d4,d0			/* initialize word counter */
	blt	3f			/* skip loop if zero */
    2:	movl	a2@,d6			/* read a source word */
	lsrl	d1,d6			/* align with destination */
	movw	d6,a3@-			/* write it out */
	subql	#2,a2			/* point to next word */
	dbf	d0,2b			/* loop until row finished */
    3:	cmpw	#0x0,d3			/* check if no final mask */
	jeq	7f			/* loop around if true */
	movl	a2@,d6			/* read right source word */
	lsrl	d1,d6			/* align with destination */
	andw	d3,d6			/* keep valid bits */
	notw	d3			/* reverse right mask */
	movw	a3@-,d7			/* read destination bits */
	andw	d3,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
	notw	d3			/* restore right mask */
    7:	addw	a0,a2			/* bump source address to next row */
	addw	a1,a3			/* bump destination address to next row */
	dbf	d5,1b			/* loop until block finished */
	bra	9f			/* return */

	/* Copy aligned, incrementing addresses */
alinc:	addql	#2,a2			/* point to first source word */
	tstw	d4			/* null middle section? */
	bge	aligen			/* do slow version if not */
	cmpw	#0xFFFF,d2		/* starts on word boundary? */
	bne	aligen			/* do slow version if not */
	tstw	d3			/* ends on word boundary? */
	bne	aligen			/* do slow version if not */

	/* Copy aligned, word width, word boundary alignment */
	addw	#2,a0			/* correct source increment */
	addw	#2,a1			/* correct destination increment */
    1:	movw	a2@,a3@			/* copy a word */
	addw	a0,a2			/* bump source address */
	addw	a1,a3			/* bump destination address */
	dbf	d5,1b			/* do next word */
	bra	9f			/* return */

	/* Copy aligned, incrementing address, arbitrary width */
aligen:
    1:	movw	a2@+,d6			/* read left source word */
	andw	d2,d6			/* keep valid bits */
	notw	d2			/* reverse left mask */
	movw	a3@,d7			/* read destination bits */
	andw	d2,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@+			/* write destination back */
	notw	d2			/* restore left mask */
	movw	d4,d0			/* initialize word counter */
	blt	3f			/* skip loop if zero */
    2:	movw	a2@+,a3@+		/* write it out */
	dbf	d0,2b			/* loop until row finished */
    3:	movw	a2@,d6			/* read left source word */
	andw	d3,d6			/* keep valid bits */
	notw	d3			/* reverse right mask */
	movw	a3@,d7			/* read destination bits */
	andw	d3,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
	notw	d3			/* restore right mask */
	addw	a0,a2			/* bump source address to next row */
	addw	a1,a3			/* bump destination address to next row */
	dbf	d5,1b			/* loop until block finished */
	bra	9f			/* return */

	/* Copy aligned, decrementing addresses */
aldec:	addql	#2,a2			/* point to first source word */
    1:	movw	a2@,d6			/* read right source word */
	andw	d2,d6			/* keep valid bits */
	notw	d2			/* reverse left mask */
	movw	a3@,d7			/* read destination word */
	andw	d2,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
	notw	d2			/* restore left mask */
	movw	d4,d0			/* initialize word counter */
	blt	3f			/* skip loop if zero */
    2:	movw	a2@-,a3@-		/* read a source word */
	dbf	d0,2b			/* loop until row finished */
    3:	movw	a2@-,d6			/* read right source word */
	andw	d3,d6			/* keep valid bits */
	notw	d3			/* reverse right mask */
	movw	a3@-,d7			/* read destination bits */
	andw	d3,d7			/* keep bits outside range */
	orw	d6,d7			/* combine with source bits */
	movw	d7,a3@			/* write destination back */
	notw	d3			/* restore right mask */
	addw	a0,a2			/* bump source address to next row */
	addw	a1,a3			/* bump destination address to next row */
	dbf	d5,1b			/* loop until block finished */

	/* Clean up stack and return */
    9:	moveml	sp@+,#<d2-d7,a2-a5>	/* restore registers */
	unlk	a6			/* remove stack frame */
	rts				/* return */

/******************************************************************************
* DrawString(x, y, font, color, length, string)
*
* Draws "string" at location ("x", "y") of the screen using "font" and
* "color".  The string is terminated whenever "length" characters have been
* drawn or a null character is reached.  The string is clipped to the current
* clipping rectangle.
* NOTE: whenever mote - pixel conversions are done, 8 motes/pixel is assumed.
*
* C Interface:
*
*	DrawString(x, y, plane, color, length, string)
*	int x, y;
*	font_plane *plane;
*	COLORT color;
*	int length;
*	char *string;
*
* Assembly Interface:
*
*	+---------------------------------+
*	| ///////////////| x              |  2 (W)
*	+---------------------------------+
*	|////////////////| y              |  6 (W)
*	+---------------------------------+
*	| plane                           |  8 (L)
*	+---------------------------------+
*	| color                           | 12 (L)
*	+---------------------------------+
*	|////////////////| length         | 18 (W)
*	+---------------------------------+
*	| string                          | 20 (L)
*	+---------------------------------+
*
******************************************************************************/
	.globl	_DrawString
_DrawString:
	link	a6,#0			/* establish stack frame */
	moveml	#<d2-d7,a2-a4>,sp@-	/* save working registers */
	movw	a6@(2+8),d5		/* get x coordinate */
	movw	a6@(6+8),d6		/* get y coordinate */
	movl	a6@(8+8),a3		/* get font plane descriptors address */
	movl	a6@(12+8),d0		/* get color */
	movw	a6@(18+8),d7		/* get length */
	movl	a6@(20+8),a4		/* get string address */

	/* Setup stack frame for CopyRaster */
	clrl	sp@-			/* space for pointer to char rasters */
	clrw	sp@-			/* reserve space for raster width */
	movl	d0,sp@-			/* pass color */
	clrl	sp@-			/* reserve space for height */
	clrl	sp@-			/* reserve space for width */
	clrl	sp@-			/* reserve space for destination y */
	clrl	sp@-			/* reserve space for destination x */
	movl	#_screen,sp@-		/* pass screen descriptor */
	clrl	sp@-			/* reserve space for source y */
	clrl	sp@-			/* reserve space for source x */
	movl	sp,d4			/* get copy of sp */
	addl	#32,d4			/* make pointer to character descriptor */
	movl	d4,sp@-			/* pass character descriptor */

	/* Draw characters until exhausted */
	tstw	d7			/* is it a null string */
	beq	dst2			/* exit if so */
	subqw	#1,d7			/* adjust length for dbf */
dst1:	movb	a4@+,d4			/* get a character */
	beq	dst2			/* exit loop if no more */
	andw	#0x7f,d4		/* extend to a word */
	subw	#0x20,d4		/* reduce to range 0-95 */
	bmi	3f			/* skip if not in range */
	aslw	#3,d4			/* turn it into a table index */
	lea	a3@(0,d4:W),a2		/* get dispatch entry for character */
	movb	a2@(4),d0		/* get x offset of character */
	extw	d0			/* extend to word */
#ifndef PIXELS
	aslw	#3,d0			/* * 8 to convert to motes */
#endif PIXELS
	addw	d5,d0			/* compute destination x */
	movw	d0,a0			/* save it */
	movb	a2@(5),d1		/* get y offset of character */
	extw	d1			/* extend to word */
#ifndef PIXELS
	aslw	#3,d1			/* * 8 to convert to motes */
#endif PIXELS
	addw	d6,d1			/* compute destination y */
	movw	d1,a1			/* save it */
	moveq	#0,d2			/* clear d2 */
	movb	a2@(2),d2		/* get raster width */
	movl	d2,d3			/* get copy of char width in pixels */
#ifndef PIXELS
	aslw	#3,d2			/* * 8 to convert to motes */
#endif PIXELS
#ifdef FONTS_LONGWORD
	addw	#31,d3			/* round up to a long word width */
	lsrw	#5,d3			/* pixel width in long words for char */
	aslw	#2,d3			/* pixel width in bytes */
#else FONTS_LONGWORD
	addw	#15,d3			/* round up to a word width */
	lsrw	#4,d3			/* pixel width in words for this char */
	aslw	#1,d3			/* pixel width in bytes */
#endif FONTS_LONGWORD
	movw	d3,sp@(36)		/* store in local raster descriptor */
	moveq	#0,d3			/* clear d3 */
	movb	a2@(3),d3		/* pass raster height */
#ifndef PIXELS
	aslw	#3,d3			/* * 8 to convert to motes */
#endif PIXELS
	movw	d0,sp@(18)		/* pass clipped X */
	movw	d1,sp@(22)		/* pass clipped Y */
	movw	d2,sp@(26)		/* pass clipped width */
	movw	d3,sp@(30)		/* pass clipped height */
	subw	a0,d0			/* compute x clipping offset */
	movw	d0,sp@(6)		/* pass it */
	movl	a3@(8*96+2),a0		/* get pointer to start of rasters */
	addw	a2@(6),a0		/* add character offset (bytes) */
	movl	a0,sp@(38)		/* store in raster structure */
	subw	a1,d1			/* compute y clipping offset */
	movw	d1,sp@(10)		/* pass as source y */
	jsr	_CopyRaster		/* draw the character */
    2:	moveq	#0,d4			/* clear d4 */
	movb	a2@,d4			/* get character width */
#ifndef PIXELS
	aslw	#3,d4			/* * 8 to convert to motes */
#endif PIXELS
	addw	d4,d5			/* bump destination x */
    3:	dbf	d7,dst1			/* loop until count exhausted */
dst2:	addl	#42,sp			/* remove parameters from stack */
	moveml	sp@+,#<d2-d7,a2-a4>	/* restore working registers */
	unlk	a6			/* unlink stack frame */
	rts				/* return */

/******************************************************************************
* SETUP
*
* This routine initializes the registers to the values needed by the raster
* operation routines.  On entry, the registers should contain:
*
*	D0 (W):		source rectangle X coordinate
*	D1 (W):		source rectangle Y coordinate
*	D2 (W):		destination rectangle X coordinate
*	D3 (W):		destination rectangle Y coordinate
*	D4 (W):		rectangle width
*	D5 (W):		rectangle height
*
*	A0 (W):		source raster width
*	A1 (W):		destination raster width
*	A2 (W):		source raster address (0 if no source)
*	A3 (W):		destination raster address
*
* On exit, the registers contain:
*
*	D0 (B):		?
*	D1 (W):		alignment rotation
*	D2 (W):		initial mask
*	D3 (W):		final mask
*	D4 (W):		adjusted width in words
*	D5 (W):		adjusted height in lines
*
*	A0 (W):		source vertical increment
*	A1 (W):		destination vertical increment
*	A2 (L):		source address
*	A3 (L):		destination address
*	A4 (W):		horizontal increment
*
* Also, on exit, the N condition code is set if the height of the rectangle
* is zero.
*
* This routine uses all registers except D7, A5, and A6.  Callers with values
* to preserve should use these registers.
*
******************************************************************************/
	.globl	setup
setup:	tstw	d4			/* test the width */
	bgt	1f			/* skip adjust if already positive */
	beq	zilch			/* return with N bit if width zero */
	addw	d4,d0			/* compute X before source left edge */
	addqw	#1,d0			/* X of source left edge */
	addw	d4,d2			/* compute X before destination left edge */
	addqw	#1,d2			/* X of destination left edge */
	negw	d4			/* make width positive */
    1:	tstw	d5			/* test the height */
	bgt	3f			/* skip adjust if already positive */
	beq	zilch			/* return with N bit if height zero */
	addw	d5,d1			/* compute Y before source top */
	addqw	#1,d1			/* Y of source top */
	addw	d5,d3			/* compute Y before destination top */
	addqw	#1,d3			/* Y of destination top */
	negw	d5			/* make height positive */
/*
 * Determine whether addresses should be incremented or decremented.  If the
 * source and destination overlap and the source precedes the destination,
 * addresses must be decremented to avoid overwriting portions of the source
 * before they are copied.  In all other cases, addresses are incremented.
 *
 * For aesthetic reasons, addresses are incremented whenever possible.  The
 * same raster and overlap tests can be eliminated if this is not desired.
 */
    3:	cmpl	a2,a3			/* same raster? */
	bne	isetup			/* increment addresses if not */
	cmpw	d1,d3			/* is destination before source? */
	blt	isetup			/* increment addresses if yes */
	bgt	4f			/* test for overlap addresses if no */
	cmpw	d0,d2			/* check x coordinates if maybe */
	blt	isetup			/* increment if destination is first */
	movw	d0,d6			/* get source x coordinate */
	addw	d4,d6			/* compute first x after source */
	cmpw	d2,d6			/* compare with left side of destination */
	bgt	dsetup			/* decrement if overlap */
	bra	isetup			/* otherwise increment */
    4:	movw	d1,d6			/* get source y coordinate */
	addw	d5,d6			/* compute first y after source */
	cmpw	d3,d6			/* compare with top of destination */
	bgt	dsetup			/* decrement if overlap */
/*
 * Compute source address and store in A2.  This is the address of the word
 * before the word that contains the source bit corresponding to the low order
 * bit of the left partial destination word.
 */
isetup:	movw	a0,d6			/* get source raster width */
	mulu	d6,d1			/* compute offset of row */
	addl	d1,a2			/* address of row */
	movw	d2,d1			/* get destination x */
	notw	d1			/*  */
	andw	#0xF,d1			/* size of left partial minus 1 */
	addw	d0,d1			/* position of last partial source bit */
	asrw	#4,d1			/* word offset */
	subqw	#1,d1			/* point to previous word */
	aslw	#1,d1			/* byte offset */
	addw	d1,a2			/* address of source */
/*
 * Compute destination address and store in A3.  The destination address is the
 * address of the word containing the first destination bit.
 */
	movw	a1,d6			/* get destination raster width */
	mulu	d6,d3			/* compute offset of row */
	addl	d3,a3			/* address of row */
	movw	d2,d6			/* copy destination x */
	asrw	#4,d6			/* compute word offset */
	aslw	#1,d6			/* byte offset */
	addw	d6,a3			/* address of first word */
/*
 * Compute X of last destination bit and store in D3.  This is simply the
 * X of the first bit plus the width minus 1.
 */
	movw	d2,d3			/* copy destination x */
	addw	d4,d3			/* add width */
	subqw	#1,d3			/* and adjust */
/*
 * Compute the amount of right shift necessary to align the source with the
 * destination.  This is the difference of their X's taken modulo 16.
 */
	movw	d2,d1			/* get destination x */
	subw	d0,d1			/* compute difference */
	andw	#0xF,d1			/* take result modulo 16 */
/*
 * Compute width in words of middle section and store in D4.  This is the
 * difference between the first and last addresses minus 1.
 */
	movw	d3,d4			/* get x of last point */
	asrw	#4,d4			/* find its word address */
	movw	d2,d6			/* get x of first point */
	asrw	#4,d6			/* find its word address */
	subw	d6,d4			/* compute difference */
	subqw	#1,d4			/* and adjust */
/*
 * Compute the left mask and store in D2.  This mask is right justified and
 * contains one bit for each bit of the destination in the first word.
 */
	movw	#0xFFFF,d6		/* start with all ones */
	andw	#0xF,d2			/* take initial x modulo word width */
	lsrw	d2,d6			/* clear appropriate number of bits */
	movw	d6,d2			/* put the result back in d2 */

/*
 * Compute the right mask and store in D3.  This mask is left justified and
 * contains one bit for each bit of the destination in the final word.
 */
	andw	#0xF,d3			/* take final x modulo word width */
	movw	#15,d6			/* get index of high order bit */
	subw	d3,d6			/* compute bits to right of final bit */
	movw	#0xFFFF,d3		/* start with all ones */
	lslw	d6,d3			/* clear appropriate number of bits */
/*
 * Adjust for the case where the entire width is within a single word.  In
 * this case, use just the left mask.
 */
	tstw	d4			/* single word (width negative)? */
	bge	1f			/* branch if not */
	andw	d3,d2			/* otherwise combine masks */
	clrw	d3			/* zero right mask */
	clrw	d4			/* zero width */
/*
 * Compute source and destination row increments and store in A0 and A1.  These
 * are the raster widths minus the width of the left and middle sections.
 */
    1:	subw	d4,a0			/* adjust source raster width */
	subw	d4,a0			/* do it again for byte addresses */
	subqw	#2,a0			/* account for left word */
	subw	d4,a1			/* adjust destination raster width */
	subw	d4,a1			/* do it again for byte addresses */
	subqw	#2,a1			/* account for left word */
	movw	#2,a4			/* initialize horizontal increment */
	bra	adjust			/* adjust width and height and return */
/*
 * Compute source address and store in A1.  This is the address of the source
 * word containing the bit corresponding to the last bit in the next to last
 * destination word.
 */
dsetup:	movw	a0,d6			/* get source raster width */
	addw	d5,d1			/* add height to y */
	subqw	#1,d1			/* compute y of last row */
	mulu	d6,d1			/* compute offset of row */
	addl	d1,a2			/* address of row */
	movw	d2,d1			/* get destination x */
	addw	d4,d1			/* add width to it */
	subqw	#1,d1			/* compute x of last bit */
	andw	#0xFFF0,d1		/* compute x of last word */
	subqw	#1,d1			/* x of bit before */
	subw	d2,d1			/* offset from beginning of destination */
	addw	d0,d1			/* corresponding source bit */
	asrw	#4,d1			/* word offset from beginning of row */
	aslw	#1,d1			/* byte offset */
	addw	d1,a2			/* address of source */
/*
 * Compute destination address and store in A3.  The destination address is the
 * address of the word containing the last destination bit.
 */
	movw	a1,d6			/* get destination raster width */
	addw	d5,d3			/* add height to y */
	subqw	#1,d3			/* compute y of last row */
	mulu	d6,d3			/* compute offset of row */
	addl	d3,a3			/* address of row */
	movw	d2,d6			/* copy destination x */
	addw	d4,d6			/* add width to x */
	subqw	#1,d6			/* compute x of last bit */
	asrw	#4,d6			/* compute word offset */
	aslw	#1,d6			/* byte offset */
	addw	d6,a3			/* address of destination */
/*
 * Compute X of last destination bit and store in D2.  This is simply the
 * X of the first bit plus the width minus 1.  Store the X of the first bit
 * in D3.
 */
	movw	d2,d3			/* copy destination x */
	addw	d4,d2			/* add width */
	subqw	#1,d2			/* and adjust */
/*
 * Compute the amount of right shift necessary to align the source with the
 * destination.  This is the difference of their X's taken modulo 16.
 */
	movw	d3,d1			/* get destination x */
	subw	d0,d1			/* compute difference */
	andw	#0xF,d1			/* take result modulo 16 */
/*
 * Compute width in words of middle section and store in D4.  This is the
 * difference between the first and last addresses minus 1.
 */
	movw	d2,d4			/* get x of last point */
	asrw	#4,d4			/* find its word address */
	movw	d3,d6			/* get x of first point */
	asrw	#4,d6			/* find its word address */
	subw	d6,d4			/* compute difference */
	subqw	#1,d4			/* and adjust */
/*
 * Compute the left mask and store in D3.  This mask is right justified and
 * contains one bit for each bit of the destination in the first word.
 */
	movw	#0xFFFF,d6		/* start with all ones */
	andw	#0xF,d3			/* take initial x modulo word width */
	lsrw	d3,d6			/* clear appropriate number of bits */
	movw	d6,d3			/* put the result back in d2 */
/*
 * Compute the right mask and store in D2.  This mask is left justified and
 * contains one bit for each bit of the destination in the final word.
 */
	andw	#0xF,d2			/* take final x modulo word width */
	movw	#15,d6			/* get index of high order bit */
	subw	d2,d6			/* compute bits to right of final bit */
	movw	#0xFFFF,d2		/* start with all ones */
	lslw	d6,d2			/* clear appropriate number of bits */
/*
 * Adjust for the case where the entire width is within a single word.  In
 * this case, use just the left mask.
 */
	tstw	d4			/* single word (width negative)? */
	bge	1f			/* branch if not */
	andw	d3,d2			/* otherwise combine masks */
	clrw	d3			/* zero right mask */
	clrw	d4			/* zero width */
/*
 * Compute source and destination row increments and store in A0 and A1.  These
 * are the raster widths minus the width of the left and middle sections.
 */
    1:	subw	d4,a0			/* adjust source raster width */
	subw	d4,a0			/* do it again for byte addresses */
	subqw	#2,a0			/* account for left word */
	movw	a0,d6			/* copy for negation */
	negw	d6			/* negate for going backward */
	movw	d6,a0			/* copy back */
	subw	d4,a1			/* adjust destination raster width */
	subw	d4,a1			/* do it again for byte addresses */
	subqw	#2,a1			/* account for left word */
	movw	a1,d6			/* copy for negation */
	negw	d6			/* negate for going backward */
	movw	d6,a1			/* copy back */
	movw	#-2,a4			/* initialize horizontal increment */
/*
 * Adjust width and height.  The raster routines use the dbf instruction in
 * their loops.  Since this instruction (inconveniently) loops until the
 * specified register reached -1, the counts must be adjusted.
 */
adjust:	
	subqw	#1,d4			/* adjust width */
	subqw	#1,d5			/* adjust height (and set N bit) */
	rts				/* return */
zilch:	moveq	#-1,d1			/* set N bit */
	rts				/* return */

/*
 * ASCII plane for default font schulz11bn.f to be linked into kernel.
 */
	.globl	_plane_desc
	.data
	.align	1
_plane_desc:
/* Character descriptors */
	.byte	0x0b, 0x00, 0x00, 0x00, 0x00, 0x00	/* 00 */
	.word	0x0000
	.byte	0x0b, 0x00, 0x02, 0x0b, 0x02, 0xf6	/* 01 */
	.word	0x0000
	.byte	0x0b, 0x00, 0x06, 0x05, 0x00, 0xf5	/* 02 */
	.word	0x0016
	.byte	0x0b, 0x00, 0x0a, 0x0d, 0x00, 0xf6	/* 03 */
	.word	0x0020
	.byte	0x0b, 0x00, 0x07, 0x0d, 0x01, 0xf5	/* 04 */
	.word	0x003a
	.byte	0x0b, 0x00, 0x0b, 0x0b, 0x00, 0xf6	/* 05 */
	.word	0x0054
	.byte	0x0b, 0x00, 0x09, 0x0b, 0x00, 0xf6	/* 06 */
	.word	0x006a
	.byte	0x0b, 0x00, 0x03, 0x04, 0x00, 0xf6	/* 07 */
	.word	0x0080
	.byte	0x0b, 0x00, 0x04, 0x0f, 0x01, 0xf5	/* 08 */
	.word	0x0088
	.byte	0x0b, 0x00, 0x04, 0x0f, 0x00, 0xf5	/* 09 */
	.word	0x00a6
	.byte	0x0b, 0x00, 0x0a, 0x07, 0x00, 0xf6	/* 0a */
	.word	0x00c4
	.byte	0x0b, 0x00, 0x0a, 0x0a, 0x00, 0xf7	/* 0b */
	.word	0x00d2
	.byte	0x0b, 0x00, 0x03, 0x04, 0x00, 0xff	/* 0c */
	.word	0x00e6
	.byte	0x0b, 0x00, 0x09, 0x02, 0x00, 0xfa	/* 0d */
	.word	0x00ee
	.byte	0x0b, 0x00, 0x03, 0x03, 0x00, 0xfe	/* 0e */
	.word	0x00f2
	.byte	0x0b, 0x00, 0x09, 0x10, 0x00, 0xf5	/* 0f */
	.word	0x00f8
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 10 */
	.word	0x0118
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x01, 0xf6	/* 11 */
	.word	0x012e
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 12 */
	.word	0x0144
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 13 */
	.word	0x015a
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 14 */
	.word	0x0170
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 15 */
	.word	0x0186
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 16 */
	.word	0x019c
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 17 */
	.word	0x01b2
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x01, 0xf6	/* 18 */
	.word	0x01c8
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 19 */
	.word	0x01de
	.byte	0x0b, 0x00, 0x03, 0x08, 0x01, 0xf8	/* 1a */
	.word	0x01f4
	.byte	0x0b, 0x00, 0x03, 0x09, 0x03, 0xfa	/* 1b */
	.word	0x0204
	.byte	0x0b, 0x00, 0x09, 0x09, 0x00, 0xf8	/* 1c */
	.word	0x0216
	.byte	0x0b, 0x00, 0x08, 0x06, 0x00, 0xf9	/* 1d */
	.word	0x0228
	.byte	0x0b, 0x00, 0x0a, 0x09, 0x00, 0xf8	/* 1e */
	.word	0x0234
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 1f */
	.word	0x0246
	.byte	0x0b, 0x00, 0x0a, 0x0a, 0x00, 0xf7	/* 20 */
	.word	0x025c
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 21 */
	.word	0x0270
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 22 */
	.word	0x0286
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 23 */
	.word	0x029c
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 24 */
	.word	0x02b2
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 25 */
	.word	0x02c8
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 26 */
	.word	0x02de
	.byte	0x0b, 0x00, 0x09, 0x0b, 0x00, 0xf6	/* 27 */
	.word	0x02f4
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 28 */
	.word	0x030a
	.byte	0x0b, 0x00, 0x06, 0x0b, 0x02, 0xf6	/* 29 */
	.word	0x0320
	.byte	0x0b, 0x00, 0x09, 0x0b, 0x00, 0xf6	/* 2a */
	.word	0x0336
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 2b */
	.word	0x034c
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x01, 0xf6	/* 2c */
	.word	0x0362
	.byte	0x0b, 0x00, 0x09, 0x0b, 0x00, 0xf6	/* 2d */
	.word	0x0378
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 2e */
	.word	0x038e
	.byte	0x0b, 0x00, 0x09, 0x0b, 0x00, 0xf6	/* 2f */
	.word	0x03a4
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x01, 0xf6	/* 30 */
	.word	0x03ba
	.byte	0x0b, 0x00, 0x09, 0x0d, 0x00, 0xf6	/* 31 */
	.word	0x03d0
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 32 */
	.word	0x03ea
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 33 */
	.word	0x0400
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 34 */
	.word	0x0416
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 35 */
	.word	0x042c
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 36 */
	.word	0x0442
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 37 */
	.word	0x0458
	.byte	0x0b, 0x00, 0x09, 0x0b, 0x00, 0xf6	/* 38 */
	.word	0x046e
	.byte	0x0b, 0x00, 0x0a, 0x0b, 0x00, 0xf6	/* 39 */
	.word	0x0484
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 3a */
	.word	0x049a
	.byte	0x0b, 0x00, 0x05, 0x0f, 0x02, 0xf5	/* 3b */
	.word	0x04b0
	.byte	0x0b, 0x00, 0x09, 0x10, 0x00, 0xf5	/* 3c */
	.word	0x04ce
	.byte	0x0b, 0x00, 0x05, 0x0f, 0x01, 0xf5	/* 3d */
	.word	0x04ee
	.byte	0x0b, 0x00, 0x07, 0x04, 0x01, 0xf5	/* 3e */
	.word	0x050c
	.byte	0x0b, 0x00, 0x09, 0x02, 0x00, 0x03	/* 3f */
	.word	0x0514
	.byte	0x0b, 0x00, 0x03, 0x04, 0x02, 0xf6	/* 40 */
	.word	0x0518
	.byte	0x0b, 0x00, 0x08, 0x07, 0x00, 0xfa	/* 41 */
	.word	0x0520
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 42 */
	.word	0x052e
	.byte	0x0b, 0x00, 0x08, 0x07, 0x00, 0xfa	/* 43 */
	.word	0x0544
	.byte	0x0b, 0x00, 0x08, 0x0b, 0x00, 0xf6	/* 44 */
	.word	0x0552
	.byte	0x0b, 0x00, 0x08, 0x07, 0x00, 0xfa	/* 45 */
	.word	0x0568
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x01, 0xf6	/* 46 */
	.word	0x0576
	.byte	0x0b, 0x00, 0x07, 0x0b, 0x00, 0xf9	/* 47 */
	.word	0x058c
	.byte	0x0b, 0x00, 0x09, 0x0b, 0x00, 0xf6	/* 48 */
	.word	0x05a2
	.byte	0x0b, 0x00, 0x06, 0x0a, 0x02, 0xf7	/* 49 */
	.word	0x05b8
	.byte	0x0b, 0x00, 0x07, 0x0e, 0x01, 0xf6	/* 4a */
	.word	0x05cc
	.byte	0x0b, 0x00, 0x09, 0x0b, 0x00, 0xf6	/* 4b */
	.word	0x05e8
	.byte	0x0b, 0x00, 0x06, 0x0b, 0x01, 0xf6	/* 4c */
	.word	0x05fe
	.byte	0x0b, 0x00, 0x0a, 0x07, 0x00, 0xfa	/* 4d */
	.word	0x0614
	.byte	0x0b, 0x00, 0x08, 0x07, 0x01, 0xfa	/* 4e */
	.word	0x0622
	.byte	0x0b, 0x00, 0x08, 0x07, 0x00, 0xfa	/* 4f */
	.word	0x0630
	.byte	0x0b, 0x00, 0x08, 0x0a, 0x00, 0xfa	/* 50 */
	.word	0x063e
	.byte	0x0b, 0x00, 0x08, 0x0a, 0x00, 0xfa	/* 51 */
	.word	0x0652
	.byte	0x0b, 0x00, 0x08, 0x07, 0x00, 0xfa	/* 52 */
	.word	0x0666
	.byte	0x0b, 0x00, 0x08, 0x07, 0x00, 0xfa	/* 53 */
	.word	0x0674
	.byte	0x0b, 0x00, 0x06, 0x09, 0x01, 0xf8	/* 54 */
	.word	0x0682
	.byte	0x0b, 0x00, 0x09, 0x07, 0x00, 0xfa	/* 55 */
	.word	0x0694
	.byte	0x0b, 0x00, 0x08, 0x07, 0x00, 0xfa	/* 56 */
	.word	0x06a2
	.byte	0x0b, 0x00, 0x0a, 0x07, 0x00, 0xfa	/* 57 */
	.word	0x06b0
	.byte	0x0b, 0x00, 0x08, 0x07, 0x01, 0xfa	/* 58 */
	.word	0x06be
	.byte	0x0b, 0x00, 0x08, 0x0a, 0x01, 0xfa	/* 59 */
	.word	0x06cc
	.byte	0x0b, 0x00, 0x08, 0x07, 0x00, 0xfa	/* 5a */
	.word	0x06e0
	.byte	0x0b, 0x00, 0x07, 0x0f, 0x01, 0xf5	/* 5b */
	.word	0x06ee
	.byte	0x0b, 0x00, 0x02, 0x10, 0x03, 0xf5	/* 5c */
	.word	0x070c
	.byte	0x0b, 0x00, 0x07, 0x0f, 0x01, 0xf5	/* 5d */
	.word	0x072c
	.byte	0x0b, 0x00, 0x0a, 0x03, 0x00, 0xf8	/* 5e */
	.word	0x074a
	.byte	0x0b, 0x00, 0x09, 0x07, 0x00, 0xfa	/* 5f */
	.word	0x0750
/* Plane height, baseline */
	.byte	0x10, 0xf5
/* Raster size in bytes | Pointer to Raster */
	.globl	_charastr
	.long	_charastr
_charastr:
/* Character rasters */
	.word	0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0x0000 /*  0x0000 */
	.word	0x0600, 0xc000, 0xc000, 0xcc00, 0xcc00, 0xcc00, 0xcc00, 0xcc00
	.word	0x3300, 0x3300, 0x3300, 0xffc0, 0xffc0, 0x3300, 0x3300, 0x3300 /*  0x0010 */
	.word	0xffc0, 0xffc0, 0x3300, 0x3300, 0x3300, 0x1000, 0x3c00, 0x5200
	.word	0x9000, 0x9000, 0xd000, 0x7c00, 0x1600, 0x1200, 0x1200, 0x9400 /* 0x0020 */
	.word	0x7800, 0x1000, 0x3060, 0x64c0, 0xc580, 0xcb00, 0x6600, 0x0600
	.word	0x0cc0, 0x1b20, 0x3620, 0x3640, 0x6300, 0x1c00, 0x3200, 0x3200 /* 0x0030 */
	.word	0x3200, 0x1c00, 0x1800, 0x3580, 0x6300, 0xc200, 0xc700, 0x7980
	.word	0xe000, 0xe000, 0x6000, 0xc000, 0x3000, 0x6000, 0x6000, 0xc000 /* 0x0040 */
	.word	0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000
	.word	0x6000, 0x6000, 0x3000, 0xc000, 0x6000, 0x6000, 0x3000, 0x3000 /* 0x0050 */
	.word	0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x6000
	.word	0x6000, 0xc000, 0x0c00, 0xccc0, 0x6d80, 0x1e00, 0x6d80, 0xccc0 /* 0x0060 */
	.word	0x0c00, 0x0c00, 0x0c00, 0x0c00, 0x0c00, 0xffc0, 0xffc0, 0x0c00
	.word	0x0c00, 0x0c00, 0x0c00, 0xe000, 0xe000, 0x6000, 0xc000, 0xff80 /* 0x0070 */
	.word	0xff80, 0xe000, 0xe000, 0xe000, 0x0180, 0x0180, 0x0300, 0x0300
	.word	0x0600, 0x0600, 0x0c00, 0x0c00, 0x1800, 0x1800, 0x3000, 0x3000 /* 0x0080 */
	.word	0x6000, 0x6000, 0xc000, 0xc000, 0x3800, 0x6c00, 0xc600, 0xce00
	.word	0xd600, 0xd600, 0xd600, 0xe600, 0xc600, 0x6c00, 0x3800, 0x1800 /* 0x0090 */
	.word	0x7800, 0xf800, 0x1800, 0x1800, 0x1800, 0x1800, 0x1800, 0x1800
	.word	0x1800, 0xff00, 0x7800, 0xcc00, 0x0600, 0x0600, 0x0600, 0x0c00 /* 0x00a0 */
	.word	0x3800, 0x6000, 0xc000, 0xc200, 0xfe00, 0x7800, 0xcc00, 0x0600
	.word	0x0c00, 0x3800, 0x1800, 0x0c00, 0x0600, 0xc600, 0xcc00, 0x7800 /* 0x00b0 */
	.word	0x1c00, 0x3c00, 0x3c00, 0x6c00, 0x6c00, 0xcc00, 0xcc00, 0xfe00
	.word	0x0c00, 0x0c00, 0x1e00, 0xfe00, 0xc000, 0xc000, 0xd800, 0xec00 /* 0x00c0 */
	.word	0x8600, 0x0600, 0x0600, 0x8600, 0xcc00, 0x7800, 0x3c00, 0x6600
	.word	0xc000, 0xc000, 0xf800, 0xec00, 0xc600, 0xc600, 0xc600, 0x6c00 /* 0x00d0 */
	.word	0x3800, 0xfe00, 0xc600, 0x0600, 0x0c00, 0x0c00, 0x1800, 0x1800
	.word	0x3000, 0x3000, 0x3000, 0x3000, 0x3c00, 0x6600, 0x6600, 0x6600 /* 0x00e0 */
	.word	0x3c00, 0x6600, 0xc300, 0xc300, 0xc300, 0x6600, 0x3c00, 0x3800
	.word	0x6c00, 0xc600, 0xc600, 0xce00, 0x7600, 0x0600, 0x0600, 0x0600 /* 0x00f0 */
	.word	0xcc00, 0x7800, 0xe000, 0xe000, 0xe000, 0x1000, 0x0200, 0xe000
	.word	0xe000, 0xe000, 0xe000, 0xe000, 0xe000, 0x0000, 0x0000, 0xe000 /* 0x0100 */
	.word	0xe000, 0x6000, 0xc000, 0x0180, 0x0600, 0x1800, 0x6000, 0xc000
	.word	0x6000, 0x1800, 0x0600, 0x0180, 0xff00, 0xff00, 0x0000, 0x0000 /* 0x0110 */
	.word	0xff00, 0xff00, 0xc000, 0x3000, 0x0c00, 0x0300, 0x00c0, 0x0300
	.word	0x0c00, 0x3000, 0xc000, 0x3c00, 0x6600, 0xc300, 0x0300, 0x0600 /* 0x0120 */
	.word	0x0c00, 0x1800, 0x1800, 0x0000, 0x1800, 0x1800, 0x1e00, 0x6080
	.word	0x8c40, 0x9240, 0x9240, 0x9240, 0x8e40, 0x8380, 0x6000, 0x1f00 /* 0x0130 */
	.word	0x0c00, 0x0c00, 0x1e00, 0x1e00, 0x3300, 0x3300, 0x6180, 0x7f80
	.word	0x6180, 0x6180, 0xf3c0, 0xfc00, 0x6600, 0x6300, 0x6300, 0x6600 /* 0x0140 */
	.word	0x7c00, 0x6600, 0x6300, 0x6300, 0x6600, 0xfc00, 0x1c00, 0x7300
	.word	0x6300, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0x6000, 0x7300 /* 0x0150 */
	.word	0x1c00, 0xf800, 0x6c00, 0x6600, 0x6300, 0x6300, 0x6300, 0x6300
	.word	0x6300, 0x6600, 0x6c00, 0xf800, 0xff00, 0x6100, 0x6000, 0x6000 /* 0x0160 */
	.word	0x6400, 0x7c00, 0x6400, 0x6000, 0x6000, 0x6100, 0xff00, 0xff00
	.word	0x6100, 0x6000, 0x6000, 0x6200, 0x7e00, 0x6200, 0x6000, 0x6000 /* 0x0170 */
	.word	0x6000, 0xf000, 0x3e00, 0x7700, 0xc180, 0xc000, 0xc000, 0xc000
	.word	0xc000, 0xc780, 0x6300, 0x7700, 0x1d00, 0xf3c0, 0x6180, 0x6180 /* 0x0180 */
	.word	0x6180, 0x6180, 0x7f80, 0x6180, 0x6180, 0x6180, 0x6180, 0xf3c0
	.word	0xfc00, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000 /* 0x0190 */
	.word	0x3000, 0x3000, 0xfc00, 0x1f80, 0x0600, 0x0600, 0x0600, 0x0600
	.word	0x0600, 0x0600, 0xe600, 0xc600, 0x6c00, 0x3800, 0xe180, 0x6100 /* 0x01a0 */
	.word	0x6200, 0x6400, 0x7800, 0x7800, 0x6c00, 0x6600, 0x6300, 0x6180
	.word	0xe1c0, 0xf000, 0x6000, 0x6000, 0x6000, 0x6000, 0x6000, 0x6000 /* 0x01b0 */
	.word	0x6000, 0x6000, 0x6300, 0xff00, 0xc180, 0xc180, 0xe380, 0xe380
	.word	0xb580, 0xb580, 0x9980, 0x9980, 0x8180, 0x8180, 0xc380, 0xc700 /* 0x01c0 */
	.word	0xe100, 0xe100, 0xf100, 0xd900, 0xd900, 0xcd00, 0xcd00, 0xc700
	.word	0xc700, 0xc300, 0x1c00, 0x3600, 0x6300, 0xc180, 0xc180, 0xc180 /* 0x01d0 */
	.word	0xc180, 0xc180, 0x6300, 0x3e00, 0x1c00, 0xfc00, 0x6600, 0x6300
	.word	0x6300, 0x6600, 0x7c00, 0x6000, 0x6000, 0x6000, 0x6000, 0xf000 /* 0x01e0 */
	.word	0x1c00, 0x3600, 0x6300, 0xc180, 0xc180, 0xc180, 0xc180, 0xd180
	.word	0x6b00, 0x3600, 0x1c00, 0x0480, 0x0300, 0xfe00, 0x6300, 0x6180 /* 0x01f0 */
	.word	0x6180, 0x6300, 0x7e00, 0x6c00, 0x6600, 0x6300, 0x6180, 0xf1c0
	.word	0x3d00, 0x6300, 0xc100, 0xc000, 0x7000, 0x1c00, 0x0600, 0x0300 /* 0x0200 */
	.word	0x8300, 0xc600, 0xbc00, 0xffc0, 0x8c40, 0x0c00, 0x0c00, 0x0c00
	.word	0x0c00, 0x0c00, 0x0c00, 0x0c00, 0x0c00, 0x1e00, 0xf1c0, 0x6080 /* 0x0210 */
	.word	0x6080, 0x6080, 0x6080, 0x6080, 0x6080, 0x6080, 0x6080, 0x7180
	.word	0x3f00, 0xf1c0, 0x6080, 0x6080, 0x6080, 0x3100, 0x3100, 0x3100 /* 0x0220 */
	.word	0x1a00, 0x1a00, 0x0e00, 0x0c00, 0xe0c0, 0xc040, 0xc040, 0xc040
	.word	0xc440, 0xc440, 0xca40, 0xd140, 0xe1c0, 0xc0c0, 0x8040, 0xf380 /* 0x0230 */
	.word	0x6100, 0x6100, 0x3200, 0x3200, 0x0c00, 0x2600, 0x2600, 0x4300
	.word	0x4300, 0xe780, 0xf1c0, 0x6080, 0x6080, 0x3100, 0x3100, 0x1a00 /* 0x0240 */
	.word	0x0c00, 0x0c00, 0x0c00, 0x0c00, 0x1e00, 0xff00, 0x8300, 0x8300
	.word	0x0600, 0x0c00, 0x1800, 0x3000, 0x6000, 0x6000, 0xc100, 0xff00 /* 0x0250 */
	.word	0xf800, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000
	.word	0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xf800, 0xc000 /* 0x0260 */
	.word	0xc000, 0x6000, 0x6000, 0x3000, 0x3000, 0x1800, 0x1800, 0x0c00
	.word	0x0c00, 0x0600, 0x0600, 0x0300, 0x0300, 0x0180, 0x0180, 0xf800 /* 0x0270 */
	.word	0x1800, 0x1800, 0x1800, 0x1800, 0x1800, 0x1800, 0x1800, 0x1800
	.word	0x1800, 0x1800, 0x1800, 0x1800, 0x1800, 0xf800, 0x1000, 0x3800 /* 0x0280 */
	.word	0x6c00, 0xc600, 0xff80, 0xff80, 0x6000, 0xc000, 0xe000, 0xe000
	.word	0x7c00, 0xc600, 0x0e00, 0x7e00, 0xc600, 0xc600, 0x7d00, 0xe000 /* 0x0290 */
	.word	0x6000, 0x6000, 0x6000, 0x6c00, 0x7600, 0x6300, 0x6300, 0x6300
	.word	0x7600, 0x7c00, 0x3e00, 0x6700, 0xc000, 0xc000, 0xc000, 0x6300 /* 0x02a0 */
	.word	0x3e00, 0x0e00, 0x0600, 0x0600, 0x0600, 0x3600, 0x6e00, 0xc600
	.word	0xc600, 0xc600, 0x6e00, 0x3f00, 0x3c00, 0x6600, 0xc300, 0xfe00 /* 0x02b0 */
	.word	0xc000, 0x6600, 0x3c00, 0x3c00, 0x6600, 0x6000, 0x6000, 0xf800
	.word	0x6000, 0x6000, 0x6000, 0x6000, 0x6000, 0xf000, 0x0e00, 0x7800 /* 0x02c0 */
	.word	0xcc00, 0xcc00, 0xcc00, 0x7800, 0xc000, 0x7c00, 0xc600, 0xc600
	.word	0x7c00, 0xe000, 0x6000, 0x6000, 0x6000, 0x6e00, 0x7300, 0x6300 /* 0x02d0 */
	.word	0x6300, 0x6300, 0x6300, 0xe780, 0x3000, 0x3000, 0x0000, 0xf000
	.word	0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0xfc00, 0x0e00, 0x0e00 /* 0x02e0 */
	.word	0x0000, 0x0000, 0x0e00, 0x0600, 0x0600, 0x0600, 0x0600, 0x0600
	.word	0x0600, 0x0600, 0xc600, 0x7c00, 0xe000, 0x6000, 0x6000, 0x6000 /* 0x02f0 */
	.word	0x6380, 0x6600, 0x6c00, 0x7800, 0x6c00, 0x6600, 0xe380, 0xf000
	.word	0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000 /* 0x0300 */
	.word	0x3000, 0xfc00, 0xdb80, 0xccc0, 0xccc0, 0xccc0, 0xccc0, 0xccc0
	.word	0xccc0, 0xdc00, 0x6300, 0x6300, 0x6300, 0x6300, 0x6300, 0xe700 /* 0x0310 */
	.word	0x3c00, 0x6600, 0xc300, 0xc300, 0xc300, 0x6600, 0x3c00, 0xdc00
	.word	0x7600, 0x6300, 0x6300, 0x6300, 0x7600, 0x6c00, 0x6000, 0x6000 /* 0x0320 */
	.word	0xf000, 0x3a00, 0x6e00, 0xc600, 0xc600, 0xc600, 0x6e00, 0x3600
	.word	0x0600, 0x0600, 0x0700, 0xde00, 0x7300, 0x6000, 0x6000, 0x6000 /* 0x0330 */
	.word	0x6000, 0xf000, 0x7e00, 0xc300, 0xe000, 0x3e00, 0x0700, 0xc300
	.word	0x7e00, 0x6000, 0x6000, 0xf000, 0x6000, 0x6000, 0x6000, 0x6000 /* 0x0340 */
	.word	0x6400, 0x3800, 0xe300, 0x6300, 0x6300, 0x6300, 0x6300, 0x6300
	.word	0x3d80, 0xe300, 0x6100, 0x6200, 0x3200, 0x3400, 0x1c00, 0x0800 /* 0x0350 */
	.word	0xccc0, 0xccc0, 0xccc0, 0xd480, 0x7300, 0x6200, 0x4200, 0xe300
	.word	0x6200, 0x3400, 0x1800, 0x2c00, 0x4600, 0xc700, 0xe300, 0x6200 /* 0x0360 */
	.word	0x6200, 0x3400, 0x3400, 0x1c00, 0x1800, 0x1800, 0xd800, 0x7000
	.word	0xff00, 0x8600, 0x0c00, 0x1800, 0x3000, 0x6100, 0xff00, 0x1e00 /* 0x0370 */
	.word	0x7000, 0x6000, 0x6000, 0x6000, 0x6000, 0x3800, 0xe000, 0x3800
	.word	0x6000, 0x6000, 0x6000, 0x6000, 0x7000, 0x1e00, 0xc000, 0xc000 /* 0x0380 */
	.word	0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000
	.word	0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xc000, 0xf000, 0x1c00 /* 0x0390 */
	.word	0x0c00, 0x0c00, 0x0c00, 0x0c00, 0x3800, 0x0e00, 0x3800, 0x0c00
	.word	0x0c00, 0x0c00, 0x0c00, 0x1c00, 0xf000, 0x3840, 0xccc0, 0x8700 /* 0x03a0 */
	.word	0xaa80, 0x5500, 0xaa80, 0x5500, 0xaa80, 0x5500, 0xaa80
