/*
 * Standalone driver for the IS extended RL101 disk controller.
 */

#include "saio.h"
#include "sais68k.h"
#include "../dev/elreg.h"

#define	ELADDR0	  	((struct eldevice *) 0x3FF900) /* 1st addr */
#define	ELADDR1  	((struct eldevice *) 0x3FF910) /* 2nd addr */
#define	EL_CN		2			/* number of controllers */
#define	EL_DR		4			/* drives per controller */
#define	UTOCN(x)	((x) >> 2)
#define	UTODR(x)	((x) & 0x03)
#define	EL_MAXCNT	0x1F000

struct eldevice		*elstd[EL_CN] = { ELADDR0, ELADDR1 };
int elstrat(), forbb;

elopen(io)
register struct iob *io;
{
	register int			cn = UTOCN(io->i_unit); 
	register int 			dr = UTODR(io->i_unit);
	register struct eldevice 	*eladdr;

	if (dr >= EL_DR || cn >= EL_CN) {
		printf("el: bad unit\n");
		return (-1);
	}
	eladdr = elstd[cn];
	if (!probe(&eladdr->elbae,&eladdr->elbae)) {
		printf("el: controller not present\n");
		return (-1);
	}
	eladdr->elba.getstat = 0;
	eladdr->elcs = (dr << 8) | EL_GETSTAT; /* get status */
	if (elwait(eladdr))
		return (-1);
	/* dont mistake a rl for an el; if # heads not given then rl */
	if (eladdr->elba.getstat == 0) {
		printf("el: controller not present\n");
		return (-1);
	}
	io->i_st.nbsec = 512;
	io->i_st.nsect = eladdr->elmp.getstat;
	io->i_st.ntpc = eladdr->elba.getstat;
	io->i_st.nspc = io->i_st.nsect * io->i_st.ntpc;
	io->i_st.ncyl = eladdr->elda.getstat / io->i_st.ntpc;
	io->i_st.off = NULL;
	if ((io->i_boff = diskpart(io)) < 0) {
		printf("el: bad offset\n");
		return (-1);
	}
	return (0);
}

elstrategy(io, func)
register struct iob *io;
{
	register int			bleft, 
					bpart;
	register int 			bn = io->i_bn;


	switch (func) {
	  case READ:
		func = EL_READ; 
		break;
	  case WRITE:
		func = EL_WRITE; 
		break;
	  default:
		printf("el: bad func 0x%x\n", func);
		return -1;
	}
	bleft = io->i_cc;	/* byte cnt */
	do {
		bpart = (bleft < EL_MAXCNT) ? bleft : EL_MAXCNT;
		if (forbb) {
			if (elstrat(io,bn,bpart,func) == -1)
				return -1;
			io->i_ma += bpart;
		} else if (badstrat(io,bn,bpart,func,elstrat,EL_READ) == -1)
			return -1;
		bleft -= bpart;
		bn += bpart / io->i_st.nbsec;
	} while (bleft > 0);
	return io->i_cc;
}

elstrat(io,bn,bpart,func)
register struct iob *io;
register int bn; 
int bpart, func;
{
	register int 			cn = UTOCN(io->i_unit);
	register int 			dr = UTODR(io->i_unit);
	register struct eldevice 	*eladdr = elstd[cn];
	static int 			errcnt = 0;

	while (1) {
		eladdr->elda.rw = bn / io->i_st.nsect;
		eladdr->elba.rw = loword(io->i_ma);
		eladdr->elbae = (short)(hiword(io->i_ma)&0x3f) |
				((bn % io->i_st.nsect) << 6);
		eladdr->elmp.rw = -((bpart + sizeof(short) - 1) >> 1);
		eladdr->elcs = dr<<8 | func;
		if (elwait(eladdr))
			return (-1);
		if (eladdr->elcs & EL_ERR) {
			printf("el: bn=%x cs=%b\n",bn,eladdr->elcs,ELCS_BITS);
			if (++errcnt < 5)
				continue;
			errcnt = 0;
			return -1;
		}
		if (eladdr->elcs & (EL_SOFT | EL_ECC))
			printf("el: recovered bn=%x cs=%b\n", bn,
				eladdr->elcs,ELCS_BITS);
		return 0;
	}
}

elwait(eladdr)
struct eldevice *eladdr;
{ 	
	long l = 0;

	while ((eladdr->elcs & EL_CRDY) == 0)
		if (++l == 800000) {
			printf("el: timeout\n");
			return (-1);
		}
	return (0);
}

#ifdef	DISK_FORMAT
struct elfmt_param {
	u_short	n_headcyl;
};

struct elfmt_defects {
	u_short	track_spared;
	u_short	offset;
};

static char buf[100];
	
elformat()
{
	register struct eldevice *eladdr;
	register struct elfmt_param *p;
	register struct elfmt_defects *d;
	register int i, n;
	int nheads, ncyl, ntrk, fc, exm;
	char test;
	u_short drive;
	u_short *c;

	printf("\nFormatting EL disk\n");
loop:	printf("\nFunctions available:\n");
	printf("  a) Format Disk\n");
	printf("  b) Display Defect List\n");
	printf("Enter letter of desired function ('H' for help): ");
	stripwhite(gets(buf));
	test = buf[0];
	if (test == 'H')
		goto loop;
	if (test != 'a' && test != 'A' && test != 'b' && test != 'B')
		goto loop;
cntlr:  printf("Controller number (0-%d) ? ",EL_CN-1);
	stripwhite(gets(buf));
	i = getnum(buf);
	if (i < 0 || i >= EL_CN) {
		printf("illegal controller number\n");
		goto cntlr;
	}
	eladdr = elstd[i];
	if (!probe(&eladdr->elbae,&eladdr->elbae)) {
		printf("controller %d at 0x%x not present\n", i, eladdr);
		goto cntlr;
	}
drv:    printf("Drive number (0-%d) ? ",EL_DR-1);
	stripwhite(gets(buf));
	drive = getnum(buf);
	if (drive < 0 || drive >= EL_DR) {
		printf("illegal drive number\n");
		goto drv;
	}
	drive = ((getnum(buf)<<8) & 0x0300) | EL_FORMAT;

exmn:	printf("Firmware Revision, EXM2.x or EXM3.x (3) ? ");
	stripwhite(gets(buf));
	i = getnum(buf);
	if (i < 2 || i > 3) {
		printf("illegal firmware rev\n");
		goto exmn;
	}
	exm = i;

	if (test == 'a' || test == 'A') {
head:		printf("Number of heads ? ");
		stripwhite(gets(buf));
		nheads = getnum(buf);
		if (((exm == 2) && (nheads > 8 || nheads < 2)) ||
		    ((exm == 3) && (nheads > 16 || nheads < 2))  )
			goto head;
cyl:		printf("Number of cylinders ? ");
		stripwhite(gets(buf));
		ncyl = getnum(buf);
		if (((exm == 2) && (ncyl > 1024 || ncyl < 20)) ||
		    ((exm == 3) && (ncyl > 2048 || ncyl < 20))  )
			goto cyl;
		if (exm == 2)
			fc = (1<<13) | (nheads-1)<<10 | (ncyl-1);
		else
			fc = (nheads-1)<<11 | (ncyl-1);
		printf(" Estimated time until completion is ");
		printf("%d hours and %d min\n",
			((nheads*ncyl)/100)/60,((nheads*ncyl)/100)%60);
		printf("Format constant = %x, ok ? ",fc);
		stripwhite(gets(buf));
		if (buf[0] != 'y' && buf[0] != 'Y')
			goto loop;
		eladdr->elda.rw = fc;
		eladdr->elcs = drive;
		c = (u_short *)0x1000;
		ncyl = *c;
		printf("Cylinder being formatted:\n");
		while (eladdr->elcs == 0x0000) {
			if (*c != ncyl) {
				ncyl = *c;
				printf("\r %d  ",ncyl);
			}
			DELAY(400);
		}
		printf("\n");
		if (eladdr->elcs & EL_ERR) {
			printf("DRIVE DID NOT FORMAT\n");
			printf("el: error cs=%b\n",eladdr->elcs,ELCS_BITS);
		} else
			printf("SUCCESSFUL FORMAT\n");
	} else if (test == 'b' || test == 'B') {
		eladdr->elda.rw = 0x8000;	/* read parameter/defect list */
		eladdr->elcs = drive;
		if (elwait(eladdr) || eladdr->elcs & EL_ERR) {
			printf("el: error cs=%b\n",eladdr->elcs,ELCS_BITS);
			printf("cannot read bad track map\n");
			goto loop;
		}
		p = (struct elfmt_param *)0x1000;
		d = (struct elfmt_defects *)0x1002;

		if (exm == 2) {
		    nheads = ((p->n_headcyl>>13) & 0x7) + 1;
		    ntrk = p->n_headcyl & 0xFFF;
		    printf("%d heads, %d tracks, 16 sec/track\n", nheads, ntrk);
		    printf("%d cylinders with data\n", ntrk/nheads);
		} else {
		    nheads = ((p->n_headcyl>>12) & 0xF) + 1;
		    ncyl = (p->n_headcyl & 0x7FF) + 1;
		    printf("%d heads, %d cylinders, 16 sec/track\n", nheads, ncyl);
		}
		printf("Hit RETURN for list of DEFECTS: ");
		gets(buf);
		printf("DEFECT	#	CYL  	HEAD\n");
		for (i = 0 ; ; i++) {
			if (d->offset == 0xffff)
				break;
			n = d->track_spared + d->offset;
			printf("%x:	%d	%d	%d\n",
				d, i, n/nheads, n%nheads);
			d++;
			DELAY(10000);
		}
	}
	goto loop;
}
#endif	DISK_FORMAT
