#include "tr.h"
#if NTR > 0

#undef USE_SLEEP

/*
 * VME Network Card Ethernet controller interface
 *	TI refers to the TMS380 user's guide
 */

#include "param.h"
#include "systm.h"
#include "mbuf.h"
#include "buf.h"
#include "protosw.h"
#include "ioctl.h"
#include "socket.h"
#include "vmmac.h"
#include "errno.h"
#include "time.h"
#include "kernel.h"
#include "uio.h"
#include "../net/if.h"
#include "../net/netisr.h"
#include "../net/route.h"
#include "../netinet/in.h"
#include "../netinet/in_systm.h"
#include "../netinet/in_var.h"
#include "../netinet/ip.h"
#include "../netinet/ip_var.h"
#include "../netinet/if_ether.h"
#include "../netinet/if_token.h"
#include "../is68kif/if_pkbuf.h"
#include "../is68kdev/qbvar.h"
#include "../is68k/reg.h"
#include "../is68k/board.h"
#include "../is68kif/if_tr.h"

#ifdef TRFS
#include "../wipc/wipc.h"
#include "../wipc/wipc_packet.h"
#include "../wipc/wipc_pklink.h"
#include "../wipc/wipc_minfo.h"
extern  struct iface iface[];
#endif TRFS

#ifdef	ENETFILTER
#include "../is68kif/enet.h"
#endif	ENETFILTER

extern  struct ifnet loif;

int	trdebug = 0;
int	trtxcount = 0;
int	trtxtimo();

/*
 * TR controller addresses  see /sys/conf/devaddr.c
 */
extern u_short *TRstd[];

int     trprobe(), trattach(), trintr(), trslave();
int     trinit(),trioctl(),troutput(),trreset();

struct  qb_device *trinfo[ NTR ];
struct  qb_driver TRdriver = {
	trprobe, trslave, trattach, 0, TRstd, "tr", trinfo, "TR"
};

struct  mbuf *if_rqbget();
struct  pkbuf *trget(), *GetPkBuf();
int	trtimeout();

/*
 * Token Rign software status per interface.
 *
 * Each interface is referenced by a network interface structure,
 * tr_if, which the routing code uses to locate the interface.
 * This structure contains the output queue for the interface, its address, ...
 */
struct  tr_softc {
	struct  trarpcom  tr_ac;           /* common ethernet structures */
#define tr_if           tr_ac.ac_if     /* network-visible interface */
#define tr_traddr       tr_ac.ac_traddr /* hardware token ring address */
#define tr_ipaddr       tr_ac.ac_ipaddr /* hardware token ring address */
	short		tr_enetunit;	/* unit number for enet filtering */
	short		tr_enetinit;	/* enet inetrface is initialized */
	u_short		tr_naddrptr;	/* pointer to node address in TMS */
	u_short		tr_flags;	/* Flags word */
	u_char		tr_cvec;	/* Assigned cmd interrupt vector */
	u_char		tr_ix;
	struct trcomm	*tr_vdma;		/* VDMA Communication area */
	struct trcomm	tr_comm;		/* Communication area */
} tr_softc[NTR];

#define	TRINIT	0x0001			/* Initialization flag */
#define	TRTIMO	0x0002			/* Timeout flag */
#define	TRCDONE	0x0004			/* Command done flag */
#define	TROPEN	0x0008			/* Adapter open flag */
#define	TRTXCMD	0x0010			/* Transmit command in progress */
#define	TRTXBSY	0x0020			/* Actual transmission in progress */

/*
**  Read Adapter Buffer struct.
*/
struct adb {
	u_short	adb_count;
	u_short	adb_addr;
	u_short	adb_data[10];
} adb, *adb_vdma;


/*
**	Probe for device.
*/

trprobe(trdev, unit)
register Trdevice  *trdev;
register int	unit;
{
	extern int	cvec, clev;
	register struct tr_softc  *tr = &tr_softc[unit];

	if (badaddr(trdev, 2) || badaddr(&trdev->csr, 1))
		return 0;

	trdev->csr = RESET | DMAENA | INTENA;
	tr->tr_ix = -1;

	tr->tr_cvec = cvec = freevec();
	clev = clev_impmax;		/* Fudge interrupt level */
	clevmax = clev_impmax;
	clev_imp = MAX(clev, clev_imp);
	trdev->csr = DMAENA | INTENA;
	tr->tr_vdma = (struct trcomm *)(IOPB_STD(
		iopballoc(&tr->tr_comm, sizeof (struct trcomm)),vbnum));
	if (adb_vdma == NULL)
		adb_vdma = (struct adb *)(IOPB_STD(
			iopballoc(&adb, sizeof (struct adb)),vbnum));
	return (sizeof(Trdevice));
}


trslave()
{
	return ( 1 );
}

/*
 * Interface exists: make available by filling in network interface
 * record.  System will initialize the interface when it is ready
 * to accept packets. 
 */
trattach(md)
register struct qb_device *md;
{
	register struct tr_softc *tr = &tr_softc[md->qi_unit];
	register struct ifnet 	*ifp = &tr->tr_if;
	extern char 		hostname[];
	register struct sockaddr_in 	*sin;

	ifp->if_unit = md->qi_unit;
	ifp->if_name = "tr";
	ifp->if_mtu = TOKENMTU;

	ifp->if_init = trinit;
	ifp->if_ioctl = trioctl;
	ifp->if_output = troutput;
	ifp->if_reset = trreset;
	ifp->if_flags = IFF_BROADCAST | IFF_NOTRAILERS;
	if_attach(ifp);

	/* Paranoia */
	bzero (tr->tr_traddr, sizeof(tr->tr_traddr));

#ifdef	TRFS
	{
		int tr_xmit(), tr_relse();

		tr->tr_ix = IfaceAttach(ifp->if_unit, tr_xmit, tr_relse,
					TRPKLEN, TRPKBURST, tokenbroadcastaddr,
					6);
		iface[tr->tr_ix].flags &= ~UP;
	}
#endif	TRFS
}


/*
 * Reset interface
 */
trreset(unit)
register int unit;
{
	register struct tr_softc *tr = &tr_softc[unit];
	register Trdevice *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	
	if (unit >= NTR || trinfo[unit] == 0 || trinfo[unit]->qi_alive == 0)
		return;

	tr->tr_if.if_flags &= ~(IFF_RUNNING|IFF_UP);
#ifdef TRFS
	if (tr->tr_ix != -1)
	    iface[tr->tr_ix].flags &= ~UP;
#endif TRFS

	trdev->csr = RESET;	/* reset SIF, clear DMA & INT enables */
}


/*
** Read adapter buffer (see TI)
*/

trreadadbuf(tr, addr, count)
register struct tr_softc *tr;
u_short	addr;
short	count;
{
	register Trdevice   *trdev = 
			(Trdevice *)trinfo[tr->tr_if.if_unit]->qi_mi->qm_addr;
	register struct scb *scb = &tr->tr_comm.scb;

	adb.adb_count = count;
	adb.adb_addr = addr;

	if (!trscbclear(tr->tr_if.if_unit, 0))
	{
	    trerr(tr->tr_if.if_unit, "trreadadbuf - SCB not cleared, to = %d\n", 10);
	    return(EHOSTDOWN);
	}

	scb->scb_cmd = CMD_RAB;
	trloadaddr(&scb->scb_addrh, &scb->scb_addrl, adb_vdma, SIF_MMR0);
	tr->tr_flags &= ~TRCDONE;
	trdev->sif_intr = SIFI_INTA | SIFI_RSI | SIFI_EXE;
}

/*
 * Initialize interface, clear pending operations.
 */
trinit(unit)
int unit;
{
	register struct tr_softc	*tr = &tr_softc[unit];
	register struct ifnet		*ifp = &tr->tr_if;
	register int 			ret, count = 0;
	Trdevice 	      *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	u_short 			naddr, swl;

	if (ifp->if_flags & IFF_RUNNING)
		return;

again:
	/* Did bring-up diags pass? */

	if ((ret = trinitialized(unit)) < 0)
	{
	    printf("trinitialized failed...");
	    if (count++ < 5)
	    {
	        printf("trying again.\n");
		reset_tr(unit);
		goto again;
	    }
	    else
	    {
	        printf("returning.\n");
		return 1;
	    }
	}
	else if (ret == 0)
	{
	    printf("trinitialized returned 0.\n");
	    return 1;
	}

	/* Allocate and initialize communications area */
	if ((tr->tr_flags & TRINIT) == 0)
	{
	    trinitbl(tr->tr_comm.tblhead, tr->tr_vdma->tblhead,
			 TXBUFS, SIF_MMR0, 0);
	    trinitbl(tr->tr_comm.rblhead, tr->tr_vdma->rblhead,
			 RXBUFS, SIF_MMR0, 1);
	    tr->tr_comm.tbl = &tr->tr_comm.tblhead[0];
	    tr->tr_comm.rbl = &tr->tr_comm.rblhead[0];
	    tr->tr_flags = TRINIT;
	}
	else
	{
	    trcleanbl(tr->tr_comm.rblhead, RXBUFS, 1);
	    trcleanbl(tr->tr_comm.tblhead, TXBUFS, 0);
	    tr->tr_comm.tbl = &tr->tr_comm.tblhead[0];
	    tr->tr_comm.rbl = &tr->tr_comm.rblhead[0];
	}

	/* Initialize SIF interface */
	if ((ret = trinitbrd(unit)) < 0)
	{
	    if (count++ < 5)
	    {
		reset_tr(unit);
		goto again;
	    }
	    else
		return 1;
	}
	else if (ret == 0)
		return 1;

	if (trdebug & TRDBG_INIT)
	{
	    u_char bia[2];

	    trdev->sif_addr = RAB_BIA;	
	    trdev->sif_addr = trdev->sif_data;
	    *(u_short *)bia = trdev->sif_data_inc;
	    printf("Token BIA: (%x.%x.", bia[0], bia[1]);
	    *(u_short *)bia = trdev->sif_data_inc;
	    printf("%x.%x.", bia[0], bia[1]);
	    *(u_short *)bia = trdev->sif_data_inc;
	    printf("%x.%x)\n", bia[0], bia[1]);
	}

	/* Store node address pointer for later */

	trdev->sif_addr = RAB_ADDRS;	
	tr->tr_naddrptr = trdev->sif_data;

	trdev->csr = (LED | DMAENA | INTENA);	/* Chips are initialized */
	ifp->if_flags |= IFF_RUNNING;		/* and running */
	tr->tr_flags &= ~(TROPEN|TRTXBSY|TRTXCMD); /* Can't be open or tx'ing */
	untimeout(trtxtimo, unit);

	return 0;
}


/*
 * Token Ring interface interrupt.
 */
trintr(unit)
register int unit;
{
	register struct tr_softc	*tr = &tr_softc[unit];
	register Trdevice   *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	register struct scb 		*scb = &tr->tr_comm.scb;
	register struct ssb 		*ssb = 
		(struct ssb *)(CACHE_INHIBIT(&tr->tr_comm.ssb));
	struct ifnet 			*ifp = &tr->tr_if;
	register int 			i;

	if ((trdev->sif_intr & SIFI_INTS) == 0)
	{
	   printf("tr%d: trintr stray intr (sif_intr 0x%x)\n",trdev->sif_intr);
	   return;
	}

	if (trdebug & TRDBG_INTR)
	    printf("tr%d: intr (0x%x) received\n", unit, trdev->sif_intr);

	switch	(trdev->sif_intr & SIFI_ICODE)
	{
	    case ICODE_CHECK :
		printf ("tr%d: Adapter check, status is:\n", unit);
		trdev->sif_addr = 0x05e0;	/* "Check" status address */
		for (i=0; i<4; i++)
			printf("    word %d == 0x%x\n", i, trdev->sif_data_inc);
		break;

	    case ICODE_RNGSTAT :
		if (ssb->ssb_cmd != SSB_RING)
		   printf("tr%d: Interrupt and command dont jive\n", unit);
		if (trdebug & TRDBG_RS)
		{
		    printf("tr%d: ring status 0x%x\n", unit, ssb->ssb_stat0);
		    if (ssb->ssb_stat0 & RS_SOFT)
			trreaderrlog(unit, 1);
		}
			/* check if we've de-inserted */
		if(ssb->ssb_stat0 & (RS_LOBE | RS_AREM | RS_REMR))
		    reset_tr(unit);
		break;

	   case ICODE_SCBCLR :
		scb->scb_cmd = 0;		/* Indicate free SCB, redun. */
		if((tr->tr_flags & TRTXCMD) == 0)
		    trstart(unit);	/* start the xmitter if we need to */
		break;

	   case ICODE_CMDSTAT :
		switch (ssb->ssb_cmd)
		{ 
		case CMD_RAB:
	            bcopy((caddr_t)&adb, tr->tr_traddr, sizeof(tr->tr_traddr));
		    ifp->if_flags |= IFF_UP;
#ifdef TRFS
		    if (tr->tr_ix != -1)
		        iface[tr->tr_ix].flags |= UP;
#endif TRFS
		    if (trdebug & TRDBG_INIT)
		    {
			printf("Token addr: (%x.%x.%x.%x.%x.%x)\n",
			    tr->tr_traddr[0], 
			    tr->tr_traddr[1],
			    tr->tr_traddr[2],
			    tr->tr_traddr[3],
			    tr->tr_traddr[4],
			    tr->tr_traddr[5]);
		    }
#ifdef	ENETFILTER
		    if (!tr->tr_enetinit) {
			struct endevp enp;
		    
			tr->tr_enetinit++;
			enp.end_dev_type = ENDT_TOKR;
			enp.end_addr_len = sizeof(tr->tr_traddr);
			enp.end_hdr_len = sizeof(struct token_header);
			enp.end_MTU = TOKENMTU;
			bcopy(tr->tr_traddr, enp.end_addr, 
						sizeof(tr->tr_traddr));
			bcopy(tokenbroadcastaddr, enp.end_broadaddr, 
						sizeof(tokenbroadcastaddr));
			tr->tr_enetunit = enetattach(ifp, &enp);
		    }
#endif	ENETFILTER
		    trrecv(unit);	/* issue recv and when the SCB is */
		    break;		/*   cleared we'll issue the xmit */

		case CMD_OPEN:
		    if (ssb->ssb_stat0 != OPEN_OK)
		    {
			trerr(unit, "open failed (stat 0x%x)\n",ssb->ssb_stat0);
		        break;
		    }

		    /* get node address from chip */
	            trreadadbuf(tr, tr->tr_naddrptr, sizeof(tr->tr_traddr));
		    tr->tr_flags |= TROPEN;
		    break;

		case SSB_RJCT:
		    printf("tr%d: command 0x%x rejected [0x%x]\n", unit, 
					ssb->ssb_stat1, ssb->ssb_stat0);
		    if (ssb->ssb_stat1 == CMD_XMIT && 
			(ssb->ssb_stat0 & RJ_SAME) == 0)
		    {
			tr->tr_flags &= ~(TRTXBSY|TRTXCMD);
			untimeout(trtxtimo, unit);
			trcleanbl(tr->tr_comm.tblhead, TXBUFS, 0);
			tr->tr_comm.tbl = &tr->tr_comm.tblhead[0];
		    }
		    if (ssb->ssb_stat1 == CMD_XMIT)
			trtxcount--;
		    break;

		case CMD_REL:
		    if (trdebug & TRDBG_RS)
			trreaderrlog(unit, 0);
		    break;

		case CMD_SFA:
		    /* not sure  what to do here when this is supported */
		    printf("tr%d: Set Functional Address\n",unit);
		    break;
		}
		tr->tr_flags |= TRCDONE;	/* Indicate current cmd done */
		if ((ssb->ssb_stat0 & CMD_CMPLT) == 0)
		    printf("tr%d: command [0x%x] failed\n", unit, ssb->ssb_cmd);
		break;

	   case ICODE_XMTSTAT :
		if ((trdebug & (TRDBG_TX|TRDBG_INTR)) == (TRDBG_TX|TRDBG_INTR))
		    printf("tr%d: tx intr (xmit 0x%x)\n", unit, ssb->ssb_stat0);
		if (ssb->ssb_stat0 & (CMD_CMPLT|XMIT_LERR))
		{
		    printf("Xmit complete/error (ssb 0x%x,0x%x)\n",
				ssb->ssb_stat0, *(u_long *)&ssb->ssb_stat1);
		    tr->tr_if.if_oerrors++;
		    tr->tr_flags &= ~TRTXCMD;
		}
		else
		    tr->tr_if.if_opackets++;
		tr->tr_flags &= ~TRTXBSY;
		untimeout(trtxtimo, unit);
		trtxcount--;
	        trfreebuf(tr->tr_comm.tblhead);
	        trstart(unit);
		break;

	   case ICODE_RCVSTAT :
		if ((trdebug & (TRDBG_RX|TRDBG_INTR)) == (TRDBG_RX|TRDBG_INTR))
		    printf("tr%d: rx intr (recv 0x%x)\n", unit, ssb->ssb_stat0);
		if (ssb->ssb_stat0 & RECV_SUSPND)	/* Suspended? */
		{
		    printf("tr%d: RECV suspnd (addr 0x%x rblhead 0x%x)i\n",unit,
				*(long *)&ssb->ssb_stat1, tr->tr_comm.rblhead);
		}
		trread(unit, ssb);
		break;

	   default :
		printf("tr%d: unknown interrupt 0x%x\n", unit, 
					trdev->sif_intr&SIFI_ICODE);
		break;
	}
	trdev->sif_intr = SIFI_INTA | SIFI_SSBCL;
}


/*
 * Receiver interrupt handling routine.
 */
trread(unit, ssb)
int	unit;
struct ssb *ssb;
{
	struct tr_softc			*tr = &tr_softc[unit];
	struct ifqueue			*inq;
	struct mbuf			*m;
	register struct token_header	*th;
	register struct llc_header	*llc;
	register struct pkbuf		*pkb;
	register struct snap		*snap;
	Trdevice   *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	struct ifnet			*ifp = &tr->tr_if;
	struct mbuf			*mtop;
	register int			headersize;
	unsigned short			s, ttype;
	int				len, off;
	u_short				rcntl;

/*
 * Get packet from interface
 */
	while (pkb = trget(unit, tr, &len)) {
		if ((ifp->if_flags & IFF_UP) == 0) {
			if (trdebug & TRDBG_RX)
			    printf("tr interface down, pkt dropped\n");
			FreePkBuf(pkb);
		        tr->tr_if.if_ierrors++;
			continue;
		}

		if (len < TOKENMIN) {
			printf("tr short pkt (0x%x) dropped\n", len);
			FreePkBuf(pkb);
		        tr->tr_if.if_ierrors++;
			continue;
		}

		if (trdebug & TRDBG_RX)
			printf("tr rx %x\n", len);

		headersize = sizeof(struct token_header);
		th = (struct token_header *)pkb->pkb_buffer;
#ifdef notdef
		if (trdebug & TRDBG_RX)
		{
		    register int i;

		    printf("tr rx dhost: ");
		    for (i=0; i<6; i++)
			printf ("0x%x.", th->token_dhost[i]);
		    printf("\n      shost: ");
		    for (i=0; i<6; i++)
			printf ("0x%x.", th->token_shost[i]);
		    printf("\n");
		}
#endif

/*
** check various bits and bytes to get headers
** correct. There could be a llc header, which
** could be followed by a snap header.
*/
		if (th->token_shost[0] & 0x80)	/* routing info? */
		{
		    rcntl = *(u_short *)(th+1);	/* get control field */
		    off = (rcntl & RC_LENGTH) >> RC_LEN_SFT;
		    if (off < 2 || off > 18)	/* junk */
		    {
			printf("tr junk route (0x%x) - dropped\n", off);
			FreePkBuf(pkb);
			tr->tr_if.if_ierrors++;
			continue;
		    }
		}
		else
		    off = 0;

		headersize += off;

		if ((th->token_frame & FC_TYPE)==FC_STD)
		{ 
		    /* skip route */
		    llc = (struct llc_header *)((int)(th+1)+off);
		    headersize += LLC_SIZE;
#ifdef notdef
		    if (trdebug & TRDBG_RX)
			printf("LLC header detected dsap 0x%x ssap 0x%x cntl 0x%x\n",
			llc->llc_dsap, llc->llc_ssap, llc->llc_cntl);
#endif

		    if (llc->llc_dsap == SAP_IP)
		    { 
			snap = (struct snap *)((int)llc+LLC_SIZE);
			ttype = ntohs(*(u_short *)&snap->snap_type[0]);
			if (trdebug & TRDBG_RX)
			    printf("SNAP header detected type 0x%x\n", 
							ttype);
		        headersize += SNAP_SIZE;
		    }
		    else
			ttype = llc->llc_dsap;
		}
		else
		    ttype = TOKENTYPE_MAC;

#ifdef	TRFS
		if (ttype == ETHERTYPE_WIPC && tr->tr_ix != -1 &&
					(iface[tr->tr_ix].flags & UP))
		{
		    tr_recv(unit, pkb, len, off);
		    continue;
		}
#endif	TRFS

/*
 * convert to mbufs, removeing headers.
 */
		m = if_rqbget(pkb->pkb_buffer, len-headersize, 0, headersize, 										ifp);
		FreePkBuf(pkb);		/* Don't need this anymore */
		if (m == 0)
		{
		    printf("rqbget had no mbufs\n");
		    continue;
		}

		switch(ttype) {
#ifdef INET
		case ETHERTYPE_IP:
			schednetisr(NETISR_IP);
			inq = &ipintrq;
			break;
#endif INET
		case ETHERTYPE_ARP: /* arp packet */
			trarpinput(&tr->tr_ac, m);
			continue;
		default:
#ifdef ENETFILTER
		    if (tr->tr_enetinit && tr->tr_enetunit >= 0) {
			/*
			 * We need the local net header after all.  Oh well,
			 * this could be improved.
			 */
			MGET(mtop, M_DONTWAIT, MT_DATA);
			if (mtop) 
			{
			    bcopy((struct token_header *)th,
					mtod(mtop, struct token_header *),
					headersize);
			    mtop->m_len = headersize;
			    /* remove interface pointer */
			    m->m_len -= sizeof(ifp);
			    m->m_off += sizeof(ifp);
			    mtop->m_next = m;
			    enetFilter(tr->tr_enetunit, mtop, len);
			    continue;
			}
			else
			{
			    printf("enet need header, but no bufs\n");
			    goto bad;
			}
		    }
#endif ENETFILTER
		    printf("Unknown type (0x%x)\n", ttype);
bad:
		    m_freem(m);
		    continue;
		}

		s = splimp();
		if (IF_QFULL(inq)) {
			IF_DROP(inq);
			splx(s);
			printf("trread: qfull, dropped packet\n");
			m_freem(m);
			continue;
		}
		IF_ENQUEUE(inq, m);
		splx(s);
		tr->tr_if.if_ipackets++;
	}
	trdev->sif_intr = SIFI_INTA | SIFI_RSI | SIFI_RECC | SIFI_RECV;
}

struct pkbuf *
trget(unit, tr, pktlen)
register int	unit;
register struct tr_softc *tr;
register int	*pktlen;
{
	register Trdevice   *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	register struct pkbuf		*pkb;
	register struct trbuflist	*rbl = 
		(struct trbuflist *)(CACHE_INHIBIT(tr->tr_comm.rbl));

	if (trdebug & TRDBG_RX)
	    printf("trget called (rbl 0x%x)\n", rbl);

	if (rbl->bl_cstat & RBL_VALID)		/* Any more frames? */
	    return (struct pkbuf *)0;

	/*
	** Sanity check on frame.
	*/
	if ((rbl->bl_cstat & (RBL_SOF|RBL_EOF|RBL_COMPLETE)) != 
					(RBL_SOF|RBL_EOF|RBL_COMPLETE))
	{
	    printf("trget: frame error (rbl cstat 0x%x)\n", rbl->bl_cstat);
	    return (struct pkbuf *)0;
	}

	if ((trdebug  & TRDBG_DUMP) && (trdebug & TRDBG_RX))
	{
	    printf("Recv packet:\n");
	    dumppkt(rbl->bl_data->pkb_buffer, TRMIN(rbl->bl_flen, 50));
	}

	/* more sanity */
	if ((pkb = rbl->bl_data) == 0)
	{
	    printf("rbl data zero!!\n");
	    return (struct pkbuf *)0;
	}
	trbufalloc(rbl);
	rbl->bl_cstat = RBL_VALID | RBL_INTR;
	tr->tr_comm.rbl = rbl->bl_next;

	trdev->sif_intr = SIFI_INTA | SIFI_RSI | SIFI_RECC | SIFI_RECV;

	*pktlen = rbl->bl_flen;
	return pkb;
}


/*
 * Token ring output routine. (called by user)
 * Encapsulate a packet of type family for the local net.
 * If destination is this address or broadcast, send packet to
 * loop device to kludge around the fact that most interfaces can't
 * talk to themselves.
 */
troutput(ifp, m0, dst)
struct ifnet *ifp;
struct mbuf *m0;
struct sockaddr *dst;
{
	register struct tr_softc	*tr = &tr_softc[ifp->if_unit];
	register struct mbuf 		*m = m0;
	register struct token_header 	*thdr;
	register int			off, i, s, error;
	u_short				type;
	u_char		 		tdst[6];
	struct in_addr 			idst;
	int				headersize=sizeof(struct token_header);
	struct llc_header 		*llc;
	struct snap	 		*snap;

	if (trdebug & TRDBG_TX)
	    printf("troutput called\n");

	if ((tr->tr_if.if_flags & IFF_UP) == 0)
	{
	    error = EHOSTDOWN;
	    goto bad;
	}

	switch( dst->sa_family ) {
#ifdef INET
	  case AF_INET:
		idst = ((struct sockaddr_in *)dst)->sin_addr;
		if (!trarpresolve(&tr->tr_ac, m, &idst, tdst))
		{
		    if (trdebug & TRDBG_TX)
		        printf("tr%d: output failed, no ARP\n", ifp->if_unit);
		    return(0);
		}
		off = ((u_short)mtod(m, struct ip *)->ip_len) - m->m_len;
		type = ETHERTYPE_IP;
		off = 0;
		headersize += LLC_SIZE + SNAP_SIZE;
		goto gottype;
#endif
	  case AF_IMPLINK:
	  case AF_UNSPEC:
		thdr = mtod(m, struct token_header *);
		bcopy(thdr->token_dhost, tdst, sizeof(tdst));
		goto gotheader;

	  default:
		error = EAFNOSUPPORT;
		goto bad;
	}

gottype:
	/*
         * Add local net header.  If no space in first mbuf,
         * allocate another.
         */
	if (m->m_off > MMAXOFF ||
	    MMINOFF + headersize > m->m_off) {
		m = m_get(M_DONTWAIT, MT_HEADER);
		if (m == 0) {
			printf("tr%d: troutput, no mbufs for header\n",
								ifp->if_unit);
			m_freem(m0);
			return(ENOBUFS);
		}
		m->m_next = m0;
		m->m_len = headersize;
	} else {
		m->m_off -= headersize;
		m->m_len += headersize;
	}
	thdr = mtod(m, struct token_header *);
	bcopy(tdst, thdr->token_dhost, sizeof(tdst));
	thdr->token_shost[0] = 0;  /* No routing info field */
	thdr->token_access = 0;
	thdr->token_frame = FC_STD;

	if (type == ETHERTYPE_IP || type == ETHERTYPE_ARP)
	{
	    llc = (struct llc_header *)(thdr + 1);
	    llc->llc_dsap = llc->llc_ssap = SAP_IP;
	    llc->llc_cntl = LLC_U;
	    snap = (struct snap *)((int)llc + LLC_SIZE);
	    snap->snap_spare[0]= snap->snap_spare[1]= snap->snap_spare[2]= 0;
	    *(u_short *)&snap->snap_type[0] = type;
	}

gotheader:
#ifdef notdef
	if (trdebug & TRDBG_TX)
	{
	    printf("tr tx dhost: ");
	    for (i=0; i<6; i++)
		printf ("0x%x.", thdr->token_dhost[i]);
	    if (thdr->token_frame & FC_STD)
	    {
	        llc = (struct llc_header *)(thdr + 1);
	        printf("\n      dsap 0x%x\n", llc->llc_dsap);
		if (llc->llc_dsap == SAP_IP)
		{
	          snap = (struct snap *)((int)llc + LLC_SIZE);
		  printf("      type 0x%x\n",*(u_short *)&snap->snap_type[0]);
		}
	    }
	    else
	        printf("\n");
	}
#endif notdef
	/*
	 * Queue message on interface if possible 
	 */
	s = splimp();
	if (IF_QFULL(&ifp->if_snd)) {
		IF_DROP(&ifp->if_snd);
		printf("tr%d: troutput, queue full\n", ifp->if_unit);
		error = ENOBUFS;
		splx(s);
		goto bad;
	}
	else
	{
		IF_ENQUEUE(&ifp->if_snd, m)
		error = 0;
	}
	if ((tr->tr_flags & TRTXBSY) == 0)
	    trstart(ifp->if_unit);
	else
	    if (trdebug & TRDBG_TX)
		printf("xmit bsy\n");
	splx(s);
	return(error);

bad:
	m_freem(m);
	return(error);
}



/*
 * Start or restart transmitting
 */

trstart (unit)
register int unit;
{
	register struct tr_softc 	*tr = &tr_softc[unit];
	register struct mbuf 		*m;
	register struct pkbuf 		*pkb;
	register short 			len;

	tr->tr_flags |= TRTXBSY;

	if (trdebug & TRDBG_TX)
		printf ("trstart called\n");

	IF_DEQUEUE(&tr->tr_if.if_snd, m);
	if (m == 0)
	    goto bad;

/*
**  Convert mbuf chain into single transmit list chain.
*/
	if ((pkb = GetPkBuf()) == (struct pkbuf *)0) {
		IF_PREPEND(&tr->tr_if.if_snd, m);
		goto bad;
	}
	len = if_wqbput(pkb->pkb_buffer, m);

	if (trxmit(unit, pkb, len) == 0)
	    timeout(trtxtimo, unit, TRTXTIMO);
	else
	    goto bad;


	return;

bad:
	tr->tr_flags &= ~TRTXBSY;
	return;
}


/*
** Gather mbufs into single pkbuf.  Although the chip is smart
**	enough to gather mbufs directly, it is very slow, so we will do
**	here.
*/

trxmit (unit, pkb, len)
register int unit;
register struct pkbuf *pkb;
register int len;
{
	register struct tr_softc 	*tr = &tr_softc[unit];
	register struct scb 		*scb = &tr->tr_comm.scb;
	Trdevice   *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	register struct trbuflist	*tbl = 
		(struct trbuflist *)(CACHE_INHIBIT(tr->tr_comm.tbl));
	u_char 				*addr;

	if ((tbl->bl_cstat & (TBL_VALID|TBL_COMPLETE|TBL_SOF)) == 
							(TBL_SOF|TBL_VALID))
	{
	    printf("trxmit: xmit list full (tbl 0x%x, cstat 0x%x)\n", 
							tbl, tbl->bl_cstat);
	    goto busy;
	}

	if ((tr->tr_flags & TRTXCMD) == 0 && !trscbclear(unit, 0))
	{
	    printf("trxmit - SCB not cleared, (to = 0) (scb = 0x%x)\n",
								scb->scb_cmd);
	    goto busy;
	}

	if (len > (TOKENMTU + TOKENHDRSIZE))
	{
	    printf("trxmit - frame size error (len %dd)\n", len);
	    stacktrace();
#ifdef DEBUGGER
	    trap15();
#endif DEBUGGER
	    goto busy;
	}

	if (trdebug & TRDBG_BUFS)
	    printf("trxmit: tbl 0x%x, pkb 0x%x\n", tbl, pkb);

	tbl->bl_cstat = TBL_SOF|TBL_INTR|TBL_VALID|TBL_EOF;
	tbl->bl_len = tbl->bl_flen = len;
	tbl->bl_data = pkb;	/* We'll want to free this later */

	addr = (u_char *)IOPB_STD(pkb->pkb_dma_buffer, vbnum);
	tbl->bl_addr = (u_char *)(((u_long)addr & 0x003fffff) | (SIF_MMR0<<24));
		
	if (tr->tr_flags & TRTXCMD)
	{
	    trdev->sif_intr = SIFI_INTA | SIFI_RSI | SIFI_XMTV;
	}
	else
	{
	    scb->scb_cmd = CMD_XMIT;
	    addr = (u_char *)tr->tr_vdma->tblhead;	/* start xmit at head */
	    trloadaddr(&scb->scb_addrh, &scb->scb_addrl, addr, SIF_MMR0);
	    tr->tr_flags |= TRTXCMD;
	    trdev->sif_intr = SIFI_INTA | SIFI_RSI | SIFI_EXE;
	}

	tr->tr_comm.tbl = tbl->bl_next;

	if (trdebug & TRDBG_TX)
	{
	    printf("tr tx %x\n", len);

	    if (trdebug & TRDBG_DUMP)
	    {
		printf("Xmit packet:\n");
		dumppkt(pkb->pkb_buffer, len);
	    }
	}

	tr->tr_if.if_opackets++;
	trtxcount++;
	return 0;
busy:
	FreePkBuf(pkb);
	return EBUSY;			/* drop packet */
}
	

/*
** Dump packet in hex
*/
static
dumppkt(buf, len)
u_char	*buf;
int	len;
{
	register u_char *p;

	for (p=buf; p< (u_char *)((int)buf + len); p++)
	{
	    if (((u_int)(p-buf) & 0xf) == 0x0)
	    {
		if ((p-buf) < 0x10)
		    printf("\n0%x:  ", p-buf);
		else
		    printf("\n%x:  ", p-buf);
	    }
	    if (*p < 0x10)
	        printf("  0%x", *p);
	    else
	        printf("  %x", *p);
	}
	printf("\n");
}


/*
 * Process an ioctl request.
 */
trioctl(ifp, cmd, data)
register struct ifnet *ifp;
int cmd;
caddr_t data;
{
	register struct ifaddr 		*ifa = (struct ifaddr *)data;
	register struct sockaddr 	*sa = &ifa->ifa_addr;
	register struct tr_softc 	*tr = &tr_softc[ifp->if_unit];
	register struct ifreq 		*ifr;
	struct sockaddr_in 		*sin;
	int 				error = 0;
	struct config_entry 		*cf;

	switch (cmd) {
	  case SIOCGHDADDR:
	  	ifr = (struct ifreq *)data;
		bcopy (tr->tr_traddr, ifr->ifr_hdaddr, 6);
		if (trdebug) {
			printf("SIOCGTRADDR addr: (%x.%x.%x.%x.%x.%x)\n",
			tr->tr_traddr[0], tr->tr_traddr[1],
			tr->tr_traddr[2], tr->tr_traddr[3],
			tr->tr_traddr[4], tr->tr_traddr[5]);
		}
		break;

	  case SIOCSIFADDR:
		if (sa->sa_family == AF_UNSPEC ) {
			bcopy((caddr_t)sa->sa_data, (caddr_t)tr->tr_traddr, 
							sizeof(tr->tr_traddr));
			if (trdebug & TRDBG_INIT)
			{
			    printf("New token addr: (%x.%x.%x.%x.%x.%x)\n",
				tr->tr_traddr[0], tr->tr_traddr[1],
				tr->tr_traddr[2], tr->tr_traddr[3],
				tr->tr_traddr[4], tr->tr_traddr[5]);
			}
			reset_tr (ifp->if_unit);
			if (trinit(ifp->if_unit))
			{
			    error = EHOSTDOWN;
			    break;
			}
			error = tropen(ifp->if_unit);
			break;
		}
		if (sa->sa_family != AF_INET) {
			error = EINVAL;
			break;
		}
		tr->tr_ipaddr.s_addr = 
			((struct sockaddr_in *)sa)->sin_addr.s_addr;
		reset_tr (ifp->if_unit);
		if (trinit(ifp->if_unit))
		{
		    error = EHOSTDOWN;
		    break;
		}
		error = tropen(ifp->if_unit);
		break;
	  case SIOCSFADDR:
		/* to be added... yank function address from ifreq ifr_sfaddr
		   (ifr_metric) and do CMD_SFA on chipset.  Will later get
		   interrupt on completion and not sure what to do there
		*/
		error = EOPNOTSUPP;	/* for now */
		break;
	  default:
		error = EINVAL;
	}
	return(error);
}


/*
**  Clean buffers.sically reset cstat so we don't loose bufs.
*/

trcleanbl(blist, size, rlist)
register struct trbuflist *blist;		/* Buffer list array */
register int 		  size, rlist;		/* Size of list */

{
	register struct trbuflist *bl, *eol = blist+size-1;

	for (bl=blist; bl <= eol; bl++)
	    if(rlist)
	        bl->bl_cstat = RBL_VALID | RBL_INTR;
	    else
		bl->bl_cstat = 0; 
}


/*
**  Initialize buffer lists.  Must fix one memory management register
**	for each list (transmit and receive), so TMS380 can get to them.
**	LISTS MUST NOT CROSS A 2^21 BOUNDRY!
*/

trinitbl(blist, blistvdma, size, mmr, dodata)
register struct trbuflist *blist, *blistvdma;	/* Buffer list array */
register int 		  size;			/* Size of list */
register int 		  mmr;			/* MMR to use */
register int 		  dodata;		/* Alloc mbufs? */
{
	register struct trbuflist *bl, *blvdma, *eol = blist+size-1;

	for (bl=blist, blvdma=blistvdma; bl < eol; bl++, blvdma++)
	{
	    bl->bl_link = 
		(u_char *)(((u_long)(blvdma+1) & 0x003fffff) | (mmr<<24));
	    bl->bl_next = bl+1;
	    if (dodata)
	    {
		trbufalloc(bl);
	        bl->bl_cstat = RBL_VALID | RBL_INTR;
	    }
	}

	/* Wrap list around */
	eol->bl_link = 
	    (u_char *)(((u_long)blistvdma & 0x003fffff) | (mmr<<24));
	eol->bl_next = blist;
	if (dodata)
	{
	    trbufalloc(bl);
	    bl->bl_cstat = RBL_VALID | RBL_INTR;
	}
}


/*
**  Issue an open command.
*/

struct op_param {
	u_short		opts;
	u_short		n_addrh;
	u_short		n_addrm;
	u_short		n_addrl;
	u_short		g_addrh;
	u_short		g_addrl;
	u_short		f_addrh;
	u_short		f_addrl;
	u_short		rl_size;
	u_short		tl_size;
	u_short		buf_size;
	u_short		exp_ram_start;
	u_short		exp_ram_end;
	u_char		xmit_buf_min;
	u_char		xmit_buf_max;
	u_short		prod_idh;
	u_short		prod_idl;
} tropenparam, *op_vdma;

/*
**  OPEN parameter defines. (TI p. 4-86)
*/
#define	OPEN_WRAP	0x8000		/* Wrap mode */
#define	OPEN_DHARD	0x4000		/* Disable hard errors */
#define	OPEN_DSOFT	0x2000		/* Disable soft errors */
#define	OPEN_PMAC	0x1000		/* Pass Adapter MAC frames */
#define	OPEN_PATT	0x0800		/* Pass attantion MAC frames */
#define	OPEN_PAD	0x0400		/* Pad Routing field */
#define	OPEN_HOLD	0x0200		/* Frame hold */
#define	OPEN_CONTEND	0x0100		/* Contender */
#define	OPEN_PBEACON	0x0080		/* Pass beacon */

u_char	prod_id[18];
u_char *prod_id_vdma;

tropen(unit)
register int unit;
{
	register struct tr_softc 	*tr = &tr_softc[unit];
	register struct op_param 	*trop = &tropenparam;
	register struct scb		*scb = &tr->tr_comm.scb;
	register Trdevice *trdev = 
			    (Trdevice *)trinfo[unit]->qi_mi->qm_addr;

	if (prod_id_vdma == NULL)
		prod_id_vdma = (u_char *)(IOPB_STD(
			iopballoc(prod_id, sizeof(prod_id)),vbnum));
	if (op_vdma == NULL)
		op_vdma = (struct op_param *)(IOPB_STD(
			iopballoc(&tropenparam, sizeof(tropenparam)),vbnum));

	bzero(prod_id, 18);
	bcopy((caddr_t)&tr->tr_ipaddr.s_addr, 
		(caddr_t)prod_id, sizeof(tr->tr_ipaddr.s_addr));
	trop->opts = OPEN_CONTEND;

	if (trdebug & TRDBG_INIT)
	{
	    printf("Open token addr: (%x.%x.%x.%x.%x.%x)\n",
		tr->tr_traddr[0], tr->tr_traddr[1],
		tr->tr_traddr[2], tr->tr_traddr[3],
		tr->tr_traddr[4], tr->tr_traddr[5]);
	}
	trop->n_addrh = *((u_short *)&tr->tr_traddr[0]);
	trop->n_addrm = *((u_short *)&tr->tr_traddr[2]);
	trop->n_addrl = *((u_short *)&tr->tr_traddr[4]);

	trop->g_addrh = trop->g_addrl = /*trop->f_addrh = */ trop->f_addrl = 0;
	trop->f_addrh = 0x4000;		/* kludge until SIOCSFADDR */
	trop->rl_size = trop->tl_size = 14;  	/* we use small lists */
	trop->buf_size = TOKENBUFSIZE;		/* defined in if_token.h */
	trop->xmit_buf_min = 0;
	trop->xmit_buf_max = TOKENBUFMAX;	/* defined in if_token.h */
	trop->exp_ram_start = trop->exp_ram_end = 0;
	trloadaddr(&trop->prod_idh, &trop->prod_idl, prod_id_vdma, SIF_MMR2);
	trdev->mmr2 = TRLOADMMR(prod_id_vdma);

	if (!trscbclear(unit, 10))
	{
	    trerr(unit,"tropen - SCB not cleared, to = %d\n", 10);
	    return(EHOSTDOWN);
	}

	scb->scb_cmd = CMD_OPEN;
	trloadaddr(&scb->scb_addrh, &scb->scb_addrl, op_vdma, SIF_MMR3);
	trdev->mmr3 = TRLOADMMR(op_vdma);
	trdev->sif_intr = SIFI_INTA | SIFI_RSI | SIFI_EXE;

	return(0);
}


/*
** Issue a receive command.
*/

trrecv(unit)
register int unit;
{
	register Trdevice   *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	register struct tr_softc	*tr = &tr_softc[unit];
	register struct scb		*scb = &tr->tr_comm.scb;

	/*  Initialize cstat words */

	if (!trscbclear(unit, 0))
	{
	    trerr(unit, "trrecv - SCB not cleared, to = %d\n", 0);
	    return;
	}

	scb->scb_cmd = CMD_RECV;
	trloadaddr(&scb->scb_addrh, &scb->scb_addrl, 
		(u_char *)(tr->tr_vdma->rblhead), SIF_MMR0);
	tr->tr_flags &= ~TRCDONE;
	trdev->sif_intr = SIFI_EXE | SIFI_INTA | SIFI_RSI | SIFI_SCBRQ;
}


/*
** Check for SCB command word clear.
*/

trscbclear(unit, timo)
register int	unit, timo;
{
	register struct tr_softc *tr = &tr_softc[unit];
	register struct scb *scb = 
		(struct scb *)(CACHE_INHIBIT(&tr->tr_comm.scb));

    if(scb->scb_cmd){
	if (timo)
	{
	    timeout (trtimeout, unit, timo);
	    sleep(&tr->tr_flags, PRIBIO);
	}
    }

    return (scb->scb_cmd == (u_short)0);
}

/*
** 	timeout routine.
*/

trtimeout(unit)
int	unit;
{
	register struct tr_softc *tr = &tr_softc[unit];

	tr->tr_flags |= TRTIMO;		/* Set flag for spin waits */
	wakeup(&tr->tr_flags);		/* Wakeup for sleep waits */
	return;
}


/*
**   Check for proper card initialization
*/
trinitialized(unit)
int	unit;
{
	register struct tr_softc *tr = &tr_softc[unit];
	register Trdevice *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;


	tr->tr_flags &= ~TRTIMO;
	timeout (trtimeout, unit, 3*hz);	/* Must be done in 3 seconds*/

	/*
	**  If intial boot, CPU isn't doing anything else anyway
	*/
#ifdef USE_SLEEP
	if (tr->tr_flags & TRINIT)
		sleep (&tr->tr_flags, PRIBIO);
	else
#endif USE_SLEEP
	{
	    while ((trdev->sif_intr & (SIFI_INIT|SIFI_ERR)) == 0 &&
	           (tr->tr_flags & TRTIMO) == 0)
		   ;
	    untimeout(trtimeout, unit);
	}
		
	if ((trdev->sif_intr & (SIFI_INIT|SIFI_TEST|SIFI_ERR|SIFI_ERCODE))
		== SIFI_INIT)			/* Init ok */
	{
	    return 1;
	}

	if ((trdev->sif_intr&(SIFI_ERR|SIFI_TEST)) == (SIFI_ERR|SIFI_TEST))
	    printf ("tr%d: Initialization error 0x%x\n", unit,
					trdev->sif_intr);
	else
	{
	    printf ("tr%d: Initialization timed out\n", unit);
	    return -1;
	}

	return 0;
}


/*
**  Initialization Block
*/
struct	init_cb {
	u_short	ib_iopts;			/* Initialization Options */
	u_char	ib_cmd_vec;			/* Command interrupt vector */
	u_char	ib_xmt_vec;			/* Transmit interrupt vector */
	u_char	ib_rcv_vec;			/* Receive interrupt vector */
	u_char	ib_rng_vec;			/* Ring interrupt vector */
	u_char	ib_scb_vec;			/* SCB Clear interrupt vector */
	u_char	ib_chk_vec;			/* Adapter Check intr vec.*/
	u_short	ib_rburst;			/* Receive burst size */
	u_short	ib_tburst;			/* Transmit burst size */
	u_short	ib_abthresh;			/* DMA Abort threshold */
	u_short	ib_scbaddr_hi;			/* SCB addr, high byte */
	u_short	ib_scbaddr_lo;			/* SCB addr, low word */
	u_short	ib_ssbaddr_hi;			/* SSB addr, high byte */
	u_short	ib_ssbaddr_lo;			/* SSB addr, low word */
};

/*
** Initialization option definitions (TI page 4-73)
**	Top bit bust always be one.
*/
#define	INIT_PARITY1		0xc000		/* Parity ENA 1 */
#define	INIT_PARITY2		0xa000		/* Parity ENA 2 */
#define	INIT_BSB		0x9000		/* Burst SCB/SBB */
#define	INIT_BLST		0x8800		/* Burst List */
#define	INIT_BLSTAT		0x8400		/* Burst List Status */
#define	INIT_BRDATA		0x8200		/* Burst Receive Data */
#define	INIT_BTDATA		0x8100		/* Burst transmit Data */

/*
**  Initialize chip set software.
*/

trinitbrd(unit)
int	unit;
{
	register struct tr_softc *tr = &tr_softc[unit];
	register Trdevice *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	register u_short *p;
	struct scb *scb = &tr->tr_comm.scb;
	struct	init_cb	trib;

	/* Fill in init info */

	trib.ib_iopts = INIT_BSB|INIT_BLST|INIT_BLSTAT|INIT_BRDATA|INIT_BTDATA;
	trib.ib_rng_vec = trib.ib_scb_vec = trib.ib_chk_vec = trib.ib_xmt_vec =
				trib.ib_cmd_vec = trib.ib_rcv_vec = tr->tr_cvec;
	trib.ib_rburst = trib.ib_tburst = 0;
	trib.ib_abthresh = 0x0101;

	trloadaddr(&trib.ib_scbaddr_hi, &trib.ib_scbaddr_lo,
		(u_char *)(&tr->tr_vdma->scb), SIF_MMR0);
	trdev->mmr0 = trdev->mmr1 = 
		TRLOADMMR((u_char *)(&tr->tr_vdma->scb));

	trloadaddr(&trib.ib_ssbaddr_hi, &trib.ib_ssbaddr_lo, 
		(u_char *)(&tr->tr_vdma->ssb), SIF_MMR0);

	/* Write init block */

	trdev->sif_addr = 0x0a00;	/* Initialization start address */
	for (p = (u_short *)&trib; p < (u_short *)(&trib+1); p++)
		trdev->sif_data_inc = *p;	/* Write a word */

	trdev->sif_intr = SIFI_INTA | SIFI_EXE | SIFI_RSI;

	/* Check status */

	tr->tr_flags &= ~TRTIMO;
	timeout (trtimeout, unit, 10*hz);    	/* Must be done in 10 seconds*/

	/* Once again, if we are booting, no need to sleep, CPU is ours */

#ifdef USE_SLEEP
	if (tr->tr_flags & TRINIT)
		sleep(&tr->tr_flags, PRIBIO);
	else
#endif USE_SLEEP
	{
	    while ((tr->tr_flags & TRTIMO) == 0)
	    {
		if ((trdev->sif_intr & (SIFI_INIT|SIFI_TEST|SIFI_ERR)) == 0 ||
						(trdev->sif_intr & SIFI_ERR))
		    break;
	    }
	    untimeout (trtimeout, unit);
	}
	
	if ((trdev->sif_intr & (SIFI_INIT|SIFI_TEST|SIFI_ERR)) == 0)
	    return 1;

	if (trdev->sif_intr & SIFI_ERR)		/* Init failed */
	    printf ("tr%d: Software init error 0x%x\n", unit,
					trdev->sif_intr & SIFI_ERCODE);
	else	/* Timed out */
	{
	    printf ("tr%d: Software init timed out (0x%x)\n", unit,
					trdev->sif_intr);
	    return -1;
	}
	return 0;
}


/*
** load a system address into IF format.
*/
trloadaddr(hi, lo, addr, mmr)
register u_short	*hi, *lo;
register u_char		*addr;
register u_short	mmr;
{
	*hi = (u_short)((((u_long)addr & 0x003f0000) >> 16) | mmr);
	*lo = (u_short)((u_long)addr & 0x0000ffff);
}

/*
**  Reset the whole card.
*/

static
reset_tr(unit)
int unit;
{
	register struct tr_softc	*tr = &tr_softc[unit];
	register Trdevice *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;

	tr->tr_if.if_flags &= ~(IFF_RUNNING|IFF_UP);
#ifdef TRFS
	if (tr->tr_ix != -1)
	    iface[tr->tr_ix].flags &= ~UP;
#endif TRFS

	trdev->csr = (RESET | DMAENA | INTENA);
	trdev->csr = (DMAENA | INTENA);
}

/*
**  Handle an error
*/

trerr(unit, s, a)
int	unit;
char	*s;
int	a;
{
	register struct tr_softc *tr = &tr_softc[unit];
	register struct ifnet *ifp = &tr->tr_if;
	register Trdevice *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;

	printf("tr%d: ", unit);
	printf(s, a);

	ifp->if_flags &= ~IFF_UP;		/* Clear UP bit. */
#ifdef TRFS
	if (tr->tr_ix != -1)
	    iface[tr->tr_ix].flags &= ~UP;
#endif TRFS
	trdev->csr = (DMAENA | INTENA);		/* Turn off LED */
}


/*
** Error log buffer (TI p. 4-112)
*/

struct errlog {
	u_char	el_line;		/* Line error */
	u_char	el_resv1;
	u_char	el_burst;		/* Burst error */
	u_char	el_arifci;		/* ARI/FCI error */
	u_short el_resv2;
	u_char	el_lost;		/* Lost frame error */
	u_char	el_cong;		/* Receive congestion */
	u_char	el_copied;		/* Frame copied error */
	u_char	el_resv3;
	u_char	el_token;		/* Token error */
	u_char	el_resv4;	
	u_char	el_bus;			/* DMA Bus error */
	u_char	el_parity;		/* DMA Parity error */
} el, *el_vdma;

/*
**  Read adapter's error log.
*/

trreaderrlog(unit, post)
register int	unit, post;
{
	register struct tr_softc *tr = &tr_softc[unit];
	register struct scb *scb = &tr->tr_comm.scb;
	register struct ssb 		*ssb = 
		(struct ssb *)(CACHE_INHIBIT(&tr->tr_comm.ssb));
	register u_char	*p;
	Trdevice *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	register u_char *elp;

	if (el_vdma == NULL)
		el_vdma = (struct errlog *)(IOPB_STD(
			iopballoc(&el, sizeof (struct errlog)),vbnum));
	if (post)
	{
	    if (!trscbclear(unit, 0))
	    {
		printf("trreaderrlog - SCB not cleared, to = %d\n", 0);
		return;
	    }

	    scb->scb_cmd = CMD_REL;
	    trloadaddr(&scb->scb_addrh, &scb->scb_addrl,
			 (u_char *)el_vdma, SIF_MMR0);
	    tr->tr_flags &= ~TRCDONE;
	    trdev->sif_intr = SIFI_INTA | SIFI_EXE | SIFI_RSI;
	}
	else
	{
	    if (ssb->ssb_stat0 != CMD_CMPLT)
	    {
		printf("trreaderrlog - command didn't complete (stat 0x%x)\n",
							ssb->ssb_stat0);
		return;
	    }

	    printf("Log:");
	    elp = (u_char *)(CACHE_INHIBIT(&el));
	    for (p = (u_char *)&el; p <= &el.el_parity; p++, elp++)
		printf(" 0x%x", *elp);
	    printf("\n");
	}
}


/*
** SEARCH AND DISTROY, er, um, that is, free, mbuf 
** chains on the xmit buf list.
*/

trfreebuf(tbl)
register struct trbuflist *tbl;
{
	register struct trbuflist *bl;

/*
**  Check all buffers.  Any Start Of Frame buffer with a zero VALID bit and
**	non zero data pointer gets freed.
*/

	tbl = (struct trbuflist *)(CACHE_INHIBIT(tbl));
	for (bl = tbl; bl < (tbl+TXBUFS); bl++)
	{
	    if ((bl->bl_cstat & (TBL_SOF|TBL_VALID)) == TBL_SOF &&
	         bl->bl_data != 0)
	    {
		if (trdebug & TRDBG_BUFS)
		    printf("trfreebuf: bl 0x%x, data 0x%x\n", bl, bl->bl_data);

		FreePkBuf(bl->bl_data);
		bl->bl_data = 0;
	    }
	}
}


trtxtimo(unit)
int unit;
{
	register struct tr_softc *tr = &tr_softc[unit];
	register Trdevice *trdev = (Trdevice *)trinfo[unit]->qi_mi->qm_addr;
	
	printf("tr%d: tx timeout\n", unit);
	printf("      sif_intr 0x%x\n", trdev->sif_intr);
}


trbufalloc(bl)
register struct trbuflist *bl;
{
	register struct pkbuf *pkb;
	u_long addr;

	if ((pkb = GetPkBuf()) == (struct pkbuf *)0)
	{
	    trerr (0, "trbufalloc no bufs\n", 0);
	    return EBUSY;
	}

	bl->bl_data = pkb;
	addr = IOPB_STD(pkb->pkb_dma_buffer, vbnum);
	bl->bl_addr=(u_char *)((addr & 0x003fffff) | (SIF_MMR0<<24));
	bl->bl_len = PKBUFSIZE;
}
		

#ifdef TRFS
tr_xmit(unit, naddr, p, len, fp0, fp1)
int unit;
u_char *naddr;
struct packet *p;
register unsigned long len;
struct fragment *fp0, *fp1;
{
	register struct tr_softc	*tr = &tr_softc[unit];
	register struct token_header	*th;
	register struct llc_header	*llc;
	register struct snap		*snap;
	register struct pkbuf 		*pkb;
	u_char				*data;
	register int 			error, s = splimp();

	if (trdebug & TRDBG_TRFS)
	    printf("tr_xmit dst (%x.%x.%x.%x.%x.%x) len 0x%x\n",
					naddr[0], naddr[1], naddr[2],
					naddr[3], naddr[4], naddr[5], len);

	if (tr->tr_flags & TRTXBSY) {
	    error = EBUSY;
	    tr->tr_flags &= ~TRTXBSY;
	    goto busy;
	}
	tr->tr_flags |= TRTXBSY;

	if (trdebug & TRDBG_TX)
	    printf("tr_xmit called\n");

	if ((pkb = GetPkBuf()) == (struct pkbuf *)0) {
	    printf("tr_xmit: no packet buffers\n");
	    error = EBUSY;
	    tr->tr_flags &= ~TRTXBSY;
	    goto busy;
	}

	th = (struct token_header *)pkb->pkb_buffer;
	bcopy(naddr, th->token_dhost, sizeof(th->token_dhost));
	th->token_shost[0] = 0;  /* No routing info field */
	th->token_access = 0;
	th->token_frame = FC_STD;

	llc = (struct llc_header *)(th + 1);
	llc->llc_dsap = llc->llc_ssap = SAP_IP;
	llc->llc_cntl = LLC_U;

	snap = (struct snap *)((int)llc + LLC_SIZE);
	snap->snap_spare[0]= snap->snap_spare[1]= snap->snap_spare[2]= 0;
	*(u_short *)&snap->snap_type[0] = ETHERTYPE_WIPC;

	len = pkbuild(&((u_char *)th)[TOKENHDRSIZE], p, len, fp0, fp1) + 
								TOKENHDRSIZE;

	if ((error = trxmit(unit, pkb, len)) == 0)
	    timeout(trtxtimo, unit, TRTXTIMO);
	else
	    tr->tr_flags &= ~TRTXBSY;

busy:
	if (trdebug & TRDBG_TRFS)
		printf(".");
	splx(s);
	return(error);
}


tr_recv(unit, pkb, len, off)
register int unit, len, off;
register struct pkbuf *pkb;
{
	register struct pklink *p = (struct pklink *)pkb->pkb_buffer;
	register u_char	*data = (u_char *)&pkb->pkb_buffer[TOKENHDRSIZE+off];
	register int ret;


	ovbcopy(((struct token_header *)p)->token_shost, p->netaddr, 6);

	if (trdebug & TRDBG_TRFS)
	    printf("tr_recv src (%x.%x.%x.%x.%x.%x) len 0x%x\n",
			p->netaddr[0], p->netaddr[1], p->netaddr[2], 
			p->netaddr[3], p->netaddr[4], p->netaddr[5], len);

	p->buflen = len;
	p->prelen = 0;
	p->ix = tr_softc[unit].tr_ix;
	p->handle = (int)pkb;
	ovbcopy(data, (u_char *)&p->packet, len-TOKENHDRSIZE);
	if ((ret = ReceivePacket(p)) != 0)
	{
	    if (trdebug & TRDBG_TRFS)
		printf("%d-", ret);
	    FreePkBuf(pkb);
	}

	return ret;
}

tr_relse(unit, p)
int unit;
register struct pklink *p;
{
	FreePkBuf((struct pkbuf *)p->handle);
}
#endif	TRFS
#endif	NTR > 0
