/*
 * QBUS|VMEBUS: span disk driver
 */
#include "sp.h"
#if NSP > 0 || Nsp > 0

/* TODO -
 * pick real major device numbers for block/char devices in conf.c
 * call diskpart to do consistant sizing (later, will require much re-write)
 * WON'T WORK WITH 18-BIT DRIVERS - local buffers should be allocated/qualified.
 */
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "dk.h"
#include "dkbad.h"
#include "buf.h"
#include "conf.h"
#include "dir.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"

#include "../is68kdev/qbvar.h"

#include "../h/spaninfo.h"

#define Nsp_SP		4	/* max number of sp drives on SP controler */
#define SP_MAXCNT	(63*1024)

/*
 * The internal configuration/status table, one entry per pseudo-drive.
 */
struct spst {
	dev_t	st_dev[Nsp_segs] ;	/* list of makedev()'s */
	int	st_size[Nsp_segs];	/* individual sizes of above */
	int	st_totsiz;		/* total size of pseudo-drive */
} spst[Nsp] = { {0}, {0}, {0}, {0} };

struct buf rsp_bp[Nsp];		/* local buffers for pseudo raw device xfers */
#define b_bufcount b_actf	/* # local buffers active for user buffer. */
#define b_bp	b_back		/* local buffer back pointer to user buffer */

int	spprobe(),	spslave(),	spattach(), spintr(), spminphys();
struct	qb_ctlr		*spcinfo[NSP];
struct	qb_device	*spdinfo[Nsp];
u_short *spstd[] = { (u_short *)-1, 0 } ;	/* no address */
struct	qb_driver SPdriver =
	{ spprobe, spslave, spattach, 0, spstd, "sp", spdinfo, "SP", spcinfo };

struct buf	*spgetbuf();

/*
 * spprobe - If the driver is loaded, then the device exists even if it is
 * not currently configured, so always return ok.  Same applies to spslave.
 */
spprobe()
{
	cvec = 0;
	return (4);
}

spslave()
{
	return (1);
}


spattach(qi)
register struct qb_device	*qi;
{
	extern dev_t		*sp_config[];
	int			i, maj, mnr;
	register struct spst	*st = &spst[qi->qi_slave];
	register dev_t		*cf = sp_config[qi->qi_slave];
	extern char		*bdevname[];
	extern int		nbdevnames;

	st->st_totsiz = 0;
	printf("\t(");
	for (i = 0 ; i < Nsp_segs && cf[i] ; ++i) {
		maj = major(cf[i]); 
		mnr = minor(cf[i]); 
		if (maj >= nblkdev || bdevsw[maj].d_psize == 0
		     || (*bdevsw[maj].d_psize)(cf[i]) <= 0) {
			printf("(%d,%d) **no device**", maj, mnr);
			st->st_totsiz = 0;
			break;
		}
		if (maj < nbdevnames && bdevname[maj])
			printf("%s%d%c ", bdevname[maj], mnr>>3, 'a'+(mnr&7));
		else
			printf("(%d,%d) ", maj, mnr);
		st->st_dev[i] = cf[i];
		st->st_size[i] = (*bdevsw[maj].d_psize)(st->st_dev[i]);
		st->st_totsiz += st->st_size[i];
	}
	if (st->st_totsiz == 0)
		printf(" **not configured**)");
	else
		printf("):%mM (%d)", st->st_totsiz*DEV_BSIZE, st->st_totsiz);
}

/*
 * spopen must return OK (0) even if device isn't configured so that ioctl
 * can be used to do configuration.  This may be a problem.
 */
spopen(dev)
dev_t dev;
{
	return ((minor(dev) > Nsp) ? ENXIO : 0);
}

/*
 * spstrategy takes the request from the bp an breaks it up into an arbitrary
 * number of requests to disk drivers, where the breaking is determined by
 * whether the original request overlaps a disk partition boundary.  If hte
 * request is broken up, then a buffers are allocated from a local pool
 * (splocal[]).  There are two pieces of info stored in the buffers to
 * keep track of which local buffers are associated with which original
 * request.   First, each local buffer (obtained from spgetbuf()) has its
 * b_back (refered to as b_bp) pointer set to the original buffer (bp).
 * This allows spintr() to know which request spawned this local buffer.
 * Second, the b_actf field in the original buffer (bp) is used as a counter
 * and is incremented each time spstrategy spawns a new local buffer for this
 * bp, and decrememtned by spintr each time a local i/o request is satisfied.
 * When spintr sees this field decrement to zero, the original request is
 * satisfied, and biodone() is called for te origianal request.
 *
 * (01/30/87) - There appears to be a potential race condition, involving
 * the b_bufcount field.  Consider the followin scenario:
 *	1. The original bp will require 2 or more spbp's to be satisfied.
 *	2. The "while" loop in spstrategy increments b_bufcount from zero
 *	   to one and passes off the first spbp.
 *	3. The while loop goes to sleep in spgetbuf.
 *	4. The underlying device that got the first spbp finishes, calls
 *	   iodone on the spbp, which in turn calls spintr, which decrements
 *	   bp->b_bufcount to 0, and then calls iodone with the original bp.
 *	   Meanwhile, bp is not really finished yet, because we are still
 *	   asleep waiting for a spbp to do the second part of it.
 * The fix is to initialize bp->b_bufcount to 1 instead of 0 before the while
 * loop begins, then add the decrement and test for zero after the loop
 * is finished.  The variable 'alldone' is used to avoid calling iodone()
 * at high priority.
 */
spstrategy(bp)
register struct buf *bp;
{
	register int		sz, s, unit=minor(bp->b_dev);
	register struct spst	*st = &spst[unit];
	register struct buf	*spbp;
	int	st_bn;		/* starting block on pseudo-drive
				    for currently active request */
	int	st_left;	/* bytes remaining to satisfy original
				    request to pseudo-device */
	int	i, offset, blkleft, alldone;

	if (unit >= Nsp || st->st_totsiz == 0 || st->st_dev[0] == 0) {
		bp->b_error = ENXIO;
		goto bad;
	}
	if (bp->b_flags & (B_PAGET | B_UAREA)) {
		printf("sp_flags=%b\n", bp->b_flags, B_FLAGS_BITS);
		goto bad;
	}
	if (bp->b_blkno < 0 || 
	    (bp->b_blkno+((bp->b_bcount+511)>>9)) > st->st_totsiz) {
		if (bp->b_blkno == st->st_totsiz) {
		    bp->b_resid = bp->b_bcount;
		    goto done;
		}
		bp->b_error = EINVAL;
		goto bad;
	}
	st_left = bp->b_bcount;
	st_bn = bp->b_blkno;

	/*
	 * Break up the request in bp into arbitrary many smaller requests
	 * to hardware drivers.
	 */
	bp->b_bufcount = (struct buf *)1;
	bp->b_resid = bp->b_bcount;
	alldone = 0;
	while (st_left > 0) {
		st = &spst[unit];
		spbp = spgetbuf();

		/*
		 * set spbp's back pointer to bp, and bump bp's local
		 * buffer counter.
		 */
		s = splbio();
		spbp->b_bp = bp;
		bp->b_bufcount = (struct buf *)((int)bp->b_bufcount + 1);
		splx(s);

		/* Load the buffer for the current segment */
		offset = 0;
		for (i=0 ; st_bn >= (offset+st->st_size[i]) ; i++)
			offset+=st->st_size[i];
		if (i >= Nsp_segs || st->st_size[i] == 0)
			panic("spgo illegal segment");
		spbp->b_dev = st->st_dev[i];
		spbp->b_blkno = st_bn - offset;

		/* set blkleft to # of blocks remaining in segment i. */
		blkleft = (offset + st->st_size[i]) - st_bn;
		sz = MIN(st_left, (blkleft << DEV_BSHIFT));
		if (sz <= 0)
			panic("spstrategy - size negative");
		spbp->b_bcount = spbp->b_bufsize = sz;
		spbp->b_flags =
		    (bp->b_flags & (B_WRITE|B_READ|B_PHYS))|B_BUSY|B_CALL;
		spbp->b_error = 0;
		spbp->b_un.b_addr=
		    bp->b_un.b_addr + ((st_bn-bp->b_blkno)<<DEV_BSHIFT);
		spbp->b_iodone = spintr;
		spbp->b_proc = bp->b_proc;
		st_left -= spbp->b_bcount;
		st_bn += ((spbp->b_bcount + (DEV_BSIZE-1)) >> DEV_BSHIFT);
		(* bdevsw[major(spbp->b_dev)].d_strategy)(spbp);
	}
	s = splbio();
	bp->b_bufcount = (struct buf *)((int)bp->b_bufcount - 1);

	/*
	 * if interrupts were too fast for us - spintr has already
	 * been called for all the spbp's (see the verbose race
	 * description above).
	 */
	if ((int)bp->b_bufcount == 0)
		alldone++;;
	splx(s);
	if (alldone)
		iodone(bp);
	return;
bad:	bp->b_flags |= B_ERROR;
done:	iodone(bp);
}

/*
 * spintr -
 *
 * check pseudo device status to see if more partial transfers are needed
 * to complete original request.
 */
spintr(spbp)
struct buf *spbp;
{
	register struct buf	*bp;
	register int		s;

	s = splbio();
	bp = spbp->b_bp;
	if (bp == 0)
		panic("spintr: null backpointer");

	if (bp->b_error |= spbp->b_error)
		bp->b_flags |= B_ERROR;
	bp->b_resid -= spbp->b_bcount;
	bp->b_bufcount = (struct buf *)((int)bp->b_bufcount - 1);
	splx(s);

	spbp->b_flags = 0;
	spfreebuf(spbp);
	if ((int)bp->b_bufcount == 0)
		iodone(bp);
}

spread(dev, uio)
dev_t	dev;
struct uio *uio;
{
	register int unit = minor(dev);

	if (unit >= Nsp)
		return (ENXIO);
	return (physio(spstrategy, &rsp_bp[unit], dev, B_READ, spminphys, uio));
}

spwrite(dev, uio)
dev_t dev;
struct uio *uio;
{
	register int unit = minor(dev);

	if (unit >= Nsp)
		return (ENXIO);
	return (physio(spstrategy, &rsp_bp[unit], dev, B_WRITE, spminphys, uio));
}

/*
 * minphys is a kludge that represents the smallest number currently
 * found in any of our existing drivers.  There must be a better way.
 */
spminphys(bp)
struct buf *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, SP_MAXCNT);
}

/*
 * Local buffer pool stuff.  Can't use buffers from geteblk or friends,
 * because we need to steal av_forw and av_back for local purposes.  The
 * constant NLOCAL should be a user alterable param, like in param.h.
 * and the allocation should then be in spconfig.c.  Later.
 */

#define NLOCAL 16

struct buf	splocal[NLOCAL] = {0};
struct buf	*splastbuf = &splocal[NLOCAL-1];
int		spwantbuf=0;
int		spsleep=0;

struct buf *
spgetbuf()
{
	register struct buf	*p;
	register int		s;

	s = splbio();
	while (1) {
		for (p=splocal; (p->b_flags & B_BUSY) && p <= splastbuf; p++)
			;
		if (p > splastbuf) {
			spsleep++;
			spwantbuf++;
			sleep(&spwantbuf, PRIBIO);
		} else
			break;
	}
	p->b_flags = B_BUSY;
	splx(s);
	return (p);
}

spfreebuf(p)
register struct buf *p;
{
	register int s;

	s = splbio();
	p->b_flags = 0;
	if (spwantbuf) {
		spwantbuf=0;
		wakeup(&spwantbuf);
	}
	splx(s);
}

spsize(dev)
dev_t	dev;
{
	register struct spst	*st = &spst[minor(dev)];

	if (st->st_totsiz == 0)
		return (-1);
	else
		return (st->st_totsiz);
}

spioctl(dev, cmd, data)
dev_t dev;
register struct spaninfo *data;
{
	register int unit = minor(dev), i, maj;
	register struct spst	*st;

	if (unit > Nsp)
		return (ENXIO);
	st = &spst[unit];

	if (cmd == GSPANINFO) {
		for (i=0 ; st->st_size[i] ; ++i) {
			data->sp_size[i] = st->st_size[i];
			data->sp_dev[i] = st->st_dev[i];
		}
		data->sp_size[i] = 0;
		data->sp_dev[i] = 0;
		return (0);
	} else if (cmd == SSPANINFO) {
		st->st_totsiz = 0;
		/* check for valid data...  */
		for (i=0 ; i < Nsp_segs && data->sp_dev[i] ; ++i) {
			maj = major(data->sp_dev[i]); 
			if (maj >= nblkdev || bdevsw[maj].d_psize == 0
			     || (*bdevsw[maj].d_psize)(data->sp_dev[i]) <= 0)
				return (EIO);
		}
		if (i == Nsp_segs)
			return (EIO);
		/* data looks good, use it...  */
		for (i=0 ; i < Nsp_segs && data->sp_dev[i] ; ++i) {
			maj = major(data->sp_dev[i]); 
			st->st_dev[i] = data->sp_dev[i];
			st->st_size[i] = (*bdevsw[maj].d_psize)(st->st_dev[i]);
			st->st_totsiz += st->st_size[i];

		}
		if (i < Nsp_segs-1)
			st->st_size[i] = st->st_dev[i] = 0;
		return (0);
	} else
		return (EIO);
}
#endif
