/*#define	DUTRACE	/* */
/*#define	IOCHECK /* */
/*#define HOWFAR	/* */
/*#define	MDMCNTRL	/* */

/*
 *	SIO driver for the VQX SCN68562 on-board dual serial I/O  (DMA version).
 *	    Also known as the "DUSCC".  This file was previously duscc.dma.c
 *	    Hardware-specific variables are named du* and duscc*.
 */
static char SCCSid[] = {"@(#)duscc.dma.c	1.7 7/11/88"};

#include "../machine/pte.h"

#include "param.h"
#include "conf.h"
#include "user.h"
#include "proc.h"
#include "ioctl.h"
#include "tty.h"
#include "map.h"
#include "buf.h"
#include "vm.h"

#include "../machine/board.h"

#include "../is68kdev/sioreg.h"
#include "../is68kdev/openchip.h"
#include "../is68kdev/nvram.h"

#include "bk.h"
#include "clist.h"
#include "file.h"
#include "uio.h"
#include "kernel.h"
#include "syslog.h"
#include "../is68kdev/qbvar.h"	/* for clev_tty	*/

#ifdef	GWS
#include "../is68kdev/gpreg.h"
#endif	GWS
#ifdef	SYSV
#include "../sysv/sys/termio.h"
#endif	SYSV

#define	TTY_NODELAY
#define	IFLAGS	(EVENP|ODDP|ECHO|XTABS|CRMOD);
#define	CLEVSIO	5

int siostart();
int sv_sioproc();

/*
 * The DUSCC interrupt vector scheme is rather complicated if DMA is being
 * used. It all depends on the value of bits 2 and 1. Bit 0 is of no
 * importance.
 * 00X
 *	DUSCC 1XX
 *	DMA   00X
 * 01X
 *	DUSCC 1XX
 *	DMA   01X
 * 10X
 *	DUSCC 0XX
 *	DMA   10X
 * 11X
 *	DUSCC 0XX
 *	DMA   11X
 * It can be seen that to obtain 6 consecutive vectors a pattern of 01X or
 * 10X should be chosen. In practice a 10X pattern will be easier to use.
 */

#define	CHANNEL0_VECTOR	0xa4	/* gives vectors in the range 0xa0 to 0xa6 */
#define	CHANNEL1_VECTOR	0xac	/* gives vectors in the range 0xa8 to 0xae */
char siovec0 = CHANNEL0_VECTOR;
char siovec1 = CHANNEL1_VECTOR;

#define devtodma(d)	(&OPENCHIP->op_duchan[d])

/*
 * There is no driver information for auto-configuration.  It is assumed that
 * you always have two sio ports, and their major number is 0.
 */

int sio_cnt = NSIO;
struct tty sio_tty[NSIO];

struct ttyptr {
	struct dusccchan *tt_addr;	/* addr chip control regs   */
	struct tty *tt_tty;		/* addr tty control struct  */
} sio_ttptr[] = {			/* indexed by minor(dev)    */
	{ (struct dusccchan *)IO_ADRS_TYA, &sio_tty[0] },
	{ (struct dusccchan *)IO_ADRS_TYB, &sio_tty[1] }
};

short	sio_nch[NSIO];			/* number of characters xmiting */
char sio_buf[NSIO][CBSIZE];	/* dma output buffer	     */
int	sio_overrun = 0;
int sio1init;
int sio0init;

/*
 * internal flag registers indicating why output is held up.
 */
unsigned char siooutflags[2];
#define	OUT_TTSTOP	1	/* T_SUSPEND/T_RESUME */
#define	OUT_TTYMODES	2	/* tty modes are to be changed */
#define	OUT_TO_BREAK	4	/* a break is to be sent when FIFO is empty */
#define	OUT_BREAK2	8	/* break command has been sent, wait for ack */
#define	OUT_BREAKING	16	/* a break is currently being sent */

sioouthold(nreason, xreason, dev)
unsigned char nreason, xreason;
dev_t dev;
{
	register
	struct dusccchan *addr = (struct dusccchan *)sio_ttptr[dev].tt_addr;
	struct tty *tp = sio_ttptr[dev].tt_tty;
	register struct tdduscc *tdptr = devtodma(dev);
	unsigned char origflags;

#ifdef NOTDEF
printf("sioouthold(%x, %x, %x)\n", nreason, xreason, dev);
#endif NOTDEF
#ifdef	DUTRACE
	dutrace(6, dev, (nreason<<16)+(xreason<<8)+siooutflags[dev]);
#endif	DUTRACE
	origflags = siooutflags[dev];
	siooutflags[dev] = (origflags & ~xreason) | nreason;
	if (siooutflags[dev]) {
		/*
		 * make sure that no DMA activity is being performed.
		 */
		tdptr->du_txstat = 0;
		if (siooutflags[dev] & (OUT_TTYMODES|OUT_TO_BREAK|OUT_BREAK2)) {
			/*
			 * enable the TRSR[7:3] interrupt.
			 */
			addr->du_ier |= DU_IER_TSR7_3;
		} else {
			addr->du_ier &= ~DU_IER_TSR7_3;
		}
	} else {
		/*
		 * DMA may be reenabled if it was disabled here.
		 */
		if (origflags) {
			if (tp->t_state & TS_ISOPEN) 
				tdptr->du_txstat = OP_DU_INEA | OP_DU_ACT;
		}
		addr->du_ier &= ~DU_IER_TSR7_3;
	}
#ifdef	DUTRACE
	dutrace(0x86, dev, (nreason<<16)+(xreason<<8)+siooutflags[dev]);
#endif	DUTRACE
}

/*
 * The DUSCC chip is clocked at 14.7456MHz. Unfortunately the internal baud
 * rate generator does not generate exactly the rates required by Unix so the
 * clock/timer will have to be used. The clock is divided by 4 before being
 * fed to the clock/timer and the output of the clock/timer should be 32 times
 * the required bit rate if the times 16 receive mode is to be used. However
 * it is further divided by 2 further on.
 */

#define	BD(sp)	(14745600/(4 * sp * 32 * 2))

/*
 * The internal baud rate generator does not support 1800 baud so this will
 * have to be done using the counter/timer directly.
 */
#define	RN(x)	(DU_RTR_CLK_BRG | x)
int durxspeeds[] = {
/*	0		50		75		110
/*	134.5		150		200		300
/*	600		1200		1800		2400
/*	4800		9600		19.2		38400
 */
	0,		RN(0),		RN(1),		RN(2),
	RN(3),		RN(4),		RN(5),		RN(6),
	RN(7),		RN(9),		DU_RTR_CLK_CT,	RN(11),
	RN(12),		RN(13),		RN(14),		RN(15)
};
#define	TN(x)	(DU_TTR_CLK_BRG | x)
int dutxspeeds[] = {
	0,		TN(0),		TN(1),		TN(2),
	TN(3),		TN(4),		TN(5),		TN(6),
	TN(7),		TN(9),		DU_TTR_CLK_2xOWN_CT,	TN(11),
	TN(12),		TN(13),		TN(14),		TN(15)
};


/* Map low nibble of sio field from NVRAM to baud rate */

unsigned char sio_nvram_speeds[] = {
	B9600,		B1200,		B1200,		B1200,
	B1200,		B1200,		B1200,		B1200,
	B1200,		B1200,		B2400,		B2400,
	B9600,		B9600,		B19200,		B19200
};

int sio_ispeed[2] = { 		/* Initial baud rate for ports 0 and 1 */
#ifdef GWS
		B1200,		/* port 0 -- kbd */
		B1200		/* port 1 -- mouse */
#else
		B9600,
		B9600
#endif
};

#ifdef NVR_DEBUG
int nv_dbg_speed = 0;
#endif

char *sioifaddr;
sioinit()
{
	static char sioinitflg;
	struct tty *tp;
	struct nv_ram nv_ram;

	sioifaddr = &sioinitflg;
	/*
	 * only perform the initialisation once.
	 */
	if (sioinitflg &&
		OPENCHIP->op_duchan[0].du_vector == siovec0 &&
		OPENCHIP->op_duchan[1].du_vector == siovec1)
		return;
	sioinitflg = 1;

	/* read baud rate from NVRAM */

	if (nvram_get(&nv_ram)) {
		sio_ispeed[0] = sio_nvram_speeds[nv_ram.sio[0] & 0xf];
		sio_ispeed[1] = sio_nvram_speeds[nv_ram.sio[1] & 0xf];
#ifdef NVR_DEBUG
		nv_dbg_speed = sio_ispeed[0];
#endif
	}
#ifdef NVR_DEBUG2
#ifdef GWS
	sio_ispeed[0] = sio_ispeed[1] = B1200;
#else GWS
	sio_ispeed[0] = sio_ispeed[1] = B9600;
#endif GWS
#endif NVR_DEBUG
	sio_ttptr[0].tt_tty->t_addr = (struct dusccchan *)sio_ttptr[0].tt_addr;
	sio_ttptr[0].tt_tty->t_dev = 0;
	sio_ttptr[1].tt_tty->t_addr = (struct dusccchan *)sio_ttptr[1].tt_addr;
	sio_ttptr[1].tt_tty->t_dev = 1;
	
	/* modify vector does not work unless you've stored into du_ivr	*/
	/* even though the openchip ignores contents of du_iver		*/
	((struct dusccchan *)sio_ttptr[0].tt_addr)->du_ivr = 0;

	siochaninit(0);
	siochaninit(1);

	/* bring the console part way up */
	tp = &sio_tty[0];
	tp->t_ospeed = tp->t_ispeed = sio_ispeed[0];
	tp->t_flags = IFLAGS;
	tp->t_universe = 0;
	clev_tty = CLEVSIO;
	sioparam(0);

#ifdef GWS
	/* bring the mouse part way up */
	tp = &sio_tty[1];
	tp->t_ospeed = tp->t_ispeed = sio_ispeed[1];
	tp->t_flags = IFLAGS;
	tp->t_universe = 0;
	clev_tty = CLEVSIO;
	sioparam(1);
#endif GWS
}

siochaninit(dev)
dev_t dev;
{
/*
 * initialise a single channel. Although the specific asynchronous parameters
 * will be set up later in sioparam(), set them up here in case the serial
 * ports are being used for kernel print output.
 */
	register struct dusccchan *addr;
	char dummy;
	
	siooutflags[dev] = 0;
	addr = (struct dusccchan *)sio_ttptr[dev].tt_addr;
	/*
	 * select the crystal as the source for the clock timer and do not
	 * prescale it.
	 */
	addr->du_ctcr = DU_CTCR_0_DTCT_PRESET | DU_CTCR_1_SCALE | DU_CTCR_4CLK;
	addr->du_ctprh = BD(1800) >> 8;
	addr->du_ctprl = BD(1800);
	addr->du_ccr = DU_CCR_CT_START;

	addr->du_cmr1 = DU_CMR1_NRZ | DU_CMR1_PRTY_NONE | DU_CMR1_ASYNC;
	addr->du_pcr =	DU_PCR_X2_CRYS | DU_PCR_GPO2 | DU_PCR_SY_RTS
			| DU_PCR_RTxC_INPUT | DU_PCR_TRxC_INPUT;
	addr->du_tpr = DU_TPR_2STOP | DU_TPR_TX_8BITS;
	/*
	 * transmit clock is times 32 clock/timer. The clock timer is divided by
	 * 2 before delivery to the transmitter section where the times 16 mode
	 * is used.
	 */
	addr->du_ttr = DU_TTR_CLK_32xOWN_CT;
	addr->du_rpr = DU_RPR_STRIP_PRTY|DU_RPR_RX_8BITS;
	addr->du_rtr = durxspeeds[sio_ispeed[minor(dev)]];
	addr->du_ttr = dutxspeeds[sio_ispeed[minor(dev)]];
	/*
	 * TxRDY goes active if the Tx FIFO is empty and RxRDY goes active if
	 * the Rx FIFO is not empty.
	 */
	addr->du_omr = DU_OMR_TxRDY_EMPTY | DU_OMR_RxRDY_NEMPT;
	/*
	 * interrupt enable register.
	 */
	addr->du_ier = DU_IER_RXRDY;
	addr->du_ccr = DU_CCR_ENBL_TX;
	addr->du_ccr = DU_CCR_ENBL_RX;
	/*
	 * turn on the master interrupt enable for this channel and set the
	 * DUSCC up to modify vector bits 0:2 so that it will work properly
	 * with the OpenChip.
	 */
	if (dev == 0) {
		addr->du_icr |= DU_ICR_INTLV_A | DU_ICR_VCTR0 | DU_ICR_MDFY_2_0
				| DU_ICR_VIS | DU_ICR_A_INT_ENBL;
		OPENCHIP->op_duchan[0].du_vector = siovec0;
	} else {
		((struct dusccchan *)sio_ttptr[0].tt_addr)->du_icr |=
			DU_ICR_INTLV_A | DU_ICR_VCTR0 | DU_ICR_MDFY_2_0
				| DU_ICR_VIS | DU_ICR_B_INT_ENBL;
		OPENCHIP->op_duchan[1].du_vector = siovec1;
	}
	devtodma(dev)->du_txchn.dl_cnt = 0;
	devtodma(dev)->du_txchn.dl_lnptr = (struct tdlink *)0;
	/*
	 * There is a problem with the DUSCC in that the TRANSMIT BREAK
	 * command may be asserted inadvertently when it is powered up. It can
	 * only be cancelled by putting the channel into local loopback mode,
	 * enabling the transmitter, disabling the transmitter and then
	 * resetting both transmit and receive.
	 * This bug is listed in the ANOMALIES IN THE REVISION H DUSCC by
	 * Martin Maloney of October 21st, 1986.
	 */
	addr->du_cmr2 = DU_CMR2_CONN_LOOP | DU_CMR2_POLLED | DU_CMR2_NO_FCS;
	addr->du_ccr = DU_CCR_ENBL_TX;
	dummy = 0;	/* small delay */
	addr->du_ccr = DU_CCR_DISABL_TX;
	addr->du_ccr = DU_CCR_RST_TX;
	addr->du_ccr = DU_CCR_RST_RX;
	addr->du_cmr2 = DU_CMR2_CONN_NORM | DU_CMR2_POLLED | DU_CMR2_NO_FCS;
	addr->du_ccr = DU_CCR_ENBL_TX;
	addr->du_ccr = DU_CCR_ENBL_RX;
}


sioopen(dev, flag)
register dev_t dev;
{
	register int unit = minor(dev);
	register struct tty *tp;
	register struct dusccchan *addr;
	int s, old_discipline;
	char error = 0;

	if (unit >= sio_cnt) {
		error = ENXIO;
		return(error);
	}
#ifdef	GWS	/* leave this undefined for now	*/
	/* with graphics, dont open mouse! */
	if ((gptype & GPEXISTS) && unit == 1)	/* gptype undefined */
		return (EBUSY);
#endif	GWS
	tp = sio_ttptr[unit].tt_tty;
	tp->t_addr = addr = (struct dusccchan *)sio_ttptr[unit].tt_addr;
	tp->t_oproc = siostart;
#ifdef	SYSV
	tp->svt_proc = sv_sioproc;
#endif	SYSV
	if ((tp->t_state & (TS_ISOPEN|TS_WOPEN)) == 0) 
		siochaninit(unit);
	if ((tp->t_state&TS_ISOPEN) == 0) {
		ttychars(tp);
		tp->t_ospeed = tp->t_ispeed = sio_ispeed[unit];
		tp->t_flags = IFLAGS;
#ifdef	SYSV
		/*
		 * sv_ttinit initializes svt_line to zero.  This is a
		 * very bad thing to do on the graphics console because
		 * write will then go to the keyboard instead of the screen
		 * causing all sorts of havoc.
		 * The line discipline has been setup during GP_initialize().
		 * Go check for yourself if you don't believe me.
		 */
		old_discipline = tp->svt_line;
		sv_ttinit(tp);
		tp->svt_line = old_discipline;
#endif	SYSV
		sioparam(unit);
		/*
		 * assert DTR, DSR and RTS. DTR is connected to the GPO1 pin.
		 * DSR is connected to the GPO2 pin.
		 */
		addr->du_omr |= DU_OMR_GPOUT1 | DU_OMR_GPOUT2 | DU_OMR_RTS;
	} else if ((tp->t_state&TS_XCLUDE) && 
	    ((u.u_uid!=0) || (u.u_procp->p_flag & SLOGIN)))
		return (EBUSY);
	s = spltty();

	if (siomdm(unit, 1))
#ifdef	SYSV
		if (tp->t_universe == UNIVERSE_SYSV)
			(void)(*sv_linesw[tp->svt_line].l_mdmint)(tp, 1);
		else
#endif	SYSV
		(void)(*linesw[tp->t_line].l_modem)(tp, 1);

#ifdef TTY_NODELAY
	if ((flag & FNDELAY) == 0)
#endif
	while ((tp->t_state & TS_CARR_ON) == 0) {
		tp->t_state |= TS_WOPEN;
#ifdef	SYSV
		map_state(tp, TOSV);
#endif	SYSV
		sleep((caddr_t)&tp->t_rawq, TTIPRI);
		if ((tp->t_state&TS_XCLUDE) && (u.u_procp->p_flag&SLOGIN)) {
			splx(s);
			return (EBUSY);
		}
	}
	splx(s);
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_open)(dev, tp, flag));
	else
#endif	SYSV
		return ((*linesw[tp->t_line].l_open)(dev, tp));
}

/*
 * The two variable sioconsflag and sioconsole are two variables which are
 * used by the sioputchar code. If that code is linked in then sioconsflag will
 * be non-zero and sioconsole will be the minor device number of the device
 * to which console output will be directed. If the sioputchar code is not
 * linked in they will both read as 0.
 */
char sioconsflag;
dev_t sioconsole;

sioclose(dev, flag)
dev_t dev;
{
	register struct tty *tp;
	register int unit = minor(dev);

	tp = sio_ttptr[unit].tt_tty;

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		(*sv_linesw[tp->svt_line].l_close)(tp);
	else
#endif	SYSV
		(*linesw[tp->t_line].l_close)(tp);
/* TODO: should remove break */
	if (tp->t_state&(TS_HUPCLS|TS_WOPEN) || (tp->t_state&TS_ISOPEN)==0)
		siomdm(minor(unit), 0);
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		sv_ttyclose(tp);
	else
#endif	SYSV
		ttyclose(tp);
}

sioread(dev, uio)
dev_t		dev;
struct uio	*uio;
{
	register int unit = minor(dev);
	register struct tty *tp = sio_ttptr[unit].tt_tty;

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_read)(tp, uio));
	else
#endif	SYSV
		return((*linesw[tp->t_line].l_read)(tp, uio));
}

siowrite(dev, uio)
dev_t		dev;
struct uio	*uio;
{
	register int unit = minor(dev);
	register struct tty *tp = sio_ttptr[unit].tt_tty;

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_write)(tp, uio));
	else
#endif	SYSV
		return((*linesw[tp->t_line].l_write)(tp, uio));
}

#ifdef	SYSV
sv_sioproc(tp, cmd)
register struct tty *tp;
register cmd;
{
	sv_proc(tp, cmd, siostart, sioparam);
}
#endif	SYSV


sioioctl(dev, cmd, addr, flag)
dev_t	dev;
caddr_t	addr;
{
	register int error, unit;
	register struct tty *tp = sio_ttptr[unit=minor(dev)].tt_tty;

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV) {
		if ((error = sv_ttiocom(tp, cmd, addr)) > 0)
			return error;
		if (cmd == TCSETAF || cmd == TCSETA || cmd == TCSETAW)
			sioparam(unit);
		if (cmd == TCSBRK)
			;		/* ??? */
		return 0;
	}
#endif	SYSV
	if ((error = (*linesw[tp->t_line].l_ioctl)(tp,cmd,addr,flag)) >= 0)
		return (error);
	if ((error = ttioctl(tp, cmd, addr, flag)) >= 0) {
		if (cmd == TIOCSETP || cmd == TIOCSETN ||
		    cmd == TIOCLSET || cmd == TIOCLBIS || cmd == TIOCLBIC)
			sioparam(unit);
		return (error);
	} 
	switch (cmd) {
	    case TIOCSDTR:
		siomdm(unit, 1);
		break;

	    case TIOCCDTR:
		siomdm(unit, 0);
		break;

/* TODO: support these
	    case TIOCSBRK:
	    case TIOCCBRK:
	    case TIOCMSET:
	    case TIOCMBIS:
	    case TIOCMBIC:
	    case TIOCMGET:
/**/

	    default:
		return (ENOTTY);
	}
	return (0);
}


sioparam(dev)
dev_t dev;
{
/*
 * set up the parameters for the DUSCC channel. There is a slight problem in
 * that the system seems to hang if they are changed while output is still
 * being done. If this is the case then enable transmit interrupts and hold
 * up transmit DMA. When the transmit FIFO is empty the modes may be set up.
 */
	register int s;

	s = spl6();
	sioouthold(OUT_TTYMODES, 0, dev);
	splx(s);
}


siosetup(dev)
dev_t dev;
{
	register flag;
	register struct tty *tp;
	register struct dusccchan *addr;
	register unsigned char tpr, rpr;
	register unsigned char mr1;
	register flags;
	int s;

	tp = sio_ttptr[dev].tt_tty;
	addr = (struct dusccchan *)sio_ttptr[dev].tt_addr;
	if (tp->t_ispeed == 0) { /* hang up line */
		siomdm(dev, 0);
		return;
	}
	/*
	 * The receiver and transmitter must be disabled when the CMR1, CMR2,
	 * RPR or TPR registers are written to in order to avoid glitches.
	 */
/*
	addr->du_ccr = DU_CCR_RST_TX;
	addr->du_ccr = DU_CCR_RST_RX;
*/
	addr->du_ccr = DU_CCR_DISABL_TX;
	addr->du_ccr = DU_CCR_DISABL_RX;
	addr->du_omr |= DU_OMR_GPOUT1 | DU_OMR_GPOUT2 | DU_OMR_RTS;

#ifdef	SYSV
	if (tp->t_universe == UNIVERSE_SYSV) {
		flag = tp->svt_cflag;
		if ((flag&SV_CSIZE) == SV_CS8) {
			tpr = DU_TPR_TX_8BITS;
			/*
			 * If 8 bits plus parity is selected, the receiver must
			 * be programmed to strip parity.
			 */
			if (flag & SV_PARENB) {
				rpr = DU_RPR_RX_8BITS | DU_RPR_STRIP_PRTY;
			} else {
				rpr = DU_RPR_RX_8BITS;
			}
		} else if ((flag&SV_CSIZE) == SV_CS7) {
			tpr = DU_TPR_TX_7BITS;
			rpr = DU_RPR_RX_7BITS;
		} else if ((tp->svt_cflag&SV_CSIZE) == SV_CS6) {
			tpr = DU_TPR_TX_6BITS;
			rpr = DU_RPR_RX_6BITS;
		} else {
			tpr = DU_TPR_TX_5BITS;
			rpr = DU_RPR_RX_5BITS;
		}
		/*
		 * perform stripping in hardware.
		 */
		if (flag & SV_ISTRIP) {
			rpr |= DU_RPR_STRIP_PRTY;
		}
		if (flag&SV_CSTOPB) {
			tpr |= DU_TPR_2STOP;
		} else {
			if ((flag & SV_CSIZE) == SV_CS5) {
				tpr |= DU_TPR_1STOP5;
			} else {
				tpr |= DU_TPR_1STOP;
			}
		}
		if (flag&SV_PARENB) {
			if (flag & SV_PARODD) {
				mr1 = DU_CMR1_PRTY_ODD | DU_CMR1_PRTY_WITH;
			} else {
				mr1 = DU_CMR1_PRTY_EVEN | DU_CMR1_PRTY_WITH;
			}
		}
		else
			mr1 = DU_CMR1_PRTY_NONE;
	} else {
#endif	SYSV
		flags = tp->t_flags;
		if (tp->t_ispeed == B134) {
			tpr = DU_TPR_TX_6BITS;
			rpr = DU_RPR_RX_6BITS;
			mr1 = DU_CMR1_PRTY_WITH;
		} else if (flags & (RAW|LITOUT|PASS8)) {
			tpr = DU_TPR_TX_8BITS;
			rpr = DU_RPR_RX_8BITS;
			mr1 = DU_CMR1_PRTY_NONE;
		} else {
			tpr = DU_TPR_TX_7BITS;
			rpr = DU_RPR_RX_7BITS;
			mr1 = DU_CMR1_PRTY_WITH;
		}
		if ((flags & EVENP) && (flags & ODDP)) {
			tpr = DU_TPR_TX_8BITS;
			rpr = DU_RPR_RX_8BITS;
			mr1 = DU_CMR1_PRTY_NONE;
		} else if (flags & EVENP) {
			mr1 |= DU_CMR1_PRTY_EVEN;
		} else {
			mr1 |= DU_CMR1_PRTY_ODD;
		}
		if ((tp->t_ospeed) == B110 || tp->t_state & TS_TWOSB)
			tpr |= DU_TPR_2STOP;
		else {
			tpr |= DU_TPR_1STOP;
		}
#ifdef	SYSV
	}
#endif	SYSV
	addr->du_cmr1 =  mr1 | DU_CMR1_NRZ | DU_CMR1_ASYNC;
#ifdef	DUTRACE
	dutrace(0x20,dev,addr->du_cmr1);
#endif	DUTRACE
	tpr |= DU_TPR_TX_CTS;
	rpr |= DU_RPR_DCD_ENBL_RX;
	addr->du_ier |= DU_IER_DCD_CTS;
	addr->du_tpr = tpr;
	addr->du_rpr = rpr;
	addr->du_rtr = durxspeeds[tp->t_ispeed];
	addr->du_ttr = dutxspeeds[tp->t_ospeed];
	addr->du_ccr = DU_CCR_ENBL_TX;
#ifdef	DUTRACE
	dutrace(0x22, addr->du_ccr, 0);
#endif	DUTRACE
	addr->du_ccr = DU_CCR_ENBL_RX;
#ifdef	DUTRACE
	dutrace(0x21,(addr->du_cmr1<<24)+(addr->du_ier<<16)+(addr->du_tpr<<8)+
		addr->du_rpr,(addr->du_rtr<<24)+(addr->du_ttr<<16)
		+(addr->du_ccr<<8)+addr->du_trsr);
#endif	DUTRACE
}


sio0intr(unit)
register unit;
{
/*
 * The receive interrupt routine.
 */
	register struct dusccchan *addr;
	register unsigned char c;
	register struct tty *tp;
	register unsigned char stat;
	char rdychar;

#ifdef	NOTDEF
#ifdef	HOWFAR
printf("sio0intr %d\n", unit);
#endif	HOWFAR
#endif	NOTDEF
#ifdef	DUTRACE
	dutrace(0,unit,0);
#endif	DUTRACE
	if (unit & 1) {
		rdychar = DU_GSR_B_RXRDY;
	} else {
		rdychar = DU_GSR_A_RXRDY;
	}
	addr = (struct dusccchan *)sio_ttptr[unit].tt_addr;
	tp = sio_ttptr[unit].tt_tty;
	while (addr->du_gsr & rdychar) {
		if (stat = addr->du_rsr & (DU_RSR_BRK_START | DU_RSR_FRM_ERR
					  | DU_RSR_OVERRUN | DU_RSR_PRTY_ERR)) {
			/*
			 * reset the framing error and parity error bits.
			 */
			addr->du_rsr = stat;
		}
		c = addr->du_rxfifo;
#ifdef	NOTDEF
#ifdef	HOWFAR
printf("char %c stat 0x%x\n", c, stat);
#endif	HOWFAR
#endif	NOTDEF
		if (unit == 0 && debug_input(c))
			continue;
		if ((tp->t_state & (TS_ISOPEN|TS_WOPEN)) == 0
#ifdef	GWS
			&& (tp->t_line != KYBDDISC)
#endif	GWS
								)
#ifdef	DUTRACE
		{
			dutrace(0x80,unit,0x80);
#endif	DUTRACE
			return;
#ifdef	DUTRACE
		}
#endif	DUTRACE
		/*
		 * Check for errors
		 */
		if (stat) {
			if (stat & DU_RSR_OVERRUN && sio_overrun++ == 0)
				log(LOG_WARNING, "du%d: line overflow\n", unit);
#ifdef	SYSV
			if (tp->t_universe == UNIVERSE_SYSV) {
			    if (stat & DU_RSR_PRTY_ERR) {
				if (tp->svt_iflag & SV_IGNPAR)
					continue;
				if (tp->svt_iflag & SV_PARMRK)
					;		/* ??? */
			    }
			    if (stat & (DU_RSR_FRM_ERR | DU_RSR_BRK_START)) {
				if (tp->svt_iflag & SV_IGNBRK)
				    continue;
				if (tp->svt_iflag & SV_BRKINT)
				    (*sv_linesw[tp->svt_line].l_input)
						(c, tp, L_BREAK);
				c = 0;
			    }
			   /*
			    * character stripping is done in hardware.
			    */
			} else {
#endif	SYSV
			    if (stat & DU_RSR_PRTY_ERR) 
				if ((tp->t_flags&(EVENP|ODDP))==EVENP || 
				    (tp->t_flags&(EVENP|ODDP))==ODDP )
					    continue;
			    if (stat & (DU_RSR_FRM_ERR | DU_RSR_BRK_START)) 
				if (tp->t_flags&RAW)
				    c = 0;
				else
				    c = tp->t_intrc;
#ifdef	SYSV
			}
#endif	SYSV
		}
#ifdef	SYSV
		if (tp->t_universe == UNIVERSE_SYSV)
			(*sv_linesw[tp->svt_line].l_input)(c, tp, 0);
		else {
#endif	SYSV

#if NBK > 0 || Nbk > 0
			if (tp->t_line == NETLDISC) {
				c &= 0177;
				BKINPUT(c, tp);
			} else
#endif
#ifdef MOUSEDEBUG
				if (unit == 1) {
					printf("[%x]", c);
					if (!c) { /* null mouse char */
						dudumpregs(unit);	
						trap15();
					}
				}
#endif
				(*linesw[tp->t_line].l_rint)(c, tp);

#ifdef	SYSV
		}
#endif	SYSV
	}
#ifdef	DEBUG
printf("sio0intr exit\n");
#endif	DEBUG
#ifdef	DUTRACE
	dutrace(0x80,unit,0);
#endif	DUTRACE
}

sio1intr(unit)
int	unit;
{
/*
 * The transmit interrupt routine. Transmission is done using DMA so this
 * function should not be called.
 */
#ifdef	DUTRACE
	dutrace(1,unit,0);
#endif	DUTRACE
#ifdef	IOCHECK
printf("du1intr");
#endif	IOCHECK
}

sio2intr(unit)
register unit;
{
/*
 * Tx/Rx status interrupt. This interrupt is only enabled if the tty modes
 * are to be changed once the transmitter is empty.
 */
	register struct dusccchan *addr;
	unsigned char tmp;
	extern ttrstrt();

#ifdef	HOWFAR
printf("du2intr %d\n", unit);
#endif	HOWFAR
	addr = (struct dusccchan *)sio_ttptr[unit].tt_addr;
#ifdef	DUTRACE
	dutrace(2,unit,tmp = addr->du_trsr);
	if (tmp & DU_TRSR_TX_EMPTY) {
#else	DUTRACE
	if ((tmp = addr->du_trsr) & DU_TRSR_TX_EMPTY) {
#endif	DUTRACE

		siosetup(unit);
		sioouthold(0, OUT_TTYMODES, unit);
		if (siooutflags[unit] & OUT_TO_BREAK) {
			addr->du_ccr = DU_CCR_TX_ABRT_BRK;
			sioouthold(OUT_BREAK2, OUT_TO_BREAK|OUT_BREAKING,unit);
		}
	}
	if (tmp & DU_TRSR_SND_BRK_ACK) {
		struct tty *tp = sio_ttptr[unit].tt_tty;

		sioouthold(OUT_BREAKING, OUT_TO_BREAK | OUT_BREAK2, unit);
		timeout(ttrstrt, (caddr_t)tp, hz>>2);
		tp->t_state |= TS_TIMEOUT;
#ifdef SYSV
		map_state(tp, TOSV);
#endif SYSV
	}
	/*
	 * reset the bits in the TRSR.
	 */
	addr->du_trsr = tmp &
		       (DU_TRSR_CTS_UNDRN|DU_TRSR_SND_BRK_ACK|DU_TRSR_DPLL_ERR);
#ifdef	DUTRACE
	dutrace(0x82,unit,addr->du_trsr);
#endif	DUTRACE
}

sio3intr(unit)
register unit;
{
/*
 * The external status interrupt routine. Changes in DCD are of interest if
 * not in the CLOCAL mode. Unfortunately the DUSCC chip will also interrupt with
 * changes in CTS.
 */
	register struct dusccchan *addr;
	register struct tty *tp;
	char stat;

	addr = (struct dusccchan *)sio_ttptr[unit].tt_addr;
#ifdef	NOTDEF
#ifdef	HOWFAR
printf("du3intr %d\n", unit);
dudumpregs(unit);
	if (unit) {
		int c;
			if(addr->du_gsr & DU_GSR_B_RXRDY) {
				c = addr->du_rxfifo;
		} else {
			if(addr->du_gsr & DU_GSR_A_RXRDY) {
				c = addr->du_rxfifo;
			}
		}
	}
#endif	HOWFAR
#endif	NOTDEF
	tp = sio_ttptr[unit].tt_tty;
	stat = addr->du_ictsr;
#ifdef	DUTRACE
	dutrace(3, unit, stat);
#endif	DUTRACE
	addr->du_ictsr = stat & (DU_ICTS_CT_ZERO | DU_ICTS_DLTA_DCD
				| DU_ICTS_DLTA_CTS_LC);
#ifdef	MDMCNTRL
printf("du3intr: stat 0x%x t_state 0x%x\n", stat, tp->t_state);
#endif	MDMCNTRL
	if ((tp->t_state & (TS_ISOPEN|TS_WOPEN)) == 0) {
#ifdef	DUTRACE
		dutrace(0x83, unit+0x80, addr->du_ictsr);
#endif	DUTRACE
		return;
	}
	if (stat & DU_ICTS_DCD) {
		if ((tp->t_state & TS_CARR_ON) == 0)
#ifdef	SYSV
		    if (tp->t_universe == UNIVERSE_SYSV)
			(void)(*sv_linesw[tp->svt_line].l_mdmint)(tp, 1);
		    else
#endif	SYSV
			(void)(*linesw[tp->t_line].l_modem)(tp, 1);
    	} else if (tp->t_state&TS_CARR_ON) {
#ifdef	SYSV
		if (tp->t_universe == UNIVERSE_SYSV) {
		    if ((*sv_linesw[tp->svt_line].l_mdmint)(tp, 0) == 0)
			siomdm(unit, 0);
		} else
#endif	SYSV
		if ((*linesw[tp->t_line].l_modem)(tp, 0) == 0)
			siomdm(unit, 0);
    	}
#ifdef	DUTRACE
	dutrace(0x83, unit, stat);
#endif	DUTRACE
}

sio4intr(unit)
int unit;
{
/*
 * This is the receive DMA interrupt.
 */
#ifdef	IOCHECK
printf("du4intr");
#endif	IOCHECK
#ifdef	DUTRACE
	dutrace(4, unit, 0);
#endif	DUTRACE
}

sio5intr(unit)
register unit;
{
/*
 * This is the transmit DMA interrupt.
 */
	register struct tty *tp;
	u_short cntr, stat;
	register struct tdduscc *tdptr = devtodma(unit);

#ifdef	HOWFAR
printf("du5intr %d\n", unit);
#endif	HOWFAR
	tp = sio_ttptr[unit].tt_tty;
	tp->t_state &= ~TS_BUSY;

	stat = tdptr->du_txstat;
#ifdef	DUTRACE
	dutrace(5, unit, stat);
#endif	DUTRACE
	/*
	 * reset the COC and ERR bits.
	 */
	if (tp->t_state & TS_ISOPEN) {
		tdptr->du_txstat = OP_DU_COC | OP_DU_ERR | OP_DU_INEA;
	} else {
		tdptr->du_txstat = OP_DU_COC | OP_DU_ERR;
#ifdef	HOWFAR
printf("du5intr not open\n");
#endif	HOWFAR
		return;
	}
	if(stat & OP_DU_ERR) {
		printf("du%d: DMA ERROR\n", unit);
	}
	tp->t_state &= ~TS_BUSY;
#ifdef	SYSV
	map_state(tp, TOSV);
#endif	SYSV
	if (tp->t_state&TS_FLUSH)
		tp->t_state &= ~TS_FLUSH;
	else {
		cntr = sio_nch[unit] - tdptr->du_txchn.dl_cnt;
		if(cntr)
			ndflush(&tp->t_outq, (int)cntr);
	}
#ifdef	SYSV
	map_state(tp, TOSV);
	if (tp->t_universe == UNIVERSE_SYSV) {
		if ((*sv_linesw[tp->svt_line].l_output)(tp))
			siostart(tp);
	} else {
#endif	SYSV
		if (tp->t_line)
			(*linesw[tp->t_line].l_start)(tp);
		else
			siostart(tp);
#ifdef	SYSV
	}
#endif	SYSV
#ifdef	DUTRACE
	dutrace(0x85, unit, stat);
#endif	DUTRACE
}

			/* on => assert RTS			*/
siomdm(unit, on)		/* returns true if DCD is true for unit	*/
{
	register struct dusccchan *addr;
	char stat;

#ifdef	DUTRACE
	dutrace(7, unit, on);
#endif	DUTRACE
	addr = (struct dusccchan *)sio_ttptr[unit].tt_addr;
	if(on) {
		addr->du_omr |= (DU_OMR_GPOUT1 | DU_OMR_GPOUT2 | DU_OMR_RTS);
	} else {
		addr->du_omr &= ~(DU_OMR_GPOUT1 | DU_OMR_GPOUT2 | DU_OMR_RTS);
	}
	stat = addr->du_ictsr;
#ifdef	DUTRACE
	dutrace(0x87, unit, stat);
#endif	DUTRACE
	return (stat & DU_ICTS_DCD);
}

siostart(tp)
register struct tty *tp;
{
	register struct dusccchan *addr = tp->t_addr;
	register unit = minor(tp->t_dev);
	register struct tdduscc *tdptr = devtodma(unit);
	register int c;
	int s, nch;
	long car;

#ifdef	DUTRACE
	dutrace(0x8, tp->t_dev, tp);
#endif	DUTRACE
	s = spltty();
	if (tp->t_state & (TS_TIMEOUT|TS_BUSY|TS_TTSTOP))
		goto out;
	if (tp->t_outq.c_cc <= TTLOWAT(tp)) {
		if (tp->t_state&TS_ASLEEP) {
			tp->t_state &= ~TS_ASLEEP;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			wakeup((caddr_t)&tp->t_outq);
		}
		if (tp->t_wsel) {
			selwakeup(tp->t_wsel, tp->t_state & TS_WCOLL);
			tp->t_wsel = 0;
			tp->t_state &= ~TS_WCOLL;
		}
	}
/* code from here to end borrowed from dh.c, since this is more like a dh */
	/*
	 * Now restart transmission unless the output queue is empty.
	 */
	if (tp->t_outq.c_cc == 0)
		goto out;
	if (tp->t_flags & (RAW|LITOUT))
		nch = ndqb(&tp->t_outq, 0);
	else {
		nch = ndqb(&tp->t_outq, 0200);
		/*
		 * If first thing on queue is a delay process it.
		 */
		if (nch == 0) {
			nch = getc(&tp->t_outq);
			timeout(ttrstrt, (caddr_t)tp, (nch&0x7f)+6);
			tp->t_state |= TS_TIMEOUT;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			goto out;
		}
	}
	/*
	 * If characters to transmit, restart transmission.
	 */
	if (nch) {
		tp->t_state |= TS_BUSY;
#ifdef	SYSV
		map_state(tp, TOSV);
#endif	SYSV
		car = (int)sio_buf[unit];
		sio_nch[unit] = nch;
		bcopy(tp->t_outq.c_cf, car, nch);
		tdptr->du_txchn.dl_cnt = nch;
		/* got to get the physical address of sio_buf[unit] */
		tdptr->du_txchn.dl_cptr = (char *)svtop(car);
		tdptr->du_txstat = OP_DU_ACT | OP_DU_INEA;
		addr->du_trsr = DU_TRSR_TX_EMPTY;
	}
out:
	splx(s);
#ifdef	DUTRACE
	dutrace(0x88, tp->t_dev, tp);
#endif	DUTRACE
}

/*
 * Stop output on a line, e.g. for ^S/^Q or output flush.
 */
siostop(tp, flag)
        register struct tty *tp;
{
        register int unit = minor(tp->t_dev);
        register struct tdduscc *tdptr = devtodma(unit);
        register struct dusccchan *addr = tp->t_addr;
        register int s;
        register int cntr;

        /*
         * Block input/output interrupts while messing with state.
         */

        s = spltty();
#ifdef  DUTRACE
        dutrace(0x40, flag, tp->t_state);
#endif  DUTRACE
        if (tp->t_state & TS_BUSY) {
                /*
                 * Device is transmitting; stop output
                 */
                if ((tp->t_state&TS_TTSTOP)==0) {
                        tp->t_state |= TS_FLUSH;
#ifdef  SYSV
                        map_state(tp, TOSV);
#endif  SYSV
#ifdef  DUTRACE
                        dutrace(0x41, tp, tp->t_state);
#endif  DUTRACE
                }
                /* we turning off ACT and (for alpha) wait for it to    */
                /* go away, so that we know tha dma ctlr is in a known  */
                /* state, then we drop the chars we did print (or would */
                /* have, in the case we are stopping for a flush)       */
                tdptr->du_txstat = OP_DU_INEA;
                while(tdptr->du_txstat&OP_DU_ACT) {
                        DELAY(100);
                }
                /* if stopping to flush the output queue                */
                if (tp->t_state&TS_FLUSH) {
#ifdef  DUTRACE
                        dutrace(0x60, tp->t_state, 0);
#endif  DUTRACE
                        /* drop all chars we had dma set up to print    */
                        ndflush(&tp->t_outq, sio_nch[unit]);
                } else {
                        cntr = sio_nch[unit] - tdptr->du_txchn.dl_cnt;
                        if(cntr)
                                ndflush(&tp->t_outq, (int)cntr);
#ifdef  DUTRACE
                                dutrace(0x61, cntr, tdptr->du_txchn.dl_cnt);
#endif  DUTRACE
                }
                sio_nch[unit] = 0;
                tdptr->du_txchn.dl_cnt = 0;
                tdptr->du_txchn.dl_cptr = (char *)0;
                tdptr->du_txstat = OP_DU_ACT | OP_DU_INEA;
                addr->du_trsr = DU_TRSR_TX_EMPTY;
        }
        splx(s);
}

dev_t sioconsole = 0;	/* minor device number of the console */
char sioconsflag = 1;


sioputchar(c)
char c;
{
	register struct dusccchan *addr ;
	register int s, i;
	register struct tdduscc *tdptr;
	char rdychar;
	char dmastat;
	char tpr;

	sioinit();
	if (sioconsole == 0) {
		rdychar = DU_GSR_A_TXRDY;
	} else {
		rdychar = DU_GSR_B_TXRDY;
	}
	tdptr = devtodma(sioconsole);
	dmastat = tdptr->du_txstat;
	s = spl7();
#ifdef	DUTRACE
	dutrace(0x9, 0, 0);
#endif	DUTRACE
	/*
	 * reset the ACT bit and wait for it to read as reset.
	 */
	tdptr->du_txstat = dmastat & OP_DU_INEA;
	while (tdptr->du_txstat & OP_DU_ACT) {
		for (i = 10; --i > 0; );
	}
	addr = (struct dusccchan *)sio_ttptr[sioconsole].tt_addr;
	/*
	 * set the DTR, DSR and RTS bits.
	 */
	addr->du_omr |= DU_OMR_GPOUT1 | DU_OMR_GPOUT2 | DU_OMR_RTS;
	/* if not ready for a char, but tx empty, start transmitter	*/
	if( (addr->du_trsr&DU_TRSR_TX_EMPTY) && !(addr->du_gsr&rdychar)){
		addr->du_ccr = DU_CCR_ENBL_TX;
#ifdef	DUTRACE
		dutrace(0x30, addr, c);
#endif	DUTRACE
	}
	if (c == '\n')
		sioputchar('\r');
	i = 100000;
	if ((tpr = addr->du_tpr) & DU_TPR_TX_CTS) {
		addr->du_tpr = tpr & ~DU_TPR_TX_CTS;
	}
	while ((addr->du_gsr & rdychar) == 0 && --i) ;
	for (i=1000;i>0;i--) /* DELAY */
		;
#ifdef	DUTRACE
	if ((addr->du_gsr & rdychar) == 0)
		dutrace(0xf, (addr->du_trsr<<8)+tpr, addr->du_gsr);
#endif	DUTRACE
	addr->du_txfifo = c;
	for (i=1000;i>0;i--) /* DELAY */
		;

	/*
	 * return the status register to its previous value.
	 */
	tdptr->du_txstat = dmastat & (OP_DU_ACT | OP_DU_INEA);
	addr->du_tpr = tpr;
	splx(s);
}

siogetchar()
{
	struct dusccchan *addr = (struct dusccchan *)sio_ttptr[0].tt_addr;
	int c = -1;

#ifdef	DUTRACE
	dutrace(0xa, 0, 0);
#endif	DUTRACE
	if (addr->du_gsr & DU_GSR_A_RXRDY) {
		c = addr->du_rxfifo;
		c &= 0177;
	}
	return c;
}

dudumpregs(unit)
int unit;
{
	struct dusccchan *addr = (struct dusccchan *)sio_ttptr[unit].tt_addr;
	union cmr2 {
		u_char	cmr2_ch;
		struct cmr2_b {
		unsigned int	c2_cc:2,
				c2_dti:3,
				c2_fcs:3;
		} cmr2_b;
	}cmr2;
	union pcr {
		u_char	pcr_ch;
		struct pcr_b {
		unsigned int	pcr_x2idc:1,
				pcr_gp2rcs:1,
				pcr_rcs:1,
				pcr_rtxc:2,
				pcr_trxc:3;
		}pcr_b;
	}pcr;
				
	union tpr {
		u_char	tpr_ch;
		struct tpr_b {
		unsigned int	tpr_stopb:4,
				tpr_txrts:1,
				tpr_cts:1,
				tpr_txlen:2;
		}tpr_b;
	}tpr;
				
	union ttr {
		u_char	ttr_ch;
		struct ttr_b {
		unsigned int	ttr_extsrc:1,
				ttr_txClock:3,
				ttr_brg:4;
		}ttr_b;
	}ttr;
				
	union rtr {
		u_char	rtr_ch;
		struct rtr_b {
		unsigned int	rtr_extsrc:1,
				rtr_txClock:3,
				rtr_brg:4;
		}rtr_b;
	}rtr;
				
	union rpr {
		u_char	rpr_ch;
		struct rpr_b {
		unsigned int	rpr_unused:3,
				rpr_RxRTS:1,
				rpr_strip:1,
				rpr_DCDenable:1,
				rpr_rxlen:2;
		}rpr_b;
	}rpr;

	union omr {
		u_char	omr_ch;
		struct omr_b {
		unsigned int	omr_txresid:3,
				omr_TxRDY:1,
				omr_RxRDY:1,
				omr_out2:1,
				omr_out1:1,
				omr_out0:1;
		}omr_b;
	}omr;

	union icr {
		u_char	icr_ch;
		struct icr_b {
		unsigned int	icr_intpri:2,
				icr_vect:2,
				icr_btm:1,
				icr_vis:1,
				icr_Amie:1,
				icr_Bmie:1;
		}icr_b;
	}icr;

	union ccr {
		u_char	ccr_ch;
		struct ccr_b {
		unsigned int	ccr_cmdtyp:2,
				ccr_dontcare:2,
				ccr_cmd:4;
		}ccr_b;
	}ccr;


	u_char cmr1 = addr->du_cmr1;
	cmr2.cmr2_ch = addr->du_cmr2;
	pcr.pcr_ch = addr->du_pcr;
	tpr.tpr_ch = addr->du_tpr;
	ttr.ttr_ch = addr->du_ttr;
	rtr.rtr_ch = addr->du_rtr;
	rpr.rpr_ch = addr->du_rpr;
	omr.omr_ch = addr->du_omr;
	icr.icr_ch = ((struct dusccchan *)sio_ttptr[0].tt_addr)->du_icr;
	ccr.ccr_ch = addr->du_ccr;
	printf("du_cmr1: %s, %s %s %s\n",
		((cmr1&0x80)?((cmr1&0x40)?"FM1 ":"FM0 ")
			   :((cmr1&0x40)?"NRCI ":"MANCHESTER ")),
		((cmr1&DU_CMR1_PRTY_ODD)? "ODD PARITY " : "EVEN PARITY"),
		((cmr1&0x20)? ((cmr1&0x10)?"WITH PARITY":"FORCE PARITY")
			   :"NO PARITY"),
		((cmr1&0x7)==0x7)? "ASYN":"NOT ASYNC!!!!!");
	printf("du_cmr2: ");
	switch(cmr2.cmr2_b.c2_cc) {
	case DU_CMR2_CONN_NORM>>6:
		printf("NORM,");
		break;
	case DU_CMR2_CONN_ECHO>>6:
		printf("ECHO,");
		break;
	case DU_CMR2_CONN_LOOP>>6:
		printf("LOOP,");
		break;
	}
	switch(cmr2.cmr2_b.c2_dti) {
	case DU_CMR2_HALF_S_DMA>>3:
		printf("HALF_S_DMA,");
		break;
	case DU_CMR2_HALF_D_DMA>>3:
		printf("HALF_D_DMA,");
		break;
	case DU_CMR2_FULL_S_DMA>>3:
		printf("FULL_S_DMA,");
		break;
	case DU_CMR2_FULL_D_DMA>>3:
		printf("FULL_D_DMA,");
		break;
	case DU_CMR2_WAIT_RX>>3:
		printf("WAIT_RX,");
		break;
	case DU_CMR2_WAIT_TX>>3:
		printf("WAIT_TX,");
		break;
	case DU_CMR2_WAIT_RX_TX>>3:
		printf("WAIT_RX/TX,");
		break;
	case DU_CMR2_POLLED>>3:
		printf("POLLED,");
		break;
	}
	switch(cmr2.cmr2_b.c2_fcs) {
	case DU_CMR2_NO_FCS:
		printf("NO FCS");
		break;
	case DU_CMR2_LRC8_P0:
		printf("LRC8 0");
		break;
	case DU_CMR2_LRC8_P1:
		printf("LRC8 1");
		break;
	case DU_CMR2_CRC16_P0:
		printf("CRC 16-0");
		break;
	case DU_CMR2_CRC16_P1:
		printf("CRC 16-1");
		break;
	case DU_CMR2_CCITT_P0:
		printf("CRC CCITT-0");
		break;
	case DU_CMR2_CCITT_P1:
		printf("CRC CCITT-1");
		break;
	}
	printf("\n");
	printf("du_pcr: ");

	if(unit == 0) {
		switch(pcr.pcr_b.pcr_x2idc) {
		case DU_PCR_X2_CRYS>>7:
			printf("X2,");
		break;
		case DU_PCR_IDC>>7:
			printf("IDC,");
		break;
		}
	}

	switch(pcr.pcr_b.pcr_gp2rcs){
	case DU_PCR_GPO2>>6:
		printf("GPO2,");
		break;
	case DU_PCR_GP_RTS>>6:
		printf("RTS,");
	}

	switch(pcr.pcr_b.pcr_rcs){
	case DU_PCR_SYNOUT>>5:
		printf("SYNOUT,");
		break;
	case DU_PCR_SY_RTS>>5:
		printf("RTS,");
		break;
	}

	switch(pcr.pcr_b.pcr_rtxc){
	case DU_PCR_RTxC_INPUT>>3:
		printf("INPUT,");
		break;
	case DU_PCR_RTxC_C_T>>3:
		printf("C/T,");
		break;
	case DU_PCR_RTxC_TxCLK>>3:
		printf("TxCLK 1x,");
		break;
	case DU_PCR_RTxC_RxCLK>>3:
		printf("RxCLK 1x,");
		break;
	}

	switch(pcr.pcr_b.pcr_trxc){
	case DU_PCR_TRxC_INPUT:
		printf("INPUT");
		break;
	case DU_PCR_TRxC_XTAL_2:
		printf("XTAL/2");
		break;
	case DU_PCR_TRxC_DPLL:
		printf("DPLL");
		break;
	case DU_PCR_TRxC_C_T:
		printf("C/T");
		break;
	case DU_PCR_TRxC_16TxCLK:
		printf("TxCLK 16x");
		break;
	case DU_PCR_TRxC_16RxCLK:
		printf("RxCLK 16x");
		break;
	case DU_PCR_TRxC_1xTxCLK:
		printf("TxCLK 1x");
		break;
	case DU_PCR_TRxC_1xRxCLK:
		printf("RxCLK 1x");
		break;

	}
	printf("\n");
	printf("du_tpr: ");

	printf("%d/16 stop bits,", tpr.tpr_b.tpr_stopb+9);
	if(tpr.tpr_b.tpr_txrts) {
		printf("Tx RTS,");
	}
	if(tpr.tpr_b.tpr_cts) {
		printf("CTS Enable Tx,");
	}
	printf("Tx %d bit chars", tpr.tpr_b.tpr_txlen+5);
	printf("\n");

	printf("du_ttr: ");
	if(ttr.ttr_b.ttr_extsrc) {
		printf("TRxC,");
	} else {
		printf("RTxC,");
	}
	switch(ttr.ttr_b.ttr_txClock) {
	case DU_TTR_CLK_1EXT>>4:
		printf("1x external,");
		break;
	case DU_TTR_CLK_16EXT>>4:
		printf("16x external,");
		break;
	case DU_TTR_CLK_DPLL>>4:
		printf("DPLL,");
		break;
	case DU_TTR_CLK_BRG>>4:
		printf("BRG,");
		break;
	case DU_TTR_CLK_2xOTH_CT>>4:
		printf("2x other channel C/T,");
		break;
	case DU_TTR_CLK_32xOTH_CT>>4:
		printf("32x other channel C/T,");
		break;
	case DU_TTR_CLK_2xOWN_CT>>4:
		printf("2x own channel C/T,");
		break;
	case DU_TTR_CLK_32xOWN_CT>>4:
		printf("32x own channel C/T,");
		break;
	}
	printf("BRG%x", ttr.ttr_b.ttr_brg);
	printf("\n");

	printf("du_rpr: ");
	if(rpr.rpr_b.rpr_RxRTS) {
		printf("Rx RTS Control,");
	}
	if(rpr.rpr_b.rpr_strip) {
		printf("strip parity,");
	}
	if(rpr.rpr_b.rpr_DCDenable) {
		printf("DCD Enable Rx,");
	}
	printf("Rx %dbit chars", rpr.rpr_b.rpr_rxlen+5);
	printf("\n");

	printf("du_rtr: ");
	if(rtr.rtr_b.rtr_extsrc) {
		printf("TRxC,");
	} else {
		printf("RTxC,");
	}
	switch(rtr.rtr_b.rtr_txClock) {
	case DU_RTR_CLK_1EXT>>4:
		printf("1x external,");
		break;
	case DU_RTR_CLK_16EXT>>4:
		printf("16x external,");
		break;
	case DU_RTR_CLK_BRG>>4:
		printf("BRG,");
		break;
	case DU_RTR_CLK_CT>>4:
		printf("C/T of channel,");
		break;
	case DU_RTR_CLK_D_64CLK>>4:
		printf("DPLL, 64xX1/CLK,");
		break;
	case DU_RTR_CLK_D_32EXT>>4:
		printf("DPLL, 32x external,");
		break;
	case DU_RTR_CLK_D_32BRG>>4:
		printf("DPLL, 32x BRG,");
		break;
	case DU_RTR_CLK_D_32CT>>4:
		printf("DPLL, 32x C/T,");
		break;
	}
	printf("BRG%x", rtr.rtr_b.rtr_brg);
	printf("\n");

	printf("du_omr: ");
	if( omr.omr_b.omr_txresid == 0x7) {
		printf("*%d residual tx bits", tpr.tpr_b.tpr_txlen+5);
	} else {
		printf("%d residual tx bits,", omr.omr_b.omr_txresid);
	}
	if(omr.omr_b.omr_TxRDY) {
		printf("TxRDY on FIFO empty,");
	} else {
		printf("TxRDY on FIFO not full,");
	}
	if(omr.omr_b.omr_RxRDY) {
		printf("RxRDY on FIFO not empty,");
	} else {
		printf("RxRDY on FIFO full,");
	}
	printf("out2 %d,", omr.omr_b.omr_out2);
	printf("out1 %d,", omr.omr_b.omr_out1);
	printf("out0 %d,", omr.omr_b.omr_out0);
	printf("\n");

#define RSR_BITS\
"\20\10CHARCOMP\7RTSNEG\6OVERRUN\4BRKEND\3BRKSTART\2FRAMEERR\1PARITYERR"
	printf("du_rsr: 0x%b\n", addr->du_rsr, RSR_BITS);
#define TRSR_BITS\
"\20\10TxEMPTY\7CTSUNDERRUN\5SENDBRKACK\4DPLLERROR"
	printf("du_trsr: 0x%b\n", addr->du_trsr, TRSR_BITS);
#define ICTSR_BITS\
"\20\10C/TRUN\7C/TZERO\6DELTADCD\5DELTACTS/LC\4DCD\3CTS/LC\2GPI2\1GPI1"
	printf("du_ictsr: 0x%b\n", addr->du_ictsr, ICTSR_BITS);
#define DUIER_BITS\
"\20\10DCD/CTS\7TxRDY\6TRSR[7:3]\5RxRDY\4RSR[7:6]\3RSR[5:4]\2RSR[3:2]\1RSR[1:0]"
	if(unit == 0) {
		printf("du_ivr: 0x%x\n", addr->du_ivr);
	} else {
		printf("du_ivrm: 0x%x\n", addr->du_ivr);
	}
	printf("du_ier: 0x%b\n", addr->du_ier, DUIER_BITS);
#define GSR_BITS\
"\20\10B:EXTorC/T\7B:Rx/Tx\6B:TxRDY\5B:RxRDY\4A:EXTorC/T\3A:Rx/Tx\2A:TxRDY\1A:RxRDY"
	printf("du_gsr: 0x%b\n", addr->du_gsr, GSR_BITS);

	printf("du_icr: ");
	switch(icr.icr_b.icr_intpri) {
	case DU_ICR_PRI_A>>6:
		printf("A,");
		break;
	case DU_ICR_PRI_B>>6:
		printf("B,");
		break;
	case DU_ICR_INTLV_A>>6:
		printf("intrlvd A,");
		break;
	case DU_ICR_INTLV_B>>6:
		printf("intrlvd B,");
		break;
	}

	switch(icr.icr_b.icr_vect) {
	case DU_ICR_VCTR0>>4:
		printf("VECT00,");
		break;
	case DU_ICR_VCTR1>>4:
		printf("VECT01,");
		break;
	case DU_ICR_VCTR2>>4:
		printf("VECT10,");
		break;
	case DU_ICR_NOT_VCTRD>>4:
		printf("NON VECTORED,");
		break;
	}
	if(icr.icr_b.icr_btm) {
		printf("modify 4:2,");
	} else {
		printf("modify 2:0,");
	}
	if(icr.icr_b.icr_vis) {
		printf("vector incl. stat,");
	} else {
		printf("vector no stat,");
	}
	if(icr.icr_b.icr_Amie) {
		printf("A enabled,");
	} else {
		printf("A disabled,");
	}
	if(icr.icr_b.icr_Bmie) {
		printf("B enabled");
	} else {
		printf("B disabled");
	}
	printf("\n");

	printf("du_ccr: ");
	switch(ccr.ccr_b.ccr_cmdtyp) {
	case 0:
		switch(ccr.ccr_b.ccr_cmd) {
		case DU_CCR_RST_TX&0xf:
			printf("reset Tx");
			break;
		case DU_CCR_RST_TX_CRC&0xf:
			printf("reset TxCRC");
			break;
		case DU_CCR_ENBL_TX&0xf:
			printf("enable Tx");
			break;
		case DU_CCR_DISABL_TX&0xf:
			printf("disable Tx");
			break;
		case DU_CCR_TX_SOM&0xf:
			printf("xmit SOM");
			break;
		case DU_CCR_TX_SOM_PAD&0xf:
			printf("xmit SOM with PAD");
			break;
		case DU_CCR_TX_EOM&0xf:
			printf("xmit EOM");
			break;
		case DU_CCR_TX_ABRT_BRK&0xf:
			printf("xmit ABORT/BREAK");
			break;
		case DU_CCR_TX_DLE&0xf:
			printf("xmit DLE");
			break;
		case DU_CCR_ACTV_ON_POLL&0xf:
			printf("go active on poll");
			break;
		case DU_CCR_RST_ACTV_POLL&0xf:
			printf("reset go active on poll");
			break;
		case DU_CCR_ON_LOOP&0xf:
			printf("go on-loop");
			break;
		case DU_CCR_OFF_LOOP&0xf:
			printf("go off-loop");
			break;
		case DU_CCR_EXCLD_CRC&0xf:
			printf("exclude from CRC");
			break;
		}
		break;
	case 1:

		switch(ccr.ccr_b.ccr_cmd) {
		case DU_CCR_RST_RX&0xf:
			printf("reset Rx");
			break;
		case DU_CCR_ENBL_RX&0xf:
			printf("enable Rx");
			break;
		case DU_CCR_DISABL_RX&0xf:
			printf("disable Rx");
			break;
		}
		break;

	case 2:
		switch(ccr.ccr_b.ccr_cmd) {
		case DU_CCR_CT_START&0xf:
			printf("start");
			break;
		case DU_CCR_CT_STOP&0xf:
			printf("stop");
			break;
		case DU_CCR_CT_FFFF&0xf:
			printf("preset to FFFF");
			break;
		case DU_CCR_CT_PRST&0xf:
			printf("preset from CTPRH/CTPRL");
			break;
		}
		break;

	case 3:
		switch(ccr.ccr_b.ccr_cmd) {
		case DU_CCR_DPLL_SEARCH&0xf:
			printf("DPLL search");
			break;
		}
		break;
	}
	printf("\n");
}

#ifdef	DUTRACE
#define DUTBSIZE 2048
struct dutb {
	unsigned long where;
	unsigned long what;
	unsigned long what2;
	unsigned long count;
} dutbuf[DUTBSIZE];

struct dutb *dutp = dutbuf;

dutrace(l1,l2,l3)	/* trace into memory buffer */
unsigned long l1,l2,l3;
{

	if(dutp->where == l1 && dutp->what == l2 && dutp->what2 == l3) {
		dutp->count++;
		return;
	}
	if(dutp == &dutbuf[DUTBSIZE-1]) {
		dutp = dutbuf;
	} else {
		dutp++;
	}
	dutp->where=l1;
	dutp->what = l2;
	dutp->what2 = l3;
	dutp->count = 1;
}
#endif	DUTRACE
