/*
 * VMEBUS: Integrated Solutions VME-SCSI disk (SD controller, sd disk) driver
 */
#include "sd.h"

#if	NSD > 0
#include "../h/types.h"

#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "dk.h"
#include "dkbad.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/sdreg.h"
#include "../machine/board.h"

#define Nsd_SD		4	/* max number of sd drives on SD controller */
#define SD_MAXCNT	(63*1024)
#define	SD_TIMEOUT	hz*5

/* 
 * State of controller from last transfer. Since only one transfer can be done 
 * at a time per controller, only allocate one for each controller.
 */
struct	SD_stat {
	short		sd_ndrive;	/* Number of drives on controller */
	short		sd_wticks;	/* Monitor time for function */
	u_short		sd_cmd;		/* command to issue */
	u_int		sd_bn;		/* block number */
	u_int		sd_addr;	/* physical address of transfer */
	int		sd_bleft;	/* bytes left to transfer */
	int		sd_bpart;	/* bytes transferred */
} 			SD_stat[NSD];

int	sdprobe(),	sdslave(),	sdattach(),	sdintr();
int	sdstrat(),	sddgo();	sdminphys();
struct	qb_ctlr		*sdcinfo[NSD];
struct	qb_device	*sddinfo[Nsd];
int                     sdzero, sdzero_vdma;

/* 
 * SD controller addresses --- see /sys/conf/devaddr.c
 */

extern u_short *SDstd[];

/*
 * SD/sd controller/device driver structure
 */
struct	qb_driver SDdriver =
	{sdprobe, sdslave, sdattach, sddgo, SDstd, "sd", sddinfo, "SD",sdcinfo};

/* 
 * sd drive structure: each drive can have a different geometry.
 */
struct qb_diskst sdst[Nsd];

struct	buf	rsdbuf[Nsd];
struct	dkbad	sdbad[Nsd];
struct	buf	sdutab[Nsd];

#define	b_bn		b_resid	
#define	sdunit(bp)	(minor((bp)->b_dev) >> 3)

int	sdwstart, sdwatch();		/* Have started guardian */

/* 
 * Check that controller exists
 */
sdprobe(sdaddr)
register struct sddevice	*sdaddr;
{
	if (sdwait(sdaddr))
		return (0);
	sdaddr->sdcs = SD_IE | SD_SIZE | SD_CBSY;
	/* return 0 if timeout, or if it is an ld or gs controller */
	if (sdwait(sdaddr) || (sdaddr->sdcs == (SD_IE | SD_SIZE | SD_CBSY)))
		return (0);
	DELAY(8000);
	if (sdzero_vdma == 0)
                sdzero_vdma = IOPB_STD(iopballoc(&sdzero, sizeof(int)),vbnum);
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (sizeof (struct sddevice));
}

/*
 * Check that a given drive exists
 */
sdslave(qi, sdaddr)
register struct qb_device	*qi;
register struct sddevice	*sdaddr;
{
	sdaddr->sdcs = (qi->qi_slave << 8) | SD_SIZE | SD_CBSY;
	if (sdwait(sdaddr) || (sdaddr->sdcs & SD_ERR) || (sdaddr->sdda == 0))
		return (0);
	return (1);
}

/*
 * Logically attach a unit to a drive/controller, and set up unit info
 */
sdattach(qi)
register struct qb_device	*qi;
{
	register struct sddevice *sdaddr;
	register struct qb_diskst *st;

	/*
	 * Start watchdog for lost interupts from drive
	 */
	if (sdwstart == 0) {
		timeout(sdwatch, (caddr_t)0, SD_TIMEOUT);
		sdwstart++;
	}

	SD_stat[qi->qi_ctlr].sd_ndrive++;
	sdaddr = (struct sddevice *)qi->qi_mi->qm_addr;
	st = &sdst[qi->qi_unit];

	sdaddr->sdcs = qi->qi_slave<<8 | SD_SIZE | SD_CBSY;
	if (sdwait(sdaddr))
		printf("	TIMEOUT ");
	else {
		st->st_ntpc = sdaddr->sdda;	/* # of tracks (heads) */
		st->st_ncpd = sdaddr->sdae;	/* # of cylinders */
		st->st_nspt = sdaddr->sdwc;	/* # of sectors per track */
		printf("	%mM (%d x %d x %d) ",
		    st->st_nspt * st->st_ntpc * st->st_ncpd * 512,
 		    st->st_nspt, st->st_ntpc, st->st_ncpd);
		diskpart(st->st_size, st->st_nspt, st->st_ntpc, st->st_ncpd);
		st->st_nspc = st->st_nspt * st->st_ntpc;
	}

	/* Initialize iostat values */
	if (qi->qi_dk >= 0)
		dk_bps[qi->qi_dk] = (3600/60)*512*st->st_nspt;
}

sdopen(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *qi;

	if (unit >= Nsd || (qi = sddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	return (0);
}

sdstrategy(bp)
	register struct buf *bp;
{
	register struct qb_diskst *st;
	register int unit = sdunit(bp);
	register struct qb_device *qi = sddinfo[unit];
	int part = minor(bp->b_dev) & 07;

	if (unit >= Nsd || qi == 0 || qi->qi_alive == 0) {
		bp->b_error = ENXIO;
		goto bad;
	}
	st = &sdst[qi->qi_unit];
	if (bp->b_blkno < 0 || 
	    (bp->b_blkno+((bp->b_bcount+511)>>9)) > st->st_size[part].nblocks) {
		if (bp->b_blkno == st->st_size[part].nblocks) {
		    bp->b_resid = bp->b_bcount;
		    goto done;
		}
		bp->b_error = EINVAL;
		goto bad;
	}
	badstrat(qi, bp, &sdbad[qi->qi_unit], sdstrat,
	    st->st_nspt, st->st_ntpc, st->st_ncpd, st->st_size[part].cyloff);
	return;
bad:	bp->b_flags |= B_ERROR;
done:	iodone(bp);
}

sdstrat(qi, bp)
	register struct qb_device *qi;
	register struct buf *bp;
{
	register struct qb_diskst *st = &sdst[qi->qi_unit];
	int part = minor(bp->b_dev) & 07;
	int s;
        register struct buf *dp = &sdutab[qi->qi_unit];
        register struct qb_ctlr *qm = qi->qi_mi;



	bp->b_bn = bp->b_blkno + (st->st_size[part].cyloff * st->st_nspc);
	s = splx(qm->qm_psl);
	disksort(dp, bp);
        if (dp->b_active == 0) {
                dp->b_forw = NULL;
                dp->b_active = 1;
                if (qm->qm_tab.b_actf == NULL)
                        qm->qm_tab.b_actf = dp;
                else
                        qm->qm_tab.b_actl->b_forw = dp;
                qm->qm_tab.b_actl = dp;
                if (qm->qm_tab.b_active == 0)
                        sdstart(qm);
        }
	splx(s);
}

/*
 * Start up a transfer on a drive.
 */
sdstart(qm)
	register struct qb_ctlr *qm;
{
	register struct buf *bp;
	register struct qb_device *qi;
	register struct sddevice *sdaddr;
	register struct SD_stat *stat;
	register struct qb_diskst *st;

	if ((bp = qm->qm_tab.b_actf->b_actf) == NULL)
		return;

	/*
	 * Mark controller busy, and determine destination.
	 */
	qm->qm_tab.b_active++;
	qi = sddinfo[sdunit(bp)];		/* Controller */
	stat = &SD_stat[qm->qm_ctlr];
	st = &sdst[qi->qi_unit];
	sdaddr = (struct sddevice *)qm->qm_addr;

	/* 
	 * save away current data transfer drive info
	 */
	stat->sd_bn = bp->b_bn;
	stat->sd_bleft = bp->b_bcount;
	if (bp->b_flags & B_READ)
		stat->sd_cmd = SD_IE | SD_CBSY | SD_READ | (qi->qi_slave<<8);
	else
		stat->sd_cmd = SD_IE | SD_CBSY | SD_WRITE | (qi->qi_slave<<8);
	qbgo(qi);
}

sddgo(qm)
register struct qb_ctlr *qm;
{

	SD_stat[qm->qm_ctlr].sd_addr = VDMA_STD(qm->qm_qbinfo,vbnum);
	sdio((struct sddevice *)qm->qm_addr, &SD_stat[qm->qm_ctlr]);
}


sdio(sdaddr, stat)
register struct sddevice *sdaddr;
register struct SD_stat	*stat;
{
	stat->sd_bpart = (stat->sd_bleft>SD_MAXCNT)? SD_MAXCNT: stat->sd_bleft;
	sdaddr->sdda = stat->sd_bn;
	sdaddr->sdba = stat->sd_addr;
	sdaddr->sdae = ((stat->sd_bn>>8)&0xFF00) | ((stat->sd_addr>>16)&0xFF);
	sdaddr->sdwc = stat->sd_bpart >> 1;
	sdaddr->sdcs = stat->sd_cmd;
}

/*
 * Handle a disk interrupt.
 */
sdintr(ctlr)
	register ctlr;
{
	register struct buf *bp, *dp;
	register struct qb_ctlr *qm = sdcinfo[ctlr];
	register struct qb_device *qi;
	register struct sddevice *sdaddr = (struct sddevice *)qm->qm_addr;
	register unit;
	struct SD_stat *stat = &SD_stat[ctlr];
	register struct qb_diskst *st;
	register u_short cs;

	stat->sd_wticks = 0;
        dp = qm->qm_tab.b_actf;
        bp = dp->b_actf;
	qi = sddinfo[sdunit(bp)];
	st = &sdst[qi->qi_unit];
	dk_busy &= ~(1 << qi->qi_dk);
	/*
	 * Check for and process errors on either the drive or the controller.
	 */
	if ((cs = sdaddr->sdcs) & SD_ERR) {
/*printf("ERROR cs=%b\n", sdaddr->sdcs, SDCS_BITS); /**/
		/*
		 * After 10 retries give up.
		 */
		if (++qm->qm_tab.b_errcnt > 10) {
			harderr(bp, "sd");
			printf("cs=%b sw=%x %x %x\n", cs, SDCS_BITS,
			    sdaddr->sdsw[0],sdaddr->sdsw[1],sdaddr->sdsw[2]);
			bp->b_flags |= B_ERROR;
		} else {
			sdrezero(sdaddr, qi->qi_slave);	/* rezero unit and */
			qm->qm_tab.b_active = 0;	/* force retry */
		}
	}
	/*
	 * If still ``active'', then don't need any more retries.
	 */
	if (qm->qm_tab.b_active) {
		/* 
		 * check if more data from previous request
		 */
		if ((bp->b_flags & B_ERROR) == 0 &&
		     (stat->sd_bleft -= stat->sd_bpart) > 0) {
			stat->sd_addr += stat->sd_bpart;
			stat->sd_bn += (stat->sd_bpart/512);
			sdio(sdaddr, stat);
			return;
		}
		dk_busy &= ~(1 << qi->qi_dk);
		bp->b_resid = stat->sd_bleft;
		qm->qm_tab.b_active = qm->qm_tab.b_errcnt = 0;
                qm->qm_tab.b_actf = dp->b_forw;
                dp->b_active = 0;
                dp->b_actf = bp->av_forw;
                if (dp->b_actf) {
                        dp->b_forw = NULL;
                        dp->b_active = 1;
                        if (qm->qm_tab.b_actf == NULL)
                                qm->qm_tab.b_actf = dp;
                        else
                                qm->qm_tab.b_actl->b_forw = dp;
                        qm->qm_tab.b_actl = dp;
                }
		iodone(bp);
	}

	qbdone(qm);

	/*
	 * If controller not transferring and have request, start controller.
	 */
	if (qm->qm_tab.b_actf && qm->qm_tab.b_active == 0)
		sdstart(qm);
}

sdread(dev, uio)
dev_t	dev;
struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nsd)
		return (ENXIO);
	return (physio(sdstrategy, &rsdbuf[unit], dev, B_READ, sdminphys, uio));
}

sdwrite(dev, uio)
dev_t dev;
struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nsd)
		return (ENXIO);
	return (physio(sdstrategy, &rsdbuf[unit], dev, B_WRITE, sdminphys, uio));
}

#include	"ioctl.h"

sdioctl(dev, cmd, data)
dev_t             dev;
int               cmd;
struct qb_diskst *data;
{
	register int unit = minor(dev) >> 3;

	if (unit > Nsd)
		return(ENXIO);
	if (cmd != DIOCGGEO)
		return(EIO);
	*data = sdst[unit];
	return(0);
}

/*
 * Reset driver after lost interupt. Cancsd software state of all pending 
 * transfers and restart all units and the controller.
 */
sdreset()
{
	register struct qb_ctlr *qm;
	register struct sddevice *sdaddr;
	register struct SD_stat *stat;
	register int ctlr;

	for (ctlr = 0; ctlr < NSD; ctlr++) {
		qm = sdcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		printf(" SD%d", ctlr);
		sdaddr = (struct sddevice *)qm->qm_addr;
		qm->qm_tab.b_active = 0;
		stat = &SD_stat[ctlr];
		stat->sd_bleft = 0;
		stat->sd_bpart = 0;
		sdwait(sdaddr);
		sdstart(qm);
	}
}

/*
 * Wake up every second and if an interrupt is pending but nothing has happened
 * increment a counter. If nothing happens for 20 seconds, reset the controller
 * and begin anew.
 */
sdwatch()
{
	register struct qb_ctlr *qm;
	register ctlr, unit;
	register struct SD_stat *stat;

	timeout(sdwatch, (caddr_t)0, SD_TIMEOUT);
	for (ctlr = 0; ctlr < NSD; ctlr++) {
		qm = sdcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		stat = &SD_stat[ctlr];
		stat->sd_wticks++;
		if (stat->sd_wticks >= 20) {
			stat->sd_wticks = 0;
			printf("SD%d: lost interrupt\n", ctlr);
			sdreset();
		}
	}
}

sdrezero(sdaddr, drive)
register struct sddevice *sdaddr;
{
	sdaddr->sdda = 0;	/* no rezero command, read block 0 instead */
	sdaddr->sdba = sdzero_vdma;
	sdaddr->sdae = 0;
	sdaddr->sdwc = 1;
	sdaddr->sdcs = SD_CBSY | SD_READ | (drive << 8);
	sdwait(sdaddr);
}

int sddumpstrat();
extern struct buf dumpbuf;
extern int dumpsize;

sddump(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register int part = minor(dev) & 07;
	register struct buf *bp = &dumpbuf;
	register struct qb_device *qi;
	register struct qb_diskst *st;
	int num, start, bcount, vdma;

	if (unit >= Nsd || (qi = sddinfo[unit]) == 0 || qi->qi_alive == 0)
		return ENXIO;
	num = ctod(dumpsize);		/* memory size in disk blocks */
	start = 0x0;			/* start dumping at physical 0 */
	st = &sdst[unit];
	if (dumplo < 0 || dumplo+num > st->st_size[part].nblocks)
		return EINVAL;
	while (num > 0) {
		bcount = (dbtob(num) > SD_MAXCNT) ? SD_MAXCNT : dbtob(num);
		bp->b_dev = dev;
		bp->b_blkno = dumplo + btodb(start);
		bp->b_bcount = bcount;
		vdma = qbdumpsetup((caddr_t)start, bcount);
		bp->b_un.b_addr = (caddr_t)(VDMA_STD(vdma,vbnum));
		bp->b_flags = B_WRITE; 
		badstrat(qi, bp, &sdbad[qi->qi_unit], sddumpstrat,
		    st->st_nspt, st->st_ntpc, st->st_ncpd, 
		    st->st_size[part].cyloff);
		qbrelse(&vdma);
		if (bp->b_flags & B_ERROR || bp->b_resid)
			return EIO;
		start += bcount;
		num -= btodb(bcount);
	}
	return 0;
}

sddumpstrat(qi, bp)
	register struct qb_device *qi;
	register struct buf *bp;
{
	register struct SD_stat	*stat = &SD_stat[qi->qi_ctlr];
	register struct sddevice *sdaddr=(struct sddevice *)qi->qi_mi->qm_addr;
	register struct qb_diskst *st = &sdst[qi->qi_unit];
	register int errcount;

	errcount = 0;
	while (1) {
		stat->sd_bn = bp->b_blkno + 
		    (st->st_size[minor(bp->b_dev)&07].cyloff*st->st_nspc);
		stat->sd_addr = (u_int) bp->b_un.b_addr;
		stat->sd_bleft = bp->b_bcount;
		if (bp->b_flags & B_READ)
			stat->sd_cmd = SD_CBSY | SD_READ | (qi->qi_slave<<8);
		else
			stat->sd_cmd = SD_CBSY | SD_WRITE | (qi->qi_slave<<8);
		sdio(sdaddr,stat);
		if (sdwait(sdaddr)) {
			printf("sd%d%c: HUNG sn%d ", minor(bp->b_dev) >> 3, 
				'a'+(minor(bp->b_dev)&07), bp->b_blkno);
			printf("cs=%b sw=%x %x %x: ", sdaddr->sdcs,
				SDCS_BITS, sdaddr->sdsw[0], 
				sdaddr->sdsw[1], sdaddr->sdsw[2]);
			bp->b_flags |= B_ERROR;
			bp->b_resid = bp->b_bcount;
			return EIO;
		} else if (sdaddr->sdcs & SD_ERR) {
			if (++errcount > 10) {
				harderr(bp, "sd");
				printf("cs=%b sw=%x %x %x: ", sdaddr->sdcs, 
					SDCS_BITS, sdaddr->sdsw[0],
					sdaddr->sdsw[1],sdaddr->sdsw[2]);
				bp->b_flags |= B_ERROR;
				bp->b_resid = bp->b_bcount;
				return EIO;
			}
		} else {
			bp->b_resid = 0;
			break;
		}
	}
	bp->b_flags |= B_DONE;
	return 0;
}

sdsize(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *qi;
	register struct qb_diskst *st;

	if (unit >= Nsd || (qi = sddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	st = &sdst[qi->qi_unit];
	return (st->st_size[minor(dev)&0x7].nblocks);
}

static 
sdwait(sdaddr)
	register struct sddevice *sdaddr;
{
	register int l = 0;
	
	DELAY(400);
	while (sdaddr->sdcs == 0xffff)  {
		if (++l == 50000) {
			return (-1);
		}
		DELAY(400);
	}
	return (0);
}

sdminphys(bp)
	struct buf *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, SD_MAXCNT);
}
#endif	NSD > 0
