#define	DEBUG
/*
 * Unix driver for the Ciprico Rimfire 3x00 disk controllers
 */
#include "rf.h"

#if NRF > 0
#include "../machine/pte.h"
#include "../machine/board.h"

#include "param.h"
#include "systm.h"
#include "dk.h"
#include "dkbad.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"
#include "syslog.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/rfreg.h"

#define	UTOCN(x)	((x) >> 3)
#define	UTODR(x)	((x) & 7)
#define	UTOVOL(x)	((x) & 1)
#define RF_MAXCNT	MAX_RF_BSIZE
#ifdef	M68020
#define	RF_XFER_WID	((((*BSR)&BSR_MEMWIDTH)==BSR_16BITMEM)?0:ABP_CTLAM_WID)
#ifdef	M68025
#define	RF_ADDRMOD	(PB_ADDRMOD_EXT)
#else	M68025
#define	RF_ADDRMOD	(PB_ADDRMOD_STD)
#endif	M68025
#else	M68020
#define	RF_XFER_WID	(0)
#define	RF_ADDRMOD	(PB_ADDRMOD_STD)
#endif	M68020

int	rfprobe(), rfslave(), rfattach(), rfintr();
int 	rfstrat(), rfminphys();
static struct qb_ctlr	*rfcinfo[NRF];
static struct qb_device	*rfdinfo[Nrf];

u_short *RFstd[] = {	(u_short *)&vme_short[0x1000],
			(u_short *)&vme_short[0x1200],
		 	0};

struct qb_driver RFdriver = 
	{ rfprobe, rfslave, rfattach, RFstd, "rf", rfdinfo, "RF", rfcinfo };

struct	RF_stat {
	short		rf_wticks;	/* Monitor time for function */
	u_short		rf_cmd;		/* command to issue */
	u_int		rf_daddr;	/* block number */
	u_int		rf_vmeaddr;	/* physical address of transfer */
	int		rf_blkcnt;	/* number of blocks to transfer */
	int		rf_unit;	/* drive/volume to issue */
	int		rf_dint;	/* done interrupt value */
	int		rf_ctype;	/* controller type */
	int		rf_bsybit;	/* bsybit */
	int		rf_nspt;	/* sectors/track for next volume */
	int		rf_shead;	/* start head for next volume */
	struct sb_ident	rf_ident;	/* firmware information */
} RF_stat[NRF];

struct qb_diskst	rfst[Nrf];
struct buf		rrfbuf[Nrf];
struct dkbad		rfbad[Nrf];
int			rf_offset[Nrf];	/* sector offset past uib/remap */

#define	b_bn		b_resid	
#define	rfunit(bp)	(minor((bp)->b_dev) >> 3)

int			rfwstart, rfwatch();	/* Have started guardian */

#define	RF_INTLEV	3
#define RF_MAXCNT	MAX_RF_BSIZE
#define	RF_TIMEOUT	hz*5
#define	RF_RETRYCNT	10
union rf_uib		rf_uib;
struct rf_xpb		rf_xpb[NRF];
#ifdef	DEBUG
int			rf_debug = 0x0;
#endif	DEBUG

rfprobe(rfaddr, cn)
	register struct rfdevice	*rfaddr;
{
	register struct rf_xpb		*xpb = &rf_xpb[cn];
	register struct rf_pb		*pb = &xpb->xpb_pb;

	if (rfreset(rfaddr, cn))
		return(0);
	XPB_SET(xpb, PB_COMMAND_GOPT, PB_UNIT_NODISK, PB_CONTROL_IOCG_0, 0);
	pb->gopt_throttle		= GOPT_THROTTLE;
	pb->pb.pb_gopt.gopt_sbenable	= GOPT_SB_WDL|GOPT_SB_DTE|GOPT_SB_IDE;
	pb->pb.pb_gopt.gopt_sbintr	= GOPT_SB_WDL;
	RF_stat[cn].rf_dint = (RF_INTLEV << DINT_LEV_SHFT) | freevec();
	xpb->xpb_dint = RF_stat[cn].rf_dint;
	if (rfexec(xpb, cn, rfaddr, 0))
		return (0);
	DELAY(50000);
	xpb->xpb_dint = 0;

	XPB_SET(xpb, PB_COMMAND_IDENT, PB_UNIT_NODISK, PB_CONTROL_IOCG_0, 0);
	if (rfexec(xpb, cn, rfaddr, 1))
		return (0);
	RF_stat[cn].rf_ident = xpb->xpb_sb.sb_ident;
	RF_stat[cn].rf_nspt = 1;
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return(sizeof(struct rfdevice));
}

rfslave(qi, rfaddr)
	register struct qb_device	*qi;
	register struct rfdevice	*rfaddr;
{
	int			cn = qi->qi_ctlr;
	int			dr = qi->qi_unit;
	register struct rf_xpb	*xpb = &rf_xpb[cn];
	register struct rf_pb	*pb = &xpb->xpb_pb;
	register struct uib	*uib = &rf_uib.uib;
	register int		i, oldsum, newsum;
	register u_short	*wp = (u_short *)uib;

	XPB_SET(xpb, PB_COMMAND_SETCONF, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb.pb_confd.confd_nbps	= 512;
	pb->pb.pb_confd.confd_ncyl	= 1;
	pb->pb.pb_confd.confd_nspt	= RF_stat[cn].rf_nspt;
	pb->pb.pb_confd.confd_nhead	= 1;
	pb->pb.pb_confd.confd_shead	= RF_stat[cn].rf_shead;
	if (RF_stat[cn].rf_ctype == 3200)
	    pb->pb.pb_confd.confd_flags	= CONFD_FLAGS_SSP;
	if (rfexec(xpb, cn, rfaddr, 1))
		return (0);

	if (RF_stat[cn].rf_ctype = 3400) {
	    XPB_SET(xpb, PB_COMMAND_START, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	    if (rfexec(xpb, cn, rfaddr, 1))
		return(0);
	}

	XPB_SET(xpb, PB_COMMAND_READ, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb_addrmod			= RF_ADDRMOD;
	pb->pb.pb_std.std_daddr		= 0;
	pb->pb.pb_std.std_vmeaddr	= svtop(uib);
	pb->pb.pb_std.std_blkcnt	= 1;
	if (rfexec(xpb, cn, rfaddr, 1))
		return (0);
	if (uib->uib_magic != RF_UIB_MAGIC) {
		printf("rf: drive not formatted\n");
		return (0);
	}
	oldsum = uib->uib_cksum;
	uib->uib_cksum = newsum = 0;
	for (i = 0; i < sizeof(struct uib); i += sizeof(u_short))
		newsum ^= *wp++;
	if (newsum != oldsum) {
		printf("rf: drive not formatted (checksum)\n");
		return (0);
	}
	if (uib->uib_confd_nhead == 0)
		return (0);

	XPB_SET(xpb, PB_COMMAND_SETCONF, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb.pb_confd.confd_nbps	= uib->uib_confd_nbps;
	pb->pb.pb_confd.confd_ncyl	= uib->uib_confd_ncyl;
	pb->pb.pb_confd.confd_nspares	= uib->uib_confd_nspares;
	pb->pb.pb_confd.confd_nspt	= uib->uib_confd_nspt;
	pb->pb.pb_confd.confd_nhead	= uib->uib_confd_nhead;
	pb->pb.pb_confd.confd_shead	= uib->uib_confd_shead;
	if ((UTOVOL(dr) == 0) && (RF_stat[cn].rf_ctype == 3200)) {
	    RF_stat[cn].rf_nspt = uib->uib_confd_nspt;
	    RF_stat[cn].rf_shead = uib->uib_confd_shead1;
	} else {
	    RF_stat[cn].rf_nspt = 1;
	    RF_stat[cn].rf_shead = 0;
	}
	pb->pb.pb_confd.confd_flags	= uib->uib_confd_flags;
	if (rfexec(xpb, cn, rfaddr, 1))
		return (0);

	XPB_SET(xpb, PB_COMMAND_SETDPARAM, PB_UNIT(dr), PB_CONTROL_IOCG_0, 0);
	pb->pb.pb_dparam.dparam_intlv	= uib->uib_dparam_intlv;
	pb->pb.pb_dparam.dparam_hdskew	= uib->uib_dparam_hdskew;
	pb->pb.pb_dparam.dparam_cyskew	= uib->uib_dparam_cyskew;
	pb->pb.pb_dparam.dparam_hgskew	= uib->uib_dparam_hgskew;
	pb->pb.pb_dparam.dparam_rcvry	= uib->uib_dparam_rcvry;
	pb->pb.pb_dparam.dparam_gap1	= uib->uib_dparam_gap1;
	pb->pb.pb_dparam.dparam_gap2	= uib->uib_dparam_gap2;
	if (rfexec(xpb, cn, rfaddr, 1))
		return(0);

	XPB_SET(xpb, PB_COMMAND_SETIOCG, PB_UNIT_NODISK, PB_UNIT(dr), 0);
	pb->pb.pb_iocg.iocg_rcvry	= uib->uib_iocg_rcvry;
	pb->pb.pb_iocg.iocg_rahead	= uib->uib_iocg_rahead;
	pb->pb.pb_iocg.iocg_cache	= uib->uib_iocg_cache;
	pb->pb.pb_iocg.iocg_ndretry	= uib->uib_iocg_ndretry;
	pb->pb.pb_iocg.iocg_dretry	= uib->uib_iocg_dretry;
	if (rfexec(xpb, cn, rfaddr, 1))
		return (0);
	return (1);
}

rfattach(qi)
register struct qb_device	*qi;
{
	register struct qb_diskst	*st;
	register struct uib		*uib = &rf_uib.uib;
	struct sb_ident			*ident = &RF_stat[qi->qi_ctlr].rf_ident;

	if (rfwstart == 0) {
		timeout(rfwatch, (caddr_t)0, RF_TIMEOUT);
		rfwstart++;
	}
	st = &rfst[qi->qi_unit];
	st->st_ntpc = uib->uib_confd_nhead;
	st->st_ncpd = uib->uib_confd_ncyl - uib->uib_cyl_offset;
	st->st_nspt = uib->uib_confd_nspt;
	rf_offset[qi->qi_unit] = uib->uib_cyl_offset * st->st_nspt * st->st_ntpc;
	printf("	%mM %s (%d x %d x %d)\n",
	    st->st_nspt * st->st_ntpc * st->st_ncpd * uib->uib_confd_nbps, uib->uib_label,
	    st->st_nspt, st->st_ntpc, st->st_ncpd);
	printf("\t\tRF%d %d.%d: cache %b ra %d rc %b re %d/%d ",
	    RF_stat[qi->qi_ctlr].rf_ctype, ident->ident_fwrev, 
	    ident->ident_engrev, uib->uib_iocg_cache, IOCG_CACHE_BITS, 
	    uib->uib_iocg_rahead, uib->uib_iocg_rcvry, IOCG_RCVRY_BITS, 
	    uib->uib_iocg_dretry, uib->uib_iocg_ndretry);
	diskpart(st->st_size, st->st_nspt, st->st_ntpc, st->st_ncpd);
	st->st_nspc = st->st_nspt * st->st_ntpc;
	if (qi->qi_dk >= 0)
		dk_bps[qi->qi_dk] = (3600/60)*512*st->st_nspt;
}

rfopen(dev)
	dev_t dev;
{
	register int		 	unit = minor(dev) >> 3;
	register struct qb_device 	*qi;

	if (unit >= Nrf || (qi = rfdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	return (0);
}

rfstrategy(bp)
	register struct buf *bp;
{
	register struct qb_diskst	 *st;
	register struct buf		 *bbp;
	register int			 unit = rfunit(bp);
	register struct qb_device	 *qi = rfdinfo[unit];
	int				 part = minor(bp->b_dev) & 07;

	if (unit >= Nrf || qi == 0 || qi->qi_alive == 0) {
		bp->b_error = ENXIO;
		goto bad;
	}
	st = &rfst[qi->qi_unit];
	if (bp->b_blkno < 0 || 
	    (bp->b_blkno+((bp->b_bcount+511)>>9)) > st->st_size[part].nblocks) {
		if (bp->b_blkno == st->st_size[part].nblocks) {
		    bp->b_resid = bp->b_bcount;
		    goto done;
		}
		bp->b_error = EINVAL;
		goto bad;
	}

	/* 
	 * The controller can only do sector transfers. On a read we must
	 * do the transfer into a kernel buffer and then out to memory so
	 * we do not overwrite memory beyond the requested location.
	 */
	if ((bp->b_bcount & 511) && (bp->b_flags&B_READ)) {
		bbp = (struct buf *)getqeblk((bp->b_bcount+511) & ~511, 
			BQUALIFY(bp->b_dev));
		bbp->b_dev = bp->b_dev;
		bbp->b_blkno = bp->b_blkno;
		bbp->b_flags |= (bp->b_flags & (B_READ|B_WRITE));
		badstrat(qi, bbp, &rfbad[qi->qi_unit], rfstrat, st->st_nspt, 
		    st->st_ntpc, st->st_ncpd, st->st_size[part].cyloff);
		biowait(bbp);
		pcopyout(bbp->b_un.b_addr, bptop(bp), bp->b_bcount);
		bp->b_flags |= (bbp->b_flags & B_ERROR);
		bp->b_resid = bbp->b_resid;
		bbp->b_dev = NODEV;
		bfree(bbp);
		brelse(bbp);
		goto done;
	}
	badstrat(qi, bp, &rfbad[qi->qi_unit], rfstrat, st->st_nspt, st->st_ntpc, 
		st->st_ncpd, st->st_size[part].cyloff);
	return;
bad:	bp->b_flags |= B_ERROR;
done:	iodone(bp);
}

rfstrat(qi, bp)
register struct qb_device	 *qi;
register struct buf		 *bp;
{
	register struct qb_diskst	 *st = &rfst[qi->qi_unit];
	int				 part = minor(bp->b_dev) & 07;
	int				 s;
	    register struct buf		 *dp;

	bp->b_bn = bp->b_blkno + (st->st_size[part].cyloff * st->st_nspc);
	s = splx(qi->qi_mi->qm_psl);
	dp = &qi->qi_mi->qm_tab;
	disksort(dp, bp);
	if (dp->b_active == 0)
		rfstartup(qi->qi_mi);
	splx(s);
}

rfstartup(qm)
register struct qb_ctlr		 *qm;
{
	register struct buf		 *bp;
	register struct qb_device	 *qi;
	register struct RF_stat		 *stat;

	if ((bp = qm->qm_tab.b_actf) == NULL)
		return;

	/* Mark controller busy, and determine destination.  */
	qm->qm_tab.b_active++;
	qi = rfdinfo[rfunit(bp)];		/* Controller */
	stat = &RF_stat[qm->qm_ctlr];

	/* save away current data transfer drive info */
	stat->rf_cmd = (bp->b_flags & B_READ) ? 
				PB_COMMAND_READ : PB_COMMAND_WRITE;
	stat->rf_unit = PB_UNIT(qi->qi_slave & 0x7);
	stat->rf_daddr = bp->b_bn + rf_offset[qi->qi_unit];
	stat->rf_vmeaddr = qbaddr(bp);
	stat->rf_blkcnt = (bp->b_bcount+511) >> 9;
	rfio(qm->qm_ctlr, (struct rfdevice *)qm->qm_addr, stat,1);

	if (qi->qi_dk >= 0) {
		dk_busy |= 1 << qi->qi_dk;
		dk_xfer[qi->qi_dk]++;
		dk_wds[qi->qi_dk] += bp->b_bcount >> 6;
	}
}

rfio(cn, rfaddr, stat, int_enable)
register struct rfdevice	*rfaddr;
register struct RF_stat		*stat;
int			int_enable;
{
	register struct rf_xpb	*xpb = &rf_xpb[cn];
	register struct rf_pb	*pb = &xpb->xpb_pb;
	register struct rf_sb	*sb = &xpb->xpb_sb.sb;

	XPB_SET(xpb, stat->rf_cmd, stat->rf_unit, stat->rf_unit, 0);
	pb->pb_addrmod			= RF_ADDRMOD;
	pb->pb.pb_std.std_daddr		= stat->rf_daddr;
	pb->pb.pb_std.std_vmeaddr	= stat->rf_vmeaddr;
	pb->pb.pb_std.std_blkcnt	= stat->rf_blkcnt;
	if (int_enable) {
		xpb->xpb_dint = RF_stat[cn].rf_dint;
		rfexec(xpb, cn, rfaddr, 0);
	} else {
		xpb->xpb_dint = 0;
		rfexec(xpb, cn, rfaddr, 1);
	}
}

/*
 * Handle a disk interrupt.
 */
rfintr(cn)
{
	register struct qb_ctlr		*qm = rfcinfo[cn];
	register struct buf		*bp = qm->qm_tab.b_actf;
	register struct qb_device	*qi = rfdinfo[rfunit(bp)];
	register struct RF_stat		*stat = &RF_stat[cn];
	register struct sb		*sb = &rf_xpb[cn].xpb_sb.sb;
	int 				*p;

	stat->rf_wticks = 0;
	dk_busy &= ~(1 << qi->qi_dk);

#ifdef	DEBUG
	if (rf_debug & (1 << cn)) {
	    p = (int *)sb;
	    printf("          SB: %8x %8x %8x\n", *p, *(p+1), *(p+2));
	}
#endif	DEBUG
	if (sb->sb_flag & SB_FLAG_ERR) {
	    if (++qm->qm_tab.b_errcnt > RF_RETRYCNT) {
		harderr(bp, "rf");
		printf("error %x: sn %d: flags %b: dstatus %b\n",
		    sb->sb_error, sb->sb_daddr, sb->sb_flag, SB_FLAG_BITS,
		    sb->sb_dstatus, (RF_stat[cn].rf_ctype == 3200) ? 
		    SB_DSTATUS_BITS_3200 : SB_DSTATUS_BITS_3400);
		bp->b_flags |= B_ERROR;
	    } else
		qm->qm_tab.b_active = 0;	/* force retry */
	}

	if (qm->qm_tab.b_active) {
		dk_busy &= ~(1 << qi->qi_dk);
		qm->qm_tab.b_active = qm->qm_tab.b_errcnt = 0;
		bp->b_resid = 0;
		qm->qm_tab.b_actf = bp->av_forw;
		iodone(bp);
	}
	if (qm->qm_tab.b_actf && qm->qm_tab.b_active == 0)
		rfstartup(qm);
}

rfexec(xpb, cn, rfaddr, wait)
register struct rf_xpb		*xpb;
register struct rfdevice 	*rfaddr;
{
	register struct sb	*sb = &xpb->xpb_sb.sb;
	int			timer = 50000;
	int			*p;

#ifdef	DEBUG
	if (rf_debug & (1 << cn)) {
	    p = (int *)xpb;
	    printf("PB:%8x %8x %8x %8x %8x\n",*p,*(p+1),*(p+2),*(p+3),*(p+4));
	}
#endif	DEBUG
	while ((rfaddr->rf_status & STATUS_BSY) != RF_stat[cn].rf_bsybit)
		if (--timer)
			DELAY(400);
		else {
			printf("RF%d: timeout\n", cn);
			return (-1);
		}
	RF_stat[cn].rf_bsybit ^= STATUS_BSY;

	/* Write the control and address modifier to the address buffer port */
	rfaddr->rf_abp_ctlam = ABP_CTLAM_SET | RF_XFER_WID | RF_ADDRMOD;
	rfaddr->rf_abp_pb_msb = svtop(xpb) >> 16;
	rfaddr->rf_abp_pb_lsb = svtop(xpb);
	rfaddr->rf_atn = ATN_TYPE0;

	if (wait) {
	    timer = 0;
	    while ((sb->sb_flag & SB_FLAG_CC) == 0) {
		    if (++timer == 50000) {
			    printf("rf: timeout\n");
			    return (-1);
		    } else
			    DELAY(400);
	    }
#ifdef	DEBUG
	    if (rf_debug & (1 << cn)) {
		p = (int *)sb;
		printf("          SB: %8x %8x %8x\n", *p, *(p+1), *(p+2));
	    }
#endif	DEBUG
	    if (sb->sb_flag & SB_FLAG_ERR)
		    return(-1);
	}
	return(0);
}

rfread(dev, uio)
dev_t		dev;
struct uio	*uio;
{
	register int		unit = minor(dev) >> 3;

	if (unit >= Nrf)
		return (ENXIO);
	return (physio(rfstrategy, &rrfbuf[unit], dev, B_READ, rfminphys, uio));
}

rfwrite(dev, uio)
dev_t		dev;
struct uio	*uio;
{
	register int		unit = minor(dev) >> 3;

	if (unit >= Nrf)
		return (ENXIO);
	return (physio(rfstrategy, &rrfbuf[unit], dev, B_WRITE, rfminphys, uio));
}

rfsizeof(dev)
dev_t	dev;
{
	register int			unit = minor(dev) >> 3;
	register struct qb_device	*qi;
	register struct qb_diskst	*st;

	if (unit >= Nrf || (qi = rfdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	st = &rfst[qi->qi_unit];
	return (st->st_size[minor(dev)&0x7].nblocks);
}

/*
 * Wake up every second and if an interrupt is pending but nothing has happened
 * increment a counter. If nothing happens for 20 seconds, reset the controller
 * and begin anew.
 */
rfwatch()
{
	register struct qb_ctlr		*qm;
	register int			cn;
	register struct RF_stat		*stat;

	timeout(rfwatch, (caddr_t)0, RF_TIMEOUT);
	for (cn = 0; cn < NRF; cn++) {
		qm = rfcinfo[cn];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		stat = &RF_stat[cn];
		stat->rf_wticks++;
		if (stat->rf_wticks >= 20) {
			stat->rf_wticks = 0;
			printf("RF%d: lost interrupt\n", cn);
			rfreset((struct rfdevice *)qm->qm_addr, cn);
		}
	}
}

rfminphys(bp)
struct buf	*bp;
{
	bp->b_bcount = MIN(bp->b_bcount, RF_MAXCNT);
}

rfreset(rfaddr, cn)
	register struct rfdevice	*rfaddr;
{ 	
	rfaddr->rf_reset = RF_RESET;
	RF_stat[cn].rf_bsybit = 0;
	DELAY(500000);
	if ((rfaddr->rf_status & STATUS_RDY) == 0) {
	    printf("RF%d: not ready: %x\n", cn, rfaddr->rf_status);
	    return (-1);
	}
	if (rfaddr->rf_status & STATUS_EMASK) {
	    printf("RF%d: diagnostics failed: %b\n", cn,
		1<<((rfaddr->rf_status&STATUS_EMASK)>>STATUS_ESHFT), 
		STATUS_EBITS);
	    return (-1);
	}
	if ((rfaddr->rf_status&STATUS_BMASK) == STATUS_B3200)
		RF_stat[cn].rf_ctype = 3200;
	else
		RF_stat[cn].rf_ctype = 3400;
	return (0);
}

rfdump() {printf("rfdump\n");}
#ifdef	DUMP
int rfdumpstrat();
extern struct buf dumpbuf;
extern int dumpsize;

rfdump(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register int part = minor(dev) & 07;
	register struct buf *bp = &dumpbuf;
	register struct qb_device *qi;
	register struct qb_diskst *st;
	int num, start, bcount;

	if (unit >= Nrf || (qi = rfdinfo[unit]) == 0 || qi->qi_alive == 0)
		return ENXIO;
	num = ctod(dumpsize);		/* memory size in disk blocks */
	start = 0x0;			/* start dumping at physical 0 */
	st = &rfst[unit];
	if (dumplo < 0 || dumplo+num > st->st_size[part].nblocks)
		return EINVAL;
	while (num > 0) {
		bcount = (dbtob(num) > RF_MAXCNT) ? RF_MAXCNT : dbtob(num);
		bp->b_dev = dev;
		bp->b_blkno = dumplo + btodb(start);
		bp->b_bcount = bcount;
		bp->b_un.b_addr = (caddr_t) start;
		bp->b_flags = B_WRITE; 
		badstrat(qi, bp, &rfbad[qi->qi_unit], rfdumpstrat,
		    st->st_nspt, st->st_ntpc, st->st_ncpd, st->st_size[part].cyloff);
		if (bp->b_flags & B_ERROR || bp->b_resid)
			return EIO;
		start += bcount;
		num -= btodb(bcount);
	}
	return 0;
}

rfdumpstrat(qi, bp)
	register struct qb_device *qi;
	register struct buf *bp;
{
	register struct RF_stat		*stat = &RF_stat[qi->qi_ctlr];
	register struct qb_diskst	*st = &rfst[qi->qi_unit];
	register int			drive = qi->qi_slave & 0x7;
	register int			errcount;
	int				i;
	register RF32REG		*rfaddr = (RF32REG *)qi->qi_mi->qm_addr;
	register STATBLK		*sb = (STATBLK *)stat->rf_status;

	errcount = 0;
	while (1) {
		stat->rf_daddr = bp->b_blkno + 
		    (st->st_size[minor(bp->b_dev)&07].cyloff * st->st_nspc) + 
		    rf_offset[qi->qi_unit];
		stat->rf_vmeaddr = (u_int) bp->b_un.b_addr;
		stat->rf_blkcnt = (bp->b_bcount+511) >> 9;
		stat->rf_unit = PB_UNIT(drive);
		stat->rf_cmd = (bp->b_flags & B_READ) ? C_READ : C_WRITE;
		rfio(qi->qi_ctlr, rfaddr, stat, 0);
		if (sb->status & ST_ERR) {
			printf("rf%d: sn%d HARD %s err: %x, stat:%b\n",
				drive, i - st->st_nspc, "WRITE", 
				sb->errcode, sb->status, ST_BITS);
			errcount++;
			if (++errcount == RF_RETRYCNT) {
				bp->b_flags |= B_ERROR;
				bp->b_resid = bp->b_bcount;
				return EIO;
			}
		} else {
			bp->b_resid = 0;
			break;
		}
	}
	bp->b_flags |= B_DONE;
	return 0;
}
#endif	DUMP
#endif
