/*
 *	 XICOR X2444 Non-Volatile Static RAM
 *
 * This driver currently supports reading NVRAM;
 *	 Write to NVRAM using the PROM command "&s"
 */

#include "param.h"
#include "../machine/board.h"
#include "../is68kdev/sioreg.h"
#include "../is68kdev/openchip.h"
#include "../is68kdev/nvram.h"

#define	RAM_SEL_DELAY	10
#define RAM_LOW_DELAY	10
#define RAM_HIGH_DELAY	10

#define	SETSTROBE	(Q_OP_PORT2 = OP2_SET_STROBE)
#define	RESETSTROBE	(Q_OP_PORT2 = OP2_RES_STROBE)
#define	SETDATAOUT	(Q_OP_PORT2 = OP2_SET_DATOUT)
#define	RESETDATAOUT	(Q_OP_PORT2 = OP2_RES_DATOUT)

struct nv_ram nvram_buf;	/* Driver's copy for private use */
int nvram_finit = 0;		/* Initialization flag */

/*
 * nvram_get - return NVRAM contents 
 */
nvram_get(pnvram)
    struct nv_ram *pnvram;
{
    register int i;
    register int *p = &nvram_buf, *q = pnvram;

    if (nvram_init()) {
	for (i=0; i < (sizeof (struct nv_ram))>>2; i++)
	    *q++ = *p++;
	return(1);
    }
    return(0);
}

/*
 * nvram_init - read NVRAM into driver buf -- once only
 */
nvram_init()
{
	int	i, ret;
	unsigned short *q;

	if (nvram_finit)
	    return(1);
	/*
	 * send an RCL instruction to the non-volatile RAM chip.
	 * The chip may not then be accessed again for 2.5 microseconds. 
	 */
	ramselect();
	shiftout(8, 0x85, RAM_LOW_DELAY, RAM_HIGH_DELAY);
	ramdeselect();
	delaycycles(100);
	q = &nvram_buf;
	for (i=0;i<16;++i)
		*q++ = nv_readw(i);
	if (ret = chkram(&nvram_buf))
	    nvram_finit++;
	return(ret);
}

chkram(pnvram)
struct nv_ram *pnvram;
{
	int             i;
	unsigned char	cksum;
	unsigned char *p = pnvram;

	cksum = 0;
	for (i = 0; i < 31; ++i)
		cksum += *p++;
	if (*p == cksum)
		return (1);
	else
		return (0);
}

/*
 * Output the byte 1AAAA11X where the A's give the address. The X value
 * should be a 1 for the QX. Then read 16 bits of data. 
 */
nv_readw(ind)
{
	unsigned short  res;

	ramselect();
	shiftout(8, 0x87 | (ind << 3), RAM_LOW_DELAY, RAM_HIGH_DELAY);
	res = shiftin(16, RAM_LOW_DELAY, RAM_HIGH_DELAY);
	ramdeselect();
	return res;
}

ramselect()
{
	SETDATAOUT;		/* Always have DATAOUT at logic 1, except for
				 * reading */
	Q_OP_PORT2 = OP2_SET_RAMSEL;	/* enable RAM */
	delaycycles(RAM_SEL_DELAY);
}

ramdeselect()
{
	Q_OP_PORT2 = OP2_RES_RAMSEL;	/* disable RAM */
}

/*
 * read a number of bits from the serial RAM or clock chip. Data is read most
 * significant bit first. Data is read on the negative going edge. 
 */
shiftin(nbits, clklowdel, clkhighdel)
{
	register int    shftreg;

	for (shftreg = 0; nbits-- > 0;) {
		shftreg <<= 1;
		delaycycles(clklowdel);
		if (DATIN)
			shftreg++;
		SETSTROBE;
		delaycycles(clkhighdel);
		RESETSTROBE;
	}
	return (shftreg);
}

/*
 * write a number of bits to the serial RAM or clock chip. Data is written
 * most signiicant bit first. Data is clocked on the positive going edge. 
 */
shiftout(nbits, shftreg, clklowdelay, clkhighdelay)
	register        nbits;
	register unsigned shftreg;
{
	register unsigned mask;

	for (mask = 1 << (nbits - 1); nbits-- > 0; mask >>= 1) {
		if (shftreg & mask) {
			SETDATAOUT;
		} else {
			RESETDATAOUT;
		}
		delaycycles(clklowdelay);
		SETSTROBE;
		delaycycles(clkhighdelay);
		RESETSTROBE;
	}
	SETDATAOUT;		/* Always leave DATAOUT set */
}

delaycycles(del)
	register short  del;
{
	del /= 3;		/* Assuming best case, cached DBRA in loop */
	if (del > 1) {
		del--;
		do {
		} while (--del != -1);
	}
}

