/*
 *	Copyright (c) 1985, 1986, 1987, 1988 Morning Star Technologies
 *	All rights reserved
 *
 *	This software may not be duplicated or used in any way without
 *	express written permission from Morning Star Technologies.
 */
# ifndef lint
static char sccs_id[] = "%W% -- ISI 4.3 BSD VMUnix";
# endif /* lint */

# ifndef	KERNEL
# define	KERNEL
# endif

# include	"param.h"
# include	"buf.h"
# include	"systm.h"
# include	"dir.h"
# include	"user.h"
# include	"uio.h"
# include	"proc.h"
# include	"conf.h"
# include	"ioctl.h"
# include	"tty.h"
# include	"file.h"
# include	"map.h"
# include	"vm.h"
# include	"kernel.h"
# include	"../is68kdev/qbvar.h"

# define	resume()	longjmp(&u.u_qsave, 1)

/*
 *	Simulate spinlocks and semaphores on single-threaded operating systems.
 *	Make sure that lock() protects against interrupts from the controller.
 */
# define	MSTPRI		(PZERO - 1)
# define	MSTLOPRI	(PZERO + 1)

# define	lock()		(mstspl_t)spl4()
# define	unlock(i)	splx(i)
# define	initsem(s, v)	(*(s) = (v))

typedef int mstsema_t;
typedef int mstspl_t;

static P(sema, pri)
mstsema_t *sema;
int pri;
    {
    mstspl_t ipl = lock();
    label_t q_save;

    if (pri < PZERO)
	{
	while (*sema == 0)
	    sleep(sema, pri);

	--*sema;
	unlock(ipl);
	return (0);
	}
    else
	{
	q_save = u.u_qsave;

	if (setjmp(&u.u_qsave) == 0)
	    {
	    while (*sema == 0)
		sleep(sema, pri);

	    --*sema;
	    unlock(ipl);
	    return (0);
	    }
	else
	    {
	    u.u_qsave = q_save;
	    unlock(ipl);
	    return (-1);
	    }
	}
    }

static int P_nowait(sema)
mstsema_t *sema;
    {
    mstspl_t ipl = lock();

    if (*sema == 0)
	{
	unlock(ipl);
	return (0);
	}
    else
	{
	--*sema;
	unlock(ipl);
	return (1);
	}
    }

static V(sema)
mstsema_t *sema;
    {
    mstspl_t ipl = lock();

    ++*sema;
    wakeup(sema);
    unlock(ipl);
    }

/*
 *	Define macros for dealing with multiple devices.  A total of 256
 *	minor devices are available for all of the boards in a single system.
 *	To change the number of minor devices per board, and therefore also
 *	the maximum number of boards, change the definition of UNIT_BITS.
 *	Do not change any of the other definitions.
 */
# define	UNIT_BITS	7	/* This must be either 5, 6 or 7 */
# define	N_UNITS		(1 << UNIT_BITS)
# define	MAX_BOARDS	(1 << (8 - UNIT_BITS))
# define	BOARD(dev)	(minor(dev) >> UNIT_BITS)
# define	UNIT(dev)	(minor(dev) & (N_UNITS - 1))

/*
 *	Maximum number of outstanding data transfer type requests to the
 *	board.  Additional requests are queued.
 */
# define	N_OUTSTANDING_IO	12
# define	N_PREREADS		(N_UNITS - 1)
# define	N_CANCELS		20

/*
 *	Size of the board's memory
 */
# define	REGISTER_SIZE	0x10000

/*
 *	Minor device N_UNITS-1 is used only for booting; N_UNITS-2 is used
 *	only for loading, and device N_UNITS-3 is used for reading the mst_t
 *	structure.
 */
# define	BOOT_DEV	(N_UNITS-1)
# define	LOAD_DEV	(N_UNITS-2)
# define	DEBUG_DEV	(N_UNITS-3)
# define	MSTINITTIMEOUT	(60/4)
# define	MSTBOOTTIMEOUT	(2*60)
# define	MSTIOTIMEOUT	(2*60)

# define	FALSE		0
# define	TRUE		1

/*
 *	Copyright (c) 1985, 1987, 1988 Morning Star Technologies
 *	All rights reserved
 *
 *	This software may not be duplicated or used in any way without
 *	express written permission from Morning Star Technologies.
 */

/* @(#)mstdriver.h	3.7 */

# define	INT_CLEAR	0
# define	INT_SET		1
# define	INT_PEND_CLEAR	2

# define	SUCCESS		0

# define	QUIESCENT	(char)(-1)
# define	IN_PROGRESS	(char)(-2)
# define	COMPLETED	(char)(-3)

# define	OP_OPEN		1
# define	OP_CLOSE	2
# define	OP_READ		3
# define	OP_WRITE	4
# define	OP_IOCTL	5

# ifdef __STDC__
# define	MIOC_CLEAR	_IO('M', 1)
# define	MIOC_START	_IO('M', 2)
# define	MIOC_REWIND	_IO('M', 3)
# define	MIOC_DIAGNOSE	_IOWR('M', 4, int)
# define	MIOC_CANCEL	_IO('M', 5)
# else
# define	MIOC_CLEAR	_IO(M, 1)
# define	MIOC_START	_IO(M, 2)
# define	MIOC_REWIND	_IO(M, 3)
# define	MIOC_DIAGNOSE	_IOWR(M, 4, int)
# define	MIOC_CANCEL	_IO(M, 5)
# endif

# define	DIOC		('d' << 8)
# define	DIOC_MEMSHORT	(DIOC | 1)
# define	DIOC_MEMLONG	(DIOC | 2)
# define	DIOC_CLOCK	(DIOC | 3)
# define	DIOC_DMAC	(DIOC | 4)
# define	DIOC_SIO	(DIOC | 5)
# define	DIOC_DMAMEM	(DIOC | 6)
# define	DIOC_DMASIO	(DIOC | 7)
# define	DIOC_ROMS	(DIOC | 8)
# define	DIOC_LOOP	(DIOC | 9)

/*
 *	Total number of I/O's that can be batched for transfer to/from the
 *	front end.
 */
# define	BATCH_SIZE	10

/*
 *	Structure of an I/O being completed.
 */
typedef struct
    {
    unsigned char i_key;
    unsigned char i_status;
# define	i_signal	i_status
    short i_resid;
    long i_proc;
    } in_t;

typedef struct
    {
    unsigned char o_key;
    unsigned char o_op_code;
    unsigned char o_unit;
    char o_pad1;
    long o_proc;
    short o_length;
    short o_pad2;
    char *o_addr;
    long o_cmd;
# define	o_flags		o_cmd
    } out_t;

typedef struct
    {
    long m_boot;
    long m_state;
    long m_long;
    short m_short[2];
    char m_char[4];
    long m_version;
    char m_int_bus;
    char m_int_cpu;
    short m_unused;
    short m_outstanding_io;
    short m_channels;
    char m_out_state;
    char m_in_state;
    char m_out_count;
    char m_in_count;
    out_t m_out[BATCH_SIZE];
    in_t m_in[BATCH_SIZE];
    char m_enable[128];
    } mst_t;

/*
 *	Values in m_version
 */
# define	MST_VERSION	7
# define	V_NUM_MASK	0xFF
# define	V_RQ_64KBFR	0x100
# define	V_OK_64KBFR	0x200

/*
 *	Define system-dependent functions and parameters
 */
# define	SIGNAL(pgrp, sig)	gsignal(pgrp, sig)
# define	U_COUNT			uio -> uio_resid
# define	U_OFFSET		uio -> uio_offset
# define	U_PUTC(c)		ureadc(c, uio)
# define	b_paddr			b_un.b_addr
# define	IOCTL_SIZE		128

/*
 *	Amount of space at the end of memory allocated to the mst_t structure.
 */
# define	MST_T_SIZE	512

/*
 *	Offset from the bus address of the mst_t structure.
 */
# define	MST_OFFSET	(REGISTER_SIZE - MST_T_SIZE)

/*
 *	Driver information for autoconfiguration.
 */
extern int mstprobe(), mstintr(), mstslave(), mstattach();
struct mb_device *mstinfo[MAX_BOARDS];

/*u_short *mstaddrs[] = { (u_short *)&vme_stdio[0xF90000+MST_OFFSET], (u_short *)0 };*/
extern u_short *MSTstd[];

struct qb_driver MSTdriver =
    { mstprobe, mstslave, mstattach, 0, MSTstd, "mst", 0, "MST", 0 };

# define	N_BOARDS	MAX_BOARDS

/*
 *	Maximum size of a DMA transfer.  Used when allocating bus map
 *	registers and by mst_minphys() when calling physio().
 */
# define	TRANSFER_SIZE	512

# define	get_char(a)	(*(char *)(a))
# define	put_char(a, v)	(*(char *)(a) = (v))
# define	get_short(a)	(*(short *)(a))
# define	put_short(a, v)	(*(short *)(a) = (v))
# define	get_long(a)	(((long)get_short(a) << 16) |\
				((long)get_short((short *)(a) + 1) & 0xFFFFL))
# define	put_long(a, v)	(put_short((short *)(a), (short)((v) >> 16)),\
				put_short((short *)(a) + 1, (short)(v)))

struct mstbuf
    {
    struct buf mb_bufhdr;		/* struct buf for this I/O */
    mstsema_t mb_allocated;		/* semaphore for allocation */
    mstsema_t mb_iowait;		/* semaphore for I/O completion */
    unsigned char mb_op_code;		/* read/write/open/close/ioctl */
    long mb_cmd;			/* cmd of an ioctl */
# define	mb_flags	mb_cmd
    long mb_proc;			/* process info for any outbound I/O */
    long mb_busaddr;			/* Bus address of I/O buffer */
    char *mb_buffer;			/* Data buffer for read/write */
    };

typedef struct mstbuf mstbuf_t;

# define	N_IOCBUF	8

typedef struct
    {
    mstsema_t i_allocated;
    char *i_data;
    } iocbuf_t;

# define	MAX_WAIT	10000

short mst_debug = 4;
# define	DEBUG1		if (mst_debug & 1) printf
# define	DEBUG2		if (mst_debug & 2) printf
# define	DEBUG4		if (mst_debug & 4) printf
# define	DEBUG8		if (mst_debug & 8) printf
# define	DEBUG16		if (mst_debug & 16) printf
# define	DEBUG32		if (mst_debug & 32) printf
# define	DEBUG64		if (mst_debug & 64) printf
# define	DEBUG128	if (mst_debug & 128) printf
int mst_wait[MAX_BOARDS];

typedef struct
    {
    mst_t *sc_csr;
    int sc_vector;
    mstsema_t sc_buf_sema;
    mstsema_t sc_ioc_sema;
    mstsema_t sc_bt_sema;
    mstbuf_t *sc_head;
    mstbuf_t *sc_tail;
    iocbuf_t sc_iocbuf[N_IOCBUF];
    char sc_unlocked;
    char sc_booted;
    char sc_boot_state;
    char sc_act_timeout;
    int sc_req_time;
    char sc_can_timeouts;
    mstbuf_t sc_bufhdr[N_OUTSTANDING_IO + N_PREREADS + N_CANCELS];
    mstsema_t sc_open_sema[N_UNITS];
    char sc_rdbusy[N_PREREADS];
    struct tty sc_tty[N_UNITS];
    } m_softc_t;

m_softc_t m_softc[N_BOARDS];

/*
 *	Get a free buffer header.
 */
static mstbuf_t *get_bufhdr(dev)
dev_t dev;
    {
    register mstbuf_t *bp;
    register m_softc_t *sp = &m_softc[BOARD(dev)];

    DEBUG8("get_bufhdr(0x%x)\n", dev);

    P(&sp -> sc_buf_sema, MSTPRI);

    for (bp = sp -> sc_bufhdr; bp < &sp -> sc_bufhdr[N_OUTSTANDING_IO]; ++bp)
	if (P_nowait(&bp -> mb_allocated))
	    {
	    bp -> mb_bufhdr.b_dev = dev;
	    DEBUG8("get_bufhdr(0x%x) returning buffer %d\n", dev,
		    bp - sp -> sc_bufhdr);
	    return (bp);
	    }

    panic("mst: get_bufhdr");
    /*NOTREACHED*/
    }

/*
 *	Free up this buffer header and toss it to the snapping dog pack.
 */
static free_bufhdr(bp)
register mstbuf_t *bp;
    {
    register m_softc_t *sp = &m_softc[BOARD(bp -> mb_bufhdr.b_dev)];

    DEBUG8("free_bufhdr(0x%x) freeing buffer %d\n",
	    bp -> mb_bufhdr.b_dev, bp - sp -> sc_bufhdr);
    bp -> mb_bufhdr.b_dev = 0;
    V(&bp -> mb_allocated);
    V(&sp -> sc_buf_sema);
    }

/*
 *	Get a free IOCTL buffer.
 */
static iocbuf_t *get_iocbuf(dev)
dev_t dev;
    {
    register m_softc_t *sp = &m_softc[BOARD(dev)];
    register iocbuf_t *ip;

    P(&sp -> sc_ioc_sema, MSTPRI);

    for (ip = sp -> sc_iocbuf; ip < &sp -> sc_iocbuf[N_IOCBUF]; ++ip)
	if (P_nowait(&ip -> i_allocated))
	    {
	    DEBUG8("get_iocbuf(0x%x) returning iocbuf %d\n", dev,
		    ip - sp -> sc_iocbuf);
	    return (ip);
	    }

    panic("mst: get_iocbuf");
    /*NOTREACHED*/
    }

/*
 *	Free up this IOCTL buffer and toss it to the snapping dog pack.
 */
static free_iocbuf(dev, ip)
dev_t dev;
register iocbuf_t *ip;
    {
    register m_softc_t *sp = &m_softc[BOARD(dev)];

    DEBUG8("free_iocbuf(0x%x) freeing iocbuf %d\n", dev, ip - sp -> sc_iocbuf);
    V(&ip -> i_allocated);
    V(&sp -> sc_ioc_sema);
    }

/*
 *	Get a free cancel buffer.
 */
static mstbuf_t *get_cancel(sp)
register m_softc_t *sp;
    {
    register mstbuf_t *bp;

    DEBUG8("get_cancel(%d)\n", sp - m_softc);

    for (bp = &sp -> sc_bufhdr[N_OUTSTANDING_IO + N_PREREADS];
	    bp < &sp -> sc_bufhdr[N_OUTSTANDING_IO + N_PREREADS + N_CANCELS];
	    ++bp)
	if (P_nowait(&bp -> mb_allocated))
	    {
	    DEBUG8("get_cancel(%d) returning buffer %d\n", sp - m_softc,
		    bp - sp -> sc_bufhdr);
	    return (bp);
	    }

    DEBUG8("get_cancel(%d) returning 0\n", sp - m_softc);
    return ((mstbuf_t *)0);
    }

/*
 *	Free up this cancel buffer.
 */
static free_cancel(bp)
register mstbuf_t *bp;
    {
    register m_softc_t *sp = &m_softc[BOARD(bp -> mb_bufhdr.b_dev)];

    DEBUG8("free_cancel(0x%x) freeing buffer %d\n",
	    bp -> mb_bufhdr.b_dev, bp - sp -> sc_bufhdr);
    V(&bp -> mb_allocated);
    }

/*
 *	This routine forces all pending I/O's to complete with an error.
 *	Must be called under lock.
 */
tmo_io(sp)
register m_softc_t *sp;
    {
    register mstbuf_t *bp;

    DEBUG32("tmo_io()\n");
    sp -> sc_unlocked = FALSE;
    sp -> sc_booted = FALSE;
    sp -> sc_head = sp -> sc_tail = 0;

    for (bp = sp -> sc_bufhdr;
	    bp < &sp -> sc_bufhdr[N_OUTSTANDING_IO + N_PREREADS]; ++bp)
	if (bp -> mb_bufhdr.b_error == IN_PROGRESS)
	    mstdone(bp, EPERM);

    DEBUG32("tmo_io() returning\n");
    }

/*
 *	Probe routine called at system startup -- see if the board is there.
 */
int mstprobe(reg, unit)
caddr_t reg;
int unit;
    {
    extern int cvec;

    register int i;
    register mst_t *csr = (mst_t *)reg;
    register m_softc_t *sp = &m_softc[unit];
    register mstbuf_t *bp;

    sp -> sc_vector = 0xFF;

    /*printf("Probing mst %d address 0x%x\n", unit, reg);*/

    put_long(&csr -> m_long, 0x00010203);
    put_short(&csr -> m_short[0], 0x0000);
    put_short(&csr -> m_short[1], 0x0001);
    put_char(&csr -> m_char[0], 0x00);
    put_char(&csr -> m_char[1], 0x01);
    put_char(&csr -> m_char[3], 0x03);
    put_char(&csr -> m_char[2], 0x02);
    put_long(&csr -> m_version, MST_VERSION);
    put_long(&csr -> m_boot, 2 | (sp -> sc_vector << 8));

    /*
     *	Give the front end time to respond.
     */
    for (i = 0; i < 81; ++i)
	{
	if (get_char(&csr -> m_int_bus) == INT_SET)
	    break;

	DELAY(100000);
	}

    /*
     *	Did he respond?
     */
    if (get_char(&csr -> m_int_bus) != INT_SET)
	return(0);

    /*
     *	Initialize the I/O buffers.
     */
    put_char(&csr -> m_int_bus, INT_CLEAR);
    put_short(&csr -> m_outstanding_io,
	    N_OUTSTANDING_IO + N_PREREADS + N_CANCELS);
    put_short(&csr -> m_channels, N_UNITS);
    put_char(&csr -> m_out_state, QUIESCENT);
    put_char(&csr -> m_in_state, QUIESCENT);
    put_char(&csr -> m_out_count, 0);
    put_char(&csr -> m_in_count, 0);

    /*
     *	Let the front end know we got it.
     */
    sp -> sc_csr = csr;
    put_char(&csr -> m_int_cpu, INT_SET);
    init_softc(sp);
    return (REGISTER_SIZE);
    }

/*
 *	Required (but unused) routines
 */
int mstslave()
    {
    return (1);
    }

int mstattach()
    {
    return (1);
    }

/*
 *	Initialize data structures for this controller
 */
static init_softc(sp)
register m_softc_t *sp;
    {
    register char *buf_addr;
    register iocbuf_t *ip;
    register mstbuf_t *bp;
    register int i;

    /*
     *	Attach read/write/ioctl data buffers
     */
    buf_addr = (char *)(56 * 1024);

    for (ip = sp -> sc_iocbuf; ip < &sp -> sc_iocbuf[N_IOCBUF]; ++ip)
	{
	ip -> i_data = buf_addr;
	buf_addr += IOCTL_SIZE;
	}

    for (bp = sp -> sc_bufhdr; bp < &sp -> sc_bufhdr[N_OUTSTANDING_IO]; ++bp)
	{
	bp -> mb_buffer = buf_addr;
	buf_addr += TRANSFER_SIZE;
	}

    /*
     *	Initialize semaphores
     */
    for (bp = sp -> sc_bufhdr;
	    bp < &sp -> sc_bufhdr[N_OUTSTANDING_IO + N_PREREADS + N_CANCELS];
	    ++bp)
	{
	initsem(&bp -> mb_allocated, 1);
	initsem(&bp -> mb_iowait, 0);
	}

    for (i = 0; i < N_UNITS; ++i)
	initsem(&sp -> sc_open_sema[i], 1);

    for (i = 0; i < N_PREREADS; ++i)
	sp -> sc_rdbusy[i] = QUIESCENT;

    for (ip = sp -> sc_iocbuf; ip < &sp -> sc_iocbuf[N_IOCBUF]; ++ip)
	initsem(&ip -> i_allocated, 1);

    initsem(&sp -> sc_buf_sema, N_OUTSTANDING_IO);
    initsem(&sp -> sc_ioc_sema, N_IOCBUF);
    initsem(&sp -> sc_bt_sema, 0);
    }

/*
 *	Process a command from the line discipline.
 */
int mstproc()
    {
    }

/*
 *	An MST device has been opened.  Format a command in a buffer
 *	header and let the strategy routine send it to the board.
 */
int mstopen(dev, flags)
dev_t dev;
int flags;
    {
    register m_softc_t *sp = &m_softc[BOARD(dev)];
    register int unit = UNIT(dev);
    register struct tty *tp = &sp -> sc_tty[unit];
    register mstbuf_t *bp;
    int was_closed = 0, error;

    DEBUG2("mstopen(0x%x, 0x%x)\n", dev, flags);

    if (BOARD(dev) >= N_BOARDS)
	return (u.u_error = ENXIO);

    if ( ! sp -> sc_csr)
	return (u.u_error = ENXIO);

    if (unit == BOOT_DEV)
	return (u.u_error = mst_boot(dev));

    if (unit == DEBUG_DEV)
	return (0);

    if (unit != LOAD_DEV)
	if (P(&sp -> sc_open_sema[unit], MSTLOPRI) < 0)
	    resume();

    if ((tp -> t_state & TS_ISOPEN) == 0)
	{
	ttychars(tp);
	tp -> t_line = 0;
	tp -> t_ispeed = tp -> t_ospeed = EXTB;
	tp -> t_flags = 0;
	tp -> t_rsel = tp -> t_wsel = 0;
	tp -> t_oproc = mstproc;
	tp -> t_state |= TS_CARR_ON | TS_ISOPEN;
	was_closed = 1;
	}

    error = (*linesw[tp -> t_line].l_open)(dev, tp);

    if (error)
	{
	if (unit != LOAD_DEV)
	    V(&sp -> sc_open_sema[unit]);

	DEBUG4("mstopen(0x%x, 0x%x) tty error %d\n", dev, flags, error);
	return (u.u_error = error);
	}

    bp = get_bufhdr(dev);

    bp -> mb_op_code = OP_OPEN;
    bp -> mb_proc = (long)u.u_procp -> p_pgrp;
    bp -> mb_bufhdr.b_dev = dev;
    bp -> mb_bufhdr.b_bcount = 0;
    bp -> mb_bufhdr.b_paddr = 0;
    bp -> mb_flags = flags;

    mststrategy(bp);
    error = bp -> mb_bufhdr.b_error;

    free_bufhdr(bp);

    if (error)
	{
	if (was_closed)
	    {
	    (*linesw[tp -> t_line].l_close)(tp);
	    ttyclose(tp);
	    tp -> t_state &= ~(TS_CARR_ON | TS_ISOPEN);
	    }

	DEBUG4("mstopen(0x%x, 0x%x) error %d\n", dev, flags, error);
	}

    if (unit != LOAD_DEV)
	V(&sp -> sc_open_sema[unit]);

    return (u.u_error = error);
    }

/*
 *	An MST device has been closed.  If this is a normal device, send
 *	a message to the board.
 */
int mstclose(dev, flags)
dev_t dev;
int flags;
    {
    register m_softc_t *sp = &m_softc[BOARD(dev)];
    register mstbuf_t *bp;
    register int unit = UNIT(dev);
    register struct tty *tp = &sp -> sc_tty[unit];

    DEBUG2("mstclose(0x%x, 0x%x)\n", dev, flags);

    if (unit == BOOT_DEV || unit == DEBUG_DEV)
	return (0);

    if (unit != LOAD_DEV)
	if (P(&sp -> sc_open_sema[unit], MSTLOPRI) < 0)
	    resume();

    bp = get_bufhdr(dev);
    bp -> mb_op_code = OP_CLOSE;

    /*
     *	Pass the real UID of this process on CLOSE, since that is a
     *	reasonable time to want to do any sort of accounting and
     *	since signals don't make sense any more after closing.
     */
    bp -> mb_proc = u.u_ruid;
    bp -> mb_bufhdr.b_dev = dev;
    bp -> mb_bufhdr.b_bcount = 0;
    bp -> mb_bufhdr.b_paddr = 0;
    bp -> mb_flags = flags;

    mststrategy(bp);

    if (bp -> mb_bufhdr.b_error)
	DEBUG4("mstclose(0x%x, 0x%x) error %d\n", dev, flags,
		bp -> mb_bufhdr.b_error);

    free_bufhdr(bp);

    (*linesw[tp -> t_line].l_close)(tp);
    ttyclose(tp);
    tp -> t_state &= ~(TS_CARR_ON | TS_ISOPEN);

    if (sp -> sc_rdbusy[unit] == COMPLETED)
	{
	bp = &sp -> sc_bufhdr[unit + N_OUTSTANDING_IO];

	if (P_nowait(&bp -> mb_iowait))
	    sp -> sc_rdbusy[unit] = QUIESCENT;
	else
	    panic("mstclose: can't allocate completed preread semaphore");
	}

    if (unit != LOAD_DEV)
	V(&sp -> sc_open_sema[unit]);

    return (u.u_error = 0);
    }

/*
 *	Do a special command.
 */
int mstioctl(dev, cmd, addr, flag)
dev_t dev;
int cmd;
caddr_t addr;
int flag;
    {
    register m_softc_t *sp = &m_softc[BOARD(dev)];
    register mstbuf_t *bp;
    register iocbuf_t *ip;
    register int unit = UNIT(dev);
    register struct tty *tp = &sp -> sc_tty[unit];
    unsigned int size;
    int tty_error, error;

    DEBUG2("mstioctl(0x%x, 0x%x, 0x%x, 0x%x)\n", dev, cmd, addr, flag);

    /*
     *	Use the 'diagnose' ioctl to enable debugging messages
     */
    if (unit == DEBUG_DEV && cmd == MIOC_DIAGNOSE)
	{
	size = mst_debug;
	mstcopyin(sp,addr, (caddr_t)&error, sizeof error);
	mstcopyout(sp,(caddr_t)&size, addr, sizeof size);
	mst_debug = error & 0xFF;
	return (0);
	}

    if (unit == BOOT_DEV || unit == DEBUG_DEV)
	return (0);


    tty_error = ttioctl(tp, cmd, addr, flag);

    if (tty_error == -1)
	tty_error = EINVAL;

    while (getc(&tp -> t_rawq) >= 0)
	;

    while (getc(&tp -> t_outq) >= 0)
	;

    /*
     *	If the tty ioctl routine doesn't recognize this command, go on;
     *	else it found some sort of error.
     */
    if (tty_error != 0 && tty_error != EINVAL)
	{
	DEBUG4("mstioctl(0x%x, 0x%x, 0x%x, 0x%x) TTY error %d\n",
		dev, cmd, addr, flag, tty_error);
	return (tty_error);
	}

    bp = get_bufhdr(dev);

    bp -> mb_op_code = OP_IOCTL;
    bp -> mb_proc = (long)u.u_procp -> p_pgrp;
    bp -> mb_bufhdr.b_dev = dev;
    size = (cmd >> 16) & IOCPARM_MASK;
    bp -> mb_cmd = mstconvert(cmd, addr, &size) & 0xFFFF;
    bp -> mb_bufhdr.b_bcount = size;

    if (size > IOCTL_SIZE)
	{
	free_bufhdr(bp);
	return (u.u_error = EINVAL);
	}

    if (size)
	{
	/*
	 *	Allocate an IOCTL buffer to hold the IOCTL data during DMA.
	 */
	ip = get_iocbuf(dev);
	bp -> mb_bufhdr.b_paddr = ip -> i_data;
	mstcopyin(sp,addr, ip -> i_data + ((int)sp -> sc_csr - MST_OFFSET), size);
	}
    else
	bp -> mb_bufhdr.b_paddr = 0;

    mststrategy(bp);
    error = bp -> mb_bufhdr.b_error;

    if (error == EINVAL)
	error = tty_error;

    if (size)
	{
	/*
	 *	Copy the user's IOCTL buffer back.
	 */
	if (error == 0)
	    mstcopyout(sp,ip -> i_data + ((int)sp -> sc_csr - MST_OFFSET),
		    addr, size - bp -> mb_bufhdr.b_resid);

	if (cmd == FIONBIO && error == 0 && *(long *)addr == 0L)
	    if (P_nowait(&sp -> sc_bufhdr[unit + N_OUTSTANDING_IO].mb_iowait))
		if (sp -> sc_rdbusy[unit] == COMPLETED)
		    sp -> sc_rdbusy[unit] = QUIESCENT;
		else
		    panic("mstioctl: allocated preread semaphore isn't completed");

	/*
	 *	Free up this IOCTL buffer.
	 */
	free_iocbuf(dev, ip);
	}

    if (error)
	DEBUG4("mstioctl(0x%x, 0x%x, 0x%x, 0x%x) error %d\n",
		dev, cmd, addr, flag, error);

    free_bufhdr(bp);
    mstrevert(cmd, addr);

    /*
     *	If a loaded protocol was successfully started, then allow
     *	I/O on all other channels.
     */
    if (cmd == MIOC_START && error == 0)
	sp -> sc_unlocked = TRUE;

    return (error);
    }

/*
 *	Translate this IOCTL command if necessary.
 */
/*ARGSUSED*/
int mstconvert(cmd, addr, size_ptr)
int cmd;
caddr_t addr;
unsigned int *size_ptr;
    {
    switch (cmd)
	{
	case TIOCSETP:	/* Translate sgttyb into MST sgttyb */
	case TIOCSETN:
	    addr[7] = addr[5];
	    addr[6] = addr[4];
	    addr[4] = 0;
	    addr[5] = 0;	/* Fall through to set size */

	case TIOCGETP:	/* Copy back all 8 bytes of sgttyb */
	    if (*size_ptr < 8)
		*size_ptr = 8;

	    return (cmd);
	}

    return (cmd);
    }

/*
 *	Translate this IOCTL command's buffer back if necessary.
 */
/*ARGSUSED*/
mstrevert(cmd, addr)
int cmd;
caddr_t addr;
    {
    switch (cmd)
	{
	case TIOCGETP:	/* Translate MST sgttyb into 4.2 BSD sgttyb */
	    addr[5] = addr[7];
	    addr[4] = addr[6];
	    break;
	}
    }

/*
 *	Get the board running back in its boot ROM.
 */
int mst_boot(dev)
dev_t dev;
    {
    register m_softc_t *sp = &m_softc[BOARD(dev)];
    register mst_t *csr = sp -> sc_csr;
    mstspl_t ipl;

    ipl = lock();

    if (sp -> sc_boot_state == IN_PROGRESS)
	{
	unlock(ipl);
	return (EPERM);
	}

    /*
     *	Boot the front-end.
     */
    sp -> sc_unlocked = FALSE;
    sp -> sc_booted = FALSE;
    sp -> sc_boot_state = IN_PROGRESS;
    unlock(ipl);

    put_long(&csr -> m_long, 0x00010203);
    put_short(&csr -> m_short[0], 0x0000);
    put_short(&csr -> m_short[1], 0x0001);
    put_char(&csr -> m_char[0], 0x00);
    put_char(&csr -> m_char[1], 0x01);
    put_char(&csr -> m_char[3], 0x03);
    put_char(&csr -> m_char[2], 0x02);
    put_long(&csr -> m_version, MST_VERSION | V_RQ_64KBFR);
    set_timeout(sp, MSTBOOTTIMEOUT);

    if (sp -> sc_vector)
	put_long(&csr -> m_boot, 2 | (sp -> sc_vector << 8));
    else
	{
	printf("mst%d: Warning: Booting with default vector\n", sp - m_softc);
	put_long(&csr -> m_boot, 1);
	}

    /*
     *	Give the front end time to respond.
     */
    DEBUG16("mst_boot: 1\n");
    P(&sp -> sc_bt_sema, MSTPRI);

    /*
     *	Force all current outstanding I/O's to complete.
     */
    ipl = lock();
    tmo_io(sp);
    unlock(ipl);

    if (sp -> sc_boot_state != COMPLETED)
	{
	DEBUG16("mst_boot: 2 (status = %d)\n", sp -> sc_boot_state);
	return (sp -> sc_boot_state);
	}

    if ( ! (get_long(&csr -> m_version) & V_OK_64KBFR))
	{
	printf("mst%d: Controller won't grant on-board buffering\n",
		sp - m_softc);
	return (EIO);
	}

    /*
     *	Initialize the I/O buffers.
     */
    put_char(&csr -> m_int_bus, INT_CLEAR);
    put_short(&csr -> m_outstanding_io,
	    N_OUTSTANDING_IO + N_PREREADS + N_CANCELS);
    put_short(&csr -> m_channels, N_UNITS);
    put_char(&csr -> m_out_state, QUIESCENT);
    put_char(&csr -> m_in_state, QUIESCENT);
    put_char(&csr -> m_out_count, 0);
    put_char(&csr -> m_in_count, 0);
    DEBUG16("mst_boot: 3\n");

    /*
     *	Let the front end know we got it.
     */
    sp -> sc_boot_state = 0;
    sp -> sc_booted = TRUE;
    put_char(&csr -> m_int_cpu, INT_SET);
    return (0);
    }

/*
 *  These timeout routines are used to timeout activity if the buffers being
 *  passed between the host and the front end are not responded to within a
 *  minimum amount of time.  That response does not, of course, mean that any
 *  I/O has been done, only that the front end has noticed the existance of
 *  the buffer and taken the I/O packets from it.
 */
static set_timeout(sp, ticks)
register m_softc_t *sp;
int ticks;
    {
    extern mst_timeout();

    DEBUG32("set_timeout(%d)\n", ticks);

    sp -> sc_can_timeouts = FALSE;

    if (sp -> sc_act_timeout)
	sp -> sc_req_time = ticks;
    else
	{
	sp -> sc_act_timeout = TRUE;
	timeout(mst_timeout, (caddr_t)sp, ticks);
	}
    }

static cancel_timeout(sp)
register m_softc_t *sp;
    {
    DEBUG32("cancel_timeout:\n");
    sp -> sc_can_timeouts = TRUE;
    sp -> sc_req_time = 0;
    }

static mst_timeout(sp)
register m_softc_t *sp;
    {
    register mstspl_t ipl;

    DEBUG32("mst_timeout:\n");

    ipl = lock();

    if (sp -> sc_can_timeouts)
	sp -> sc_act_timeout = FALSE;
    else
	if (sp -> sc_req_time != 0)
	    {
	    timeout(mst_timeout, (caddr_t)sp, sp -> sc_req_time);
	    sp -> sc_req_time = 0;
	    }
	else
	    {
	    DEBUG32("mst_timeout: actually timing out\n");
	    sp -> sc_act_timeout = FALSE;

	    if (sp -> sc_boot_state)
		{
		if (sp -> sc_boot_state == IN_PROGRESS)
		    {
		    sp -> sc_boot_state = EIO;
		    V(&sp -> sc_bt_sema);
		    }
		}
	    else
		{
		printf("mst%d: Timeout\n", sp - m_softc);
		tmo_io(sp);
		}
	    }

    unlock(ipl);
    }

/*
 *	Reduce size of physio() transfer to fit available mapping hardware
 */
mst_minphys(bp)
register struct buf *bp;
    {
    if ((int)bp -> b_bcount > TRANSFER_SIZE)
	bp -> b_bcount = TRANSFER_SIZE;
    }

/*
 *	Send a pre-read message to the front-end.  Must be called under lock.
 */
mstpreread(dev)
dev_t dev;
    {
    register m_softc_t *sp = &m_softc[BOARD(dev)];
    register int unit = UNIT(dev);
    register mstbuf_t *bp;
    mstspl_t ipl;

    DEBUG2("mstpreread(0x%x)\n", dev);
    ipl = lock();

    if (sp -> sc_rdbusy[unit] == QUIESCENT)
	{
	sp -> sc_rdbusy[unit] = IN_PROGRESS;
	bp = &sp -> sc_bufhdr[unit + N_OUTSTANDING_IO];
	bp -> mb_op_code = OP_READ;
	bp -> mb_proc = (long)u.u_procp -> p_pgrp;
	bp -> mb_cmd = 0;
	bp -> mb_bufhdr.b_dev = dev;
	bp -> mb_busaddr = 0;
	bp -> mb_bufhdr.b_bcount = 0;
	bp -> mb_bufhdr.b_error = IN_PROGRESS;
	mststart(sp, bp);
	}

    unlock(ipl);
    }

/*
 *	Read a block of data from the front end.
 */
int mstread(dev, uio)
dev_t dev;
struct uio *uio;
    {
    register m_softc_t *sp = &m_softc[BOARD(dev)];
    register int unit = UNIT(dev);
    register mstbuf_t *bp;
    int error;

    DEBUG2("mstread(0x%x)\n", dev);

    if (unit == BOOT_DEV)
	return (0);

    if (unit == DEBUG_DEV)
	{
	u.u_error = 0;

	while (U_COUNT > 0 && U_OFFSET < (8 * 1024) && u.u_error == 0)
	    {
	    char ch = get_char((char *)sp -> sc_csr - (8 * 1024 - MST_T_SIZE) + U_OFFSET);

	    U_PUTC(ch);
	    }

	return (0);
	}

    bp = &sp -> sc_bufhdr[unit + N_OUTSTANDING_IO];
    mstpreread(dev);

    if (P(&bp -> mb_iowait, MSTLOPRI) < 0)
	resume();

    bp = get_bufhdr(dev);

    bp -> mb_op_code = OP_READ;
    bp -> mb_proc = (long)u.u_procp -> p_pgrp;
    bp -> mb_cmd = 0;
    bp -> mb_bufhdr.b_dev = dev;

    mstphysio(bp, B_READ, uio);

    if (bp -> mb_bufhdr.b_error)
	DEBUG4("mstread(0x%x) error %d\n", dev, bp -> mb_bufhdr.b_error);

    error = bp -> mb_bufhdr.b_error;
    free_bufhdr(bp);
    sp -> sc_rdbusy[unit] = QUIESCENT;
    mstpreread(dev);
    return (error);
    }

/*
 *	Write a block of data to the front end.
 */
int mstwrite(dev, uio)
dev_t dev;
struct uio *uio;
    {
    register mstbuf_t *bp;
    register int unit = UNIT(dev);
    int error;

    DEBUG2("mstwrite(0x%x)\n", dev);

    if (unit == BOOT_DEV)
	return (0);

    if (unit == DEBUG_DEV)
	return (u.u_error = EACCES);

    bp = get_bufhdr(dev);

    bp -> mb_op_code = OP_WRITE;
    bp -> mb_proc = (long)u.u_procp -> p_pgrp;
    bp -> mb_cmd = 0;
    bp -> mb_bufhdr.b_dev = dev;

    mstphysio(bp, B_WRITE, uio);

    if (bp -> mb_bufhdr.b_error)
	DEBUG4("mstwrite(0x%x) error %d\n", dev, bp -> mb_bufhdr.b_error);

    error = bp -> mb_bufhdr.b_error;
    free_bufhdr(bp);
    return (error);
    }

/*
 *	Simulate physio(), but buffer through our own pre-allocated buffers.
 */
mstphysio(bp, flags, uio)
register mstbuf_t *bp;
int flags;
struct uio *uio;
    {
    register int size;
    register m_softc_t *sp = &m_softc[BOARD(bp -> mb_bufhdr.b_dev)];

    u.u_error = 0;

    while (U_COUNT > 0 && u.u_error == 0)
	{
	size = U_COUNT > TRANSFER_SIZE ? TRANSFER_SIZE : U_COUNT;

	if (flags == B_WRITE)
	    mstuiomove(sp, bp -> mb_buffer + ((int)sp -> sc_csr - MST_OFFSET),
		    size, UIO_WRITE, uio);

	bp -> mb_bufhdr.b_bcount = size;
	bp -> mb_bufhdr.b_paddr = bp -> mb_buffer;
	mststrategy(bp);

	if (bp -> mb_bufhdr.b_error)
	    break;

	if (flags == B_READ)
	    mstuiomove(sp, bp -> mb_buffer + ((int)sp -> sc_csr - MST_OFFSET),
		    size - bp -> mb_bufhdr.b_resid, UIO_READ, uio);

	if (bp -> mb_bufhdr.b_resid)
	    {
	    if (flags == B_WRITE)
		{
		U_OFFSET -= bp -> mb_bufhdr.b_resid;
		U_COUNT += bp -> mb_bufhdr.b_resid;
		}

	    break;
	    }
	}
    }

/*
 *	Return 1 if reading or writing won't block
 */
int mstselect(dev, rw)
dev_t dev;
int rw;
    {
    extern int selwait;

    register m_softc_t *sp = &m_softc[BOARD(dev)];
    register struct tty *tp = &sp -> sc_tty[UNIT(dev)];
    register mstbuf_t *bp;
    register mstspl_t ipl;

    DEBUG2("mstselect(0x%x, %d)\n", dev, rw);

    switch (rw)
	{
	case FREAD:
	    /*
	     *	If this device is quiescent, issue a pre-read.
	     *	If a pre-read has completed and nobody is using it, then
	     *	a read should not block.
	     */
	    bp = &sp -> sc_bufhdr[UNIT(dev) + N_OUTSTANDING_IO];

	    if (P_nowait(&bp -> mb_iowait))
		{
		V(&bp -> mb_iowait);
		return (1);
		}

	    ipl = lock();
	    mstpreread(dev);
	    unlock(ipl);

	    if (tp -> t_rsel
		    && (caddr_t)tp -> t_rsel -> p_wchan == (caddr_t)&selwait)
		tp -> t_state |= TS_RCOLL;
	    else
		tp -> t_rsel = u.u_procp;

	    return (0);

	case FWRITE:
	    /*
	     *	Return 1 if no reads are outstanding on this device.
	     *	This does not guarantee that a write will not block, but
	     *	there is a better chance.
	     */
	    for (bp = sp -> sc_bufhdr; bp < &sp -> sc_bufhdr[N_OUTSTANDING_IO];
		    ++bp)
		{
		ipl = lock();

		if (bp -> mb_bufhdr.b_dev == dev
			&& bp -> mb_op_code == OP_WRITE)
		    {
		    unlock(ipl);

		    if (tp -> t_wsel && (caddr_t)tp -> t_wsel -> p_wchan ==
			    (caddr_t)&selwait)
			tp -> t_state |= TS_WCOLL;
		    else
			tp -> t_wsel = u.u_procp;

		    return (0);
		    }

		unlock(ipl);
		}

	    return (1);

	default:
	    return (0);
	}

    /*NOTREACHED*/
    }
/*
 *	Send a cancel IOCTL for this operation
 */
static mstcancel(bp)
register mstbuf_t *bp;
    {
    register m_softc_t *sp = &m_softc[BOARD(bp -> mb_bufhdr.b_dev)];
    register mstbuf_t *bp1;
    register mstspl_t ipl;

    if (bp1 = get_cancel(sp))
	{
	DEBUG2("mstcancel(0x%x)\n", bp -> mb_bufhdr.b_dev);

	bp1 -> mb_op_code = OP_IOCTL;
	bp1 -> mb_proc = 0;
	bp1 -> mb_cmd = MIOC_CANCEL;
	bp1 -> mb_bufhdr.b_dev = bp -> mb_bufhdr.b_dev;
	bp1 -> mb_busaddr = bp -> mb_op_code;
	bp1 -> mb_bufhdr.b_bcount = 0;
	bp1 -> mb_bufhdr.b_error = IN_PROGRESS;
	ipl = lock();
	mststart(sp, bp1);
	unlock(ipl);
	}
    else
	printf("mst%d: Out of cancel buffers\n", sp - m_softc);
    }

/*
 *	Do an I/O to/from the front end.
 */
mststrategy(bp)
register mstbuf_t *bp;
    {
    register m_softc_t *sp = &m_softc[BOARD(bp -> mb_bufhdr.b_dev)];
    register mstspl_t ipl;

    DEBUG128("mststrategy(buffer %d)\n", bp - sp -> sc_bufhdr);

    bp -> mb_bufhdr.b_flags &= ~B_ERROR;

    bp -> mb_busaddr = (long)bp -> mb_bufhdr.b_paddr;

    ipl = lock();
    bp -> mb_bufhdr.b_error = IN_PROGRESS;
    mststart(sp, bp);
    unlock(ipl);

    if (bp -> mb_op_code == OP_READ || bp -> mb_op_code == OP_WRITE)
	{
	if (P(&bp -> mb_iowait, MSTLOPRI) < 0)
	    {
	    mstcancel(bp);
	    P(&bp -> mb_iowait, MSTPRI);
	    }
	}
    else
	P(&bp -> mb_iowait, MSTPRI);

    if (bp -> mb_bufhdr.b_error != SUCCESS)
	bp -> mb_bufhdr.b_flags |= B_ERROR;

    DEBUG128("mststrategy(buffer %d) returning status %d\n",
	    bp - sp -> sc_bufhdr, bp -> mb_bufhdr.b_error);
    DEBUG128("mststrategy(): op_code = %d\n", bp -> mb_op_code);
    }

/*
 *	Give transactions to the front end.  Must be called under lock.
 */
mststart(sp, bp)
register m_softc_t *sp;
register mstbuf_t *bp;
    {
    register mst_t *csr = sp -> sc_csr;
    register out_t *op;

    DEBUG128("mststart()\n");

    if (bp)
	{
	if ( ! sp -> sc_booted || ( ! sp -> sc_unlocked
		&& UNIT(bp -> mb_bufhdr.b_dev) != LOAD_DEV))
	    {
	    mstdone(bp, EPERM);
	    return;
	    }

	/*
	 *	Enqueue the I/O onto the I/O queue.
	 */
	bp -> mb_bufhdr.b_forw = 0;

	if (sp -> sc_head)
	    {
	    sp -> sc_tail -> mb_bufhdr.b_forw = &bp -> mb_bufhdr;
	    sp -> sc_tail = bp;
	    }
	else
	    sp -> sc_head = sp -> sc_tail = bp;
	}
    else
	if (sp -> sc_head == 0)
	    return;

    if ( ! sp -> sc_booted)
	return;

    /*
     *	Don't send messages yet if the board isn't ready.
     */
    if (get_char(&csr -> m_out_state) != QUIESCENT)
	return;

    /*
     *	Move as many I/O's from the queue into the outgoing buffer as we can.
     */
    put_char(&csr -> m_out_count, 0);

    for (op = csr -> m_out; op < &csr -> m_out[BATCH_SIZE]; ++op)
	{
	if ((bp = sp -> sc_head) == 0)
	    break;

	DEBUG128("mststart(): bp = %d\n", bp - sp -> sc_bufhdr);
	sp -> sc_head = (mstbuf_t *)bp -> mb_bufhdr.b_forw;

	if (sp -> sc_head == 0)
	    sp -> sc_tail = 0;

	put_char(&op -> o_key, bp - sp -> sc_bufhdr);
	put_char(&op -> o_op_code, bp -> mb_op_code);
	put_char(&op -> o_unit, UNIT(bp -> mb_bufhdr.b_dev));
	put_long(&op -> o_proc, bp -> mb_proc);
	put_short(&op -> o_length, bp -> mb_bufhdr.b_bcount);
	put_long(&op -> o_cmd, bp -> mb_cmd);
	put_long(&op -> o_addr, bp -> mb_busaddr);
	}

    put_char(&csr -> m_out_count, op - csr -> m_out);

    /*
     *	Set the outgoing buffer's state to IN_PROGRESS.
     */
    put_char(&csr -> m_out_state, IN_PROGRESS);

    /*
     *  Set a timeout up, so that if the front end doesn't respond, then
     *  the units will be timed out.
     */
    DEBUG128("mststart(): csr -> m_out_count = %d\n", op - csr -> m_out);
    set_timeout(sp, MSTIOTIMEOUT);

    /*
     *	Interrupt the front end.
     */
    put_char(&csr -> m_int_cpu, INT_SET);
    }

/*
 *	Complete the operation pending on this bp.
 */
mstdone(bp, status)
register mstbuf_t *bp;
int status;
    {
    register m_softc_t *sp = &m_softc[BOARD(bp -> mb_bufhdr.b_dev)];
    register struct tty *tp = &sp -> sc_tty[UNIT(bp -> mb_bufhdr.b_dev)];

    if (bp -> mb_bufhdr.b_error == IN_PROGRESS)
	bp -> mb_bufhdr.b_error = status;
    else
	panic("mstdone: Bad status");

    if (bp -> mb_op_code == OP_READ && tp -> t_rsel)
	{
	selwakeup(tp -> t_rsel, tp -> t_state & TS_RCOLL);
	tp -> t_state &= ~TS_RCOLL;
	tp -> t_rsel = 0;
	}

    if (bp -> mb_op_code == OP_WRITE && tp -> t_wsel)
	{
	selwakeup(tp -> t_wsel, tp -> t_state & TS_WCOLL);
	tp -> t_state &= ~TS_WCOLL;
	tp -> t_wsel = 0;
	}

    if (bp - sp -> sc_bufhdr >= N_OUTSTANDING_IO + N_PREREADS)
	{
	free_cancel(bp);
	return;
	}

    /*
     *	Mark pre-reads QUIESCENT if their device is closed.
     */
    if (bp - sp -> sc_bufhdr >= N_OUTSTANDING_IO)
	if (tp -> t_state & TS_ISOPEN)
	    sp -> sc_rdbusy[UNIT(bp -> mb_bufhdr.b_dev)] = COMPLETED;
	else
	    {
	    sp -> sc_rdbusy[UNIT(bp -> mb_bufhdr.b_dev)] = QUIESCENT;

	    /*
	     *	Don't V() the semaphore -- setting rdbusy = QUIESCENT is
	     *	equivalent to undoing the pre-read.
	     */
	    return;
	    }

    V(&bp -> mb_iowait);
    }

/*
 *	We (may) have just been interrupted.  Complete (or continue) I/Os.
 */
int mstintr()
    {
    register m_softc_t *sp;
    register int serviced = 0;
    register int in_state, out_state, i;
    register mstspl_t ipl;
    register mst_t *csr;

    /*
     *	No lock() is done here because it is implicit on a single-processor
     *	system at interrupt time.
     */
    for (sp = m_softc; sp < &m_softc[N_BOARDS]; ++sp)
	if ((csr = sp -> sc_csr) && get_char(&csr -> m_int_bus) == INT_SET)
	    {
	    DEBUG64("mstintr(%d):\n", sp - m_softc);

	    /*
	     *	First, get some volatile locations
	     */
	    in_state = get_char(&csr -> m_in_state);
	    out_state = get_char(&csr -> m_out_state);

	    /*
	     *	Clear the interrupt bit.
	     */
	    put_char(&csr -> m_int_bus, INT_PEND_CLEAR);

	    if ( ! sp -> sc_booted)
		{
		if (sp -> sc_boot_state == IN_PROGRESS
			&& get_long(&csr -> m_boot) == 0)
		    {
		    cancel_timeout(sp);
		    sp -> sc_boot_state = COMPLETED;
		    V(&sp -> sc_bt_sema);
		    }
		}
	    else
		if (out_state != COMPLETED && in_state != COMPLETED)
		    {
		    DEBUG64("mstintr: calling tmo_io\n");
		    tmo_io(sp);
		    printf(
	    "mst%d: Unexpected interrupt: in_state=%d out_state=%d\n",
			    sp - m_softc, in_state, out_state);
		    }
		else
		    {
		    if (in_state == COMPLETED)
			{
			register in_t *ip, *end;
			long pgrp;

			DEBUG64("mstintr(%d): in_state == COMPLETED\n",
				sp - m_softc);

			/*
			 *	Set the b_resid and b_error fields and do a
			 *	wakeup for each I/O in the incoming buffer.
			 */
			for (ip = csr -> m_in, end = &csr ->
				m_in[get_char(&csr -> m_in_count)];
				ip < end; ++ip)
			    /*
			     *	Process signals from the board
			     */
			    if (pgrp = get_long(&ip -> i_proc))
				{
				SIGNAL((short)pgrp, get_char(&ip -> i_signal));
				DEBUG2("mstintr(%d): signal(%d, %d)\n",
					sp - m_softc, pgrp,
					get_char(&ip -> i_signal));
				}
			    else
				{
				register int key =
					get_char(&ip -> i_key) & 0xFF;
				register mstbuf_t *bp = &sp -> sc_bufhdr[key];

				bp -> mb_bufhdr.b_resid =
					get_short(&ip -> i_resid);
				mstdone(bp, get_char(&ip -> i_status));
				}

			put_char(&csr -> m_in_count, 0);
			put_char(&csr -> m_in_state, QUIESCENT);
			}

		    if (out_state == COMPLETED)
			{
			DEBUG64("mstintr(%d): out_state == COMPLETED\n",
				sp - m_softc);

			/*
			 *	Cancel the timeout
			 *	Change the outgoing buffer's state to QUIESCENT
			 */
			cancel_timeout(sp);
			put_char(&csr -> m_out_state, QUIESCENT);
			mststart(sp, (mstbuf_t *)0);
			}
		    }

	    /*
	     *	Synchronize with the board interrupt routine, but not if it
	     *	has gone awry.
	     */
	    for (i = 0; i < MAX_WAIT; ++i)
		if (get_char(&csr -> m_int_bus) != INT_PEND_CLEAR)
		    break;

	    if (i > mst_wait[sp - m_softc])
		mst_wait[sp - m_softc] = i;

	    serviced = 1;
	    }

    return (serviced);
    }

mstcopyin(sp, src, dest, size)
m_softc_t *sp;
char *src, *dest;
int size;
    {
    if (size == 0)
	return;

    if (size < 0)
	panic("size < 0 in mstcopyin");

    if (((int)src & (int)dest & 1) == 1)
	put_char(dest++, *src++), --size;

    if (((int)src | (int)dest) & 1)
	while (size--)
	    put_char(dest++, *src++);
    else
	{
	while (size > 1)
	    {
	    put_short(dest, *(short *)src);
	    dest += 2;
	    src += 2;
	    size -= 2;
	    }

	if (size != 0)
	    put_char(dest, *src);
	}
    }

mstcopyout(sp, src, dest, size)
m_softc_t *sp;
char *src, *dest;
int size;
    {
    if (size == 0)
	return;

    if (size < 0)
	panic("size < 0 in mstcopyout");

    if (((int)src & (int)dest & 1) == 1)
	*dest++ = get_char(src++);

    if (((int)src | (int)dest) & 1)
	while (size--)
	    *dest++ = get_char(src++);
    else
	{
	while (size > 1)
	    {
	    *(short *)dest = get_short(src);
	    dest += 2;
	    src += 2;
	    size -= 2;
	    }

	if (size != 0)
	    *dest = get_char(src);
	}
    }

mstuiomove(sp, buffer, length, sense, uio)
struct uio *uio;
m_softc_t *sp;
char *buffer;
int length;
enum uio_rw sense;
    {
    static char buf[TRANSFER_SIZE];

    if (sense == UIO_READ)
	mstcopyin(sp, buffer, buf, length);

     uiomove(buf,  length,  sense, uio);

    if (sense == UIO_WRITE)
	mstcopyout(sp, buf, buffer, length);
    }
