/*
 * VMEBUS: Integrated Solutions SCSI laser disk (LD controler, ld disk) driver
 */
#include "ld.h"
#if	NLD > 0
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "dk.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"
#include "ioctl.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/ldreg.h"
#include "../machine/board.h"

/* LD controler addresses  see /sys/conf/devaddr.c*/
extern u_short *LDstd[];

#define LD_MAXCNT	256*LD_SEC_SIZE	/* the max count per r/w command */
#define	LD_TIMEOUT	60		/* number of seconds for timeout */
#define	LD_NRETRY	5
#define	LD_WAIT		0xFFF0

struct	size {
	daddr_t	nblocks;
	int	cyloff;
};

/*
 * State of controller from last transfer. Since only one transfer can be done
 * at a time per controller, only allocate one for each controller.
 */
struct	LD_stat {
	short		ld_wticks;	/* Monitor time for function */
	u_short		ld_command;	/* read/write command to issue */
	dev_t		ld_dev;		/* device */
	u_int		ld_bn;		/* block number */
	u_int		ld_addr;	/* physical address of transfer */
	int		ld_bleft;	/* bytes left to transfer */
	int		ld_bpart;	/* bytes transferred */
	u_short		ld_cmd[5];	/* registers for ioctl command */
}			LD_stat[NLD];

/* Buffer headers used  to pass ioctl commands to the internal routines. */
struct	buf	cldbuf[NLD];

int	ldprobe(),	ldslave(),	ldattach(),	ldintr();
int	ldminphys(),	lddgo();
struct	qb_ctlr		*ldcinfo[NLD];
struct	qb_device	*lddinfo[Nld];
#define LDADDR(dev)	((struct lddevice *)(lddinfo[UNIT(dev)])->qi_mi->qm_addr)

/* LD/ld controler/device driver structure */
struct	qb_driver LDdriver =
	{ldprobe, ldslave, ldattach, lddgo, LDstd, "ld", lddinfo, "LD",ldcinfo};

/* ld drive structure: each drive can have a different geometry.  */
struct qb_diskst	ldst[Nld];

struct	buf		rldbuf[Nld];
struct	buf		ldutab[Nld];
#define	b_bn		b_resid
#define	ldunit(bp)	(minor((bp)->b_dev) >> 3)

#ifdef	DEBUG
#define	dprintf(x)	printf x
#else	DEBUG
#define	dprintf(x)
#endif	DEBUG

ldprobe(ldaddr)
	register struct lddevice	*ldaddr;
{
	ldaddr->ld_cmd[0] = 0x84;		/* sd/gs controller (size) ? */
	DELAY(8000);
	if (ldaddr->ld_cmd[0] != 0x84)
		return (0);			/* yes */
	ldcommand(ldaddr->ld_cmd, LD_RST, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	ldaddr->ld_cs = LDCS_IE | LDCS_GO;
	DELAY(8000);
	if (ldwait(ldaddr) || (ldaddr->ld_cs&LDCS_OK_MASK) != LDCS_OK)
		return (0);
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (sizeof (struct lddevice));
}

ldslave(qi, ldaddr)
	register struct qb_device	*qi;
	register struct lddevice	*ldaddr;
{
	ldcommand(ldaddr->ld_cmd, LD_SENSE, 0, 0, 0, 0x10, 0, 0, 0, 0, 0);
	ldaddr->ld_cs = (qi->qi_slave<<LDCS_TID_SHIFT)|LDCS_GO;
	if (ldwait(ldaddr) || (ldaddr->ld_cs&LDCS_OK_MASK) != LDCS_OK)
		return (0);
	return (1);
}

ldattach(qi)
	register struct qb_device	*qi;
{
	static int			ldwstart, ldwatch();
	register struct qb_diskst	*st = &ldst[qi->qi_unit];

	if (ldwstart++ == 0)		/* Start watchdog for lost interupts */
		timeout(ldwatch, (caddr_t)0, LD_TIMEOUT*hz);
	st->st_ncpd = LD_CYL;		/* # of cylinders */
	st->st_ntpc = LD_TRACK;		/* # of tracks (heads) */
	st->st_nspt = LD_SPT;		/* # of sectors per track */
	printf("	%mM (%d x %d x %d) %d byte sectors", 
		st->st_nspt * st->st_ntpc * st->st_ncpd * LD_SEC_SIZE, 
		st->st_nspt, st->st_ntpc, st->st_ncpd, LD_SEC_SIZE);
	diskpart(st->st_size, st->st_nspt, st->st_ntpc, st->st_ncpd);
	st->st_nspc = st->st_nspt * st->st_ntpc;

	/* Initialize iostat values */
	if (qi->qi_dk >= 0)
		dk_bps[qi->qi_dk] = (LD_RPM/60)*LD_SEC_SIZE*st->st_nspt;
}

ldopen(dev)
	dev_t				dev;
{
	register int			unit = UNIT(dev);
	register struct qb_device	*qi;

	if (unit >= Nld || (qi = lddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	return(ldioctl(dev,LD_IOC_TSTRDY,NULL));
}

ldstrategy(bp)
	register struct buf		*bp;
{
	register int			unit = ldunit(bp);
	register struct qb_device	*qi = lddinfo[unit];
	int				part = minor(bp->b_dev) & 07;
	register struct buf		*bbp;
	register struct qb_diskst	*st;

	if (unit >= Nld || qi == 0 || qi->qi_alive == 0) {
		bp->b_error = ENXIO;
		goto bad;
	}
	st = &ldst[qi->qi_unit];
	if (bp->b_blkno < 0 || 
	    (bp->b_blkno+((bp->b_bcount+511)>>9)) > st->st_size[part].nblocks) {
		if (bp->b_blkno == st->st_size[part].nblocks) {
		    bp->b_resid = bp->b_bcount;
		    goto done;
		}
		bp->b_error = EINVAL;
		goto bad;
	}

	/* controller does 1024 byte transfers; use a padded kernel buffer */
	if (bp->b_bcount & LD_SEC_MASK) {
		bbp = (struct buf *)getqeblk((bp->b_bcount+LD_SEC_MASK)
			& ~LD_SEC_MASK, BQUALIFY(bp->b_dev));
		bbp->b_dev = bp->b_dev;
		bbp->b_blkno = bp->b_blkno;
		bbp->b_flags |= (bp->b_flags & (B_READ|B_WRITE));
		if ((bp->b_flags & B_READ) == B_WRITE) {
			bpcopyin(bbp->b_un.b_addr, bp);
			bzero((char *)bbp->b_un.b_addr + bp->b_bcount,
			    LD_SEC_SIZE - (bp->b_bcount&LD_SEC_MASK));
		}
		ldstrat(qi, bbp);
		biowait(bbp);
		if (bp->b_flags & B_READ)
			bpcopyout(bbp->b_un.b_addr, bp);
		bp->b_flags |= (bbp->b_flags & B_ERROR);
		bp->b_resid = bbp->b_resid;
		bbp->b_dev = NODEV;
		bfree(bbp);
		brelse(bbp);
		goto done;
	}
	ldstrat(qi, bp);
	return;
bad:	bp->b_flags |= B_ERROR;
done:	iodone(bp);
}

ldstrat(qi, bp)
	register struct qb_device	*qi;
	register struct buf		*bp;
{
	register struct qb_diskst	*st = &ldst[qi->qi_unit];
	register int			part = minor(bp->b_dev) & 07, s;
        register struct buf *dp = &ldutab[qi->qi_unit];
        register struct qb_ctlr *qm = qi->qi_mi;


	s = splx(qm->qm_psl);
	if (bp == &cldbuf[qi->qi_ctlr]) {/* do ioctl command first  */
		bp->av_forw = dp->b_actf;
		dp->b_actf = bp;
		if (dp->b_actl == NULL)
			dp->b_actl = bp;
	} else {
	    	bp->b_bn = bp->b_blkno +
			(st->st_size[part].cyloff * st->st_nspc);
		disksort(dp, bp);
	}
        if (dp->b_active == 0) {
                dp->b_forw = NULL;
                dp->b_active = 1;
                if (qm->qm_tab.b_actf == NULL)
                        qm->qm_tab.b_actf = dp;
                else
                        qm->qm_tab.b_actl->b_forw = dp;
                qm->qm_tab.b_actl = dp;
                if (qm->qm_tab.b_active == 0)
                        ldstart(qm);
        }
	splx(s);
}

/*
 * Start up a transfer on a drive.
 */
ldstart(qm)
	register struct qb_ctlr		*qm;
{
	register struct buf		*bp;
	register struct qb_device	*qi;
	register struct lddevice	*ldaddr;
	register struct LD_stat		*stat;
	int				i;

	if ((bp = qm->qm_tab.b_actf->b_actf) == NULL)
		return;
	qm->qm_tab.b_active++;			/* mark controller busy */
	qi = lddinfo[ldunit(bp)];
	stat = &LD_stat[qm->qm_ctlr];
	ldaddr = (struct lddevice *)qm->qm_addr;

	if (bp == &cldbuf[qm->qm_ctlr]) {	/* issue ioctl command */
		for (i = 0 ; i < 5 ; i++)
			ldaddr->ld_cmd[i] = stat->ld_cmd[i];
		i = (int)bp->b_un.b_addr;
		ldaddr->ld_ba[0] = byteswap(byte0(i), byte1(i));
		ldaddr->ld_ba[1] = byteswap(byte2(i), byte3(i));
		ldaddr->ld_cs = (TARGET(bp->b_dev)<<LDCS_TID_SHIFT)|LDCS_IE|LDCS_GO;
		return;
	}

	stat->ld_bn = bp->b_bn>>(LD_SEC_SHIFT-9);
	stat->ld_bleft = bp->b_bcount;
	stat->ld_dev = bp->b_dev;
	if (bp->b_flags & B_READ)
		stat->ld_command = LD_READ;
	else
		stat->ld_command = LD_WRITE;
	qbgo(qi);
}


lddgo(qm)
register struct qb_ctlr *qm;
{
	LD_stat[qm->qm_ctlr].ld_addr = VDMA_STD(qm->qm_qbinfo,vbnum);
	ldio((struct lddevice *)qm->qm_addr, &LD_stat[qm->qm_ctlr]);
}


ldio(ldaddr, stat)
	register struct lddevice	*ldaddr;
	register struct LD_stat		*stat;
{
	stat->ld_bpart = (stat->ld_bleft>LD_MAXCNT)? LD_MAXCNT: stat->ld_bleft;
	ldcommand(ldaddr->ld_cmd, stat->ld_command, 
		byte1(stat->ld_bn), byte2(stat->ld_bn), byte3(stat->ld_bn), 
		stat->ld_bpart>>LD_SEC_SHIFT, 0, 0, 0, 0, 0);
	ldaddr->ld_ba[0] = byteswap(byte0(stat->ld_addr), byte1(stat->ld_addr));
	ldaddr->ld_ba[1] = byteswap(byte2(stat->ld_addr), byte3(stat->ld_addr));
	dprintf(("ldio cmd=%x,%x,%x,%x,%x ba=%x,%x\n", ldaddr->ld_cmd[0],
		ldaddr->ld_cmd[1],ldaddr->ld_cmd[2],ldaddr->ld_cmd[3],
		ldaddr->ld_cmd[4],ldaddr->ld_ba[0],ldaddr->ld_ba[1]));
	ldaddr->ld_cs = (TARGET(stat->ld_dev)<<LDCS_TID_SHIFT)|LDCS_IE|LDCS_GO;
}

ldintr(ctlr)
	register int			ctlr;
{
	register struct qb_ctlr		*qm = ldcinfo[ctlr];
	struct LD_stat 			*stat = &LD_stat[ctlr];
	register struct buf		*bp, *dp;
	register struct lddevice	*ldaddr=(struct lddevice *)qm->qm_addr;
	register struct qb_device	*qi;

	stat->ld_wticks = 0;
        dp = qm->qm_tab.b_actf;
        bp = dp->b_actf;
	qi = lddinfo[ldunit(bp)];


	/* Check for and process errors */
	if ((ldaddr->ld_cs & LDCS_OK_MASK) != LDCS_OK) {
	    if (++qm->qm_tab.b_errcnt > LD_NRETRY) {
	    	if ((ldaddr->ld_cs&LDCS_HA_SNS) && ((ldaddr->ld_cmd[1]&0xf)<=1))
                        ;
		      /*printf("ld%d: SOFT ERROR\n", ldunit(bp)); */
		else {
			harderr(bp, "ld");
			bp->b_flags |= B_ERROR;
		}
	    } else if (bp != &cldbuf[ctlr])
		qm->qm_tab.b_active = 0;	/* force retry */
	}

	/* If still ``active'', then don't need any more retries.  */
	if (qm->qm_tab.b_active) {
	    /* check if more data from previous request on non ioctl commands */
	    if ( bp != &cldbuf[ctlr] &&
		(bp->b_flags&B_ERROR)==0 && (stat->ld_bleft-=stat->ld_bpart)>0){
		stat->ld_addr += stat->ld_bpart;
		stat->ld_bn += (stat->ld_bpart>>LD_SEC_SHIFT);
		ldio(ldaddr, stat);
		return;
	    }
	    bp->b_resid =  stat->ld_bleft;
	    dk_busy &= ~(1 << qi->qi_dk);
    	    qm->qm_tab.b_active = qm->qm_tab.b_errcnt = 0;
            qm->qm_tab.b_actf = dp->b_forw;
            dp->b_active = 0;
            dp->b_actf = bp->av_forw;
            if (dp->b_actf) {
                    dp->b_forw = NULL;
                    dp->b_active = 1;
                    if (qm->qm_tab.b_actf == NULL)
                            qm->qm_tab.b_actf = dp;
                    else
                            qm->qm_tab.b_actl->b_forw = dp;
                    qm->qm_tab.b_actl = dp;
            }
	    iodone(bp);					/* operation done */
	}

	qbdone(qm);

	/* If controller not transferring and have request, start controller. */
	if (qm->qm_tab.b_actf && qm->qm_tab.b_active == 0)
		ldstart(qm);
}

ldread(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	register int	unit = UNIT(dev);

	if (unit >= Nld)
		return (ENXIO);
	return (physio(ldstrategy, &rldbuf[unit], dev, B_READ, ldminphys, uio));
}

ldwrite(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	register int	unit = UNIT(dev);

	if (unit >= Nld)
		return (ENXIO);
	return (physio(ldstrategy, &rldbuf[unit], dev, B_WRITE, ldminphys, uio));
}

/*
 * Reset driver after lost interupt. Cancld software state of all pending
 * transfers and restart all units and the controller.
 */
ldreset(ctlr)
{
	register struct qb_ctlr		*qm = ldcinfo[ctlr];
	register struct LD_stat		*stat = &LD_stat[ctlr];
	register struct lddevice	*ldaddr =(struct lddevice *)qm->qm_addr;

	printf(" LD%d", ctlr);
	qm->qm_tab.b_active = 0;
	stat->ld_bleft = 0;
	stat->ld_bpart = 0;
	ldwait(ldaddr);
	ldstart(qm);
}

/*
 * Wake up every LD_TIMEOUT seconds and if an interrupt is pending but nothing 
 * has happened increment a counter. If nothing happens for 10 times, reset 
 * controler and begin anew.
 */
ldwatch()
{
	register struct qb_ctlr	*qm;
	register int		ctlr, unit;
	register struct LD_stat *stat;

	timeout(ldwatch, (caddr_t)0, LD_TIMEOUT*hz);
	for (ctlr = 0; ctlr < NLD; ctlr++) {
		qm = ldcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		stat = &LD_stat[ctlr];
		stat->ld_wticks++;
		if (stat->ld_wticks >= 10) {
			stat->ld_wticks = 0;
			printf("LD%d: lost interrupt\n", ctlr);
			ldreset(ctlr);
		}
	}
}

lddump(ldaddr)
{
	return(ENODEV);
}

ldsize(dev)
	dev_t				dev;
{
	register int			unit = UNIT(dev);
	register struct qb_device	*qi;

	if (unit >= Nld || (qi = lddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	return (ldst[qi->qi_unit].st_size[minor(dev)&0x7].nblocks);
}

static
ldwait(ldaddr)
	struct lddevice	*ldaddr;
{
	int		i = 0;

	while (ldaddr->ld_cs == LDCS_BUSY)
		if (i++ == LD_WAIT)
			return (-1);
	return (0);
}

ldminphys(bp)
	struct buf	*bp;
{
	bp->b_bcount = MIN(bp->b_bcount, LD_MAXCNT);
}

ldioctl(dev, cmd, data)
	register u_short	*data;
	register int		cmd;
	dev_t			dev;
{
	int				unit = UNIT(dev), xxx, error = 0;
	register int			i, s;
	register struct lddevice	*ldaddr = LDADDR(dev);
	register struct qb_device	*qi;
	register struct buf		*bp;
	register u_short		*p1 = (u_short *)ldaddr, *cmdpkt;

	if (unit >= Nld || (qi = lddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	if (cmd == LD_IOC_STAT) {		/* driver pseudo command */
		for (i=0; i < 8; i++)
			*data++ = *p1++;
		return (0);
	}

	bp = &cldbuf[qi->qi_ctlr];		/* get ioctl buffer header */
	s = splx(qi->qi_mi->qm_psl);
	while (bp->b_flags&B_BUSY) {
		bp->b_flags |= B_WANTED;
		sleep((caddr_t)bp, PRIBIO);
	}
	bp->b_flags = B_BUSY|B_READ;
	splx(s);
/* KLUDGE: the following code is WRONG!  The structure 'data' is on the system
stack; by the time we issue the command we could be running on another uarea.
This could CRASH the system with a trashed stack. 'data' should be coppied to 
a per controller array, and b_addr should point to that.  CTH */
/* TODO: VDMA */
	bp->b_un.b_addr = (caddr_t)svtop(data);
	bp->b_dev = dev;
	cmdpkt = LD_stat[qi->qi_ctlr].ld_cmd;

	switch (cmd) {
	    case LD_IOC_TSTRDY:
	    case LD_IOC_RZRO:
	    case LD_IOC_RST:
	    case LD_IOC_FOLOUP:
	    case LD_IOC_VRSN:
	    case LD_IOC_RDIAG:
	    case LD_IOC_RCAP:
		ldcommand(cmdpkt, cmd&0xFF, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		break;

	    case LD_IOC_SDIAG:
		ldcommand(cmdpkt, cmd&0xFF, 0x04, 0, 0, 0, 0, 0, 0, 0, 0);
		break;

	    case LD_IOC_INQ:
		ldcommand(cmdpkt, cmd&0xFF, 0, 0, 0, 0x08, 0, 0, 0, 0, 0);
		break;

	    case LD_IOC_MSEL:
		ldcommand(cmdpkt, cmd&0xFF, 0, 0, 0, 0x10, 0, 0, 0, 0, 0);
		break;

	    case LD_IOC_STSP:
		ldcommand(cmdpkt, cmd&0xFF, 0, 0, 0, (int)data, 0, 0, 0, 0, 0);
		break;

	    case LD_IOC_SENSE:
	    case LD_IOC_MDSN:
		ldcommand(cmdpkt, cmd&0xFF, 0, 0, 0, 0x10, 0, 0, 0, 0, 0);
		ldstrategy(bp);
		iowait(bp);
		if (ldaddr->ld_cs&LDCS_OK_MASK != LDCS_OK)
			goto err;
		for (i=0; i < 4; i++)
			*data++ = *p1++;
		bp->b_flags = B_BUSY|B_READ;
		ldcommand(cmdpkt, LD_FOLOUP, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		p1 = (u_short *)ldaddr;
		break;

	    default:
		goto err;
	}
	ldstrategy(bp);
	iowait(bp);
	if (cmd&IOC_OUT)
		for (i=0; i < 4; i++)
			*data++ = *p1++;
	if (ldaddr->ld_cs&LDCS_OK_MASK != LDCS_OK)
  err:		error = ENXIO;
	if (bp->b_flags&B_WANTED)		/* wakeup other ioctl's */
		wakeup((caddr_t)bp);
	bp->b_flags &= B_ERROR;
	return (error);
}
#endif	NLD > 0
