#include "../machine/pte.h"
#include "param.h"
#include "conf.h"
#include "user.h"
#include "proc.h"
#include "file.h"
#include "ioctl.h"
#include "tty.h"
#include "map.h"
#include "buf.h"
#include "vm.h"
#include "../is68kdev/qbvar.h"

#include "../machine/board.h"

#include "../dev/openchip.h"
#include "../dev/video.h"
#include "console.h"
#include "keybd.h"
#ifdef  SYSV
#include "../sysv/sys/termio.h"
#endif  SYSV

#define CON_DEBUG
int con_debug = 0;

#define NKBDS		3	/* mouse, keboard, and scancode multiplexed */
#define KEYBUNIT	0	/* keyboard unit number */
#define MOUSEUNIT	1	/* mouse unit number */
#define XWINUNIT	2	/* X Windows multiplexed unit number */
#define MOUSECODE	0x80000000	/* This is a mouse key */

#define NORM		1
#define INESC		2
#define INFF		3

struct tty      con_tty[NKBDS];
#define	IFLAGS	(EVENP|ODDP|ECHO|XTABS|CRMOD);

#ifdef  SYSV
int             sv_kbproc();
#endif  SYSV
int             kbstart();
int             kbparam();
unsigned int    con_getc();

/* #define CTRLT	/* */


unsigned int    sctoascii();


keybinit()
{
	/* reset the keyboard */
	InitKey();
}

int             keyerr = 0;	/* Count of framing errors from keyboard */
static          curkey = 0;	/* Current matrix code               */
static          ascii;		/* Ascii code of current key         */
static          inesc = 0;	/* No active escape sequence         */
static          escff = 0;
static          kstat = 0;	/* special key status                */

short           m2a[] =		/* matrix position to ascii table    */
{
 -1, 27, '1', '2', '3', '4', '5', '6',
 '7', '8', '9', '0', '-', '=', DELK, TABK,
 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i',
 'o', 'p', '[', ']', CRK, CTRLK, 'a', 's',
 'd', 'f', 'g', 'h', 'j', 'k', 'l', ';',
 '\'', '`', SHFTK, '\\', 'z', 'x', 'c', 'v',
 'b', 'n', 'm', ',', '.', '/', SHFTK, HELPK,
 CAPSDOWN, ' ', CAPSUP, F1, F2, F3, F4, F5,
 F6, F7, F8, F9, F10, -1, DK, '7',
 '8', '9', '-', '4', '5', '6', '+', '1',
 '2', '3', '0', '.', CRK, '*', '#', CUPK,
 CUDK, CULK, CURK, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ
};

short           m2da[] =	/* matrix position to diamond ascii table    */
{
 -1, -1, -1, -1, 0x9c, 0x9b, 0xf6, -1,	/* pound, cent, divide */
 -1, 0xfa, -1, -1, 0xf1, 0xf0, -1, -1,	/* bullet, plus/minus, equivalent */
 -1, -1, -1, -1, -1, 0x9d, -1, -1,	/* yen */
 -1, -1, 0xae, 0xaf, -1, -1, -1, -1,	/* left guilemot, right guilemot */
 -1, -1, -1, -1, -1, -1, 0xbb, 0xac,	/* lightning bolt, 1/4 */
 0xab, 0xf7, -1, -1, -1, -1, -1, -1,	/* 1/2, approximately equal to */
 -1, -1, -1, 0xf3, 0xf2, 0xbc, -1, -1,	/* <=, >=, 3/4 */
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, '_', PTRUK,
 PTRDK, PTRLK, PTRRK, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ
};

short	m2ua[] =	/* matrix position to upper case ascii table    */
{
 -1, 27, '!', '\@', '#', '$', '%', '^',
 '&', '*', '(', ')', '_', '+', DELK, TABK,
 'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I',
 'O', 'P', '{', '}', CRK, CTRLK, 'A', 'S',
 'D', 'F', 'G', 'H', 'J', 'K', 'L', ':',
 '"', '~', SHFTK, '|', 'Z', 'X', 'C', 'V',
 'B', 'N', 'M', '<', '>', '?', SHFTK, HELPK,
 CAPSDOWN, ' ', CAPSUP, F1, F2, F3, F4, F5,
 F6, F7, F8, F9, F10, -1, DK, '7',
 '8', '9', '-', '4', '5', '6', '+', '1',
 '2', '3', '0', '.', CRK, '*', '#', CUPK,
 CUDK, CULK, CURK, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ, ESCSEQ
};

short           a2c[] =		/* Ascii to control code table */
{
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 -1, -1, -1, -1, -1, -1, -1, -1,
 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
 -1, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, -1, -1, -1, -1, -1
};

/*
 * INITKEY
 * Initialise keyboard variables.                                             
 */

InitKey()
{
	int            c;
	int count;

	curkey = 0;		/* No current key   */
	kstat = 0;		/* Status of special keys */
	inesc = 0;		/* Not in an escape sequence */
	escff = 0;		/* Not an escaped 0xff */
	kb_reset();		/* Reset the keyboard */

	count = 0;
	while ((c = GetScan()) == 0x100){
		if (count++ == 10000)
			break;
	}
	if (c == 0x100)
		return (-1);
	else {
		OPENCHIP->op_kbdstat |= OP_KB_INEA;	/* Enable kb ints */
		return (0);
	}
}

/*
 * KBINTR
 * Keyboard interrupt routine.                                                
 */

kbintr()
{
	register unsigned int c;
	register struct tty *tp = &con_tty[0];

	while ((c = con_getc()) != 0) {
		if (c & MOUSECODE) {
			tp = &con_tty[MOUSEUNIT];
			c &= ~(MOUSECODE);
		} else {
			if (debug_input(c))
				continue;
			tp = &con_tty[KEYBUNIT];
		}

		if ((tp->t_state & (TS_ISOPEN | TS_WOPEN)) == 0
#ifdef  GWS
		    && (tp->t_line != KYBDDISC)
#endif  GWS
		    )
			return;

#ifdef SYSV
		if (tp->t_universe == UNIVERSE_SYSV) {
			if (tp->svt_iflag & SV_ISTRIP)
				c &= 0177;
			else
				c &= 0377;
		}
		if (tp->t_universe == UNIVERSE_SYSV)
			(*sv_linesw[tp->svt_line].l_input) (c, tp, 0);
		else {
#endif	SYSV

#if NBK > 0 || Nbk > 0
			if (tp->t_line == NETLDISC) {
				c &= 0177;
				BKINPUT(c, tp);
			} else
#endif
				(*linesw[tp->t_line].l_rint) (c, tp);
#ifdef	SYSV
		}
#endif SYSV
	}
}

unsigned int
con_getc()
{
	register unsigned key;
	register unsigned short modifier;
	register        s;

	/* Go round till no more keys */

	for (key = GetScan(); key < 0x100; key = GetScan()) {
		if (con_tty[XWINUNIT].t_state & TS_ISOPEN)
			xprocess(key);
		/* Are in the throws of processing an escape sequence? */
		if (inesc || (key == 0x7f && escff)) {
			if (key == 0xff) {	/* End of escape sequence */
				inesc = 0;
				escff = 1;	/* May be the start of
						 * 0x7f,0xff sequence */
				continue;
			}
			if (escff)
				key = 0xff;
			inesc = 1;
			escff = 0;	/* Can't be in above sequence */
			key |= MOUSECODE;
			return (key);
			/*
			 * MoveMouse (key); continue; 
			 */
		}
		/* Is this the start of something big? */

		if (key == 0x7d) {
			keyerr++;
			continue;
		}
		modifier = m2a[key & 0x7f];
		if (modifier == ESCSEQ && key < 0x80) {
			inesc = 1;	/* Start of escape sequence */
			continue;
		}
		escff = 0;	/* possible sequence of 0xff,0x7f interrupted */

		/* Is this the shift key? */

		if (modifier == SHFTK || modifier == CTRLK || modifier == DK) {
			if (key & 0x80)
				kstat &= ~modifier;	/* modifier gone up   */
			else
				kstat |= modifier;	/* "       "  down */
#ifdef CON_DEBUG
			if (con_debug > 5)
				printf("(shift/control/diamond)");
#endif CON_DEBUG
			goto kexit;
		}
		if (modifier == CAPSUP) {	/* Caps lock key changed */
			if (key & 0x80)	/* Caps lock key gone up */
				kstat &= modifier;	/* Caps lock off  */
#ifdef CON_DEBUG
			if (con_debug > 5)
				printf("(capsup)");
#endif CON_DEBUG
			goto kexit;
		}
		if (modifier == CAPSDOWN) {	/* Caps lock key changed */
			if (key & 0x80)	/* Caps lock key gone up */
				kstat |= modifier;	/* Caps lock on  */
#ifdef CON_DEBUG
			if (con_debug > 5)
				printf("(capsdown)");
#endif CON_DEBUG
			goto kexit;
		}
		if (key == curkey) {	/* Means 2 down or up codes in a row
					 * for same key */
#ifdef CON_DEBUG
			if (con_debug > 5)
				printf("(dup)");
#endif CON_DEBUG
			goto kexit;	/* something is wrong, so forget it */
		}
		/* key-up or key-down code received         */
		if (key == (curkey | 0x80))	/* current key has gone up */
			curkey = 0;	/* and zero curkey */

		if (key < 128) {/* new key has gone down so.. */
			curkey = key;
#ifdef CON_DEBUG
			if (con_debug > 5)
				printf("(newkey%X))", key);
#endif CON_DEBUG
			return (sctoascii(key));
		}
kexit:		;
	}			/* End for */
	return (0);
}

/*
 * Change scan code to ascii 
 */

unsigned int
sctoascii(mcode)
register byte   mcode;
{

	keyerr = 0;
	if (kstat & SHFTK)
		ascii = m2ua[mcode];	/* get upper case ascii */
	else
		ascii = m2a[mcode];	/* Get ascii code from matrix address */

	if (kstat & DK)
		ascii = m2da[mcode];	/* get diamond key ascii */

	if (ascii < 0)
		return (0);	/* There is no corresponding code */

	if (ascii < 256) {	/* Standard extended ascii code   */
		if (kstat & CTRLK)
			ascii = a2c[ascii];	/* Get ctrl code */
		if (ascii < 0)
			return (0);	/* Invalid ctrl code */
		if (kstat & CAPSLK && (ascii >= 'a' && ascii <= 'z'))
			ascii &= 0xdf;	/* Upper case alphabetics */
	}
#ifdef	CTRLT
	if (ascii == 'T' - '\@') {
		stack_trace();
		return (0);
	}
#endif	CTRLT

	return (ascii);
}

/*
 * GETSCAN a key-up, key-down code                                               *
 */

GetScan()
{
	register struct openchip *oc = OPENCHIP;
	register int    ch = 0x100;
	register byte   stat, dummy;

	if ((stat = oc->op_kbdstat) & OP_KB_RDY) {
		if (stat & OP_KB_FRM) {
			dummy = oc->op_kbdata;
			++keyerr;
		} else
			ch = oc->op_kbdata;
	}
	if (stat & OP_KB_OVFL) {
		oc->op_kbdstat = stat & (OP_KB_OVFL | OP_KB_INEA);
		keyerr++;
	}
	return (ch);
}

/*
 * KB_RESET
 * Reset the keyboard. Uses clock interrupts to time the reset
 * pulse.         
 */

kb_reset()
{
	char            c;
	int             i;

	Q_OP_PORT2 = OP2_SET_KEYRESET;	/* reset keyboard */
	for (i = 50000; i > 0; i--);
	Q_OP_PORT2 = OP2_RES_KEYRESET;	/* reenable keyboard */

	c = OPENCHIP->op_kbdata;
	OPENCHIP->op_kbdstat = OP_KB_OVFL;	/* clear overflow bit in case */

}

/*
 * unix device stuff 
 */
kbopen(dev, flag)
dev_t           dev;
int             flag;
{
	register int    unit = minor(dev);
	register struct tty *tp = &con_tty[unit];
	int             s, old_discipline;

#ifdef CON_DEBUG
	if (con_debug > 2)
		printf("kbopen(%X, %X)\n", dev, flag);
#endif CON_DEBUG
	keybinit();

	tp->t_oproc = kbstart;
	tp->t_state |= TS_CARR_ON;

#ifdef  SYSV
	tp->svt_proc = sv_kbproc;
#endif  SYSV
	if ((tp->t_state & TS_ISOPEN) == 0) {
		ttychars(tp);
		tp->t_ospeed = tp->t_ispeed = B9600;
		tp->t_flags = IFLAGS;
#ifdef  SYSV
		/*
		 * sv_ttinit initializes svt_line to zero.  This is a very
		 * bad thing to do on the graphics console because write will
		 * then go to the keyboard instead of the screen causing all
		 * sorts of havoc. The line discipline has been setup during
		 * GP_initialize(). Go check for yourself if you don't
		 * believe me. 
		 */
		old_discipline = tp->svt_line;
		sv_ttinit(tp);
		tp->svt_line = old_discipline;
#endif  SYSV
	} else if ((tp->t_state & TS_XCLUDE) &&
		   ((u.u_uid != 0) || (u.u_procp->p_flag & SLOGIN)))
		return (EBUSY);

	s = spltty();
#ifdef TTY_NODELAY
	if ((flag & FNDELAY) == 0)
#endif
		while ((tp->t_state & TS_CARR_ON) == 0) {
			tp->t_state |= TS_WOPEN;
#ifdef  SYSV
			map_state(tp, TOSV);
#endif  SYSV
			sleep((caddr_t) & tp->t_rawq, TTIPRI);
			if ((tp->t_state & TS_XCLUDE) && (u.u_procp->p_flag & SLOGIN)) {
				splx(s);
				return (EBUSY);
			}
		}
	splx(s);
#ifdef  SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_open) (dev, tp, flag));
	else
#endif  SYSV
	{
#ifdef CON_DEBUG
		if (con_debug > 5)
			printf("calling lineswitch(%X, %a)\n", dev, tp);
#endif CON_DEBUG
		s = (*linesw[tp->t_line].l_open) (dev, tp);
#ifdef CON_DEBUG
		if (con_debug > 5)
			printf("lineswitch returned %X\n", s);
#endif CON_DEBUG
		return (s);
		/*
		 * return ((*linesw[tp->t_line].l_open)(dev, tp)); 
		 */
	}
}

kbclose(dev, flag)
dev_t           dev;
{
	register struct tty *tp = &con_tty[minor(dev)];

	tp->t_state &= ~TS_CARR_ON;
#ifdef  SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		(*sv_linesw[tp->svt_line].l_close) (tp);
	else
#endif  SYSV
		(*linesw[tp->t_line].l_close) (tp);
#ifdef  SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		sv_ttyclose(tp);
	else
#endif  SYSV
		ttyclose(tp);

}

kbread(dev, uio)
dev_t           dev;
struct uio     *uio;
{
	register struct tty *tp = &con_tty[minor(dev)];

#ifdef CON_DEBUG
	if (con_debug > 5)
		printf("kbread(%X, %a)\n", dev, uio);
#endif CON_DEBUG
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_read) (tp, uio));
	else
#endif	SYSV
		return ((*linesw[tp->t_line].l_read) (tp, uio));
}

kbwrite(dev, uio)
dev_t           dev;
struct uio     *uio;
{
	register struct tty *tp = &con_tty[minor(dev)];

#ifdef  SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_write) (tp, uio));
	else
#endif  SYSV
		return ((*linesw[tp->t_line].l_write) (tp, uio));
}



kbioctl(dev, cmd, addr, flag)
dev_t           dev;
caddr_t         addr;
{
	register int    error, unit;
	register struct tty *tp;

	tp = &con_tty[(unit = minor(dev))];
#ifdef  SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV) {
		if ((error = sv_ttiocom(tp, cmd, addr)) > 0)
			return error;
		if (cmd == TCSETAF || cmd == TCSETA || cmd == TCSETAW);	/* ??? */
		if (cmd == TCSBRK);	/* ??? */
		return 0;
	}
#endif  SYSV
	if ((error = (*linesw[tp->t_line].l_ioctl) (tp, cmd, addr, flag)) >= 0)
		return (error);
	if ((error = ttioctl(tp, cmd, addr, flag)) >= 0) {
		if (cmd == TIOCSETP || cmd == TIOCSETN ||
		    cmd == TIOCLSET || cmd == TIOCLBIS || cmd == TIOCLBIC);
		return (error);
	}
	switch (cmd) {
	case TIOCSDTR:
		;		/* ??? */
		break;

	case TIOCCDTR:
		;		/* ??? */
		break;

		/*
		 * TODO: support these case TIOCSBRK: case TIOCCBRK: case
		 * TIOCMSET: case TIOCMBIS: case TIOCMBIC: case TIOCMGET: /* 
		 */

	default:
		return (ENOTTY);
	}
	return (0);
}

#ifdef	SYSV
sv_kbproc(tp, cmd)
register struct tty *tp;
register        cmd;
{
	sv_proc(tp, cmd, kbstart, kbparam);
}
#endif	SYSV

kbstart(tp)
register struct tty *tp;
{
	register int    c;
	int             s;

	s = spltty();
	if (tp->t_state & (TS_TIMEOUT | TS_BUSY | TS_TTSTOP))
		goto out;
	if (tp->t_outq.c_cc <= TTLOWAT(tp)) {
		if (tp->t_state & TS_ASLEEP) {
			tp->t_state &= ~TS_ASLEEP;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			wakeup((caddr_t) & tp->t_outq);
		}
		if (tp->t_wsel) {
			selwakeup(tp->t_wsel, tp->t_state & TS_WCOLL);
			tp->t_wsel = 0;
			tp->t_state &= ~TS_WCOLL;
		}
	}
	if (tp->t_outq.c_cc == 0)
		goto out;
	tp->t_state |= TS_BUSY;
#ifdef	SYSV
	map_state(tp, TOSV);
#endif	SYSV
	c = getc(&tp->t_outq);
	printf("%c", c);
out:	splx(s);
}

kbparam()
{
}

unsigned int con_state = NORM;

xprocess(key)
register unsigned int key;
{
	register unsigned short modifier;
	register        s;

	switch (con_state){
	case INESC:
		switch (key){
			case 0xFF:
				con_state = INFF;
				return;
			default:
				mouseproc(key);
				return;
		}
		return;
	case INFF:

		if (key == 0x7F){
			key = 0xFF;
			mouseproc(key);
			con_state = INESC;
			return;
		}
		if (key == 0x7D){
			keyerr++;
			return;
			/* change to norm? */
		}

	case NORM:
		modifier = m2a[key & 0x7f];

		switch (modifier) {
		case ESCSEQ:
			if (key < 0x80){
				con_state = INESC;
				return;
			}
#ifdef CON_DEBUG
			if (con_debug > 1)
				printf("escap seq twice\n");
#endif CON_DEBUG
			emit(key);
			return;

		default:
			emit(key);
		}	
	}	
}

struct 	ioevent mouseevent, keybdevent;
extern struct timeval time;

output_packet(packet)
struct ioevent *packet;
{

		register unsigned char *bp = (unsigned char *)packet;
		register unsigned int i;
		register struct tty *tp = &con_tty[XWINUNIT];

		packet->timestamp = time;	/* structure assignment */
		
#ifdef CON_DEBUG
		if (con_debug > 5){
			printf("type: %X, time: %X,%X\n",
				packet->type, packet->timestamp.tv_sec,
				packet->timestamp.tv_usec);
			printf("data1: %X, data2: %X, data3: %X\n",
				packet->data1, packet->data2, packet->data3);
		}
#endif CON_DEBUG

#ifdef	SYSV
		if (tp->t_universe == UNIVERSE_SYSV){
			for (i = 0; i < sizeof(struct ioevent); i++)
				(*sv_linesw[tp->svt_line].l_input)
				    (*bp++, tp, 0);
		} else {
#endif	SYSV
			for (i = 0; i < sizeof(struct ioevent); i++)
#ifdef notdef
				(*linesw[tp->t_line].l_rint) (*bp++, tp);
#endif notdef
				xdrint(*bp++, tp);
#ifdef	SYSV
		}
#endif	SYSV

}

/*
 * note that this in not reentrant
 */
emit(key)
unsigned int key;
{
	keybdevent.type = KEYBDEVENT;
	keybdevent.data1 = key;
	output_packet(&keybdevent);
}


/*
 * MOUSEPROC called from an interrupt routine, therefore is not re-entered       */
static byte     store[3];
static          sin = 4;

mouseproc(value)
register byte   value;
{
	register        newkstat, bchange, leftbut, rightbut;


	keyerr = 0;
	if (value & 0x80)	/* 1st byte has bit 6 set */
		sin = 0;

	/* Only need to move if mouse has moved */
	if (sin < 3)
		store[sin++] = value;	/* save it */

	if (sin == 3) {
		sin = 4;	/* In case things go adrift */

		mouseevent.data1 =
		    (char)((store[1] & 0x7f) | (((char) store[0] & 0x8) << 4));
		mouseevent.data2 =
		    (char)((store[2] & 0x7f) | (((char) store[0] & 0x10) << 3));

		/* Update button status */
		/* New value */
		newkstat = kstat & ~MMASK | (int) (store[0] & 0x3) << 12;
#ifdef CON_DEBUG
		if (con_debug)
		    printf("%X  ", newkstat);
#endif CON_DEBUG
		/* Which btns have changed */
		bchange = kstat & MMASK ^ newkstat & MMASK;
		
		kstat = newkstat;	/* Update kstat */
		if (bchange & MLMASK) {
			if (kstat & MLMASK)
				leftbut = 1;
			else
				leftbut = 0;
#ifdef CON_DEBUG
		if (con_debug)
		    printf("!", newkstat);
#endif CON_DEBUG
		}
		if (bchange & MRMASK) {
			if (kstat & MRMASK)
				rightbut = 1;
			else
				rightbut = 0;
#ifdef CON_DEBUG
		if (con_debug)
		    printf("?", newkstat);
#endif CON_DEBUG
		}
#ifdef CON_DEBUG
		if (con_debug)
		    printf("\r");
#endif CON_DEBUG

		mouseevent.type = MOUSEVENT;
		mouseevent.data3 = 0;
		if (kstat & MRMASK)
			mouseevent.data3 |= 1;
		if (kstat & MLMASK)
			mouseevent.data3 |= 2;
		output_packet(&mouseevent);
	}
}

xdrint(c, tp)
unsigned int c;
struct tty *tp;
{
	register unsigned int s, error = 0;
	static unsigned short count = 0;
	static struct ioevent xdevent;
	char *bp = (char *)&xdevent;

#ifdef CON_DEBUG
	if (con_debug > 1)
		printf("%x)", c);
#endif CON_DEBUG
	bp[count++] = c; 
	if (tp->t_rawq.c_cc > TTYHOG) 
		ttyflush(tp, FREAD|FWRITE);
	else {
		if (putc(c, &tp->t_rawq) != 0)
			ttwakeup(tp);
	}
	if (count == sizeof(struct ioevent)){
		count = 0;
#ifdef CON_DEBUG
	if (con_debug > 1)
		printf("w\n");
#endif CON_DEBUG
		ttwakeup(tp);
	}
}

