#define DEBUG
/*
 * VMEBUS: Integrated Solutions GP-SCSI tape driver
 */
#include "gt.h"
#if NGT > 0 || Ngt > 0
#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/file.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"
#include "../h/ioctl.h"
#include "../h/mtio.h"

#ifdef	VQX
#include "../is68kdev/openchip.h"
#endif	VQX
#include "../is68kdev/scsi.h"
#include "../is68kdev/gsreg.h"
#include "../is68kdev/gsvar.h"
#include "../is68kdev/qbvar.h"
#include "../machine/board.h"

/*		_________________________________
 * minor(dev):	| 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 *		---------------------------------
 *		\__flags___/ \_cont/ \__targ___/
 *			     \______ctlr_______/
 *			     \______unit_______/
 */
#define Ngt_targ_cont	   	8
#define Ngt_lun_targ	    	1
#define GT_CTLR(cont, targ)     (((cont) << 3) | ((targ) - 0))
#define GT_MKUNIT(ctlr, lun)	(ctlr)
#define GT_UNIT(dev)	    	(minor(dev) & 0x1F)
#define GTU_LUN(unit)	   	(0)
#define GTU_CTLR(unit)	  	(unit)
#define GTU_TARG(unit)	  	((GTU_CTLR(unit) & 7) + 0)
#define GTU_CONT(unit)	  	(GTU_CTLR(unit) >> 3)
#define GTF_REWIND		0x80
#define GTF_ISI			0x40
#define GTF_SWAB		0x20
#define GT_MAXCNT	       	0x1FE00
#define GT_MAXBA	       	0xFFFFFFFF

/*
 * State of controller from last command. Since only one command can be done
 * at a time per target device, allocate one for each device on each controller.
 */
struct gtstat	{	/* pseudoctlr status block */
	struct		{
		unsigned f_open		: 1;	/* device open */
		unsigned f_nodisc	: 1;	/* disallow disconnects */
		unsigned f_swcd		: 1;	/* swap command data */
		unsigned f_swmd		: 1;	/* swap media data */
		unsigned f_busy		: 1;	/* using raw buffer */
		unsigned f_want		: 1;	/* want to use raw buffer */
		unsigned f_sense	: 2;	/* times sense done on device */
		unsigned f_lastiow	: 1;	/* last i/o was a write */
		unsigned f_seeking	: 1;	/* seeking to block address */
		unsigned f_hdone	: 1;	/* isi header read/written */
		unsigned f_buffered	: 1;	/* using buffered mode */
		unsigned f_notraw	: 1;	/* opened as block device */
		unsigned f_reject	: 1;	/* not a valid device combo */
		unsigned f_lastrfm	: 1;	/* last i/o was read and a */
						/* file mark was found */
		unsigned f_		: 1;	/* */
	}		fbits;
#define				s_open		fbits.f_open
#define				s_nodisc	fbits.f_nodisc
#define				s_swcd	 	fbits.f_swcd
#define				s_swmd 		fbits.f_swmd
#define				s_busy 		fbits.f_busy
#define				s_want 		fbits.f_want
#define				s_sense		fbits.f_sense
#define				s_lastiow	fbits.f_lastiow
#define				s_seeking	fbits.f_seeking
#define				s_hdone		fbits.f_hdone
#define				s_buffered	fbits.f_buffered
#define				s_notraw	fbits.f_notraw
#define				s_reject	fbits.f_reject
#define				s_lastrfm	fbits.f_lastrfm
	struct tty      	*ttyp;  /* record users tty for errors */
	u_int			resid;	/* most recent residual count */
	u_int			fileno; /* current file number */
	u_int			fileba; /* current ba in file */
	u_int			blksz;	/* size tape blocks (0 if variable) */
	u_int			count;	/* count for current op */
	u_int			roundeven;	/* bytecount rounded up */
	u_int			dsreg;	/* saved status of last failed cmd */
	u_short			needreinit;	/* device needs reiniting */
	u_short			dtype;	/* brand of drive (for quirks) */
#define	EXABYTE	1
#define	WANGTEK	2
#define	ARCHIVE	3
	struct isihead	{
		u_short	ih_id;
		u_short	ih_fileno;
		u_short	ih_fileba;
		u_short	ih_blksz;
		u_short	ih_count;
	}			isihead; /* header block for isi tapes */
	int			isihead_vdma;
	struct scsi_blklim	blklim;	/* store block size limits */
	int			blklim_vdma;
	struct hacb_dcb 	dcb;    /* for direct commands to be done  */
	struct hacb_dcb 	*gt_dcb; 
	union scsi_sns	  	sense;
	int			sense_vdma;
	struct scsi_msen	msen;	/* to sense write protect during open */
	int			msen_vdma;
	struct buf      	rbuf; 
	struct buf      	qhd;   
	int			gt_free_vdma;	/* free vdma on done */
	int			gt_lun;
	char			gt_direct;
}		gtstat[Ngt];

#ifdef MULTILUN
int	     nextlun[NGT * Ngt_targ_cont];	  /* selected lun */
#endif MULTILUN

struct scsi_inq	gt_inq;
int		gt_inq_vdma;
int		gt_stat_vdma;

int     gtprobe(),      gtslave(),      gtattach(),     gtintr(), 	gtdgo();
int     gtgetcmd(),     gtdoerr(),      gtminphys();

struct qb_device	*gtdinfo[Ngt];
struct qb_ctlr		*gtcinfo[NGT];
extern u_short		*GSstd[];
/*
 * gt/gt controller/unit driver structure
 */
struct  qb_driver GTdriver =
	{gtprobe, gtslave, gtattach, gtdgo, GSstd, "gt", gtdinfo,"GT", gtcinfo};
/*
 * gtq is used to tell gs routines that we have commands for it, and to pass
 * pointers to the routines for getting commands and command completion.
 */
struct gs_queue	gtq[NGT * Ngt_targ_cont];

/* Check that controller exists, initialize it, fake it for pseudocontroller */
gtprobe(gtaddr, cont)
	u_short *gtaddr;
{
	return (gsprobe(gtaddr, cont));
}

/* Check that a given unit exists */
gtslave(qi, gtaddr)
	struct qb_device  *qi;
	u_short	    *gtaddr;
{
	int			unit = qi->qi_unit;
	int			ctlr = GTU_CTLR(unit);
	register struct gtstat *stat = &gtstat[unit];

	stat->s_nodisc = (qi->qi_flags & 1) ? DCB_NODISC : DCB_DISC;
	stat->s_swcd = (qi->qi_flags & 2) ? DCB_NOSW : DCB_SW;
	stat->s_swmd = (qi->qi_flags & 4) ? DCB_SW : DCB_NOSW;
	stat->s_buffered = (qi->qi_flags & 8) ? 0 : 1;
	gsqinit(&gtq[ctlr], gtgetcmd, gtdoerr);
        if (gt_inq_vdma == 0) {
                gt_inq_vdma = IOPB_STD(iopballoc(&gt_inq, sizeof(gt_inq)),vbnum);
                cache_inhibit(&gt_inq, sizeof(gt_inq));
                gt_stat_vdma = IOPB_STD(iopballoc(gtstat, sizeof(gtstat)),vbnum);
	}
        stat->sense_vdma = (int)&gtstat[unit].sense-(int)gtstat + gt_stat_vdma;
        stat->msen_vdma = (int)&gtstat[unit].msen - (int)gtstat + gt_stat_vdma;
        stat->blklim_vdma = (int)&gtstat[unit].blklim-(int)gtstat+gt_stat_vdma;
        stat->isihead_vdma =(int)&gtstat[unit].isihead-(int)gtstat+gt_stat_vdma;
	return gtinq(qi);
}

/* Logically attach a unit to a drive/controller, and set up unit info */
gtattach(qi)
	struct qb_device  *qi;
{
	int				unit = qi->qi_unit;
	int				lun = GTU_LUN(unit);
	register struct gtstat		*stat = &gtstat[unit];
	register struct hacb_dcb *dcb = &stat->dcb;
	int				count;

	printf("%8s %10s SCSI tape drive",
		gt_inq.inq_vendor, gt_inq.inq_product);
	if (!strncmp(gt_inq.inq_vendor,"EXABYTE ",8))
		stat->dtype = EXABYTE;
	if (!strncmp(gt_inq.inq_vendor,"WANGTEK ",8))
		stat->dtype = WANGTEK;
	if (!strncmp(gt_inq.inq_vendor,"ARCHIVE ",8))
		stat->dtype = ARCHIVE;
	SET_CDB_0(&dcb->dcb_cdb, CMD_BLKLIM, lun, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, stat->s_nodisc,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 
	    stat->blklim_vdma, sizeof(stat->blklim), 1);
	flush_cache_entry(&stat->blklim, sizeof(stat->blklim));
	if (gtissue(unit))
		printf("\nBLKLIM failed\n");
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, 0, sizeof(stat->msen), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, stat->s_nodisc,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 
	    stat->msen_vdma, sizeof(stat->msen), 1);
	flush_cache_entry(&stat->msen, sizeof(stat->msen));
	if (gtissue(unit))
		printf("\nMSENSE failed\n");
	stat->blksz = SCSI_HML(stat->msen.msen_bd.bd_bl);

	printf("\n        blocklimits: %x to %x;",
		stat->blklim.blklim_min, stat->blklim.blklim_max);
	if (stat->blksz)
		printf(" fixed blocksize %x;",stat->blksz);
	else
		printf(" variable blocksize;");
	printf(" density code %x",stat->msen.msen_bd.bd_density);

	stat->msen.msen_len = 0;
	stat->msen.msen_mtype = 0;
	stat->msen.msen_wprot = 0;
	stat->msen.msen_rsvd = stat->s_buffered * 0x10;
	stat->msen.msen_bdl = 8;
	stat->msen.msen_bd.bd_density = 0;
	SCSI_HML_SET(stat->msen.msen_bd.bd_nb, 0);
	SCSI_HML_SET(stat->msen.msen_bd.bd_bl, stat->blksz);
	count = 12;
	if (stat->dtype == EXABYTE) {
		/* evenbytedisconnect & parityenable */
		stat->msen.msen_pd.pd_code = 6;
		count = 14;
	}
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSELECT, lun, 0, count, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_OUT, stat->s_swcd, stat->s_nodisc,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 
	    stat->msen_vdma, count, 1);
	flush_cache_entry(&stat->msen, sizeof(stat->msen));
	if (gtissue(unit))
		printf("\nMSELECT failed\n");
	return(0);
}

gtissue(unit)
{
	int				ctlr = GTU_CTLR(unit);
	int				cont = GTU_CONT(unit);
	int				targ = GTU_TARG(unit);
	register struct gtstat		*stat = &gtstat[unit];
	register struct hacb_dcb 	*dcb = &stat->dcb;
	int				retrycount = 3;

    while (1) {
	stat->gt_direct = 1;
	if (gsfakeint(cont, targ, &gtq[ctlr], 800000) ||
	    dcb->dcb_err) { /* didn't complete or cerr or sense error */
		stat->s_sense = 0;
		return (-1);
	}
	if (stat->s_sense == 0) /* commmand completed ok */
		return (0);
	stat->s_sense = 0;
	switch (stat->sense.sns_7.sns_7_key) {
		case SNS_7_KEY_NO_SENSE:
		case SNS_7_KEY_RECOVERED:
			return(0);
		case SNS_7_KEY_UNIT_ATTN:
		case SNS_7_KEY_ABORT_CMD:
			if (--retrycount)
				continue;
		default:
			return(stat->sense.sns_7.sns_7_key);
	}
    }
}

gtinq(qi)
	struct qb_device		*qi;
{
	int				unit = qi->qi_unit;
	int				lun = GTU_LUN(unit);
	register struct gtstat	*stat = &gtstat[unit];
	register struct hacb_dcb	*dcb = &stat->dcb;

	SET_CDB_0(&dcb->dcb_cdb, CMD_INQUIRY, lun, 0,
	    sizeof(struct scsi_inq), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
	    gt_inq_vdma, sizeof(struct scsi_inq), 1);
	bzero(&gt_inq, sizeof(struct scsi_inq));
	flush_cache_entry(&gt_inq, sizeof(gt_inq));
	if (gtissue(unit) || (gt_inq.inq_pdtype != INQ_PDT_TAPE))
		return (0);
	return (1);
}

gtcommand(bp)
	register struct buf	*bp;
{

	bp->b_flags = B_WANTED;
	gtstrategy(bp);
	iowait(bp);
	return(bp->b_error);
}

gtintr(cont)
{
	return (gsintr(cont));
}

gtreset(unit)
{
}

gtsize(dev)
{
}

gtchropen(dev,flag) dev_t dev; { return(gtopen(dev,flag,0)); }
gtblkopen(dev,flag) dev_t dev; { return(gtopen(dev,flag,1)); }
gtopen(dev,flag,notraw)
	dev_t dev;
{
	int				unit = GT_UNIT(dev);
	int				lun = GTU_LUN(unit);
	register struct gtstat		*stat = &gtstat[unit];
	register struct buf		*bp;
	register struct hacb_dcb	*dcb;
	struct qb_device		*qi;

	if (unit >= Ngt || (qi = gtdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	if (stat->s_open)
		return (EBUSY);
	stat->s_open = 1;
	gtwait(stat);
	if ((notraw && (dev&GTF_ISI || stat->blksz != 512)) ||
	    (dev&GTF_ISI && stat->blksz != 512))
		stat->s_reject = 1; /* reject illegal combinations */
	stat->s_notraw = notraw;
	stat->s_swmd = dev&GTF_SWAB ? DCB_SW : DCB_NOSW;
	bp = &stat->rbuf;
	bp->b_dev = dev;
	stat->ttyp = u.u_ttyp;
	dcb = &stat->dcb;
	bp->b_un.b_addr = (caddr_t )dcb;

	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, 0, sizeof(stat->msen), 0, 0);
	SET_DCB(dcb, DCB_IE, DCB_DIR_IN, stat->s_swcd, stat->s_nodisc,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 
	    stat->msen_vdma, sizeof(stat->msen), 1);
	flush_cache_entry(&stat->msen, sizeof(stat->msen));
	if (gtcommand(bp)) {
		tprintf(stat->ttyp, "gt%d: MSENSE failed\n", unit);
		gtdone(stat);
		stat->s_open = 0;
		return (EIO);
	}
	if (((flag&(FREAD|FWRITE)) == FWRITE) && (stat->msen.msen_wprot)) {
		tprintf(stat->ttyp, "gt%d: write protected\n", unit);
		gtdone(stat);
		stat->s_open = 0;
		return (EIO);
	}
	stat->s_lastiow = 0;
	stat->s_lastrfm = 0;
	gtdone(stat);
	return (0);
}

gtclose(dev,flag)
	dev_t				dev;
{
	int				unit = GT_UNIT(dev);
	int				lun = GTU_LUN(unit);
	register struct gtstat 		*stat = &gtstat[unit];
	register struct buf		*bp = &stat->rbuf;
	register struct hacb_dcb	*dcb = &stat->dcb;

	gtwait(stat);
	bp->b_dev = dev;
	bp->b_un.b_addr = (caddr_t )dcb;
	if (flag == FWRITE || (flag & FWRITE) && stat->s_lastiow) {
		stat->fileba = 0;
		stat->fileno++;
		SET_CDB_0(&dcb->dcb_cdb, CMD_FILEMARK, lun, 0, 1, 0, 0);
		SET_DCB(dcb, DCB_IE, DCB_DIR_NO, stat->s_swcd, stat->s_nodisc,
	 	   sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 0, 0, 0);
		if (gtcommand(bp))
			tprintf(stat->ttyp, "gt%d: FILEMARK failed\n", unit);
	}
	if (((minor(dev) & GTF_REWIND) == 0) && (stat->fileba||stat->fileno)) {
		stat->fileba = 0;
		stat->fileno = 0;
		SET_CDB_0(&dcb->dcb_cdb, CMD_REZERO, lun, 0, 0, 0, 0);
		SET_DCB(dcb, DCB_IE, DCB_DIR_NO, stat->s_swcd, stat->s_nodisc,
	 	   sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 0, 0, 0);
		if (gtcommand(bp))
			tprintf(stat->ttyp, "gt%d: REWIND failed\n", unit);
	}
	stat->s_open = 0;
	stat->s_reject = 0;
	gtdone(stat);
	return (0);
}

gtread(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int		unit = GT_UNIT(dev);
	int		count = uio->uio_resid;
	int		rc;

	if (unit >= Ngt)
		return (ENXIO);
	rc = physio(gtstrategy, &gtstat[unit].rbuf, dev, B_READ,
		    gtminphys, uio);
	/*
	 * If the read produced zero bytes, then clear
	 * the last read was file mark flag.  The zero return
	 * to the user will indicate a file mark on this read.
	 * (see also: gtstrategy)
	 */
	if(count == uio->uio_resid) {
		gtstat[unit].s_lastrfm = 0;
	}
	return(rc);
}

gtwrite(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int		unit = GT_UNIT(dev);

	if (unit >= Ngt)
		return (ENXIO);
	return (physio(gtstrategy, &gtstat[unit].rbuf, dev, B_WRITE,
		gtminphys, uio));
}


gtgetcmd(cont, targ, dcb)
	register struct hacb_dcb	*dcb;
{
	register struct qb_ctlr *qm;
	register struct buf *bp;
	register int ctlr = GT_CTLR(cont,targ);
	register struct gtstat	*stat = &gtstat[ctlr];
	register struct qb_device *qi;
	register unit, lun;
	u_int		blocksize;

	if (stat->needreinit)
		return(gtreinit(dcb,stat,stat->gt_lun,1));

	if (stat->gt_direct) {
		stat->gt_lun = stat->dcb.dcb_cdb.cdb_0.cdb_0_lun;
		bcopy(&stat->dcb, dcb, sizeof(struct hacb_dcb));
		return(1);
	}

	if (((qm = gtcinfo[cont]) == NULL)
	    || (qm->qm_tab.b_actf == NULL) 
	    || ((bp = qm->qm_tab.b_actf->b_actf) == NULL)) {
		gsdeq(cont, targ, &gtq[ctlr]);
		return(0);
	}

	stat->s_lastiow = 0;
	unit = GT_UNIT(minor(bp->b_dev));
	stat->gt_lun = lun = GTU_LUN(unit);

	if (bp->b_un.b_addr == &stat->dcb) { /* is a direct simple command */
		bcopy(&stat->dcb, dcb, sizeof(struct hacb_dcb));
		return(1);
	}

	/* else must be a standard read/write */
	if (stat->s_reject || bp->b_bcount > GT_MAXCNT) {
		stat->resid = bp->b_bcount;
		gtiodone(stat,EINVAL);
		return(0);
	}
	if (stat->s_notraw && (stat->count = bp->b_blkno - stat->fileba)) {
		stat->s_seeking = 1; /* do a seek to correct block first */
		SET_CDB_0(&dcb->dcb_cdb,CMD_SPACE,lun,
			(stat->count>>8)&0xFFFF,stat->count&0xFF,0,0);
		SET_DCB(dcb,DCB_IE,DCB_DIR_NO,0,stat->s_nodisc,
			sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
 		return (1);
	}
	stat->roundeven = bp->b_bcount + (bp->b_bcount & 1);
	if (!(blocksize = stat->blksz)) blocksize++;
	stat->count =  ((stat->roundeven + blocksize - 1) / blocksize);
	if (bp->b_dev&GTF_ISI)
		if (isiheader(stat,bp,dcb,lun))
			return(1);

	qi = gtdinfo[GT_UNIT(minor(bp->b_dev))];
	qi->qi_mi = qm;
	stat->gt_dcb = dcb;
	stat->gt_free_vdma = (int)qm;
	qbgo(qi);
	return(2);
}

gtdgo(qm)
	register struct qb_ctlr *qm;
{
	register struct hacb_dcb	*dcb;
	int			unit;
	int			ctlr;
	int			cont;
	int			targ;
	int			lun;
	struct gtstat	*stat;
	struct buf	*bp;
	u_char		cmd, dir;

	bp = qm->qm_tab.b_actf->b_actf;
	unit = GT_UNIT(minor(bp->b_dev));
	cont = GTU_CONT(unit);
	targ = GTU_TARG(unit);
 	ctlr = GT_CTLR(cont, targ);
	lun = GTU_LUN(unit);
 	stat = &gtstat[GT_MKUNIT(ctlr, lun)];
	dcb = stat->gt_dcb;

	if (bp->b_flags & B_READ) {
		cmd = CMD_READ;
		dir = DCB_DIR_IN;
	} else {
		cmd = CMD_WRITE;
		dir = DCB_DIR_OUT;
	}
	SET_CDB_0(&dcb->dcb_cdb, cmd, lun,
		(stat->count>>8)|(stat->blksz?0x10000:0),stat->count&0xFF,0,0);
	SET_DCB(dcb, DCB_IE, dir, stat->s_swmd, stat->s_nodisc,
		sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
		VDMA_STD(qm->qm_qbinfo,vbnum), stat->roundeven, stat->blksz);
	gsrun(cont, targ);
	return(1);
}
/* do upper level interrupt processing */
gtdoerr(cont, targ, dcb)
	struct hacb_dcb *dcb;
{
	int		ctlr = GT_CTLR(cont, targ);
#ifdef MULTILUN
	int		lun = nextlun[ctlr];
#else MULTILUN
	int		lun = 0;
#endif MULTILUN
	int		unit = GT_MKUNIT(ctlr, lun);
	struct gtstat	*stat = &gtstat[ctlr];
	struct qb_ctlr *qm;
	struct buf	*bp;
	int		filemark = 0;
	int		key;

	stat->s_lastrfm = 0;
	if (dcb->dcb_err) {
	    if (dcb->dcb_scsi_status & SCSI_STATUS_BUSY)
		return (0); /* do it again! (be nice to sleep here) */
	    if (stat->s_sense > 2) {
		dcb->dcb_scsi_status = 0; /* handle in fatal error routine */
		dcb->dcb_cerr = 0xFF; /* handle in fatal error routine */
	    }

	    if ((dcb->dcb_scsi_status & SCSI_STATUS_CHECK) ||
                (dcb->dcb_cerr == DCB_CERR_NOSTAT)) {
		stat->s_sense++;
		SET_CDB_0(&dcb->dcb_cdb, CMD_SENSE, lun, 0,
		    sizeof(union scsi_sns), 0,0);
		SET_DCB(dcb, stat->s_open ? DCB_IE : DCB_NOIE, DCB_DIR_IN,
		    stat->s_swcd, stat->s_nodisc, sizeof(struct cdb_0),
		    DCB_BW_16, DCB_AM_STD_S_D, 0,
		    stat->sense_vdma, sizeof(union scsi_sns), 1);
		flush_cache_entry(&stat->sense, sizeof(stat->sense));
		return (1);
	    }
	}
	if (stat->gt_direct) { /* in autoconf procedures */
		stat->gt_direct = 0;
		stat->dcb.dcb_err = dcb->dcb_err;
		return(0);
	}
	if (stat->needreinit)
		return(gtreinit(dcb,stat,lun,0));
	if (dcb->dcb_err) { /* error in command, but sense not available */
		tprintf(stat->ttyp,"gt%d: FATAL ERROR:",unit);
		stat->resid = stat->count;
		stat->dsreg = (dcb->dcb_err << 15) |
			(dcb->dcb_cerr << 8) | dcb->dcb_scsi_status;
		if (dcb->dcb_scsi_status)
		    tprintf(stat->ttyp,"scsi status %b",
			dcb->dcb_scsi_status, SCSI_STATUS_BITS);
		else if (dcb->dcb_cerr)
		    tprintf(stat->ttyp," cerr %x",dcb->dcb_cerr);
		else
		    tprintf(stat->ttyp," could not get sense");
		if (stat->s_sense < 3) {
			stat->s_sense++;
			tprintf(stat->ttyp,": retried\n");
			return(1);
		}
		tprintf(stat->ttyp,": FAILED\n");
		gtiodone(stat, EIO);
		return (0);
	}
	stat->resid = 0;
	if (stat->s_sense) {
		stat->s_sense = 0;
		if (stat->sense.sns_7.sns_7_valid)
			stat->resid = SCSI_HMML(stat->sense.sns_7.sns_7_info);
		if (stat->sense.sns_7.sns_7_fil_mk) {
			filemark = 1;
			stat->fileba = 0;
			stat->s_lastrfm = 1;
		}
		if (stat->sense.sns_7.sns_7_eom) {
			tprintf(stat->ttyp,
			    "gt%d: unexpected physical EOT/BOT\n",unit);
			gtiodone(stat,ENOSPC);
			return (0);
		}
		if (stat->s_notraw && stat->sense.sns_7.sns_7_ili) {
			tprintf(stat->ttyp,
			    "gt%d: incorrect record length\n",unit);
			gtiodone(stat,EIO);
			stat->fileba += (stat->count - stat->resid);
			return (0);
		}
		switch (key = stat->sense.sns_7.sns_7_key) {
		    case SNS_7_KEY_NO_SENSE: case SNS_7_KEY_RECOVERED: 
			break; /* go on as if no error */
		    case SNS_7_KEY_ABORT_CMD:
			return(0); /* have it retried */
		    case SNS_7_KEY_DATA_PROTECT:
			tprintf(stat->ttyp,
			    "gt%d: CARTRIDGE WRITE PROTECTED\n",unit);
			gtiodone(stat, EIO);
			return (0);
		    case SNS_7_KEY_UNIT_ATTN:
			tprintf(stat->ttyp,
	"gt%d: DRIVE RESET OR CARTRIDGE CHANGED: reinitializing unit\n", unit);
			stat->needreinit = 1;
			if (stat->fileno || stat->fileba)
				gtiodone(stat, EIO);
				/* REWIND SOMEHOW! */
			stat->fileba = 0;
			stat->fileno = 0;
			return (0);
		    case SNS_7_KEY_BLANK_CHECK:
			filemark = 1; /* treat like a second filemark */
			break;
		    default:
			tprintf(stat->ttyp,"gt%d: HARD ERROR: %s\n",
				unit, ascii_sns_7_key[key]);
			gtiodone(stat, EIO);
			return (0);
		}
	}
	if (((qm = gtcinfo[cont]) == NULL)
	    || (qm->qm_tab.b_actf == NULL) 
	    || ((bp = qm->qm_tab.b_actf->b_actf) == NULL)) {
		tprintf(stat->ttyp, "gt%d: lost buffer\n", unit);
		gtiodone(stat, EIO);
		return(0);
	}
    	if (bp->b_un.b_addr == &stat->dcb) {
	    /* is a direct simple command */
	    switch (dcb->dcb_cdb.cdb_0.cdb_0_cmd) {
	    case CMD_SPACE:
		if (dcb->dcb_cdb.cdb_0.cdb_0_lba_h) {	/* skip files */
			if (stat->count > 0)
				stat->fileba = 0;
			else
				stat->fileba = GT_MAXBA;
			stat->fileno += (stat->count - stat->resid);
		} else {				/* space records */
			if (filemark) {
				if (stat->count > 0) {
					stat->fileno++;
					stat->fileba = 0;
				} else {
					stat->fileno--;
					stat->fileba = GT_MAXBA;
				}
				gtiodone(stat, EIO);
				return(0);
			} else {
				stat->fileba += (stat->count - stat->resid);
			}
		}
	    case CMD_REZERO:
		stat->fileba = 0;
		stat->fileno = 0;
		break;
	    default:
		break;
	    } /* end switch */
	    gtiodone(stat, 0);
	    return (0);
	}
	/* read/write commands (or seek) */
	if (bp->b_dev&GTF_ISI)
		return(isiheader1(stat,bp,dcb,filemark));
	if (filemark) {
		stat->fileba = 0;
		stat->fileno++;
	} else {
		if (stat->blksz)
			stat->fileba += (stat->count - stat->resid);
		else
			stat->fileba++;
	}
	if (stat->s_seeking) {
		stat->s_seeking = 0;
		return (0);
	}
	if (stat->blksz)
		stat->resid *= stat->blksz;
	if ((bp->b_flags & B_READ) == 0)
		stat->s_lastiow = 1;
	gtiodone(stat, 0);
	return (0);
}

gtioctl(dev, cmd, data, flag)
	caddr_t data;
	dev_t dev;
{
	int				unit = GT_UNIT(dev);
	int				lun = GTU_LUN(unit);
	struct gtstat	*stat = &gtstat[unit];
	register struct buf		*bp = &stat->rbuf;
	register struct hacb_dcb	*dcb = &stat->dcb;
	struct mtop			*mtop;
	struct mtget			*mtget;
	int				fmks,count;

	switch (cmd) {
	    case MTIOCTOP:  /* tape operation */
		gtwait(stat);
		bp->b_un.b_addr = (caddr_t )&stat->dcb;
		bp->b_dev = dev;
		bp->b_error = 0;
		mtop = (struct mtop *)data;
		switch (mtop->mt_op) {
		    case MTFSF: case MTBSF:
		    case MTFSR: case MTBSR:
			count = mtop->mt_count;
			stat->count = count;
			switch (mtop->mt_op) {
			    case MTBSF:
				count = -count;
				/* fall through */
			    case MTFSF:
				fmks = 0x10000;
				break;
			    case MTBSR:
				count = -count;
				/* fall through */
			    case MTFSR:
				break;
			}
			if (stat->dtype == WANGTEK && count < 0) {
				/* WANG don't support reverse! */
				gtdone(stat);
				return(EINVAL);
			}
			SET_CDB_0(&dcb->dcb_cdb,CMD_SPACE,lun,
				(((count>>8)&0xffff)|fmks),(count&0xff),0,0);
			SET_DCB(dcb,DCB_IE,DCB_DIR_NO,0,stat->s_nodisc,
				sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
			break;
		    case MTREW:
			SET_CDB_0(&dcb->dcb_cdb,CMD_REZERO,lun,0,0,0,0);
			SET_DCB(dcb,DCB_IE,DCB_DIR_NO,0,stat->s_nodisc,
				sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
			break;
		    case MTWEOF:
			SET_CDB_0(&dcb->dcb_cdb, CMD_FILEMARK, lun, 0,
				mtop->mt_count, 0, 0);
			SET_DCB(dcb,DCB_IE,DCB_DIR_NO,0,stat->s_nodisc,
				sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
			break;
		    case MTERASE:
			SET_CDB_0(&dcb->dcb_cdb,CMD_ERASE,lun,0x10000,0,0,0);
			SET_DCB(dcb,DCB_IE,DCB_DIR_NO,0,stat->s_nodisc,
				sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
			break;
		    case MTRST:
			gtreset(unit);
			gtdone(stat);
			return (0);
		    case MTNOP:
			gtdone(stat);
			return (0);
		    case MTOFFL: case MTRET: case GTLOAD:
			switch (mtop->mt_op) {
			    case MTOFFL:
				count = 0;
				break;
			    case MTRET:
				count = 3;
				break;
			    case GTLOAD:
				count = mtop->mt_count;
				break;
			}
			SET_CDB_0(&dcb->dcb_cdb,CMD_STARTSTOP,lun,0,count,0,0);
			SET_DCB(dcb,DCB_IE,DCB_DIR_NO,0,stat->s_nodisc,
				sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
			break;
		    case GTPREVENT:
			SET_CDB_0(&dcb->dcb_cdb,CMD_PREVENT,lun,0,
				mtop->mt_count,0,0);
			SET_DCB(dcb,DCB_IE,DCB_DIR_NO,0,stat->s_nodisc,
				sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
			break;
		    case GTSETBKSZ:
			if ((mtop->mt_count < stat->blklim.blklim_min) ||
				(mtop->mt_count > stat->blklim.blklim_max)) {
					gtdone(stat);
					return (EINVAL);
			}
			stat->blksz = mtop->mt_count;
			stat->needreinit = 1;
			gtdone(stat);
			return (0);
		    default:
			gtdone(stat);
			return (ENXIO);
		}
		gtcommand(bp);
		gtdone(stat);
		return (bp->b_error);
	    case MTIOCGET:
		mtget = (struct mtget *)data;
		mtget->mt_type = MT_ISGT;
		mtget->mt_dsreg = stat->dsreg;
		mtget->mt_erreg = (stat->sense.sns_7.sns_7_key << 8) |
			stat->sense.sns_7.sns_7_err;
		mtget->mt_resid = stat->resid;
		mtget->mt_fileno = stat->fileno;
		mtget->mt_blkno = stat->fileba;
		break;
	    default:
		return (ENXIO);
	}
	return (0);
}

/*
 * Queue a pseudotape operation.
 */
gtstrategy(bp)
register struct buf *bp;
{
	int			unit = GT_UNIT(bp->b_dev);
	int			cont = GTU_CONT(unit);
	struct gs_queue		*gtqp = &gtq[GTU_CTLR(unit)];
	struct gtstat	*stat = &gtstat[unit];
	register struct buf	*dp = &stat->qhd;
	register int		s;
	register struct qb_device *qi = gtdinfo[unit];
	register struct qb_ctlr *qm = gtcinfo[cont];

	if(bp->b_flags & B_READ) {
	    /*
	     * If the last operation was a read and it came back with
	     * a filemark flag, then skip this read and report
	     * the UNIX standard eof
	     */
	     if(stat->s_lastrfm) {
		 bp->b_resid = bp->b_bcount;
		 /* *** what else needs to be done here to clean up????? */
		 bp->b_flags |= B_DONE;
		 stat->s_lastrfm = 0;
		 return;
	     }
	}
	/*
	 * Put transfer at end of controller queue
	 */
	bp->av_forw = NULL;
	s = splx(qm->qm_psl);
	if (dp->b_actf == NULL)
		dp->b_actf = bp;
	else
		dp->b_back->av_forw = bp;
	dp->b_back = bp;

	if (dp->b_active == 0) {
		dp->b_forw = NULL;
		dp->b_active = 1;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
	}
	if (gtqp->gsq_forw == (struct gs_queue *)0)
		gsenq(cont, GTU_TARG(unit), gtqp);
	splx(s);
}

gtminphys(bp)
	struct buf      *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, GT_MAXCNT);
}

gtiodone(stat, error)
	struct gtstat	*stat;
{
	struct buf	*bp = stat->qhd.b_actf;
	register struct qb_ctlr *qm;
	struct buf	*dp;

	qm = gtcinfo[GTU_CONT(GT_UNIT(bp->b_dev))];
	dp = qm->qm_tab.b_actf;
	qm->qm_tab.b_actf = dp->b_forw;
	dp->b_active = 0;
	dp->b_actf = bp->av_forw;
	if (dp->b_actf) {
		dp->b_forw = NULL;
		dp->b_active = 1;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
	}

	bp->b_resid = stat->resid;
	bp->b_error = error;

	if (error) {
		bp->b_flags |= B_ERROR;
	}
	iodone(bp);
	if (stat->gt_free_vdma) {
		qbdone(stat->gt_free_vdma);
		stat->gt_free_vdma = 0;
	}
	return (0);
}

gtdone(stat)
	register struct gtstat *stat;
{
	stat->s_busy = 0;
	if (stat->s_want)
		wakeup(stat);
}

gtwait(stat)
	register struct gtstat *stat;
{
	while (stat->s_busy) {
		stat->s_want = 1;
		sleep(stat, PRIBIO);
	}
	stat->s_busy = 1;
	stat->s_want = 0;
}

#define GT_NO_STEPS	2
gtreinit(dcb,stat,lun,fromgetcmd)
	struct hacb_dcb	*dcb;
	struct gtstat	*stat;
	int		lun;
	int		fromgetcmd;
{
	int		count;

  if (fromgetcmd) {
    switch (stat->needreinit) {
      case 1:	 /* rewind tape */
	SET_CDB_0(&dcb->dcb_cdb, CMD_REZERO, lun, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_IE, DCB_DIR_NO, stat->s_swcd, stat->s_nodisc,
		sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 0, 0, 0);
	break;
      case 2:	 /* set mode select */
	stat->msen.msen_len = 0;
	stat->msen.msen_mtype = 0;
	stat->msen.msen_wprot = 0;
	stat->msen.msen_rsvd = stat->s_buffered * 0x10;
	stat->msen.msen_bdl = 8;
	stat->msen.msen_bd.bd_density = 0;
	SCSI_HML_SET(stat->msen.msen_bd.bd_nb, 0);
	SCSI_HML_SET(stat->msen.msen_bd.bd_bl, stat->blksz);
	count = 12;
	if (stat->dtype == EXABYTE) {
		/* evenbytedisconnect & parityenable */
		stat->msen.msen_pd.pd_code = 6;
		count = 14;
	}
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSELECT, lun, 0, count, 0, 0);
	SET_DCB(dcb, DCB_IE, DCB_DIR_OUT, stat->s_swcd, stat->s_nodisc,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 
	    stat->msen_vdma, count, 1);
	flush_cache_entry(&stat->msen, sizeof(stat->msen));
	break;
      default:
	return(0);
    }
    return (1);
  } else {				/* from doerr */
	if (dcb->dcb_err) {
		tprintf(stat->ttyp,"fatal error while reiniting\n");
		stat->needreinit = GT_NO_STEPS;
	}
	if (stat->s_sense)
		switch (stat->sense.sns_7.sns_7_key) {
		    case SNS_7_KEY_NO_SENSE: case SNS_7_KEY_RECOVERED: 
			break;
		    case SNS_7_KEY_ABORT_CMD:
			return(0); /* have it retried */
		    case SNS_7_KEY_UNIT_ATTN:
			tprintf(stat->ttyp,
			"DRIVE RESET OR CARTRIDGE CHANGED: rereiniting unit\n");
			stat->needreinit = 0;
			break;
		    default:
			tprintf(stat->ttyp,"fatal error while reiniting\n");
			stat->needreinit = GT_NO_STEPS;
			break;
		}
	if (++stat->needreinit > GT_NO_STEPS)
		stat->needreinit = 0;
	return (0);
  }
}

isiheader(stat,bp,dcb,lun)
	struct gtstat	*stat;
	struct buf	*bp;
	struct hacb_dcb	*dcb;
{
	struct isihead	*ih = &stat->isihead;

	if (stat->s_hdone) {
		if (bp->b_flags & B_READ) {
			stat->count = ih->ih_count;
			stat->roundeven = ih->ih_blksz;
		}
		return(0);
	}
	if (bp->b_flags & B_READ) {
		ih->ih_id = 0;
		SET_CDB_0(&dcb->dcb_cdb, CMD_READ, lun, 0x10000, 1, 0, 0);
		SET_DCB(dcb, DCB_IE, DCB_DIR_IN, stat->s_swmd, stat->s_nodisc,
			sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
			stat->isihead_vdma, sizeof(struct isihead), 512);
		flush_cache_entry(&stat->isihead, sizeof(stat->isihead));
	} else {
		ih->ih_id = 0xA1FE;
		ih->ih_fileno = stat->fileno + 1;
		ih->ih_fileba = stat->fileba + 1;
		ih->ih_blksz = bp->b_bcount;
		ih->ih_count = stat->count;
		SET_CDB_0(&dcb->dcb_cdb, CMD_WRITE, lun, 0x10000, 1, 0, 0);
		SET_DCB(dcb, DCB_IE, DCB_DIR_OUT, stat->s_swmd, stat->s_nodisc,
			sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
			stat->isihead_vdma, sizeof(struct isihead), 512);
			flush_cache_entry(&stat->isihead,sizeof(stat->isihead));
	}
	return(1);
}

isiheader1(stat,bp,dcb,filemark)
	struct gtstat	*stat;
	struct buf	*bp;
	struct hacb_dcb	*dcb;
{
	struct isihead	*ih = &stat->isihead;

	if (stat->s_hdone) {
		stat->s_hdone = 0;
		stat->resid = bp->b_bcount - ih->ih_blksz;
		stat->fileba++;
		if (filemark) {
			stat->fileba = 0;
			stat->fileno++;
		}
		if ((bp->b_flags & B_READ) == 0)
			stat->s_lastiow = 1;
		gtiodone(stat,0);
	} else {
		if (bp->b_flags & B_READ) {
			if (filemark) {
				stat->resid = bp->b_bcount;
				stat->fileba = 0;
				stat->fileno++;
				gtiodone(stat,0);
				return(0);
			}
			if (ih->ih_id != 0xA1FE) {
				gtiodone(stat,EIO);
				return(0);
			}
			stat->fileno = ih->ih_fileno;
		}
		stat->s_hdone = 1;
	}
	return(0);
}

extern	dumpsize;
gtdump(dev)
	dev_t	dev;
{
	int			unit = GT_UNIT(dev);
	int			lun = GTU_LUN(unit);
	struct gtstat	*stat = &gtstat[unit];
	register struct hacb_dcb	*dcb = &stat->dcb;
	struct qb_device		*qi;
	u_int				blksz, num, count, blks, fixed, start;
	u_int				vdma;

	if (unit >= Ngt || (qi = gtdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	stat->s_swmd = dev&GTF_SWAB ? DCB_SW : DCB_NOSW;
	if (stat->blksz) {
		fixed = 0x10000;
		blksz = stat->blksz;
	} else
		blksz = 1;
	start = 0x0;
	num = ctob(dumpsize);
	while (num > 0) {
		count = (num > GT_MAXCNT) ? GT_MAXCNT : (num + (num&0x1));
		blks = (count + stat->blksz - 1) / stat->blksz;
		vdma = qbdumpsetup(start, dtob(blks));
		SET_CDB_0(&dcb->dcb_cdb, CMD_WRITE, lun, (blks >> 8) | fixed,
			blks & 0xFF, 0, 0);
		SET_DCB(dcb, DCB_NOIE,DCB_DIR_OUT,stat->s_swmd,stat->s_nodisc,
			sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
			VDMA_STD(vdma,vbnum), count, blksz);
		if (gtissue(unit)) {
			qbrelse(&vdma);
			return (EIO);
		}
		qbrelse(&vdma);
		start += count;
		num -= count;
	}
	SET_CDB_0(&dcb->dcb_cdb, CMD_FILEMARK, lun, 0, 1, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, stat->s_swcd, stat->s_nodisc,
		sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 0, 0, 0);
	if (gtissue(unit))
		return (EIO);
	SET_CDB_0(&dcb->dcb_cdb, CMD_REZERO, lun, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, stat->s_swcd, stat->s_nodisc,
		sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, 0, 0, 0);
	if (gtissue(unit))
		return (EIO);
	return (0);
}

#endif
