/* NOTE qdtrace, which this causes to be defined is also referenced in other */
/* files.  Be sure you turn off QDTRACE (and GDTRACE?) there if you do here  */
#define QDTRACE		/* SEE NOTE ABOVE */

/*
 * OPENCHIP: Liberator On-board SCSI subdriver subroutines
 */

#include "../is68kdev/qsreg.h"
#include "../is68kdev/qscsi.h"

#ifndef	MAXDEVBKSZ
#define	MAXDEVBKSZ      0x200
#endif	MAXDEVBKSZ

#define	QS_ID		7	/* SCSI controller ID - 7 has highest pri */
#define	QS_TIMEOUT	30	/* TIMEOUT * 8ms = 240ms timeout on sel */

/* Macros */

#define GETMST()	(OPENCHIP->op_scstat & OP_ST_MST_MASK)
#define GETPHASE()	(OPENCHIP->op_scstat & OP_ST_PHSMASK)
#define GETSTATE()	(OPENCHIP->op_scstat & OP_ST_STATEMASK)
#define NEWMST(x)	\
	(OPENCHIP->op_scstat = (OPENCHIP->op_scstat & ~OP_ST_MST_MASK) | x)
#define NEWPHASE(x)	\
	(OPENCHIP->op_scstat = (OPENCHIP->op_scstat & ~OP_ST_PHSMASK) | x)
#define NEWSTATE(x)	\
	(OPENCHIP->op_scstat = (OPENCHIP->op_scstat & ~OP_ST_STATEMASK) | x)

#define GSCT_CTLR(cont, targ)	(((cont) << 2) | (targ))

#define ptov(X) (((unsigned long)X)+VEQR_ADDR)

char		gs_vqx_version[64];
int		gs_vqx_vers();
int		gs_vqx_cmdwait(), gs_vqx_run(), gs_vqx_intr();

/* this code knows that there can only be one on-board scsi controller	*/
struct	scsi_cmnd gs_vqx_cmd[Ngs_targ_cont];

u_char	qsmsginbuf;	/* controller can only talk to a single target	*/
u_char	qsmsgout;	/* so there is only one of these		*/
int	gs_vqx_started;	/* pseudo-controller internal busy flag		*/
char			gs_vqx_local[Ngs_targ_cont][MAXDEVBKSZ];
#ifdef DEBUG
u_char prevstat, prevopstat;
#endif DEBUG

/*
 * State of controller from last transfer. Since only one transfer can be done
 * at a time per target device, allocate one for each device on controller.
 */
struct	gs_vqx_stat       {	       /* pseudoctlr status block 	*/
	int		qs_sgphase;	/* scatter/gather phase		*/
	unsigned long	qs_sgaddr;	/* scatter/gather address	*/
	unsigned long	qs_sglen;	/* scatter/gather length	*/
	unsigned long	qs_sglba;	/* s/g logical block address	*/
} gs_vqx_stat[Ngs_targ_cont];

struct gs_vqx_qe {		/* work queue entries, one per target	*/
#define QE_FL_BUSY	0x0001		/* already busy	(sanity check)	*/
	u_short	qe_flags;		/* state info about this entry	*/
	u_short qe_targ;		/* for gs_vqx_dequeue to return	*/
	struct qs_vqx_qd *qe_next;	/* run queue next entry		*/
} gs_vqx_qe[Ngs_targ_cont];		/* indexed by targ		*/

struct gs_vqx_qh {
	/* could be removed, useful for debugging			*/
	u_long qh_count;		/* number of entries enqueued	*/
	struct gs_vqx_qe *qh_head;	/* next one to run		*/
	struct gs_vqx_qe *qh_tail;	/* where to insert next one	*/
} gs_vqx_qh;

gs_vqx_intr(cont)
{
	register struct gs_hacb	*hacb = &gs_hacb[cont];	/* no CACHE on VQX */
	int			targ;
	struct hacb_dcb		*dcb;
#ifdef	DEBUG
	unsigned char status, rstatus;
#endif	DEBUG
	register struct gs_vqx_device *wdptr;
	struct scsi_cmnd	*cmd;
	u_short			rdcr;
	char			done = 0;
	struct gs_vqx_stat	*stat;
	register int		s;
	
	wdptr = (struct gs_vqx_device *)IO_ADRS_SCSI;
	status = rstatus = OPENCHIP->op_scsistatus;
	wdptr->wd_addr_stat = WD_DEST_ID;
	targ = wdptr->wd_data; 

#ifdef	QDTRACE
	qdtrace(0x10004, targ, rstatus);
#endif	QDTRACE
	stat = &gs_vqx_stat[targ];

	/*
	 * The WD scsi chip knows which unit needs dealing with
	 * we turn off the GO and BSY bits to keep qswatch and friends
	 * happy
	 */
	rdcr = ((DCR_BUSY|DCR_GO)<<targ);
	if((hacb->hacb_dcr & rdcr) != rdcr) {
		printf("qs_vqx_intr: expected BUSY|GO, got %x (targ %x)\n",
			hacb->hacb_dcr, targ);
#ifdef	DEBUGGER
		trap(15);
#endif	DEBUGGER
	}
	
	dcb = &hacb->hacb_dcb[targ];
	
	if(dcb->dcb_ie) {		/* command was with interrupts	 */
		OCsanity("gs_vqx_intr");
	}
	cmd = &gs_vqx_cmd[targ];	/* OPENCHIP command block	*/

	/*
	 * There are several different versions of status which encode the
	 * new phase. In order to reduce the number of case's in a switch
	 * such multiple choices are reduced to one. 
	 */
	if (status & 0x08)
		status &= 0x0f;
	switch (GETSTATE()) {
	case SC_SELECTING:
		switch (status) {
		case WDST_SEL_CMPLT:	/* select completed successfully */
			if(OPENCHIP->op_sccntrl & OP_SC_SELATN) 
				NEWSTATE(SC_SELMSGOUT);
			else
				NEWSTATE(SC_CONNECTED);
			break;
		case WDST_TIMEOUT_CMD_TERM:	/* selection timed out */
#ifdef	DEBUG_PHASE
			printf("WDST_TIMEOUT_CMD_TERM\n");
#endif	DEBUG_PHASE
			qscmddone(targ,cmd,SFN_SLCT_TIMEOUT);
			done = 1;
			break;
		default:
			printf("SELECTING with status 0x%x\n", rstatus);
#ifdef	DEBUG
printf("op_scsistatus=0x%x, op_scstat=0x%x, prevstat=0x%x, prevopstat=0x%x\n",
				OPENCHIP->op_scsistatus, OPENCHIP->op_scstat,
				prevstat,prevopstat);
#endif	DEBUG
			break;
		}
		break;

	case SC_SELMSGOUT:
		switch(status) {
			case WDST_REQ_SERV_MSG_OUT:
				qsmsgout = MSG_IDENTIFY | cmd->qs_lun;
				setupDMA(wdptr,&qsmsgout, 1, OP_ST_PHS_MSG_OUT);
				break;
			default:
				NEWSTATE(SC_CONNECTED);
				goto connected;
				
		}

	case SC_CONNECTED:
		/*
		 * If DMA was in use, it must be tidied up. 
		 */
connected:	
		if (OPENCHIP->op_sccntrl & OP_SC_DACT) {
			unsigned long   lngth;
			char           *cptr;

			OPENCHIP->op_sccntrl &= ~OP_SC_DACT;
			/*
			 * wait for DACT to read as reset. Perform a slight
			 * delay in the loop so as not to access the OpenChip
			 * too much. 
			 */
			do {
				DELAY(10);
			} while (OPENCHIP->op_sccntrl & OP_SC_DACT);

			/*
			 * get the remaining count out of the SCSI chip. 
			 */
			wdptr->wd_addr_stat = WD_TRANS_COUNT0;
			lngth = (unsigned char) wdptr->wd_data;
			lngth <<= 8;
			lngth |= (unsigned char) wdptr->wd_data;
			lngth <<= 8;
			lngth |= (unsigned char) wdptr->wd_data;
			/*
			 * calculate the corrected DMA pointer. 
			 */

			cptr = (char *)ptov(OPENCHIP->op_scdmaptr);
			cptr -= (unsigned char) ((unsigned char) cptr +
				(unsigned char) lngth) - OPENCHIP->op_scsum;

			switch (GETPHASE()) {
			case OP_ST_PHS_DATA_IN:
			case OP_ST_PHS_DATA_OUT:
				if (!(cmd->qs_flags & SCF_DATA_OVFL)) {
					cmd->qs_datlnk.sl_cnt = lngth;
					cmd->qs_datlnk.sl_ptr = cptr;
				}
				break;
			case OP_ST_PHS_STATUS:
				if (!(cmd->qs_flags & SCF_STATUS_OVFL)) {
					cmd->qs_nstat = lngth;
					cmd->qs_pstat = cptr;
				}
				break;
			case OP_ST_PHS_COMMAND:
				if (!(cmd->qs_flags & SCF_COMM_OVFL)) {
					cmd->qs_ncmnd = lngth;
					cmd->qs_pcmnd = cptr;
				}
				break;
			}
		}
		switch (status) {
			/*
			 * statuses 0x1? indicate previous transfer completed
			 * normally. 0x4? indicates transfer completed with
			 * count != 0. 0x8? indicates first transfer after
			 * connection. 
			 */
		case 0x08 | PHS_MSG_IN:
#ifdef	DEBUG_PHASE
			printf("PHS_MSG_IN\n");
#endif	DEBUG_PHASE
			setupDMA(wdptr,&qsmsginbuf, 1, OP_ST_PHS_MSG_IN);
			break;
		case 0x08 | PHS_MSG_OUT:
#ifdef	DEBUG_PHASE
			printf("PHS_MSG_OUT\n");
#endif	DEBUG_PHASE
			qsmsgout = MSG_IDENTIFY | cmd->qs_lun;
			setupDMA(wdptr,&qsmsgout, 1, OP_ST_PHS_MSG_OUT);
			break;
		case 0x08 | PHS_STATUS:
#ifdef	DEBUG_PHASE
			printf("PHS_STATUS\n");
#endif	DEBUG_PHASE
			if (cmd->qs_nstat) {
			    setupDMA(wdptr,cmd->qs_pstat, cmd->qs_nstat, 
				OP_ST_PHS_STATUS);
			} else {
			    cmd->qs_flags |= SCF_STATUS_OVFL;
			    gs_vqx_setmanual(wdptr, OP_ST_PHS_STATUS);
			}
			break;
		case 0x08 | PHS_COMMAND:
#ifdef	DEBUG_PHASE
			printf("PHS_COMMAND\n");
#endif	DEBUG_PHASE
			if (cmd->qs_ncmnd) {
#ifdef	XDEBUG
			    int i;
			    char *cptr;
			    for(i=0,cptr=cmd->qs_pcmnd;i<cmd->qs_ncmnd;i++) {
			    	printf("PHS_COMMAND cdb[%d]=0x%x\n",i,*cptr++);
			    }
#endif	XDEBUG
			    setupDMA(wdptr,cmd->qs_pcmnd, cmd->qs_ncmnd, 
				OP_ST_PHS_COMMAND);
			} else {
			    gs_vqx_setmanual(wdptr, OP_ST_PHS_COMMAND);
			}
			break;
		case 0x08 | PHS_DATA_IN:
		case 0x08 | PHS_DATA_OUT:
#ifdef	DEBUG_PHASE
			printf("PHS_DATA_IN/PHS_DATA_OUT\n");
#endif	DEBUG_PHASE
			if (cmd->qs_datlnk.sl_cnt == 0) {
			    cmd->qs_flags |= SCF_DATA_OVFL;
			    if ((status & PHS_MASK) == PHS_DATA_IN) 
				gs_vqx_setmanual(wdptr, OP_ST_PHS_DATA_IN);
			    else
				gs_vqx_setmanual(wdptr, OP_ST_PHS_DATA_OUT);
			} else {
#ifdef	QDTRACE
	qdtrace(0x411, cmd->qs_datlnk.sl_ptr, cmd->qs_datlnk.sl_cnt);
#endif	QDTRACE
			    if ((status & PHS_MASK) == PHS_DATA_IN) {
				setupDMA(wdptr,cmd->qs_datlnk.sl_ptr, 
				    cmd->qs_datlnk.sl_cnt, OP_ST_PHS_DATA_IN);
			    } else {
				setupDMA(wdptr,cmd->qs_datlnk.sl_ptr, 
				    cmd->qs_datlnk.sl_cnt, OP_ST_PHS_DATA_OUT);
			    }
			}
			break;
		case WDST_DISCONN_SERV:	/* normal disconnect */
		case WDST_DISCONN_CMD_TERM:	/* unexpected disconnect */
			/*
			 * Notice that l_state is guaranteed to be L_ST_NULL
			 * since it gets set so when a reselection takes
			 * place. 
			 */
			switch (GETMST()) {
			case MST_NULL:
			case MST_DISC:
#ifdef	DEBUG
				printf("UNEXPECTED DISCONNECT\n");
#endif	DEBUG
				qscmddone(targ,cmd,SFN_NO_CMND_CMPLT);
				done = 1;
				break;
			case MST_CMPLT:
			case MST_CMPLT2:
				qscmddone(targ,cmd,SFN_NORMAL_END);
				done = 1;
				break;
			}
			/*
			 * The code may now go ahead and try to issue another
			 * command. 
			 */
			break;
		case WDST_MSG_IN_PAUSE:	/* message in has paused */
			switch (cmd->qs_status[0]) {
			case MSG_CMD_COMPLETE:
				NEWMST(MST_CMPLT);
				wdptr->wd_addr_stat = WD_COMMAND;
				wdptr->wd_data = WD_CMD_NEG_ACK;
				break;
			case MSG_EXTENDED:
#ifdef	DEBUG_PHASE
				printf("WDST_MSG_IN_PAUSE: MSG_EXTENDED\n");
#endif	DEBUG_PHASE
				gs_vqx_setmanual(wdptr, OP_ST_PHS_MSG_IN);
			case MSG_SAVE_DPTR:
			case MSG_RESTORE_PTRS:
			default:
#ifdef	DEBUG_PHASE
				printf("WDST_MSG_IN_PAUSE:\
 MSG_SAVE_DPTR/MSG_RESTORE_PTRS\n");
#endif	DEBUG_PHASE
				cmd->qs_status[0] = MSG_MSG_REJECT;
				wdptr->wd_addr_stat = WD_COMMAND;
				wdptr->wd_data = WD_CMD_NEG_ACK;
				break;
			case MSG_DISCONNECT:
#ifdef	DEBUG_PHASE
				printf("WDST_MSG_IN_PAUSE: MSG_DISCONNECT\n");
#endif	DEBUG_PHASE
				NEWMST(MST_DISC);
				wdptr->wd_addr_stat = WD_COMMAND;
				wdptr->wd_data = WD_CMD_NEG_ACK;
				break;
			case MSG_MSG_REJECT:
#ifdef	DEBUG_PHASE
				printf("WDST_MSG_IN_PAUSE: MSG_REJECT\n");
#endif	DEBUG_PHASE
				/*
				 * It is hard to see what to do if this is
				 * received but the specification says that a
				 * message reject message may not be rejected
				 * itself. 
				 */
				wdptr->wd_addr_stat = WD_COMMAND;
				wdptr->wd_data = WD_CMD_NEG_ACK;
				break;
			}
			break;
		default:
			printf("CONNECTED with status 0x%x\n", rstatus); 
#ifdef	DEBUG 
				printf("op_scsistatus=0x%x, op_scstat=0x%x,\
 prevstat=0x%x, prevopstat=0x%x\n",
				OPENCHIP->op_scsistatus, OPENCHIP->op_scstat,
				prevstat,prevopstat);
#endif	DEBUG
			break;
		}
		break;
	}
#ifdef	DEBUG
	prevstat = OPENCHIP->op_scsistatus;
	prevopstat = OPENCHIP->op_scstat;
#endif	DEBUG
	/* map the openchip and wd status info into the dcb		*/
	/* unfortunately we have too many bits, but we do our best	*/
	dcb->dcb_scsi_status = cmd->qs_status[0];
	if(cmd->qs_finstat == SFN_NORMAL_END
	    && !(cmd->qs_flags&SCF_FATAL_COMB)) {
		dcb->dcb_cerr = 0;
	} else {
		if(cmd->qs_finstat == SFN_UNSPEC)	/* fold to 4 bits */
			cmd->qs_finstat = SFN_NORMAL_END;
		dcb->dcb_cerr = ( ((cmd->qs_flags&0x1e)<<3) | cmd->qs_finstat);
	}
	OPENCHIP->op_scsintr = 0xff;
	if(!done) {
		return;	/* wait for next phase interrupt	*/
	}

	gsrunning[cont]--;
	hacb->hacb_dcr &= ~rdcr;	/* appease watchdog	*/

	s = spl7();
	gs_vqx_started--;		/* controller no longer running	*/
	splx(s);
	gs_vqx_start(cont);	/* give another target a chance	*/

	/*
	 * sgphase routine	action
	 *   0     gsintr	normal
	 *   0     gsrun	check for need to split xfer, set up if so,
	 *			and change sgphase to 1
	 *   1	   gsintr	upon completion of cmd, call gsrun; if error,
	 *			clear sgphase so retry starts all over
	 *   1     gsrun	copy data in if write, start I/O, sgphase = 2
	 *   2     gsintr	copy data out if read, sgphase = 0
	 *   2     gsrun	panic
	 */
	if(stat->qs_sgphase == 1) {	/* do second part of sg		*/
		gs_vqx_run(cont, targ);
		return;
	}
	if(stat->qs_sgphase == 2) {	/* finish 2nd part of sg operation */
		unsigned char xcmd;
		if(((xcmd=dcb->dcb_cdb.cdb_0.cdb_0_cmd) != CMD_XWRITE)
		   && (xcmd != CMD_XREAD)) {
#ifdef	DEBUG
			printf("gs_vqx_intr: dcb %x, sgphase 1, command %x\n",
				dcb, xcmd);
#ifdef	DEBUGGER
			trap15();
#endif	DEBUGGER
#else	DEBUG
			panic("gs_vqx_intr: sgphase 1, command %x\n", xcmd);
#endif	DEBUG
		}
		if(xcmd == CMD_XREAD) {		/* is read operation	*/
			bcopy(gs_vqx_local[targ], stat->qs_sgaddr,
				stat->qs_sglen);
		}
		stat->qs_sgphase = 0;
	}
	gs_intr_common(cont, targ, dcb);
	return (0);
}

gs_vqx_run(cont, targ)
{
	/* no CACHE on vqx	*/
	register struct gs_hacb	*hacb = &gs_hacb[cont];
	register struct hacb_dcb *dcb = &hacb->hacb_dcb[targ];
	register struct gs_vqx_stat *stat = &gs_vqx_stat[targ];
	unsigned int res_len;
	unsigned char xcmd = dcb->dcb_cdb.cdb_0.cdb_0_cmd;

	if (hacb->hacb_dcr & ((DCR_GO | DCR_BUSY) << targ))
		panic("gs: target busy but not active");
	gsrunning[cont]++;

	/* we always get either a system virtual address, or a physical	*/
	/* address (from qm_qbinfo), and since system virtual starts	*/
	/* past the possible end of physical memory (16 MB + 64MB)	*/
	/* if it is < SYSV_BASE, it must be physical (ignoring mmap)	*/
	/* the phys address may have had PHYSMEMBASE subtracted from it	*/
	/* because it is a VDMA_STD address, and VME sees phys mem at 0	*/
	
	if((unsigned long) dcb->dcb_dadr < PHYSMEMBASE) {
		dcb->dcb_dadr += PHYSMEMBASE;
	}
	if((unsigned long) dcb->dcb_dadr < SYSV_BASE) {
		/* convert to virtual	*/
		dcb->dcb_dadr += VEQR_ADDR;
	}

	/* fake scatter/gather operatons				*/
	/* next version of firmware ought to support it directly	*/
	/*
	 * (for the following discussion insert _vqx_ into routine names)
	 * when a need for a scatter/gather opeation is detected (in gsrun)
	 * sgphase is set to 1 by gsrun, and gsaddr and gslen are set up.
	 * When the first part is  complete gsintr sets sgphase to 2 and
	 * gsrun is called again from gsintr.  gsrun schedules the second
	 * phase from gsaddr and gslen, (ignoring contents of the dcb)
	 * and gsintr clears sgphase to 0 if it finds it 2 (after copying
	 * data out if it were a read operation
	 * sgphase routine	action
	 *   0     gsintr	normal
	 *   0     gsrun	check for need to split xfer, set up is so,
	 *			and change sgphase to 1
	 *   1	   gsintr	upon completion of cmd, call gsrun; if error,
	 *			clear sgphase so retry starts all over
	 *   1     gsrun	copy data in if write, start I/O, sgphase = 2
	 *   2     gsintr	copy data out if read, sgphase = 0
	 *   2     gsrun	panic
	 */
	if(stat->qs_sgphase == 2)
		panic("gs_vqx_run, qs_sgphase 2!!\n");
	/* check here, cause may be only a short xfer			*/
	if(stat->qs_sgphase == 0 && dcb->dcb_dbsz
	   && (res_len=(dcb->dcb_dlen % dcb->dcb_dbsz))
	   && (dcb->dcb_dlen == res_len)) {
		if (dcb->dcb_dbsz  > MAXDEVBKSZ)
			panic("gs: leftover too large");
		stat->qs_sgphase = 2;	/* "2nd" phase of scatter-gather*/
		/* record target/destination of 2nd part of xfer	*/
		stat->qs_sgaddr = (unsigned int) dcb->dcb_dadr;
		stat->qs_sglen  = res_len;	/* len of "2nd" part	*/
		dcb->dcb_dlen = dcb->dcb_dbsz;
		/* block count will be 1, we can leave it so		*/
		stat->qs_sglba = dcb->dcb_cdb.cdb_1.cdb_1_lba;
		if(xcmd == CMD_XWRITE) {	/* is write operation	*/
			bcopy(stat->qs_sgaddr, gs_vqx_local[targ],
				stat->qs_sglen);
		}
		dcb->dcb_dadr = gs_vqx_local[targ];
	} else if(stat->qs_sgphase == 1) {	/* start 2nd part of I/O */
		if((xcmd != CMD_XWRITE) && (xcmd != CMD_XREAD)) {
#ifdef	DEBUG
			printf("gs_vqx_run: dcb %x, sgphase 1, command %x\n",
				dcb, xcmd);
#ifdef	DEBUGGER
			trap15();
#endif	DEBUGGER
#else	DEBUG
			panic("gs_vqx_run: sgphase 1, command %x\n", xcmd);
#endif	DEBUG
		}
		if(xcmd == CMD_XWRITE) {	/* is write operation	*/
			bcopy(stat->qs_sgaddr, gs_vqx_local[targ],
				stat->qs_sglen);
		}
		dcb->dcb_dadr = gs_vqx_local[targ];
		dcb->dcb_dlen = dcb->dcb_dbsz;
		SCSI_HL_SET(dcb->dcb_cdb.cdb_1.cdb_1_len, 1);
		stat->qs_sgphase = 2;
	} else if(dcb->dcb_dbsz && (res_len=(dcb->dcb_dlen % dcb->dcb_dbsz))) {
		unsigned long lba;
		unsigned int bcnt;
		if (dcb->dcb_dbsz  > MAXDEVBKSZ)
			panic("gs: leftover too large");
		stat->qs_sgphase = 1;	/* first phase of scatter-gather*/
		/* compute target/destination of 2nd part of xfer	*/
		stat->qs_sgaddr = (unsigned int) dcb->dcb_dadr + dcb->dcb_dlen
				   - res_len;
		stat->qs_sglen  = res_len;	/* len of 2nd part	*/
		dcb->dcb_dlen -= res_len;	/* shorten first part	*/
		/* also shorten block count				*/
		bcnt = SCSI_HL(dcb->dcb_cdb.cdb_1.cdb_1_len) - 1;
		SCSI_HL_SET(dcb->dcb_cdb.cdb_1.cdb_1_len, bcnt);
		lba = dcb->dcb_cdb.cdb_1.cdb_1_lba;
		stat->qs_sglba = lba + dcb->dcb_dlen/dcb->dcb_dbsz;
	}
	/* and we fire up the I/O					*/
	gs_vqx_enqueue(targ);
	gs_vqx_start(cont);
}

/* unlike the other supported controllers, the onboard scsi can only	*/
/* handle one operation at a time, so we must schedule it internally	*/
/* we will someday also add disconnect/reconnect support here		*/

gs_vqx_start(cont)	/* start up the controller, if anything to do	*/
{
	register struct scsi_cmnd	*cmd;
	register struct gs_vqx_device	*addr;
	register struct gs_hacb		*hacb = &gs_hacb[cont];
	register struct hacb_dcb	*dcb;
	register struct gs_vqx_stat	*stat;
	register int			targ;	/* range 0-7	*/
	register int			s;	/* saved spl	*/

	s = spl7();
	if(gs_vqx_started) {	/* already running			*/
		splx(s);
		return;
	}
	if((targ=gs_vqx_dequeue()) == -1) {	/* nothing to do	*/
		splx(s);
		return;
	}
	gs_vqx_started++;	/* mark already running			*/
	splx(s);
	hacb->hacb_dcr |= ((DCR_GO|DCR_BUSY) << targ);
	dcb = &hacb->hacb_dcb[targ];
	stat = &gs_vqx_stat[targ];

	/* copy dcb data into scsi_cmnd struct	*/
	cmd = &gs_vqx_cmd[targ];
	/* set_cmd bzero's cmd struct first				*/
	SET_CMD(cmd, 0, &dcb->dcb_cdb, dcb->dcb_cdblen, cmd->qs_status,
		SFN_NOT_FINISHED, LEN_STATUS, dcb->dcb_dadr, dcb->dcb_dlen,
		0, 0);
	cmd->qs_lun = dcb->dcb_cdb.cdb_0.cdb_0_lun;
	if(stat->qs_sgphase == 2) {	/* doing 2nd scatter/gather I/O	*/
		dcb->dcb_cdb.cdb_1.cdb_1_lba = stat->qs_sglba;
	}

	while (s=OPENCHIP->op_scsintr) {	/* forget any pending interrupts */
#ifdef	QDTRACE
		qdtrace(10002, s, 0);
#endif	QDTRACE
		OPENCHIP->op_scsintr = 0xff;
		DELAY(15);
	}
	if(dcb->dcb_ie) {		/* command is with interrupts	 */
			/* allow them	 */
		do {
			OPENCHIP->op_sccntrl |= OP_SC_INEA;
			if(! OPENCHIP->op_sccntrl&OP_SC_INEA)
				printf("scsi INEA did not stick\n");
		} while(! OPENCHIP->op_sccntrl&OP_SC_INEA);

	} else {
#ifdef	QDTRACE
		qdtrace(10003, 0, 0);
#endif	QDTRACE
		OPENCHIP->op_sccntrl &= ~OP_SC_INEA;	/* dis-allow	 */
	}

	addr = (struct gs_vqx_device *)IO_ADRS_SCSI;

	addr->wd_addr_stat = WD_DEST_ID;	
	addr->wd_data = targ;

	addr->wd_addr_stat = WD_SYNCHRNS;
	addr->wd_data = 0;
	addr->wd_addr_stat = WD_CONTROL;
	addr->wd_data = WCNT_DMA;

	OPENCHIP->op_sccntrl |= OP_SC_SELATN;
	NEWSTATE(SC_SELECTING);
	addr->wd_addr_stat = WD_COMMAND;
	addr->wd_data = WD_CMD_SEL_ATN;
	if(dcb->dcb_ie) {		/* command is with interrupts	 */
		OCsanity("gs_vqx_start");
	}
}

/* there is work for targ, put it onto the run queue			*/
gs_vqx_enqueue(targ)
register int targ;
{
	register struct gs_vqx_qe *qe = &gs_vqx_qe[targ];
	register struct gs_vqx_qh *qh = &gs_vqx_qh;
	register int s;
	
	s = spl7();	/* need to use our own spl (autoconf knows it?)	*/
	if(qe->qe_flags & QE_FL_BUSY) {
		panic("gs_vqx_enqueue: target %x already in queue\n", targ);
	}
	qe->qe_flags |= QE_FL_BUSY;	/* can't be on queue twice	*/
	if(qh->qh_count++) {	/* queue not empty		*/
		qh->qh_tail->qe_next = qe;
		qh->qh_tail = qe;
		qe->qe_next = (struct gs_vqx_qe *) 0;
	} else {
		qh->qh_tail = qh->qh_head = qe;
		qe->qe_next = (struct gs_vqx_qe *) 0;
	}
	splx(s);
}

/* return the next targ that has work pending (-1 for none)	*/
/* is called with interrupts disabled				*/
gs_vqx_dequeue()
{
	register struct gs_vqx_qe *qe;
	register struct gs_vqx_qh *qh = &gs_vqx_qh;
	register int s;
	
	if(qh->qh_count == 0) {	/* queue  empty		*/
		return(-1);
	}
	qh->qh_count--;
	qe = qh->qh_head;
	if((qh->qh_head = qe->qe_next)==(struct gs_vqx_qe *) 0) {
		qh->qh_tail = (struct gs_vqx_qe *) 0;
	}
	qe->qe_flags &= ~QE_FL_BUSY;	/* can't be on queue twice	*/
	return(qe->qe_targ);
}

gs_vqx_init(gsaddr, cont)
	register struct gs_vqx_device	*gsaddr;
{
	register struct gs_hacb	*hacb = &gs_hacb[cont];
	register int gs_vqx_fwversion;
	register int gs_vqx_fwcode;
	register struct gs_vqx_device	*addr;
	addr = (struct gs_vqx_device *)IO_ADRS_SCSI;

	bzero(hacb, sizeof(struct gs_hacb));
	bzero(&gs_vqx_qh, sizeof gs_vqx_qh);
	bzero(gs_vqx_qe, sizeof gs_vqx_qe);
	{
		register int i;
		for(i=0; i<Ngs_targ_cont; i++)
			gs_vqx_qe[i].qe_targ = i;
	}
	OPENCHIP->op_sccntrl = 0;
	while(OPENCHIP->op_sccntrl & OP_SC_DACT) {
		DELAY(8);
	}

	/* Initialize the WD33C93 chip */

	if((addr->wd_addr_stat &
		(WD_AUX_BSY | WD_AUX_CIP)) == (WD_AUX_BSY | WD_AUX_CIP)) {
		IO_ADRS_PORT1 = OP1_SET_RESSCSI;
		DELAY(100000);
		IO_ADRS_PORT1 = OP1_RES_RESSCSI;
		DELAY(100000);
	}
	do {
		OPENCHIP->op_scsintr = 0xff;
		DELAY(15);
	} while (OPENCHIP->op_scsintr);

	/*
	 * Set the SCSI ID and issue a reset cmd. Then field the
	 * interrupt which results from the reset.
	 */

	addr->wd_addr_stat = WD_OWNID;
	addr->wd_data = QS_ID;
	addr->wd_addr_stat = WD_COMMAND;
	addr->wd_data = WD_CMD_RESET;
	while (!(OPENCHIP->op_scsintr)) {
		DELAY(15);
	}

	/* Set the source id register and the timeout */
	addr->wd_addr_stat = WD_SOURCE_ID;
	addr->wd_data = 0;
	addr->wd_addr_stat = WD_CONTROL;
	addr->wd_data = WCNT_DMA; 
	addr->wd_addr_stat = WD_TIMEOUT;
	addr->wd_data = QS_TIMEOUT;

	/* we enable interrupts so that the configure code can catch one  */

	OPENCHIP->op_sccntrl = OP_SC_INEA;	
	/* our interrupt vector is not attached until after we return	  */
	/* so we give some time for the interrupt to happen, then we turn */
	/* off the interrupt						  */
	DELAY(10);		/* now or never				  */
	OPENCHIP->op_scsintr = 0xff;

	gsident[cont] = QS_ID;
	gs_run[cont] = gs_vqx_run;
	gs_intr[cont] = gs_vqx_intr;
	gs_cmdwait[cont] = gs_vqx_cmdwait;
	gs_vers[cont] = gs_vqx_vers;
	GSaddr[cont] = (int) gsaddr;

	gs_vqx_fwversion = OPENCHIP->op_version;
	gs_vqx_fwcode = (OPENCHIP->op_system&0x1f)>>1;
	sprintf(gs_vqx_version, "Openchip FW version %d.%d%d, Code %d\n",
		gs_vqx_fwversion/100, (gs_vqx_fwversion%100)/10,
		gs_vqx_fwversion%10, gs_vqx_fwcode);

	hacb->hacb_dcr = 0;
	gs_vqx_started = 0;	/* controller is idle	*/
	return (0);
}

gs_vqx_cmdwait(cont, targ, timeout)
{
	register struct gs_hacb	*hacb = &gs_hacb[cont];
	int			targo = DCR_GO << targ;
	long			l = 0;
	register struct scsi_cmnd *cmd;
	cmd = &gs_vqx_cmd[targ];	/* OPENCHIP command block	*/
	while(!OPENCHIP->op_scsintr) {	/* while would not interrupt	*/
		DELAY(400);
		if(timeout-- == 0) {
			return(-1);
		}
	}
	/* Caller processes errors */
	return(0);
}

gs_vqx_cmdabort(cont, targ)
{
	printf("gs_vqx_cmdabort(%x, %x) called!\n", cont, targ);
#ifdef	DEBUGGER
	trap15();
#endif	DEBUGGER
}

gs_vqx_vers()
{
	printf(" %s", gs_vqx_version);
}

char *
gs_vqx_vtop(vaddr)
unsigned long vaddr;
{
	if(vaddr >= VEQR_ADDR) {
		return((char *)vaddr-VEQR_ADDR);
	} else {
		if(vaddr < MAXU_ADDR) {
			printf("vtop(%x)!!\n", vaddr);
			panic("vtop passed user address\n");
		}
		return((char *)svtop(vaddr));
	}
}

/*
 *
 * setupDMA - setup an OpenChip DMA transfer
 *
 */


setupDMA(addr,ptr,cnt,phase)
	struct gs_vqx_device *addr;
	char *ptr;
	char phase;
{
	short i;
#ifdef	QDTRACE
	qdtrace(0x10001, ptr, (phase<<16)+cnt);
#endif	QDTRACE

	addr->wd_addr_stat = WD_TRANS_COUNT0;
	addr->wd_data = cnt >> 16;
	addr->wd_data = cnt >> 8;
	addr->wd_data = cnt;
	NEWPHASE(phase);

	OPENCHIP->op_scdmaptr = gs_vqx_vtop(ptr);
	OPENCHIP->op_scsum = (char) ptr + (char) cnt;
	addr->wd_addr_stat = WD_COMMAND;
	addr->wd_data = WD_CMD_TRANS_INFO;
	while(addr->wd_addr_stat & WD_AUX_CIP)
		for(i=0;i<8;i++) ;
	OPENCHIP->op_sccntrl |= OP_SC_DACT;
}


/*
 *
 * gs_vqx_setmanual - finish command protocol
 *
 */

gs_vqx_setmanual(wdptr,phase)
	struct gs_vqx_device *wdptr;
	char            phase;
{
	static char     non_val;

	wdptr->wd_addr_stat = WD_TRANS_COUNT0;
	wdptr->wd_data = 0xff;
	/*
	 * It isn't really worth writing a value to any but the most
	 * significant byte of the transfer counter. 
	 */
	wdptr->wd_addr_stat = WD_COMMAND;
	wdptr->wd_data = WD_CMD_TRANS_PAD;
	NEWPHASE(phase);
	if (!(phase & OP_ST_PHS_IO)) {
		OPENCHIP->op_scdmaptr = gs_vqx_vtop(&non_val);
		OPENCHIP->op_sccntrl |= OP_SC_DACT;
	}
}
/*
 *
 * qscmddone - wrap up command processing 
 *
 */

qscmddone(targ,cmd,finstat)
	unsigned char  targ;
	struct scsi_cmnd *cmd;
{
	register unsigned int ctlr;

	cmd->qs_finstat = finstat;
	NEWSTATE(SC_DISCONNECT); 
}
#ifdef	QDTRACE
#define QDTBSIZE 2048
struct qdtb {
	unsigned long where;
	unsigned long what;
	unsigned long what2;
	unsigned long count;
} qdtbuf[QDTBSIZE];

unsigned int qdtraceon = 0;
struct qdtb *qdtp = qdtbuf;

qdtrace(l1,l2,l3)	/* trace into memory buffer */
unsigned long l1,l2,l3;
{
	int s;

	if(!qdtraceon)
		return;
	s = spl7();	/* keep interrupt code out of here	*/
	if(qdtp->where == l1 && qdtp->what == l2 && qdtp->what2 == l3) {
		qdtp->count++;
		splx(s);
		return;
	}
	if(qdtp == &qdtbuf[QDTBSIZE-1]) {
		qdtp = qdtbuf;
		qdtp->where=0;	/* don't increment count based on old data */
	} else {
		qdtp++;
		qdtp->where=0;	/* don't increment count based on old data */
	}
	if(qdtp >= &qdtbuf[QDTBSIZE]) {
		panic("qdtrace: total insanity\n");
	}
	qdtp->where=l1;
	qdtp->what = l2;
	qdtp->what2 = l3;
	qdtp->count = 1;
	splx(s);
}
#endif	QDTRACE
