/*
 * VMEBUS: Integrated Solutions GP-SCSI general purpose interface driver
 */
#include "gg.h"
#if NGG > 0 || Ngg > 0
#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"


#include "../machine/board.h"
#include "../is68kdev/qbvar.h"

#ifdef  VQX
#include "../is68kdev/openchip.h"
#endif  VQX
#include "../h/ggio.h"

/*                _________________________________
 * minor(dev):    | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 *                ---------------------------------
 *                         \___cont__/ \___targ__/
 *                         \_________ctlr________/

 */
#define Ngg_targ_cont		8
#define GG_CTLR(cont, targ)	(((cont) << 3) | (targ))
#define GG_CONT(ctlr)		((ctlr) >> 3)
#define GG_TARG(ctlr)		((ctlr) & 7)
#define GG_MAXCNT		0x1FE00

/*
 * State of controller from last command. Since only one command can be done
 * at a time per target device, allocate one for each device on each controller.
 */
struct	ggstat       {	      /* pseudoctlr status block */
	struct	    {
	    unsigned f_nodisc : 1;	/* disallow disconnects */
	    unsigned f_swcd   : 1;	/* swap command data */
	    unsigned f_swmd   : 1;	/* swap media data */
	    unsigned f_open   : 1;	/* device open */
	    unsigned f_busy   : 1;	/* using dirc control struct */
	    unsigned f_ready  : 1;	/* ctrl struct filled,ready*/
	    unsigned f_want   : 1;	/* want to use ctrl struct */
	    unsigned f_sense  : 1;	/* did a sense on device */
	}		fbits;
#define			s_nodisc	fbits.f_nodisc
#define			s_swcd		fbits.f_swcd
#define			s_swmd		fbits.f_swmd
#define			s_open		fbits.f_open
#define			s_busy		fbits.f_busy
#define			s_ready		fbits.f_ready
#define			s_want		fbits.f_want
#define			s_sense		fbits.f_sense
	int		s_lun;		/* selected lun */
	int		s_bs;		/* block size */
	struct hacb_dcb	s_dcb;		/* direct command to be done  */
	struct	buf	gg_utab;
	int		gg_dcb;
} ggstat[NGG * Ngg_targ_cont];

union scsi_sns		gg_sns;
int 			gg_sns_vdma;
struct scsi_inq		gg_inq;
int 			gg_inq_vdma;
struct scsi_rcap        gg_rcap;
int 			gg_rcap_vdma;

int	ggprobe(),	ggslave(),	ggattach(),	ggintr();
int	gggetcmd(),	ggdoerr(),	ggdgo(),	ggminphys();
caddr_t ggmalloc();
struct buf	*ggmembhp;

struct qb_device	*ggdinfo[NGG * Ngg_targ_cont];
struct buf 		rggbuf[NGG * Ngg_targ_cont];
struct	qb_ctlr 	*ggcinfo[NGG];
extern u_short		*GSstd[];
extern u_short		gsident[];
extern char		*ascii_sns_7_key[], *ascii_pdtype[];

/*
 * GG/gg controller/unit driver structure
 */
struct	qb_driver GGdriver =
	{ggprobe, ggslave, ggattach, ggdgo, GSstd, "gg", ggdinfo, "GG",ggcinfo};

/*
 * ggq is used to tell gs routines that we have commands for it, and to pass
 * pointers to the routines for getting commands and command completion.
 */
struct	gs_queue ggq[NGG * Ngg_targ_cont];

/* Check that controller exists, initialize it, fake it for pseudocontroller */
ggprobe(ggaddr, cont)
	u_short *ggaddr;
{
	return (gsprobe(ggaddr, cont));
}

/* Check that a given unit exists */
ggslave(qi, ggaddr)
	struct qb_device  *qi;
	u_short		   *ggaddr;
{
	int			ctlr = qi->qi_unit;
	register struct ggstat	*stat = &ggstat[ctlr];

	stat->s_nodisc = (qi->qi_flags & 1) ? DCB_NODISC : DCB_DISC;
	stat->s_swcd = (qi->qi_flags & 2) ? DCB_NOSW : DCB_SW;
	stat->s_swmd = (qi->qi_flags & 4) ? DCB_SW : DCB_NOSW;
        if (gg_inq_vdma == 0) {
                gg_inq_vdma = IOPB_STD(iopballoc(&gg_inq, sizeof(gg_inq)),vbnum);
                cache_inhibit(&gg_inq, sizeof(gg_inq));
                gg_sns_vdma = IOPB_STD(iopballoc(&gg_sns, sizeof(gg_sns)),vbnum);
                cache_inhibit(&gg_sns, sizeof(gg_sns));
                gg_rcap_vdma = IOPB_STD(iopballoc(&gg_rcap, sizeof(gg_rcap)),vbnum);
                cache_inhibit(&gg_rcap, sizeof(gg_rcap));
        }

	gsqinit(&ggq[ctlr], gggetcmd, ggdoerr);
	return (1);
}

/* Logically attach a unit to a drive/controler, and set up unit info */
ggattach(qi)
	struct qb_device  *qi;
{
	int	ctlr = qi->qi_unit;
	int	targ = qi->qi_slave;
	int	lun;

	if (targ == (gsident[GG_CONT(ctlr)] & GS_LOC_HID_MASK))
		gsvers(GG_CONT(ctlr));
	else
	    for (lun = 0; lun < 8; lun++)
		gginq(qi, lun);
}

gggetcmd(cont, targ, dcb)
	struct hacb_dcb	   *dcb;
{
	int		ctlr = GG_CTLR(cont, targ);
	struct ggstat   *stat = &ggstat[ctlr];
	register struct qb_device *qi;
        register struct qb_ctlr *qm;

	if (stat->s_ready) {		/* a direct cmd needs to be done */
		bcopy(&stat->s_dcb, dcb, sizeof(struct hacb_dcb));
		if (stat->gg_dcb == 0) {
			return (1);	/* and get it sent off to target */
		} else { 
			stat->gg_dcb = (int)dcb;
			qi = ggdinfo[ctlr];
			qi->qi_mi = ggcinfo[cont];
			qbgo(qi);
			return(2);
		}
	} else {
		gsdeq(cont, targ, &ggq[ctlr]);
		return (0);
	}
}

ggdgo(qm)
        register struct qb_ctlr *qm;
{
        register struct hacb_dcb        *dcb;
	struct ggstat   *stat;
        struct buf      *bp;
        u_char          cmd, dir;
	int		cont, targ, ctlr;

        bp = qm->qm_tab.b_actf->b_actf;
        ctlr = minor(bp->b_dev);
	stat = &ggstat[ctlr];
	dcb = (struct hacb_dcb *)stat->gg_dcb;
	dcb->dcb_dadr = (u_char *)VDMA_STD(qm->qm_qbinfo,vbnum);
	cont = GG_CONT(ctlr);
	targ = GG_TARG(ctlr);
	gsrun(cont, targ);
}

/* do upper level interrupt processing */
ggdoerr(cont, targ, dcb)
	struct hacb_dcb *dcb;
{
	int			ctlr = GG_CTLR(cont, targ);
	register struct ggstat	*stat = &ggstat[ctlr];
	int			lun;
        struct buf      *bp = stat->gg_utab.b_forw;
        register struct qb_ctlr *qm;
        struct buf      *dp;

	if (stat->s_sense) {
		stat->s_sense = 0;
		return (0);
	}
	/* clear ready flag, copy status, and wakeup caller if inte.  */
	stat->s_ready = 0;
	stat->s_dcb.dcb_err = dcb->dcb_err;
	stat->s_dcb.dcb_cerr = dcb->dcb_cerr;
	stat->s_dcb.dcb_scsi_status = dcb->dcb_scsi_status;
	if (!stat->s_open &&	/* in configure routines */
	    (dcb->dcb_scsi_status & SCSI_STATUS_CHECK)) {
		stat->s_sense = 1;
		lun = dcb->dcb_cdb.cdb_0.cdb_0_lun;
		SET_CDB_0(&dcb->dcb_cdb, CMD_SENSE, lun, 0,
		    sizeof(union scsi_sns), 0, 0);
		SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, DCB_NODISC,
		    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
		    gg_sns_vdma, sizeof(union scsi_sns), 1);
		bzero(&gg_sns, sizeof(struct scsi_sns));
		flush_cache_entry(&gg_sns, sizeof(struct scsi_sns));
		return (1);
	}
	if (dcb->dcb_ie)		/* we're in full interrupt mode */
		wakeup(&stat->s_dcb);	/* command done - wake him up */
	if (stat->gg_dcb) {
		stat->gg_dcb = 0;
		qm = ggcinfo[cont];
		qbdone(qm);
		dp = qm->qm_tab.b_actf;
		qm->qm_tab.b_actf = dp->b_forw;
		dp->b_active = 0;
		dp->b_actf = bp->av_forw;
		if (dp->b_actf) {
			dp->b_forw = NULL;
			dp->b_active = 1;
			if (qm->qm_tab.b_actf == NULL)
				qm->qm_tab.b_actf = dp;
			else
				qm->qm_tab.b_actl->b_forw = dp;
			qm->qm_tab.b_actl = dp;
		}
		stat->gg_utab.b_forw = bp->av_forw;
	}
	gsdeq(cont, targ, &ggq[ctlr]);
	return (0);
}

ggstrategy(bp)
	register struct buf	*bp;
{
	int				ctlr = minor(bp->b_dev);
	int				cont = GG_CONT(ctlr);
	int				targ = GG_TARG(ctlr);
	register struct ggstat		*stat = &ggstat[ctlr];
	register struct hacb_dcb *dcb = &stat->s_dcb;
        register struct buf     *bqp = &stat->gg_utab;
        register int            s;
        register struct qb_device *qi = ggdinfo[ctlr];
        register struct qb_ctlr *qm = qi->qi_mi;

	if (stat->s_bs == 0)
		goto bad;
	while (stat->s_busy) {
		stat->s_want = 1;
		sleep(stat, PRIBIO);
	}
	stat->s_busy = 1;
	bp->av_forw = NULL;
        s = splx(qm->qm_psl);
        if (bqp->b_forw == NULL)
                bqp->b_forw = bp;
        else
                bqp->b_back->av_forw = bp;
        bqp->b_back = bp;

        if (bqp->b_active == 0) {
                bqp->b_forw = NULL;
                bqp->b_active = 1;
                if (qm->qm_tab.b_actf == NULL)
                        qm->qm_tab.b_actf = bqp;
                else
                        qm->qm_tab.b_actl->b_forw = bqp;
                qm->qm_tab.b_actl = bqp;
        }

	SET_CDB_1(&dcb->dcb_cdb, (bp->b_flags & B_READ)?CMD_XREAD:CMD_XWRITE,
	    stat->s_lun, 0, bp->b_blkno/(stat->s_bs/512), 
	    (bp->b_bcount + (stat->s_bs-1))/stat->s_bs, 0, 0);
	SET_DCB(dcb, DCB_IE, (bp->b_flags & B_READ)?DCB_DIR_IN:DCB_DIR_OUT,
	    stat->s_swmd, stat->s_nodisc, sizeof(struct cdb_1), DCB_BW_16,
	    DCB_AM_STD_S_D, 0, 0, bp->b_bcount & ~1, stat->s_bs);
	stat->s_ready = 1;
	stat->gg_dcb = (int)dcb;
	if (ggq[ctlr].gsq_forw == 0)
		gsenq(cont, targ, &ggq[ctlr]);
	sleep(dcb, PRIBIO);
	splx(s);
	stat->s_busy = 0;
	if (stat->s_want) {
		stat->s_want = 0;
		wakeup(stat);
	}
	if (stat->s_dcb.dcb_err)
		goto bad;
	bp->b_resid = 0;
	iodone(bp);
	return;
bad:	bp->b_flags |= B_ERROR;
	iodone(bp);
}

ggread(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int	ctlr = minor(dev);

	if (ctlr >= NGG * Ngg_targ_cont)
		return (ENXIO);
	return (physio(ggstrategy, &rggbuf[ctlr], dev, B_READ, ggminphys, uio));
}

ggwrite(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int	ctlr = GG_TARG(minor(dev));

	if (ctlr >= NGG * Ngg_targ_cont)
		return (ENXIO);
	return (physio(ggstrategy, &rggbuf[ctlr], dev, B_WRITE, ggminphys, uio));
}

ggioctl(dev, cmd, op_dcb)
	dev_t				dev;
	register struct hacb_dcb	*op_dcb;
{
	int				ctlr = minor(dev);
	int				cont = GG_CONT(ctlr);
	int				targ = GG_TARG(ctlr);
	register struct ggstat		*stat = &ggstat[ctlr];
	register struct hacb_dcb 	*dcb = &stat->s_dcb;
	caddr_t				ump, smp;
	int				s;
	int				error = 0;
	int				iopb;	

	if (stat->s_open == 0)
		return (NODEV);
	while (stat->s_busy) {
		stat->s_want = 1;
		sleep(stat, PRIBIO);
	}
	stat->s_busy = 1;
	bcopy(op_dcb, dcb, sizeof(struct hacb_dcb));
	dcb->dcb_ie = 1;
	if (dcb->dcb_dlen) {
		ump = dcb->dcb_dadr;
		if (useracc(ump, dcb->dcb_dlen,
		    (dcb->dcb_dir == DCB_DIR_IN) ? B_WRITE : B_READ) == NULL) {
			error = EFAULT;
			goto err;
		}
		s = splx(ggcinfo[ctlr]->qm_psl);
		smp = ggmalloc(dcb->dcb_dlen);
		splx(s);
		if (dcb->dcb_dir == DCB_DIR_OUT)
			copyin(ump, smp, dcb->dcb_dlen);
		else
			bzero(smp, dcb->dcb_dlen);
		iopb = iopballoc(smp,dcb->dcb_dlen);
		dcb->dcb_dadr = (caddr_t)IOPB_STD(iopb,vbnum);
		flush_cache_entry(smp, dcb->dcb_dlen);
	}
	s = splx(ggcinfo[ctlr]->qm_psl);
	stat->s_ready = 1;
	if (ggq[ctlr].gsq_forw == 0)
		gsenq(cont, targ, &ggq[ctlr]);
	sleep(dcb, PRIBIO);
	splx(s);
	op_dcb->dcb_err = dcb->dcb_err;
	op_dcb->dcb_cerr = dcb->dcb_cerr;
	op_dcb->dcb_scsi_status = dcb->dcb_scsi_status;
	if (dcb->dcb_dlen) {
		if (dcb->dcb_dir == DCB_DIR_IN)
			copyout(smp, ump, dcb->dcb_dlen);
		iopbfree(iopb);
		s = splx(ggcinfo[ctlr]->qm_psl);
		ggmfree(smp);
		splx(s);
	}
err:	stat->s_busy = 0;
	if (stat->s_want) {
		stat->s_want = 0;
		wakeup(stat);
	}
	return (error);
}

ggopen(dev)
	dev_t dev;
{
	int			ctlr = minor(dev);
	struct qb_device	*qi;
	struct ggstat		*stat;

	if (!suser())
		return (u.u_error);
	if (ctlr >= NGG * Ngg_targ_cont || (qi = ggdinfo[ctlr]) == 0 ||
	    qi->qi_alive == 0)
		return (ENXIO);
	stat = &ggstat[ctlr];
	if (stat->s_open)
		return (EBUSY);
	stat->s_open = 1;
	return (0);
}

ggclose(dev)
	dev_t dev;
{
	int			ctlr = minor(dev);
	struct qb_device	*qi;

	ggstat[ctlr].s_open = 0;
	return (0);
}

ggissue(qi, stat, dcb, sense)
	register struct qb_device	*qi;
	register struct ggstat		*stat;
	register struct hacb_dcb	*dcb;
{
	u_char				retrycount = 0;

    while (1) {
	stat->s_ready = 1;
	if (gsfakeint(qi->qi_ctlr, qi->qi_slave, &ggq[qi->qi_unit], 800000))
		return (-1);
	if (dcb->dcb_err == 0)
		return (0);
	if ((dcb->dcb_scsi_status & SCSI_STATUS_CHECK) &&
	    (gg_sns.sns_7.sns_7_class == 7))
		switch (gg_sns.sns_7.sns_7_key) {
			case SNS_7_KEY_NO_SENSE:
			case SNS_7_KEY_RECOVERED:
				return (0);
			case SNS_7_KEY_UNIT_ATTN:
			case SNS_7_KEY_ABORT_CMD:
				if (--retrycount)
					continue;
			default:
				return (gg_sns.sns_7.sns_7_key);
		}
	return (-1);
    }
}

gginq(qi, lun)
	struct qb_device	*qi;
{
	register struct ggstat		*stat = &ggstat[qi->qi_unit];
	register struct hacb_dcb 	*dcb = &stat->s_dcb;

	SET_CDB_0(&dcb->dcb_cdb, CMD_INQUIRY, lun, 0, sizeof(struct scsi_inq),
	    0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
	    gg_inq_vdma, sizeof(struct scsi_inq), 1);
	bzero(&gg_inq, sizeof(struct scsi_inq));
	flush_cache_entry(&gg_inq, sizeof(struct scsi_inq));
	if (ggissue(qi, stat, dcb, 1) || (gg_inq.inq_pdtype == INQ_PDT_NOLUN))
		return (-1);
	SET_CDB_0(&dcb->dcb_cdb, CMD_TESTREADY, lun, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, 0, DCB_NODISC,
	    sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
	switch (ggissue(qi, stat, dcb, 1)) {
		case SNS_7_KEY_NO_SENSE:
		case SNS_7_KEY_NOT_READY:
			break;
		default:
			return (-1);
	}
	if (lun)
		printf("\n				");
	printf("	%d:", lun);
	if (gg_inq.inq_pdtype <= INQ_PDT_WORM)
		printf("%6s: ", ascii_pdtype[gg_inq.inq_pdtype]);
	else
		printf("PDT %2x: ", gg_inq.inq_pdtype);
	printf("%8s %10s %4s", gg_inq.inq_vendor, gg_inq.inq_product,
		gg_inq.inq_rev);
#ifdef UNDEF
	/* start the unit and do a read capacity for read/write interface */
	SET_CDB_0(&dcb->dcb_cdb, CMD_STARTSTOP, lun, 0, 1, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, 0, DCB_NODISC, sizeof(struct cdb_0),
	    0, 0, 0, 0, 0, 0);
	if (ggissue(qi, stat, dcb, 1))
		return (0);
	DELAY(200000);			/* warm up, dont want BUSY status */
	SET_CDB_1(&dcb->dcb_cdb, CMD_RCAPAC, lun, 0, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, DCB_NODISC,
	    sizeof(struct cdb_1), DCB_BW_16, DCB_AM_STD_S_D, 0, 
	    gg_rcap_vdma, sizeof(gg_rcap), 1);
	flush_cache_entry(&gg_rcap, sizeof (gg_rcap));
	if (ggissue(qi, stat, dcb, 1) ||
	    gg_rcap.rcap_bl < 512 || (gg_rcap.rcap_bl % 512) != 0)
		return (0);
	stat->s_bs = gg_rcap.rcap_bl;
#endif UNDEF
	return (0);
}

ggintr(cont)
{
	return (gsintr(cont));
}

ggminphys(bp)
	struct buf	*bp;
{
	bp->b_bcount = MIN(bp->b_bcount, GG_MAXCNT);
}

caddr_t
ggmalloc(len)
{
	struct buf	*bp = (struct buf *)geteblk(len);

	bp->b_actf = ggmembhp;
	ggmembhp = bp;
	return (bp->b_un.b_addr);
}

ggmfree(addr)
	register caddr_t addr;
{
	register struct buf	*bp, **op = &ggmembhp;

	for (bp = ggmembhp; bp; bp = bp->b_actf) {
		if (bp->b_un.b_addr == addr) {
			*op = bp->b_actf;
			bfree(bp);
			brelse(bp);
			return (0);
		}
		op = &(bp->b_actf);
	}
	panic("gg: lost buffer pointer");
}
#endif
