/*
 * VMEBUS: Integrated Solutions GP-SCSI disk driver
 */
#include "gd.h"

#if NGD > 0 || Ngd > 0
#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dk.h"
#include "../h/dkbad.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"

#ifdef	VQX
#include "../is68kdev/openchip.h"
#endif	VQX
#include "../is68kdev/qbvar.h"
#include "../is68kdev/scsi.h"
#include "../is68kdev/gsreg.h"
#include "../is68kdev/gsvar.h"
#include "../h/gdio.h"
#include "../machine/board.h"
/*                _________________________________
 * minor(dev):    | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 *                ---------------------------------
 *                 \_cont/ \_targ/ \l/ \partition/
 *                 \_____ctlr____/
 *                 \_______unit______/
 */
#define Ngd_lun_targ		2
#define Ngd_targ_cont		4
#define	GDD_PART(dev)		((dev) & 7)
#define	GDD_UNIT(dev)		((dev) >> 3)
#define	GDU_LUN(unit)		((unit)&1)
#define	GDU_CTLR(unit)		(((unit) >> 1)&0xF)
#define	GDU_TARG(unit)		(GDU_CTLR(unit) & 3)
#define	GDU_CONT(unit)		(GDU_CTLR(unit) >> 2)
#define GDCL_UNIT(ctlr, lun)	(((ctlr) << 1) | (lun))
#define GDCT_CTLR(cont, targ)	(((cont) << 2) | (targ))

#define GD_MAXCNT	0x1FE00
#define	GD_MAXRETRY	3

/*
 * State of controller from last transfer. Since only one transfer can be done
 * at a time per target device, allocate one for each device on each controller.
 */
struct	gdstat       {	       /* pseudoctlr status block */
	struct	{
	    u_char f_nodisc : 1;	/* disallow disconnects */
	    u_char f_swcd   : 1;	/* swap command data */
	    u_char f_swmd   : 1;	/* swap media data */
	    u_char f_noerr  : 1;	/* ignore errors */
	    u_char f_sense  : 1;	/* sense was requested */
	    u_char f_direct : 1;	/* direct polled mode command */
	    u_char f_restart : 1;	/* restart motor after unitattn */
			    : 1;
	    u_char f_lun;		/* current/next lun to access */
	}		gd_statbits;
#define			    gd_nodisc	gd_statbits.f_nodisc
#define			    gd_swcd	gd_statbits.f_swcd
#define			    gd_swmd	gd_statbits.f_swmd
#define			    gd_noerr	gd_statbits.f_noerr
#define			    gd_sense	gd_statbits.f_sense
#define			    gd_direct	gd_statbits.f_direct
#define			    gd_restart	gd_statbits.f_restart
#define			    gd_lun	gd_statbits.f_lun
	struct hacb_dcb *gd_dcb;
	caddr_t		gd_addr;	/* physical address of transfer */
	u_int		gd_lba;		/* block number */
	int		gd_bleft;	/* bytes left to transfer */
	int		gd_bpart;	/* bytes transferred */
	struct hacb_dcb	gd_dirc;	/* direct command block */
	struct buf	gd_qhd[Ngd_lun_targ]; /* heads of drive queues */
	struct qb_ctlr	gd_cinfo;	/* controller queue */
	union scsi_sns	gd_sns;		/* for sense on normal r/w */
	int		gd_sns_vdma;	/* vdma address for sense on normal r/w */
} gdstat[NGD * Ngd_targ_cont];

int	gdprobe(),	gdslave(),	gdattach(),	gdintr(),	gddgo();
int	gdstrat(),	gdminphys();
int	gdgetcmd(),	gddoerr();

struct	qb_device	*gddinfo[Ngd];
struct	qb_ctlr 	*gdcinfo[NGD];
extern	u_short		*GSstd[];
extern	char		*ascii_sns_7_key[], *ascii_pdtype[];

/* GD/gd controller/unit driver structure */
struct	qb_driver GDdriver =
 	{gdprobe, gdslave, gdattach, gddgo, GSstd, "gd", gddinfo, "GD",gdcinfo};

struct qb_diskst gdst[Ngd];

struct scsi_inq		gd_inq;
int			gd_inq_vdma;
struct scsi_msen	gd_msen;
int			gd_msen_vdma;
struct scsi_msel	gd_msel;
int			gd_msel_vdma;
struct scsi_rcap	gd_rcap;
int			gd_rcap_vdma;
int			gd_stat_vdma;

/*
 * gdq is used to tell gs routines that we have commands for it, and to pass
 * pointers to the routines for getting commands and handling completion.
 */
struct gs_queue	gdq[NGD * Ngd_targ_cont];
struct buf	rgdbuf[Ngd];
struct dkbad	gdbad[Ngd];
#define b_lba	b_resid

/* Check that controller exists */
gdprobe(gdaddr, cont)
	ushort	*gdaddr;
{
	return (gsprobe(gdaddr, cont));
}

/* Check that a given unit exists and is really a drive */
gdslave(qi, gdaddr)
	struct qb_device	*qi;
	struct gddevice		*gdaddr;
{
	int		ctlr = GDU_CTLR(qi->qi_unit);
	struct gdstat	*stat = &gdstat[ctlr];

#ifdef	ZYCAD
	stat->gd_nodisc = (qi->qi_flags & 1) ? DCB_DISC : DCB_NODISC;
#else
	stat->gd_nodisc = (qi->qi_flags & 1) ? DCB_NODISC : DCB_DISC;
#endif
	stat->gd_swcd = (qi->qi_flags & 2) ? DCB_NOSW : DCB_SW;
	stat->gd_swmd = (qi->qi_flags & 4) ? DCB_SW : DCB_NOSW;
        if (gd_inq_vdma == 0) {
                gd_inq_vdma = IOPB_STD(iopballoc(&gd_inq, sizeof(gd_inq)),vbnum);
                cache_inhibit(&gd_inq, sizeof(gd_inq));
                gd_msen_vdma = IOPB_STD(iopballoc(&gd_msen, sizeof(gd_msen)),vbnum);
                cache_inhibit(&gd_msen, sizeof(gd_msen));
                gd_msel_vdma = IOPB_STD(iopballoc(&gd_msel, sizeof(gd_msel)),vbnum);
                cache_inhibit(&gd_msel, sizeof(gd_msel));
                gd_rcap_vdma = IOPB_STD(iopballoc(&gd_rcap, sizeof(gd_rcap)),vbnum);
                cache_inhibit(&gd_rcap, sizeof(gd_rcap));
		gd_stat_vdma = IOPB_STD(iopballoc(gdstat, sizeof(gdstat)),vbnum);
                cache_inhibit(gdstat, sizeof(gdstat));
        }
        stat->gd_sns_vdma = (int)&stat->gd_sns - (int)gdstat + gd_stat_vdma;
	gsqinit(&gdq[ctlr], gdgetcmd, gddoerr);
	stat->gd_cinfo.qm_driver = &GDdriver;
	return gdinq(qi);
}

/* Logically attach a unit to a drive/controler, and set up unit info */
gdattach(qi)
	struct qb_device	*qi;
{
	if (gdgeom(qi))
		qi->qi_alive = 0;
	gdslctmod(qi);
}


gdgetcmd(cont, targ, dcb)
	register struct hacb_dcb	*dcb;
{
	register struct qb_ctlr *qm;
	register struct buf *bp;
	register int ctlr = GDCT_CTLR(cont,targ);
	register struct gdstat	*stat;
	register struct qb_device *qi;

 	stat = &gdstat[ctlr];
	/* if direct command, copy stuff into dcb and return (1); */
	if (stat->gd_direct) {
		stat->gd_lun = stat->gd_dirc.dcb_cdb.cdb_0.cdb_0_lun;
		bcopy(&stat->gd_dirc, dcb, sizeof(struct hacb_dcb));
		return (1);
	}
	qm = &gdstat[ctlr].gd_cinfo;
		
	if (qm->qm_tab.b_actf == NULL){
		gsdeq(cont, targ, &gdq[ctlr]);
		return(0);
	}

	if ((bp = qm->qm_tab.b_actf->b_actf) == NULL) {
		gsdeq(cont, targ, &gdq[ctlr]);
		return(0);
	}

	/* save away current data transfer drive info */
	if (stat->gd_bleft == 0) { /* not already working on a buffer */
		stat->gd_lba	= bp->b_lba;
		stat->gd_bleft	= bp->b_bcount;
		qi = gddinfo[GDD_UNIT(minor(bp->b_dev))];
		stat->gd_dcb = dcb;
		qi->qi_mi = qm;
		qbgo(qi);
		return(2);
	}
	stat->gd_bpart = MIN(stat->gd_bleft, GD_MAXCNT);

	SET_CDB_1(&dcb->dcb_cdb, (bp->b_flags & B_READ)?CMD_XREAD:CMD_XWRITE,
	    stat->gd_lun, 0, stat->gd_lba, (stat->gd_bpart + 511) >> 9, 0, 0);
	SET_DCB(dcb, DCB_IE, (bp->b_flags & B_READ)?DCB_DIR_IN:DCB_DIR_OUT,
	    stat->gd_swmd, stat->gd_nodisc, sizeof(struct cdb_1), DCB_BW_16,
	    DCB_AM_STD_S_D, 0, stat->gd_addr, stat->gd_bpart & ~1, 512);
	return(1);
}

gddgo(qm)
	register struct qb_ctlr *qm;
{
	register struct hacb_dcb	*dcb;
	int			unit;
	int			ctlr;
	int			cont;
	int			targ;
	register struct gdstat	*stat;
	int			lun;
	struct buf		*bp;

	bp = qm->qm_tab.b_actf->b_actf;
	unit = GDD_UNIT(minor(bp->b_dev));
	cont = GDU_CONT(unit);
	targ = GDU_TARG(unit);
 	ctlr = GDCT_CTLR(cont, targ);
 	stat = &gdstat[ctlr];
	dcb = stat->gd_dcb;
	stat->gd_lun = lun = GDU_LUN(unit);
	stat->gd_addr	= (caddr_t )(VDMA_STD(qm->qm_qbinfo,vbnum));
	stat->gd_bpart = MIN(stat->gd_bleft, GD_MAXCNT);
	SET_CDB_1(&dcb->dcb_cdb, (bp->b_flags & B_READ)?CMD_XREAD:CMD_XWRITE,
	    lun, 0, stat->gd_lba, (stat->gd_bpart + 511) >> 9, 0, 0);
	SET_DCB(dcb, DCB_IE, (bp->b_flags & B_READ)?DCB_DIR_IN:DCB_DIR_OUT,
	    stat->gd_swmd, stat->gd_nodisc, sizeof(struct cdb_1), DCB_BW_16,
	    DCB_AM_STD_S_D, 0, stat->gd_addr, stat->gd_bpart & ~1, 512);
	gsrun(cont, targ);
}

/* do upper level interrupt processing */
gddoerr(cont, targ, dcb)
	register struct hacb_dcb	*dcb;
{
	int			ctlr = GDCT_CTLR( cont, targ );
	register struct gdstat	*stat = &gdstat[ctlr];
	register union scsi_sns	*sns = &stat->gd_sns;
	int			lun = stat->gd_lun;
	register struct buf	*bp;
#ifdef notdef
	int			dk = gddinfo[GDCL_UNIT(ctlr, lun)]->qi_dk;
#endif notdef
	register struct qb_ctlr *qm;
	register struct buf	*dp;

	if (dcb->dcb_scsi_status & SCSI_STATUS_BUSY)
	    return (0); /* do it again!  (be nice to sleep here) */

	if (stat->gd_direct == 0) {
		qm = &gdstat[ctlr].gd_cinfo;
		if (qm->qm_tab.b_actf == NULL)
			return(stat->gd_sense = 0);
		bp = qm->qm_tab.b_actf->b_actf;
	}
	if (stat->gd_noerr) {
	    stat->gd_direct = 0;
	    if (stat->gd_sense) {
#ifdef	ZYCAD
		if (sns->sns_7.sns_7_class == 7 &&
		    sns->sns_7.sns_7_key == SNS_7_KEY_RECOVERED)
			stat->gd_dirc.dcb_err = 0;
#endif
		return (stat->gd_sense = 0);
	    }
	    stat->gd_dirc.dcb_err = dcb->dcb_err;
	    stat->gd_dirc.dcb_cerr = dcb->dcb_cerr;
	    stat->gd_dirc.dcb_scsi_status = dcb->dcb_scsi_status;
	    if (dcb->dcb_err && (dcb->dcb_scsi_status & SCSI_STATUS_CHECK)) {
		stat->gd_sense = 1;
		SET_CDB_0(&dcb->dcb_cdb, CMD_SENSE, lun, 0,
		    sizeof(union scsi_sns), 0, 0);
		SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
		    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
		    stat->gd_sns_vdma, sizeof(union scsi_sns), 1);
		bzero(sns, sizeof(union scsi_sns));
		flush_cache_entry(sns, sizeof (union scsi_sns));
		return (1);
	    }
	    return (0);
	}
	if (stat->gd_restart) {
		if (dcb->dcb_err)
			printf("failed\n");
		else
			printf("succeeded\n");
		return(stat->gd_restart = 0);
	}
	/* if call was a sense request; finish off processing.  */
	if (stat->gd_sense) {
		if (dcb->dcb_err) {
			printf("sense failed: cerr %x: scsi %b", dcb->dcb_cerr,
				dcb->dcb_scsi_status, SCSI_STATUS_BITS);
			if ((stat->gd_sense++ < 3) &&
			    (dcb->dcb_scsi_status &  SCSI_STATUS_CHECK)) {
				printf(": retried\n");
				return (1);
			}
			printf("no more retries");
			bzero(sns, sizeof(union scsi_sns));
			dcb->dcb_err = 0;
		}
		stat->gd_sense = 0;
		if (sns->sns_7.sns_7_class != 7) {
			printf("invalid sense class %d", 
				sns->sns_7.sns_7_class);
			bzero(sns, sizeof(union scsi_sns));
		}
		if (sns->sns_7.sns_7_key)
			printf("%s: code %x",
				ascii_sns_7_key[sns->sns_7.sns_7_key],
				sns->sns_7.sns_7_err);
		if (sns->sns_7.sns_7_key == SNS_7_KEY_UNIT_ATTN) {
			printf(": restarting: ");
			stat->gd_restart = 1;
			SET_CDB_0(&dcb->dcb_cdb, CMD_STARTSTOP, lun, 0,1,0,0);
			SET_DCB(dcb, DCB_IE, DCB_DIR_NO, 0, stat->gd_nodisc,
				sizeof(struct cdb_0), 0, 0, 0, 0, 0, 0);
			return(1);
		}
		if (sns->sns_7.sns_7_valid)
			printf(": bn %d (0x%x)",
				SCSI_HMML(sns->sns_7.sns_7_info),
				SCSI_HMML(sns->sns_7.sns_7_info));
		if (sns->sns_7.sns_7_key == SNS_7_KEY_RECOVERED)
			printf(": soft error\n");
		else if (++stat->gd_qhd[lun].b_errcnt < GD_MAXRETRY) {
			printf(": reissued(%d)\n", stat->gd_qhd[lun].b_errcnt);
			return (0);
		} else if (!stat->gd_direct) {
			printf("\n");
			harderr(bp, "gd");
			bp->b_flags |= B_ERROR;
		} else
			stat->gd_dirc.dcb_err = 0x8000;
	}

	/* check errors, get sense if available */
	if (dcb->dcb_err) {
		printf("gd%d: ", GDCL_UNIT(ctlr, lun));
		if (dcb->dcb_scsi_status & SCSI_STATUS_CHECK) {
			/* start sense request processing */
			stat->gd_sense = 1;
			SET_CDB_0(&dcb->dcb_cdb, CMD_SENSE, lun, 0,
			    sizeof(union scsi_sns), 0, 0);
			SET_DCB(dcb, (stat->gd_direct?DCB_NOIE:DCB_IE),
			    DCB_DIR_IN, stat->gd_swcd, stat->gd_nodisc,
			    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
			    stat->gd_sns_vdma, sizeof(union scsi_sns), 1);
			flush_cache_entry(sns, sizeof (union scsi_sns));
			return (1);
		} else if (dcb->dcb_cerr)
			printf("cerr %x", dcb->dcb_cerr);
		else
			printf("scsi status %b",
				dcb->dcb_scsi_status, SCSI_STATUS_BITS);
		if (++stat->gd_qhd[lun].b_errcnt < GD_MAXRETRY) {
			printf(": retried\n");
			return (0);
		}
		if (!stat->gd_direct)
			bp->b_flags |= B_ERROR;
		printf("\n");
	}
	stat->gd_qhd[lun].b_errcnt = 0;

	/* return if direct command */
	if (stat->gd_direct)
		return (stat->gd_direct = 0);

	/*
	 * check if more data from previous request; if so, update stat struct.
	 * Leave gd_lun alone so we're next.
	 */
	if (((bp->b_flags & B_ERROR) == 0) &&
	    ((stat->gd_bleft -= stat->gd_bpart) > 0)) {
		stat->gd_addr += stat->gd_bpart;
		stat->gd_lba += (( stat->gd_bpart + 511 ) >> 9 );
		return (0);
	}

	/*
	 * if no more data, or had unrecoverable error, clear dk_busy, reset
	 * errcnt, set b_resid, remove buf from qhead, and mark buffer iodone.
	 * Advance to next gd_lun so others get a chance.
	 */
#ifdef notdef
	dk_busy &= ~(1 << dk);
#endif
	bp->b_resid = stat->gd_bleft;
	stat->gd_bleft = 0;
	dp = qm->qm_tab.b_actf;
	qm->qm_tab.b_actf = dp->b_forw;
	dp->b_active = 0;
	dp->b_actf = bp->av_forw;
	if (dp->b_actf) {
		dp->b_forw = NULL;
		dp->b_active = 1;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
	}
	iodone(bp);
	qbdone(qm);
	return (0);
}

gdioctl(dev, cmd, data)
	dev_t	dev;
	char	*data;
{
	int 		unit = GDD_UNIT(minor(dev));
	struct gdstat	*stat = &gdstat[GDU_CTLR(unit)];

	switch (cmd) {
	    case GDIOCDISC:
		stat->gd_nodisc = DCB_DISC;
		return(0);
		break;
	    case GDIOCNODISC:
		stat->gd_nodisc = DCB_NODISC;
		return(0);
		break;
	    case DIOCGGEO:
		if (unit > Ngd)
			return(ENXIO);
		*(struct qb_diskst *)data = gdst[unit];
		return(0);
		break;
	    default:
		return(EIO);
		break;
	}
}

gdopen(dev)
	dev_t	dev;
{
	int			unit = GDD_UNIT(minor(dev));
	struct qb_device	*qi;

	if (unit >= Ngd || (qi = gddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	return (0);
}

gdstrategy(bp)
	register struct buf	*bp;
{
	register struct qb_diskst	*st;
	int			unit = GDD_UNIT(minor((bp)->b_dev));
	struct qb_device	*qi = gddinfo[unit];
	int			part = GDD_PART(minor(bp->b_dev));

	if (unit >= Ngd || qi == 0 || qi->qi_alive == 0) {
		bp->b_error = ENXIO;
		goto bad;
	}
	st = &gdst[unit];
	if (bp->b_blkno < 0 || 
	    (bp->b_blkno+((bp->b_bcount+511)>>9)) > st->st_size[part].nblocks) {
		if (bp->b_blkno == st->st_size[part].nblocks) {
		    bp->b_resid = bp->b_bcount;
		    goto done;
		}
		bp->b_error = EINVAL;
		goto bad;
	}
	badstrat(qi, bp, &gdbad[qi->qi_unit], gdstrat, st->st_nspt, st->st_ntpc,
		st->st_ncpd, st->st_size[part].cyloff);
	return;
bad:	bp->b_flags |= B_ERROR;
done:	iodone(bp);
}

gdstrat(qi, bp)
	struct qb_device	*qi;
	register struct buf	*bp;
{
	int		part = GDD_PART(minor(bp->b_dev)), s;
	int		unit = GDD_UNIT(minor(bp->b_dev));
	int		ctlr = GDU_CTLR(unit);
	int		cont = GDU_CONT(unit);
	struct qb_diskst	*st = &gdst[unit];
	register struct buf *dp = &gdstat[ctlr].gd_qhd[GDU_LUN(unit)];
	register struct qb_ctlr *qm = &gdstat[ctlr].gd_cinfo;

	bp->b_lba = bp->b_blkno + (st->st_size[part].cyloff * st->st_nspc);

	/* put buffer on units queue for gdgetcmd to remove when called */
	s = splx(gdcinfo[cont]->qm_psl);
	disksort(dp, bp);

	if (dp->b_active == 0) {
		dp->b_forw = NULL;
		dp->b_active = 1;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
	}

	/* if we aren't queued for this target, put us on */
	if (gdq[ctlr].gsq_forw == (struct gs_queue *)0)
		gsenq(cont, GDU_TARG(unit), &gdq[ctlr]);
	splx(s);
}

gdread(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int	unit = GDD_UNIT(minor(dev));

	if (unit >= Ngd)
		return (ENXIO);
	return (physio(gdstrategy, &rgdbuf[unit], dev, B_READ, gdminphys, uio));
}

gdwrite(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int	unit = GDD_UNIT(minor(dev));

	if (unit >= Ngd)
		return (ENXIO);
	return (physio(gdstrategy, &rgdbuf[unit], dev, B_WRITE, gdminphys, uio));
}

gdreset(ctlr)
{
}

int gddumpstrat();
extern struct buf dumpbuf;
extern int dumpsize;

gddump(dev)
	dev_t	dev;
{
	register struct buf		*bp = &dumpbuf;
	register struct qb_device	*qi;
	register struct qb_diskst		*st;
	int				num, start, bcount;
	int				unit = GDD_UNIT(minor(dev));
	int				part = GDD_PART(minor(dev));
	int				vdma;

	if (unit >= Ngd || (qi = gddinfo[unit]) == 0 || qi->qi_alive == 0)
		return ENXIO;
	num = ctod(dumpsize);		/* memory size in disk blocks */
	start = 0x0;			/* start dumping at physical 0 */
	st = &gdst[unit];
	if (dumplo < 0 || dumplo+num > st->st_size[part].nblocks)
		return EINVAL;
	while (num > 0) {
		bcount = (dbtob(num) > GD_MAXCNT) ? GD_MAXCNT : dbtob(num);
		bp->b_dev = dev;
		bp->b_blkno = dumplo + btodb(start);
		bp->b_lba = bp->b_blkno +
			(st->st_size[part].cyloff * st->st_nspc);
		bp->b_bcount = bcount;
#ifdef  VQX
                vdma = qbdumpsetup((caddr_t)start+PHYSMEMBASE, bcount);
#else   VQX
                vdma = qbdumpsetup((caddr_t)start, bcount);
#endif  VQX
		bp->b_un.b_addr = (caddr_t)(VDMA_STD(vdma,vbnum));
		bp->b_flags = B_WRITE;
		badstrat(qi, bp, &gdbad[qi->qi_unit], gddumpstrat, st->st_nspt,
		    st->st_ntpc, st->st_ncpd, st->st_size[part].cyloff);
		qbrelse(&vdma);
		if (bp->b_flags & B_ERROR || bp->b_resid)
			return EIO;
		start += bcount;
		num -= btodb(bcount);
	}
	return 0;
}

gddumpstrat(qi, bp)
	register struct qb_device	*qi;
	register struct buf		*bp;
{
	register struct gdstat	*stat = &gdstat[GDU_CTLR(qi->qi_unit)];
	struct hacb_dcb	 *dcb = &stat->gd_dirc;

	SET_CDB_1(&dcb->dcb_cdb, CMD_XWRITE, GDU_LUN(qi->qi_unit), 0,
	    bp->b_lba, (bp->b_bcount + 511) >> 9, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_OUT, stat->gd_swmd, DCB_NODISC,
	    sizeof(struct cdb_1), DCB_BW_16, DCB_AM_STD_S_D, 0, bp->b_un.b_addr,
	    bp->b_bcount, 512);
	if (gdissue("WRITE", dcb, qi->qi_unit))
		bp->b_flags |= B_ERROR;
	bp->b_resid = 0;
	bp->b_flags |= B_DONE;
	return 0;
}

gdsize(dev)
	dev_t	dev;
{
	struct qb_device	*qi;
	struct qb_diskst		*st;
	int			unit = GDD_UNIT(minor(dev));

	if (unit >= Ngd || (qi = gddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	st = &gdst[unit];
	return (st->st_size[GDD_PART(minor(dev))].nblocks);
}

gdminphys(bp)
	struct buf	*bp;
{
	bp->b_bcount = MIN(bp->b_bcount, GD_MAXCNT);
}

gdissue(command, dcb, unit)
	register char                   *command;
	register struct hacb_dcb        *dcb;
{
	struct hacb_dcb                 dcbsav;
	int				count = 0;

	bcopy(dcb, &dcbsav, sizeof(struct hacb_dcb));
	while (1) {
	    switch (gdissue2(command, dcb, unit)) {
		case  0:	return (0);			/* ok */
		case -1:	if (command)
				    printf("%s: command failed\n", command);
				return (-1);			/* hard error */
		case 1:		if (++count == GD_MAXRETRY) {     /* retry */
				    if (command)
					printf("%s: command failed\n", command);
				    return (-1);
				}
	    }
	    bcopy(&dcbsav, dcb, sizeof(struct hacb_dcb));
	}
}

gdissue2(command, dcb, unit)
	register char			*command;
	register struct hacb_dcb	*dcb;
{
	register struct gdstat	*stat = &gdstat[GDU_CTLR(unit)];

	stat->gd_direct = 1;
	stat->gd_noerr = 1;
	if (gsfakeint(GDU_CONT(unit), GDU_TARG(unit),
	    &gdq[GDU_CTLR(unit)], 800000)) {
		stat->gd_noerr = 0;
		return(-1);
	}
	stat->gd_noerr = 0;
        if (dcb->dcb_err == 0)          /* return if no error */
		return (0);
        if (((dcb->dcb_scsi_status & SCSI_STATUS_CHECK) == 0) ||
          (stat->gd_sns.sns_7.sns_7_class != 7)) {
            if (command)
		printf("gd%d: cerr %x: fatal error: %s: scsi %b\n", 
		    unit, dcb->dcb_cerr, command, dcb->dcb_scsi_status,
                    SCSI_STATUS_BITS);
            return (-1);
        }
        if (command)
		printf("gd%d: %s: %s: code %x", unit, command,
            		ascii_sns_7_key[stat->gd_sns.sns_7.sns_7_key],
			stat->gd_sns.sns_7.sns_7_err);
        if (stat->gd_sns.sns_7.sns_7_valid)
            if (command)
                printf(": bn %d (0x%x)",
			SCSI_HMML(stat->gd_sns.sns_7.sns_7_info),
                	SCSI_HMML(stat->gd_sns.sns_7.sns_7_info));
        switch (stat->gd_sns.sns_7.sns_7_key) {
            case SNS_7_KEY_RECOVERED:
                if (command)
                	printf(": soft error\n");
                return (0);
            case SNS_7_KEY_ABORT_CMD:
                if (command)
                	printf(": reissued\n");
		/* fall through */
            case SNS_7_KEY_UNIT_ATTN:
                return (1);
            default:
                if (command)
                	printf(": hard error\n");
                return (-1);
        }
}

gdinq(qi)
	struct qb_device  *qi;
{
	int				unit = qi->qi_unit;
	int				ctlr = GDU_CTLR(unit);
	register struct gdstat		*stat = &gdstat[ctlr];
	register struct hacb_dcb 	*dcb = &stat->gd_dirc;
	register int			lun = GDU_LUN(unit);

	SET_CDB_0(&dcb->dcb_cdb, CMD_INQUIRY, lun, 0, sizeof(gd_inq), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_inq_vdma,
	    sizeof(gd_inq), 1);
	gd_inq.inq_pdtype = INQ_PDT_NOLUN;
	flush_cache_entry(&gd_inq, sizeof (gd_inq));
	if (gdissue(0, dcb, unit))
		return (0);
	if (gd_inq.inq_pdtype != INQ_PDT_DISK)
		return (0);

	SET_CDB_0(&dcb->dcb_cdb, CMD_STARTSTOP, lun, 0, 1, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, 0, DCB_NODISC, sizeof(struct cdb_0),
	    0, 0, 0, 0, 0, 0);
	if (gdissue(0, dcb, unit))
		return (0);
	DELAY(600000);			/* warm up, dont want BUSY status */

	SET_CDB_0(&dcb->dcb_cdb, CMD_TESTREADY, lun, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, 0, DCB_NODISC, sizeof(struct cdb_0),
	    0, 0, 0, 0, 0, 0);
	if (gdissue("TESTREADY", dcb, unit))
		return (0);
	return (1);
}

gdgeom(qi)
	struct qb_device  *qi;
{
	int				unit = qi->qi_unit;
	int				ctlr = GDU_CTLR(unit);
	struct qb_diskst			*st = &gdst[unit];
	register struct gdstat		*stat = &gdstat[ctlr];
	register struct hacb_dcb 	*dcb = &stat->gd_dirc;
	register int			lun = GDU_LUN(unit);
	register struct pag_desc	*pd;
	register int			ncpd, ntpc, nspt, spd;

	/* get capacity information */
	SET_CDB_1(&dcb->dcb_cdb, CMD_RCAPAC, lun, 0, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_1), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_rcap_vdma,
	    sizeof(gd_rcap), 1);
	flush_cache_entry(&gd_rcap, sizeof (gd_rcap));
	if (gdissue("RCAPAC", dcb, unit))
		return (-1);
	if (gd_rcap.rcap_bl != 512) {
		printf("gd%d: bad blocksize %d\n", unit, gd_rcap.rcap_bl);
		return (-1);
	}
	spd = gd_rcap.rcap_lba + 1;	/* sectors per disk */

	/* get current value geometry information */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, PD_GEOMETRY<<8,
	    sizeof(gd_msen), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msen_vdma,
	    sizeof(gd_msen), 1);
	flush_cache_entry(&gd_msen, sizeof (gd_msen));
	if (gdissue("MSENSE GEO", dcb, unit))
		return (-1);
	pd = (struct pag_desc *)(((char *)&gd_msen.msen_bd)+gd_msen.msen_bdl);
	ncpd = SCSI_HML(pd->pd_pg.pg_geo.geo_cyl);
	ntpc =  pd->pd_pg.pg_geo.geo_heads;

	/* get current value format information */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, PD_FORMAT<<8,
	    sizeof(gd_msen), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msen_vdma,
	    sizeof(gd_msen), 1);
	flush_cache_entry(&gd_msen, sizeof (gd_msen));
	if (gdissue("MSENSE FMT", dcb, unit))
		return (-1);
	pd = (struct pag_desc *)(((char *)&gd_msen.msen_bd)+gd_msen.msen_bdl);
	nspt =  pd->pd_pg.pg_fmt.fmt_spt;

	/* test geometry for consistency */
	if ((nspt * ntpc * ncpd) == 0) {
		printf("gd%d: nonsense geometry (%d x %d x %d)\n",
			unit, nspt, ntpc, ncpd);
		return (-1);
	}

	/* if capacity not equal to calculated size, try various adjustments */
	if ((nspt * ntpc * ncpd) != spd) {
	    /* subtract alternate sectors if a zone == a track */
	    if (pd->pd_pg.pg_fmt.fmt_tpz == 1)
		nspt -= pd->pd_pg.pg_fmt.fmt_alt_spz;
	    /* subtract alternate sectors & tracks if a zone == a cylinder */
	    if (pd->pd_pg.pg_fmt.fmt_tpz == ntpc) {
		nspt -= pd->pd_pg.pg_fmt.fmt_alt_spz / pd->pd_pg.pg_fmt.fmt_tpz;
		ntpc -= pd->pd_pg.pg_fmt.fmt_alt_tpz;
	    }
	    /* subtract alternate cylinders if a zone == a disk (volume) */
	    if (pd->pd_pg.pg_fmt.fmt_tpz == 0)
		ncpd -= (pd->pd_pg.pg_fmt.fmt_alt_tpz + ntpc - 1) / ntpc;
	    else
	        /* subtract alternate cylinders if alternate tracks/volume */
	        ncpd -= (pd->pd_pg.pg_fmt.fmt_alt_tpv + ntpc - 1) / ntpc;
	}
	/* if still unequal, drop/add cylinders at end */
	if ((nspt * ntpc * ncpd) != spd)
	    ncpd = spd / (nspt * ntpc);
#define TOSHIBA_HACK
	/* hard code old style geometry calculation for Toshiba 156 only. */
#ifdef TOSHIBA_HACK
	if (strncmp(gd_inq.inq_vendor, "TOSHIBA MK156FB", 15) == 0) {
		nspt = 35; ntpc = 10; ncpd = 825;
	}
#endif TOSHIBA_HACK

	/* form drive name from first 3 characters of vendor and size */

#ifdef  JUPITER_MODS
	/*
	 * This hack is needed so that we can mount a Jupiter disk using
	 * its partition sizes and offsets.  I'm hard coding it here because
	 * it only needed to get a disk made for Sumitomo(sp?)
	 */
	st->st_size[0].nblocks = 29832;
	st->st_size[0].cyloff = 4;
	st->st_size[6].nblocks = 144144;
	st->st_size[6].cyloff = 230;
#endif
	printf("        %mM %3s%3m (%d x %d x %d)", nspt * ntpc * ncpd * 512,
	    gd_inq.inq_vendor, nspt * ntpc * ncpd * 512, nspt, ntpc, ncpd);
	diskpart(st->st_size, nspt, ntpc, ncpd);
	st->st_ncpd = ncpd;
	st->st_ntpc = ntpc;
	st->st_nspt = nspt;
	st->st_nspc = nspt * ntpc;
	if (qi->qi_dk >= 0)
		dk_bps[qi->qi_dk] = (3600/60) * 512 * nspt;
	return (0);
}

gdslctmod(qi)
	struct qb_device  *qi;
{
	int				unit = qi->qi_unit;
	int				ctlr = GDU_CTLR(unit);
	register struct gdstat		*stat = &gdstat[ctlr];
	register struct hacb_dcb 	*dcb = &stat->gd_dirc;
	register int			lun = GDU_LUN(unit);
	register struct pag_desc	*pdn, *pdl;
	int				len;

	/* get the current error recovery parameter page in msen */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, PD_ERROR<<8, sizeof(gd_msen),
	    0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msen_vdma,
	    sizeof(gd_msen), 1);
	flush_cache_entry(&gd_msen, sizeof (gd_msen));
	if (gdissue("MSENSE ERR", dcb, unit))
		return (-1);
	pdn = (struct pag_desc *)(((char *)&gd_msen.msen_bd)+gd_msen.msen_bdl);
	if (pdn->pd_pg.pg_err.err_per)
		return (0);
	len = gd_msen.msen_len + 1;
	gd_msen.msen_len = 0;
	gd_msen.msen_wprot = 0;
	gd_msel = *((struct scsi_msel *)&gd_msen);

	/* get changable status of error recovery parameters page in msen */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, (0x40 | PD_ERROR)<<8,
	    sizeof(gd_msen), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msen_vdma,
	    sizeof(gd_msen), 1);
	flush_cache_entry(&gd_msen, sizeof (gd_msen));
	if (gdissue("MSENSE CERR", dcb, unit))
		return (-1);

	pdl = (struct pag_desc *)(((char *)&gd_msel.msel_bd)+gd_msel.msel_bdl);
	if (pdn->pd_pg.pg_err.err_per)
		pdl->pd_pg.pg_err.err_per = 1;

	/* send page to drive */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSELECT, lun, pdl->pd_save<<16, len, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_OUT, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msel_vdma,
	    len, 1);
	pdl->pd_save = 0;
	if (gdissue("MSELECT ERR", dcb, unit, 0))
		return (-1);
	return (0);
}

gdintr(cont)
{
	return (gsintr(cont));
}
#endif
