#ifdef	SYSV
/* @(#)name.c	6.1 */

#include "reg.h"
#include "pte.h"
#include "psl.h"

#include "param.h"
#include "types.h"
#include "systm.h"
#include "vmparam.h"
#include "user.h"
#include "proc.h"
#include "buf.h"
#include "dir.h"
#include "kernel.h"
#include "vnode.h"
#include "../ufs/inode.h"
#include "quota.h"

#include "frame.h"
#include "m68k.h"
#include "trap.h"
#include "board.h"
#include "../sysv/sys/utsname.h"

#define SYS "unix"
#define NODE ""
#define REL "5"
#define VER "3"
#define MACH "is68k"

struct utsname utsname = {
	SYS,
	NODE,
	REL,
	VER,
	MACH,
};

/*
 * Simulate vax sigcontext in <signal.h>, but with 680[12]0 stack frame format. 
 * Note that pc and psl are exchanged (they are in vax order).
 */
struct sigctx {
	int		sc_onstack;		/* sigstack state to restore */
	int		sc_mask;		/* signal mask to restore */
	int		sc_sp;			/* user sp to restore */
	union xsf	sc_xsf;			/* extended exception frame */
};						/* with pc and psl exchanged */
#define	sc_pc	sc_xsf.xf0.xsf_psl		/* vax pc position */
#define	sc_psl	sc_xsf.xf0.xsf_pc		/* vax psl position */

/*
 * Args to sigcode, plus eventual arg to sigcleanup, pushed on interrupt stack.
 * sigcode copies the first 3 items to make args to user's handler.
 */
struct sv_sigframe {
	int		sf_sig;			/* signal # */
	int		sf_code;		/* extra hdware code */
	int		filemagic;		/* file magic in header */
	struct sigctx	*sf_sctxp;		/* ptr to sigcontext */
	int		(*sf_handler)();	/* user's handler */
	struct sigctx	*sf_copy_sctxp;		/* arg for sigcleanup */
};

/*
 * Sysv_sendsig :
 * Send an interrupt to a process: First sigcode (see sysv_sigcode.s) gets
 * pushed on the current user stack, for lack of any better place to put it.  
 * Then a max size sigctx structure (an extended sigcontext structure from 
 * <signal.h>) gets pushed on the user stack, to be used by sigcleanup (below)
 * to return to the user state before the interrupt occurred.
 * We push a sigframe structure, providing the arguments to the signal handler.
 * When we return from here to the user, we will return to the stacked sigcode.
 * After the user's signal handler returns to the sigcode, it will trap to 
 * sigcleanup (below) to finish the job. There is a unique trap #code for each 
 * 680[12]0 stack format to clean up.
 */
sysv_sendsig(handler, sig, mask)
int		(*handler)();			/* user's signal handler */
register int	sig;				/* signal number */
int		mask;				/* mask restore */
{
	register caddr_t	sigp;		/* where sigcode goes */
	struct sigctx		sctx;		/* sigctx is built here */
	register struct sigctx	*sctxp;		/* where sigctx goes */
	struct sv_sigframe	sframe;		/* sigframe is built here */
	register struct sv_sigframe *sframep;	/* where sigframe goes */
	register int		*regs;		/* ptr to user's regs */
	u_short			*vec;		/* stack format vector ptr */
	register int		xsfsize;	/* size of xsf stack frame */
	extern caddr_t		sysv_sigcode;	/* sigcode, in locore.s */
	extern int		sysv_szsigcode;	/* sizeof sigcode */

	regs = u.u_ar0;
	if (regs[SP] >= USRSTACK)	/* out of range */
		goto bad;
	sigp = (char *)(regs[SP] - sysv_szsigcode);/* claim space for sigcode */
	sctxp = (struct sigctx *)sigp - 1; /* claim space for sigctx */

	/*	There is no signal stack */
	sframep = (struct sv_sigframe *)sctxp - 1;

	/*
	 * Must build signal handler context on stack to be returned to so that
	 * sigcleanup (trapped to from sigcode) will pop ps and pc off correct 
	 * stack.  The remainder of the signal state used in calling the handler
	 * must be placed on the stack on which the handler is to operate so 
	 * that the procedure entry code in sigcode will save the registers and
	 * such correctly.
	 */
	if ((u_int)sframep <= USRSTACK - ctob(u.u_ssize)) 
		(void) grow((u_int)sframep);
	if (!useracc((caddr_t)sframep, sizeof (struct sv_sigframe), B_WRITE))
		goto bad;
	if ((u_int)sctxp <= USRSTACK - ctob(u.u_ssize))
		(void) grow((u_int)sctxp);
	if (!useracc((caddr_t)sctxp, sizeof (struct sigctx), B_WRITE))
		goto bad;
	if (copyout(sysv_sigcode, sigp, (u_int)sysv_szsigcode))
		goto bad;
	sframe.sf_sig = sig;
	if (sig == SIGILL || sig == SIGFPE) {
		sframe.sf_code = u.u_code;
		u.u_code = 0;
	} else
		sframe.sf_code = 0;
	sframe.sf_sctxp = sctxp;
	sframe.sf_handler = handler;
	sframe.filemagic = u.u_filemagic;

	/*
	 * Duplicate the pointer to the sigctx structure. This one doesn't get 
	 * popped by the rtd, and is used by sigcleanup to reset the signal 
	 * state on inward return.
	 */
	sframe.sf_copy_sctxp = sctxp;

	/* put sigframe structure on user's interrupt stack */
	if (copyout((caddr_t)&sframe, (char *)sframep, (u_int)sizeof(sframe)))
		goto bad;

	/* 
	 * Compute extended exception stack frame size and its sigcode entry
	 */
	sctx.sc_onstack = 0;	/* sigctx goes on previous stack */
	sctx.sc_mask = mask;
	sctx.sc_sp = regs[SP];
	vec = (u_short *)(&((struct esf0 *)(&regs[PC + 1]))->esf_vec);
	switch (*vec & (ESF_FMTMSK & ~ESF_SOFT)) {
	  case ESF_FMT0:		/* sigcode returns via trap #8 */
		xsfsize = XSF_ISIZE0; sigp += 0; break;
	  case ESF_FMT2:		/* sigcode returns via trap #6 */
		xsfsize = XSF_ISIZE2; sigp += 2; break;
	  case ESF_FMT8:		/* sigcode returns via trap #5 */
		xsfsize = XSF_ISIZE8; sigp += 4; break;
	  case ESF_FMT9:		/* sigcode returns via trap #4 */
		xsfsize = XSF_ISIZE9; sigp += 6; break;
	  case ESF_FMTA:		/* sigcode returns via trap #3 */
		xsfsize = XSF_ISIZEA; sigp += 8; break;
	  case ESF_FMTB:		/* sigcode returns via trap #2 */
		xsfsize = XSF_ISIZEB; sigp += 10; break;
	  default:
		panic("Sendsig: bad stack format");
	}

	/* copy appropriate amount of kernel exception stack to local buffer */
	bcopy((caddr_t)&regs[PS], (caddr_t)&sctx.sc_pc, xsfsize);
	sctx.sc_psl = regs[PS];		/* make VAX order pc/psl */
	sctx.sc_pc = regs[PC];		/* make VAX order pc/psl */

	/* copy local sigctx to previous user stack */
	if (copyout((char *)&sctx, (char *)sctxp, (u_int)sizeof(sctx)))
		goto bad;

	/* 
	 * If we have a long frame, mark it so locore will collapse it to a 
	 * short 4-word frame to enter user sigcode.
	 */
	if (xsfsize != XSF_ISIZE0)	/* tell locore to collapse stack */
		*vec |= ESF_SOFT;	/* use illegal stack format bit */
	regs[SP] = (int)sframep;
	regs[PC] = (int)sigp;
	return;

bad:	sendill();
}
#endif	SYSV
