/*
 * 680{12}0 exception stack frame layouts and extended layouts.
 *
 * The "esf" structures describe the 68020 hardware stack frames.
 * The "xsf" structures extend the short status register to a full
 * longword to match the kernel stack (the pc and psl must look like
 * elements in an array of ints).
 */
#ifndef	LOCORE

/* 68010, and 68020
 * format 0:	Normal 4-word stack frame - interrupts, format errors,
 *		trap #n, illegal instructions, A-line and F-line emulator traps,
 *		privilege violations, coprocessor pre-instruction exceptions.
 */
struct esf0 {	
	u_short	esf_sw;		/* status register */
	u_int	esf_pc;		/* program counter */
	u_short	esf_vec;	/* stack format and vector offset */
};
struct xsf0 {	
	u_int	xsf_psl;	/* program status longword */
	u_int	xsf_pc;		/* program counter */
	struct	esf0 xsf_esf;
};

/* 68020
 * format 1:	Throwaway 4-word stack frame - interrupt while in master state.
 */
struct esf1 {
  	u_short	esf_sw;		/* status register */
	u_int	esf_pc;		/* program counter */
	u_short	esf_vec;	/* stack format and vector offset */
};
struct xsf1 {	
	u_int	xsf_psl;	/* program status longword */
	u_int	xsf_pc;		/* program counter */
	struct	esf1 xsf_esf;
};

/* 68020
 * format 2:	Normal 6-word stack frame - chk, chk2, cptrapcc, trapv trace,
 *		zero divide, and coprocessor post-instruction exceptions.
 */
struct esf2 {	
	u_short	esf_sw;		/* status register */
	u_int	esf_pc;		/* program counter */
	u_short	esf_vec;	/* stack format and vector offset */
	u_int	esf_iaddr;	/* addr of instr causing trap */
};
struct xsf2 {	
	u_int	xsf_psl;	/* program status longword */
	u_int	xsf_pc;		/* program counter */
	struct	esf2 xsf_esf;
};

/* 68010
 * format 8:	Bus Error on 68010.
 */
struct esf8 {	
	u_short	esf_sw;		/* status register */
	u_int	esf_pc;		/* program counter */
	u_short	esf_vec;	/* stack format and vector offset */
	u_short	esf_ssw_10;	/* special status word */
	u_int	esf_faddr_10;	/* data cycle fault addr */
	u_short	esf_rsv0;	/* reserved by Motorola */
	u_short	esf_dob;	/* data output buffer */
	u_short	esf_rsv1;	/* reserved by Motorola */
	u_short	esf_dib;	/* data input buffer */
	u_short	esf_rsv2;	/* reserved by Motorola */
	u_short	esf_ii[16];	/* internal information */
};
struct xsf8 {	
	u_int	xsf_psl;	/* program status longword */
	u_int	xsf_pc;		/* program counter */
	struct	esf8 xsf_esf;
};

/* 68020
 * format 9:	Coprocessor mid-instruction 10-word stack frame.
 */
struct esf9 {	
	u_short	esf_sw;		/* status register */
	u_int	esf_pc;		/* program counter */
	u_short	esf_vec;	/* stack format and vector offset */
	u_int	esf_iaddr;	/* addr of coproc instr */
	u_int	esf_ireg[2];	/* internal regs */
};
struct xsf9 {	
	u_int	xsf_psl;	/* program status longword */
	u_int	xsf_pc;		/* program counter */
	struct	esf9 xsf_esf;
};

/* 68020
 * format A:	Short bus cycle fault stack frame - address error or bus error
 *		on instruction boundary.
 */
struct esfa {	
	u_short	esf_sw;		/* status register */
	u_int	esf_pc;		/* program counter */
	u_short	esf_vec;	/* stack format and vector offset */
	u_short	esf_reg1;	/* internal reg */
	u_short	esf_ssw_20;	/* special status word */
	u_short	esf_ipipec;	/* instr pipe stage C */
	u_short	esf_ipipeb;	/* instr pipe stage B */
	u_int	esf_faddr_20;	/* data cycle fault addr */
	u_int	esf_ireg2;	/* internal reg */
	u_int	esf_idob;	/* data output buffer */
	u_int	esf_ireg3;	/* internal reg */
};
struct xsfa {	
	u_int	xsf_psl;	/* program status longword */
	u_int	xsf_pc;		/* program counter */
	struct	esfa xsf_esf;
};

/* 68020
 * format B:	Long bus cycle fault stack frame - address error or bus error
 *		not on instruction boundary.
 */
struct esfb {
	u_short	esf_sw;		/* status register */
	u_int	esf_pc;		/* program counter */
	u_short	esf_vec;	/* stack format and vector offset */
	u_short	esf_sreg1;	/* internal reg */
	u_short	esf_ssw_20;	/* special status word */
	u_short	esf_ipipec;	/* instr pipe stage C */
	u_short	esf_ipipeb;	/* instr pipe stage B */
	u_int	esf_faddr_20;	/* data cycle fault addr */
	u_int	esf_ireg2;	/* internal reg */
	u_int	esf_idob;	/* data output buffer */
	u_int	esf_aireg3[2];	/* internal reg */
	u_int	esf_ibaddr;	/* stage B addr */
	u_int	esf_ireg4;	/* internal reg */
	u_int	esf_idib;	/* data input buffer */
	u_int	esf_aireg5[11];	/* internal regs */
};
struct xsfb {
	u_int	xsf_psl;	/* program status longword */
	u_int	xsf_pc;		/* program counter */
	struct	esfb xsf_esf;
};

/*
 * Union of all stack frame layouts
 */
union esf {
	struct esf0	f0;
	struct esf1	f1;
	struct esf2	f2;
	struct esf8	f8;
	struct esf9	f9;
	struct esfa	fa;
	struct esfb	fb;
};
union xsf {
	struct xsf0	xf0;
	struct xsf1	xf1;
	struct xsf2	xf2;
	struct xsf8	xf8;
	struct xsf9	xf9;
	struct xsfa	xfa;
	struct xsfb	xfb;
};
#endif	!LOCORE

/*
 * Stack frame formats in vector offset word.
 */
#define	ESF_VECMSK	0x0fff		/* mask for sVec vector offset */
#define	ESF_VECSHIFT	2		/* shift for sVec vector */
#define	ESF_FMTMSK	0xf000		/* mask for sVec formats */
#define	ESF_FMTSHIFT	12		/* shift for sVec formats */
#define	ESF_FMT0	0x0000		/* stack format 0 68010, 68020*/
#define	ESF_FMT1	0x1000		/* stack format 1 68020*/
#define	ESF_FMT2	0x2000		/* stack format 2 68020*/
#define	ESF_FMT8	0x8000		/* stack format 8 68010*/
#define	ESF_FMT9	0x9000		/* stack format 9 68020*/
#define	ESF_FMTA	0xa000		/* stack format A 68020*/
#define	ESF_FMTB	0xb000		/* stack format B 68020*/
#define	ESF_SOFT	0x4000		/* Software collapse frame bit */
#define	ESF_SOFT_BIT	6		/* bit in upper byte */

/*
 * Stack frame sizes in bytes.
 */
#define	ESF_ISIZE0	8
#define	ESF_ISIZE1	8
#define	ESF_ISIZE2	12
#define	ESF_ISIZE8	58
#define	ESF_ISIZE9	20
#define	ESF_ISIZEA	32
#define	ESF_ISIZEB	92

#define	XSF_ISIZE0	(ESF_ISIZE0 + 8)
#define	XSF_ISIZE1	(ESF_ISIZE1 + 8)
#define	XSF_ISIZE2	(ESF_ISIZE2 + 8)
#define	XSF_ISIZE8	(ESF_ISIZE8 + 8)
#define	XSF_ISIZE9	(ESF_ISIZE9 + 8)
#define	XSF_ISIZEA	(ESF_ISIZEA + 8)
#define	XSF_ISIZEB	(ESF_ISIZEB + 8)

/*
 * Special status word on 68020
 */
#define	ESF_SSW_20_AFC	0x0007		/* data cycle address function code */
#define	ESF_SSW_20_SIZ	0x0030		/* data cycle size code */
#define	ESF_SSW_20_RW	0x0040		/* data cycle read/write (read==1) */
#define	ESF_SSW_20_RM	0x0080		/* data cycle read/modify/write  */
#define	ESF_SSW_20_DF	0x0100		/* data cycle fault/rerun flag  */
#define	ESF_SSW_20_RB	0x1000		/* instr pipe stage B rerun flag */
#define	ESF_SSW_20_RC	0x2000		/* instr pipe stage C rerun flag */
#define	ESF_SSW_20_FB	0x4000		/* instr pipe stage B fault flag */
#define	ESF_SSW_20_FC	0x8000		/* instr pipe stage C fault flag */
#define	ESF_SSW_20_BITS	\
"\20\20IPCF\17IPBF\16IPCR\15IPBR\11DF\10RMW\7READ\6SIZ1\5SIZ0\3FC2\2FC1\1FC0"
#define ESF_SSW_20_SOFTMSK \
	~(ESF_SSW_20_DF|ESF_SSW_20_RB|ESF_SSW_20_RC) /* dont rerun */

/*
 * Special status word on 68010
 */
#define	ESF_SSW_10_AFC	0x0007		/* data cycle address function code */
#define	ESF_SSW_10_RW	0x0100		/* data cycle read/write (read==1) */
#define	ESF_SSW_10_BY	0x0200		/* byte transfer flag */
#define	ESF_SSW_10_HB	0x0400		/* high byte */
#define	ESF_SSW_10_RM	0x0800		/* read/modify/write cycle */
#define	ESF_SSW_10_DF	0x1000		/* data fetch flag */
#define	ESF_SSW_10_RR	0x8000		/* Re-run flag, 1-> software rerun */
#define	ESF_SSW_10_BITS	\
"\20\20RERUN\16IFETCH\16DFETCH\14RMW\13HBYTE\12BYTE\11READ\3FC2\2FC1\1FC0"

/*
 * 68881 exception vectors
 */

#define	    FPE_TRAPCC_VEC	0x07	/* FTRAPcc instruction */
#define	    FPE_FLINE_VEC	0x0b	/* F-Line emulation trap */
#define	    FPE_COPROC_VEC	0x0d	/* coprocessor protocol violation */
#define	    FPE_FBSUN_VEC	0x30	/* branch/set on unordered */
#define	    FPE_INEXCT_VEC	0x31	/* inexact result */
#define	    FPE_FLTDIV_VEC	0x32	/* divide by zero floating fault */
#define	    FPE_FLTUND_VEC	0x33	/* floating underflow */
#define	    FPE_FOPERR_VEC	0x34	/* operand error */
#define	    FPE_FLTOVF_VEC	0x35	/* floating overflow fault */
#define	    FPE_FLTNAN_VEC	0x36	/* divide by zero floating fault */

#if defined(M68030) || defined(VQX)

#ifndef	LOCORE
/*
 * This file defines the data structures for the mmu.
 * One major data structure, the short format early 
 * termination page descriptor, is not defined here
 * but rather in pte.h as struct pte.
 */

struct rpd {			/* root pointer descriptor */
u_int	rp_lu:1,		/* lower or upper page range */
	rp_limit:15,		/* limit on table index */
#ifndef VQX
	:14,			/* must be zero */
#else VQX
	:6,
	rp_sg:1,
	:7,
#endif VQX
	rp_dt:2,		/* descriptor type */
	rp_addr:32;		/* address of table at next level */
};

struct tdlong {			/* long table descriptor */
u_int	td_lu:1,		/* lower or upper page range */
	td_limit:15,		/* limit on table index */
#ifndef VQX
	td_con7e:7,		/* must be constant 0x7e */
#else VQX
	td_ral:3,
	td_wal:3,
	td_sg:1,
#endif VQX
	td_s:1,			/* supervisor access */
	:1,			/* must be zero */
	td_ci:1,		/* cache inhibit bit */
	:1,			/* must be zero */
	td_m:1,			/* descriptor modified bit */
	td_u:1,			/* descriptor reference bit */
	td_wp:1,		/* write protect */
	td_dt:2,		/* descriptor type */
	td_addr:32;		/* table address */
};


struct tcr {			/* translatiopn control register */
u_int	tc_e:1,			/* translation enable */
	:5,
	tc_sre:1,		/* supervisor root pointer enable */
	tc_fcl:1,		/* function code lookup enable */
	tc_ps:4,		/* page size */
	tc_is:4,		/* initial shift */
	tc_tia:4,		/* level a index */
	tc_tib:4,		/* level b index */
	tc_tic:4,		/* level c index */
	tc_tid:4;		/* level d index */
};


struct ttr {			/* transparent translation register */
u_int	tt_lab:8,		/* logical address base */
	tt_lam:8,		/* logical address mask */
	tt_e:1,			/* enable */
	:4,
	tt_ci:1,		/* cache inhibit */
	tt_rw:1,		/* read/write */
	tt_rwm:1,		/* read/write mask */
	:1,
	tt_fcbase:3,		/* function code value */
	:1,
	tt_fcmask:3;		/* function code mask */
};
#endif	LOCORE

#define	DTPAGE	1		/* page descriptor type */
#define DTSHORT	2		/* short format descriptor type */
#define DTLONG	3		/* long format descriptor type */
#define	PS4K	0x0c		/* 4K page size encoding */
#define	TIA	10		/* number of bits to index into level A */
#define	TIB	10		/* number of bits to index into level B */
				/* 2**TIB should be NPTEPG */
#define PPTASPACE	(1<<22)	/* address space of each level A slot */

#ifdef M68030
/* 68030 ON CHIP CACHE */
#define	ICACHE_ENABLE	0x01
#define	ICACHE_FREEZE	0x02
#define	ICACHE_CE	0x04
#define	ICACHE_CLEAR	0x08
#define	ICACHE_BURST	0x10
#define	DCACHE_ENABLE	0x0100
#define	DCACHE_FREEZE	0x0200
#define	DCACHE_CE	0x0400
#define	DCACHE_CLEAR	0x0800
#define	DCACHE_BURST	0x1000
#define	DCACHE_WA	0x2000

#endif M68030

#define	MMUCRP	0		/* cpu root pointer */
#define	MMUSRP	1		/* sup root pointer */
#define	MMUTC	2		/* translation control register */
#define MMUSR	3		/* mmu status register */
#ifdef M68030
#define MMUTT0	4		/* mmu transparent translation register 0 */
#define MMUTT1	5		/* mmu transparent translation register 1 */
#endif M68030

#define FC_ALT		3		/* alternate function code */
					/* for on board device addressing */
#endif	M68030 || VQX
