/*
 * Copyright (c) 1982 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)clock.c	6.3 (Berkeley) 6/8/85
 */

#include "param.h"
#include "time.h"
#include "kernel.h"

#include "clock.h"
#include "../is68k/board.h"
#ifdef  VQX
#include "../is68kdev/sioreg.h"
#include "../is68kdev/openchip.h"
#else   VQX
#include "../is68kdev/ctcreg.h"
#endif  VQX

#ifdef  VQX
        /* used to read and write the clock and nvram   */
#define RAM_SEL_DELAY   10
#define RAM_LOW_DELAY   10
#define RAM_HIGH_DELAY  10

#define CLK_DELAY       20      /* should be 14 microsecs */

#define SETSTROBE       (Q_OP_PORT2 = OP2_SET_STROBE)
#define RESETSTROBE     (Q_OP_PORT2 = OP2_RES_STROBE)
#define SETDATAOUT      (Q_OP_PORT2 = OP2_SET_DATOUT)
#define RESETDATAOUT    (Q_OP_PORT2 = OP2_RES_DATOUT)
#endif  VQX


/*
 * Machine-dependent clock routines.
 *
 * Startrtclock restarts the real-time clock, which provides
 * hardclock interrupts to kern_clock.c.
 *
 * Inittodr initializes the time of day hardware which provides
 * date functions.  Its primary function is to use some file
 * system information in case the hardware clock lost state.
 *
 * Resettodr restores the time of day hardware after a time change.
 */
unsigned short *battery_clock = 0;	/* address of battery clock */
short	have_cpu_clock = 0;		/* on-board day/date clock on 20cpus */

/*
 * Start the real-time clock.
 */
#ifdef VQX
static int clock_idle = 1;      /* clock not started yet        */
startrtclock()
{
        if ( clock_idle ) {     /* only needs a kick once       */
                clkinit();
                clock_idle = 0;
        }
}
#else   VQX
startrtclock()
{
#ifdef  M68030
	sio_rt_start();
#else   M68030
	ctx_bits |= CTX_CLOCK;		/* turn on 60 Hz clock */
	*CTX = ctx | ctx_bits;
#endif	M68030
	prcpuspeed();
	memenable();
}
#endif VQX

/*
 * Initialize the time of day register, based on the time base which is, e.g.
 * from a filesystem. Called after configure.
 *
 * called from init_main with base = -1 for diskless/cluster node
 * 
 */
inittodr(base)
	time_t base;
{
	time.tv_sec = gettod(base);

	if (time.tv_sec < 5*SECYR) {
		printf("***  WARNING:  preposterous time in file system  ***\n");
		time.tv_sec = 6*SECYR + 186*SECDAY + SECDAY/2;
		resettodr();
		printf("***  WARNING:  CHECK AND RESET THE DATE  ***\n");
	}
}

gettod(base)
{
	short year, month, day, hour, mins, secs;
	register struct tm *tm;
	struct tm *gmtime();
	register int i;

#ifdef TRFS
	if( base == -1 ) {
		/* use ConnectTime.tv_sec for base instead of the -1 */
		extern struct timeval ConnectTime;
		base = (time_t) ConnectTime.tv_sec;
	}
#endif TRFS

	/* see if 68020/68225 on-board clock is there */
	have_cpu_clock = check_cpu_clock();

	if (!battery_clock && !have_cpu_clock)
		return(base);

	/* on-board cpu clock has precedence */
	if(have_cpu_clock) {
		battery_clock = (short *)1;   /* set it so /bin/date is happy */
#ifdef VQX
		read_cpu_clock(&year,&month,&day,&hour,&mins,&secs);
#else VQX
		read_cpu_clock(&month,&day,&hour,&mins,&secs);
#endif VQX
	} else { /* use clock on qic2 or tc50 */
		register int s = spl7();
		BCLOCK_GET();
		BCLOCK_READ( 7, month);
		BCLOCK_READ( 6, day);
		BCLOCK_READ( 4, hour);
		BCLOCK_READ( 3, mins);
		BCLOCK_READ( 2, secs);
		BCLOCK_RESET();
		splx(s);
	}
	tm = gmtime(&base);
	/* VQX clock has year -- we don't need to get it from fs */
#ifndef VQX
	year = tm->tm_year - 70;
	if (tm->tm_mon >= 6 && month <= 6)
		year += 1; /* merry XMAS, happy new year */
#endif VQX
	base = 0;
	for (i = 0; i < year; i++)
		base += dysize(70 + i);
	/* Leap year */
	if (dysize(70 + year) == 366 && month >= 3)
		base++;
	while (--month)
		base += dmsize[month-1];
	base += day-1;
	base = 24*base + hour;
	base = 60*base + mins;
	base = 60*base + secs;
	return(base);
}


/*
 * Reset the TODR based on the time value; used when the TODR has a preposterous
 * value and also when the time is reset by the stime system call.
 */
resettodr()
{
	register struct tm *tm;
	struct tm *gmtime();
	register int s;
	
	tm = gmtime(&time.tv_sec);
	tm->tm_mon += 1;

	if (!(battery_clock || have_cpu_clock))
		return;
	if(have_cpu_clock)
		write_cpu_clock( tm );
	else {
		s = spl7();
		BCLOCK_WRITE( 2, tm->tm_sec);
		BCLOCK_WRITE( 3, tm->tm_min);
		BCLOCK_WRITE( 4, tm->tm_hour);
		BCLOCK_WRITE( 6, tm->tm_mday);
		BCLOCK_WRITE( 7, tm->tm_mon);
		BCLOCK_WRITE( 8, tm->tm_year);
		BCLOCK_WRITE( 9, 1);	/* set 24 hour mode on 1/2" tape */
		BCLOCK_SET();
		BCLOCK_RESET();
		splx(s);
	}
}

struct tm *
gmtime(tim)
unsigned long *tim;
{
	register int d0, d1;
	long hms, day;
	register int *tp;
	static struct tm xtime;

	/* break initial number into days */
	hms = *tim % 86400;
	day = *tim / 86400;
	if (hms<0) {
		hms += 86400;
		day -= 1;
	}
	tp = (int *)&xtime;

	/* generate hours:minutes:seconds */
	*tp++ = hms%60;
	d1 = hms/60;
	*tp++ = d1%60;
	d1 /= 60;
	*tp++ = d1;

	/*
	 * day is the day number. generate day of the week.
	 * The addend is 4 mod 7 (1/1/1970 was Thursday)
	 */

	xtime.tm_wday = (day+7340036)%7;

	/* year number */
	if (day>=0) for(d1=70; day >= dysize(d1); d1++)
		day -= dysize(d1);
	else for (d1=70; day<0; d1--)
		day += dysize(d1-1);
	xtime.tm_year = d1;
	xtime.tm_yday = d0 = day;

	/* generate month */
	if (dysize(d1)==366)
		dmsize[1] = 29;
	for(d1=0; d0 >= dmsize[d1]; d1++)
		d0 -= dmsize[d1];
	dmsize[1] = 28;
	*tp++ = d0+1;
	*tp++ = d1;
	xtime.tm_isdst = 0;
	return(&xtime);
}

dysize(y)
{
	if((y%4) == 0)
		return(366);
	return(365);
}


#ifdef  M68030
binary_to_bcd(i)
{
	return((i / 10) << 4 | (i % 10));
}

bcd_to_binary(i)
unsigned char i;
{
	return((i >> 4) * 10 + (i & 0x0f));
}

#else  M68030

/* on-board day/date battery backup clock stuff */
/* static char clk_pat[8] = {0xc5, 0x3a, 0xa3, 0x5c, 0xc5, 0x3a, 0xa3, 0x5c};*/
/* above is the documented byte patterns - note inverted data pattern */

u_char mtch_pat[CLKBITS] = {1,0,1,0,0,0,1,1,
			   0,1,0,1,1,1,0,0,
			   1,1,0,0,0,1,0,1,
			   0,0,1,1,1,0,1,0,
			   1,0,1,0,0,0,1,1,
			   0,1,0,1,1,1,0,0,
			   1,1,0,0,0,1,0,1,
			   0,0,1,1,1,0,1,0};


/*
  organization of clocks registers (all #s in BCD format)

  clk_data[0:0-3] = hundredths of seconds
  clk_data[0:4-7] = tenths of seconds
  clk_data[1:0-3] =  seconds
  clk_data[1:4-7] = tens of seconds
  clk_data[2:0-3] =  minutes
  clk_data[2:4-7] = tens of minutes
  clk_data[3:0-3] = hours 
  clk_data[3:4-5] = tens of hours(or am/pm flag + tens of hours)
  clk_data[3:7] = 12/24 hr mode (0 = 24hr mode )
  clk_data[4:5] = NOT oscillator
  clk_data[4:4] = NOT reset !!!! NEVER SET THIS TO ZERO !!!
  clk_data[5:4-5] = tens of days (leftmost digit of day of month)
  clk_data[5:0-3] = days
  clk_data[6:4] = tens of months 
  clk_data[6:0-3] = months 
  clk_data[7:4-7] = tens of years
  clk_data[7:0-3] = years 	
*/

int clk_data[CLKBITS];

Init_Clock()
{
	register int i;
	register char *clock;
	char tmp;

	tmp = *CLKRD; /* force chip to read next 64 bits */
	for(i=0;i<CLKBITS;i++) {
		clock = (char *)(CLKWT + mtch_pat[i]);
		tmp = *clock;
	}
}

ClkSleep()
{
	int i;
	for(i=0; i<20000; i++);
	return(i);
}

Read_Clock()
{
	register int i,j;

	Init_Clock();
	ClkSleep();
	for(i=0; i<(CLKBITS/8); i++) {
		clk_data[i] = 0;
		for(j=0; j<(CLKBITS/8); j++) {
			clk_data[i] += (int)((*CLKRD & 0x01) << j);
		}
	}

	ClkSleep();


}

Write_Clock()
{
	register int i,j;
	register char *clock;
	register char tmp;

	Init_Clock(); /* prepare for writing */
	ClkSleep();
	for(i=0;i<(CLKBITS/8);i++) {
		for(j=0; j<(CLKBITS/8); j++) {
			clock = (char *)(CLKWT + ((clk_data[i] >> j) & 0x1));
			tmp = *clock;
		}
	}
	ClkSleep();
}
#endif  M68030

check_cpu_clock()
{
	short have_it = 0;
	register int i,j;
	register char *clock;
	int clk_data2[(CLKBITS/8)];
	char c;
	
#if	defined(M68030) || defined(VQX)
	return(1);
#else	M68030 || VQX
	/*see if on-board clock is present */
	/* force a reset */
	c = *CLKRD;
	Read_Clock();
	if((clk_data[4] & 0x20)){ /* oscillator is not on */
		for(i=0; i<(CLKBITS/8); i++)
			clk_data[i] = 0;
		clk_data[6] = 7; /* jul */
		clk_data[5] = 1; /* jul  1*/
		clk_data[7] = 0x18; /* 1988 */
		clk_data[4] |= 0x10; /* reset off */
		/*clk_data[3] bit 7 = 0 -> 24 hr mode on */
		/*clk_data[4] bit 5 = 0 -> osc on*/
		Write_Clock();
	}
	/* read clock twice in a row - if something changes then we
	   have a clock chip there - if not, no clock */
	Read_Clock();
	Init_Clock();
	ClkSleep();
	for(i=0; i<(CLKBITS/8); i++){
		clk_data2[i] = 0;
		for(j=0; j<(CLKBITS/8); j++)
			clk_data2[i] += (int )((*CLKRD & 0x01) << j);
	}
	ClkSleep();
	for(i=0; i<(CLKBITS/8); i++) {
		if(clk_data[i] != clk_data2[i]) {
			have_it = 1;
			break;
		}
	}
	if(have_it)
		printf("Local Battery Backup Clock\n");

	return( have_it );
#endif	M68030 || VQX
}

#ifdef	VQX
read_cpu_clock(yr,mo,dy,hr,mn,sc) /*year, month, day, hour, min, sec*/
#else VQX
read_cpu_clock(mo,dy,hr,mn,sc) /*month, day, hour, min, sec*/
#endif VQX
#ifdef VQX
short *yr;
#endif VQX
short *mo;
short *dy;
short *hr;
short *mn;
short *sc;
{
	/*read clock and load the appropriate data into the provided address*/

#ifdef	M68020
#ifdef	VQX
	short  day_of_week;
	/*
	 * perform a read of register 7. 7 bytes are read back.
	 * From most to least significant they
	 * are : hour, minute, date, month, year, day of week, second.
	 * Each byte contains 2 BCD digits with the least significant
	 * bit being clocked out first and seconds first, hours last.
	 *
	 * The clock chip has been installed such that the first read
	 * bit arrives on the negative edge of the 5 clock pulse not
	 * the fourth. Therefore transmit 5 bits with the extra bit a
	 * 1.
	 */
	shftout(5, 0x1f, CLK_DELAY);
	*hr = srvindgt();
	*mn = srvindgt();
	*dy = srvindgt();
	*mo= srvindgt();
	*yr = srvindgt();
	day_of_week = srvindgt();       /* ignored      */
	*sc = srvindgt();
	srv_unselect();
#else	VQX
	Read_Clock(); /* fill clk_data with bits */
	*mo = (short)((clk_data[6] & 0xF) + (((clk_data[6] >> 4) & 0x1) * 10));
	*dy = (short)((clk_data[5] & 0xF) + (((clk_data[5] >> 4) & 0x3) * 10));
	*sc = (short)((clk_data[1] & 0xF) + (((clk_data[1] >> 4) & 0x7) * 10));
	*mn = (short)((clk_data[2] & 0xF) + (((clk_data[2] >> 4) & 0x7) * 10));
	if( !(clk_data[3] & 0x80) ) { /* 24 hr mode */
   	    *hr = (short)((clk_data[3] & 0xF)+(((clk_data[3] >> 4) & 0x3)*10));
	} else { /* 12 hr mode */
	    *hr = (short)((clk_data[3] & 0xF)+(((clk_data[3] >> 4) & 0x1)*10));
	}
#endif	VQX
#else	M68020
	afcwb(NVRAM_CONTROL, NVRAM_READ);
	*mo = bcd_to_binary(afcrb(NVRAM_MONTH));
	*dy = bcd_to_binary(afcrb(NVRAM_DATE));
	*hr = bcd_to_binary(afcrb(NVRAM_HOUR));
	*mn = bcd_to_binary(afcrb(NVRAM_MINUTE));
	*sc = bcd_to_binary(afcrb(NVRAM_SECOND));
	afcwb(NVRAM_CONTROL, 0);
#endif	M68020
	return(0);
}

write_cpu_clock(tm)
struct tm *tm;
{
	register int i;

#ifdef	M68030
        afcwb(NVRAM_CONTROL, NVRAM_WRITE);
        afcwb(NVRAM_YEAR, binary_to_bcd(tm->tm_year));
        afcwb(NVRAM_MONTH, binary_to_bcd(tm->tm_mon));
        afcwb(NVRAM_DATE, binary_to_bcd(tm->tm_mday));
        afcwb(NVRAM_DAY, binary_to_bcd(tm->tm_wday));
        afcwb(NVRAM_HOUR, binary_to_bcd(tm->tm_hour));
        afcwb(NVRAM_MINUTE, binary_to_bcd(tm->tm_min));
        afcwb(NVRAM_SECOND, binary_to_bcd(tm->tm_sec));
        afcwb(NVRAM_CONTROL, 0);
#else	M68030
#ifdef	VQX
        /*
         * write to register 7. The most significant
         * bits get written first.
         */
        srv_clkselect();
        shftout(4, 14, CLK_DELAY);
        srvoutdgt(tm->tm_hour);
        srvoutdgt(tm->tm_min);
        srvoutdgt(tm->tm_mday);
        srvoutdgt(tm->tm_mon);
        srvoutdgt(tm->tm_year);
        srvoutdgt(0);           /* day_of_week  */
        srvoutdgt(tm->tm_sec);
        srv_unselect();
#else	VQX
	/* write the time of day clock from data in the tm structure */

	for(i=0;i<(CLKBITS/8);i++)
		clk_data[i] = 0;
	/* build seconds image */
	clk_data[1] = (tm->tm_sec%10) + ((tm->tm_sec/10)<<4);
	/*build minutes image */
	clk_data[2] = (tm->tm_min%10) + ((tm->tm_min/10)<<4);
	/*build hours image */
	clk_data[3] = (tm->tm_hour%10) + ((tm->tm_hour/10)<<4);
	/*build day of month image */
	clk_data[5] = (tm->tm_mday%10) + ((tm->tm_mday/10)<<4);
	/*build month image */
	clk_data[6] = (tm->tm_mon%10) + ((tm->tm_mon/10)<<4);
	/*build year image*/
	clk_data[7] = (tm->tm_year%10) + ((tm->tm_year/10)<<4);

	clk_data[4] &= 0xDF ; /* set oscillator on */
	clk_data[4] |= 0x10;  /* >>>>> ALWAYS set reset off<<<<<<< */
	clk_data[3] &= 0x7F ; /* set 24 hr mode */

	Write_Clock();
#endif	VQX
#endif	M68030
	return(0);
}

#ifdef	VQX
	/* VQX clock manipulation routines--perhaps belong elsewhere */

srvoutdgt(val)
unsigned char val; {
/*
 * take an 8 bit value in the range 0 to 99 and convert it into 2 BCD digits.
 */
	shftout(8, ((val / 10) << 4) + val % 10, CLK_DELAY);
}
srvindgt() {
/*
 * read in an 8 bit packed BCD value and convert it to binary. Notice that the
 * bits are read LSB first but this is handled by the shift routine.
 */
	unsigned char val;

	val = shftlsbin(8, CLK_DELAY);
	return(((val >> 4) & 0x0f) * 10 + (val & 0x0f));
}
srv_clkselect() {
	SETDATAOUT;
	Q_OP_PORT2 = OP2_RES_RAMSEL;	/* disable RAM */
	Q_OP_PORT2 = OP2_SET_TIMSEL;	/* enable clock */
}
srv_unselect() {
	Q_OP_PORT2 = OP2_RES_RAMSEL;	/* disable RAM */
	Q_OP_PORT2 = OP2_RES_TIMSEL;	/* disable clock */
}

shftout(nbits, shftreg, delay)
short nbits;
long shftreg;
int delay; {
/*
 * write a number of bits to the serial RAM or clock chip. Data is written most
 * signiicant bit first. Data is clocked on the positive going edge.
 */
	long mask;

	mask = 1L << (nbits - 1);
	for (;nbits > 0; nbits--, mask >>= 1) {
		if (shftreg & mask) {
			SETDATAOUT;
		} else {
			RESETDATAOUT;
		}
		srvdelay(delay);
		SETSTROBE;
		srvdelay(delay);
		RESETSTROBE;
	}
	SETDATAOUT;
}


shftin(nbits, delay)
short nbits;
int delay; {
/*
 * read a number of bits from the serial RAM or clock chip. Data is read most
 * significant bit first. Data is read on the negative going edge.
 */
	long shftreg;

	shftreg = 0;
	for (; --nbits >= 0;) {
		srvdelay(delay);
		shftreg <<= 1;
		if (DATIN) {
			shftreg++;
		}
		SETSTROBE;
		srvdelay(delay);
		RESETSTROBE;
	}
	return(shftreg);
}


shftlsbin(nbits, delay)
short nbits;
int delay; {
/*
 * read a number of bits from the serial RAM or clock chip.
 * Data is read on the negative going edge.
 * This routine differs from that above in that data is read least significant
 * bit first.
 */
	long shftreg;
	long mask;

	shftreg = 0;
	for (mask = 1; nbits > 0; nbits--, mask <<= 1) {
		srvdelay(delay);
		if (DATIN) {
			shftreg |= mask;
		}
		SETSTROBE;
		srvdelay(delay);
		RESETSTROBE;
	}
	return(shftreg);
}


/*
 * The non-volatile RAM chip and the clock chip are relatively slow devices
 * so delays are needed when accessing them.
 * There are three delay values. The first is the half clock cycle time of the
 * RAM chip. That is 500ns.
 * The second is the minimum deselect time of the RAM chip which is 800ns.
 * The other is the half clock cycle time of the clock chip which is
 * 14microseconds.
 */
srvdelay(delay)
register delay; {
	for (;--delay > 0;);
}

#endif	VQX
