/* Copyright (c) 1980 Regents of the University of California */
#include <stdio.h>
#include "as.h"
#include "asexpr.h"
#include "asscan.h"
#include "assyms.h"

int	lgensym[10];
char	genref[10];

long	bitfield;
int	bitoff;
int	curlen;			/* current length of literals */

/*
 *	The following three variables are communication between various
 *	modules to special case a number of things.  They are properly
 *	categorized as hacks.
 */
extern	struct	symtab *lastnam;/*last name seen by the lexical analyzer*/
int	exprisname;		/*last factor in an expression was a name*/
int	droppedLP;		/*one is analyzing an expression beginning with*/
				/*a left parenthesis, which has already been*/
				/*shifted. (Used to parse (<expr>)(rn)*/

char	yytext[NCPS+2];		/*the lexical image*/
int	yylval;			/*the lexical value; sloppy typing*/
/*
 *	Expression and argument managers
 */
struct	exp	*xp;		/*next free expression slot, used by expr.c*/
struct	exp	explist[NEXP];	/*max of 20 expressions in one opcode*/
struct	arg	arglist[NARG];	/*building up operands in instructions*/
/*
 *	Sets to accelerate token discrimination
 */
char	tokensets[(LASTTOKEN) - (FIRSTTOKEN) + 1];

static	char	UDotsname[32];	/*name of the assembly source*/

int	yyparse()
{
	register	struct	exp	*locxp;
			/*
			 *	loc1xp and ptrloc1xp are used in the
			 * 	expression lookahead
			 */
			struct	exp	*loc1xp;	/*must be non register*/
			struct	exp	**ptrloc1xp = & loc1xp;
			struct	exp	*pval;		/*hacking expr:expr*/

	register	struct	symtab	*np;
	register	int		argcnt;

	register	int		val;		/*what yylex gives*/
	register	int		auxval;		/*saves val*/

	register	struct 	arg	*ap;		/*first free argument*/

			struct	symtab	*p;
	register	struct	symtab	*stpt;

			struct	strdesc	*stringp;	/*handles string lists*/

			int		regno;		/*handles arguments*/
			int		*ptrregno = &regno;
			int		seg_type; 	/*the kind of segment: data or text*/
			int		seg_number;	/*the segment number*/
			int		space_value;	/*how much .space needs*/
			int		fill_rep;	/*how many reps for .fill */
			int		fill_size;	/*how many bytes for .fill */

			int		field_width;	/*how wide a field is to be*/
			int		field_value;	/*the value to stuff in a field*/
			char		*stabname;	/*name of stab dealing with*/
			ptrall		stabstart;	/*where the stab starts in the buffer*/
			int		reloc_how;	/* how to relocate expressions */

	val = yylex();

    while (val != PARSEEOF){	/* primary loop */

	xp = explist;
	ap = arglist;

	while (INTOKSET(val, LINSTBEGIN)){
		if (val == INT) {
			int i = ((struct exp *)yylval)->e_xvalue;
			shift;
			if (val != COLON)
				goto nocolon;
			if (i < 0 || i > 9) {
				yyerror("Local labels are 0-9");
				goto errorfix;
			}
			sprintf(yytext, "L%d\001%d", i, lgensym[i]);
			lgensym[i]++;
			genref[i] = 0;
			yylval = (int)*lookup(passno == 1);
			val = NAME;
			np = (struct symtab *)yylval;
			goto restlab;
		}
		if (val == NL){
			lineno++;
			shift;
		} else
		if (val == SEMI) 
			shift;
		else {	/*its a name, so we have a label or def */
			if (val != NAME){
				ERROR("Name expected for a label");
			}
			np = (struct symtab *)yylval;
			shiftover(NAME);
			if (val == ASSIGN) {
				shift;
				goto assign;
			}
nocolon:
			if (val != COLON) {
#ifdef FLEXNAMES
				yyerror("\"%s\" is not followed by a ':' for a label definition",
#else not FLEXNAMES
				yyerror("\"%.*s\" is not followed by a ':' for a label definition",
					NCPS,
#endif not FLEXNAMES
					np->s_name);
				goto  errorfix;
			}
restlab:
			shift;
			flushfield(NBPW/4);
			if ((np->s_type&XTYPE)!=XUNDEF &&
			    (np->s_type&XTYPE)!=XCOMM) {
				if((np->s_type&XTYPE)!=dotp->e_xtype ||
				   np->s_value!=dotp->e_xvalue ||
				   (passno==1 && np->s_index!=dotp->e_xloc)
				  ){
					{
						if (passno == 1)
#ifdef FLEXNAMES
						  yyerror("%s redefined",
#else not FLEXNAMES
						  yyerror("%.*s redefined",
							NCPS,
#endif not FLEXNAMES 
							np->s_name);
						else
#ifdef FLEXNAMES
						  yyerror("%s redefined: PHASE ERROR, 1st: %d, 2nd: %d",
#else not FLEXNAMES
						  yyerror("%.*s redefined: PHASE ERROR, 1st: %d, 2nd: %d",
							NCPS,
#endif not FLEXNAMES
							np->s_name,
							np->s_value,
							dotp->e_xvalue);
					}
				}
			}
			np->s_type &= (XXTRN|XFORW);
			np->s_type |= dotp->e_xtype;
			np->s_value = dotp->e_xvalue;
			if (passno == 1){
				np->s_index = dotp-usedot;
				if (np->s_name[0] == 'L' ||
				    (np->s_name[0] >= '0' &&
						np->s_name[0] <= '9')) {
					nlabels++;
				}
				np->s_tag = LABELID;
			}
		}	/*end of this being a label*/
	}	/*end of to consuming all labels, NLs and SEMIS */ 

	/*
	 *	process the INSTRUCTION body
	 */
	switch(val){

    default:
	ERROR("Unrecognized instruction or directive");

   case IABORT:
	shift;
	sawabort();
	/*NOTREACHED*/
	break;

   case PARSEEOF:
	tokptr -= sizeof(toktype);
	*tokptr++ = VOID;
	tokptr[1] = VOID;
	tokptr[2] = PARSEEOF;
	break;

   case IFILE:
	shift;
	stringp = (struct strdesc *)yylval;
	shiftover(STRING);
	dotsname = &UDotsname[0];
	movestr(dotsname, stringp->str,
		stringp->str_lg >= 32? 32 :stringp->str_lg);
	dotsname[stringp->str_lg] = '\0';
	break;

   case ILINENO:
	shift;		/*over the ILINENO*/
	expr(locxp, val);
	lineno = locxp->e_xvalue;
	break;

   case ISET: 	/* .set  <name> , <expr> */
	shift;
	np = (struct symtab *)yylval;
	shiftover(NAME);
	shiftover(CM);
    assign:
	expr(locxp, val);
	np->s_type &= (XXTRN|XFORW);
	np->s_type |= locxp->e_xtype&(XTYPE|XFORW);
	np->s_value = locxp->e_xvalue;
	if (passno==1)
		np->s_index = locxp->e_xloc;
	if ((locxp->e_xtype&XTYPE) == XUNDEF)
		yyerror("Illegal set?");
	break;

   case ILSYM: 	/*.lsym name , expr */
	shift;
	np = (struct symtab *)yylval;
	shiftover(NAME);
	shiftover(CM);
	expr(locxp, val);
	/*
	 *	Build the unique occurance of the
	 *	symbol.
	 *	The character scanner will have
	 *	already entered it into the symbol
	 *	table, but we should remove it
	 */
	if (passno == 1){
		stpt = (struct symtab *)symalloc();
#ifdef FLEXNAMES
		stpt->s_name = np->s_name;
#else
		movestr(stpt->s_name, np->s_name, NCPS);
#endif
		np->s_tag = OBSOLETE;	/*invalidate original */
		nforgotten++;
		np = stpt;
		if ((locxp->e_xtype & XTYPE) != XABS) 
			yyerror("Illegal second argument to lsym");
		np->s_value=locxp->e_xvalue;
		np->s_type=XABS;
		np->s_tag = ILSYM;
	}
	break;

   case IGLOBAL: 	/*.globl <name> */
	do {
		shift;
		np = (struct symtab *)yylval;
		shiftover(NAME);
		np->s_type |= XXTRN;
	}
	while (val == CM);
	break;

   case IDATA: 	/*.data [ <expr> ] */
   case ITEXT: 	/*.text [ <expr> ] */
	seg_type = -val;
	shift;
	if (INTOKSET(val, EBEGOPS+YUKKYEXPRBEG+SAFEEXPRBEG)){
		expr(locxp, val);
		seg_type = -seg_type;   /*now, it is positive*/
	}

	if (seg_type < 0) {	/*there wasn't an associated expr*/
		seg_number = 0;
		seg_type = -seg_type;
	} else {
		if (locxp->e_xtype != XABS || (seg_number=locxp->e_xvalue) >= NLOC) {
			yyerror("illegal location counter");
			seg_number = 0;
		}
	}
	if (seg_type == IDATA)
		seg_number += NLOC;
	flushfield(NBPW/4);
	dotp = &usedot[seg_number];
	if (passno==2) {	/* go salt away in pass 2*/
		txtfil = usefile[seg_number];
		relfil = rusefile[seg_number];
	}
	break;

	/*
	 *	Storage filler directives:
	 *
	 *	.byte	[<exprlist>]
	 *
	 *	exprlist:  empty | exprlist outexpr
	 *	outexpr:   <expr> | <expr> : <expr>
	 */
   case IBYTE:	curlen = NBPW/4; goto elist;

   case IINT:
   case ILONG:
	curlen = NBPW;
	if (dotp->e_xvalue & 1) {
		if (passno ==1)
			yywarning(".long (or .int) occurs at odd address");
		outb(0);
	}
	goto elist;

   case IWORD:
	curlen = NBPW/2;
	if (dotp->e_xvalue & 1) {
		if (passno ==1)
			yywarning(".word occurs at odd address");
		outb(0);
	}
   elist:
	seg_type = val;
	shift;

	/*
	 *	Expression List processing
	 */
	if (INTOKSET(val, EBEGOPS+YUKKYEXPRBEG+SAFEEXPRBEG)){
	    do{
		/*
		 *	expression list consists of a list of :
		 *	<expr>
		 *	<expr> : <expr> 
		 *		(pack expr2 into expr1 bits
		 */
		expr(locxp, val);
		/*
		 *	now, pointing at the next token
		 */
		if (val == COLON){
			shiftover(COLON);
			expr(pval, val);
			if (locxp->e_xtype != XABS)
			  yyerror("Width not absolute");
			field_width = locxp->e_xvalue;
			locxp = pval;
			if (bitoff + field_width >
			  curlen)
				flushfield(curlen);
			if (field_width > curlen)
				yyerror("Expression crosses field boundary");
		} else {
			field_width = curlen;
			flushfield(curlen);
		}

		 if ((locxp->e_xtype&XTYPE)!=XABS) {
			if (bitoff)
				yyerror("Illegal relocation in field");
			switch(curlen){
				case NBPW/4:	reloc_how = TYPB; break;
				case NBPW/2:	reloc_how = TYPW; break;
				case NBPW:	reloc_how = TYPL; break;
			}
			if (passno == 1){
				dotp->e_xvalue += ty_nbyte[reloc_how];
			} else {
				outrel(locxp, reloc_how);
			}
		} else {
			field_value = locxp->e_xvalue & ( (1L << field_width)-1);
			bitfield |= field_value << bitoff;
			bitoff += field_width;
		}
		if ( auxval = (val == CM)) shift;
		xp = explist;
	    } while (auxval);
	}	/*existed an expression  at all*/

	flushfield(curlen);
	if ( ( curlen == NBPW/4) && bitoff)
		dotp->e_xvalue ++;
	break;
	/*end of case IBYTE, IWORD, ILONG, IINT*/

   case ISPACE: 	/* .space <expr> */
	shift;
	expr(locxp, val);
	if (locxp->e_xtype != XABS)
		yyerror("Space size not absolute");
	space_value = locxp->e_xvalue;
  ospace:
	flushfield(NBPW/4);
	while (space_value > 96){
		outs(strbuf[2].str, 96);
		space_value -= 96;
	}
	outs(strbuf[2].str, space_value);
	break;

	/*
	 *	.fill rep, size, value
	 *	repeat rep times: fill size bytes with (truncated) value
	 *	size must be between 1 and 8
	 */
   case	IFILL:
	shift;
	expr(locxp, val);
	if (locxp->e_xtype != XABS)
		yyerror("Fill repetition count not absolute");
	fill_rep = locxp->e_xvalue;
	shiftover(CM);
	expr(locxp, val);
	if (locxp->e_xtype != XABS)
		yyerror("Fill size not absolute");
	fill_size = locxp->e_xvalue;
	if (fill_size <= 0 || fill_size > 8)
		yyerror("Fill count not in in 1..8");
	shiftover(CM);
	expr(locxp, val);
	if (passno == 2 && locxp->e_xtype != XABS)
			yyerror("Fill value not absolute");
	flushfield(NBPW/4);
	dotp->e_xvalue += fill_rep * fill_size;
	if (passno == 2) {
#ifdef VAX_CROSS
		char *cp = (char *)&locxp->e_xvalue;
#else VAX_CROSS
		char *cp = ((char *)&locxp->e_xvalue) +
			((fill_size==1)?3 : (fill_size==2)?2 : 0);
#endif VAX_CROSS
		while(fill_rep-- > 0)
			bwrite(cp, fill_size, txtfil);
	}
	break;

   case IASCII:	/* .ascii [ <stringlist> ] */
   case IASCIZ: 	/* .asciz [ <stringlist> ] */
	auxval = val;
	shift;

	/*
	 *	Code to consume a string list
	 *
	 *	stringlist: empty | STRING | stringlist STRING
	 */
	while (val ==  STRING){
		flushfield(NBPW/4);
		if (bitoff)
		  dotp->e_xvalue++;
		stringp = (struct strdesc *)yylval;
		outs(stringp->str, stringp->str_lg);
		shift;		/*over the STRING*/
		if (val == CM)	/*could be a split string*/
			shift;
	}

	if (auxval == IASCIZ){
		flushfield(NBPW/4);
		outb(0);
	}
	break;
	
   case IORG: 	/* .org <expr> */
	shift;
	expr(locxp, val);

	if (locxp->e_xtype==XABS)
		orgwarn++;
	else if ((locxp->e_xtype & ~XXTRN) != dotp->e_xtype)
		yyerror("Illegal expression to set origin");
	space_value = locxp->e_xvalue - dotp->e_xvalue;
	if (space_value < 0)
		yyerror("Backwards 'org'");
	goto ospace;
	break;

/*
 *
 *	Process stabs.  Stabs are created only by the f77
 *	and the C compiler with the -g flag set.
 *	We only look at the stab ONCE, during pass 1, and
 *	virtually remove the stab from the intermediate file
 *	so it isn't seen during pass2.  This makes for some
 *	hairy processing to handle labels occuring in
 *	stab entries, but since most expressions in the
 *	stab are integral we save lots of time in the second
 *	pass by not looking at the stabs.
 *	A stab that is tagged floating will be bumped during
 *	the jxxx resolution phase.  A stab tagged fixed will
 *	not be be bumped.
 *
 *	.stab:	Old fashioned stabs
 *	.stabn: For stabs without names
 *	.stabs:	For stabs with string names
 *	.stabd: For stabs for line numbers or bracketing,
 *		without a string name, without
 *		a final expression.  The value of the
 *		final expression is taken to be  the current
 *		location counter, and is patched by the 2nd pass
 *
 *	.stab{<expr>,}*NCPS,<expr>, <expr>, <expr>, <expr>
 *	.stabn		 <expr>, <expr>, <expr>, <expr>
 *	.stabs   STRING, <expr>, <expr>, <expr>, <expr>
 *	.stabd		 <expr>, <expr>, <expr> # . 
 */
   case ISTAB: 
#ifndef FLEXNAMES
	stabname = ".stab";
	if (passno == 2)	goto errorfix;
	stpt = (struct symtab *)yylval;
	/*
	 *	Make a pointer to the .stab slot.
	 *	There is a pointer in the way (stpt), and
	 *	tokptr points to the next token.
	 */
	stabstart = tokptr;
	(char *)stabstart -= sizeof(struct symtab *);
	(char *)stabstart -= sizeof(toktype);
	shift;
	for (argcnt = 0; argcnt < NCPS; argcnt++){
		expr(locxp, val);
		stpt->s_name[argcnt] = locxp->e_xvalue;
		xp = explist;
		shiftover(CM);
	}
	goto tailstab;
#else	FLEXNAMES
	yyerror(".stab directive not supported in; report this compiler bug to system administrator");
	goto errorfix;
#endif FLEXNAMES

  tailstab:
	expr(locxp, val);
	if (! (locxp->e_xvalue & STABTYPS)){
		yyerror("Invalid type in %s",stabname);
		goto errorfix;
	}
	stpt->s_ptype = locxp->e_xvalue;
	shiftover(CM);
	expr(locxp, val);
	stpt->s_other = locxp->e_xvalue;
	shiftover(CM);
	expr(locxp, val);
	stpt->s_desc = locxp->e_xvalue;
	shiftover(CM);
	exprisname = 0;
	expr(locxp, val);
	p = locxp->e_xname;
	if (p == NULL) {	/*absolute expr to begin with*/
		stpt->s_value = locxp->e_xvalue;
		stpt->s_index = dotp - usedot;
		if (exprisname){
			switch(stpt->s_ptype){
				case N_GSYM:
				case N_FNAME:
				case N_RSYM:
				case N_SSYM:
				case N_LSYM:
				case N_PSYM:
				case N_BCOMM:
				case N_ECOMM:
				case N_LENG:
					stpt->s_tag = STABFIXED;
					break;
				default:
					stpt->s_tag = STABFLOATING;
					break;
			}
		} else
			stpt->s_tag = STABFIXED;
	}
	else {		/*really have a name*/
		stpt->s_dest = locxp->e_xname;
		stpt->s_index = p->s_index;
		stpt->s_type = p->s_type | STABFLAG;
		/*
		 *	We will assign a more accruate
		 *	guess of locxp's location when
		 *	we sort the symbol table
		 *	The final value of value is
		 *	given by stabfix()
		 */
		stpt->s_tag = STABFLOATING;
	}
	/*
	 *	tokptr now points at one token beyond
	 *	the current token stored in val and yylval,
	 *	which are the next tokens after the end of
	 *	this .stab directive.  This next token must
	 *	be either a SEMI or NL, so is of width just
	 *	one.  Therefore, to point to the next token
	 *	after the end of this stab, just back up one..
	 */
	buildskip(stabstart, (char *)tokptr - sizeof(toktype));
	break;	/*end of the .stab*/

   case ISTABDOT:	
	stabname = ".stabd";
	stpt = (struct symtab *)yylval;
	/*
	 *	We clobber everything after the
	 *	.stabd and its pointer... we MUST
	 *	be able to get back to this .stabd
	 *	so that we can resolve its final value
	 */
	stabstart = tokptr;
	shift;		/*over the ISTABDOT*/
	if (passno == 1){
		expr(locxp, val);
		if (! (locxp->e_xvalue & STABTYPS)){
			yyerror("Invalid type in .stabd");
			goto errorfix;
		}
		stpt->s_ptype = locxp->e_xvalue;
		shiftover(CM);
		expr(locxp, val);
		stpt->s_other = locxp->e_xvalue;
		shiftover(CM);
		expr(locxp, val);
		stpt->s_desc = locxp->e_xvalue;
		/*
		 *
		 *	Now, clobber everything but the
		 *	.stabd pseudo and the pointer
		 *	to its symbol table entry
		 *	tokptr points to the next token,
		 *	build the skip up to this
		 */
		buildskip(stabstart, (toktype *)tokptr - sizeof(toktype));
	}
	/*
	 *	pass 1:	Assign a good guess for its position
	 *		(ensures they are sorted into right place)/
	 *	pass 2:	Fix the actual value
	 */
	stpt->s_value = dotp->e_xvalue;
	stpt->s_index = dotp - usedot;
	stpt->s_tag = STABFLOATING;	/*although it has no effect in pass 2*/
	break;

   case ISTABNONE:	stabname = ".stabn"; goto shortstab;

   case ISTABSTR: 	stabname = ".stabs";
   shortstab:
	auxval = val;
	if (passno == 2) goto errorfix;
	stpt = (struct symtab *)yylval;
	stabstart = tokptr;
	(char *)stabstart -= sizeof(struct symtab *);
	(char *)stabstart -= sizeof(toktype);
	shift;
	if (auxval == ISTABSTR){
		stringp = (struct strdesc *)yylval;
		shiftover(STRING);
#ifndef FLEXNAMES
		auxval = stringp->str_lg > NCPS ? NCPS : stringp->str_lg;
#else
		stringp->str[stringp->str_lg] = 0;
#endif
		shiftover(CM);
	} else {
		stringp = &(strbuf[2]);
#ifndef FLEXNAMES
		auxval = NCPS;
#endif
	}
#ifndef FLEXNAMES
	movestr(stpt->s_name, stringp->str, auxval);
#else
	stpt->s_name = savestr(stringp->str);
#endif
	goto tailstab;
	break;

   case ICOMM:		/* .comm  <name> , <expr> */
   case ILCOMM: 	/* .lcomm <name> , <expr> */
	auxval = val;
	shift;
	np = (struct symtab *)yylval;
	shiftover(NAME);
	shiftover(CM);
	expr(locxp, val);

	if (locxp->e_xtype != XABS)
		yyerror("comm size not absolute");
	if (passno == 1) {
		if ((np->s_type&XTYPE) != XUNDEF &&
		    ((np->s_type&XTYPE) != XCOMM ||
		      (np->s_value != locxp->e_xvalue &&
		       np->s_value != 0)))
#ifdef FLEXNAMES
			yyerror("Redefinition of %s",
#else not FLEXNAMES
			yyerror("Redefinition of %.*s",
				NCPS,
#endif not FLEXNAMES
				np->s_name);

		np->s_value = (locxp->e_xvalue+3) & ~3;	/* long align */
		np->s_type &= ~XTYPE;
		if (auxval == ICOMM)
			np->s_type |= XXTRN | XCOMM;
		else {
			np->s_type |= XBSS;
		}
	}
	break;

   case IEVEN:	 		/* .even */
	shift;
	flushfield(NBPW/4);
	if (dotp->e_xvalue & 1)
		outb(0);
	break;

   case IALIGN: 		/* .align <expr> */
	shift;
	expr(locxp, val);
	if (locxp->e_xtype != XABS  ||
	    locxp->e_xvalue < 0  ||
	    locxp->e_xvalue > 16) {
		yyerror("Illegal `align' argument");
		return;
	}
	flushfield(NBPW/4);
	if (passno == 1  &&  locxp->e_xvalue > 2) {
		yywarning(".align %d in any segment is NOT preserved by the loader",
			locxp->e_xvalue);
		yywarning(".align %d converted to .align 1", locxp->e_xvalue);
		locxp->e_xvalue = 1;
	}
	if (dotp->e_xvalue & 1)
		outb(0);
	if (locxp->e_xvalue == 2 && (dotp->e_xvalue & 2))
		outw(0);
	break;

   case INST0: 		/* instructions w/o arguments*/
	insout(yylval, (struct arg *)0, 0);
	shift;	
	break;

   case INSTn:		/* instructions with arguments*/
   case IJXXX: 		/* UNIX style jump instructions */
	auxval = val;
	seg_type = yylval;
	/*
	 *	Code to process an argument list
	 */
	ap = arglist;
	xp = explist;	

	shift;		/* bring in the first token for the arg list*/

	for (argcnt = 1; argcnt <= 6; argcnt++, ap++){
		/*
		 *	code to process an argument proper
		 */
		switch(val) {

		   case REG:
		   case REGOP: 
			findreg(regno);
			ap->a_areg1 = regno;
			if (val == COLON) {
				if (regno > 7 && !(regno >= 16 && regno <= 23)) {
					ERROR("bad regpair component");
				}
				shift;
				findreg(regno);
				if ((regno > 7 && ap->a_areg1 <= 7) ||
				    (regno <= 7 && ap->a_areg1 >= 16)) {
					ERROR("bad regpair component");
				}
				ap->a_areg2 = regno;
				if (ap->a_areg1 >= 16)
					ap->a_atype = FREGPAIR;
				else
					ap->a_atype = REGPAIR;
				ap->a_amode = M_INVALID;
				break;
			} else if (val != IND) {
				if (regno < 8) {
					ap->a_atype = DREG;
					ap->a_amode = M_DREG;
				} else if (regno < 16) {
					ap->a_atype = AREG;
					ap->a_amode = M_AREG;
				} else if (regno < 24) {
					ap->a_atype = FREG;
					ap->a_amode = M_INVALID;
				} else {
					ap->a_amode = M_INVALID;
					switch (regno) {
					case R_ZA: /* ZA */
					case R_PC: /* PC */
					case R_ZPC: /* ZPC */
						ERROR("bad reg"); break;
					case R_CC: ap->a_atype = CC; break;
					case R_SR: ap->a_atype = SR; break;
					case R_USP: ap->a_atype = USP; break;
					case R_VB: /* VB */
					case R_CAC: /* CAC */
					case R_CAA: /* CAA */
					case R_MSP: /* MSP */
					case R_ISP: /* ISP */
						ap->a_atype = CREG;
						break;
					case R_SFC: /* SFC */
						ap->a_atype = SFC; break;
					case R_DFC: /* DFC */
						ap->a_atype = DFC; break;
					case R_FPCR:
						ap->a_atype = FCNTL; break;
					case R_FPSR:
						ap->a_atype = FSTAT; break;
					case R_FPIAR:
						ap->a_atype = FADDR; break;

					case R_MMUTC:
						ap->a_atype = MMUTC; break;
					case R_MMUSRP:
						ap->a_atype = MMUSRP; break;
					case R_MMUCRP:
						ap->a_atype = MMUCRP; break;
					case R_MMUTT0:
						ap->a_atype = MMUTT0; break;
					case R_MMUTT1:
						ap->a_atype = MMUTT1; break;
					case R_MMUSR:
						ap->a_atype = MMUSR; break;
					}
				}
				break;
			}
			if (!((regno >= 8 && regno <= 15) ||
			      (regno >= R_PC && regno <= R_ZA))) {
				ERROR("not address register");
			}
			shiftover(IND);
			if (val == PLUS) {
				if (regno > 15) {
					ERROR("no such mode");
				}
				shift;
				ap->a_atype = INC;
				ap->a_amode = M_INC;
				break;
			} else if (val == MINUS) {
				if (regno > 15) {
					ERROR("no such mode");
				}
				shift;
				ap->a_atype = DEC;
				ap->a_amode = M_DEC;
				break;
			} else if (val == COLON) {
				shift;
				findreg(regno);
				if (regno < 8 || regno > 15) {
					ERROR("bad regpair component");
				}
				ap->a_areg2 = regno;
				ap->a_atype = REGPAIR;
				ap->a_amode = M_INVALID;
				if (val != IND) {
					ERROR("missing '@' in regpair");
				}
				shift;
				break;
			} else if (val != LP) {
				if (regno > 15) {
					ERROR("no such mode");
				}
				ap->a_atype = ANINDIR;
				ap->a_amode = M_INDR;
				break;
			}
			shiftover(LP);

			if( !(INTOKSET(val,
				 EBEGOPS
				+YUKKYEXPRBEG
				+SAFEEXPRBEG)) ) {
				ERROR("index expression expected");
			}
			expr(ap->a_xp,val);
			if (val == SIZESPEC) {
				ap->a_bsize = yylval;
				shift;
				if (passno == 1 && !for68020)
					yywarning("68020 disp. size spec. used");
				if (ap->a_bsize != 2  &&  ap->a_bsize != 4) {
					ERROR ("illegal size spec.");
				}
			} else
				ap->a_bsize = 0;  /* size not specified */
			if (val == RP) {
				if (regno == R_PC || regno == R_ZPC) {
					ap->a_atype = PCINDIR;
					ap->a_amode = M_PCDISP;
				} else {
					ap->a_atype = DISP;
					ap->a_amode = M_DISP;
				}
				shift;
				if (val == LB) {
					shift;
					val = post_index(ap, val);
				} else
					ap->a_od = NULL;
				break;
			} else if (val != CM) {
				ERROR ("')' or ',' expected");
			}
			shiftover(CM);

			if (regno == R_PC || regno == R_ZPC) {
				ap->a_atype = PCINDIR;
				ap->a_amode = M_PCINDEX;
			} else {
				ap->a_atype = ANINDIR;
				ap->a_amode = M_INDEX;
			}
			if (val != REG  &&  val != REGOP) {
				ERROR ("register expected");
			}
			findreg(regno);
			ap->a_areg2 = regno;
			if (val == SIZESPEC) {
				ap->a_xsize = yylval;
				shift;
				if (ap->a_xsize != 2  &&  ap->a_xsize != 4) {
					ERROR ("illegal size spec.");
				}
			} else
				ap->a_xsize = 4;  /* default: long index */
			if (val == MUL) {
				shift;
				if (val != INT) {
					ERROR ("illegal scale spec");
				}
				switch ( ((struct exp *)yylval)->e_xvalue ) {
				case 1: ap->a_ascale = 0; break;
				case 2: ap->a_ascale = 1; break;
				case 4: ap->a_ascale = 2; break;
				case 8: ap->a_ascale = 3; break;
				default:
					ERROR ("illegal scale spec");
				}
				shift;
			} else
				ap->a_ascale = 0;  /* default: byte scale */
			if (val != RP) {
				ERROR ("')' expected");
			}
			shiftover(RP);

			if (val == LB) {
				shift;
				if( !(INTOKSET(val,
					 EBEGOPS
					+YUKKYEXPRBEG
					+SAFEEXPRBEG)) ) {
					ERROR("outer displacement expression expected");
				}
				expr(ap->a_od,val);
				if (val == SIZESPEC) {
					ap->a_osize = yylval;
					shift;
					if (passno == 1 && !for68020)
						yywarning("68020 disp. size spec. used");
					if (ap->a_osize != 2  &&  ap->a_osize != 4) {
						ERROR ("illegal size spec.");
					}
				} else
					ap->a_osize = 0;  /* size not specified */
				if (val != RB) {
					ERROR ("']' expected");
				}
				shift;
				ap->a_amode = (ap->a_amode == M_INDEX) ?
					M_PREX : M_PCPREX;
			} else
				ap->a_od = NULL;
			break;

		   case LITOP:
		   {
			int littype;
			shift;
			if (val == LA) {
				val = reg_list(ap);
				break;
			} else if( !(INTOKSET(val,
					 EBEGOPS
					+YUKKYEXPRBEG
					+SAFEEXPRBEG)) ) {
				ERROR("immediate expression expected");
			}
			littype = val;
			expr(ap->a_xp,val);
			ap->a_amode = M_IMM;
			if (((ap->a_xp->e_xtype&XTYPE)==XABS)  &&
			    (!(ap->a_xp->e_xtype&XFORW))) {
				ap->a_atype = 0;
				if (littype == FLTNUM) {
					double fabs();
					double dvalue =
						fabs(((union Double *)ap->a_xp)->dvalue);
					if (dvalue == 0.0 ||
#ifndef IEEE
					    (dvalue <= 1.7e38 &&
					     dvalue >= 0.294e-38))
#else IEEE
					    (dvalue <= 3.4028234e38 &&
					     dvalue >= 1.1754944e-38))
#endif IEEE
						ap->a_atype |= IMM_F;
					ap->a_atype |= IMM_D;

				} else {
					register v = ap->a_xp->e_xvalue;

					if (v == 1)
						ap->a_atype |= IMM_1;
					if (v >= 0  &&  v <= 7)
						ap->a_atype |= IMB_3;
					if (v >= 1  &&  v <= 8)
						ap->a_atype |= IMM_3;
					if (v >= 0  &&  v <= 31)
						ap->a_atype |= IMB_5;
					if (v >= -128  &&  v <= 127)
						ap->a_atype |= IMM_SB;
					if (v >= -128  &&  v <= 255)
						ap->a_atype |= IMM_B;
					if (v >= -32768  &&  v <= 32767)
						ap->a_atype |= IMM_SW;
					if (v >= -32768  &&  v <= 65535)
						ap->a_atype |= IMM_W;
					ap->a_atype |= IMM_L;
				}
			} else
				ap->a_atype = IMM;
			break;
		   }

		   default:
			if( !(INTOKSET(val,
				 EBEGOPS
				+YUKKYEXPRBEG
				+SAFEEXPRBEG)) ) {
				ERROR("expression expected");
			}
			expr(ap->a_xp,val);
			ap->a_atype = ABS;
			if (val == SIZESPEC) {
				ap->a_xsize = yylval;
				shift;
				if (ap->a_xsize != 2  &&  ap->a_xsize != 4) {
					ERROR ("illegal size spec.");
				}
			} else
				ap->a_xsize = 0;	/* unknown for now */
			break;

		}	/*end of the switch to process an arg*/

		if (val == LCB) {
			shift;
			/* bitfield/k-factor specs will be uncommon, */
			/* so proc call overhead is okay here. */
			val = bfselect(ap, val);
		} else {
			/* flag as without any spec */
			ap->a_hasbf = 0;
			ap->a_haskfac = 0;
		}
		if (val != CM) break;
		shiftover(CM);
	}	/*processing all the arguments*/

	if (argcnt > 6){
		yyerror("More than 6 arguments");
		goto errorfix;
	}

	insout(seg_type, arglist,
		auxval == INSTn ? argcnt : - argcnt);
	break;

   case IFLOAT:
	curlen = 4;
	if (dotp->e_xvalue & 1) {
		if (passno ==1)
			yywarning(".float occurs at odd address");
		outb(0);
	}
	goto floatlist;
   case IQUAD:
   case IDOUBLE: 
	curlen = 8;
	if (dotp->e_xvalue & 1) {
		if (passno ==1)
			yywarning(".double (or .quad) occurs at odd address");
		outb(0);
	}
      floatlist:	
	/*
	 *	eat a list of floating point numbers
	 */
	shift;
	if (val == FLTNUM){
		/* KLS MOD */
		float flocal;
#ifdef VAX_CROSS
		union fltpt_kludge {
			double dbl;
			float flt;
			unsigned short sh[4];
		} kluj;
#endif VAX_CROSS
		do{
			if (val == CM) shift;
			if (val != FLTNUM) {
			  ERROR("floating number expected");
			}
			if (passno == 1)
				dotp->e_xvalue += curlen;
			else {
#ifdef VAX_CROSS
			  /* can't use bwrite() for these: byte order!.
			   * also, outw increments dotp
			   */
			  if(curlen == 8) {
			    kluj.dbl = ((union Double *)yylval)->dvalue;
			    outw(kluj.sh[0]);
			    outw(kluj.sh[1]);
			    outw(kluj.sh[2]);
			    outw(kluj.sh[3]);
			  } else  {
			    kluj.flt = ((union Double *)yylval)->dvalue;
			    outw(kluj.sh[0]);
			    outw(kluj.sh[1]);
			  }
#else VAX_CROSS
			  dotp->e_xvalue += curlen;
			  if(curlen == 8) {
			    bwrite((char *)&(((union Double *)yylval)->dvalue),
				curlen, txtfil);
			  } else  {
			    flocal = ((union Double *)yylval)->dvalue;
			    bwrite((char *)&flocal, curlen, txtfil);
			  }
#endif VAX_CROSS
			}
			shift;
			xp = explist;
		} while (val == CM);
	}
	break;
    }	/*end of the switch for looking at each reserved word*/

     continue;

   errorfix: 
	/*
	 *	got here by either requesting to skip to the
	 *	end of this statement, or by erroring out and
	 *	wanting to apply panic mode recovery
	 */
	while (    (val != NL) 
		&& (val != SEMI) 
		&& (val != PARSEEOF)
	      ){
		shift;
	}
	if (val == NL)
		lineno++;
	shift;

    }	/*end of the loop to read the entire file, line by line*/

}	/*end of yyparse*/
	
/* Process a register list for MOVEM.
 * Format is: '<' regs {','regs} '>'
 * where regs is: reg['-'reg]
 */
reg_list (ap)
register struct arg *ap;		/*first free argument*/
{
	register int	val;		/*what yylex gives*/
	register int	reg1;
	register unsigned reg_mask;
		int	regno;		/*handles registers*/
		int	*ptrregno = &regno;

	reg_mask = 0;
	ap->a_areg1 = -1;
	do {
		shift;		/* skip over "," (or "<") */
		findreg(regno);
		if (ap->a_areg1 < 0)
			ap->a_areg1 = regno;
		if (regno >= R_PC ) {
			ERROR ("Invalid register in list.");
		} else if (val == MINUS) {
			/* reg1-reg2 sequence */
			shift;
			reg1 = regno;
			findreg(regno);
			if (regno <= reg1) {
				ERROR ("Invalid register list range.");
			} else
				for (; reg1 <= regno; reg1++)
					reg_mask |= 1 << reg1;
		} else {
			reg_mask |= 1 << regno;
		}
	} while (val == CM);

	shiftover(RA);

	/* check to see that registers have not been mixed */
	if (ap->a_areg1 < 16) {
		/* only d0..a7 allowed here */
		if (reg_mask & ~((1 << 16) - 1)) {
			yyerror("Invalid register list");
			reg_mask &= ~((1 << 16) - 1);
		}
		ap->a_xsize = 2;
	} else if (ap->a_areg1 >= 16 && ap->a_areg1 <= 23) {
		/* only f0..f7 allowed here */
		if (reg_mask & ~(((1 << 8) - 1) << 16)) {
			yyerror("Invalid register list");
			reg_mask &= ~((1 << 8) - 1) << 16;
		}
		reg_mask >>= 16;
		ap->a_xsize = 1;
	} else if (ap->a_areg1 >= R_FPIAR && ap->a_areg1 <= R_FPCR) {
		/* only FPCR,FPSR,FPIAR allowed here */
		if (reg_mask & ~(07 << R_FPIAR)) {
			yyerror("Invalid register list");
			reg_mask &= ~(07 << R_FPIAR);
		}
		reg_mask >>= R_FPIAR;
		ap->a_xsize = 0;
	}


	ap->a_xp = xp++;
	ap->a_xp->e_xtype = XABS;
	ap->a_xp->e_xvalue = reg_mask;
	ap->a_xp->e_xloc = 0;
	ap->a_xp->e_xname = NULL;
	ap->a_atype = REGLIST;
	ap->a_amode = M_INVALID;
	/* ap->a_areg1 is left unchanged for further later error checks. */
	return val;

   errorfix: 
	/*
	 *	got here by either requesting to skip to the
	 *	end of this statement, or by erroring out and
	 *	wanting to apply panic mode recovery
	 */
	while (    (val != NL) 
		&& (val != SEMI) 
		&& (val != PARSEEOF)
	      ){
		shift;
	}
	if (val == NL)
		lineno++;
	shift;
	return val;

}

/*
 *	Process a post-indexed mode of the form
 *
 *		<disp mode>[Rn:<sz>*<scale>, <outer disp>]
 *
 *	The <disp mode> and the leading [ have already been processed.
 *	ap points to the <disp mode> processed so far.
 */
int
post_index(ap, val)
register struct arg *ap;
register int  val;		/*what yylex gives*/
{

	int	regno;		/*handles arguments*/
	int	*ptrregno = &regno;
	/*
	 *	loc1xp and ptrloc1xp are used in the
	 * 	expression lookahead
	 */
	struct	exp	*loc1xp;	/*must be non register*/
	struct	exp	**ptrloc1xp = & loc1xp;

	if (val == REG || val == REGOP) {
		findreg(regno);
		ap->a_areg2 = regno;
		if (val == SIZESPEC) {
			ap->a_xsize = yylval;
			shift;
			if (ap->a_xsize != 2  &&  ap->a_xsize != 4) {
				ERROR ("illegal size spec.");
			}
		} else
			ap->a_xsize = 4;  /* default: long index */
		if (val == MUL) {
			shift;
			if (val != INT) {
				ERROR ("illegal scale spec");
			}
			switch ( ((struct exp *)yylval)->e_xvalue ) {
			case 1: ap->a_ascale = 0; break;
			case 2: ap->a_ascale = 1; break;
			case 4: ap->a_ascale = 2; break;
			case 8: ap->a_ascale = 3; break;
			default:
				ERROR ("illegal scale spec");
			}
			shift;
		} else
			ap->a_ascale = 0;  /* default: byte scale */
		if (val != CM) {
			ERROR ("',' expected");
		}
		shiftover(CM);
	} else
		ap->a_areg2 = -1;

	if( !(INTOKSET(val,
		 EBEGOPS
		+YUKKYEXPRBEG
		+SAFEEXPRBEG)) ) {
		ERROR("outer displacement expression expected");
	}
	expr(ap->a_od,val);
	if (val == SIZESPEC) {
		ap->a_osize = yylval;
		shift;
		if (!for68020)
			yywarning("68020 disp. size spec. used");
		if (ap->a_osize != 2  &&  ap->a_osize != 4) {
			ERROR ("illegal size spec.");
		}
	} else
		ap->a_osize = 0;  /* size not specified */
	if (val != RB) {
		ERROR ("']' expected");
	}
	shift;

	if (ap->a_areg1 == R_PC || ap->a_areg1 == R_ZPC) {
		ap->a_atype = PCINDIR;
		ap->a_amode = M_PCPOSTX;
	} else {
		ap->a_atype = ANINDIR;
		ap->a_amode = M_POSTX;
	}
	return (val);

   errorfix: 
	while (    (val != NL) 
		&& (val != SEMI) 
		&& (val != PARSEEOF)
	      ){
		shift;
	}
	return(val);
}

/*
 *	Process a bit field selection of the form
 *
 *		<ea>{ offset:width }
 *
 *	where offset and width can be either a data register
 *	or an integer in the range 0-31.
 *	The <ea> and the leading { have already been processed.
 *	ap points to the <ea> processed so far.
 *
 *	This routine also handles the k-factor for the fmovep instruction.
 */
int
bfselect(ap, val)
register struct arg *ap;
register int  val;		/*what yylex gives*/
{

	register long iv;
	short needint = 0;

	if (val == LITOP) {
		needint++;
		shift;
	}

	if (val == REG) {
		if (needint) {
			ERROR("invalid k-factor");
		}
		ap->a_bforeg = 1;
		if ((ap->a_bfoff = yylval) > 7) {
			ERROR("invalid selection offset register");
		}
	} else if (val == INT) {
		ap->a_bforeg = 0;
		iv = ((struct exp *)yylval)->e_xvalue;
		/* range checks are below, when we know what to check */
		ap->a_bfoff = iv;
	} else if (val == MINUS) {
		shift;
		if (val == INT) {
			ap->a_bforeg = 0;
			iv = -((struct exp *)yylval)->e_xvalue;
			/* range checks below */
			ap->a_bfoff = iv;
		} else {
			ERROR("invalid k-factor");
		}
	} else {
		ERROR("invalid selection offset");
	}
	shift;

	if (val != COLON) {
		if (val == RCB) {
			ap->a_haskfac = 1;
			if (passno == 1 &&
			    !ap->a_bforeg && (iv < -64 || iv > 17))
				yywarning("k-factor out of range");
		} else {
			ERROR("bogus selection syntax");
		}
	} else {
		ap->a_hasbf = 1;
		if (passno == 1 &&
		    !ap->a_bforeg && (iv < 0 || iv > 31))
			yywarning("bitfield offset truncated");

		shift;
		if (val == REG) {
			ap->a_bfwreg = 1;
			if ((ap->a_bfwid = yylval) > 7) {
				ERROR("invalid bitfield width register");
			}
		} else if (val == INT) {
			ap->a_bfwreg = 0;
			iv = ((struct exp *)yylval)->e_xvalue;
			if (passno == 1 &&
			    (iv < 1 || iv > 32))
				yywarning("bitfield width truncated");
			ap->a_bfwid = iv;
		} else {
			ERROR("invalid bitfield width");
		}
		shift;
	}

	if (val != RCB) {
		ERROR("'}' missing in selection");
	}
	shift;

	return(val);

   errorfix: 
	while (    (val != NL) 
		&& (val != SEMI) 
		&& (val != PARSEEOF)
	      ){
		shift;
	}
	return(val);
}

/*
 *	Process a register declaration of the form
 *	% <expr>
 *
 *	Note:
 *		The scanner has already processed funny registers of the form
 *	%dd[+-]*, where dd is a decimal number in the range 00 to 15 (optional
 *	preceding zero digit).  If there was any space between the % and
 *	the digit, the scanner wouldn't have recognized it, so we
 *	hack it out here.
 */
int funnyreg(val, regnoback)		/*what the read head will sit on*/
	int	val;			/*what the read head is sitting on*/
	int	*regnoback;		/*call by return*/
{
	register	struct	exp *locxp;
			struct	exp *loc1xp;
			struct	exp **ptrloc1xp = & loc1xp;

	expr(locxp, val);	/*and leave the current read head with value*/
	if ( (passno == 2) &&
	    (   (locxp->e_xtype & XTYPE) != XABS
	     || locxp->e_xvalue < 0
	     || locxp->e_xvalue >= R_PC 
	    )
	  ){
		yyerror("Illegal register");
		return(0);
	}
	*regnoback = locxp->e_xvalue;
	return(val);
} 

/*VARARGS1*/
yyerror(s, a1, a2,a3,a4,a5)
	char	*s;
{

#define	sink stdout

	if (anyerrs == 0 && ! silent) 
		fprintf(sink, "Assembler:\n");
	anyerrs++;
	if (silent) return;
	
	fprintf(sink, "\"%s\", line %d: ", dotsname, lineno);
	fprintf(sink, s, a1, a2,a3,a4,a5);
	fprintf(sink, "\n");
}

/*VARARGS1*/
yywarning(s, a1, a2,a3,a4,a5)
	char	*s;
{

#define	sink stdout

	if (anyerrs == 0 && ! silent) 
		fprintf(sink, "Assembler:\n");
	if (silent) return;
	
	fprintf(sink, "\"%s\", line %d: WARNING: ", dotsname, lineno);
	fprintf(sink, s, a1, a2,a3,a4,a5);
	fprintf(sink, "\n");
}
