/*
 *	Tiles are the essence of virtual terminals.  Tiles are the
 *	indivisible units dealt with by the virtual terminal manager.  A tile
 *	is a rectanglular area which is either fully hidden or exposed to
 *	another tile or rectangle.  Tiles are kept in lists represnting such
 *	things as windows and are constantly being recomputed to check for
 *	overlapp and other nasty things.  Tiles are doubly linked forward 
 *	(points to) and backwards (points from) other tiles.  The routines 
 *	below are the main tile manipulation entities and work as described.
 */

#include <vt_hdrs.h>		    /* system global defines for sherwood */
#include <rectangle.h>		    /* defines & functions for rectangle */
#include <tile.h>		    /* structures et al for tiles */
#include <window.h>		    /* structures et al for windows */
#include <cursor.h>		    /* structures et al for cursors */


extern struct window *desk;	    /* the desktop, declared in window.c */
struct tile *freetiles;		    /* tile free list */

/*
 *	Routine to create a new empty tile list
 */
struct tile *CreateTileList()
{
    register struct tile *list;

    if (freetiles == NULL)	/* get new block if tile freelist is empty */
	list = (struct tile *)CreateBlock(sizeof(struct tile));
    else			/* otherwise use one from the tile freelist */
    {
	list = freetiles;
	freetiles = freetiles->next;	/* point to next free tile */
    }

    list->prev = list;			/* point tile to self for now */
    list->next = list;

    return(list);			/* return a tile pointer */
}

/*
 *	Routine to return an entire tile list to the tile freelist
 */
DestroyTileList(list)
register struct tile *list;
{
    list->prev->next = freetiles;
    freetiles = list;
}

/*
 *	Routine to get a new tile and insert it into a tile list after the
 *	given tile.
 */
struct tile *InsertTile(tile)
register struct tile *tile;
{
    register struct tile *nile;	    /* new tile */

    if (freetiles == NULL)   /* get a new block if tile freelist is empty */
	nile = (struct tile *)CreateBlock(sizeof(struct tile));
    else		    /* otherwise get block from tile freelist */
    {
	nile = freetiles;
	freetiles = freetiles->next;	/* point to next free tile on list */
    }

    /* insert new tile after given tile */
    nile->next = tile->next;	/* point to where given tile pointed to */
    nile->prev = tile;		/* point from given tile */

    nile->next->prev = nile;	/* tile that given pointed from/to now */
    nile->prev->next = nile;	/* points to new tile */

    return(nile);		/* return new tile */
}

/*
 *	Routine to remove the given tile from its tile list
 */
DeleteTile(tile)
register struct tile *tile;
{
    tile->prev->next = tile->next;  /* fix pointers to point around tile */
    tile->next->prev = tile->prev;

    tile->next = freetiles;	    /* put this tile back on free tile list */
    freetiles = tile;
}

/*
 * routine to reorder the given tile by y value using a simple bubble sort
 */
SortTileList(tiles, lessthan)
register struct tile * tiles;
register lessthan;
{
    register struct tile * looptile, * prevtile, *stoptile;

    /*
     * loop until all tiles are bubbled
     */
    stoptile = tiles;
    do {
	/*
	 * loop through all tiles starting at end of list
	 */
	looptile = tiles->prev; 
	while (looptile->prev != stoptile) {
	    prevtile = looptile->prev; 
	    /*
	     * swap this pair of tiles if needed, otherwise reset pointer
	     */
	    if ((lessthan) ? (looptile->bounds.y < prevtile->bounds.y)
		 : (prevtile->bounds.y < looptile->bounds.y)) {
		SwapTiles(prevtile, looptile);
	    } else {
		looptile = looptile->prev;
	    }
	}
	stoptile = stoptile->next;
    } while (stoptile != tiles->prev);
}  				/* end of SortTileList */

/*
 * routine to swap the two given tiles in a tile list, assume tile1 is 
 * immediatily before tile2 in list.
 */
SwapTiles(tile1, tile2)
register struct tile * tile1, *tile2;
{
    tile1->prev->next = tile2;
    tile2->next->prev = tile1;
    tile1->next = tile2->next;
    tile2->prev = tile1->prev;
    tile2->next = tile1;
    tile1->prev = tile2;
}

/*
 *	Routine to put a tile list back together again, i.e merge tiles with
 *	common borders and the same visibility - remove redundancies.
 */
Optimize(tiles)
struct tile *tiles;
{
    struct tile *hitile, *lotile;   /* temporary tile poniters */

    /* loop through all tiles in this tile list */
    for (hitile = tiles->next; hitile != tiles; hitile = hitile->next)
    {
	/* loop through rest of tiles in list, stopping if two tiles have the
	 * same visibility and touching borders */
	for (lotile = tiles->next; 
	     lotile != tiles &&	!(lotile->hidden == hitile->hidden
				 && Adjacent(hitile->bounds, lotile->bounds));
						    lotile = lotile->next);
	/* if we stopped early above, remove extra tile, remake tile bounds
	 * and do it all again */
	if (lotile != tiles)
	{
	    DeleteTile(lotile);
	    Union(&hitile->bounds, hitile->bounds, lotile->bounds);
	    hitile = hitile->prev;
	}
    }
}

/*
 *	Routine to remove any tiles of the given visibility from a tile list
 */
Filter(tiles, visibility)
register struct tile *tiles;
register bool visibility;
{
    register struct tile *tile;

    /* loop thru all tiles in this list */
    for (tile = tiles->next; tile != tiles; tile = tile->next)
	if (tile->hidden == visibility)  /* remove tile if visability is as
					  * specified */
	{
	    tile = tile->prev;
	    DeleteTile(tile->next);
	}
} 

/*
 *	Routine to partition a given tile list into a null intersected
 *	covering set of rectangluar tiles such that each tile is either fully
 *	hidden or exposed with respect to the given rectangle - i.e. this will
 *	patition a window into tiles with respect to a given rectangle.  A
 *	picture is worth a thousand words.
 *
 *	      +-------------------------|---------------+
 *	      |				|  Added 	|
 *	      |	  Given tile		|  Vertical     |
 *	      |				|  tile		|
 *	      |-------------------------+-----------------------+
 *	      |	added horizonal tile	|		|	|
 *	      +------------------------	|-----Given Rectangle	|
 *					|			|
 *					+-----------------------+
 *
 */
Partition(tiles, bounds)
register struct tile *tiles;
struct rectangle bounds;
{
    register struct tile *tile, *nile;
    struct rectangle common;

    /* loop thru all tiles in current tile list */
    for (tile = tiles->next; tile != tiles; tile = tile->next)
    {
	/* check if this tile overlaps given bounds */
	if (Overlap(bounds, tile->bounds))
	{
	    /* find the area of commonality between the tile and bounds */
	    Intersection(&common, bounds, tile->bounds);
	    /* add horizonal tile to bottom of given tile if needed */
	    if (common.y != tile->bounds.y)
	    {
		nile = InsertTile(tile);     /* insert new tile in list */
		nile->hidden = tile->hidden; /* same visibility for now */
		/* make bounds for new tile and recompute for old */
		Bottom(&nile->bounds, tile->bounds, common.y); 
		Top(&tile->bounds, tile->bounds, common.y-1);
	    }
	    /* add horizonal tile to top of given tile if needed */
	    else if (common.y+common.h != tile->bounds.y+tile->bounds.h)
	    {
		nile = InsertTile(tile);     /* insert new tile in list */
		nile->hidden = tile->hidden; /* same visibility for now */
		/* make bounds for new tile and recompute for old */
		Top(&nile->bounds, tile->bounds, common.y+common.h-1);
		Bottom(&tile->bounds, tile->bounds, common.y+common.h);
	    }
	    /* add vertical tile to right of given tile if needed */
	    else if (common.x != tile->bounds.x)
	    {
		nile = InsertTile(tile);
		nile->hidden = tile->hidden;
		Right(&nile->bounds, tile->bounds, common.x);
		Left(&tile->bounds, tile->bounds, common.x-1);
	    }
	    /* add vertical tile to left of given tile if needed */
	    else if (common.x+common.w != tile->bounds.x+tile->bounds.w)
	    {
		nile = InsertTile(tile);
		nile->hidden = tile->hidden;
		Left(&nile->bounds, tile->bounds, common.x+common.w-1);
		Right(&tile->bounds, tile->bounds, common.x+common.w);
	    }
	    /* otherwise tile is totally obsured by rectangle */
	    else
		tile->hidden = TRUE;
	}
    }
    Optimize(tiles);	/* remove any redundant tiles from list and return */
}    

/*
 *	Routine to create a list of visible tiles with respect to all windows
 *	currently on screen with repect to some arbitrary rectangle.
 */
struct tile *Visible(hiwin, lowin, bounds) 
register struct window *hiwin, *lowin;	    /* ordered linked list of windows */
struct rectangle bounds;	    /* physical bounds of screen */
{
    register struct tile *result, *tile;
    register struct window *w;

    result = CreateTileList();	    /* initialize a dummy tile list */

    /* do nothing if rectangle is not on screen */
    if (Overlap(bounds, desk->ibounds))
    {
	tile = InsertTile(result);  /* put a visible tile in the list */
	tile->hidden = FALSE;
	/* compute bounds of rectangle and screen */
	Intersection(&tile->bounds, bounds, desk->ibounds);
    }

    /* loop thru each window and partition tile list correctly */
    for (w = hiwin; w != lowin; w = w->next)
      	Partition(result, w->obounds);
    /* Remove all tiles in list which are not visible */
    Filter(result, TRUE);

    return(result);
}

/*
 *	Routine to create a partition of visible tiles in a rectangle
 *	with respect to another	rectangle.
 */
struct tile *RVisible(upper, lower)
struct rectangle    upper;		/* rectangle of bounds */
struct rectangle    lower;		/* rectangle to partition */
{
    register struct tile *result, *tile;

    result = CreateTileList();		/* make a dummy tile list */

    tile = InsertTile(result);
    tile->hidden = FALSE;
    tile->bounds = lower;		/* set bounds to lower rectangle */

    Partition(result, upper);		/* partition lower rectangle */
    Filter(result, TRUE);		/* remove all non-visible tiles */

    return(result);
}

/*
 *	Routine to create a list of invisible tiles with respect to windows
 *	currently on screen with repect to some arbitrary rectangle (inverse
 *	of Visible above)
 */
struct tile *Covered(hiwin, lowin, bounds) 
struct window *hiwin, *lowin;
struct rectangle bounds;
{
    register struct tile *result, *tile;
    register struct window *w;

    result = CreateTileList();

    if (Overlap(bounds, desk->ibounds))
    {
	tile = InsertTile(result);
	tile->hidden = FALSE;
	Intersection(&tile->bounds, bounds, desk->ibounds);
    }
    
    for (w = hiwin; w != lowin; w = w->next)
	Partition(result, w->obounds);
    Filter(result, FALSE);

    return(result);
}

/*
 *	Routine to create a partition of invisible tiles in a rectangle
 *	with respect to another	rectangle (inverse of RVisible above)
 */
struct tile *RCovered(upper, lower)
struct rectangle upper, lower;
{
    register struct tile *result, *tile;

    result = CreateTileList();

    tile = InsertTile(result);
    tile->hidden = FALSE;
    tile->bounds = lower;

    Partition(result, upper);
    Filter(result, FALSE);

    return(result);
}

/*
 *	Routine to create a tile list of intersecting tiles from two given
 *	tile lists.
 */
struct tile *TileIntersection(t1, t2)
register struct tile *t1, *t2;
{
    register struct tile *tile, *newtile, *result;

    result = CreateTileList();	    /* create a dummy tile list */

    /* copy tile list t1 to newtile */
    for (tile = t1->next; tile != t1; tile = tile->next)
    {
	newtile = InsertTile(result);
	newtile->bounds = tile->bounds;
	newtile->hidden = FALSE;    /* mark it all visible */
    }

    /* loop thru each tile in t2 and partion copy of t1 */
    for (tile = t2->next; tile != t2; tile = tile->next)
	Partition(result, tile->bounds);

    Filter(result, FALSE);	/* remove all visiable tiles from list */

    return(result);
}

/*
 *	Routine to compute the non-intersecting portions of two tile lists
 */
struct tile *TileDifference(t1, t2)
register struct tile *t1, *t2;
{
    register struct tile *tile, *newtile, *result;

    result = CreateTileList();		/* create a dummy tile list */

    /* copy tile list 1 into a temporay location */
    for (tile = t1->next; tile != t1; tile = tile->next)
    {
	newtile = InsertTile(result);
	newtile->bounds = tile->bounds;
	newtile->hidden = FALSE;
    }

    /* loop through all tiles of list two and partition list 1 */
    for (tile = t2->next; tile != t2; tile = tile->next)
	Partition(result, tile->bounds);

    Filter(result, TRUE);   /* remove all invisible tiles from list */

    return(result);
}


/*
 *	Routine to move a tile on the screen a distance dx,dy */
struct tile *TileTranslation(tiles, dx, dy)
register struct tile *tiles;
register int dx, dy;
{
    register struct tile *tile, *newtile, *result;

    result = CreateTileList();	    /* create a dummy tile list */

    /* loop thru all tiles in list and copy with translation */
    for (tile = tiles->next; tile != tiles; tile = tile->next)
    {
	newtile = InsertTile(result);
	newtile->bounds.x = tile->bounds.x + dx;
	newtile->bounds.y = tile->bounds.y + dy;
	newtile->bounds.w = tile->bounds.w;
	newtile->bounds.h = tile->bounds.h;
	newtile->hidden = tile->hidden;
    }

    return(result);
}

/*
 *	Routine to compute a tile list from a given tile list which
 *	represents the intersection of each tile with a given rectangle
 */
struct tile *RestrictTiles(tiles, bounds)
register struct tile *tiles;
struct rectangle bounds;
{
    register struct tile *tile, *result, *nile;
    
    result = CreateTileList();	    /* create a dummy tile list */

    /* loop thru all tiles in list */
    for (tile = tiles->next; tile != tiles; tile = tile->next)
	/* add a new tile of intersection of need be */
	if (Overlap(tile->bounds, bounds))
	{
	    nile = InsertTile(result);
	    Intersection(&nile->bounds, tile->bounds, bounds);
	    nile->hidden = tile->hidden;
	}
	
    Optimize(result);

    return(result);
}

/*
 *	Routine to actually move tiles in a tile list on the screen to a new
 *	location, computing the manner to do it so that no pixel dies and
 *	ugly death.
 */
SlideTiles(tiles, dx, dy)
register struct tile *tiles;
int dx, dy;
{
    register struct tile *unmoved, *t1, *t2;
    struct rectangle bounds;

    /* set global clipping to bounds of the desktop */
    SetClipping(desk->ibounds.x, desk->ibounds.y,
		desk->ibounds.w, desk->ibounds.h);

    /* look at first unmoved tile, loop thru all in list */
    unmoved = tiles->next;
    while (unmoved != tiles)
    {
	/* loop through remaining tiles to find how to copy */
	for (t1 = unmoved; t1 != tiles; t1 = t1->next)
	{
	    /* make temporary copy of this tile in new location */
	    bounds = t1->bounds;
	    bounds.x += dx;
	    bounds.y += dy;

	    /* stop to copy if overlap does occur */
	    for (t2 = unmoved; t2 != tiles; t2 = t2->next)
		if (t1 != t2 && Overlap(bounds, t2->bounds))
		    break;

	    /* break on first file as well */
	    if (t2 == tiles)
		break;
	}

	/* copy tile t1 to new location */
	CopyArea(t1->bounds.x, t1->bounds.y, bounds.x, bounds.y,
		  bounds.w, bounds.h);

	/* if same as started loop then do next one, otherwise move tile to
	 * head of list and relink.
	 */
	if (t1 == unmoved)
	    unmoved = unmoved->next;
	else
	{
	    /* remove tile from current position */
	    t1->next->prev = t1->prev;
	    t1->prev->next = t1->next;

	    /* replace tile at head of list */
	    t1->next = tiles->next;
	    t1->prev = tiles;
	    t1->prev->next = t1;
	    t1->next->prev = t1;
	}
    }
    UnsetClipping();
}
