#include "../h/ucode.h"
/******************************************************************************
* DrawInvertedVector(x, y, dx, dy, thickness)
*
* Draws an inveted vector with a x component of "dx" and a y component of "dy" 
* at screen coordinates ("x", "y").  "Thickness" is the width of the vector in
* pixels.
*
* C Interface:
*
*	DrawInvertedVector(x, y, dx, dy, thickness)
*	int x, y, dx, dy, thickness;
*
* Assembly Interface:
*
*	+---------------------------------+
*	|////////////////| x              |  2 (W)
*	+---------------------------------+
*	|////////////////| y              |  6 (W)
*	+---------------------------------+
*	|////////////////| dx             | 10 (W)
*	+---------------------------------+
*	|////////////////| dy             | 14 (W)
*	+---------------------------------+
*	|////////////////| thickness      | 16 (W)
*	+---------------------------------+
*
******************************************************************************/
	.text
	.globl	_DrawInvertedVector
_DrawInvertedVector:
	link	a6,#0			/* establish frame pointer */
	moveml	#<d2-d7,a2-a5>,sp@-	/* save working registers */

	/* Get C parameters */
	movw	a6@(10),d0		/* get initial x */
	movw	a6@(14),d1		/* get initial y */
	movw	a6@(18),d2		/* get delta x */
	movw	a6@(22),d3		/* get delta y */
	movw	a6@(26),d5		/* get thickness */

	/* Set up deltas and increments */
	movw	d5,d7			/* get a copy of thickness */
	asrw	#1,d7			/* half it for adjustment */
	tstw	d2			/* test delta x */
	bpl	1f			/* branch if left to right vector */

	negw	d2			/* take absolute value of delta x */
	movl	#-1,a0			/* and reverse horizontal increment */
	addw	d7,d0			/* center brush on point */
	bra	2f

    1:	movl	#1,a0			/* initialize horizontal increment */
	subw	d7,d0			/* center brush on point */

    2:	tstw	d3			/* compute delta y */
	bpl	3f			/* branch if top to bottom vector */

	negw	d3			/* take absolute value of delta y */
	movl	#-1,a1			/* reverse horizontal increment */
	addw	d7,d1			/* center brush on point */
	bra	4f

    3:	movl	#1,a1			/* initialize vertical increment */
	subw	d7,d1			/* center brush on point */

    4:	movw	d2,a2			/* put dx in proper place */
	movw	d3,a3			/* put dy in proper place */

	/* Test for shallow or steep */
	cmpw	a2,a3			/* compare x and y deltas */
	blt	hvect			/* branch if vector is shallow */

	/* Draw a steep vector */
vvect:	movw	a3,d7			/* get dy */
	asrw	#1,d7			/* half it */
	movw	d7,a4			/* initialize top accumulator */
	movw	d7,a5			/* and bottom accumulator too */
	movw	d0,d2			/* initialize left x */
	movw	d0,d3			/* initialize right x */
	cmpw	#0,a0			/* test horizontal increment */
	blt	1f			/* branch if negative */

	addw	d5,d3			/* left x on left if positive */
	bra	2f

    1:	subw	d5,d3			/* on right if negative */

    2:	movw	d1,d0			/* initialize y, keeping previous y */

	asrw	#1,d5			/* set left count to thickness over 2 */
	movw	a3,d6			/* initialize right count to dy */
	movw	d5,d7			/* overall count is the sum */
	addw	d6,d7			/* of the two */

    3:	addw	a1,d0			/* increment vertically */

	subqw	#1,d5			/* decrement left count */
	bge	5f			/* skip dda if not time yet */

	addw	a2,a4			/* bump left accumulator by dx */
	cmpw	a4,a3			/* has it exceeded dy yet? */
	bge	5f			/* branch if not */

	bsr	vblast			/* draw a rectangle */
	movw	d0,d1			/* remember previous y */

	addl	a0,d2			/* increment horizontally */
	subw	a3,a4			/* adjust accumulator */

    5:	subqw	#1,d6			/* decrement right count */
	blt	6f			/* skip dda if all finished */

	addw	a2,a5			/* bump right accumulator by dx */
	cmpw	a5,a3			/* has it exceeded dy yet? */
	bge	6f			/* branch if not */

	bsr	vblast			/* draw a rectangle */
	movw	d0,d1			/* remember previous y */

	addl	a0,d3			/* increment horizontally */
	subw	a3,a5			/* adjust accumulator */


    6:	dbf	d7,3b			/* draw another pixel */
	bsr	vblast			/* finish off vector */
	bra	drvr			/* return when finished */
	
	/* Draw a shallow vector */
hvect:	movw	a2,d7			/* get dx */
	asrw	#1,d7			/* half it */
	movw	d7,a4			/* initialize top accumulator */
	movw	d7,a5			/* and bottom accumulator too */
	movw	d1,d2			/* initialize top y */
	movw	d1,d3			/* initialize bottom y */
	cmpw	#0,a1			/* test vertical increment */
	blt	1f			/* branch if negative */

	addw	d5,d3			/* bottom y on bottom if positive */
	bra	2f

    1:	subw	d5,d3			/* bottom y on top if negative */

    2:	movw	d0,d1			/* remember previous x */

	asrw	#1,d5			/* set top count to thickness over 2 */
	movw	a2,d6			/* initialize bottom count to dx */
	movw	d5,d7			/* overall count is the sum */
	addw	d6,d7			/* of the two */

drv7r:	addw	a0,d0			/* increment horizontally */

drv8r:	subqw	#1,d5			/* decrement top count */
	bge	drv9r			/* skip dda if not time yet */

	addw	a3,a4			/* bump top accumulator by dy */
	cmpw	a4,a2			/* has it exceeded dx yet? */
	bge	drv9r			/* branch if not */

	bsr	hblast			/* draw a rectangle */
	movw	d0,d1			/* remember previous x */

	addl	a1,d2			/* increment vertically */
	subw	a2,a4			/* adjust accumulator */

drv9r:	subqw	#1,d6			/* decrement bottom count */
	blt	drv10r			/* skip dda if all finished */

	addw	a3,a5			/* bump bottom accumulator by dy */
	cmpw	a5,a2			/* has it exceeded dx yet? */
	bge	drv10r			/* branch if not */

	bsr	hblast			/* draw a rectangle */
	movw	d0,d1			/* remember previous x */

	addl	a1,d3			/* increment vertically */
	subw	a2,a5			/* adjust accumulator */

drv10r:	dbf	d7,drv7r		/* draw another pixel */

	bsr	hblast			/* finish off vector */

drvr:   moveml	sp@+,#<d2-d7,a2-a5>	/* restore working registers */
	unlk	a6			/* unlink frame */
	rts				/* and return */

hblast:	moveml	#<d0-d3,a0-a1>,sp@-		/* save registers */
	subw	d1,d0			/* compute width of region */
	beq	1f			/* forget it if nothing */

	subw	d2,d3			/* height is bottom y minus top y */
	exg	d0,d2			/* width in d2 */
	exg	d0,d1			/* y in d1, x in d0 */

	jsr	Adjust			/* make rectangle positive */
	jsr	RClip			/* clip to clipping bounds */
	ble	1f			/* return if nothing visible */

	movl	d3,sp@-			/* pass height */
	movl	d2,sp@-			/* pass width */
	movl	d1,sp@-			/* pass y */
	movl	d0,sp@-			/* pass x */
	movl	#_screen,sp@-		/* pass screen raster descriptor */
	jsr	_InvertRaster		/* invert a block */
	addl	#20,sp			/* clean parameters off stack */

    1:	moveml	sp@+,#<d0-d3,a0-a1>	/* restore registers */
	rts				/* return */

vblast:	moveml	#<d0-d3,a0-a1>,sp@-	/* save registers */

	subw	d1,d0			/* compute height of region */
	beq	1f			/* forget it if nothing */

	subw	d2,d3			/* width is right x minus left x */
	exg	d0,d3			/* height in d3 */
	exg	d0,d2			/* width in d2, x in d0 */

	jsr	Adjust			/* make rectangle positive */
	jsr	RClip			/* clip to clipping bounds */
	ble	1f			/* return if nothing visible */

	movl	d3,sp@-			/* pass height */
	movl	d2,sp@-			/* pass width */
	movl	d1,sp@-			/* pass y */
	movl	d0,sp@-			/* pass x */
	movl	#_screen,sp@-		/* pass screen raster descriptor */
	jsr	_InvertRaster		/* invert a block */
	addl	#20,sp			/* clean parameters off stack */
    1:	moveml	sp@+,#<d0-d3,a0-a1>	/* restore registers */
	rts				/* return */
