/*
 *	routines to manage a free space of storage for the virtual terminal
 *	manager.  This space is allocated by the kernel, pointed to by vtpool
 *	and has a size of VTSIZE.  These routines are used throughout the vt
 *	manager.
 */
#include <vt_hdrs.h>			/* sherwood defines */

struct header
{
    long	    size;
    struct header   *link;
};

#define HEADSZ	    sizeof(struct header)   /* size of header on each block */
#define SIZESZ	    sizeof(long)	    /* size of unit allocation      */

/*
 *	find size and end of vt free space pool for appropriate module
 */
extern char 		*vtpool;
#define POOLEND		((int)vtpool+VTSIZE)	/* compute size */
#define pool ((struct header *) (vtpool))

/*
 *	Set up vt free space pool for use
 */
InitializeBlock()
{
    pool->size = -1;		    /* mark first block empty */
    pool->link = pool+1;	    /* look at second block, size is whole */
    pool->link->size = (POOLEND-(int)pool) - HEADSZ - SIZESZ;  /*  pool size*/
    pool->link->link = NULL;	    /* next block does not exist */
}


/*
 *	determine if a block of the requested size exists, return 1 if it does
 */
HaveBlock(size)
register int size;
{
    int s;
    register struct header *this, *prev;

    s = spl6();	
    prev = pool;
    this = prev->link;
    while (this != NULL && this->size < size) {
	prev = this;
	this = this->link;
    }
    splx(s);
    return (this != NULL);
}

/*
 *	allocate a block of the requested size, return a pointer to it.
 */
char * CreateBlock(size)
register int size;
{
    int s, alloc;
    register struct header *this, *prev, *next;

    s = spl6();				/* don't bother me I'm working */
    prev = pool;			/* look at head of pool */
    this = prev->link;			/* look at first valid block */

    /* loop to find empty block of at least this big */
    while (this != NULL && this->size < size) {
	prev = this;
	this = this->link;
    }
    if (this == NULL)			/* panic if nothing is there */
	panic("out of virtual terminal space");

    /* link around this block if it is not big enough to split up */
    if (this->size-size < HEADSZ)
	prev->link = this->link;
    else {
	/* link space at end of block into free pool, next points to it */
	alloc = (SIZESZ+size+HEADSZ-1)/HEADSZ*HEADSZ; /* round size up */
	next = (struct header *)((int)this+alloc);
	next->size = this->size-alloc;	/* set size and link */
	next->link = this->link;
	prev->link = next;		/* put this in free list */
    }
    this->size = size;
    splx(s);
    return((char *)this+SIZESZ);	/* return pointer to it  */
}


/*
 *	Routine to reinsert a used block into the free pool for use again
 */
DestroyBlock(block)
char * block;
{
    register struct header *prev, *this, *next;
    int s;

    s = spl6();
    this = (struct header *)(block-SIZESZ);	/* point to this block */
    prev = pool;				/* point to head of pool */
    next = prev->link;

    /* loop to find first unused block less than this one */
    while (next != NULL && next < this) {
	prev = next;
	next = next->link;
    }

    /* save size of this block and reinsert it with links back and forward */
    this->size = (this->size+SIZESZ+HEADSZ-1)/HEADSZ*HEADSZ - SIZESZ;
    this->link = next;
    prev->link = this;

    /* concatinate with next block if also empty */
    if ((int)this+SIZESZ+this->size == (int)next) {
	this->size += next->size+SIZESZ;
	this->link = next->link;
    }

    /* concatinate with previous block if empty */
    if ((int)prev+SIZESZ+prev->size == (int)this) {
	prev->size += this->size+SIZESZ;
	prev->link = this->link;
    }
    splx(s);
}
