/* 
 * Virtual terminal driver for virtual terminals called directly from cdevsw.
 * The device argument to each of the calls is parsed to give the following 
 * information - physical terminal (for now this is hard coded to cons) and 
 * the window number (minor macro used to get this). These numbers are then 
 * used to write to and/or manipulate windows.
 */

#include "param.h"		/* system parameters */
#include "types.h"
#include "systm.h"		/* system sturcts et al */
#include "dir.h"		/* directories and files */
#include "user.h"		/* user block */
#include "file.h"		/* file structure */
#include "proc.h"		/* process strucutures */
#include "vnode.h"		/* inodes */
#include "conf.h"		/* configuration */
#include "buf.h"		/* buffers */
#include "dk.h"			/* disk stuff */
#include "uio.h"		/* user input/output struct */
#include "kernel.h"		/* need lbolt and hz */
#include "signal.h"		/* signal defines */
#include "errno.h"		/* error defines */
#include "../vt/vt_hdrs.h"	/* virtual terminal parameters */
#include "../vt/vt_fonts.h"	/* font defines and structures */
#include "../vt/vt_output.h"	/* vt output defines */
#include "../vt/vt_kernel.h"	/* kernel only vt defines */
#include "../vt/vt_key.h"	/* external virtual key names */
#include "../vt/vt_keyin.h"	/* internal virtual key names */
#include "../is68kdev/pdma.h"	/* pseudo dma defs */
#include "../is68kdev/sioreg.h"	/* serial I/O register defs */
#include "../is68kdev/gpreg.h"
#include "../machine/board.h"
#include "ttychars.h"

static short 	lastmousex, lastmousey;
char		flushvtok = 0;		/* pending request for gpdaemon */
char		flushmouseok = 0;	/* pending mouse/cursor request */
int		vtstart();

extern unsigned char	mousetab[];    /* Mouse multiplication factor */
extern short		minx, miny, maxx, maxy, mousex, mousey;
extern int		mousestate;
extern int		console_no;	/* major number of console */
extern short		consx, consy, consw, consh;
extern struct pdma 	sio_pdma[];
extern struct tty	sio_tty[];
extern int		sio_ispeed;
extern struct tty	vt_tty[];   	/* system wide virtual terms */
extern short		globallut[];	/* global lookup table status */
extern int 		localmask[], localface[],
			globalmask[], globalface[];
extern int 		*v_color[];	/* different colors for system */
extern struct itable	v_itable[];	/* system defined icons */
extern struct statline	v_status;	/* status line */
extern struct popquiz	popquiz;	/* popup menu */
extern struct trackcursor trackcursor;	/* cursor tracking box */
extern	int		mactive, kactive;
extern struct mark	vt_marks[];
extern unsigned char	Mdelta, mousetab[];
extern int		CopyURaster(), ToggleURaster(), PaintURaster();
extern short		vt_cnpoptop, vt_cnpoptopsave;

#ifdef	SYSV
int sv_vtproc();
#endif	SYSV

#define ISPEED 	B9600
#define IFLAGS	(EVENP|ODDP|ECHO|CRMOD|CRTBS|CRTERA|CRTKIL|CTLECH)

vtopen(dev, flag)
    register dev_t	dev;
{
    register struct tty *vp;
    register int	wnum, i;
    int	error;

    if (!(gptype & GPEXISTS))
	return(ENXIO);

    /* find window number and return if not valid */
    if ((wnum = minor(dev)) > NVT - 1)
	return(ENOVT);

    /* check to see if this vt is already open */
    vp = &vt_tty[wnum];
    if (flag&FEXCL && vp->t_state&(TS_WOPEN|TS_ISOPEN))
	return(EEXIST);
    vp->t_oproc = vtstart;
    vp->t_state |= TS_WOPEN;
#ifdef	SYSV
    map_state(vp, TOSV);
    vp->svt_proc = sv_vtproc;
#endif	SYSV

    if ((vp->t_state&TS_ISOPEN)==0) {
	kactive = wnum;
	ttychars(vp);
	vp->t_ispeed = vp->t_ospeed = ISPEED;
	vp->t_flags = IFLAGS;
#ifdef	SYSV
	sv_ttinit(vp);
#endif	SYSV

	/* if this is the user agent vt, then init screen et al */
	if (wnum == 0) {
	    for (i=1; i<=NVT; i++)
		if (vt_tty[i].t_state & TS_ISOPEN)
		    vtclose(i);

	    /* init screen, give desktop as window and set a mouse cursor */
	    gpget();
	    setupvt(vp);
	    SetCharacterAttributes(vp->v_win, CLEARBACK);	/* for icons */

	    /* set up default terminal emulator. WARNING: keyboard first! */
	    vp->v_keyb = (*keybsw[vp->v_te=deskem].k_init)(vp);
	    vp->v_term = (*termsw[vp->v_te].t_init)(vp);
	    gpgive();
	} else {
	    /* define regular default window */
	    gpget();
	    vp->v_win = (int *)CreateWindow(2000, 2000, 10, 10, 0, NORMAL);
	    SetWindowTitle(vp->v_win, "XYZZY"); /**/
	    gsignal(vt_tty[0].t_pgrp, SIGREF);	/* tell dm about it */
	    vtopen1(vp);
	    gpgive();
	} 
    } else if (vp->t_state&TS_XCLUDE && u.u_uid != 0)
	return (EBUSY);

    vp->t_state |= TS_CARR_ON;
#ifdef	SYSV
    map_state(vp, TOSV);
    if (u.u_procp->p_universe == UNIVERSE_SYSV)
    	error = (*sv_linesw[vp->svt_line].l_open)
		((wnum == 0) ? makedev(console_no,0) : dev, vp, flag);
    else
#endif	SYSV
    	error = (*linesw[vp->t_line].l_open)
		((wnum == 0) ? makedev(console_no,0) : dev, vp);
    if (error)
	return(error);
    if (vp->t_pgrp == 0 && u.u_procp)
	vp->t_pgrp = u.u_procp->p_pgrp;
    return(0);
}

vtopen1(vp)
    register struct tty	*vp;	/* pointer to a vt */
{
    /* set up default cursors */
    DefineGlobalCursor(vp->v_win, globalmask, globalface, 5, 3, 0);
    DefineLocalCursor(vp->v_win, localmask, localface, 8, 20, 0);  

    /* set up default fonts, colors and position */
    SetFont(vp->v_win, DEFWINDOWFONT);
    SetColor(vp->v_win, BLACK);
    SetBColor(vp->v_win, WHITE);
    SetPosition(vp->v_win, 0, 0); 

    /* set up default terminal emulator. WARNING: keyboard first! */
    vp->v_keyb = (*keybsw[vp->v_te=defterm].k_init)(vp);
    vp->v_term = (*termsw[vp->v_te].t_init)(vp);
}

vtclose(dev, flag)
    dev_t	dev;
{
    register struct tty	*vp;
    register int	wnum, i;

    /* find window number and return if not valid */
    if ((wnum = minor(dev)) > NVT)
	return(ENOVT);
    vp = &vt_tty[wnum];
    if ((vp->t_state & TS_ISOPEN) == 0)
	return(ENXIO);

    /* clean up if this window is in the middle of a mark */
    if (vt_marks[wnum].req&MARKING) {
	vt_marks[wnum].req = 0;
	vp->t_state &= ~TS_BUSY;
#ifdef	SYSV
	map_state(vp, TOSV);
#endif	SYSV
	gpget();
	(*termsw[vp->v_te].t_mark) (vp->v_term, MARK_CANCEL, 0, 0);
	gpgive();
    }
#ifdef	SYSV
    if (u.u_procp->p_universe == UNIVERSE_SYSV) {
	(*sv_linesw[vp->svt_line].l_close)(vp);
	sv_ttyclose(vp);
    } else {
#endif	SYSV
	(*linesw[vp->t_line].l_close)(vp);
	ttyclose(vp);
#ifdef	SYSV
    }
#endif	SYSV
    /*
     * if this is the desktop then clean up screen and die otherwise make 
     * sure desktop manager is active
     */
    gpget();
    if (wnum == 0)
	unsetup(vp);
    else { 
	/* get rid of any emulators */
	(*termsw[vp->v_te].t_exit)(vp->v_term);
	(*keybsw[vp->v_te].k_exit)(vp->v_keyb);

	/* remove window from screen */
	if (wnum == mactive)
	    mactive = 0; 
	if (wnum == kactive)
	    kactive = 0; 
	DestroyWindow(vp->v_win);
	vp->v_win = NULL;
	gsignal(vt_tty[0].t_pgrp, SIGREF);	/* tell dm about it */
    }
    gpgive();
    return(0);
}

vtread(dev, uio)
    dev_t	dev;
    struct uio	*uio;
{
    struct tty	*vp = &vt_tty[minor(dev)];
    int		error;

#ifdef	SYSV
    if (u.u_procp->p_universe == UNIVERSE_SYSV)
	error =  ((*sv_linesw[vp->svt_line].l_read)(vp, uio));
    else
#endif	SYSV
	error =  ((*linesw[vp->t_line].l_read)(vp, uio));
    return (error);
}

vtwrite(dev, uio)
    dev_t	dev;
    struct uio	*uio;
{
    struct tty	*vp = &vt_tty[minor(dev)];
    int		error;
    
#ifdef	SYSV
    if (u.u_procp->p_universe == UNIVERSE_SYSV)
	error =  ((*sv_linesw[vp->svt_line].l_write)(vp, uio));
    else
#endif	SYSV
	error =  ((*linesw[vp->t_line].l_write)(vp, uio));
    return (error);
}

vtioctl(dev, cmd, data, flag)
    register caddr_t	data;
{
    register struct tty *vp = &vt_tty[minor(dev)];
    register short	*shp = (short *)data;
    register int	error;
    
    /* enshure pending writes occur before the ioctl is executed */
    ttywait(vp);

#ifdef	SYSV
    /* for now, only allow 'vanilla' ioctls in the SYSV universe */
    if (u.u_procp->p_universe == UNIVERSE_SYSV) {
	error = sv_ttiocom(vp, cmd, data);
	return error;
    }
#endif	SYSV
    /* try to let the emulation deal with the ioctl */
    if (termsw[vp->v_te].t_ioctl != termsw_noop) {
	gpget();
	error=(*termsw[vp->v_te].t_ioctl)(vp->v_term,cmd,data,flag);
	gpgive();
	if (error >= 0)
	    return(error);
    }
    /* emulation independent ioctl */
    switch(cmd) {
	case TIOVMYGIP : 
		if (*shp) {
			gpget();
			gptype |= GPEXCLUSIVE;
			vt_cnpoptopsave = vt_cnpoptop;
			vt_cnpoptop = 0;
		} else {
			gptype &= ~GPEXCLUSIVE;
			vt_cnpoptop = vt_cnpoptopsave;
			gpgive();
		}
		break;

	case TIOVSETE: 
	    if ((*data < 0) || (*data >= nterms))
		return(ENOVT);
	    gpget();
	    (*termsw[vp->v_te].t_exit)(vp->v_term);
	    (*keybsw[vp->v_te].k_exit)(vp->v_keyb);
	    vp->v_keyb = (*keybsw[vp->v_te=(*data)].k_init)(vp);
	    vp->v_term = (*termsw[vp->v_te].t_init)(vp);
	    gpgive();
	    break;

	case TIOCSETD: 
	    if (vp->t_line == *(int*)data) 
		break;

	    /* soon this will be the old way to change emulators */
	    if ((vp->t_line != TWSDISC) && (*(int *) data == TWSDISC)) {
		gpget();
		(*termsw[vp->v_te].t_exit)(vp->v_term);
		(*keybsw[vp->v_te].k_exit)(vp->v_keyb);
		vp->v_keyb = (*keybsw[vp->v_te=1].k_init)(vp);
		vp->v_term = (*termsw[vp->v_te].t_init)(vp);
		gpgive();
	    } else if ((vp->t_line == TWSDISC) && (*(int *)data != TWSDISC)) {
		gpget();
		(*termsw[vp->v_te].t_exit)(vp->v_term);
		(*keybsw[vp->v_te].k_exit)(vp->v_keyb);
		vp->v_keyb = (*keybsw[vp->v_te=defterm].k_init)(vp);
		vp->v_term = (*termsw[vp->v_te].t_init)(vp);
		gpgive();
	    }
	    error = ttioctl(vp, cmd, data, flag);
	    return (error);

	/* Get some of the window state */
	case TIOVGSTATE: {
	    register struct wstate	*wp = (struct wstate *) data;
	    unsigned short		font;
	    COLORT			fcolor, bcolor;
	    PATTERN			fpat, bpat;
	    register int		i;
	    
	    gpget();
	    GetCurrentColors(vp->v_win, &fcolor, &bcolor);
	    GetCurrentPatterns(vp->v_win, &fpat, &bpat);
	    wp->fcolor = wp->bcolor = 0;
	    for (i = 0; i < NCOLOR; i++) {
		if (v_color[i] == fpat) 
		    wp->fcolor = -i;
		if (v_color[i] == bpat) 
		    wp->bcolor = -i;
	    }
	    if (wp->fcolor >= -1) 
		wp->fcolor = fcolor;
	    if (wp->bcolor >= -1) 
		wp->bcolor = bcolor;
	    GetCurrentFont(vp->v_win, &font);
	    wp->font = font;
	    GetCurrentPosition(vp->v_win, &(wp->xpos), &(wp->ypos));
	    GetCurrentAddressing(vp->v_win, &(wp->addressing));
	    GetStringAttributes(vp->v_win,&(wp->justification), 
		&(wp->orientation), &(wp->direction), &(wp->ctype));
	    GetCurrentPhase(vp->v_win, &(wp->xphase), &(wp->yphase));
	    GetLineAttributes(vp->v_win,&(wp->thickness),&(wp->mask),
		&(wp->count));
	    GetWindowSize(vp->v_win, &(wp->width), &(wp->height));
	    GetTemporaryClipping(vp->v_win, &(wp->tx), &(wp->ty), &(wp->tw), 
		&(wp->th));
	    GetPermanentClipping(vp->v_win, &(wp->px), &(wp->py), &(wp->pw), 
		&(wp->ph));
	    wp->mmode = (*keybsw[vp->v_te].k_ctl)(vp->v_keyb, KGMASK, 0);
	    wp->ldisc = vp->t_line;
	    wp->cvisible = GetCursorVisibility(vp->v_win);
	    gpgive();
	    break;
	}

	case TIOVGETHW: {
	    register struct gconfig	*cp = (struct gconfig *) data;

	    cp->w = SCREENW; 
	    cp->h = SCREENH;
	    cp->x_res = 90; 
	    cp->y_res = 90;
	    if (gptype & GPCOLOR) {
		cp->realcolors = LUTSIZE;
		cp->lutsize = LUTSIZE;
		cp->nred    = 16;
		cp->ngreen  = 16;
		cp->nblue   = 16;
	    } else {
		cp->realcolors = 2;
		cp->lutsize = 0;
		cp->nred    = 0;
		cp->ngreen  = 0;
		cp->nblue   = 0;
	    }
	    cp->nwindows = NVT;
	    cp->nistyles = NISTYLES;
	    cp->nicons   = NICON;
	    break;
	}

	/* Get Character Attributes */
	case TIOVGCHR: {
	    register struct charatt	*cp = (struct charatt *) data;
	    COLORT			fcolor;

	    gpget();
	    switch (cp->type) {
		case Cbaseline:
		    cp->val=CharacterBaseline(GetFontPlane(cp->font,DEFPLANE));
		    DoneWithPlane( cp->font, DEFPLANE);
		    break;

		case Cheight:
		    cp->val=CharacterHeight(GetFontPlane(cp->font, DEFPLANE));
		    DoneWithPlane( cp->font, DEFPLANE);
		    break;

		case Cwidth:
		    cp->val=CharacterWidth(GetFontPlane( cp->font, DEFPLANE), 
			 cp->val);
		    DoneWithPlane( cp->font, DEFPLANE);
		    break;

		case Ctype:
		    GetStringAttributes(vp->v_win, &fcolor, &fcolor, &fcolor,
			&cp->val); 
		    break;
	    }
	    gpgive();
	    break;
	}

	/* Put a title on the window */
	case TIOVSETT:
	    gpget();
	    SetWindowTitle(vp->v_win,data);	    /* set title */
	    gpgive();
	    break;

	/* Change the emulator font */
	case TIOVSETEF: {
	    register short		*sp = (short *) data;

	    gpget();
	    (*termsw[vp->v_te].t_font)(vp->v_term, *sp);
	    vpsetwinsize(vp);
	    gpgive();
	    break;
	}

	/* Display a pop-up menu, and return the selected item */
	case TIOVPOPQ: 
	    gpget();
	    *data = DisplayPopQuiz(vp, data);
	    gpgive();
	    break;
	    
	case TIOVTRCKM: {
	    register struct trackmouse	*t = (struct trackmouse *) data;
	    short	x, y;
	    short	min_x, min_y, max_x, max_y;
	    int		s;

	    GetWindowPosition( vp->v_win, &x, &y);
	    min_x = minx; 
	    min_y = miny;
	    max_x = maxx; 
	    max_y = maxy;
	    s = spltty();
	    while (trackcursor.type)	/* somebody else is tracking cursor */
		sleep((caddr_t)&trackcursor.type, PZERO-1);
	    trackcursor.type = t->type;
	    trackcursor.x = x + t->x;
	    trackcursor.y = y + t->y;
	    trackcursor.w = t->w;
	    trackcursor.h = t->h;
	    trackcursor.thickness = t->thickness;

	    switch(t->type) {
		case TrackFixed:
		    trackcursor.xoffset = (x + t->x) - mousex;
		    trackcursor.yoffset = (y + t->y) - mousey;
		    minx = (x + t->bx) - trackcursor.xoffset;
		    miny = (y + t->by) - trackcursor.yoffset;
		    maxx = (x + t->bx + t->bw) - t->w - trackcursor.xoffset;
		    maxy = (y + t->by + t->bh) - t->h - trackcursor.yoffset;
		    if ((mousestate&7) != 7)
			sleep((caddr_t)&trackcursor.thickness, PZERO-8);
		    t->x = trackcursor.x - x;
		    t->y = trackcursor.y - y;
		    break;

		case TrackRubber:
		    trackcursor.xoffset = (x + t->x + t->w) - mousex;
		    trackcursor.yoffset = (y + t->y + t->h) - mousey;
		    minx = (x + t->bx) - trackcursor.xoffset;
		    miny = (y + t->by) - trackcursor.yoffset;
		    maxx = (x + t->bx + t->bw) - trackcursor.xoffset;
		    maxy = (y + t->by + t->bh) - trackcursor.yoffset;
		    if ((mousestate&7) != 7)
			sleep((caddr_t)&trackcursor.thickness, PZERO-8);
		    t->w = trackcursor.w;
		    t->h = trackcursor.h;
		    break;
	    }
	    splx(s);
	    trackcursor.type = 0;
	    gpget();
	    RemoveGlobalRubberBox();
	    minx = min_x; miny = min_y;
	    maxx = max_x; maxy = max_y;
	    wakeup((caddr_t)&trackcursor.type);
	    gpgive();
	    break;
	}
	
	/* Define Cursors */
	case TIOVDEFC: {
	    register struct curpat	*cp = (struct curpat *) data;
	    int				mask[32], face[32];

	    if ( copyin(cp->face, face, 128) || copyin(cp->mask, mask, 128))
		return(EFAULT);
	    gpget();
	    if (cp->type == LocalCur)
		DefineLocalCursor(vp->v_win, mask, face, 
				    cp->xref, cp->yref, cp->blink);
	    else if (cp->type == GlobalCur)
		DefineGlobalCursor(vp->v_win, mask, face, 
				    cp->xref, cp->yref, cp->blink); 
	    gpgive();
	    break;
	}

	case TIOVRASTOP: {
	    register struct rasterop	*rp = (struct rasterop *) data;
	    RASTER 			uraster;
	    register char		*first, *last;
	    register int		count;

	    first = (char *)((char *)rp->ras +
		(rp->sy*rp->width)		+ ((rp->sx+7)>>3));
	    last = (short *)((char *)rp->ras +
		((rp->sy + rp->h - 1)*rp->width)+ ((rp->sx+rp->w+7)>>3));
	    count = last - first + 1;

	    if (!useracc(first, count, B_READ))
		return(EFAULT);

	    /* Cannot tollerate page fault while owner of GIP */
	    u.u_procp->p_flag |= SPHYSIO;
	    vslock(first, count);
	    uraster.address = rp->ras;
	    uraster.width = rp->width;
	    gpget();

	    switch( rp->type) {
		case COPYRASTER:
		    DoRasterOp(vp->v_win, CopyURaster, &uraster, 
			rp->sx, rp->sy, rp->x, rp->y, rp->w, rp->h, rp->color);
		    break;

		case TOGGLERASTER:
		    DoRasterOp(vp->v_win, ToggleURaster, &uraster, 
			rp->sx, rp->sy, rp->x, rp->y, rp->w, rp->h, rp->color);
		    break;

		case PAINTRASTER:
		    DoRasterOp(vp->v_win, PaintURaster, &uraster, 
			rp->sx, rp->sy, rp->x, rp->y, rp->w, rp->h, rp->color);
		    break;
	    }
	    gpgive();
	    vsunlock(first, count, B_WRITE);	/* device write, memory read */
	    u.u_procp->p_flag &= ~SPHYSIO;
	    break;
	}

	/* Set the console open size/position */
	case TIOUSETCP:
	    consx = *shp++;
	    consy = *shp++;
	    consw = *shp++;
	    consh = *shp;
	    break;

	/* Display/delete status line */
	case TIOUSSTAT: {
	    register unsigned char	*sp;
	    register unsigned char	*sp1;
	    register int		i;

	    sp = (unsigned char *) shp;	/* point to characters and copy in */
	    sp1 = v_status.message;
	    for (i = 0; i < 128; i++)
		*sp1++ = *sp++;
	    v_status.visible = 1;
	    gpget();
	    DisplayStatusLine();
	    gpgive();
	    break;
	}

	case TIOUDSTAT:
	    gpget();
	    EraseStatusLine();
	    gpgive();
	    break;
	
	/* Turn on/off rubber baby band boxes */
	case TIOUMVBOX:
	    if (((struct clipbnd *) data)->type == BabyBox) {
		gpget();
		HideGlobalCursor();		/* hide cursor for ua */
		DisplayGlobalRubberBox(((struct clipbnd *) data)->x,
				   ((struct clipbnd *) data)->y,
				   ((struct clipbnd *) data)->w,
				   ((struct clipbnd *) data)->h,2);
		gpgive();
	    }
	    break;

	case TIOUNOBOX:
	    gpget();
	    RemoveGlobalRubberBox();	/* kill baby box of rubber */
	    ShowGlobalCursor();
	    gpgive();
	    break;

	/* Get the internal vt number for a window */
	case TIOUGETWN:
	    *(short *)data = vp - vt_tty;
	    break;

	/* Get the pgrp of a window */
	case TIOUWPGRP:{
	    register struct uwindow	*wp = (struct uwindow *) data;

	    /* return -1 if trying to get bad window pgrp */
	    wp->val1 = -1;
	    if ((wp->win < 0) || (wp->win > NVT))
		break;
	    if (wp->win == NVT) {	/* is this how we say its the console */
		wp->val1 = -2;
		break;
	    }
	    if (vt_tty[wp->win].t_state & TS_ISOPEN) 
		wp->val1 = vt_tty[wp->win].t_pgrp;
	    break;
	}

	/* Set Pane Colors */
	case TIOUSETPC:
	    /* return if trying to change bad window panes */
	    if ((*shp < 0) || (*shp > NVT)) 
		break;
	    vp = &vt_tty[*shp++];
	    if ((vp->t_state & TS_ISOPEN) == 0) 
		break;
	    gpget();
	    SetPaneColors(vp->v_win, *shp, *(shp +1), *(shp + 2));
	    gpgive();
	    break;

	/* Set a new active vt */
	case TIOUNEWVT:
	    /* return if trying to change bad window panes */
	    if ((*shp < 0) || (*shp > NVT)) 
		break;
	    kactive = *shp;
	    mactive = *shp;
	    break;

	/* Set mouse clipping boundries */
	case TIOUMBND:
	    /* just copy the little buggers in */
	    minx = *shp++;
	    maxx = *shp++;
	    miny = *shp++;
	    maxy = *shp;

	    /* clip the mouse if needed */
	    if (mousex < minx) 
		mousex = minx;
	    if (mousey < miny) 
		mousey = miny;
	    if (mousex > maxx) 
		mousex = maxx;
	    if (mousey > maxy) 
		mousey = maxy;
	    break;

	/* Get/Set a windows size */
        case TIOUGETWS: {
	    register struct uwindow	*wp = (struct uwindow *) data;

	    /* return if trying to change bad window panes */
	    if ((wp->win < 0) || (wp->win > NVT)) 
		break;
	    vp = &vt_tty[wp->win];
	    if ((vp->t_state & TS_ISOPEN) == 0)
		return(ESRCH);

	    /* no quiz window */
	    gpget();
	    GetWindowSize(vp->v_win, &(wp->val1), &(wp->val2));
	    gpgive();
	    break;
	}

	case TIOUSETWS:  {
	    register struct uwindow	*wp = (struct uwindow *) data;

	    /* return if trying to change bad window panes */
	    if ((wp->win < 0) || (wp->win > NVT)) 
		break;
	    vp = &vt_tty[wp->win];
	    if ((vp->t_state & TS_ISOPEN) == 0) 
		break;
	    gpget();
	    ChangeWindowSize(vp->v_win, wp->val1, wp->val2);
	    vpsetwinsize(vp);
	    gpgive();
	    break;
	}
	
	/* Get/set a windows position */
	case TIOUGETWP: {
	    register struct uwindow	*wp = (struct uwindow *) data;

	    /* return if trying to change bad window panes */
	    if ((wp->win < 0) || (wp->win > NVT)) 
		break;
	    vp = &vt_tty[wp->win];
	    if ((vp->t_state & TS_ISOPEN) == 0)
		return(ESRCH);

	    gpget();
	    GetWindowPosition(vp->v_win, &(wp->val1), &(wp->val2));
	    gpgive();
	    break;
	}

	case TIOUSETWP: {
	    register struct uwindow	*wp = (struct uwindow *) data;

	    /* return if trying to change bad window panes */
	    if ((wp->win < 0) || (wp->win > NVT)) 
		break;
	    vp = &vt_tty[wp->win];
	    if ((vp->t_state & TS_ISOPEN) == 0) 
		break;
	    gpget();
	    ChangeWindowPosition(vp->v_win, wp->val1, wp->val2);
	    gpgive();
	    break;
	}

	/*  Find the top window */
	case TIOUTOPW: {
	    register	w;

	    gpget();
	    for (w=0; w<=NVT; w++)
		if (vt_tty[w].t_state & TS_ISOPEN) {
		    GetWindowDepth(vt_tty[w].v_win, shp); /* check its depth */
		    if (!*shp) 
			break;
		}
	    gpgive();
	    *shp = w;
	    break;
	}

	/* Get/set a windows depth */
	case TIOUGETWD: {
	    register struct uwindow	*wp = (struct uwindow *) data;

	    /* return -1 if trying to change bad window panes */
	    wp->val1 = -1;
	    if ((wp->win < 0) || (wp->win > NVT)) 
		break;
	    vp = &vt_tty[wp->win];
	    if ((vp->t_state & TS_ISOPEN) == 0)
		return(ESRCH);
	    gpget();
	    GetWindowDepth(vp->v_win, &(wp->val1));
	    gpgive();
	    break;
	}

	case TIOUSETWD: {
	    register struct uwindow	*wp = (struct uwindow *) data;

	    /* return if trying to change bad window panes */
	    if ((wp->win < 0) || (wp->win > NVT)) 
		break;
	    vp = &vt_tty[wp->win];
	    if ((vp->t_state & TS_ISOPEN) == 0) 
		break;
	    gpget();
	    ChangeWindowDepth(vp->v_win,wp->val1);
	    gpgive();
	    gsignal(vt_tty[0].t_pgrp, SIGREF);
	    break;
	}
    
	/* Configuration ioctl's */
	case TIOUGMC:	{
	    register struct mconfig	*cp = (struct mconfig *) data;
	    register int		i;
	
	    cp->mc_delta = Mdelta;
	    for ( i=0; i<=VT_MTABSIZE; i++)
		cp->mc_table[i] = mousetab[i];
	    break;
	}

	case TIOUSMC:	{
	    register struct mconfig	*cp = (struct mconfig *) data;
	    register int		i;

	    Mdelta = cp->mc_delta;
	    for ( i=0; i<=VT_MTABSIZE; i++)
		mousetab[i] = cp->mc_table[i];
	    break;
	}

	case TIOUSGLUT: {
	    register struct colorlut	*cp = (struct colorlut *) data;

	    if (!(gptype & GPCOLOR))
		return(ENXIO);
	    gpget();
	    LoadGlobalLUT(cp->color, cp->red, cp->green, cp->blue);
	    gpgive();
	    break;
	}

	case TIOUGGLUT: {
	    register struct colorlut	*cp = (struct colorlut *) data;
	    short			r, g, b;

	    if (!(gptype & GPCOLOR))
		return(ENXIO);
	    gpget();
	    GetGlobalLUT(cp->color, &r, &g, &b);
	    gpgive();
	    cp->red = r;
	    cp->green = g;
	    cp->blue = b;
	    break;
	}

	/* close console window */
	case TIOUCCON:
	    if (vt_tty[NVT].t_state & TS_ISOPEN)
		return(vtclose(NVT, 0));
	    else
		return(ESRCH);

	case TIOUSETWF: {
	    register struct uwindow	*wp = (struct uwindow *) data;

	    if ((wp->win < 0) || (wp->win > NVT)) 
		break;
	    vp = &vt_tty[wp->win];
	    gpget();
	    (*termsw[vp->v_te].t_font)(vp->v_term, wp->val1);
	    vpsetwinsize(vp);
	    gpgive();
	    break;
	}
	
	default:
	    if (((error=(*linesw[vp->t_line].l_ioctl)(vp,cmd,data,flag))>=0) ||
	        ((error=ttioctl(vp, cmd, data, flag))>=0) )
		return(error);
	    return(ENOTTY);
    }
    return(0);
}


/*
 * This routine is called when the user agent vt is opened for the first time.
 * It sets up and initializes all the little things needed for the screen 
 */
setupvt(tp)
    register struct tty	*tp;
{				
    register int    	i, j;
    WINDOW	    	DesktopWindow();
    register struct tty	*stp;

    /* initialize icon table */
    for (i = 0; i < NICON; i++) 
	v_itable[i].iaddr = 0;

    /* initialize the global lookup table */
    if (gptype & GPCOLOR) {
	globallut[0] = 1;		/* claim black */
	LoadGlobalLUT(0, 0, 0, 0);
	globallut[1] = 1;		/* and white */
	LoadGlobalLUT(1, 15, 15, 15);
	globallut[2] = 1;		/* also gray50 */
	LoadGlobalLUT(2, 11, 11, 11);
	for (i=3; i<LUTSIZE; i++)
	    globallut[i] = VT_FREE;
    }

    /* give desktop as tty window, clear background & set up refresh */
    tp->v_win = (int *) DesktopWindow();
    v_status.visible = FALSE;
    consx = (SCREENW - CONSW)>>1;
    consy = (SCREENH - CONSH)>>1;
    consw = CONSW;
    consh = CONSH;

    /* set up the mouse and keyboard to work with the desktop. KLUDGE */
    sioinit();
    for ( i=0; i<NSIO; i++) {
	stp = &sio_tty[i];
	stp->t_addr = (caddr_t)&sio_pdma[i];
	if ((tp->t_state&TS_ISOPEN) == 0) {
	    stp->t_state |= TS_ISOPEN;
	    stp->t_flags = (EVENP|ODDP|ECHO|XTABS|CRMOD);
#ifdef	SYSV
	    map_state(stp, TOSV);
#endif	SYSV
	} 

	if (i==0) {			/* keyboard */
	    stp->t_ospeed = stp->t_ispeed = sio_ispeed;
	    stp->t_line = KYBDDISC;
#ifdef	SYSV
	    stp->svt_line = SV_KYBDDISC;
#endif	SYSV
	} else {			/* mouse */
	    stp->t_ospeed = stp->t_ispeed = B1200;
	    stp->t_line = MOUSDISC;
#ifdef	SYSV
	    stp->svt_line = SV_MOUSDISC;
#endif	SYSV
	}
#ifdef	SYSV
	map_flags(stp, TOSV);
#endif	SYSV
	sioparam(i);
    }

    /* show cursor, set up mouse bounds and keyboard state */
    minx = miny = 0;
    maxx = SCREENW;
    maxy = SCREENH;
    DefineGlobalCursor(tp->v_win, globalmask, globalface, 5, 3, 0);
    MoveGlobalCursor(mousex, mousey);
    ShowGlobalCursor();
    SetActiveCursor(tp->v_win);
}

/*
 * Routine used when the last vt (user agent vt ) is closed to shutdown screen 
 */
unsetup(tp)
    register struct tty	*tp;
{
    register int	i;

    /* zero out icon table */
    for (i = 0; i < NICON; i++) 
        v_itable[i].iaddr = 0;

    /* clear the whole screen */
    if (gptype & GPCOLOR)
    	SetColor(tp->v_win, GRAY50);
    else
    	SetPattern(tp->v_win, gray50);

    SetPosition(tp->v_win, 0, 0);
    PaintRectangleInterior(tp->v_win, SCREENW, SCREENH);
    if (!(gptype & GPCOLOR))
    	SetPattern(tp->v_win, solid);
    consx = (SCREENW - CONSW)>>1;
    consy = (SCREENH - CONSH)>>1;
    consw = CONSW;
    consh = CONSH;
}

flushvtecho()
{
    register struct tty	*vp;
    register int 	i, more;
    register int 	flushed = 0;

    /* do the mouse first */
    if (flushmouseok)
    	updatemouse(1);

    /* let terminal emulator know its ok to buffer windows who need service */
    for (i = 0, vp = &vt_tty[0]; i <= NVT; i++, vp++)
	if ( (vp->t_state & TS_ISOPEN) && !(vp->v_state&VT_STARTED) &&
	    !(vp->t_state&(TS_TIMEOUT|TS_BUSY|TS_TTSTOP)) && vp->t_outq.c_cc) {
	    	vp->v_state |= VT_STARTED;
		(*termsw[vp->v_te].t_bufon)(vp->v_term);
		flushed |= (1 << i);
	    }

    /* display one characher per window till all of outqs sent to screen */
    do {
        more = 0;
        for (i = 0, vp = &vt_tty[0]; i <= NVT; i++, vp++)
	    if ((flushed&(1<<i)) && !(vp->t_state&TS_TTSTOP) && 
		vp->t_outq.c_cc) {
    	            more = 1;
	    	    (*termsw[vp->v_te].t_disp)(vp->v_term, getc(&vp->t_outq));
	    }
        if (flushmouseok)
            updatemouse(1);
#ifdef	M68020
        if (runrun)
	    break;
#endif	M68020
    } while (more);

    for (i = 0, vp = &vt_tty[0]; i <= NVT; i++, vp++)
	if ( flushed & (1 << i)) {
	    /* tell the emulator to stop buffering */
	    (*termsw[vp->v_te].t_bufoff)(vp->v_term);
	    vp->v_state &= ~VT_STARTED;

            /* wakeup any sleepers */
            if (vp->t_outq.c_cc <= TTLOWAT(vp)) {
	        if (vp->t_state&TS_ASLEEP) {
		    vp->t_state &= ~TS_ASLEEP;
#ifdef	SYSV
		    map_state(vp, TOSV);
#endif	SYSV
		    wakeup((caddr_t)&vp->t_outq);
	        }
	        if (vp->t_wsel) {
		    selwakeup(vp->t_wsel, vp->t_state & TS_WCOLL);
		    vp->t_wsel = 0;
		    vp->t_state &= ~TS_WCOLL;
	        }
            }
	}

    if (flushmouseok)
    	updatemouse(1);

    if (more) {
	gpgive();
	i = splclock();
	u.u_procp->p_pri = u.u_procp->p_usrpri;
	setrq(u.u_procp);
	swtch();
	splx(i);
	curpri = u.u_procp->p_pri;
	flushvtok = 1;
	gpget();
    }
}

/* 
 * update the mouse position
 */
updatemouse(flag)
    int	flag;			/* update popups or trackboxes too! */
{
    short	absx;
    short	absy;
    short	new;
    int		relx, rely;

    flushmouseok = 0;

    if (lastmousex != mousex || lastmousey != mousey) {
	absx = mousex;
	absy = mousey;
	if (flag && ((int)popquiz.win > 1)) {
	    /* find the new choice */
	    if (WithinPane(popquiz.win,DISPLAY,absx,absy)) { 
		MapCoordinates(popquiz.win,DISPLAY,absx,absy,&relx,&rely);
		new = (rely / popquiz.cheight) + 1;
	    } else 
		new = 0;

	    if ((new != popquiz.current)) {
		if (popquiz.current) { /* unhighlight current */
		    InvertRegion(popquiz.win,0,
			popquiz.cheight*(popquiz.current-1), 1000, 
			popquiz.cheight);
		}
		if (new) { /* hightlight new current */
		    InvertRegion(popquiz.win,0,popquiz.cheight*(new-1),
				 2000, popquiz.cheight);
		}
		popquiz.current = new;
	    }
	} else if (flag && trackcursor.type) {
	    switch(trackcursor.type) {
		case 1 /* TrackFixed */:
		    DisplayGlobalRubberBox(
			trackcursor.x = mousex + trackcursor.xoffset,
			trackcursor.y = mousey + trackcursor.yoffset,
			trackcursor.w, trackcursor.h, trackcursor.thickness);
		    break;
		case 2 /* TrackRubber */:
		    DisplayGlobalRubberBox( trackcursor.x, trackcursor.y,
			trackcursor.w = (mousex + trackcursor.xoffset)
					- trackcursor.x,
			trackcursor.h = (mousey + trackcursor.yoffset)
					- trackcursor.y, trackcursor.thickness);
		    break;
	    }
	}
	MoveGlobalCursor(mousex, mousey);
	lastmousex = mousex; lastmousey = mousey;
    }
}

/*
 * Routine used to start output from a virtual terminal to the screen. 
 * this routine will tell the correct terminal emulator when to buffer its 
 * output.  It will give up the processor if the output takes too long.
 */
vtstart(vp)
    register struct tty	*vp;
{
    register unsigned char	c;

    /* if at interrupt priority return after waking up the type-ahead daemon */
    if ((spl()>>8) & 7) {
	flushvtok = 1;
	wakeup((caddr_t)&flushvtok);
	return;
    }

    /* check to see that we aren't already running */	
    gpget();
    if (vp->t_state&(TS_TIMEOUT|TS_BUSY|TS_TTSTOP) || (vp->v_state&VT_STARTED)) {
	gpgive();
	return;
    }
    vp->v_state |= VT_STARTED;

    if (vp->t_outq.c_cc == 0)
	goto out;

    /* let terminal emulator know its ok to buffer */
    (*termsw[vp->v_te].t_bufon)(vp->v_term);

    while (vp->t_outq.c_cc > 0) {	/* display all of outq to screen    */
	/* quit if output has become stopped */
    	if (vp->t_state & TS_TTSTOP) {
	    /* tell the emulator to stop buffering */
	    (*termsw[vp->v_te].t_bufoff)(vp->v_term);
	    vp->v_state &= ~VT_STARTED;
	    gpgive();
	    return;
	}

    	if (flushmouseok)
	    updatemouse(1);
#ifdef	M68020
	/* give-up the processor */
	if (runrun) {
	    flushvtok = 1;
	    wakeup((caddr_t)&flushvtok);
	    break;
	}
#endif	M68020
	(*termsw[vp->v_te].t_disp)(vp->v_term, getc(&vp->t_outq));
    }

    (*termsw[vp->v_te].t_bufoff)(vp->v_term);
    gpgive();

out:
    /* wakeup any sleepers */
    if (vp->t_outq.c_cc <= TTLOWAT(vp)) {
	if (vp->t_state&TS_ASLEEP) {
	    vp->t_state &= ~TS_ASLEEP;
#ifdef	SYSV
	    map_state(vp, TOSV);
#endif	SYSV
	    wakeup((caddr_t)&vp->t_outq);
	}
	if (vp->t_wsel) {
	    selwakeup(vp->t_wsel, vp->t_state & TS_WCOLL);
	    vp->t_wsel = 0;
	    vp->t_state &= ~TS_WCOLL;
	}
    }
    vp->v_state &= ~VT_STARTED;
}

#ifdef	SYSV
/*
 *	device specific output routine for sysv TTY code
 */
sv_vtproc(vp, cmd)
register struct tty *vp;
register cmd;
{
	sv_proc(vp, cmd, vtstart, NULL);
}
#endif	SYSV

#ifdef	NOTDEF
/*
 *	device specific output routine for sysv TTY code
 */
sv_vtproc(vp, cmd)
register struct tty *vp;
register cmd;
{
	int s;
	extern sv_ttrstrt();

	s = spltty();
	switch (cmd) {
	    case T_TIME:
		vp->svt_state &= ~TIMEOUT;
		goto start;

	    case T_WFLUSH:
		break;

	    case T_RESUME:
		vp->svt_state &= ~TTSTOP;
		goto start;

	    case T_OUTPUT:
start:
		map_state(vp, TO43);
		if ((vp->svt_state & (TIMEOUT|TTSTOP|BUSY)) == 0)
			vtstart(vp);
		break;

	    case T_SUSPEND:
		vp->svt_state |= TTSTOP;
		map_state(vp, TO43);
		break;

	    case T_BLOCK:
		vp->svt_state &= ~TTXON;
		vp->svt_state |= TBLOCK;

		if (vp->svt_state & BUSY) 
			vp->svt_state |= TTXOFF;
		else {
			vp->svt_state |= BUSY;
		}
		map_state(vp, TO43);
		break;

	    case T_RFLUSH:
		if (!(vp->svt_state & TBLOCK))
			break;

	    case T_UNBLOCK:
		vp->svt_state &= ~(TTXOFF | TBLOCK);
		if (vp->svt_state & BUSY)
			vp->svt_state |= TTXON;
		else  {
			vp->svt_state |= BUSY;
		}
		goto start;
		break;

	    case T_BREAK:
		vp->svt_state |= TIMEOUT;
					/* should send BRK character here */
		map_state(vp, TO43);
		timeout(sv_ttrstrt, vp, hz/4);
		break;

	    case T_PARM:
				/* ??? */
		break ;
	}
	splx(s);
}
#endif	NOTDEF

vpsetwinsize(vp)
    register struct tty	*vp;
{
    unsigned short		font;
    struct font_plane		*charinfo;
    register int		i;

    GetWindowSize(vp->v_win, &vp->t_winsize.ws_xpixel, &vp->t_winsize.ws_ypixel);
    vp->t_winsize.ws_row = 0;
    vp->t_winsize.ws_col = 0;
    GetCurrentFont( vp->v_win, &font );
    charinfo = GetFontPlane( font, DEFPLANE );
    if ( i = CharacterWidth( charinfo, 'W'))
	vp->t_winsize.ws_col = vp->t_winsize.ws_xpixel / i;
    if ( i = CharacterHeight( charinfo))
	vp->t_winsize.ws_row = vp->t_winsize.ws_ypixel / i;
    DoneWithPlane( font, DEFPLANE );
}
