#include "param.h"
#include "user.h"
#include "types.h"
#include "errno.h"
#include "proc.h"
#include "file.h"
#include "uio.h"
#include "../wipc/wipc.h"
#include "../trfs/trfs.h"

/*
 * Operation specific routines to encode request messages on the client 
 * side ("Sd"), and form replys on the server/cousin side ("Sv")
 */

/* Macro to allocate and initialize request message headers */
#define	TRFS_SD(f, F, RFLAGS, XS0, XS1)		\
	union {					\
		struct f	msg;		\
		struct wmsg	pad;		\
	} m;					\
	m.msg.h.rcode = F;			\
	m.msg.h.rflags = RFLAGS;		\
	m.msg.h.xs0 = XS0;			\
	m.msg.h.xs1 = XS1

/* ------- system calls ------ */

SdExit(rpid, rval)
	unsigned long rpid;
{
	TRFS_SD(rq_exit, RQ_EXIT, 0, 0, 0);
	return TrfsSend(&m, rpid);
}

SvExit(mp, rpid)
	struct rq_exit *mp;
{
	ResetCousinRpid();
}

SdFork(rpid)
	unsigned long rpid;
{
	int err;

	TRFS_SD(rq_fork, RQ_FORK, RF_NONIDEM, 0, 0);
	if (err = TrfsSend(&m, rpid))
		return -err;
	return (int)(MID(rpid) | PID(m.msg.rpid));
}

SvFork(mp, rpid)
	struct rq_fork *mp;
{
	return mp->rpid = (unsigned long)CoFork();
}

SdChexec(rpid, path, comlen, exdata, exdatalen, uidp, gidp)
	unsigned long rpid;
	char *path, *exdata;
	uid_t *uidp;
	gid_t *gidp;
{
	int err;

	TRFS_SD(rq_chexec, RQ_CHEXEC, RF_NONIDEM | RF_CRCOUSIN,
		setpath(&m.msg.path, path), XS_KERN | XS_WRITE);
	m.msg.path.len = MAX(m.msg.path.len, comlen);
	m.msg.exbuf.addr = exdata;
	m.msg.exbuf.len = exdatalen;
	setuser(&m.msg.id, rpid);
	m.msg.newimage = 1;
	if ((err = TrfsSend(&m, rpid)) == 0) {
		if (uidp)
			*uidp = m.msg.id.uid;
		if (gidp)
			*gidp = m.msg.id.gid;
	}
	return err;
}

SvChexec(mp, rpid)
	register struct rq_chexec *mp;
{
	uid_t uid = mp->id.uid;
	gid_t gid = mp->id.gid;

	getuser(&mp->id);
	mp->h.rcode = CoChexec(mp->path.addr, mp->exbuf.addr, mp->exbuf.len,
		&uid, &gid);
	mp->id.uid = uid;
	mp->id.gid = gid;
	mp->newimage = 1;			/* compatibility */
}

SdLoad(rpid, addr, len, offset)
	unsigned long rpid;
	char *addr;
{
	TRFS_SD(rq_load, RQ_LOAD, RF_NONIDEM, XS_USER | XS_WRITE, 0);
	m.msg.buf.addr = addr;
	m.msg.buf.len = len;
	m.msg.offset = offset;
	return TrfsSend(&m, rpid);
}

SvLoad(mp, rpid)
	register struct rq_load *mp;
{
	int segflg = UIO_USERSPACE;

	if ((mp->h.xs0 & XS_SEGMASK) == XS_KERN) {
		setdstxs(XS_KERN | XS_WRITE);
		segflg = UIO_SYSSPACE;
	}
	xReplyOnCopy(mp, rpid, mp->buf.addr + mp->buf.len - 1, mp->h.xs0);
	mp->h.rcode = CoLoad(mp->buf.addr, mp->buf.len, mp->offset, segflg);
}

/* Returns nonnegative file descriptor, or negative error number. */
SdOpen(rpid, path, mode, cmode, devp, inump, modep, rdevp)
	unsigned long rpid;
	char *path;
	unsigned long *devp, *inump, *modep, *rdevp;
{
	int err;

	TRFS_SD(rq_open, RQ_OPEN, RF_NONIDEM | RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 0);
	setuser(&m.msg.rq_u.id, rpid);
	m.msg.accessmode = mode;
	m.msg.createmode = cmode;
	if (err = TrfsSend(&m, rpid))
		return -err;
	if (devp)
		*devp = m.msg.rq_u.is.dev;
	if (inump)
		*inump = m.msg.rq_u.is.inum;
	if (modep)
		*modep = m.msg.rq_u.is.mode;
	if (rdevp)
		*rdevp = m.msg.rq_u.is.rdev;
	return m.msg.rfd;
}

SvOpen(mp, rpid)
	register struct rq_open *mp;
{
	unsigned long dev = 0, inum = 0, mode = 0, rdev = 0;

	getuser(&mp->rq_u.id);
	mp->rfd = CoCopen(mp->path.addr, mp->accessmode, mp->createmode,
			&dev, &inum, &mode, &rdev);
	mp->rq_u.is.dev = dev;
	mp->rq_u.is.inum = inum;
	mp->rq_u.is.mode = mode;
	mp->rq_u.is.rdev = rdev;
}

int
SdStat(rpid, path, follow, rfd, buf, bufsize, segflg)
	unsigned long rpid;
	char *path, *buf;
{
	TRFS_SD(rq_stat, RQ_STAT, (rfd < 0) ? RF_CRCOUSIN : 0, 
		(rfd < 0) ? setpath(&m.msg.path, path) : 0,
		(segflg == UIO_SYSSPACE ? XS_KERN : XS_USER) | XS_WRITE);
	if (rfd < 0)
		setuser(&m.msg.id, rpid);
	else
		bzero(&m.msg.id, sizeof m.msg.id);
	m.msg.buf.addr = buf;
	m.msg.buf.len = bufsize;
	m.msg.rfd = rfd;
	m.msg.follow = follow;
	return TrfsSend(&m, rpid);
}

SvStat(mp, rpid)
	register struct rq_stat *mp;
	unsigned long rpid;
{
	if ((mp->h.xs1 & XS_SEGMASK) == XS_KERN)
		setdstxs(XS_KERN | XS_WRITE);
	xReplyOnCopy(mp, rpid, mp->buf.addr + mp->buf.len - 1, mp->h.xs1);
	if (mp->h.xs0) {
		getuser(&mp->id);
		mp->h.rcode = CoStat(mp->path.addr, mp->follow, mp->buf.addr);
	} else if (mp->rfd >= 0) {
		mp->h.rcode = CoFstat(mp->rfd, mp->buf.addr);
	} else
		mp->h.rcode = EINVAL;		/* shouldn't happen */
}

SdAccess(rpid, path, mode)
	unsigned long rpid;
	char *path;
{
	TRFS_SD(rq_access, RQ_ACCESS, RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 0); 
	setuser(&m.msg.id, rpid);
	m.msg.mode = mode;
	return TrfsSend(&m, rpid);
}

SvAccess(mp, rpid)
	register struct rq_access *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoAccess(mp->path.addr, mp->mode, 
			mp->id.ruid, mp->id.rgid);
}

SdReadlink(rpid, path, buf, lenp, segflg)
	unsigned long rpid;
	char *path, *buf;
	int *lenp;
{
	int err;

	TRFS_SD(rq_readlink, RQ_RDLINK, RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 
		(segflg == UIO_SYSSPACE ? XS_KERN : XS_USER) | XS_WRITE);
	setuser(&m.msg.id, rpid);
	m.msg.buf.addr = buf;
	m.msg.buf.len = *lenp;
	if ((err = TrfsSend(&m, rpid)) == 0)
		*lenp = m.msg.buf.len;
	return err;
}

SvReadlink(mp, rpid)
	register struct rq_readlink *mp;
{
	register int n;

	if ((mp->h.xs1 & XS_SEGMASK) == XS_KERN)
		setdstxs(XS_KERN | XS_WRITE);
	xReplyOnCopy(mp, rpid, mp->buf.addr + mp->buf.len - 1,
				mp->h.xs1 & (XS_SEGMASK | XS_WRITE));
	getuser(&mp->id);
	if ((n = CoReadlink(mp->path.addr, mp->buf.addr, mp->buf.len)) < 0)
		mp->h.rcode = -n;
	else
		mp->buf.len = n;
}

SdUnlink(rpid, path)
	unsigned long rpid;
	char *path;
{
	TRFS_SD(rq_unlink, RQ_UNLINK, RF_NONIDEM | RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 0); 
	setuser(&m.msg.id, rpid);
	return TrfsSend(&m, rpid);
}

SvUnlink(mp, rpid)
	register struct rq_access *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoUnlink(mp->path.addr);
}

SdChmod(rpid, path, rfd, mode)
	unsigned long rpid;
	char *path;
	unsigned short mode;
{
	TRFS_SD(rq_chmod, RQ_CHMOD, (rfd < 0) ? RF_CRCOUSIN : 0, 
		(rfd < 0) ? setpath(&m.msg.path, path) : 0, 0);
	if (rfd < 0)
		setuser(&m.msg.id, rpid);
	else
		bzero(&m.msg.id, sizeof m.msg.id);
	m.msg.rfd = rfd;
	m.msg.mode = mode;
	return TrfsSend(&m, rpid);
}

SvChmod(mp, rpid)
	register struct rq_chmod *mp;
{
	if (mp->h.xs0) {
		getuser(&mp->id);
		mp->h.rcode = CoChmod(mp->path.addr, mp->mode);
	} else if (mp->rfd >= 0)
		mp->h.rcode = CoFchmod(mp->rfd, mp->mode);
	else
		mp->h.rcode = EINVAL;		/* shouldn't happen */
}

SdChown(rpid, path, rfd, uid, gid)
	unsigned long rpid;
	char *path;
	uid_t uid;
	gid_t gid;
{
	TRFS_SD(rq_chown, RQ_CHOWN, (rfd < 0) ? RF_CRCOUSIN : 0, 
		(rfd < 0) ? setpath(&m.msg.path, path) : 0, 0);
	if (rfd < 0)
		setuser(&m.msg.id, rpid);
	else
		bzero(&m.msg.id, sizeof m.msg.id);
	m.msg.rfd = rfd;
	m.msg.uid = uid;
	m.msg.gid = gid;
	return TrfsSend(&m, rpid);
}

SvChown(mp, rpid)
	register struct rq_chown *mp;
{
	if (mp->h.xs0) {
		getuser(&mp->id);
		mp->h.rcode = CoChown(mp->path.addr, mp->uid, mp->gid);
	} else if (mp->rfd >= 0)
		mp->h.rcode = CoFchown(mp->rfd, mp->uid, mp->gid);
	else
		mp->h.rcode = EINVAL;		/* shouldn't happen */
}

SdTruncate(rpid, path, rfd, len)
	unsigned long rpid;
	char *path;
{
	TRFS_SD(rq_truncate, RQ_TRUNC, (rfd < 0) ? RF_CRCOUSIN : 0, 
		(rfd < 0) ? setpath(&m.msg.path, path) : 0, 0);
	if (rfd < 0)
		setuser(&m.msg.id, rpid);
	else
		bzero(&m.msg.id, sizeof m.msg.id);
	m.msg.rfd = rfd;
	m.msg.len = len;
	return TrfsSend(&m, rpid);
}

SvTruncate(mp, rpid)
	register struct rq_truncate *mp;
{
	if (mp->h.xs0) {
		getuser(&mp->id);
		mp->h.rcode = CoTruncate(mp->path.addr, mp->len);
	} else if (mp->rfd >= 0)
		mp->h.rcode = CoFtruncate(mp->rfd, mp->len);
	else
		mp->h.rcode = EINVAL;		/* shouldn't happen */
}

SdTimes(rpid, path, tv, tvlen)
	unsigned long rpid;
	char *path;
	struct timeval *tv;
{
	TRFS_SD(rq_times, RQ_TIMES, RF_CRCOUSIN, setpath(&m.msg.path, path),
			(tv ? (XS_KERN | XS_READ) : 0));
	setuser(&m.msg.id, rpid);
	m.msg.tv.addr = (char *)tv;
	m.msg.tv.len = tvlen;
	return TrfsSend(&m, rpid);
}

SvTimes(mp, rpid)
	register struct rq_times *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoTimes(mp->path.addr, (struct timeval *)mp->tv.addr);
}

SdRename(rpid, from, to)
	unsigned long rpid;
	char *from, *to;
{
	TRFS_SD(rq_rename, RQ_RENAME, RF_NONIDEM | RF_CRCOUSIN, 
		setpath(&m.msg.from, from), setpath(&m.msg.to, to)); 
	setuser(&m.msg.id, rpid);
	return TrfsSend(&m, rpid);
}

SvRename(mp, rpid)
	register struct rq_rename *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoRename(mp->from.addr, mp->to.addr);
}

SdLink(rpid, target, linkname, sym)
	unsigned long rpid;
	char *target, *linkname;
{
	TRFS_SD(rq_link, RQ_LINK, RF_NONIDEM | RF_CRCOUSIN | (sym ? RF_ALT : 0),
		setpath(&m.msg.target, target), 
		setpath(&m.msg.linkname, linkname)); 
	setuser(&m.msg.id, rpid);
	return TrfsSend(&m, rpid);
}

SvLink(mp, rpid)
	register struct rq_link *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoLink(mp->target.addr, mp->linkname.addr,
						(mp->h.rflags & RF_ALT));
}

SdMkdir(rpid, path, mode)
	unsigned long rpid;
	char *path;
{
	TRFS_SD(rq_mkdir, RQ_MKDIR, RF_NONIDEM | RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 0);
	setuser(&m.msg.id, rpid);
	m.msg.mode = mode;
	return TrfsSend(&m, rpid);
}

SvMkdir(mp, rpid)
	register struct rq_mkdir *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoMkdir(mp->path.addr, mp->mode);
}

SdRmdir(rpid, path)
	unsigned long rpid;
	char *path;
{
	TRFS_SD(rq_rmdir, RQ_RMDIR, RF_NONIDEM | RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 0);
	setuser(&m.msg.id, rpid);
	return TrfsSend(&m, rpid);
}

SvRmdir(mp, rpid)
	register struct rq_rmdir *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoRmdir(mp->path.addr);
}

SdMknod(rpid, path, mode, dev)
	unsigned long rpid;
	char *path;
{
	TRFS_SD(rq_mknod, RQ_MKNOD, RF_NONIDEM | RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 0);
	setuser(&m.msg.id, rpid);
	m.msg.mode = mode;
	m.msg.dev = dev;
	return TrfsSend(&m, rpid);
}

SvMknod(mp, rpid)
	register struct rq_mknod *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoMknod(mp->path.addr, mp->mode, mp->dev);
}

SdFsync(rpid, rfd)
	unsigned long rpid;
{
	TRFS_SD(rq_fsync, RQ_FSYNC, 0, 0, 0);
	m.msg.rfd = rfd;
	return TrfsSend(&m, rpid);
}

SvFsync(mp, rpid)
	struct rq_fsync *mp;
{
	mp->h.rcode = CoFsync(mp->rfd);
}

SdChdir(rpid, path, isroot, onoff)
	unsigned long rpid;
	char *path;
	unsigned short isroot, onoff;
{
	TRFS_SD(rq_chdir, RQ_CHDIR, RF_NONIDEM |
		(((*path == '/') || (*path == 0)) ? RF_CRCOUSIN : 0) | 
		(isroot ? RF_ALT : 0), setpath(&m.msg.path, path), 0);
	setuser(&m.msg.id, rpid);
	m.msg.onoff = onoff;
	return TrfsSend(&m, rpid);
}

SvChdir(mp, rpid)
	register struct rq_chdir *mp;
{
	getuser(&mp->id);
	mp->h.rcode = CoChdirec(mp->path.addr, (mp->h.rflags & RF_ALT),
								mp->onoff);
}

SdGetdir(rpid, rfd, buf, len, basep, baseplen, offset, rval)
	unsigned long rpid;
	char *buf, *basep;
	long *offset, *rval;
{
	int err;

	TRFS_SD(rq_getdir, RQ_GETDIR, 0, XS_USER|XS_WRITE, XS_USER|XS_WRITE);
	m.msg.buf.addr = buf;
	m.msg.buf.len = len;
	m.msg.basep.addr = basep;
	m.msg.basep.len = baseplen;
	m.msg.offset = *offset;
	m.msg.rfd = rfd;
	if ((err = TrfsSend(&m, rpid)) == 0) {
		*offset = m.msg.offset;
		*rval = m.msg.rval;
	}
	return err;
}

SvGetdir(mp, rpid)
	register struct rq_getdir *mp;
{
	mp->h.rcode = CoGetdir(mp->rfd, mp->buf.addr, mp->buf.len, 
		mp->basep.addr, &mp->offset, &mp->rval);
}

SdStatfs(rpid, path, rfd, buf, bufsize, segflg)
	unsigned long rpid;
	char *path, *buf;
{
	TRFS_SD(rq_statfs, RQ_STATFS, (rfd < 0) ? RF_CRCOUSIN : 0, 
		(rfd < 0) ? setpath(&m.msg.path, path) : 0,
		(segflg == UIO_SYSSPACE ? XS_KERN : XS_USER) | XS_WRITE);
	if (rfd < 0)
		setuser(&m.msg.id, rpid);
	else
		bzero(&m.msg.id, sizeof m.msg.id);
	m.msg.buf.addr = buf;
	m.msg.buf.len = bufsize;
	m.msg.rfd = rfd;
	return TrfsSend(&m, rpid);
}

SvStatfs(mp, rpid)
	register struct rq_statfs *mp;
	unsigned long rpid;
{
	if ((mp->h.xs1 & XS_SEGMASK) == XS_KERN)
		setdstxs(XS_KERN | XS_WRITE);
	xReplyOnCopy(mp, rpid, mp->buf.addr + mp->buf.len - 1,
				mp->h.xs1 & (XS_SEGMASK | XS_WRITE));
	if (mp->h.xs0) {
		getuser(&mp->id);
		mp->h.rcode = CoStatfs(mp->path.addr, mp->buf.addr);
	} else if (mp->rfd >= 0)
		mp->h.rcode = CoFstatfs(mp->rfd, mp->buf.addr);
	else
		mp->h.rcode = EINVAL;		/* shouldn't happen */
}

SdFlock(rpid, rfd, how)
	unsigned long rpid;
{
	TRFS_SD(rq_flock, RQ_FLOCK, RF_NONIDEM, 0, 0);
	m.msg.rfd = rfd;
	m.msg.how = how;
	return TrfsSend(&m, rpid);
}

SvFlock(mp, rpid)
	struct rq_flock *mp;
{
	mp->h.rcode = CoFlock(mp->rfd, mp->how);
}

SdLockf(rpid, rfd, flag, size, offset)
	unsigned long rpid, size;
	off_t offset;
{
	TRFS_SD(rq_lockf, RQ_LOCKF, RF_NONIDEM, 0, 0);
	m.msg.rfd = rfd;
	m.msg.flag = flag;
	m.msg.size = size;
	m.msg.offset = offset;
	return TrfsSend(&m, rpid);
}

SvLockf(mp, rpid)
	struct rq_lockf *mp;
{
	mp->h.rcode = CoLockf(mp->rfd, mp->flag, mp->size, mp->offset);
}

SdFcntl(rpid, rfd, cmd, arg, offset, buf, bufsize, read, write)
	unsigned long rpid;
	off_t offset;
	char *buf;
{
	register int rval;

	TRFS_SD(rq_fcntl, RQ_FCNTL, RF_NONIDEM,
		((read && write) ? XS_USER | 
			(read ? XS_READ : 0) | 
			(write ? XS_WRITE : 0) : 0), 0);
	m.msg.cmd = cmd;
	m.msg.arg = arg;
	m.msg.offset = offset;
	m.msg.buf.addr = buf;
	m.msg.buf.len = bufsize;
	m.msg.rfd = rfd;
	if ((rval = TrfsSend(&m, rpid)) == 0)
		u.u_r.r_val1 = m.msg.arg;
	return rval;
}

SvFcntl(mp, rpid)
	struct rq_fcntl *mp;
{
	u.u_r.r_val1 = 0;
	if (mp->buf.addr) {
		xReplyOnCopy(mp, rpid,mp->buf.addr + mp->buf.len - 1,mp->h.xs0);
		mp->h.rcode = CoFcntl(mp->rfd, mp->cmd, mp->buf.addr,
							mp->offset);
	} else
		mp->h.rcode = CoFcntl(mp->rfd, mp->cmd, mp->arg, mp->offset);
	if (mp->h.rcode == 0)
		mp->arg = u.u_r.r_val1;
}

/* ------- file ops ------ */

SdClose(rpid, rfd)
	unsigned long rpid;
{
	TRFS_SD(rq_close, RQ_CLOSE, RF_NONIDEM, 0, 0);
	m.msg.rfd = rfd;
	return TrfsSend(&m, rpid);
}

SvClose(mp, rpid)
	register struct rq_close *mp;
{
	mp->h.rcode = CoClose(mp->rfd);
}

SdRw(rpid, rfd, rw, base, offsetp, residp, segflg, maxfsize, nonidem)
	unsigned long rpid;
	enum uio_rw rw;
	char *base;
	unsigned long *offsetp, *residp;
{
	int rval;

	TRFS_SD(rq_rw, RQ_RW, nonidem ? RF_NONIDEM : 0, 
	        (segflg == UIO_SYSSPACE ? XS_KERN : XS_USER) | 
		(rw == UIO_READ ? XS_WRITE : (XS_READ | XS_RMBURST)), 0);
	m.msg.buf.addr = base;
	m.msg.buf.len = *residp;
	m.msg.offset = *offsetp;
	m.msg.maxfsize = maxfsize;
	m.msg.rfd = rfd;
	if (rval = TrfsSend(&m, rpid))
		return rval;
	*residp = m.msg.buf.len;
	*offsetp = m.msg.offset;
	return 0;
}

SvRw(mp, rpid)
	register struct rq_rw *mp;
	unsigned long rpid;
{
	enum uio_rw rw;
	int segflg;
	unsigned long resid, offset;

	switch (mp->h.xs0 & (XS_READ | XS_WRITE)) {
	  case XS_READ:	
		rw = UIO_WRITE;
		break;
	  case XS_WRITE:
		rw = UIO_READ;
		break;
	  default:
		mp->h.rcode = EINVAL;
		return;
	}
	switch (mp->h.xs0 & XS_SEGMASK) {
	  case XS_USER:
		segflg = UIO_USERSPACE;
		break;
	  case XS_KERN:
		segflg = UIO_SYSSPACE;
		break;
	  default:
		mp->h.rcode = EINVAL;
		return;
	}
	resid = mp->buf.len;
	offset = mp->offset;
	mp->buf.len = 0;
	mp->offset = 0;
	xReplyOnCopy(mp, rpid, mp->buf.addr + resid - 1, mp->h.xs0);
	mp->h.rcode = CoRw(mp->rfd, rw, mp->buf.addr, &offset, &resid,
							segflg, mp->maxfsize);
	mp->buf.len = resid;
	mp->offset = offset;
}

SdIoctl(rpid, rfd, com, buf, buflen)
	unsigned long rpid;
	char *buf;
{
	TRFS_SD(rq_ioctl, RQ_IOCTL, RF_NONIDEM, XS_READ | XS_WRITE | XS_KERN, 0);
	setuser(&m.msg.id, rpid);
	m.msg.buf.addr = buf;
	m.msg.buf.len = buflen;
	m.msg.com = com;
	m.msg.rfd = rfd;
	return TrfsSend(&m, rpid);
}

SvIoctl(mp, rpid)
	register struct rq_ioctl *mp;
{
	getuser(&mp->id);
	setsrcxs(XS_KERN | XS_READ);
	setdstxs(XS_KERN | XS_WRITE);
	mp->h.rcode = CoIoctl(mp->rfd, mp->com, mp->buf.addr);
}

SdSelect(rpid, rfd, flag)
	unsigned long rpid;
{
	TRFS_SD(rq_select, RQ_SELECT, RF_NONIDEM, 0, 0);
	m.msg.rfd = rfd;
	m.msg.flag = flag;
	return TrfsSend(&m, rpid);
}

SvSelect(mp, rpid)
	register struct rq_select *mp;
{
	mp->h.rcode = CoSelect(mp->rfd, mp->flag);
}

/* ------- misc calls ------ */

SdUnpBind(rpid, path, so, rfdp)	
	unsigned long rpid;
	char *path;
	struct socket *so;
	int *rfdp;
{
	int err;

	TRFS_SD(rq_unp_bind, RQ_UNP_BIND, RF_NONIDEM | RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 0);
	setuser(&m.msg.id, rpid);
	m.msg.so = (unsigned long)so;
	if ((err = TrfsSend(&m, rpid)) == 0)
		*rfdp = m.msg.rfd;
	return err;
}

SvUnpBind(mp, rpid)
	struct rq_unp_bind *mp;
	unsigned long rpid;
{
	int rfd;

	getuser(&mp->id);
	if ((mp->h.rcode = CoUnpBind(mp->path.addr, mp->so, MID(rpid), &rfd))
									== 0)
		mp->rfd = rfd;
}

SdUnpDetach(rpid, rfd)
	unsigned long rpid;
	int rfd;
{
	TRFS_SD(rq_unp_detach, RQ_UNP_DETACH, RF_NONIDEM | RF_CRCOUSIN, 0, 0);
	m.msg.rfd = rfd;
	return TrfsSend(&m, rpid);
}

SvUnpDetach(mp, rpid)
	struct rq_unp_detach *mp;
{
	mp->h.rcode = CoUnpDetach(mp->rfd);
}

SdUnpConnect(rpid, path, sop)
	unsigned long rpid;
	char *path;
	struct socket **sop;
{
	int err;

	TRFS_SD(rq_unp_connect, RQ_UNP_CONNECT, RF_NONIDEM | RF_CRCOUSIN, 
		setpath(&m.msg.path, path), 0);
	setuser(&m.msg.id, rpid);
	m.msg.so = 0;
	if ((err = TrfsSend(&m, rpid)) == 0)
		*sop = (struct socket *)m.msg.so;
	return err;
}

SvUnpConnect(mp, rpid)
	struct rq_unp_connect *mp;
	unsigned long rpid;
{
	struct socket *so;

	getuser(&mp->id);
	if ((mp->h.rcode = CoUnpConnect(mp->path.addr, &so, MID(rpid))) == 0)
		mp->so = (unsigned long)so;
}

SdCore(rpid, addr, len, offset, seg)
	unsigned long rpid;
	char *addr;
	unsigned long len, offset;
{
	TRFS_SD(rq_core, RQ_CORE, 0, (seg ? XS_KERN : XS_USER) | XS_READ, 0);
	m.msg.buf.addr = addr;
	m.msg.buf.len = len;
	m.msg.offset = offset;
	return TrfsSend(&m, rpid);
}

SvCore(mp, rpid)
	register struct rq_core *mp;
{
	int segflg = 0;

	if ((mp->h.xs0 & XS_SEGMASK) == XS_KERN) {
		setsrcxs(XS_KERN | XS_READ);
		segflg = 1;
	}
	mp->h.rcode = CoCore(mp->buf.addr, mp->buf.len, mp->offset, segflg);
}

SdSwap(rpid, rw, blkno, len, addr)
	unsigned long rpid, blkno, len;
	char *addr;
{
	TRFS_SD(rq_swap, RQ_SWAP, 0, XS_PHYS | (rw ? XS_WRITE : XS_READ), 0);
	m.msg.buf.addr = addr;
	m.msg.buf.len = len;
	m.msg.blkno = blkno;
	return TrfsSend(&m, rpid);
}

SvSwap(mp, rpid)
	register struct rq_swap *mp;
	unsigned long rpid;
{
	setsrcxs(XS_PHYS | XS_READ);
	setdstxs(XS_PHYS | XS_WRITE);
	xReplyOnCopy(mp, rpid, mp->buf.addr + mp->buf.len - 1, mp->h.xs0);
	mp->h.rcode = CoSwap(rpid, (mp->h.xs0&XS_WRITE) ? UIO_READ : UIO_WRITE,
					mp->buf.addr, mp->buf.len, mp->blkno);
}

/* ------- heartbeat calls ------ */

SdAmAlive(name, rpid)
	char *name;
	unsigned long rpid;
{
	union {
		struct msg_connect	msg;
		struct wmsg		pad;
	} m;
	extern struct timeval time;

	bzero(&m, sizeof m);
	m.msg.h.rcode = RQ_ALIVE;
	m.msg.h.rflags = RF_NOREP;
	strncpy(m.msg.mname, name, sizeof m.msg.mname - 2);
	m.msg.sec = time.tv_sec;
	m.msg.usec = time.tv_usec;
	return xSend(&m, rpid);
}

SvIsAlive(mp, rpid)
	struct msg_connect *mp;
{
	char name[64];

	strncpy(name, mp->mname, sizeof name);
	name[sizeof name - 1] = '\0';
	UpdateRef(name);
}

/* ------- internet address translation calls ------ */

SdINAddr(rpid, name, namelen, addp, addlen)
{
	union {
		struct rq_get_in_addr	msg;
		struct wmsg		pad;
	} m;
	int err;

	if (namelen > sizeof m.msg.mname  ||  addlen > sizeof m.msg.addbuf)
		return EINVAL;
	bzero(&m, sizeof m);
	m.msg.h.rcode = RQ_GET_IN_ADDR;
	m.msg.h.rflags = 0;
	m.msg.h.xs0 = 0;
	m.msg.h.xs1 = 0;
	strcpy(m.msg.mname, name);
	if ((err = TrfsSend(&m, rpid)) == 0)
		strcpy(addp, m.msg.addbuf);
	return err;
}

SvINAddr(mp, rpid)
	struct rq_get_in_addr *mp;
{
	mp->h.rcode = HostToINAddr(mp->mname, mp->addbuf, sizeof mp->addbuf);
}

SdGetlink(rpid, path, buf, lenp, followp, cindexp, clenp)
	unsigned long rpid;
	char *path, *buf;
	int *lenp, *followp, *cindexp, *clenp;
{
	int err;

	TRFS_SD(rq_getlink, RQ_GETLINK, RF_CRCOUSIN, 
		setpath(&m.msg.path, path), XS_KERN | XS_WRITE);
	setuser(&m.msg.id, rpid);
	m.msg.buf.addr = buf;
	m.msg.buf.len = *lenp;
	m.msg.follow = *followp;
	err = TrfsSend(&m, rpid);
	*lenp = m.msg.buf.len;
	*followp = m.msg.follow;
	*cindexp = m.msg.cindex;
	*clenp = m.msg.path.len;		/* KLUDGE */
	return err;
}

SvGetlink(mp, rpid)
	register struct rq_getlink *mp;
{
	int len, follow, cindex, clen;

	setdstxs(XS_KERN | XS_WRITE);
	getuser(&mp->id);
	len = mp->buf.len;
	follow = mp->follow;
	cindex = clen = 0;
	mp->h.rcode = CoGetlink(mp->path.addr, mp->buf.addr, &len, 
		&follow, &cindex, &clen);
	mp->buf.len = len;
	mp->follow = follow;
	mp->cindex = cindex;
	mp->path.len = clen;			/* KLUDGE */
}

/* ------- protocol entry route ------ */

TrfsSend(mp, rpid)
	register struct wmsg *mp;
	unsigned long rpid;
{
	register unsigned long rval;

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		mp->h.rflags |= RF_UNIVERSE_SYSV;
	else
		mp->h.rflags &= ~RF_UNIVERSE_SYSV;
#endif	SYSV
	/* Currently xSend does not return error numbers! */
	if ((int)(rval = xSend(mp, rpid)) < 0)
		return -(int)rval;
	if (rval == 0)
		return EIO;
	if (mp->h.rcode == ENETRESET  &&  PID(rpid) == SERVER_PID)
		RestartSys();
	else if (rval != rpid  &&  PID(rval) != SERVER_PID)
		EnterRpid(rval);
	return mp->h.rcode;
}

#ifdef	SYSV

SdSysv_Statfs(rpid, path, rfd, len, fstyp, buf, bufsize, segflg)
	unsigned long rpid;
	char *path;
	char *buf;
{
	TRFS_SD(rq_sysv_statfs, RQ_SYSV_STATFS, (rfd < 0) ? RF_CRCOUSIN : 0, 
		(rfd < 0) ? setpath(&m.msg.path, path) : 0,
		(segflg == UIO_SYSSPACE ? XS_KERN : XS_USER) | XS_WRITE);
	if (rfd < 0)
		setuser(&m.msg.id, rpid);
	else
		bzero(&m.msg.id, sizeof m.msg.id);
	m.msg.buf.addr = buf;
	m.msg.buf.len = bufsize;
	m.msg.len = len;
	m.msg.fstyp = fstyp;
	m.msg.rfd = rfd;
	return TrfsSend(&m, rpid);
}

SvSysv_Statfs(mp, rpid)
	register struct rq_sysv_statfs *mp;
	unsigned long rpid;
{
	if ((mp->h.xs1 & XS_SEGMASK) == XS_KERN)
		setdstxs(XS_KERN | XS_WRITE);
	xReplyOnCopy(mp, rpid, mp->buf.addr + mp->buf.len - 1,
				mp->h.xs1 & (XS_SEGMASK | XS_WRITE));
	if (mp->h.xs0) {
		getuser(&mp->id);
		mp->h.rcode = CoSysv_Statfs(mp->path.addr, mp->buf.addr,
				mp->len, mp->fstyp);
	} else if (mp->rfd >= 0)
		mp->h.rcode = CoSysv_Fstatfs(mp->path.addr, mp->buf.addr,
				mp->len, mp->fstyp);
	else
		mp->h.rcode = EINVAL;		/* shouldn't happen */
}

SdGetdents(rpid, rfd, buf, nbytes, offset, rval)
	unsigned long rpid;
	char *buf;
	long *offset, *rval;
{
	int err;

	TRFS_SD(rq_getdents, RQ_GETDENTS, 0, XS_USER|XS_WRITE, 0);
	m.msg.buf.addr = buf;
	m.msg.buf.len = nbytes;
	m.msg.offset = *offset;
	m.msg.rfd = rfd;
	if ((err = TrfsSend(&m, rpid)) == 0) {
		*offset = m.msg.offset;
		*rval = m.msg.rval;
	}
	return err;
}

SvGetdents(mp, rpid)
	register struct rq_getdents *mp;
{
	mp->h.rcode = CoGetdents(mp->rfd, mp->buf.addr, mp->buf.len,
		&mp->offset, &mp->rval);
}

SdSysv_Ioctl(rpid, rfd, cmd, arg)
	unsigned long rpid;
{
	TRFS_SD(rq_sysv_ioctl, RQ_SYSV_IOCTL, RF_NONIDEM, 0, 0);
	setuser(&m.msg.id, rpid);
	m.msg.cmd = cmd;
	m.msg.arg = arg;
	m.msg.rfd = rfd;
	return TrfsSend(&m, rpid);
}

SvSysv_Ioctl(mp, rpid)
	register struct rq_sysv_ioctl *mp;
{
	getuser(&mp->id);
	setsrcxs(XS_USER | XS_READ);
	setdstxs(XS_USER | XS_WRITE);
	mp->h.rcode = CoIoctl(mp->rfd, mp->cmd, mp->arg);
}

SdSysv_Lseek(rpid, rfd, off, sbase, voffsetp, foffsetp)
	unsigned long rpid;
	off_t off;
	off_t *voffsetp, *foffsetp;
{
	register int err;

	TRFS_SD(rq_sysv_lseek, RQ_SYSV_LSEEK, 0, 0, 0);
	m.msg.off = off;
	m.msg.sbase = sbase;
	m.msg.foffset = *foffsetp;
	m.msg.rfd = rfd;
	if ((err = TrfsSend(&m, rpid)) == 0) {
		*voffsetp =  m.msg.voffset;
		*foffsetp =  m.msg.foffset;
	}
	return err;
}

SvSysv_Lseek(mp, rpid)
	register struct rq_sysv_lseek *mp;
{
	mp->h.rcode = CoSysv_Lseek(mp->rfd, mp->off, mp->sbase,
		&mp->voffset, &mp->foffset);
}

SdPload(rpid, path, addr, len, offset, segflg)
	unsigned long rpid;
	char *path;
	char *addr;
	unsigned long len, offset;
{
	TRFS_SD(rq_pload, RQ_PLOAD, RF_NONIDEM | RF_CRCOUSIN,
		setpath(&m.msg.path, path),
		(segflg == UIO_USERSPACE) ? XS_USER : XS_KERN | XS_WRITE);
	m.msg.buf.addr = addr;
	m.msg.buf.len = len;
	m.msg.offset = offset;
	return TrfsSend(&m, rpid);
}

SvPload(mp, rpid)
	register struct rq_pload *mp;
{
	int segflg = UIO_USERSPACE;

	if ((mp->h.xs1 & XS_SEGMASK) == XS_KERN) {
		setdstxs(XS_KERN | XS_WRITE);
		segflg = UIO_SYSSPACE;
	}
	xReplyOnCopy(mp, rpid, mp->buf.addr + mp->buf.len - 1, mp->h.xs1);
	mp->h.rcode = CoPload(mp->path.addr, mp->buf.addr, mp->buf.len,
				mp->offset, segflg);
}

SdGethead(rpid, path, exdatap, exdatalen)
	unsigned long rpid;
	char *path, *exdatap;
{
	TRFS_SD(rq_gethead, RQ_GETHEAD, RF_NONIDEM | RF_CRCOUSIN,
			setpath(&m.msg.path, path), XS_KERN | XS_WRITE);
	m.msg.buf.addr = exdatap;
	m.msg.buf.len = exdatalen;
	return TrfsSend(&m, rpid);
}

SvGethead(mp, rpid)
	register struct rq_gethead *mp;
{
	mp->h.rcode = CoGethead(mp->path.addr, mp->buf.addr);
}

int
SdSysv_Stat(rpid, path, follow, rfd, buf, bufsize, segflg)
	unsigned long rpid;
	char *path, *buf;
{
	TRFS_SD(rq_stat, RQ_SYSV_STAT, (rfd < 0) ? RF_CRCOUSIN : 0, 
		(rfd < 0) ? setpath(&m.msg.path, path) : 0,
		(segflg == UIO_SYSSPACE ? XS_KERN : XS_USER) | XS_WRITE);
	if (rfd < 0)
		setuser(&m.msg.id, rpid);
	else
		bzero(&m.msg.id, sizeof m.msg.id);
	m.msg.buf.addr = buf;
	m.msg.buf.len = bufsize;
	m.msg.rfd = rfd;
	m.msg.follow = follow;
	return TrfsSend(&m, rpid);
}

SvSysv_Stat(mp, rpid)
	register struct rq_stat *mp;
	unsigned long rpid;
{
	if ((mp->h.xs1 & XS_SEGMASK) == XS_KERN)
		setdstxs(XS_KERN | XS_WRITE);
	xReplyOnCopy(mp, rpid, mp->buf.addr + mp->buf.len - 1, mp->h.xs1);
	if (mp->h.xs0) {
		getuser(&mp->id);
		mp->h.rcode = CoSysv_Stat(mp->path.addr, mp->buf.addr);
	} else if (mp->rfd >= 0)
		mp->h.rcode = CoSysv_Fstat(mp->rfd, mp->buf.addr);
	else
		mp->h.rcode = EINVAL;		/* shouldn't happen */
}

#endif	SYSV
