/*	Copyright (c) 1984 AT&T	*/
/*	  All Rights Reserved  	*/

/*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF AT&T	*/
/*	The copyright notice above does not evidence any   	*/
/*	actual or intended publication of such source code.	*/

/*	#ident	"@(#)kern-port:nudnix/remcall.c	10.27"		*/
#if	defined(SYSV) && defined(RFS)
/*
 *	R E M O T E   C A L L
 *
 *	Format a msg representing the i/o system call that
 *	hit a remote mount point.  Send the msg to the
 *	remote system for further execution.
 *
 *	If successful, remote_call will do a long jump
 *	and so will never return to its caller.
 *
 *	If, however, the pathname evaluation ends up back on
 *	the local system, then remote_call resets pathname structure
 *	to the remainder of the pathname and returns a pointer
 *	to the vnode in which the search should continue
 *	locally. (This can be possible as lookuppn saves the pointer
 *	to pathname structure in u.u_rfspnp!!).
 *
 *	If there is any error, u.u_error is set and NULL returned.
 */

#include "../h/types.h"
#include "../sysv/sys/sema.h"
#include "../h/param.h"
#include "../sysv/sys/errno.h"
#include "../h/user.h"
#include "../sysv/sys/comm.h"
#include "../sysv/sys/message.h"
#include "../sysv/sys/stat.h"
#include "../h/proc.h"
#include "../sysv/sys/idtab.h"
#include "../sysv/sys/stream.h"
#include "../sysv/sys/nserve.h"
#include "../sysv/sys/cirmgr.h"
#include "../sysv/sys/debug.h"
#include "../sysv/sys/sysinfo.h"
#include "../sysv/sys/rdebug.h"
#include "../h/vnode.h"
#include "../h/vfs.h"
#include "../sysv/sys/rfsnode.h"
#include "../sysv/sys/mount.h"
#include "../ufs/inode.h"
#include "../ufs/mount.h"
#include "../sysv/sys/sysmacros.h"
#include "../h/file.h"
#include "../sysv/sys/fcntl.h"
#include "../h/pathname.h"
#include "../h/uio.h"

extern	struct timeval	time;
extern	struct	vnode *rem_vnode();
extern	caddr_t	kmem_alloc();

#define JUMP 1
#define NOJUMP 0

struct vnode *
remote_call (vp, arg)
register struct vnode *vp;
caddr_t arg;
{
	register struct sndd *out_port = (struct sndd *)(VTORFS(vp))->rf_fsptr;
	mblk_t	*in_bp;
	register mblk_t	*out_bp;
	register struct	request	*msg_out;
	sndd_t	ret_gift = NULL;/* gift w/in-bound msg		*/
	struct	vnode *vnode_p;	/* return value		 */
	int	c;		/* c and cp are used to move the */
	register char	*cp;	/* pathname to the msg_out buffer*/
	int	jump = 0;	/* whether to longjmp at end  */
	register struct pathname *pnp;

	if(u.u_procp->p_universe != UNIVERSE_SYSV) {
		u.u_error = EINVAL;
		return(NULL);
	}
#ifdef TRFS
	if (u.u_tclient) {
		u.u_error = EREMOTE_SV;
		return NULL;
	}
#endif
#if	DUDEBUG
	DUPRINT4(DB_SYSCALL, "remote_call: syscall is %d (%s), rp %x\n",
		 u.u_procp->p_systemcall, sysname(u.u_procp->p_systemcall),
		 VTORFS(vp));
	if(strcmp(sysname(u.u_procp->p_systemcall), "unknown") == 0) {
		u.u_error = EREMOTE_SV;
		return NULL;
	}
#endif

	if (server()) {	/* if server, this is multihop	*/
		u.u_error = EMULTIHOP;
		return(NULL);
	}
	sysinfo.remcall++;
	out_port->sd_refcnt++;	/* rmount will not succeed if cnt > 1 */
	out_bp = alocbuf(sizeof (struct request), BPRI_LO);
	if (out_bp == NULL) {
		u.u_error = EINTR;
		goto out;
	}
	msg_out = (struct request *) PTOMSG(out_bp->b_rptr);
	msg_out->rq_type = REQ_MSG;
	msg_out->rq_mntindx = out_port->sd_mntindx;
	cp = msg_out->rq_data;
	pnp = u.u_rfspnp;
	/* Copy remaining path name */
	if(pnp) {
		pn_skipslash(pnp); /* Knock off leading slashes */
		while (c = pn_getchar(pnp)) {
			if (cp < &msg_out->rq_data[DATASIZE - 2])
				*cp++ = c;
			else {
				u.u_error = E2BIG;
				freemsg (out_bp);
				goto out;
			}
		}
	}
	*cp++ = '\0';	/*  namei on the other side looks one char past the
			    end of the string.  Don't take this out!!*/
			
	rcallmsg (msg_out, out_port, &ret_gift,arg, vp);
	if (u.u_error) {
		freemsg (out_bp);
		goto out;
	}
	dinfo.osyscall++;	/* outgoing system calls */

	/* Locking may not be necessary! */
	if (!(u.u_procp->p_systemcall == DUCLOSE))
	  	rfslock(VTORFS(vp));

	rfs_trace (3, 0);
	/* send the message and wait for the response */
	(void)rsc(out_port, out_bp, cp - (char *) msg_out, &in_bp,
				ret_gift, (struct uio *)NULL);
	if (u.u_error) {
		if (ret_gift)
			free_sndd(ret_gift);
		rfsunlock(VTORFS(vp));
		goto out;
	}
	out_port->sd_refcnt--;
	jump = rcallret (in_bp, ret_gift, &vnode_p, out_port, vp);

	rfs_trace (4, jump);
	rfsunlock(VTORFS(vp));

	/*
	 * Ensure u.u_nextcp doesn't point to a slash.
	 */
	pnp = u.u_rfspnp;
	pn_skipslash(pnp); /* NEEDED? */
	if (jump == JUMP) {
		pn_free(pnp);
		u.u_rflags |= U_RSYS;
		VN_RELE(vp);
		longjmp (&u.u_qsave, 1);
	} else
		return (vnode_p);
out:
	out_port->sd_refcnt--;
	return (NULL);
}

/*  Prepare a remote call message to send to the server side.
 *  The arguments depend on the call that is continuing remotely.
 *  For operations that are just plain namei's, make sure that
 *  the right intention (sought, create, delete) is made clear.
 */

/*static*/
rcallmsg (rmp, out_port, gift, arg, vp)
register struct	request	*rmp;
sndd_t out_port, *gift;
caddr_t arg;
struct vnode *vp;
{
	extern short dufstyp;

	rmp->rq_sysid = get_sysid(out_port);
	rmp->rq_uid = u.u_uid;
	rmp->rq_gid = u.u_gid;
	rmp->rq_opcode = u.u_procp->p_systemcall;
	rmp->rq_ulimit = (u.u_rlimit[RLIMIT_FSIZE].rlim_max) >> 9;
	rmp->rq_cmask = u.u_cmask;

	if (u.u_rdir && (u.u_rdir->v_flag & VRFSNODE))
		/* put in index of remote root rcvd */
		rmp->rq_rrdir =
			((sndd_t)VTORFS(u.u_rdir)->rf_fsptr)->sd_sindex;
	else
		rmp->rq_rrdir = 0;

	/*  system calls that do namei's to do more later  */
	switch (u.u_procp->p_systemcall)  {
	case DUEXEC:
	case DUEXECE:
		dinfo.osysexec++;	/* outgoing exec's */
	case DUCHDIR:
	case DUCHROOT:
		if ((*gift = cr_sndd()) == NULL)
			u.u_error = ENOMEM;
		break;
			
	case DULINK1:
	case DULINK:
	    	if (vp->v_vfsp->vfs_flag & VFS_RDONLY) {
			u.u_error = EROFS;
			break;
		}

		if (u.u_pdir == NULL) {	
			if ((*gift = cr_sndd()) == NULL)
				u.u_error = ENOMEM;
		} else {
			register struct vnode *tvp = u.u_pdir;
			register struct sndd *tsd;

			u.u_procp->p_systemcall = DULINK1;
			rmp->rq_opcode = u.u_procp->p_systemcall;
			if (tvp->v_flag & VRFSNODE) {
				tsd = (struct sndd *)VTORFS(tvp)->rf_fsptr;
				if(tsd->sd_queue == out_port->sd_queue)
					rmp->rq_link = tsd->sd_sindex;
				else	
					rmp->rq_link = 0;
			} else
				rmp->rq_link = 0;
		}
		break;

	case DUUNLINK:
		if (rmp->rq_data[0] == '\0' ||
			 (rmp->rq_data[0] == '.' && rmp->rq_data[1] == '\0'))
		  	u.u_error = EBUSY;
	    	else if (vp->v_vfsp->vfs_flag & VFS_RDONLY)
			u.u_error = EROFS;
		break;

	case DUIUPDATE:
	case DUUPDATE:
	    	if (vp->v_vfsp->vfs_flag & VFS_RDONLY)
			u.u_error = EROFS;

		break;

	case DUCLOSE:
	{
		register struct a {
			int mode;
			int count;
			off_t off;
		} *temp;

		temp = (struct a *)arg;
		rmp->rq_fmode = temp->mode;
		rmp->rq_foffset = temp->off;
		rmp->rq_count  = temp->count;
		break;
	}
	case DUACCESS:
	{
		struct a {
			char *fname;
			int fmode;
		} *uap = (struct a *) u.u_ap;
		
		if ((uap->fmode & W_OK) && (vp->v_vfsp->vfs_flag & VFS_RDONLY))
		  u.u_error = EROFS;
		else {
			rmp->rq_uid = u.u_ruid;
			rmp->rq_gid = u.u_rgid;
			rmp->rq_fmode = uap->fmode;
		}
		break;
	}
	case DUADVERTISE:
		u.u_error = EREMOTE_SV;	/* can't adv remote resource */
		break;
	case DUCHMOD:
	{
		struct a { char *fname; int fmode; } *uap =
		  (struct a *) u.u_ap;
	    	if (vp->v_vfsp->vfs_flag & VFS_RDONLY)
		  u.u_error = EROFS;
		else
		  rmp->rq_fmode = uap->fmode;
		break;
	}
	case DUCHOWN:
	{
		register struct a { char *fname; int uid; int gid; }
		*uap = (struct a *) u.u_ap;
	    	if (vp->v_vfsp->vfs_flag & VFS_RDONLY)
		  u.u_error = EROFS;
		else {
		    rmp->rq_newuid = uap->uid;
		    rmp->rq_newgid = uap->gid;
		}
		break;
	}
	case DUCOREDUMP:
		if ((*gift = cr_sndd()) == NULL)
			u.u_error = ENOMEM;
		rmp->rq_cmask = u.u_cmask;
		break;

	case DUOPEN:
	{
		register struct a { char *fname; int mode; int crtmode; }
		*uap = (struct a *) u.u_ap;
		register i;

		if (uap->mode & (FWRITE|FTRUNC)) {
		    if (vp->v_vfsp->vfs_flag & VFS_RDONLY) {
			u.u_error = EROFS;
			break;
		    }
		}

		if ((*gift = cr_sndd()) == NULL) {
		    u.u_error = ENOMEM;
		    break;
		}
		rmp->rq_mode = uap->mode;
		/*
		 * Reverse jugglery done here to counter what was done in
		 * sysv_open!
		 */
		i = rmp->rq_mode & (SYSV_BITS << 1);
		rmp->rq_mode &= ~(SYSV_BITS << 1);
		rmp->rq_mode |= (i >> 1);
		rmp->rq_crtmode = uap->crtmode;
		rmp->rq_cmask = u.u_cmask;
		break;
	}
	case DUCREAT:
	{
	    	register struct a {
		    char *fname;
		    int fmode;
		} *uap = (struct a *) u.u_ap;

		if ((*gift = cr_sndd()) == NULL)
		  	u.u_error = ENOMEM;
		else if (vp->v_vfsp->vfs_flag & VFS_RDONLY) {
			free_sndd(*gift);
			u.u_error = EROFS;
		}
		else {
		    rmp->rq_fmode = uap->fmode;
		    rmp->rq_cmask = u.u_cmask;
		}
		break;
	}
	case DULBMOUNT:
		if ((*gift = cr_sndd()) == NULL) {
			u.u_error = ENOMEM;
			return;
		}
		rmp->rq_newmntindx = u.u_mntindx;  /*  set in smount()  */
		break;
	case DUMKDIR:
	{
		register struct a { char *fname; int fmode;  }
		*uap = (struct a *) u.u_ap;
	    	if (vp->v_vfsp->vfs_flag & VFS_RDONLY)
		  u.u_error = EROFS;
		else
		  rmp->rq_fmode = uap->fmode;
		break;
	}
	case DUMKNOD:
	{
	    	register struct a {
		    char *fname;
		    int fmode;
		    dev_t pad;
		    dev_t dev;
		} *uap = (struct a *) u.u_ap;

	    	if (vp->v_vfsp->vfs_flag & VFS_RDONLY)
		  u.u_error = EROFS;
		else {
		    rmp->rq_fmode = uap->fmode;
		    rmp->rq_dev = uap->dev;
		    rmp->rq_cmask = u.u_cmask;
		}
		break;
	}
	case DURMDIR:
		if (rmp->rq_data[0] == '\0')
		  	u.u_error = EBUSY;
		else if (rmp->rq_data[0] == '.' && rmp->rq_data[1] == '\0')
		  	u.u_error = EINVAL;
	    	else if (vp->v_vfsp->vfs_flag & VFS_RDONLY)
			u.u_error = EROFS;
		break;

	case DUMOUNT:
	case DUUMOUNT:
		u.u_error = EREMOTE_SV;
		break;
	case DURMOUNT:
	{
		struct a { char *mntfs; char *mntpt; } *uap=
		  (struct a *) u.u_ap;
		/******************
		u.u_dirp = uap->mntpt;
		******************/
		break;
	}
	case DUSTAT:
	{
		struct a { char *fname; struct sysv_stat *bs; } *uap =
		  (struct a *) u.u_ap;
		rmp->rq_bufptr = (int) uap->bs;
		break;
	}
	case DUSTATFS:
	{
		register struct a {
			char	*fname;
			struct	sysv_statfs *bufp;
			int	len;
			int	fstyp;
		} *uap = (struct a *) u.u_ap;

		rmp->rq_bufptr = (int) uap->bufp;
		rmp->rq_len = uap->len;
		rmp->rq_fstyp = uap->fstyp;
		break;
	}
	case DUUTIME:
	{
		register struct a {
			char *fname;
			time_t *tptr;
		} *uap = (struct a *) u.u_ap;

	    	if (vp->v_vfsp->vfs_flag & VFS_RDONLY)
		  	u.u_error = EROFS;
		else if ((*gift = cr_sndd()) == NULL)
		  	u.u_error = ENOMEM;
		else {
			rmp->rq_bufptr = (int) uap->tptr;
			rmp->rq_prewrite = 0;
		}
		break;
	}

	case DUSYNCTIME:
		rmp->rq_synctime = time.tv_sec;
		break;

	default:
		printf ("rcallmsg: syscall type %d unexpected\n", u.u_procp->p_systemcall);
		u.u_error = EINVAL;
	}
	return;
}


/*  return from remote call.
 *  Put arguments back into this process's u area.
 *  Return argument tells remcall whether to do longjump.
 */

/*static*/
rcallret (in_bp, gift, vnop, out_port, vp)
mblk_t	*in_bp;
sndd_t	gift;
struct vnode **vnop;	
sndd_t	out_port;
register struct vnode *vp;
{
	int jump;
	struct	response *resp;
	register struct pathname *pnp;
	register int residue;

	resp = (struct response *) PTOMSG(in_bp->b_rptr);
	u.u_procp->p_sig |= resp->rp_sig;

	DUPRINT2(DB_SIGNAL,"rcallret:rp_sig=%x\n",resp->rp_sig);
	if (resp->rp_errno != 0) {
		DUPRINT2(DB_SYSCALL, "rcallret: remote u.u_error %d\n",
			resp->rp_errno);
		u.u_error = resp->rp_errno;
		u.u_r.r_val1 = resp->rp_rval;

		if (gift) {
			free_sndd (gift);
		}
		*vnop = NULL;
		freemsg (in_bp);
		return (NOJUMP);
	}

	/* Check for crossing machine boundary (ECROSSMNT). */
	if ((resp->rp_opcode == DUDOTDOT) || (resp->rp_opcode == DULBIN)) {
		pnp = u.u_rfspnp;	/* Set by lookuppn */
		residue = strlen(resp->rp_data);
		pnp->pn_path -= residue;
		pnp->pn_pathlen += residue;
		u.u_rflags |= U_DOTDOT;
		if (resp->rp_opcode == DUDOTDOT) {
			/*
			 * Will the following hold good?
			 */
			*vnop =  rfsmount[resp->rp_mntindx].m_vfsp->vfs_vnodecovered;
		} else {
			*vnop =  mounttab[resp->rp_mntindx].m_devvp;
		}
		VN_HOLD(*vnop);
		if (gift) {
			free_sndd (gift);
		}
		freemsg (in_bp);
		return (NOJUMP);
	}
	jump = JUMP;
	*vnop = NULL;

	if (gift)
		gift->sd_mntindx = resp->rp_mntindx;
		
	switch (u.u_procp->p_systemcall)  {
	case DUEXEC:
	case DUEXECE:
	case DUCHDIR:
	case DUCHROOT:
	case DULBMOUNT:
	case DUOPEN:
	case DUCREAT:
	case DUCOREDUMP:
	case DULINK:
		*vnop = rem_vnode (vp, gift, in_bp);
		jump = NOJUMP;
		break;
	case DUCLOSE:
	case DUIUPDATE:
	case DUUPDATE:
		jump = NOJUMP;
	case DUCHMOD:
	case DUCHOWN:
	case DUMKNOD:
	case DUUNLINK:
	case DURMDIR:
	case DUMKDIR:
		freemsg (in_bp);
		break;

	case DULINK1:
		freemsg (in_bp);
		VN_RELE(u.u_pdir);
		break;			/* Our implementation does long jump */
		
	case DUUTIME:
		free_sndd (gift);
		freemsg (in_bp);
		break;

	case DUACCESS:
	{
#ifndef is68k
		register int	svuid, svgid;
		/*  switch uid, gid with ruid, rgid  */
		svuid = u.u_ruid;
		svgid = u.u_rgid;
		u.u_ruid = u.u_uid;
		u.u_rgid = u.u_gid;
		u.u_uid = svuid;
		u.u_gid = svgid;
#endif
		freemsg (in_bp);
		break;
	}
	case DUSTAT:
	{
		register struct sysv_stat	*statptr;
		register struct gdp	*gdpp;
		extern struct gdp gdp[];

		statptr = (struct sysv_stat *) resp->rp_data;
		gdpp = GDP(out_port->sd_queue);
		hibyte(statptr->st_dev) = ~(gdpp - gdp);
		u.u_error = copyout(resp->rp_data, (caddr_t)resp->rp_bufptr,
					(u_int)sizeof(struct sysv_stat));
		freemsg (in_bp);
		break;
	}
	case DUSTATFS:
		u.u_error =  copyout(resp->rp_data, resp->rp_bufptr,
						(u_int)resp->rp_count);
		freemsg (in_bp);
		break;

	case DUSYNCTIME:
		GDP(out_port->sd_queue)->time = resp->rp_synctime - time.tv_sec;
		jump = NOJUMP;
		freemsg (in_bp);
		break;

	default:
		printf ("rcallret: unknown system returned %d\n",
			u.u_procp->p_systemcall);
		freemsg (in_bp);
				
	}  /*  end switch  */

	return (jump);
}

/*  This should be called only by a server process, in iget() and namei().
 *  A mount point was crossed, return to the server level of the remote
 *  call and send back the mount index and the reason for the crossing.
 *  In the DOT_DOT case, need to back up u.u_nextcp to point to the '..'.
 */

goback (mntindx, why, pnp)
int	mntindx;	/*  index into the srmount index	*/
int	why;		/*  LBIN or DOT_DOT			*/
struct pathname *pnp;	/* Relevant path name			*/
{
	register struct pathname *npnp;

	if (why == DOT_DOT) {	/*  point u.u_nextcp to the first '.'  */
		/*
		 * Back up over "..".
		 */
		while ((*--pnp->pn_path) == '/')
			pnp->pn_pathlen++;
		--pnp->pn_path;
		pnp->pn_pathlen += 2;
		u.u_error |= EDOTDOT;
		npnp = (struct pathname *)kmem_alloc((u_int)sizeof(struct pathname));
		npnp->pn_buf = (char *)kmem_alloc((u_int)MAXPATHLEN);
		npnp->pn_path = npnp->pn_buf;
		npnp->pn_pathlen = pnp->pn_pathlen; /* length remaining */
		(void)strncpy(npnp->pn_buf, pnp->pn_path, pnp->pn_pathlen);
		u.u_rfspnp = npnp;
	} else		  /*  LBIN */
		u.u_error |= ELBIN;
	u.u_mntindx = mntindx;
}


/* Send an interrupt message across the network.  Finds the
 * correct place to send the message from the minwd field (this means
 * this won't work with servers).  Does not wait for a reply because
 * the reply is a natural (albeit) interrupted return from the remote
 * system call.
 */

sendrsig()
{
	register struct message *smsg;
	register struct request *req;
	register mblk_t	*bp;
	register struct sndd *sd;

	while ((bp = alocbuf(sizeof(struct request)-DATASIZE, BPRI_MED)) == NULL) {
		u.u_procp->p_sig = u.u_procp->p_cursig = 0;
	}
	req = (struct request *) PTOMSG(bp->b_rptr);
	sd = (struct sndd *)u.u_procp->p_minwd;
	req->rq_type = REQ_MSG;
	req->rq_opcode = DURSIGNAL;	/* this is a interrupt message */
	req->rq_sysid = get_sysid(sd);
	req->rq_mntindx = sd->sd_mntindx;
	smsg = (struct message *)bp->b_rptr;
	smsg->m_stat |= SIGNAL;
	if (sndmsg(sd, bp, sizeof(struct request) - DATASIZE,
	 			(rcvd_t)NULL) == FAILURE)
		return (FAILURE);
	u.u_procp->p_flag |= SRSIG;  /*so rsc() knows proc is signalled*/	
	return (SUCCESS);
}


/* server() comes here from iget() when ip->i_flag & ILBIN is true */
/* return code: 0: server crossing mount point, go back
		1: continue on this server machine */
/*************
THE FOLLOWING CODE NEEDS TO BE CONVERTED AS NEEDED!!
WE MAY NOT NEED IT AT ALL!!
riget(ip)
register struct inode *ip;
{
	register struct	srmnt	*smp;
	extern struct srmnt *getsrmount();

 	if (smp = getsrmount (ip, u.u_procp->p_sysid))  {
		goback (smp->sr_mntindx, LBIN);
		strcpy (u.u_arg[0], u.u_nextcp);
		return (0);
	}
	return (1);
}
****************/

struct vnode *
rem_vnode (vp, gift, in_bp)
register struct vnode *vp;
register struct sndd *gift;
register mblk_t *in_bp;
{
	register struct rfsnode *rp;
	register struct response *resp;
	struct vnode *makerfsnode();
	struct vnode *vvp;
	struct inode tip;
	ushort mode, ftype;

	resp = (struct response *) PTOMSG(in_bp->b_rptr);
	tip.i_uid = gluid(GDP(gift->sd_queue), resp->rp_uid);
	tip.i_gid = glgid(GDP(gift->sd_queue), resp->rp_gid);
	tip.i_nlink = resp->rp_nlink;
	tip.i_size = resp->rp_size;
	mode = resp->rp_mode;
	ftype = resp->rp_ftype;
	freemsg (in_bp);
	if((vvp = makerfsnode(VTORFS(vp)->rf_mntdev, gift->sd_sindex)) == NULL) {
		del_sndd (gift);
		return (NULL);
	}
	rp = VTORFS(vvp);
	DUPRINT2 (DB_FSS,"rem_vnode: makerfsnode rp = %x\n",rp);
	rp->rf_uid = tip.i_uid;
	rp->rf_gid = tip.i_gid;
	rp->rf_nlink = tip.i_nlink;
	rp->rf_size = tip.i_size;
	rp->rf_ftype = ftype;
	vvp->v_type = IFTOVT(ftype);
	if (rp->rf_fsptr) {
		DUPRINT2 (DB_FSS,"rcallret: cache hit sndd = %x\n",
			rp->rf_fsptr);
                if (gift->sd_sindex!=((struct sndd *)rp->rf_fsptr)->sd_sindex) {
                        printf ("rem_vnode: panic rp = %x fsptr = %x gift %x\n",rp,rp->rf_fsptr,gift);
                        panic ("rem_vnode cache hit error");
                }
		del_sndd (gift);
		if (u.u_error) {
			VN_RELE(vvp);
			return (NULL);
		}
	} else {
		/* populate the independent inode */
		DUPRINT1 (DB_FSS,"rcallret: populate rp \n");
		rp->rf_fsptr = gift;
	}
	if ((u.u_procp->p_systemcall == DUEXEC) || (u.u_procp->p_systemcall == DUEXECE))
		((struct sndd *)(rp->rf_fsptr))->sd_mode = mode;
	return (vvp);
}


duustat()
{
	struct rfsnode rp;
	register struct a { char *cbuf; int dev; int cmd; }
	*uap = (struct a *) u.u_ap;
	extern struct gdp gdp[];
	short dev;
	sndd_t sdp;
	short index;
	queue_t *qp;
	if ((sdp = cr_sndd ()) == NULL) {
		u.u_error = ENOMEM;
		return;
	}
	rp.rf_vnode.v_data = (caddr_t)&rp;
	rp.rf_fsptr = sdp;
	dev = (short) uap->dev;
	index = (short)((~(hibyte (dev))) & 0x00ff) ;
	if(index >= maxgdp || !(gdp[index].flag & GDPCONNECT)) {
		u.u_error = ENOENT;
		free_sndd(sdp);
		return;
	}
	qp = gdp[index].queue;
	DUPRINT3 (DB_FSS,"duustat: qp %x sdp %x\n", qp, sdp);
	set_sndd (sdp, qp, (index_t)CFRD, 0);
	remfileop (RFSTOV(&rp),NULL,NULL);
	free_sndd(sdp);
}



rem_date ()
{
	register struct gdp *tmp;

	DUPRINT1(DB_SYSCALL, "rem_date: starting to update links\n");
	for (tmp = gdp; tmp < &gdp[maxgdp]; tmp++)
		if ((tmp->queue != NULL) && (tmp->flag & GDPCONNECT))
			date_sync (tmp->queue);
}

date_sync (qp)
queue_t *qp;
{
	register sndd_t sd;
	register short tmp;
	struct rfsnode rp;

	if ((sd = cr_sndd ()) == NULL) {
		u.u_error = ENOMEM;
		return;
	}
	set_sndd (sd, qp, (index_t)RECOVER_RD, RECOVER_RD);
	tmp = u.u_procp->p_systemcall;
	u.u_procp->p_systemcall = DUSYNCTIME;
	rp.rf_vnode.v_data = (caddr_t)&rp;
	rp.rf_fsptr = sd;
	(void)remote_call (RFSTOV(&rp), (caddr_t)NULL);
	u.u_procp->p_systemcall = tmp;
	free_sndd (sd);
}
#endif	defined(SYSV) && defined(RFS)
