/*	@(#)vfs_pathname.c 1.1 86/02/03 SMI	*/

#include "param.h"
#include "types.h"
#include "systm.h"
#include "dir.h"
#include "uio.h"
#include "errno.h"
#include "pathname.h"
#include "user.h"
#include "proc.h"
#include "kernel.h"

#ifdef	SYSV
#include "../sysv/sys/fs/s5dir.h"
#ifdef	RFS
#include "../sysv/sys/sysmacros.h"
#endif	RFS
#endif	SYSV


/*
 * Pathname utilities.
 *
 * In translating file names we copy each argument file
 * name into a pathname structure where we operate on it.
 * Each pathname structure can hold MAXPATHLEN characters
 * including a terminating null, and operations here support
 * allocating and freeing pathname structures, fetching
 * strings from user space, getting the next character from
 * a pathname, combining two pathnames (used in symbolic
 * link processing), and peeling off the first component
 * of a pathname.
 */

char *pn_freelist;

/*
 * Allocate contents of pathname structure.
 * Structure itself is typically automatic
 * variable in calling routine for convenience.
 */
pn_alloc(pnp)
	register struct pathname *pnp;
{
	if (pn_freelist) {
		pnp->pn_buf = pn_freelist;
		pn_freelist = *(char **) pnp->pn_buf;
	} else {
		pnp->pn_buf = (char *)kmem_alloc((u_int)MAXPATHLEN);
	}
	pnp->pn_path = (char *)pnp->pn_buf;
	pnp->pn_pathlen = 0;
}

/*
 * Pull a pathname from user user or kernel space
 */
int
pn_get(str, seg, pnp)
	register struct pathname *pnp;
	int seg;
	register char *str;
{
	register int error;

	pn_alloc(pnp);
	if (seg == UIOSEG_USER)
		error = copyinstr(str, pnp->pn_path,
			MAXPATHLEN, &pnp->pn_pathlen);
	else
		error = copystr(str, pnp->pn_path,
			MAXPATHLEN, &pnp->pn_pathlen);
	pnp->pn_pathlen--;			/* dont count null */
	if (error)
		pn_free(pnp);
#ifdef	SYSV
	else if (u.u_procp->p_universe == UNIVERSE_SYSV && !pnp->pn_pathlen
#ifdef	RFS
		&& !server()
#endif	RFS
								) {
		error = ENOENT;
		pn_free(pnp);
	}
#endif	SYSV
	return(error);
}

#ifdef is68k
/*
 * Get next character from a path. Return null at end forever.
 */
pn_getchar(pnp)
	register struct pathname *pnp;
{
	if (pnp->pn_pathlen > 0) {
		pnp->pn_pathlen--;
		return (*pnp->pn_path++);
	} else
		return (0);
}

/*
 * add a character to end of path.
 */
pn_putchar(pnp, c)
	register struct pathname *pnp;
{
	if (pnp->pn_pathlen >= MAXPATHLEN)
		return;
	pnp->pn_path[pnp->pn_pathlen] = c;
	pnp->pn_pathlen++;
}

pn_ungetchar(pnp)
	register struct pathname *pnp;
{
	pnp->pn_path--;
	pnp->pn_pathlen++;
}
#endif	is68k

/*
 * Set pathname to argument string.
 */
pn_set(pnp, path)
	register struct pathname *pnp;
	register char *path;
{
	register int error;

	pnp->pn_path = pnp->pn_buf;
	error = copystr(path, pnp->pn_path, MAXPATHLEN, &pnp->pn_pathlen);
	pnp->pn_pathlen--;		/* don't count null byte */
	return (0);
}

/*
 * Combine two argument pathnames by putting second argument before first in 
 * first's buffer, and freeing second argument. This isn't very general: it is 
 * designed specifically for symbolic link processing.
 */
pn_combine(pnp, sympnp)
	register struct pathname *pnp;
	register struct pathname *sympnp;
{
	if ((pnp->pn_pathlen + sympnp->pn_pathlen) >= MAXPATHLEN)
		return (ENAMETOOLONG);
	ovbcopy(pnp->pn_path, pnp->pn_buf + sympnp->pn_pathlen,
	    (u_int)pnp->pn_pathlen);
	bcopy(sympnp->pn_path, pnp->pn_buf, (u_int)sympnp->pn_pathlen);
	pnp->pn_pathlen += sympnp->pn_pathlen;
	pnp->pn_path = pnp->pn_buf;
	return (0);
}

#ifdef	is68k
/*
 * Expand any macros found in the pathname.
 * NOTE: many of these macros should be done in the context of the client!!
 */
pn_macroexp(pnp, mname, clientserver)
	register struct pathname *pnp;
	char *mname;
{
	register char *p = pnp->pn_path; 
	register char *s;
	register int slen;
	int len, i = pnp->pn_pathlen;

	/* Search path for MACRO references and expaned. */
	while (*p && i) {
	    if (*p == '$') {
		len = 0;
		if (clientserver & 1) {
		    if (strncmp(p, "$BUS", 4) == 0) {
			len = 4;
#ifdef	QBUS
			s = "QBUS";
#else	QBUS
			s = "VME";
#endif	QBUS
		    } else if (strncmp(p, "$CPU", 4) == 0) {
			len = 4;
#ifdef	M68020
			s = "68020";
#else	M68020
			s = "68010";
#endif	M68020
		    } else if (strncmp(p, "$MACHINE", 8) == 0) {
			len = 8;
#ifdef	QBUS
#ifdef	M68020
			s = "Q20";
#else	M68020
			s = "Q10";
#endif	M68020
#else	QBUS
#ifdef	M68020
			s = "V20";
#else	M68020
			s = "V10";
#endif	M68020
#endif	QBUS
		    } else if (strncmp(p, "$UNIVERSE", 9) == 0) {
			len = 9;
			if (u.u_procp->p_universe == UNIVERSE_BSD)
				s = "BSD";
			else if (u.u_procp->p_universe == UNIVERSE_SYSV)
				s = "SYSV";
			else
				s = "UNKNOWN";
		    }
		}
		if ((len == 0) && (clientserver & 2)) {
#ifdef	TRFS
			char *trfs_slink();

			s = trfs_slink(p, &len, mname, u.u_tclient);
#else	TRFS
			if (strncmp(p, "$HOST", 5) == 0) {
				len = 5;
				s = hostname;
			} else if (strncmp(p, "$RHOST", 6) == 0) {
				len = 6;
				s = "";
			} else if (strncmp(p, "$REMOTE", 7) == 0) {
				len = 7;
				s = "";
			}
#endif	TRFS
		}
		if (len) {
			slen = strlen(s);
			if (pnp->pn_pathlen + (slen - len) >= MAXPATHLEN)
				return (ENAMETOOLONG);
			ovbcopy(p + len, p + slen, i - len);
			bcopy(s, p, slen);
			pnp->pn_pathlen += (slen - len);
			p += slen;
			i -= len;
		} else {
			p++;
			i--;
		}
	    } else {
		p++;
		i--;
	    }
	}
}
#endif	is68k

/*
 * Strip next component off a pathname and leave in
 * buffer comoponent which should have room for
 * MAXNAMLEN bytes and a null terminator character.
 */
pn_getcomponent(pnp, component)
	register struct pathname *pnp;
	register char *component;
{
	register char *cp;
	register int l;
	register int n;
	register char c;
#ifdef	SYSV
	char *sysv_component;
	sysv_component = component;
#endif	SYSV


	cp = pnp->pn_path;
	l = pnp->pn_pathlen;
	n = MAXNAMLEN;
	while ((l > 0) && (*cp != '/')) {
		if (--n < 0)
			return(ENAMETOOLONG);
		c = *cp++;
		if (c & 0x80)
			return (EINVAL);
		*component++ = c;
		--l;
	}
#ifdef	is68k
	/* trailing / */
	if (l == 1 && *cp == '/') {
		l = 0;
		*cp = '\0';
	}
#endif	is68k
	pnp->pn_path = cp;
	pnp->pn_pathlen = l;
	*component = 0;
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV) {
		sysv_component += SYSV_DIRSIZ;
		*sysv_component = 0;
	}
#endif	SYSV
	return (0);
}

/*
 * skip over consecutive slashes in the pathname
 */
void
pn_skipslash(pnp)
	register struct pathname *pnp;
{
	while ((pnp->pn_pathlen > 0) && (*pnp->pn_path == '/')) {
		pnp->pn_path++;
		pnp->pn_pathlen--;
	}
}

/*
 * Free pathname resources.
 */
void
pn_free(pnp)
	register struct pathname *pnp;
{
	if (pnp->pn_buf) {
		*(char **) pnp->pn_buf = pn_freelist;
		pn_freelist = pnp->pn_buf;
	}
	pnp->pn_path = pnp->pn_buf = 0;
}

#ifdef	is68k			/* ISI: TRFS: */
/*
 * Make a duplicate of a pathname structure
 */
struct pathname *
pn_dup(pnp)
	register struct pathname *pnp;
{
	register struct pathname *npnp;

	npnp = (struct pathname *)kmem_alloc(sizeof(struct pathname));
	npnp->pn_buf = pnp->pn_buf;
	npnp->pn_path = pnp->pn_path;
	npnp->pn_pathlen = pnp->pn_pathlen;
	return(npnp);
}

/*
 * Free duplicate pathname resources.
 */
void
pn_dfree(pnp)
	register struct pathname *pnp;
{
	pn_free(pnp);
	kmem_free((caddr_t)pnp, sizeof(struct pathname));
}

pn_printf(pnp)
	register struct pathname *pnp;
{
	register char *cp = pnp->pn_path;
	register int l = pnp->pn_pathlen;

	printf(":");
	while (l > 0) {
		printf("%c",*cp++);
		--l;
	}
	printf(":");
}
#endif is68k
