/*
 * Standalone RL02 disk driver
 */
#include "saio.h"
#include "sais68k.h"
#include "../is68kdev/rlreg.h"

static struct rldevice	*rlstd[] = { 	(struct rldevice *)0x3FF900,
					(struct rldevice *)0x3FF910, };
static int		rl02_off[] = { 0, 398, 0, 398, -1, -1, 0, -1 };
static int		rl01_off[] = { 0, 0, 0, 0, -1, -1, 0, -1 };
static struct st 	rlst =	{ 512, 2, 20, 40, 512, rl02_off }; /* RL02 */

#define	UTOCN(x)	((x) >> 3)
#define	UTODR(x)	((x) & 7)

/* 
 * struct to keep state info about the controller
 */
static struct	rl_stat {
	short	rl_dn;		/* drive number */
	short	rl_cylnhd;	/* cylinder and head */
	u_short	rl_bleft;	/* bytes left to transfer */
	u_short	rl_bpart;	/* bytes transferred */
	short	rl_22bits;	/* true if controller allows 22bit addresses */
}		rl_stat[] = { -1, 0, 0, 0};

rlopen(io)
	register struct iob *io;
{
	register int	ctn = UTOCN(io->i_unit); 
	register struct rldevice *rladdr = rlstd[ctn];
	register struct rl_stat *st = &rl_stat[0];
	register int ctr = 0;
	int status;

	if (!probe(&rladdr->rlba,&rladdr->rlba)) {
		printf("rl: controller not present\n");
		return (-1);
	}
	/*
	 * DEC reports that: For some unknown reason the RL02 (seems to be only
	 * drive 1) does not return a valid drive status the first time that a 
	 * GET STATUS request is issued for the drive, in fact it can take up to
	 * three or more GET STATUS requests to obtain the correct status. In 
	 * order to overcome this, the driver has been modified to issue a GET 
	 * STATUS request and validate the drive status returned.  If a valid 
	 * status is not returned after eight attempts, then an error message is
	 * printed.
	 */
	do {
		rladdr->rlba = 0;
		rladdr->rlda.getstat = RL_RESET;
		rladdr->rlcs = (UTODR(io->i_unit) <<8) | RL_GETSTAT;
		if (rlwait(rladdr))
			return (-1);
		status = rladdr->rlmp.getstat;
	} while( (status&RLMP_STATUS) != RLMP_STATOK && ++ctr<8 );
	/* dont mistake an erl for an rl, erl will return # of heads in rlba */
	if (rladdr->rlba != 0) {
		printf("rl: controller not present\n");
		return (-1);
	}
	if ((rladdr->rlcs & RL_DE) || (ctr >= 8)) {
		printf("rl: unit does not respond\n");
		return (-1);
	}
	if ((status & RLMP_DT) == 0 ) {	/* NO RL01'S */
		printf("rl: RL01 unit not supported\n");
		return (-1);
	}
	/* Determine disk posistion */
	rladdr->rlcs = (UTODR(io->i_unit) << 8) | RL_RHDR;
	if (rlwait(rladdr))
		return (-1);
	/* save disk drive posistion */
	st->rl_cylnhd = (rladdr->rlmp.readhdr & 0177700) >> 6;
	st->rl_dn = UTODR(io->i_unit);
	st->rl_22bits = probe(&rladdr->rlbae, &rladdr->rlbae);
	io->i_st = rlst;
	if ((io->i_boff = diskpart(io)) < 0) {
		printf("rl: bad offset\n");
		return (-1);
	}
	return (0);
}

rlstrategy(io, func)
	register struct iob *io;
{
	register int	ctn = UTOCN(io->i_unit); 
	register struct rldevice *rladdr = rlstd[ctn];
	register struct rl_stat *st = &rl_stat[0];
	int com;
	daddr_t bn;
	short cn, sn, head;
	int diff, errcnt = 0;

retry:
	bn = io->i_bn;			/* block number */
	cn = bn / 40;			/* 40 512 byte blocks per cylinder */
	sn = (bn % 20) << 1;
	head = (bn / 20) & 1;
	st->rl_bleft = io->i_cc;	/* total number of bytes to trans */
stupid_rl:
	/* find out how many cylinders to seek */
	diff = (st->rl_cylnhd >> 1) - cn;
	if ( diff == 0 && (st->rl_cylnhd & 1) == head )
		goto noseek;
	/* first time or we switched drives */
	st->rl_dn = UTODR(io->i_unit);	/* drive number */
	if ( diff < 0 )
		rladdr->rlda.seek = -diff<<7 | RLDA_HGH | head << 4;
	else
		rladdr->rlda.seek = diff<<7 | RLDA_LOW | head << 4;
	rladdr->rlcs = (st->rl_dn << 8) | RL_SEEK;
	/* reset position of drive */
	st->rl_cylnhd = (cn << 1) | head;
noseek:
	/* wait for controller and drive */
	while( (rladdr->rlcs & RL_DCRDY) != RL_DCRDY)
		continue;
	/* calculate the max number of bytes we can trans */
	st->rl_bpart = NRLSECT * NRLBPSC - (sn * NRLBPSC);
	if ( st->rl_bleft < st->rl_bpart )
		st->rl_bpart = st->rl_bleft;
	rladdr->rlda.rw = (st->rl_cylnhd << 6) | sn;
	rladdr->rlmp.rw = -(st->rl_bpart >> 1);
	rladdr->rlba = loword(io->i_ma);
	if ( st->rl_22bits )
		rladdr->rlbae = hiword(io->i_ma)&0x3F;
	else if (!addr18((long)io->i_ma)) {
		printf("rl: 22 bit address\n");
		return (-1);
	}
	com = (st->rl_dn << 8) | (((long)io->i_ma>>12)&RL_BAE);
	if (func == READ)
		com |= RL_READ;
	else
		com |= RL_WRITE;
	rladdr->rlcs = com;
	/* wait for controller and drive */
	while( (rladdr->rlcs & RL_DCRDY) != RL_DCRDY)
		continue;
	if (rladdr->rlcs & RL_ERR) {
		int status;
		if ( rladdr->rlcs & RL_DE ) {
			rladdr->rlda.getstat = RL_GSTAT;
			rladdr->rlcs = (st->rl_dn << 8) | RL_GETSTAT;
			if (rlwait(rladdr))
				return (-1);
			status = rladdr->rlmp.getstat;
			rladdr->rlda.getstat = RL_RESET;
			rladdr->rlcs = (st->rl_dn <<8) | RL_GETSTAT;
			if (rlwait(rladdr))
				return (-1);
		}
		printf("rl: error (cyl,head,sec)=(%d,%d,%d) cs=%b mp=%b\n",
		    cn, head, sn, rladdr->rlcs & 0xffff, RLCS_BITS,
		    status, RLER_BITS);
		/* Determine disk posistion */
		rladdr->rlcs = (st->rl_dn << 8) | RL_RHDR;
		if (rlwait(rladdr))
			return (-1);
		/* save disk drive posistion */
		st->rl_cylnhd = (rladdr->rlmp.readhdr & 0177700) >> 6;
		if (errcnt == 10) {
			printf("rl: unrecovered error\n");
			return (-1);
		}
		errcnt++;
		goto retry;
	}
	/* 
	 * do we have to finish off the rest of the transfer?
	 */
	if ( (st->rl_bleft -= st->rl_bpart) > 0 ) {
		/* 
		 * increment head and/or cylinder
		 */
		if ( ++head > 1 ) {
			cn++;		/* want next cyl, head 0 sector 0 */
			head = 0;
		}
		/* 
		 * we always want sector to be zero
		 */
		sn = 0;
		io->i_ma += io->i_cc - st->rl_bleft;
		goto stupid_rl;
	}
	if (errcnt)
		printf("rl: recovered by retry\n");
	return (io->i_cc);
}

static
rlwait(rladdr)
	register struct rldevice *rladdr;
{
	register int i;
	short cs;

	for ( i = 800000; i ; i--)
		if (probe(&rladdr->rlcs, &cs) && (cs & RL_CRDY) != 0)
			break;
	if (i == 0) {
		printf("rl: timeout\n");
		return(-1);
	}
	return (0);
}
