/*
 * Integrated Solutions VME bus interprocessor communications driver.
 */

#include "vb.h"

#if NVB > 0
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "mbuf.h"
#include "buf.h"
#include "protosw.h"
#include "ioctl.h"
#include "socket.h"
#include "vmmac.h"
#include "errno.h"
#include "time.h"
#include "kernel.h"
#include "syslog.h"
#include "uio.h"
#include "../net/if.h"
#include "../net/netisr.h"
#include "../net/route.h"

#ifdef	INET
#include "../netinet/in.h"
#include "../netinet/in_systm.h"
#include "../netinet/in_var.h"
#include "../netinet/ip.h"
#include "../netinet/ip_var.h"
#include "../netinet/if_ether.h"
#endif	INET

#ifdef	NS
#include "../netns/ns.h"
#include "../netns/ns_if.h"
#endif	NS

#ifdef	TRFS
#include "../wipc/wipc.h"
#include "../wipc/wipc_packet.h"
#include "../wipc/wipc_pklink.h"
#define	PKLEN		(BCBBUFSIZE - 32 - 4)
#define	PKBURST		10
#endif	TRFS

#ifdef	ENETFILTER
#include "../is68kif/enet.h"
#endif	ENETFILTER

#include "../is68kif/if_qb.h"
#include "../is68kif/if_pkbuf.h"
#include "../is68kif/if_ring.h"
#include "../is68kif/if_vbreg.h"
#include "../is68kdev/qbvar.h"

/*
 * possible message area address/length tuples
 */
struct vb_addr_tuple {
	struct ma	*ma;		/* physical address of message area */
	int		malen;		/* area length */
} vb_addrs[] = {
	{ (struct ma *)VBPHYS0,	VBSIZE0},
	{ (struct ma *)0, 0}		/* null terminated */
};

int	vbprobe(), vbslave(), vbattach(), vbintr(), vbservice();
int	vbinit(), vbioctl(), vboutput(), vbreset();
#ifdef	TRFS
int	vb_xmit(), vb_relse();
#endif	TRFS
struct	qb_device *vbinfo[NVB];
u_short	*vbstd[NVB+1];			/* will be zero ! */
struct  qb_driver VBdriver = 
	{ vbprobe, vbslave, vbattach, vbstd, "vb", vbinfo, "VB" };
struct bcb	*vbbufferinit();

extern struct ma	vb;		/* virtual address for message area */
extern short		fastclock;
extern int		cvec;		/* autoconf interrupt vector */
extern int		clev;		/* autoconf interrupt level */

/*
 * Net software status per interface.
 * Each interface is referenced by a network interface structure,
 * vs_if, which the routing code uses to locate the interface.
 */
struct	vb_softc {
	struct arpcom	vs_ac;		/* common ethernet structures */
#define	vs_if		vs_ac.ac_if	/* network-visible interface */
#define	vs_addr		vs_ac.ac_enaddr	/* hardware ethernet address */
	char		vs_active;
	char		vs_flags;	/* private flags */
#define		VB_RUNNING	(1<<0)	/*   board is running */
	int		vs_nbuf;	/* number of buffers/message area */
	struct ma	*vs_map;	/* pointer to message area */
	int		vs_malen;	/* size of message area */
	struct ring	*vs_freeringp;	/* pointer to free buffer ring */
	struct ring	*vs_inputringp;	/* pointer to our pending input ring */
	short		vs_enetunit;	/* ENET: unit number for filtering */
	char		vs_enetinit;	/* ENET: interface is initialized */
	char		vs_ix;		/* TRFS: */
} vb_softc[NVB];

#define	MINPKTSIZE	60

/*
 * This routine is called early in kernel startup (before autoconf) to find 
 * and map in message area. Size of map allocated in Sysmap (locore.s)
 * must be >= btop(MAX(vb_addrs[i].malen)).
 */
VB_initialize(top_of_mem)
	u_short *top_of_mem;
{
	register int i, n;
	register struct ma *vbp;
	register int unit = 0;
	register char *p;

	if (vbnum >= NCPU)
		panic("too many clusters");
	for (i = 0, vbp = vb_addrs[i].ma; vbp; i++, vbp = vb_addrs[i].ma)
	    if ((u_short *)vbp >= top_of_mem) {
		/* map memory into kernel virtual space */
		mapin(vbmap, btop((caddr_t)&vb), btop((caddr_t)vbp), 
			btop(vb_addrs[i].malen), PG_V|PG_FOD|PG_KW|PG_VMESTD);

		/* determine if message area */
		if (badaddr((caddr_t)&vb,1) || badaddr(((caddr_t)&vb)+1,1) ||
		    badaddr((caddr_t)&vb,2)	 )
			continue;
		n = vb.ma_icpu.ic_tickle;
		vb.ma_icpu.ic_tickle = n + 1;
		if (vb.ma_icpu.ic_tickle == n)
			continue;

		/* size message area */
		for (n = 0, p = (char *)&vb; n < vb_addrs[i].malen; 
					n += NBPG, p += NBPG)
			if (badaddr(p, 2))
				break;
		vbstd[unit] = (u_short *)&vb;
		vb_softc[unit++].vs_malen = n;
#ifdef	M68020
		/* reserve IPI space */
		iosused(&vme_short[VBINTRBASE], VBINTRLEN);
#endif	M68020
		return;
	    }
	if (vbnum)
	    panic("no cluster message area");
}

vbprobe(map, unit)
	struct ma *map;
{
	register struct vb_softc *vs = &vb_softc[unit];
	register int cluster;

	splhigh();
#ifdef	M68020
	cvec = VBINTVECTOR;
	clev = VBINTLEVEL;
#else	M68020
	cvec = 0;
#endif	M68020
	clevmax = clev_impmax;
	clev_imp = MAX(clev, clev_imp);
	vs->vs_map = map;
	if ((cluster = vbconnect(vs)) < 0) {
		vs->vs_map = 0;
		return 0;
	}
	vs->vs_freeringp = ItoKv(map->ma_icpu.ic_freeringp, struct ring *, map);
	vs->vs_inputringp = ItoKv(map->ma_icpu.ic_inputringp[cluster], 
				struct ring *, map);
	bcopy(map->ma_icpu.ic_ethaddr, vs->vs_addr, sizeof(vs->vs_addr));
	vs->vs_addr[VB_EADDRX] = cluster;
	vs->vs_active = VB_ACTIVE;
	return vs->vs_malen;
}

vbslave()
{
	return 1;
}

/*
 * Interface exists: make available by filling in network interface record. 
 * System will initialize the interface when it is ready to accept packets. 
 */
vbattach(qi)
	register struct qb_device *qi;
{
	register struct vb_softc *vs = &vb_softc[qi->qi_unit];
	register struct ifnet *ifp = &vs->vs_if;

#ifdef	M68010
	fastclock = 3;
#endif	M68010
	printf("	(%e) %mM", vs->vs_addr, vs->vs_malen);
	if (vbnum == 0)
		printf(" %d %dK buffers", vs->vs_nbuf, BCBBUFSIZE/1024);
#ifdef	M68020
	printf("\n\t\t\t\t\tIPI @ shortio 0x%x", VBINTADDR(vbnum));
#endif	M68020
	ifp->if_unit = qi->qi_unit;
	ifp->if_name = "vb";
	ifp->if_mtu = BCBBUFSIZE;
	ifp->if_flags = IFF_BROADCAST;
	ifp->if_init = vbinit;
	ifp->if_output = vboutput;
	ifp->if_ioctl = vbioctl;
	ifp->if_reset = vbreset;
	if_attach(ifp);
#ifdef	TRFS
	vs->vs_ix = IfaceAttach(ifp->if_unit, vb_xmit, vb_relse,
		PKLEN, PKBURST, etherbroadcastaddr, sizeof(vs->vs_addr));
#endif	TRFS
}

/* Reset of interface after completion of initialization or bus reset. */
vbreset(unit)
	int unit;
{
	register struct qb_device *qi;
	register struct vb_softc *vs = &vb_softc[unit];
	register struct bcb *bcbp;

	if (unit >= NVB || (qi = vbinfo[unit]) == 0 || qi->qi_alive == 0)
		return;
	vs->vs_active = VB_INACTIVE;
	if (vs->vs_map) {
		vs->vs_map->ma_icpu.ic_active[vbnum] = VB_INACTIVE;
		while(bcbp = (struct bcb *)vbringget(vs->vs_inputringp))
			vbringput(vs->vs_freeringp, bcbp);
	}
	vb_softc[unit].vs_if.if_flags &= ~IFF_RUNNING;
	vb_softc[unit].vs_flags &= ~VB_RUNNING;
}

/* Initialization of interface: clear pending operations. */
vbinit(unit)
	int unit;
{
	register struct vb_softc *vs = &vb_softc[unit];
	register struct ifnet *ifp = &vs->vs_if;

#ifndef	TRFS
	/* not yet, if address still unknown */
	if (ifp->if_addrlist == (struct ifaddr *)0)
		return;
#endif	TRFS
	if (vs->vs_flags & VB_RUNNING)
		return;
	ifp->if_flags |= IFF_RUNNING;
	vs->vs_flags |= VB_RUNNING;
#ifdef	TRFS
	IfaceUp(vs->vs_ix);
#endif	TRFS
#ifdef	ENETFILTER
	if (!vs->vs_enetinit) {
		struct endevp enp;

		vs->vs_enetinit++;
		enp.end_dev_type = ENDT_10MB;
		enp.end_addr_len = sizeof(vs->vs_addr);
		enp.end_hdr_len = sizeof(struct ether_header);
		enp.end_MTU = ETHERMTU;
		bcopy(vs->vs_addr, enp.end_addr, sizeof(vs->vs_addr));
		bcopy(etherbroadcastaddr, enp.end_broadaddr,
			sizeof(vs->vs_addr));
		vs->vs_enetunit = enetattach(ifp, &enp);
	}
#endif	ENETFILTER
}

/* Called by clock interrupt routine; fake an interprocessor interrupt. */
vbcheck()
{
	register int unit;

	for (unit = 0; unit < NVB; unit++)
		vbintr(unit);
}

/* Interprocessor interrupt; If there is pending input post a software AST */
vbintr(unit)
{
	register struct vb_softc *vs = &vb_softc[unit];

	if (vs->vs_map && !vbringempty(vs->vs_inputringp))
		softcall(vbservice, 0);
}

/* software AST service */
vbservice()
{
	register int unit;
	register struct vb_softc *vs = vb_softc;
	register struct ma *map;
	register struct bcb *bcbp;

	for (unit = 0; unit < NVB; unit++, vs++) {
	    if ((map = vs->vs_map) == 0)
		continue;
	    while (bcbp = (struct bcb *)vbringget(vs->vs_inputringp))
		if (vbread(vs, ItoKv(bcbp, struct bcb *, map)) == 0)
		    vbringput(vs->vs_freeringp, bcbp);
	}
}

vbread(vs, bcbp)
	register struct vb_softc *vs;
	register struct bcb *bcbp;
{
	register struct ifnet *ifp = &vs->vs_if;
	register struct ether_header *eh;
	register struct mbuf *m;
	int len, off, resid, s;
	register struct ifqueue *inq;
	char	*bp;

	ifp->if_ipackets++;
	if (bcbp->b_addr == 0) {
		if (ifp->if_ierrors++ % 100 == 0)
			log(LOG_WARNING, "vb: 100 receive errors\n");
		return 0;
	}
	eh = ItoKv(bcbp->b_addr, struct ether_header *, vs->vs_map);
	len = bcbp->b_len - sizeof(struct ether_header);
	eh->ether_type = ntohs((u_short)eh->ether_type);
#define vbdataaddr(eh, off, type)	((type)(((caddr_t)((eh)+1)+(off))))
	if (eh->ether_type >= ETHERTYPE_TRAIL &&
	    eh->ether_type < ETHERTYPE_TRAIL+ETHERTYPE_NTRAILER) {
		if ((off = (eh->ether_type - ETHERTYPE_TRAIL)*512) >= BCBBUFSIZE)
			return 0;			/* sanity */
		eh->ether_type = ntohs(*vbdataaddr(eh, off, u_short *));
		resid = ntohs(*(vbdataaddr(eh, off+2, u_short *)));
		if (off + resid > len)
			return 0;			/* sanity */
		len = off + resid;
	} else
		off = 0;
	if (len == 0)
		return 0;

#ifdef	TRFS
	if (eh->ether_type == ETHERTYPE_WIPC)
		return !vb_recv(vs - vb_softc, bcbp);
#endif	TRFS

	/*
	 * Pull packet off interface. Off is nonzero if packet has trailing 
	 * header; if_rqbget will then force this header information to be at 
	 * the front and drop the type and length at the head of trailers.
	 */
	if ((m=if_rqbget(eh, len, off, sizeof(struct ether_header), ifp)) == 0)
		return 0;

	switch (eh->ether_type) {
#ifdef INET
	    case ETHERTYPE_IP:
		schednetisr(NETISR_IP);
		inq = &ipintrq;
		break;

	    case ETHERTYPE_ARP:
		arpinput(&vs->vs_ac, m);
		return 0;
#endif
#ifdef NS
	    case ETHERTYPE_NS:
		schednetisr(NETISR_NS);
		inq = &nsintrq;
		break;

#endif
	    default:
#ifdef	ENETFILTER
		if (vs->vs_enetinit && vs->vs_enetunit >= 0) {
			register struct mbuf *mtop;

			/*
			 * We need the local net header after all. Oh well,
			 * this could be improved.
			 */
			MGET(mtop, M_DONTWAIT, MT_DATA);
			if (mtop) {
				/* undo the swap done earlier */
				eh->ether_type = htons((u_short)eh->ether_type);
				bcopy((struct ether_header*)eh->ether_dhost,
					mtod(mtop, struct ether_header *),
					sizeof(struct ether_header));
				mtop->m_len = sizeof(struct ether_header);
				/* remove interface pointer */
				m->m_len -= sizeof(ifp);
				m->m_off += sizeof(ifp);
				mtop->m_next = m;
				enetFilter(vs->vs_enetunit, mtop,
					(len + sizeof(struct ether_header)) );
				return 0;
			}
		}
#endif	ENETFILTER
		m_freem(m);
		return 0;
	}

	s = splimp();
	if (IF_QFULL(inq)) {
		IF_DROP(inq);
		m_freem(m);
	} else
		IF_ENQUEUE(inq, m);
	splx(s);
	return 0;
}

/*
 * Net output routine. Encapsulate a packet of type family for the local net.
 * Use trailer local net encapsulation if enough data in first packet leaves 
 * a multiple of 512 bytes of data in remainder.
 */
vboutput(ifp, m0, dst)
	register struct ifnet *ifp;
	struct mbuf *m0;
	struct sockaddr *dst;
{
	register struct vb_softc *vs = &vb_softc[ifp->if_unit];
	register struct mbuf *m = m0;
	register struct ether_header *eh;
	register int off;
	int type, s, error;
	u_char edst[6];
	struct in_addr idst;
	int usetrailers;

	switch( dst->sa_family ) {
#ifdef INET
	    case AF_INET:
		idst = ((struct sockaddr_in *)dst)->sin_addr;
		if (!arpresolve(&vs->vs_ac, m, &idst, edst, &usetrailers))
			return 0;	/* if not yet resolved */
		off = ((u_short)mtod(m, struct ip *)->ip_len) - m->m_len;
		if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		    m->m_off >= MMINOFF + 2 * sizeof (u_short)) {
			type = ETHERTYPE_TRAIL + (off>>9);
			m->m_off -= 2 * sizeof(u_short);
			m->m_len += 2 * sizeof(u_short);
			*mtod(m, u_short *) = htons((u_short)ETHERTYPE_IP);
			*(mtod(m, u_short *) + 1) = htons((u_short)m->m_len);
			goto gottrailertype;
		}
		type = ETHERTYPE_IP;
		off = 0;
		goto gottype;
#endif
#ifdef NS
	    case AF_NS:
		type = ETHERTYPE_NS;
		bcopy((caddr_t)&(((struct sockaddr_ns *)dst)->sns_addr.x_host),
			edst, sizeof (edst));
		off = 0;
		goto gottype;
#endif
	    case AF_IMPLINK:
		eh = mtod(m, struct ether_header *);
		goto gotheader;

	    case AF_UNSPEC:
		eh = (struct ether_header *)dst->sa_data;
		bcopy(eh->ether_dhost, edst, sizeof (edst));
		type = eh->ether_type;
		goto gottype;

	    default:
		log(LOG_WARNING, "vb%d: can't handle af%d\n", ifp->if_unit,
			dst->sa_family);
		error = EAFNOSUPPORT;
		goto bad;
	}

gottrailertype:
	/*
	 * Packet to be sent as trailer: move first packet
	 * (control information) to end of chain.
	 */
	while (m->m_next)
		m = m->m_next;
	m->m_next = m0;
	m = m0->m_next;
	m0->m_next = 0;
	m0 = m;

gottype:
	/* Add local net header. If no space in mbuf, allocate another */
	if (m->m_off > MMAXOFF ||
	    MMINOFF + sizeof(struct ether_header) > m->m_off) {
		m = m_get(M_DONTWAIT, MT_DATA);
		if (m == 0) {
			error = ENOBUFS;
			goto bad;
		}
		m->m_next = m0;
		m->m_off = MMINOFF;
		m->m_len = sizeof(struct ether_header);
	} else {
		m->m_off -= sizeof(struct ether_header);
		m->m_len += sizeof(struct ether_header);
	}
	eh = mtod(m, struct ether_header *);
	eh->ether_type = htons((u_short)type);
	bcopy(edst, eh->ether_dhost, sizeof (edst));

gotheader:
	bcopy(vs->vs_addr, eh->ether_shost, sizeof(vs->vs_addr));

	/* transfer message to destination input queue if possible */
	if (vbput(vs, m)) {
/*		printf("vb%d: constipated transmitter\n", ifp->if_unit); /**/
		IF_DROP(&ifp->if_snd);
		m_freem(m);
		return (ENOBUFS);
	}
	vs->vs_if.if_opackets++;
	return 0;

bad:	m_freem(m0);
	return error;
}

/* Routine to copy from mbuf chain to a message area buffer. */
vbput(vs, m)
	register struct vb_softc *vs;
	struct mbuf *m;
{
	register struct ma *map = vs->vs_map;
	register struct bcb *bcbp;
	register u_char *bp;

	if ((bcbp = (struct bcb *)vbringget(vs->vs_freeringp)) == NULL)
		return 1;
	ItoK(bcbp, struct bcb *, map);
	bp = ItoKv(bcbp->b_addr, u_char *, map);
	bcbp->b_len = if_wqbput(bp, m, 0);
	bcbp->b_len = MAX(bcbp->b_len, MINPKTSIZE);
	vbxmit(vs, bcbp, ICPUX(bp));
	return 0;
}

/* link a message area buffer onto its destination's input queue */
vbxmit(vs, bcbp, x)
	register struct vb_softc *vs;
	register struct bcb *bcbp;
	register int x;
{
	register struct ring *recieve_ringp;
	register struct ma *map = vs->vs_map;
	register struct bcb *bcbp0;
	register u_char *bp;

	if (x < 0 || x >= NCPU) {		/* simulate broadcast */
	    bp = ItoKv(bcbp->b_addr, u_char *, map);
	    for (x = NCPU - 1; x > 0; x--) {
		if (map->ma_icpu.ic_active[x] == VB_INACTIVE ||
		    map->ma_icpu.ic_inputringp[x] == NULL)
			continue;
		if ((bcbp0 = (struct bcb *)vbringget(vs->vs_freeringp)) == NULL) {
/*			printf("vb: constipated broadcast to %d\n", x); /**/
			continue;
		}
		ItoK(bcbp0, struct bcb *, map);
		bcbp0->b_len = bcbp->b_len;
		bcopy(bp, ItoKv(bcbp0->b_addr, u_char *, map), bcbp->b_len);
		if (vbringput(
		    ItoKv(map->ma_icpu.ic_inputringp[x], struct ring *, map),
		    KtoIv(bcbp0, struct bcb *, map)) == 1)
			vbipi(map, x);
	    }
	}
	if (x > 0 && map->ma_icpu.ic_active[x] == VB_INACTIVE)
	    recieve_ringp = vs->vs_freeringp;
	else
	    recieve_ringp = 
		    ItoKv(map->ma_icpu.ic_inputringp[x], struct ring *, map);
	if (vbringput(recieve_ringp, KtoIv(bcbp, struct bcb *, map)) == 1)
	    vbipi(map, x);
}

/* issue an interprocessor interrupt to cluster x */
vbipi(map, x)
	register struct ma *map;
{
	register int ipi = (int)map->ma_icpu.ic_intrinfo[x];

	if (ipi && (x != vbnum))
	    if (badaddr((u_short *)&vme_short[ipi], 2))
		printf("vb: cluster %d's IPI is NOT at shortio %x!\n", x, ipi);
}

/*
 * Process an ioctl request. this can be called via the "socket" route 
 * for SIOCSIFADDR or by the cdev/inode route for SIOCSIFCCFWR/RD
 */
vbioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	register struct ifaddr *ifa = (struct ifaddr *)data;
	struct vb_softc *vs = &vb_softc[ifp->if_unit];
	int s = splimp(), error = 0;

	switch (cmd) {
	    case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;
		vbinit(ifp->if_unit);

		switch (ifa->ifa_addr.sa_family) {
#ifdef INET
		    case AF_INET:
			((struct arpcom *)ifp)->ac_ipaddr =
				IA_SIN(ifa)->sin_addr;
			arpwhohas((struct arpcom *)ifp, &IA_SIN(ifa)->sin_addr);
			break;
#endif
#ifdef NS
		    case AF_NS:
		    {
			register struct ns_addr *ina = &(IA_SNS(ifa)->sns_addr);
			
			if (ns_nullhost(*ina))
				ina->x_host = *(union ns_host *)(vs->vs_addr);
/*			else
				vbsetaddr(ina->x_host.c_host,ifp->if_unit); /**/
			break;
		    }
#endif
		}
		break;

	    case SIOCSIFFLAGS:
		if ((ifp->if_flags & IFF_UP)==0 && vs->vs_flags & VB_RUNNING) {
			vs->vs_flags &= ~VB_RUNNING;
#ifdef	TRFS
			IfaceDown(vs->vs_ix);
#endif	TRFS
		} else if (ifp->if_flags & IFF_UP && 
		    (vs->vs_flags & VB_RUNNING) == 0)
			vbinit(ifp->if_unit);
		break;

	    default:
		error = EINVAL;
	}
	splx(s);
	return error;
}

/* connect to message area, initialize it if we are cluster 0 */
vbconnect(vs)
	register struct vb_softc *vs;
{
	register struct ma *map = vs->vs_map;
	register int i;

	if (vbnum)
		return vbjoin(map);
	else
		return vbinitialize(vs);
}

/* cluster client initilization */
vbjoin(map)
	register struct ma *map;
{
	register int i;

	/* 
	 * set active[0] to VB_INACTIVE, and wait for cluster server to set 
	 * active[0] to VB_ACTIVE, indicating that the message area is 
	 * initialized.
	 */
	map->ma_icpu.ic_active[0] == VB_INACTIVE;
	for (i = VB_DELAY; i >= 0; i--)
	    if (map->ma_icpu.ic_active[0] == VB_ACTIVE)
		if (tas(&map->ma_icpu.ic_active[vbnum])) {
			map->ma_icpu.ic_active[vbnum] = VB_ACTIVE;
			map->ma_icpu.ic_intrinfo[vbnum] = VBINTADDR(vbnum);
			return vbnum;
		} else {
			printf("vb: more than one cluster %d\n", vbnum);
			panic("cluster number in use");
		}
	return -1;
}

/* cluster server initilization */
vbinitialize(vs)
	register struct vb_softc *vs;
{
	register struct ma *map = vs->vs_map;
	static char vb_eaddr[6] = VB_EADDR;
	register int i;

	for (i = 1; i < MAXCPU; i++) {
		map->ma_icpu.ic_active[i] = VB_ACTIVE;
		map->ma_icpu.ic_inputringp[i] = (struct ring *)0;
		map->ma_icpu.ic_intrinfo[i] = 0;
	}
	map->ma_icpu.ic_ncpu = NCPU;
	bcopy(vb_eaddr, map->ma_icpu.ic_ethaddr, sizeof(vb_eaddr));

	/* initialize free ring */
	vbringinit(&map->ma_freering, MAXCPURING);
	map->ma_icpu.ic_freeringp = 
			KtoIv(&map->ma_freering, struct ring *, map);

	/* initialize input rings */
	for (i = 0; i < NCPU; i++) {
		vbringinit(&map->ma_inputring[i], MAXRING);
		map->ma_icpu.ic_inputringp[i] = 
			KtoIv(&map->ma_inputring[i], struct ring *, map);
	}

	/* put all buffers on freering */
	for (i = 0; KtoIv(&map->ma_bufs[i+1], int, map) < vs->vs_malen; i++ )
		vbringput(&map->ma_freering, vbbufferinit(map, &map->ma_bufs[i]));
	vs->vs_nbuf = i;

	for (i = 1; i < NCPU; i++)
		map->ma_icpu.ic_active[i] = VB_INACTIVE;
	vbtictoc(hz/20);		/* 20 times a sec */
	map->ma_icpu.ic_intrinfo[0] = VBINTADDR(0);
	return 0;
}

struct bcb *
vbbufferinit(map, bp)
	register struct ma *map;
	register struct bufs *bp;
{
	bp->buf_head.b_link = (struct bcb *)0;
	bp->buf_head.b_stat = 0;
	bp->buf_head.b_len = BCBBUFSIZE;
	bp->buf_head.b_addr = KtoIv(&bp->buf[0], char *, map);
	bp->buf_head.b_msglen = 0;
	bp->buf_head.b_reserved = 0;
	return KtoIv(bp, struct bcb *, map);
}

/*
 * vbtictoc -- called every tic-th clock interrupt. For each active 
 * message area, set ic_active[0] to vs_active. This routine is executed 
 * by the cluster server, and is used to allow other clusters to detect 
 * that the message area has been initialized.
 */
vbtictoc(tic)
	register int tic;
{
	register struct vb_softc *vs;

	for (vs = vb_softc; vs < &vb_softc[NVB]; vs++)
		if (vs->vs_map)
			vs->vs_map->ma_icpu.ic_active[0] = vs->vs_active;
	timeout(vbtictoc, tic, tic);
}

#ifdef	TRFS
vb_xmit(unit, naddr, p, len, fp0, fp1)
	int unit;
	struct ether_addr *naddr;
	struct packet *p;
	long len;
	struct fragment *fp0, *fp1;
{
	register struct bcb *bcbp;
	register struct ether_header *eh;
	register struct vb_softc *vs = &vb_softc[unit];
	register struct ma *map = vs->vs_map;

	if ((bcbp = (struct bcb *)vbringget(vs->vs_freeringp)) == NULL)
		return EBUSY;
	ItoK(bcbp, struct bcb *, map);
	eh = ItoKv(bcbp->b_addr, struct ether_header *, map);
	bcopy(naddr, eh->ether_dhost, sizeof(eh->ether_dhost));
	bcopy(vs->vs_addr, eh->ether_shost, sizeof (eh->ether_shost));
	eh->ether_type = ETHERTYPE_WIPC;
	bcbp->b_len = (sizeof(struct pklink) - sizeof(struct packet)) +
		pkbuild(&((struct pklink *)eh)->packet, p, len, fp0, fp1);
	bcbp->b_len = MAX(MINPKTSIZE, bcbp->b_len);
	vbxmit(vs, bcbp, ICPUX(eh));
	vs->vs_if.if_opackets++;
	return 0;
}

vb_recv(unit, bcbp)
	int unit;
	register struct bcb *bcbp;
{
	register struct vb_softc *vs = &vb_softc[unit];
	register struct pklink *p;

	p = ItoKv(bcbp->b_addr, struct pklink *, vs->vs_map);
	ovbcopy(((struct ether_header *)p)->ether_shost, p->netaddr, 6);
	p->buflen = BCBBUFSIZE;
	p->prelen = 0;
	p->ix = vs->vs_ix;
	p->handle = (int)bcbp;
	return ReceivePacket(p);
}

vb_relse(unit, p)
	int unit;
	struct pklink *p;
{
	register struct vb_softc *vs = &vb_softc[unit];

	vbringput(vs->vs_freeringp, KtoIv(p->handle, struct bcb *, vs->vs_map));
}
#endif	TRFS
#endif
