/*      enet.c	Stanford	25 April 1983 */

/*
 *  Ethernet packet filter layer,
 *  	formerly: Ethernet interface driver
 *
 **********************************************************************
 * HISTORY
 * 17 July 1984	Jeff Mogul	Stanford
 *	Bug fix: in enetwrite(), several problems with sleeping on
 *	IF_QFULL:
 *	- don't do it for kernel mode writes.
 *	- count # of procs sleeping, to avoid lost wakeups.  Old
 *		scheme would only wake up the first sleeper.
 *	- using sleeper-count, avoid using more than one timeout
 *		table entry per device; old scheme caused timeout table panics
 *	- trap interupted sleeps using setjmp, so that we can deallocate
 *		packet header and mbufs; otherwise we lost them and panicked.
 *
 * 5 July 1984	Jeff Mogul	Stanford
 *	Bug fix: in enetwrite() make sure enP_RefCount is zero before
 *	deallocating "packet".  Otherwise, "packets" get lost, and
 *	take mbufs (and ultimately, the system) with them.
 *
 * 8 December 1983	Jeffrey Mogul	Stanford
 *	Fixed bug in enetwrite() that eventually caused	allocator
 *	to run out of packets and panic.  If enetwrite() returns
 *	an error it should first deallocate any packets it has allocated.
 *
 * 10 November 1983	Jeffrey Mogul	Stanford
 *	Slight restructuring for support of 10mb ethers;
 *	- added the EIOCDEVP ioctl
 *	- removed the EIOCMTU ioctl (subsumed by EIOCDEVP)
 *	This requires an additional parameter to the enetattach
 *	call so that the device driver can specify things.
 *
 *	Also, cleaned up the enDebug scheme by adding symbolic
 *	definitions for the bits.
 *
 * 25-Apr-83	Jeffrey Mogul	Stanford
 *	Began conversion to 4.2BSD.  This involves removing all
 *		references to the actual hardware.
 *	Changed read/write interface to use uio scheme.
 *	Changed ioctl interface to "new style"; this places a hard
 *		limit on the size of a filter (about 128 bytes).
 *	"Packets" now point to mbufs, not private buffers.
 *	Filter can only access data in first mbuf (about 50 words worst case);
 *		this is long enough for all Pup purposes.
 *	Added EIOCMTU ioctl to get MTU (max packet size).
 *	Added an enetselect() routine and other select() support.
 *	Other stuff is (more or less) left intact.
 *	Most previous history comments removed.
 *	Changed some names from enXXXX to enetXXXX to avoid confusion(?)
 *
 * 10-Aug-82  Mike Accetta (mja) at Carnegie-Mellon University
 *	Added new EIOCMBIS and EIOCMBIC ioctl calls to set and clear
 *	bits in mode word;  added mode bit ENHOLDSIG which suppresses
 *	the resetting of an enabled signal after it is sent (to be
 *	used inconjunction with the SIGHOLD mechanism);  changed
 *	EIOCGETP to zero pad word for future compatibility;  changed enwrite()
 *	to enforce correct source host address on output packets (V3.05e).
 *	(Stanford already uses long timeout value and has no pad word - JCM)
 *	[Last change before 4.2BSD conversion starts.]
 *
 * 01-Dec-81  Mike Accetta (mja) at Carnegie-Mellon University
 *	Fixed bug in timeout handling caused by missing "break" in the
 *	"switch" state check within enetread().  This caused all reads
 *	to be preceeded by a bogus timeout.  In addition, fixed another
 *	bug in signal processing by also recording process ID of
 *	process to signal when an input packet is available.  This is
 *	necessary because it is possible for a process with an enabled
 *	signal to fork and exit with no guarantee that the child will
 *	reenable the signal.  Thus under appropriately bizarre race
 *	conditions, an incoming packet to the child can cause a signal
 *	to be sent to the unsuspecting process which inherited the
 *	process slot of the parent.  Of course, if the PID's wrap around
 *	AND the inheriting process has the same PID, well ... (V3.03d).
 *
 * 22-Feb-80  Rick Rashid (rfr) at Carnegie-Mellon University
 *	Rewritten to provide multiple user access via user settable
 *	filters (V1.05).
 *
 * 18-Jan-80  Mike Accetta (mja) at Carnegie-Mellon University
 *      Created (V1.00).
 *
 **********************************************************************
 */

#include "il.h"
#include "ex.h"
#include "nw.h"

/* number of potential units */
#define	NENET	4	/* (Nec + Nen + Nil + Nde + Nex) */

#ifdef	ENETFILTER
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/mbuf.h"
#include "../h/buf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/ioctl.h"
#include "../h/map.h"
#include "../machine/pte.h"
#include "../h/proc.h"
#include "../h/vnode.h"
#include "../h/file.h"
#include "../h/tty.h"
#include "../h/uio.h"

#include "../h/protosw.h"
#include "../h/socket.h"
#include "../net/if.h"

#undef	queue
#undef	dequeue
#define	queue	Queue
#include "../is68kif/enet.h"
#include "../is68kif/enetdefs.h"

#define	enprintf(flags)	if (enDebug&(flags)) printf

/*
 * Symbolic definitions for enDebug flag bits
 */

#define	ENDBG_TRACE	1	/* trace most operations */
#define	ENDBG_DESQ	2	/* trace descriptor queues */
#define	ENDBG_INIT	4	/* initialization info */
#define	ENDBG_SCAV	8	/* scavenger operation */
#define	ENDBG_ABNORM	16	/* abnormal events */

#define DEBUG	ENDBG_ABNORM 	/* standard debug level */
/*#define DEBUG	ENDBG_TRACE|ENDBG_DESQ|ENDBG_INIT|ENDBG_SCAV|ENDBG_ABNORM /**/

#define min(a,b)        ( ((a)<=(b)) ? (a) : (b) )

#define	splenet	splimp	/* used to be spl6 but I'm paranoid */

#define PRINET  26			/* interruptible */

/*
 *  'enQueueElts' is the pool of packet headers used by the driver.
 *  'enPackets'   is the pool of packets used by the driver (these should
 *	          be allocated dynamically when this becomes possible).
 *  'enFreeq'     is the queue of available packets
 *  'enState'     is the driver state table per logical unit number
 *  'enUnit'  	  is the physical unit number table per logical unit number;
 *		  the first "attach"ed ethernet is logical unit 0, etc.
 *  'enFreeqMin'  is the minimum number of packets ever in the free queue
 *		  (for statistics purposes)
 *  'enAllocCount'is the number of packets allocated to external kernel
 *		  applications
 *  'enScavenges' is the number of scavenges of the active input queues
 *		  (for statustics purposes)
 *  'enDebug'	  is a collection of debugging bits which enable trace and/or
 *		  diagnostic output as defined above (ENDBG_*)
 *  'enUnits'	  is the number of attached units
 */
struct enPacket	enQueueElts[ENPACKETS];
struct enQueue	enFreeq;
struct enState  enState[NENET];
int		enFreeqMin = ENPACKETS;
int		enAllocCount = 0;
int		enScavenges = 0;
int		enDebug = DEBUG;
int		enUnits = 0;

/*
 *  Forward declarations for subroutines which return other
 *  than integer types.
 */
extern boolean enDoFilter();


/*
 * Linkages to if_en.c
 */

struct enet_info {
	struct	ifnet *ifp;	/* which ifp for output */
} enet_info[NENET];

struct sockaddr enetaf = { AF_IMPLINK };


/****************************************************************
 *								*
 *		Various Macros & Routines			*
 *								*
 ****************************************************************/

/*
 *  forAllOpenDescriptors(p) -- a macro for iterating
 *  over all currently open devices.  Use it in place of
 *      "for ( ...; ... ; ... )"
 *  and supply your own loop body.  The loop variable is the
 *  parameter p which is set to point to the descriptor for
 *  each open device in turn.
 */

#define forAllOpenDescriptors(p)					\
	for ((p) = (struct enOpenDescriptor *)enDesq.enQ_F;		\
	      (struct Queue *)(&enDesq) != &((p)->enOD_Link);		\
	      (p) = (struct enOpenDescriptor *)(p)->enOD_Link.F)

/*
 *  enInitQueue - initialize ethernet queue
 */

#define	enInitQueue(q)							\
{									\
	initqueue((struct queue *)(q));					\
	(q)->enQ_NumQueued = 0;						\
}



/*
 *  enEnqueue - add an element to a queue
 */

#define	enEnqueue(q, elt)						\
{									\
	enqueue((struct queue *)(q), (struct queue *)(elt));		\
	(q)->enQ_NumQueued++;						\
}



/*
 *  enFlushQueue - release all packets from queue, freeing any
 *  whose reference counts drop to 0.  Assumes caller
 *  is at high IPL so that queue will not be modified while
 *  it is being flushed.
 */

enFlushQueue(q)
register struct enQueue *q;
{

    register struct enPacket *qelt;

    while((qelt=(struct enPacket *)dequeue((struct queue *)q)) != NULL)
    {
	if (0 == --(qelt->enP_RefCount))
	{
	    enEnqueue(&enFreeq, qelt);
	}
    }
    q->enQ_NumQueued = 0;

}

/*
 *  enInitWaitQueue - initialize an empty packet wait queue
 */

enInitWaitQueue(wq)
register struct enWaitQueue *wq;
{			

    wq->enWQ_Head = 0;
    wq->enWQ_Tail = 0;
    wq->enWQ_NumQueued = 0;
    wq->enWQ_MaxWaiting = ENDEFWAITING;

}



/*
 *  enEnWaitQueue - add a packet to a wait queue
 */

enEnWaitQueue(wq, p)
register struct enWaitQueue *wq;
struct enPacket *p;
{

    wq->enWQ_Packets[wq->enWQ_Tail] = p;
    wq->enWQ_NumQueued++;
    enNextWaitQueueIndex(wq->enWQ_Tail);

}



/*
 *  enDeWaitQueue - remove a packet from a wait queue
 */

struct enPacket *
enDeWaitQueue(wq)
register struct enWaitQueue *wq;
{

    struct enPacket *p;

    wq->enWQ_NumQueued--;
    if (wq->enWQ_NumQueued < 0)
	panic("enDeWaitQueue"); 
    p = wq->enWQ_Packets[wq->enWQ_Head];
    enNextWaitQueueIndex(wq->enWQ_Head);

    return(p);

}



/*
 *  enTrimWaitQueue - cut a wait queue back to size
 */
enTrimWaitQueue(wq, threshold)
register struct enWaitQueue *wq;
{

    register int Counter = (wq->enWQ_NumQueued - threshold);
    register struct enPacket *p;

#ifdef	DEBUG
    enprintf(ENDBG_SCAV)
    		("enTrimWaitQueue(%x, %d): %d\n", wq, threshold, Counter);
#endif
    while (Counter-- > 0)
    {
	wq->enWQ_NumQueued--;
	enPrevWaitQueueIndex(wq->enWQ_Tail);
	p = wq->enWQ_Packets[wq->enWQ_Tail];
	if (0 == --(p->enP_RefCount))
	{
	    m_freem(p->enP_mbuf);
	    enEnqueue(&enFreeq, p);
	}
    }
}



/*
 *  enFlushWaitQueue - remove all packets from wait queue
 */

#define	enFlushWaitQueue(wq)	enTrimWaitQueue(wq, 0)

/*
 *  scavenging thresholds:
 *
 *  index by number of active files;  for N open files, each queue may retain
 *  up to 1/Nth of the packets not guaranteed to be freed on scavenge.  The
 *  total number of available packets is computed less any which are currently
 *  allocated to other kernel applications and one for sending.
 *
 *  (assumes high IPL)
 */
char enScavLevel[(NENET*ENMAXOPENS)+1];

/*
 *  enInitScavenge -- set up ScavLevel table
 */
enInitScavenge()
{
    register int PoolSize = (ENPACKETS-enAllocCount-ENMINSCAVENGE);
    register int i = (NENET*ENMAXOPENS)+1;
    register struct enState *enStatep;

    PoolSize--;		/* leave one for transmitter */
    while (--i>0)
        enScavLevel[i] = (PoolSize / i);
}

/*
 *  enScavenge -- scan all OpenDescriptors for all ethernets, releasing
 *    any queued buffers beyond the prescribed limit and freeing any whose
 *    refcounts drop to 0.
 *    Assumes caller is at high IPL so that it is safe to modify the queues.
 */
enScavenge()
{

    register struct enOpenDescriptor *d;
    register int threshold = 0;
    register struct enState *enStatep;

    for (enStatep=enState; enStatep < &enState[NENET]; enStatep++)
	threshold += enCurOpens;
    threshold = enScavLevel[threshold];

    /* recalculate thresholds based on current allocations */
    enInitScavenge();

    enScavenges++;
#ifdef	DEBUG
    enprintf(ENDBG_SCAV)("enScavenge: %d\n", threshold);
#endif
    for (enStatep=enState; enStatep < &enState[NENET]; enStatep++)
    {
	if (enDesq.enQ_F == 0)
	    continue;			/* never initialized */
	forAllOpenDescriptors(d)
	{
	    enTrimWaitQueue(&(d->enOD_Waiting), threshold);
	}
    }

}



/*
 *  enAllocatePacket - allocate the next packet from the free list
 *
 *  Assumes IPL is at high priority so that it is safe to touch the
 *  packet queue.  If the queue is currently empty, scavenge for
 *  more packets.
 */

struct enPacket *
enAllocatePacket()
{

    register struct enPacket *p;

    if (0 == enFreeq.enQ_NumQueued)
	enScavenge();
    p = (struct enPacket *)dequeue((struct queue *)&enFreeq);
    if (p == NULL)
	panic("enAllocatePacket");
    if (enFreeqMin > --enFreeq.enQ_NumQueued)
	enFreeqMin = enFreeq.enQ_NumQueued;

    p->enP_RefCount = 0;	/* just in case */

    return(p);

}



/*
 *  enDeallocatePacket - place the packet back on the free packet queue
 *
 *  (High IPL assumed).
 */

#define	enDeallocatePacket(p)						\
{									\
	if (p->enP_RefCount) panic("enDeallocatePacket: refcount != 0");\
	enqueue((struct queue *)&enFreeq, (struct queue *)(p));		\
	enFreeq.enQ_NumQueued++;					\
}

/****************************************************************
 *								*
 *	    Routines to move uio data to/from mbufs		*
 *								*
 ****************************************************************/

/*
 * These two routines were inspired by/stolen from ../sys/uipc_socket.c
 *	   Both return error code (or 0 if success).
 */

/*
 * read: return contents of mbufs to user.  DO NOT free them, since
 *	there may be multiple claims on the packet!
 */
enrmove(m, uio, count)
register struct mbuf *m;
register struct uio *uio;
register int count;
{
	register int len;
	register int error = 0;
	
	enprintf(ENDBG_TRACE)("enrmove(%x, %x, %x):\n", m, uio, count);
	count = min(count, uio->uio_resid);	/* # of bytes to return */
	
	while ((count > 0) && m && (error == 0)) {
	    len = min(count, m->m_len);	/* length of this transfer */
	    count -= len;
	    error = uiomove(mtod(m, caddr_t), (int)len, UIO_READ, uio);
	    
	    m = m->m_next;
	}

	return(error);
}

enwmove(uio, mbufp)
register struct uio *uio;
register struct mbuf **mbufp;	/* top mbuf is returned by reference */
{
	struct mbuf *mtop = 0;
	register struct mbuf *m;
	register struct mbuf **mp = &mtop;
	register struct iovec *iov;
	register int len;
	int error = 0;
	
	enprintf(ENDBG_TRACE)("enwmove(%x, %x):\n", uio, mbufp);

	while ((uio->uio_resid > 0) && (error == 0)) {
	    iov = uio->uio_iov;
	    
	    if (iov->iov_len == 0) {
	    	uio->uio_iov++;
		uio->uio_iovcnt--;
		if (uio->uio_iovcnt < 0)
		    panic(
		    	"enwmove: uio_iovcnt %d < 0 while uio_resid %d > 0\n",
				uio->uio_iovcnt, uio->uio_resid);
	    }
	    MGET(m, M_WAIT, MT_DATA);
	    if (m == NULL) {
	    	error = ENOBUFS;
		break;
	    }
#ifdef	DRIVERS_SUPPORT_CLUSTERD_MBUFS
	    if (iov->iov_len >= MCLBYTES) {	/* big enough to use a page */
		MCLGET(m);
		if (m->m_len != MCLBYTES)
		    goto nopages;
		len = MCLBYTES;
	    }
	    else {
nopages:
		len = MIN(MLEN, iov->iov_len);
	    }
#else	DRIVERS_SUPPORT_CLUSTERD_MBUFS
	    len = MIN(MLEN, iov->iov_len);
#endif	DRIVERS_SUPPORT_CLUSTERD_MBUFS
	    error = uiomove(mtod(m, caddr_t), len, UIO_WRITE, uio);
	    m->m_len = len;
	    *mp = m;
	    mp = &(m->m_next);
	}

	if (error) {		/* probably uiomove fouled up */
	    if (mtop)
		m_freem(mtop);
	}
	else {
	    *mbufp = mtop;	/* return ptr to top mbuf */
	}

	return(error);
}

/*
 *  enetopen - open ether net device
 *
 *  Callable from user or kernel mode.  Bit ENKERNEL of the flag argument
 *  indicates whether or not the open is being done by the kernel.  This
 *  bit is remembered for later use by read, write, and ioctl.
 *
 *  Errors:	ENXIO	- illegal minor device number
 *		EBUSY	- minor device already in use
 */

/* ARGSUSED */
enetopen(dev, flag)
{
    register int md = ENINDEX(dev);
    register struct enState *enStatep;
    int ipl;

    /*
     * Each open enet file has a different minor device number.
     * In general, a prospective net user must try successively
     * to open the devices 'enet', 'enet1', ... 'enetn' where n
     * is MAXOPENS-1.  This is not elegant, but UNIX will call
     * open for each new open file using the same vnode but calls
     * close only when the last open file referring to the vnode 
     * is released. This means that we cannot know inside the
     * driver code when the resources associated with a particular
     * open of the same vnode should be deallocated.  Thus, we have
     * simply made up a number of different vnodes to represent the
     * simultaneous opens of the ethernet.  Each of these has
     * a different minor device number.
     *
     * When opening an ethernet device in the kernel, simply iterate
     * through minor device numbers until an open fails to return
     * EBUSY.  
     */

#ifdef	DEBUG
    enprintf(ENDBG_TRACE)("enetopen(%d, %x):\n", md, flag);
#endif
    /* check for illegal minor dev */
    if (md >= NENET*ENMAXOPENS		/* minor dev too high */
	|| (((u_char)minor(dev)>>ENINDEXBITS) >= enUnits) /* bad unit */
        || (enet_info[ENUNIT(dev)].ifp == 0)		/* ifp not known */
	|| ((enet_info[ENUNIT(dev)].ifp->if_flags & IFF_UP) == 0) )
					/* or if down */
    {
	return(ENXIO);
    }
    enStatep = &enState[ENUNIT(dev)];
    if (enOpenFlag[md])
    {
	return(EBUSY);
    }
    enOpenFlag[md] = TRUE;

    enprintf(ENDBG_DESQ)
    	("enetopen: Desq: %x, %x\n", enDesq.enQ_F, enDesq.enQ_B);
    enInitDescriptor(&enAllDescriptors[md], flag);
    enInsertDescriptor(&(enDesq), &enAllDescriptors[md]);

    return(0);
}


/*
 *  enInit - intialize ethernet unit (called by enetattach)
 */

enInit(enStatep)
register struct enState *enStatep;
{

#ifdef	DEBUG
    enprintf(ENDBG_INIT)("enInit(%x):\n", enStatep);
#endif

    /*  initialize free queue if not already done  */
    if (enFreeq.enQ_F == 0)
    {
	register int i;

	initqueue((struct queue *)&enFreeq);
	for (i=0; i<ENPACKETS; i++)
	{
	    register struct enPacket *p;

	    p = &enQueueElts[i];
	    p->enP_Func = NULL;
	    p->enP_RefCount = 0;
	    enDeallocatePacket(p);
	}
    }
    initqueue((struct queue *)&enDesq);	/* init descriptor queue */

}        

/*
 *  enetclose - ether net device close routine
 *
 *  Callable from user or kernel mode.
 */

/* ARGSUSED */
enetclose(dev, flag)
{
    register int md = ENINDEX(dev);
    register struct enState *enStatep = &enState[ENUNIT(dev)];
    register struct enOpenDescriptor *d = &enAllDescriptors[md];
    int ipl;

    enOpenFlag[md] = FALSE;      

#ifdef	DEBUG
    enprintf(ENDBG_TRACE)("enetclose(%d, %x):\n", md, flag);
#endif

    /*
     *  insure that receiver doesn't try to queue something
     *  for the device as we are decommissioning it.
     *  (I don't think this is necessary, but I'm a coward.)
     */
    ipl = splenet();
    dequeue((struct queue *)d->enOD_Link.B);
    enCurOpens--;
    enprintf(ENDBG_DESQ)
    		("enetclose: Desq: %x, %x\n", enDesq.enQ_F, enDesq.enQ_B);
    enFlushWaitQueue(&(d->enOD_Waiting));
    splx(ipl);

}

/*
 *  enetread - read next packet from net
 *
 *  Callable from user or kernel mode (checks OD_Flag)
 */

/* VARARGS */
enetread(dev, uio)
dev_t dev;
register struct uio *uio;
{

    register struct enState *enStatep = &enState[ENUNIT(dev)];
    register struct enOpenDescriptor *d = &enAllDescriptors[ENINDEX(dev)];
    register struct enPacket *p;
    int ipl;
    int error;
    extern enTimeout();

#if	DEBUG
    enprintf(ENDBG_TRACE)("enetread(%x):", dev);
#endif

    ipl = splenet();
    /*
     *  If nothing is on the queue of packets waiting for
     *  this open enet file, then set timer and sleep until
     *  either the timeout has occurred or a packet has
     *  arrived.
     */

    while (0 == d->enOD_Waiting.enWQ_NumQueued)
    {
	if (d->enOD_Timeout < 0)
	{
	    splx(ipl);
	    return(0);
	}
        if (d->enOD_Timeout)
	{
	    /*
	     *  If there was a previous timeout pending for this file,
	     *  cancel it before setting another.  This is necessary since
	     *  a cancel after the sleep might never happen if the read is
	     *  interrupted by a signal.
	     */
	    if (d->enOD_RecvState == ENRECVTIMING)
		untimeout(enTimeout, (caddr_t)d);
            timeout(enTimeout, (caddr_t)d, d->enOD_Timeout);
            d->enOD_RecvState = ENRECVTIMING;
	}
        else
            d->enOD_RecvState = ENRECVIDLE;

        sleep((caddr_t)d, PRINET);

        switch (d->enOD_RecvState)
	{
            case ENRECVTIMING:
	    {
                untimeout(enTimeout, (caddr_t)d);
                d->enOD_RecvState = ENRECVIDLE;
		break;
	    }
            case ENRECVTIMEDOUT:
	    {
                splx(ipl);
		return(0);
	    }
	}
    }

    p = enDeWaitQueue(&(d->enOD_Waiting));
    splx(ipl);

    /*
     * Kernel mode read
     *
     * Return pointer to packet.  It must be subsequently freed via an
     * EIOCDEALLOCP ioctl() call and may not be changed since it may
     * have also been on some other wait queue.
     *
     * (we don't use any fields of the U area since we can't guarantee
     * our context).
     */
    if (d->enOD_Flag & ENKERNEL)
    {
	enAllocCount++;		/* packet allocated to kernel */
	*(struct enPacket **)uio = p;
	return(0);
    }

    /*  
     * User mode read
     *
     * Move data from packet into user space.
     */
    error = enrmove(p->enP_mbuf, uio, p->enP_ByteCount);

    ipl = splenet();
    if (0 == --(p->enP_RefCount))	/* if no more claims on this packet */
    {
	m_freem(p->enP_mbuf);	/* release mbuf */
	enDeallocatePacket(p);	/* and packet */
    }
    splx(ipl);

    return(error);
}



/*
 *  enTimeout - process ethernet read timeout
 */

enTimeout(d)
register struct enOpenDescriptor * d;
{
    register int ipl;

#ifdef	DEBUG
    enprintf(ENDBG_TRACE)("enTimeout(%x):\n", d);
#endif
    ipl = splenet();
    d->enOD_RecvState = ENRECVTIMEDOUT;
    wakeup((caddr_t)d);
    enetwakeup(d);
    splx(ipl);

}

/*
 *  enetwrite - write next packet to net
 *
 *  Callable from user or kernel mode (checks OD_Flag).
 */

int enKludgeSleep[NENET];	/* Are we sleeping on IF_QFULL? */
				/*  really, # of procs sleeping on IF_QFULL */

/* VARARGS */
enetwrite(dev, uio)
dev_t dev;
register struct uio *uio;
{
    register struct enState *enStatep = &enState[ENUNIT(dev)];
    register struct enOpenDescriptor *d = &enAllDescriptors[ENINDEX(dev)];
    register struct enPacket *p;
    register struct ifnet *ifp = enet_info[ENUNIT(dev)].ifp;
    int ipl;
    int error;
    int enKludgeTime();

#if	DEBUG
    enprintf(ENDBG_TRACE)("enetwrite(%x):\n", dev);
#endif

    /*
     * User mode write
     *
     * Allocate packet for transmit.
     *
     * Copy user data into packet (if ubaremap() handled user page tables
     * we might be able to get away with direct mapping without copying here).
     */
    if ((d->enOD_Flag&ENKERNEL) == 0)
    {
	if (uio->uio_resid == 0)
	    return(0);
	ipl = splenet();
	p = enAllocatePacket();
	p->enP_RefCount++;
	splx(ipl);

	if (uio->uio_resid > ifp->if_mtu) {	/* too large */
	    ipl = splenet();
	    p->enP_RefCount--;
	    enDeallocatePacket(p);
	    splx(ipl);
	    return(EMSGSIZE);
	}

	p->enP_ByteCount = uio->uio_resid;

	if (error = enwmove(uio, &(p->enP_mbuf))) {
	    ipl = splenet();
	    p->enP_RefCount--;
	    enDeallocatePacket(p);
	    splx(ipl);
	    return(error);
	}

	p->enP_Data = mtod(p->enP_mbuf, u_short *);
		/* get address of beginning of data area */
    }
    else
    /*
     * Kernel mode write
     *
     * Use packet supplied by caller.
     *
     * (Again, we avoid using any fields of the U area since we may be called
     * in interrupt context).
     */
    {
	p = (struct enPacket *)uio;
    }

    ipl = splenet();
    /*
     * if the queue is full, and write is not kernel mode,
     * hang around until there's room or until process is interrupted
     */
    while (IF_QFULL(&(ifp->if_snd)) && ((d->enOD_Flag&ENKERNEL) == 0)) {
	extern int hz;
	/* if nobody else has a timeout pending for this unit, set one */
	if (enKludgeSleep[ENUNIT(dev)] == 0)
	    timeout(enKludgeTime, ENUNIT(dev), 2 * hz);
	enKludgeSleep[ENUNIT(dev)]++;
	if (setjmp(&u.u_qsave)) {
	    /* sleep (following) was interrupted, clean up */
#if	DEBUG
	    enprintf(ENDBG_ABNORM)
	    	("enetwrite(%x): sleep %d interrupted\n", dev,
			enKludgeSleep[ENUNIT(dev)]);
#endif	DEBUG
	    p->enP_RefCount--;
	    m_freem(p->enP_mbuf);
	    enDeallocatePacket(p);
	    splx(ipl);
	    return(EINTR);
	}
	sleep((caddr_t)&(enKludgeSleep[ENUNIT(dev)]), PRINET);
	/* enKludgeTime() has decremented enKludgeSleep[unit] for us */
    }
    
    /* place mbuf chain on outgoing queue & start if necessary */
    error = (*ifp->if_output)(ifp, p->enP_mbuf, &enetaf);
    					/* returns error number */
	/* this always frees the mbuf */
    enXcnt++;
    
    if (p->enP_Func != NULL)		/* kernel mode, presumably */
        (*(p->enP_Func))(p, ENUNIT(dev));
    else if (--(p->enP_RefCount) == 0) {
    	enDeallocatePacket(p);
    }
    else
        panic("enetwrite: packet alloc count %d should be 0\n",
					p->enP_RefCount);
    splx(ipl);
    
    return(error);
}

enKludgeTime(unit)
int unit;
{
	/* XXX perhaps we should always wakeup? */
	if (enKludgeSleep[unit]) {
		/*
		 * decrement count here, in case process was zapped
		 */
		enKludgeSleep[unit]--;
		wakeup((caddr_t)&(enKludgeSleep[unit]));
		/* XXX should we restart transmitter? */
	}
}

/*
 *  enetioctl - ether net control
 *
 *  Callable from user or kernel mode (checks OD_Flag)
 *
 *  EIOCGETP	 - get ethernet parameters
 *  EIOCSETP	 - set ethernet read timeout
 *  EIOCSETF	 - set ethernet read filter
 *  EIOCENBS	 - enable signal when read packet available
 *  EIOCINHS     - inhibit signal when read packet available
 *  FIONREAD	 - check for read packet available
 *  EIOCSETW	 - set maximum read packet waiting queue length
 *  EIOCFLUSH	 - flush read packet waiting queue
 *  EIOCALLOCP	 - allocate packet (kernel only)
 *  EIOCDEALLOCP - deallocate packet (kernel only)
 *  EIOCMBIS	 - set mode bits
 *  EIOCMBIC	 - clear mode bits
 *  ECIODEVP	 - get device parameters
 *
 * Note: the following two are obsolete! should use bcopy or something.
 *  encopyin()  copies data from caller to driver based on mode of open
 *  encopyout() copies data from driver to caller based on mode of open
 */

#define encopyin(from, to, size)	bcopy(from, to, size)
#define	encopyinOLD(from, to, size)					\
	if (d->enOD_Flag&ENKERNEL)					\
		bcopy(from, to, size);					\
	else								\
		if (copyin(from, to, size))				\
		{							\
			return(EFAULT);					\
		}

#define	encopyout(from, to, size)	bcopy(from, to, size)
#define	encopyoutOLD(from, to, size)					\
	if (d->enOD_Flag&ENKERNEL)					\
		bcopy(from, to, size);					\
	else								\
		if (copyout(from, to, size))				\
		{							\
			return(EFAULT);					\
		}

/* ARGSUSED */
enetioctl(dev, cmd, addr, flag)
caddr_t addr;
dev_t flag;
{

    register struct enState *enStatep = &enState[ENUNIT(dev)];
    register struct enOpenDescriptor * d = &enAllDescriptors[ENINDEX(dev)];
    int ipl;

#if	DEBUG
    enprintf(ENDBG_TRACE)
	    	("enetioctl(%x, %x, %x, %x):\n", dev, cmd, addr, flag);
#endif

    switch (cmd)
    {
	case EIOCGETP:
	{
            struct eniocb t;

	    t.en_maxwaiting = ENMAXWAITING;
	    t.en_maxpriority = (ENINDEX(dev) < ENHIPRIDEV)
	    				? (ENHIPRI-1)
					: ENMAXPRI;
            t.en_rtout = d->enOD_Timeout;
	    t.en_addr = 0; /*enet_info[ENUNIT(dev)].ifp->if_host[0]; /**/
	    t.en_maxfilters = ENMAXFILTERS;

	    bcopy((caddr_t)&t, addr, sizeof t);
	}
        endcase

        case EIOCSETP:
	{
            struct eniocb t;

            bcopy(addr, (caddr_t)&t, sizeof t);
            d->enOD_Timeout = t.en_rtout;
	}
        endcase

        case EIOCSETF:
	{
            struct enfilter f;

            bcopy(addr, (caddr_t)&f, sizeof f);
	    if ((ENINDEX(dev) < ENHIPRIDEV && f.enf_Priority >= ENHIPRI) ||
		f.enf_FilterLen > ENMAXFILTERS)
	    {
		return(EINVAL);
	    }
            /* insure that filter is installed indivisibly */
            ipl = splenet();
            bcopy((caddr_t)&f, (caddr_t)&(d->enOD_OpenFilter), sizeof f);
	    dequeue((struct queue *)d->enOD_Link.B);
	    enDesq.enQ_NumQueued--;
	    enInsertDescriptor(&(enDesq), d);
            splx(ipl);
	}
        endcase

	/*
	 *  Enable signal n on input packet
	 */
	case EIOCENBS:
	{
            union {int n;  int (*func)();} un;

	    bcopy(addr, (caddr_t)&un, sizeof un);
	    if ((d->enOD_Flag & ENKERNEL) == 0)
	    {
		if (un.n < NSIG)
		{
		    d->enOD_SigProc = u.u_procp;
		    d->enOD_SigPid  = u.u_procp->p_pid;
		    d->enOD_SigNumb = un.n;	/* This must be set last */
		}
		else
		{
		    goto bad;
		}
	    }
	    else
	    {
		d->enOD_SigFunc = un.func;
		d->enOD_SigNumb = NSIG;		/* This must be set last */
	    }
	}
	endcase

	/*
	 *  Disable signal on input packet
	 */
	case EIOCINHS:
	{
	    if ((d->enOD_Flag & ENKERNEL) == 0)
	    {
		d->enOD_SigNumb = 0;
	    }
	    else
	    {
		d->enOD_SigFunc = NULL;
	    }
	}
	endcase

	/*
	 *  Check for packet waiting
	 */
	case FIONREAD:
	{
            int n;
            register struct enWaitQueue *wq;

	    ipl = splenet();
	    if ((wq = &(d->enOD_Waiting))->enWQ_NumQueued)
		n = wq->enWQ_Packets[wq->enWQ_Head]->enP_ByteCount;
	    else
		n = 0;
	    splx(ipl);
	    bcopy((caddr_t)&n, addr, sizeof n);
	}
	endcase

	/*
	 *  Set maximum recv queue length for a device
	 */
	case EIOCSETW:
	{
            unsigned un;

	    bcopy(addr, (caddr_t)&un, sizeof un);
	    /*
             *  unsigned un         MaxQueued
             * ----------------    ------------
             *  0               ->  DEFWAITING
	     *  1..MAXWAITING   ->  un
	     *  MAXWAITING..-1  ->  MAXWAITING
             */
	    d->enOD_Waiting.enWQ_MaxWaiting = (un) ? min(un, ENMAXWAITING)
                                        : ENDEFWAITING;
	}
	endcase

	/*
	 *  Flush all packets queued for a device
	 */
	case EIOCFLUSH:
	{
	    ipl = splenet();
	    enFlushWaitQueue(&(d->enOD_Waiting));
	    splx(ipl);
	}
	endcase

	/*
	 *  Set mode bits
	 */
	case EIOCMBIS:
	{
	    u_short mode;

	    bcopy(addr, (caddr_t)&mode, sizeof mode);
	    if (mode&ENPRIVMODES)
		return(EINVAL);
	    else
		d->enOD_Flag |= mode;
	}
	endcase

	/*
	 *  Clear mode bits
	 */
	case EIOCMBIC:
	{
	    u_short mode;

	    bcopy(addr, (caddr_t)&mode, sizeof mode);
	    if (mode&ENPRIVMODES)
		return(EINVAL);
	    else
		d->enOD_Flag &= ~mode;
	}
	endcase

	/*
	 *  Allocate an ethernet packet (kernel only)
	 */
	case EIOCALLOCP:
	{
	    register struct enPacket *p;

	    if ((d->enOD_Flag&ENKERNEL) == 0)
		goto bad;
	    ipl = splenet();
	    p = enAllocatePacket();
	    p->enP_RefCount++;
	    enAllocCount++;
	    splx(ipl);
	    *(struct enPacket **)addr = p;
	}
	endcase

	/*
	 *  Deallocate an ethernet packet (kernel only)
	 */
	case EIOCDEALLOCP:
 	{
	    register struct enPacket *p;

	    if ((d->enOD_Flag & ENKERNEL) == 0)
		goto bad;
	    p = *(struct enPacket **)addr;
	    ipl = splenet();
	    enAllocCount--;
	    if (--(p->enP_RefCount) == 0)
		enDeallocatePacket(p);
	    splx(ipl);
	}
	endcase

	/*
	 * Return hardware-specific device parameters.
	 */
	case EIOCDEVP:
	{
	    bcopy((caddr_t)&(enDevParams), addr, sizeof(struct endevp));
	}
	endcase;

        default:
	{
	bad:
	    return(EINVAL);
	}
        endcase
    }

    return(0);

}

/****************************************************************
 *								*
 *		Support for select() system call		*
 *								*
 *	Other hooks in:						*
 *		enInitDescriptor()				*
 *		enInputDone()					*
 *		enTimeout()					*
 ****************************************************************/
/*
 * inspired by the code in tty.c for the same purpose.
 */

/*
 * enetselect - returns true iff the specific operation
 *	will not block indefinitely.  Otherwise, return
 *	false but make a note that a selwakeup() must be done.
 */
enetselect(dev, rw)
register dev_t dev;
int rw;
{
	register struct enState *enStatep;
	register struct enOpenDescriptor *d;
	register struct enWaitQueue *wq;
	register int ipl;
	register int avail;
	
	switch (rw) {
	
	case FREAD:
		/*
		 * an imitation of the FIONREAD ioctl code
		 */
		enStatep = &(enState[ENUNIT(dev)]);
		d = &(enAllDescriptors[ENINDEX(dev)]);
		
		ipl = splenet();
		if ((wq = &(d->enOD_Waiting))->enWQ_NumQueued)
			avail = 1;	/* at least one packet queued */
		else {
			avail = 0;	/* sorry, nothing queued now */
			/*
			 * If there's already a select() waiting on this
			 * minor device then this is a collision.
			 * [This shouldn't happen because enet minors
			 * really should not be shared, but if a process
			 * forks while one of these is open, it is possible
			 * that both processes could select() us.]
			 */
			if (d->enOD_SelProc
			     && d->enOD_SelProc->p_wchan == (caddr_t)&selwait)
			     	d->enOD_SelColl = 1;
			else
				d->enOD_SelProc = u.u_procp;		
		}
		splx(ipl);	
		return(avail);

	case FWRITE:
		/*
		 * since the queueing for output is shared not just with
		 * the other enet devices but also with the IP system,
		 * we can't predict what would happen on a subsequent
		 * write.  However, since we presume that all writes
		 * complete eventually, and probably fairly fast, we
		 * pretend that select() is true.
		 */
		return(1);

	default:		/* hmmm. */
		return(1);		/* don't block in select() */
	}
}

enetwakeup(d)
register struct enOpenDescriptor *d;
{
	if (d->enOD_SelProc) {
		selwakeup(d->enOD_SelProc, d->enOD_SelColl);
		d->enOD_SelColl = 0;
		d->enOD_SelProc = 0;
	}
}

/*
 * enetFilter - incoming linkage from ../is68kif/if_il.c
 */

enetFilter(en, m, count)
register int en;
register struct mbuf *m;
register int count;
{

    register struct enState *enStatep = &enState[en];
    register struct enPacket *p;
    register int pullcount;	/* bytes, not words */
    int s = splenet();

#if	DEBUG
    enprintf(ENDBG_TRACE)("enetFilter(%d):\n", en);
#endif

    p = enAllocatePacket();	/* panics if not possible */

    p->enP_ByteCount = count;
    
    pullcount = min(MLEN, count);	/* largest possible first mbuf */
    if (m->m_len < pullcount) {
    	/* first mbuf not as full as it could be - fix this */
	if ((m = m_pullup(m, pullcount)) == 0) {
	    /* evidently no resources; bloody m_pullup discarded mbuf */
	    enDeallocatePacket(p);
	    enRdrops++;
	    goto out;
	}
    }
    
    p->enP_mbuf = m;
    p->enP_Data = mtod(m, u_short *);

    enInputDone(enStatep, p);
out:
    splx(s);
}

/*
 * enInputDone - process correctly received packet
 */

enInputDone(enStatep, p)
register struct enState *enStatep;
register struct enPacket *p;
{
    register struct enOpenDescriptor *d;
    register int queued = 0;
#if	DEBUG
    enprintf(ENDBG_TRACE)("enInputDone(%x): %x\n", enStatep, p);
#endif
    forAllOpenDescriptors(d)
    {
        if (enDoFilter(p,d))
	{
            if (d->enOD_Waiting.enWQ_NumQueued < d->enOD_Waiting.enWQ_MaxWaiting)
	    {
                enEnWaitQueue(&(d->enOD_Waiting), p);
                p->enP_RefCount++;
		queued++;
                wakeup((caddr_t)d);
		enetwakeup(d);
#if	DEBUG
                enprintf(ENDBG_TRACE)("enInputDone: queued\n");
#endif
	    }
	    /*  send notification when input packet received  */
	    if (d->enOD_SigNumb)
	    {
		if (d->enOD_SigNumb < NSIG)
		{
		    if (d->enOD_SigProc->p_pid == d->enOD_SigPid)
			psignal(d->enOD_SigProc, d->enOD_SigNumb);
		    if ((d->enOD_Flag & ENHOLDSIG) == 0)
			d->enOD_SigNumb = 0;		/* disable signal */
		}
		else
		{
		    register int enunit;
		    
		    enunit = enStatep - enState;	/* find unit # */

		    (*(d->enOD_SigFunc))(enunit);
		}
	    }
	    if (d->enOD_OpenFilter.enf_Priority >= ENHIPRI)
		break;
	}
    }
    if (queued == 0)			/* this buffer no longer in use */
    {
	m_freem(p->enP_mbuf);			/* free mbuf */
	enDeallocatePacket(p);			/*  and packet */
	enRdrops++;
    }
    else
	enRcnt++;

}

enInitDescriptor(d, flag)
register struct enOpenDescriptor *d;
{

#if	DEBUG
    enprintf(ENDBG_TRACE)("enInitDescriptor(%x):\n", d);
#endif
    d->enOD_RecvState = ENRECVIDLE;
    d->enOD_OpenFilter.enf_FilterLen = 0;
    d->enOD_OpenFilter.enf_Priority = 0;
    d->enOD_Timeout = 0;
    d->enOD_SigNumb = 0;
    d->enOD_Flag = (flag&ENKERNEL);
    d->enOD_SelColl = 0;
    d->enOD_SelProc = 0;		/* probably unnecessary */
    /*
     * Remember the PID that opened us, at least until some process
     * sets a signal for this minor device
     */
    if ((flag&ENKERNEL) == 0) {
	d->enOD_SigPid = u.u_procp->p_pid;
    }
    enInitWaitQueue(&(d->enOD_Waiting));
#if	DEBUG
    enprintf(ENDBG_TRACE)("=>eninitdescriptor\n");
#endif

}

#define	opx(i)	(i>>ENF_NBPA)

boolean
enDoFilter(p,d)
struct enPacket *p;
struct enOpenDescriptor *d;
{

    register unsigned short *sp;
    register unsigned short *fp;
    register unsigned short *fpe;
    register unsigned op;
    register unsigned arg;
    register maxword;
    unsigned short stack[ENMAXFILTERS+1];

#ifdef	DEBUG
    enprintf(ENDBG_TRACE)("enDoFilter(%x,%x):\n", p, d);
#endif
    sp = &stack[ENMAXFILTERS];
    maxword = (min(p->enP_ByteCount, p->enP_mbuf->m_len)>>1) + ENF_PUSHWORD;
    	/* can't address beyond end of packet or end of first mbuf */
    fp = &d->enOD_OpenFilter.enf_Filter[0];
    fpe = &fp[d->enOD_OpenFilter.enf_FilterLen];
    *sp = TRUE;

    for (; fp < fpe; )
    {
	op = (*fp >> ENF_NBPA) & ((1 << ENF_NBPO) - 1);
	arg = *fp & ((1 << ENF_NBPA) - 1);
	fp++;
	switch (arg)
	{
	    default:
		if ((arg >= ENF_PUSHWORD)&&(arg < maxword))
		    *--sp = p->enP_Data[arg-ENF_PUSHWORD];
		else
		{
#ifdef	DEBUG
		    enprintf(ENDBG_TRACE)("=>0(len)\n");
#endif
		    return(false);
		}
		break;
	    case ENF_PUSHLIT:
		*--sp = *fp++;
		break;
	    case ENF_PUSHZERO:
		*--sp = 0;
	    case ENF_NOPUSH:
		break;
	}
	if (op == ENF_NOP)
	    continue;
	if (sp > &stack[ENMAXFILTERS-2])
	{
#ifdef	DEBUG
	    enprintf(ENDBG_TRACE)("=>0(sp)\n");
#endif
	    return(false);
	}
	arg = *sp++;
	switch (op)
	{
	    default:
#ifdef	DEBUG
		enprintf(ENDBG_TRACE)("=>0(def)\n");
#endif
		return(false);
	    case opx(ENF_AND):
		*sp &= arg;
		break;
	    case opx(ENF_OR):
		*sp |= arg;
		break;
	    case opx(ENF_XOR):
		*sp ^= arg;
		break;
	    case opx(ENF_EQ):
		*sp = (*sp == arg);
		break;
	    case opx(ENF_LT):
		*sp = (*sp < arg);
		break;
	    case opx(ENF_LE):
		*sp = (*sp <= arg);
		break;
	    case opx(ENF_GT):
		*sp = (*sp > arg);
		break;
	    case opx(ENF_GE):
		*sp = (*sp >= arg);
		break;
	}
    }
#ifdef	DEBUG
    enprintf(ENDBG_TRACE)("=>%x\n", *sp);
#endif
    return((boolean)*sp);

}

/*
 *  enInsertDescriptor - insert open descriptor in queue ordered by priority
 */

enInsertDescriptor(q, d)
register struct enQueue *q;
register struct enOpenDescriptor *d;
{
    struct enOpenDescriptor * nxt;
    register int ipl;

    ipl = splenet();
    nxt = (struct enOpenDescriptor *)q->enQ_F;
    while ((struct Queue *)q != &(nxt->enOD_Link))
    {
        if (d->enOD_OpenFilter.enf_Priority > nxt->enOD_OpenFilter.enf_Priority)
	    break;
        nxt = (struct enOpenDescriptor *)nxt->enOD_Link.F;
    }
    enqueue((struct queue *)&(nxt->enOD_Link),(struct queue *)&(d->enOD_Link));
    enprintf(ENDBG_DESQ)("enID: Desq: %x, %x\n", q->enQ_F, q->enQ_B);
    q->enQ_NumQueued++;
    splx(ipl);

}

enetattach(ifp, devp)
struct ifnet *ifp;
struct endevp *devp;
{
    register struct enState *enStatep = &(enState[enUnits]);

#ifdef	DEBUG
    enprintf(ENDBG_INIT) ("enetattach: type %d, addr ", devp->end_dev_type);
    if (enDebug&ENDBG_INIT) {
	register int i;
	for (i = 0; i < devp->end_addr_len; i++)
	    printf("%o ", devp->end_addr[i]);
	printf("\n");
    }
#endif	DEBUG

    enet_info[enUnits].ifp = ifp;

    bcopy((caddr_t)devp, (caddr_t)&(enDevParams), sizeof(struct endevp));

    enInit(enStatep);
    
    return(enUnits++);
}

#else	ENETFILTER
enetopen(){}
enetclose(){}
enetread(){}
enetwrite(){}
enetioctl(){}
enetselect(){}
#endif	ENETFILTER
