/*
 * QBUS: TM11 tape driver
 */
#include "tm.h"
#if	NTM > 0
/* TODO:
 *	ISI: raw io does not necssarily use rtsbuf when calling strategy !!!!
 *	test driver with more than one slave
 *	test driver with more than one controller
 *	test reset code
 *	what happens if you offline tape during rewind?
 *	test using file system on tape
 */
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "file.h"
#include "map.h"
#include "vm.h"
#include "ioctl.h"
#include "mtio.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"

#include "../machine/cpu.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/tmreg.h"

/*
 * There is a ctmbuf per tape controller. It is used as the token to pass to 
 * the internal routines to execute tape ioctls, and also acts as a lock on the
 * slaves on the controller, since there is only one per controller. In 
 * particular, when the tape is rewinding on close we release the user process 
 * but any further attempts to use the tape drive before the rewind completes 
 * will hang waiting for ctmbuf.
 */
struct	buf	ctmbuf[NTM];

/*
 * Raw tape operations use rtmbuf.  The driver notices when rtmbuf is being used
 * and allows the user program to continue after errors and read records not of
 * the standard length (BSIZE).
 */
struct	buf	rtmbuf[NTM];

/*
 * Driver qbus interface routines and variables.
 */
int	tmprobe(), tmslave(), tmattach(), tmintr();
int	tmminphys();

struct	qb_ctlr		*TMinfo[NTM];
struct	qb_device	*tminfo[Ntm];
struct	buf		tmutab[Ntm];
short			tmtoTM[Ntm];
u_short			*TMstd[] = { (u_short *)0x3FF550, 0 };
struct	qb_driver 	TMdriver =
 	{ tmprobe, tmslave, tmattach, TMstd, "tm", tminfo, "TM", TMinfo };

/* bits in minor device */
#define	UNIT_tm(dev)	(minor(dev)&03)
#define	UNIT_TM(dev)	(tmtoTM[UNIT_tm(dev)])
#define	T_NOREWIND	04
#define	T_1600BPI	08

#define	INF	(daddr_t)1000000L

/*
 * Software state per tape transport.
 *
 * 1. A tape drive is a unique-open device; we refuse opens when it is already.
 * 2. We keep track of the current position on a block tape and seek before 
 *    operations by forward/back spacing if necessary.
 * 3. We remember if the last operation was a write on a tape, so if a tape is 
 *    open read write and the last thing done is a write we can write a standard
 *    end of tape mark (two eofs).
 * 4. We remember the status registers after the last command, using then 
 *    internally and returning them to the SENSE ioctl.
 * 5. We remember the last density the tape was used at.  If it is not a BOT 
 *    when we start using it and we are writing, we don't let the density be 
 *    changed.
 */
struct	tm_softc {
	char	sc_openf;	/* lock against multiple opens */
	char	sc_lastiow;	/* last op was a write */
	daddr_t	sc_blkno;	/* block number, for block device tape */
	daddr_t	sc_nxrec;	/* position of end of tape, if known */
	u_short	sc_erreg;	/* copy of last erreg */
	u_short	sc_dsreg;	/* copy of last dsreg */
	short	sc_resid;	/* copy of last bc */
#ifdef unneeded
	short	sc_lastcmd;	/* last command to handle direction changes */
#endif
	u_short	sc_dens;	/* prototype command with density info */
	daddr_t	sc_timo;	/* time until timeout expires */
	short	sc_tact;	/* timeout is active */
	short	sc_18bit;	/* */
} tm_softc[Ntm];
#ifdef unneeded
int	tmgapsdcnt;		/* DEBUG */
#endif

int	tmtimer();
/*
 * States for qm->qm_tab.b_active, the per controller state flag. This is used 
 * to sequence control in the driver.
 */
#define	SSEEK	1		/* seeking */
#define	SIO	2		/* doing seq i/o */
#define	SCOM	3		/* sending control command */
#define	SREW	4		/* sending a drive rewind */

/*
 * Determine if there is a controller for a tm at address reg. Our goal is to 
 * make the device interrupt.
 */
tmprobe(tmaddr)
	struct tmdevice *tmaddr;
{
	extern int cvec;

	/*
	 * There are some TM controllers which will not interupt if you do not 
	 * actualy give a command, and set the GO bit.  This code will not work
	 * for drives like this.
	 */
	tmaddr->tmcs = TM_IE;
	DELAY(100);
	/*
	 * If this is a TM, it ought to have interrupted by now, if it isn't 
	 * (ie: it is a TS) then we just hope that it didn't interrupt, so 
	 * autoconf will ignore it. Just in case, we will reference one of 
	 * the more distant TM registers, and hope for a machine check, or 
	 * similar disaster if this is a TS.
	 */
	if (badaddr((caddr_t)&tmaddr->tmrd, 2))
		return (0);
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (sizeof (struct tmdevice));
}

/*
 * Due to a design flaw, we cannot ascertain if the tape exists or not unless 
 * it is on line - ie: unless a tape is mounted. This is too servere a 
 * restriction to bear, so all units are assumed to exist.
 */
tmslave(qi, tmaddr)
	struct qb_device *qi;
	struct tmdevice *tmaddr;
{

	return (1);
}

/*
 * Record attachment of the unit to the controller.
 */
tmattach(qi)
	struct qb_device *qi;
{
	/*
	 * tmtoTM is used in UNIT_TM to index the ctmbuf and rtmbuf arrays given
	 * a tm unit number.
	 */
	tmtoTM[qi->qi_unit] = qi->qi_mi->qm_ctlr;
	if (!badaddr(&((struct tmdevice *)qi->qi_mi->qm_addr)->tmbae, 2)) {
		((struct tmdevice *)qi->qi_mi->qm_addr)->tmbae = 0xc;
		if ( ((struct tmdevice *)qi->qi_mi->qm_addr)->tmbae == 0xc) {
			printf("	22bit ");
			return;
		}
	}
	tm_softc[qi->qi_unit].sc_18bit++;
	printf("	18bit ");
}

/*
 * Open the device.  Tapes are unique open devices, so we refuse if it is 
 * already open. We also check that a tape is available, and don't block waiting
 * here; if you want to wait for a tape you should timeout in user code.
 */
tmopen(dev, flag)
	dev_t dev;
	int flag;
{
	register int tmunit;
	register struct qb_device *qi;
	register struct tm_softc *sc;
	int olddens, dens;
	int s;

	tmunit = UNIT_tm(dev);
	if (tmunit>=Ntm || (sc = &tm_softc[tmunit])->sc_openf ||
	    (qi = tminfo[tmunit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	if (sc->sc_18bit)
		bdevsw[major(dev)].d_flags |= B_18BIT;
	olddens = sc->sc_dens;
	dens = TM_IE | TM_GO | (qi->qi_slave << 8);
	if ((minor(dev) & T_1600BPI) == 0)
		dens |= TM_D800;
	sc->sc_dens = dens;

get:	tmcommand(dev, TM_SENSE, 1);
	if (sc->sc_erreg&TMER_SDWN) {
		sleep((caddr_t)&lbolt, PZERO+1);
		goto get;
	}
	sc->sc_dens = olddens;
	if ((sc->sc_erreg&(TMER_SELR|TMER_TUR)) != (TMER_SELR|TMER_TUR)) {
		uprintf("tm%d: not online\n", tmunit);
		return (EIO);
	}
	if ((flag&FWRITE) && (sc->sc_erreg&TMER_WRL)) {
		uprintf("tm%d: no write ring\n", tmunit);
		return (EIO);
	}
	if ((sc->sc_erreg&TMER_BOT) == 0 && (flag&FWRITE) &&
	    dens != sc->sc_dens) {
		uprintf("tm%d: can't change density in mid-tape\n", tmunit);
		return (EIO);
	}
	sc->sc_openf = 1;
	sc->sc_blkno = (daddr_t)0;
	sc->sc_nxrec = INF;
	sc->sc_lastiow = 0;
	sc->sc_dens = dens;
	s = spl6();
	if (sc->sc_tact == 0) {
		sc->sc_timo = INF;
		sc->sc_tact = 1;
		timeout(tmtimer, (caddr_t)dev, 5*hz);
	}
	splx(s);
	return (0);
}

/*
 * Close tape device. 
 * If tape was open for writing or last operation was a write, then write two 
 * EOF's and backspace over the last one. Unless this is a non-rewinding special
 * file, rewind the tape. Make the tape available to others.
 */
tmclose(dev, flag)
	register dev_t dev;
	register flag;
{
	register struct tm_softc *sc = &tm_softc[UNIT_tm(dev)];

	if (flag == FWRITE || (flag&FWRITE) && sc->sc_lastiow) {
		tmcommand(dev, TM_WEOF, 1);
		tmcommand(dev, TM_WEOF, 1);
		tmcommand(dev, TM_SREV, 1);
	}
	if ((minor(dev)&T_NOREWIND) == 0)
		/*
		 * 0 count means don't hang waiting for rewind complete rather 
		 * ctmbuf stays busy until the operation completes preventing 
		 * further opens from completing by preventing a TM_SENSE from 
		 * completing.
		 */
		tmcommand(dev, TM_REW, 0);
	sc->sc_openf = 0;
}

/*
 * Execute a command on the tape drive a specified number of times.
 */
tmcommand(dev, com, count)
	dev_t dev;
	int com, count;
{
	register struct buf *bp;
	register int s;

	bp = &ctmbuf[UNIT_TM(dev)];
	s = spl5();
	while (bp->b_flags&B_BUSY) {
		/*
		 * This special check is because B_BUSY never gets cleared in 
		 * the non-waiting rewind case.
		 */
		if (bp->b_repcnt == 0 && (bp->b_flags&B_DONE))
			break;
		bp->b_flags |= B_WANTED;
		sleep((caddr_t)bp, PRIBIO);
	}
	bp->b_flags = B_BUSY|B_READ;
	splx(s);
	bp->b_dev = dev;
	bp->b_repcnt = -count;
	bp->b_command = com;
	bp->b_blkno = 0;
	tmstrategy(bp);
	/*
	 * In case of rewind from close, don't wait. This is the only case where
	 * count can be 0.
	 */
	if (count == 0)
		return;
	iowait(bp);
	if (bp->b_flags&B_WANTED)
		wakeup((caddr_t)bp);
	bp->b_flags &= B_ERROR;
}

/*
 * Queue a tape operation.
 */
tmstrategy(bp)
	register struct buf *bp;
{
	int tmunit = UNIT_tm(bp->b_dev);
	register struct qb_ctlr *qm;
	register struct buf *dp;
	int s;

	/*
	 * Put transfer at end of unit queue
	 */
	dp = &tmutab[tmunit];
	bp->av_forw = NULL;
	s = spl5();
	qm = tminfo[tmunit]->qi_mi;
	if (dp->b_actf == NULL) {
		dp->b_actf = bp;
		/*
		 * Transport not already active..put at end of controller queue.
		 */
		dp->b_forw = NULL;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
	} else
		dp->b_actl->av_forw = bp;
	dp->b_actl = bp;
	/*
	 * If the controller is not busy, get it going.
	 */
	if (qm->qm_tab.b_active == 0)
		tmstart(qm);
	splx(s);
}

/*
 * Start activity on a tm controller.
 */
tmstart(qm)
	register struct qb_ctlr *qm;
{
	register struct buf *bp, *dp;
	register struct tmdevice *tmaddr = (struct tmdevice *)qm->qm_addr;
	register struct tm_softc *sc;
	register struct qb_device *qi;
	int tmunit, cmd, ioaddr;
	daddr_t blkno;

	/*
	 * Look for an idle transport on the controller.
	 */
loop:
	if ((dp = qm->qm_tab.b_actf) == NULL)
		return;
	if ((bp = dp->b_actf) == NULL) {
		qm->qm_tab.b_actf = dp->b_forw;
		goto loop;
	}
	tmunit = UNIT_tm(bp->b_dev);
	qi = tminfo[tmunit];
	/*
	 * Record pre-transfer status (e.g. for TM_SENSE)
	 */
	sc = &tm_softc[tmunit];
	tmaddr = (struct tmdevice *)qm->qm_addr;
	tmaddr->tmcs = (qi->qi_slave << 8);
	sc->sc_dsreg = tmaddr->tmcs;
	sc->sc_erreg = tmaddr->tmer;
	sc->sc_resid = tmaddr->tmbc;
	/*
	 * Default is that last command was NOT a write command; if we do a 
	 * write command we will notice this in tmintr().
	 */
	sc->sc_lastiow = 0;
	if (sc->sc_openf < 0 || (tmaddr->tmcs&TM_CUR) == 0) {
		/*
		 * Have had a hard error on a non-raw tape
		 * or the tape unit is now unavailable
		 * (e.g. taken off line).
		 */
		bp->b_flags |= B_ERROR;
		goto next;
	}
	if (bp == &ctmbuf[UNIT_TM(bp->b_dev)]) {
		/*
		 * Execute control operation with the specified count.
		 */
		if (bp->b_command == TM_SENSE)
			goto next;
		/*
		 * Set next state; give 5 minutes to complete rewind or file
		 * skip operations, or 10 seconds per iteration (minimum 60 
		 * seconds and max 5 minutes) to complete other ops.
		 */
		if (bp->b_command == TM_REW) {
			qm->qm_tab.b_active = SREW;
			sc->sc_timo = 5 * 60;
		} else {
			qm->qm_tab.b_active = SCOM;
			sc->sc_timo =
			    MIN(MAX(10*(int)-bp->b_repcnt,60),5*60);
		}
		if (bp->b_command == TM_SFORW || bp->b_command == TM_SREV) {
			tmaddr->tmbc = bp->b_repcnt;
			sc->sc_timo = 5 * 60;
		}
		goto dobpcmd;
	}
	/*
	 * The following checks handle boundary cases for operation on non-raw 
	 * tapes. On raw tapes the initialization of sc->sc_nxrec by tmphys 
	 * causes them to be skipped normally (except in the case of retries).
	 */
	if (bdbtofsb(bp->b_blkno) > sc->sc_nxrec) {
		/*
		 * Can't read past known end-of-file.
		 */
		bp->b_flags |= B_ERROR;
		bp->b_error = ENXIO;
		goto next;
	}
	if (bdbtofsb(bp->b_blkno) == sc->sc_nxrec &&
	    bp->b_flags&B_READ) {
		/*
		 * Reading at end of file returns 0 bytes.
		 */
		bp->b_resid = bp->b_bcount;
		clrbuf(bp);
		goto next;
	}
	if ((bp->b_flags&B_READ) == 0)
		/*
		 * Writing sets EOF
		 */
		sc->sc_nxrec = bdbtofsb(bp->b_blkno) + 1;
	/*
	 * If the data transfer command is in the correct place, set up all the
	 * registers except the csr.
	 */
	if ((blkno = sc->sc_blkno) == bdbtofsb(bp->b_blkno)) {
		tmaddr->tmbc = -bp->b_bcount;
		if ((bp->b_flags&B_READ) == 0) {
			if (qm->qm_tab.b_errcnt)
				cmd = TM_WIRG;
			else
				cmd = TM_WCOM;
		} else
			cmd = TM_RCOM;
		qm->qm_tab.b_active = SIO;
#ifdef notdef
		if (tmreverseop(sc->sc_lastcmd))
			while (tmaddr->tmer & TMER_SDWN)
				tmgapsdcnt++;
		sc->sc_lastcmd = TM_RCOM;		/* will serve */
#endif
		sc->sc_timo = 60;	/* premature, but should serve */
		ioaddr = qbaddr(bp);
		tmaddr->tmba = loword(ioaddr);
		if (!sc->sc_18bit)
			tmaddr->tmbae = hiword(ioaddr);
		tmaddr->tmcs = sc->sc_dens | cmd | ((ioaddr >> 12) & 0x30);
		return;
	}
	/*
	 * Tape positioned incorrectly; set to seek forwards or backwards to the
	 * correct spot. This happens for raw tapes only on error retries.
	 */
	qm->qm_tab.b_active = SSEEK;
	if (blkno < bdbtofsb(bp->b_blkno)) {
		bp->b_command = TM_SFORW;
		tmaddr->tmbc = blkno - bdbtofsb(bp->b_blkno);
	} else {
		bp->b_command = TM_SREV;
		tmaddr->tmbc = bdbtofsb(bp->b_blkno) - blkno;
	}
	sc->sc_timo = MIN(MAX(10 * -tmaddr->tmbc, 60), 5 * 60);
dobpcmd:
#ifdef notdef
	/*
	 * It is strictly necessary to wait for the tape to stop before changing
	 * directions, but the TC11 handles this for us.
	 */
	if (tmreverseop(sc->sc_lastcmd) != tmreverseop(bp->b_command))
		while (tmaddr->tmer & TM_SDWN)
			tmgapsdcnt++;
	sc->sc_lastcmd = bp->b_command;
#endif
	/*
	 * Do the command in bp.
	 */
	tmaddr->tmcs = (sc->sc_dens | bp->b_command);
	return;

next:
	/*
	 * Done with this operation due to error or the fact that it doesn't do
	 * anything. Dequeue the transfer and continue processing this slave.
	 */
	qm->qm_tab.b_errcnt = 0;
	dp->b_actf = bp->av_forw;
	iodone(bp);
	goto loop;
}

/*
 * Tm interrupt routine.
 */
/*ARGSUSED*/
tmintr(tm11)
	int tm11;
{
	struct buf *dp;
	register struct buf *bp;
	register struct qb_ctlr *qm = TMinfo[tm11];
	register struct tmdevice *tmaddr;
	register struct tm_softc *sc;
	int tmunit;
	register state;

	if ((dp = qm->qm_tab.b_actf) == NULL)
		return;
	bp = dp->b_actf;
	tmunit = UNIT_tm(bp->b_dev);
	tmaddr = (struct tmdevice *)qm->qm_addr;
	sc = &tm_softc[tmunit];
	/*
	 * If last command was a rewind, and tape is still rewinding, wait for 
	 * the rewind complete interrupt.
	 */
	if (qm->qm_tab.b_active == SREW) {
		qm->qm_tab.b_active = SCOM;
		if (tmaddr->tmer&TMER_RWS) {
			sc->sc_timo = 5*60;		/* 5 minutes */
			return;
		}
	}
	/*
	 * An operation completed... record status
	 */
	sc->sc_timo = INF;
	sc->sc_dsreg = tmaddr->tmcs;
	sc->sc_erreg = tmaddr->tmer;
	sc->sc_resid = tmaddr->tmbc;
	if ((bp->b_flags & B_READ) == 0)
		sc->sc_lastiow = 1;
	state = qm->qm_tab.b_active;
	qm->qm_tab.b_active = 0;
	/*
	 * Check for errors.
	 */
	if (tmaddr->tmcs&TM_ERR) {
		while (tmaddr->tmer & TMER_SDWN)
			;			/* await settle down */
		/*
		 * If we hit the end of the tape file, update our position.
		 */
		if (tmaddr->tmer&TMER_EOF) {
			tmseteof(bp);		/* set blkno and nxrec */
			state = SCOM;		/* force completion */
			/*
			 * Stuff bc so it will be unstuffed correctly later to 
			 * get resid.
			 */
			tmaddr->tmbc = -bp->b_bcount;
			goto opdone;
		}
		/*
		 * If we were reading raw tape and the only error was that the
		 * record was too long, then we don't consider this an error.
		 */
		if (bp == &rtmbuf[UNIT_TM(bp->b_dev)] && (bp->b_flags&B_READ) &&
		    (tmaddr->tmer&(TMER_HARD|TMER_SOFT)) == TMER_RLE)
			goto ignoreerr;
		/*
		 * If error is not hard, and this was an i/o operation retry up
		 * to 8 times.
		 */
		if ((tmaddr->tmer&TMER_HARD)==0 && state==SIO) {
			if (++qm->qm_tab.b_errcnt < 7) {
				sc->sc_blkno++;
				goto opcont;
			}
		} else
			/*
			 * Hard or non-i/o errors on non-raw cause it to close.
			 */
			if (sc->sc_openf>0 && bp != &rtmbuf[UNIT_TM(bp->b_dev)])
				sc->sc_openf = -1;
		/*
		 * Couldn't recover error
		 */
		printf("tm%d: hard error bn%d er=%b\n", minor(bp->b_dev)&03,
		    bp->b_blkno, sc->sc_erreg, TMER_BITS);
		bp->b_flags |= B_ERROR;
		goto opdone;
	}
	/*
	 * Advance tape control FSM.
	 */
ignoreerr:
	switch (state) {

	case SIO:
		/*
		 * Read/write increments tape block number
		 */
		sc->sc_blkno++;
		goto opdone;

	case SCOM:
		/*
		 * For forward/backward space record update current position.
		 */
		if (bp == &ctmbuf[UNIT_TM(bp->b_dev)])
		switch (bp->b_command) {

		case TM_SFORW:
			sc->sc_blkno -= bp->b_repcnt;
			break;

		case TM_SREV:
			sc->sc_blkno += bp->b_repcnt;
			break;
		}
		goto opdone;

	case SSEEK:
		sc->sc_blkno = bdbtofsb(bp->b_blkno);
		goto opcont;

	default:
		panic("tmintr");
	}
opdone:
	/*
	 * Reset error count and remove from device queue.
	 */
	qm->qm_tab.b_errcnt = 0;
	dp->b_actf = bp->av_forw;
	/*
	 * Check resid; watch out for resid >32767 (tmbc not negative).
	 */
	bp->b_resid = ((int) -tmaddr->tmbc) & 0xffff;
	iodone(bp);
	/*
	 * Circulate slave to end of controller queue to give others a chance.
	 */
	qm->qm_tab.b_actf = dp->b_forw;
	if (dp->b_actf) {
		dp->b_forw = NULL;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
	}
	if (qm->qm_tab.b_actf == 0)
		return;
opcont:
	tmstart(qm);
}

tmtimer(dev)
	int dev;
{
	register struct tm_softc *sc = &tm_softc[UNIT_tm(dev)];
	register short x;

	if (sc->sc_timo != INF && (sc->sc_timo -= 5) < 0) {
		printf("tm%d: lost interrupt\n", UNIT_tm(dev));
		sc->sc_timo = INF;
		x = spl5();
		tmintr(UNIT_TM(dev));
		(void) splx(x);
	}
	timeout(tmtimer, (caddr_t)dev, 5*hz);
}

tmseteof(bp)
	register struct buf *bp;
{
	register int tmunit = UNIT_tm(bp->b_dev);
	register struct tmdevice *tmaddr = 
	    (struct tmdevice *)TMinfo[UNIT_TM(bp->b_dev)]->qm_addr;
	register struct tm_softc *sc = &tm_softc[tmunit];

	if (bp == &ctmbuf[UNIT_TM(bp->b_dev)]) {
		if (sc->sc_blkno > bdbtofsb(bp->b_blkno)) {
			/* reversing */
			sc->sc_nxrec = bdbtofsb(bp->b_blkno) - tmaddr->tmbc;
			sc->sc_blkno = sc->sc_nxrec;
		} else {
			/* spacing forward */
			sc->sc_blkno = bdbtofsb(bp->b_blkno) + tmaddr->tmbc;
			sc->sc_nxrec = sc->sc_blkno - 1;
		}
		return;
	} 
	/* eof on read */
	sc->sc_nxrec = bdbtofsb(bp->b_blkno);
}

tmread(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	int errno;

	errno = tmphys(dev, uio);
	if (errno)
		return (errno);
	return (physio(tmstrategy, &rtmbuf[UNIT_TM(dev)], dev, B_READ, 
		tmminphys, uio));
}

tmwrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	int errno;

	errno = tmphys(dev, uio);
	if (errno)
		return (errno);
	return (physio(tmstrategy, &rtmbuf[UNIT_TM(dev)], dev, B_WRITE, 
		tmminphys, uio));
}

/*
 * Check that a raw device exists. If it does, set up sc_blkno and sc_nxrec
 * so that the tape will appear positioned correctly.
 */
tmphys(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register int tmunit = UNIT_tm(dev);
	register daddr_t a;
	register struct tm_softc *sc;
	register struct qb_device *qi;

	if (tmunit >= Ntm || (qi=tminfo[tmunit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	sc = &tm_softc[tmunit];
	a = bdbtofsb(uio->uio_offset >> 9);
	sc->sc_blkno = a;
	sc->sc_nxrec = a + 1;
	return (0);
}

tmreset(qbn)
	int qbn;
{
	register struct qb_ctlr *qm;
	register tm11, tmunit;
	register struct qb_device *qi;
	register struct buf *dp;

	for (tm11 = 0; tm11 < NTM; tm11++) {
		if ((qm = TMinfo[tm11]) == 0 || qm->qm_alive == 0 )
			continue;
		printf(" tm%d", tm11);
		qm->qm_tab.b_active = 0;
		qm->qm_tab.b_actf = qm->qm_tab.b_actl = 0;
		((struct tmdevice *)(qm->qm_addr))->tmcs = TM_DCLR;
		for (tmunit = 0; tmunit < Ntm; tmunit++) {
			if ((qi = tminfo[tmunit]) == 0 || qi->qi_mi != qm ||
			    qi->qi_alive == 0)
				continue;
			dp = &tmutab[tmunit];
			dp->b_active = 0;
			dp->b_forw = 0;
			if (qm->qm_tab.b_actf == NULL)
				qm->qm_tab.b_actf = dp;
			else
				qm->qm_tab.b_actl->b_forw = dp;
			qm->qm_tab.b_actl = dp;
			if (tm_softc[tmunit].sc_openf > 0)
				tm_softc[tmunit].sc_openf = -1;
		}
		tmstart(qm);
	}
}

/*ARGSUSED*/
tmioctl(dev, cmd, data, flag)
	caddr_t data;
	dev_t dev;
{
	int tmunit = UNIT_tm(dev);
	register struct tm_softc *sc = &tm_softc[tmunit];
	register struct buf *bp = &ctmbuf[UNIT_TM(dev)];
	register callcount;
	int fcount;
	struct mtop *mtop;
	struct mtget *mtget;
	/* we depend of the values and order of the MT codes here */
	static tmops[] =
	   {TM_WEOF,TM_SFORW,TM_SREV,TM_SFORW,TM_SREV,TM_REW,TM_OFFL,TM_SENSE};

	switch (cmd) {

	case MTIOCTOP:	/* tape operation */
		mtop = (struct mtop *)data;
		switch (mtop->mt_op) {

		case MTWEOF:
			callcount = mtop->mt_count;
			fcount = 1;
			break;

		case MTFSF: case MTBSF:
			callcount = mtop->mt_count;
			fcount = INF;
			break;

		case MTFSR: case MTBSR:
			callcount = 1;
			fcount = mtop->mt_count;
			break;

		case MTREW: case MTOFFL: case MTNOP:
			callcount = 1;
			fcount = 1;
			break;

		default:
			return (ENXIO);
		}
		if (callcount <= 0 || fcount <= 0)
			return (EINVAL);
		while (--callcount >= 0) {
			tmcommand(dev, tmops[mtop->mt_op], fcount);
			if ((mtop->mt_op == MTFSR || mtop->mt_op == MTBSR) &&
			    bp->b_resid)
				return (EIO);
			if ((bp->b_flags&B_ERROR) || sc->sc_erreg&TMER_BOT)
				break;
		}
		return (geterror(bp));

	case MTIOCGET:
		mtget = (struct mtget *)data;
		mtget->mt_dsreg = sc->sc_dsreg;
		mtget->mt_erreg = sc->sc_erreg;
		mtget->mt_resid = sc->sc_resid;
		mtget->mt_type = MT_ISTM;
		break;

	default:
		return (ENXIO);
	}
	return (0);
}

#define	DBSIZE	20

tmdump()
{
	return(ENODEV);
}

tmminphys(bp)
	struct buf *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, (63*1024));
}
#endif	NTM > 0
