/*
 * VMEBUS: Integrated Solutions VME-GP-SCSI driver subroutines
 */
#include "gs.h"

#if NGS > 0 || Ngs > 0
#undef	NGS
#undef	Ngs
#define NGS		4	/* max number of gp-scsi controllers per cpu */
#define Ngs_targ_cont	8	/* max number of target devs per gs cont */
#define ASCII_DCB_CERR
#define ASCII_SNS_7_KEY
#define ASCII_PDTYPE

#define DEBUG		0	/* */
#ifdef	DEBUG
int		gsdebug = DEBUG;
#endif	DEBUG

#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dk.h"
#include "../h/dkbad.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/scsi.h"
#include "../is68kdev/gsreg.h"
#include "../is68kdev/gsvar.h"

/*        _________________________________
 * ctlr:  | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 *        ---------------------------------
 *                 \_cont____/ \___targ__/
 */
#define	GS_CTLR(cont, targ)	(((cont) << 3) | (targ))

u_short *GSstd[] = {
		(u_short *)&vme_stdio[0xFFFFc0],
		(u_short *)&vme_stdio[0xFFFFd0],
		(u_short *)&vme_stdio[0xFFFFe0],
		(u_short *)&vme_stdio[0xFFFFf0],
		0 };

struct gsdevice *GSaddr[NGS];
u_short		gsident[NGS];
struct qb_ctlr	*gscinfo[NGS];
struct gs_hacb	gs_hacb[NGS];
int		gs_hacb_vdma[NGS];	/* VDMA mapping for hacb */
struct gs_queue	*gsqh[NGS * Ngs_targ_cont];
u_char		gsrunning[NGS];
int		gswstart, gswatch();	/* watchdog guardian */
u_char		gswticks[3][NGS][Ngs_targ_cont];
#define GS_WTIME  hz*5

/* Check that controller exists */
gsprobe(gsaddr, cont)
	register struct gsdevice	*gsaddr;
{
	struct gsdevice	**ap;

	if (GSaddr[cont]) {		/* already assigned and inited */
		cvec = 0;		/* vector already caught */
		goto ret;
	}
	if (gsaddr == (struct gsdevice *)-1)	/* wait for unassigned slot */
		return (0);

	/* if not a gp-scsi then ignore it */
	if ((gswait(gsaddr) & GS_LOC_ID_MASK) != GS_LOC_ID)
		return (0);

	/* do the init, this will generate an interrupt */
	if (gsinit(gsaddr, cont))
		return (0);

	/* need to save the real addr, mark slot assigned */
	GSaddr[cont] = gsaddr;
	for (ap = GSstd; *ap; ap++)
		if (*ap == gsaddr)
			*((int *)ap) = -1;

	if (gswstart == 0) {		/* start watchdog timer */
		timeout(gswatch, (caddr_t)0, GS_WTIME);
		gswstart++;
	}
ret:	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (16);
}

/* Handle a disk interrupt. */
gsintr(cont)
{
	register struct gs_hacb	*hacb = &gs_hacb[cont];
	struct gs_queue		*qp;
	int			targ;
	u_short			rdcr;
	struct hacb_dcb		*dcb;

	/*
	 * Figure out which target(s) interrupted by shifting through dcr to
	 * find a target with GO bit cleared and DB bit set.
	 */
	rdcr = hacb->hacb_dcr;
	for (targ = 0; targ < Ngs_targ_cont; targ++, rdcr = (rdcr>>1)&0x7FFF) {
		if (!(rdcr & DCR_BUSY) || (rdcr & DCR_GO))
			continue;
		gsrunning[cont]--;
		/* reset DB bit in dcr of interrupting device */
		do {
			hacb->hacb_semhost = 0;
			hacb->hacb_semhost = 1;
		} while (hacb->hacb_semhadpt);
		hacb->hacb_dcr &= ~(DCR_BUSY << targ);
		hacb->hacb_semhost = 0;
		gswticks[0][cont][targ] = gswticks[1][cont][targ] = 0;
		gswticks[2][cont][targ] = 0;
		qp = gsqh[GS_CTLR(cont, targ)];
		dcb = &hacb->hacb_dcb[targ];
#ifdef	DEBUG
		if (gsdebug)
		    printf("cont %d targ %d lun %d: %s: scsi_status %b\n",
			cont, targ, dcb->dcb_cdb.cdb_0.cdb_0_lun,
			ascii_dcb_cerr[dcb->dcb_cerr], dcb->dcb_scsi_status,
			SCSI_STATUS_BITS);
#endif	DEBUG

		/*
		 * Check for and process errors.  If doerr returns non-zero,
		 * then a new command is needed on this target immediately,
		 * and we should not move on to the next driver
		 */
		if ((*qp->gsq_doerr)(cont, targ, dcb)) {
			gsrun(cont, targ);
			continue;
		}

		/* if driver not deqd, give another driver a chance */
		if (gsqh[GS_CTLR(cont, targ)] == qp)
			gsqh[GS_CTLR(cont, targ)] = qp->gsq_forw;
		gsstart(cont, targ);
	}
	return (0);
}

/* fill in command and run */
gsstart(cont, targ)
{
	struct hacb_dcb *dcb = &gs_hacb[cont].hacb_dcb[targ];
	struct gs_queue	*qp;

	while (qp = gsqh[GS_CTLR(cont, targ)]) {
		if ((*qp->gsq_getcmd)(cont, targ, dcb)) {
			gsrun(cont, targ);
			return (0);
		}
		else
			gsqh[GS_CTLR(cont, targ)] = qp->gsq_forw;
	}
}

gsrun(cont, targ)
{
	register struct gs_hacb	*hacb = &gs_hacb[cont];
	int	c;

#ifdef	DEBUG
	if (gsdebug)
		gsprintdcb(cont, targ);
#endif	DEBUG

	if (hacb->hacb_dcr & ((DCR_GO | DCR_BUSY) << targ))
		panic("gs: target busy but not active");
	do {
		hacb->hacb_semhost = 0;
		hacb->hacb_semhost = 1;
	} while (hacb->hacb_semhadpt);
	hacb->hacb_dcr |= (DCR_GO << targ);
	hacb->hacb_semhost = 0;
	gsrunning[cont]++;
	GSaddr[cont]->gs_loc = GS_LOC_GO;
}

/* initialize queue entry */
gsqinit(gsq, getcmd, doerr)
	register struct gs_queue	*gsq;
	int				(*getcmd)(), (*doerr)();
{
	gsq->gsq_forw = (struct gs_queue *)0;
	gsq->gsq_back = (struct gs_queue *)0;
	gsq->gsq_getcmd = getcmd;
	gsq->gsq_doerr = doerr;
}

/* add entry to end of queue */
gsenq(cont, targ, gsq)
	register struct gs_queue	*gsq;
{
	register struct gs_queue	*qp = gsqh[GS_CTLR(cont, targ)];

	if (qp == 0) {
		gsq->gsq_forw = gsq;
		gsq->gsq_back = gsq;
		gsqh[GS_CTLR(cont, targ)] = gsq;
		gsstart(cont, targ);
	} else {
		gsq->gsq_forw = qp;
		gsq->gsq_back = qp->gsq_back;
		qp->gsq_back = gsq;
		gsq->gsq_back->gsq_forw = gsq;
	}
	return (0);
}

/* remove an entry from front of queue */
gsdeq(cont, targ, gsq)
	register struct gs_queue	*gsq;
{
	register struct gs_queue	*qp = gsqh[GS_CTLR(cont, targ)];

	if (gsq->gsq_forw == gsq)	/* q is empty */
		gsqh[GS_CTLR(cont, targ)] = 0;
	else {
		if (qp == gsq)		/* move head up */
			gsqh[GS_CTLR(cont, targ)] = gsq->gsq_forw;
		gsq->gsq_forw->gsq_back = gsq->gsq_back;
		gsq->gsq_back->gsq_forw = gsq->gsq_forw;
	}

	/* zero out pointers indicating driver is done */
	gsq->gsq_back = gsq->gsq_forw = 0;
	return (0);
}

gsinit(gsaddr, cont)
	register struct gsdevice	*gsaddr;
{
	register struct gs_hacb		*hacb = &gs_hacb[cont];
	register struct hacb_dcb	*dcb;
	int				hatarg, vdma;
	char				vers[8];

	bzero(hacb, sizeof(struct gs_hacb));
	/* give the init cmd */
	do {
	        gsaddr->gs_loc = GS_LOC_INIT | GS_LOC_INIT_FLG;
	} while (gsaddr->gs_loc != GS_LOC_INIT_OK);
	vdma = svqballoc(&gs_hacb[cont], sizeof(struct gs_hacb));
	gsaddr->gs_loc = 0;
	gsaddr->gs_loc = vdma >> 16;
	gsaddr->gs_loc = vdma;
	if (((gsident[cont] = gswait(gsaddr)) & GS_LOC_ID_MASK) != GS_LOC_ID)
		return (-1);

	/* issue version command to cause interrupt */
	hatarg = gsident[cont] & GS_LOC_HID_MASK;
	dcb = &hacb->hacb_dcb[hatarg];
	vdma = svqballoc(vers, sizeof(vers));
	SET_CDB_0(&dcb->dcb_cdb, CMD_VERSION, 0, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_IE, DCB_DIR_IN, 0, 0, sizeof(struct cdb_0),
		DCB_BW_16, DCB_AM_STD_S_D, 0, vdma, sizeof(vers), 1);
	gsrun(cont, hatarg);
	gsrunning[cont]--;
	if (gscmdwait(cont, hatarg, 800000))
		return (-1);
	do {
		hacb->hacb_semhost = 0;
		hacb->hacb_semhost = 1;
	} while (hacb->hacb_semhadpt);
	hacb->hacb_dcr = 0;
	hacb->hacb_semhost = 0;
	if (dcb->dcb_err)			/* make sure no errors */
		return (-1);
	return (0);
}

/* wait for polled mode command to complete */
gsfakeint(cont, targ, gsq, timeout)
	struct gs_queue *gsq;
{
	gsenq(cont, targ, gsq);		/* start up target pseudocontroller */
	do {
		if (gscmdwait(cont, targ, timeout)) {
			printf("gs%d: controller timed out\n",
				GS_CTLR(cont, targ));
			return (-1);
		}
		gsintr(cont);		/* fake interrupt */
	} while (gsrunning[cont]);
	return (0);
}

gscmdwait(cont, targ, timeout)
{
	register struct gs_hacb	*hacb = &gs_hacb[cont];
	int			targo = DCR_GO << targ;
	long			l = 0;

	/* wait for GO to be cleared */
	while (hacb->hacb_dcr & (targo)) {
	    if (l++ > timeout) {
		do {
		    hacb->hacb_semhost = 0;
		    hacb->hacb_semhost = 1;
		} while (hacb->hacb_semhadpt);
		hacb->hacb_dcr = 0;
		hacb->hacb_semhost = 0;
		return (-1);
	    }
	    DELAY(400);
	} 
	return (0);
}

gswait(gsaddr)
	register struct gsdevice	*gsaddr;
{
	long l = 0;

	while (gsaddr->gs_loc == 0xFFFF && ++l < 800000)
		;
	return (gsaddr->gs_loc);
}

gssplx(cont)
{
	return (splx(gscinfo[cont]->qm_psl));
}

/* reset target device */
gscmdabort(cont, targ)
{
	register struct gs_hacb	*hacb = &gs_hacb[cont];
	struct hacb_dcb		*dcb = &hacb->hacb_dcb[targ];
	int			rdcr;

	bzero(dcb, sizeof(struct hacb_dcb));
	dcb->dcb_ie = DCB_IE;
	do {
		hacb->hacb_semhost = 0;
		hacb->hacb_semhost = 1;
	} while (hacb->hacb_semhadpt);
	rdcr = hacb->hacb_dcr;
	hacb->hacb_dcr = (rdcr & ~(DCR_BUSY << targ)) | (DCR_GO << targ);
	hacb->hacb_semhost = 0;
}

/* has a dual function: look for lost interrupts, and for missed commands */
gswatch()
{
	u_char  cont, targ;
	u_short r, c;

	timeout(gswatch, (caddr_t)0, GS_WTIME);
	for (cont = 0; cont < NGS; cont++) {
	    if ((r = gs_hacb[cont].hacb_dcr) == 0)
		continue;
	    for (targ = 0; targ < Ngs_targ_cont; targ++, r >>= 1) {
		c = r & (DCR_GO | DCR_BUSY);
		switch (c) {
		    case DCR_GO:	/* command not accepted yet */
			if (gswticks[0][cont][targ]++ >= 20) {
				gswticks[0][cont][targ] = 0;
				printf("GS%d: possible missed cmd\n",
				    GS_CTLR(cont, targ));
				GSaddr[cont]->gs_loc = GS_LOC_GO;
				continue;
			}
		    case DCR_BUSY:	/* command done, not cleaned */
			if (gswticks[1][cont][targ]++ >= 20) {
				gswticks[1][cont][targ] = 0;
				printf("GS%d: lost interrupt\n",
				    GS_CTLR(cont, targ));
				gsintr(cont);
				continue;
			}
		    case (DCR_GO|DCR_BUSY): /* command in process */
			if (gswticks[2][cont][targ]++ >= 1000) {
				gswticks[2][cont][targ] = 0;
				printf("GS%d: hung dcr=%x: cmd aborted\n",
				    gs_hacb[cont].hacb_dcr, GS_CTLR(cont,targ));
/*				gscmdabort(cont, targ);/**/
				continue;
			}
		    case 0: /* device not in use */
				;
		}
	    }
	}
}

#ifdef	DEBUG
gsprintdcb(cont, targ)
{
	register u_char *c = (u_char *)&gs_hacb[cont].hacb_dcb[targ]; 
	register int i, s;

	s = spltty();
	printf("\nDCB: cont %d targ %d lun %d:", cont, targ,
	    gs_hacb[cont].hacb_dcb[targ].dcb_cdb.cdb_0.cdb_0_lun);
	for (i = 0 ; i < sizeof(struct hacb_dcb); i++) {
		if ((i % 16) == 0)
			printf("\n  %2d:", i);
		printf(" %2x", *c++);
	}
	printf("\nSpace to execute, 'q' to disable: ");
	while ((i = cngetc()) == -1)
		;
	if (i == 'q')
		gsdebug = 0;
	splx(s);
}
#endif	DEBUG
#endif
