/*
 * VMEBUS: Integrated Solutions GP-SCSI general purpose interface driver
 */
#include "gg.h"
#if NGG > 0 || Ngg > 0
#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"

#include "../is68kdev/qbvar.h"
#include "../h/ggio.h"

/*                _________________________________
 * minor(dev):    | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 *                ---------------------------------
 *                         \___cont__/ \___targ__/
 *                         \_________ctlr________/
 */
#define Ngg_targ_cont		8
#define GG_CTLR(cont, targ)	(((cont) << 3) | (targ))
#define GG_CONT(ctlr)		((ctlr) >> 3)
#define GG_TARG(ctlr)		((ctlr) & 7)
#define GG_MAXCNT		0x1FE00

/*
 * State of controller from last command. Since only one command can be done
 * at a time per target device, allocate one for each device on each controller.
 */
struct	ggstat       {	      /* pseudoctlr status block */
	struct	    {
	    unsigned f_nodisc : 1;	/* disallow disconnects */
	    unsigned f_swcd   : 1;	/* swap command data */
	    unsigned f_swmd   : 1;	/* swap media data */
	    unsigned f_open   : 1;	/* device open */
	    unsigned f_busy   : 1;	/* using dirc control struct */
	    unsigned f_ready  : 1;	/* ctrl struct filled,ready*/
	    unsigned f_want   : 1;	/* want to use ctrl struct */
	}		fbits;
#define			s_nodisc	fbits.f_nodisc
#define			s_swcd		fbits.f_swcd
#define			s_swmd		fbits.f_swmd
#define			s_open		fbits.f_open
#define			s_busy		fbits.f_busy
#define			s_ready		fbits.f_ready
#define			s_want		fbits.f_want
	int		s_lun;		/* selected lun */
	int		s_bs;		/* block size */
	struct hacb_dcb	s_dcb;		/* direct command to be done  */
} ggstat[NGG * Ngg_targ_cont];

union scsi_sns		gg_sns;
struct scsi_inq		gg_inq;
struct scsi_rcap        gg_rcap;

int	ggprobe(),	ggslave(),	ggattach(),	ggintr();
int	gggetcmd(),	ggdoerr(),	ggminphys();
caddr_t ggmalloc();
struct buf	*ggmembhp;

struct qb_device	*ggdinfo[NGG * Ngg_targ_cont];
struct buf 		rggbuf[NGG * Ngg_targ_cont];
extern struct	qb_ctlr *gscinfo[];
extern u_short		*GSstd[];
extern u_short		gsident[];
extern char		*ascii_dcb_cerr[], *ascii_sns_7_key[], *ascii_pdtype[];

/*
 * GG/gg controller/unit driver structure
 */
struct	qb_driver GGdriver =
	{ ggprobe, ggslave, ggattach, GSstd, "gg", ggdinfo, "GG", gscinfo };

/*
 * ggq is used to tell gs routines that we have commands for it, and to pass
 * pointers to the routines for getting commands and command completion.
 */
struct	gs_queue ggq[NGG * Ngg_targ_cont];

/* Check that controller exists, initialize it, fake it for pseudocontroller */
ggprobe(ggaddr, cont)
	u_short *ggaddr;
{
	return (gsprobe(ggaddr, cont));
}

/* Check that a given unit exists */
ggslave(qi, ggaddr)
	struct qb_device  *qi;
	u_short		   *ggaddr;
{
	int			ctlr = qi->qi_unit;
	register struct ggstat	*stat = &ggstat[ctlr];

	stat->s_nodisc = (qi->qi_flags & 1) ? DCB_DISC : DCB_NODISC;
	stat->s_swcd = (qi->qi_flags & 2) ? DCB_NOSW : DCB_SW;
	stat->s_swmd = (qi->qi_flags & 4) ? DCB_SW : DCB_NOSW;
	gsqinit(&ggq[ctlr], gggetcmd, ggdoerr);
	return (1);
}

/* Logically attach a unit to a drive/controler, and set up unit info */
ggattach(qi)
	struct qb_device  *qi;
{
	int	ctlr = qi->qi_unit;
	int	targ = qi->qi_slave;
	int	lun;

	if (targ == (gsident[GG_CONT(ctlr)] & GS_LOC_HID_MASK))
		ggvers(qi);
	else
	    for (lun = 0; lun < 8; lun++)
		gginq(qi, lun);
}

gggetcmd(cont, targ, dcb)
	struct hacb_dcb	   *dcb;
{
	int		ctlr = GG_CTLR(cont, targ);
	struct ggstat   *stat = &ggstat[ctlr];

	if (stat->s_ready) {		/* a direct cmd needs to be done */
		bcopy(&stat->s_dcb, dcb, sizeof(struct hacb_dcb));
		return (1);		/* and get it sent off to target */
	} else {
		gsdeq(cont, targ, &ggq[ctlr]);
		return (0);
	}
}

/* do upper level interrupt processing */
ggdoerr(cont, targ, dcb)
	struct hacb_dcb *dcb;
{
	int			ctlr = GG_CTLR(cont, targ);
	register struct ggstat	*stat = &ggstat[ctlr];

	/* clear ready flag, copy status, and wakeup caller if inte.  */
	stat->s_ready = 0;
	stat->s_dcb.dcb_err = dcb->dcb_err;
	stat->s_dcb.dcb_cerr = dcb->dcb_cerr;
	stat->s_dcb.dcb_scsi_status = dcb->dcb_scsi_status;
	if (dcb->dcb_ie)		/* we're in full interrupt mode */
		wakeup(&stat->s_dcb);	/* command done - wake him up */
	gsdeq(cont, targ, &ggq[ctlr]);
	return (0);
}

ggstrategy(bp)
	register struct buf	*bp;
{
	int				ctlr = minor(bp->b_dev);
	int				cont = GG_CONT(ctlr);
	int				targ = GG_TARG(ctlr);
	register struct ggstat		*stat = &ggstat[ctlr];
	register struct hacb_dcb	*dcb = &stat->s_dcb;
	int				s;

	if (stat->s_bs == 0)
		goto bad;
	while (stat->s_busy) {
		stat->s_want = 1;
		sleep(stat, PRIBIO);
	}
	stat->s_busy = 1;
	SET_CDB_1(&dcb->dcb_cdb, (bp->b_flags & B_READ)?CMD_XREAD:CMD_XWRITE,
	    stat->s_lun, 0, bp->b_blkno/(stat->s_bs/512), 
	    (bp->b_bcount + (stat->s_bs-1))/stat->s_bs, 0, 0);
	SET_DCB(dcb, DCB_IE, (bp->b_flags & B_READ)?DCB_DIR_IN:DCB_DIR_OUT,
	    stat->s_swmd, stat->s_nodisc, sizeof(struct cdb_1), DCB_BW_16,
	    DCB_AM_STD_S_D, 0, qbaddr(bp), bp->b_bcount & ~1, stat->s_bs);
	s = gssplx(cont);
	stat->s_ready = 1;
	if (ggq[ctlr].gsq_forw == 0)
		gsenq(cont, targ, &ggq[ctlr]);
	sleep(dcb, PRIBIO);
	splx(s);
	stat->s_busy = 0;
	if (stat->s_want) {
		stat->s_want = 0;
		wakeup(stat);
	}
	if (stat->s_dcb.dcb_err)
		goto bad;
	bp->b_resid = 0;
	iodone(bp);
	return;
bad:	bp->b_flags |= B_ERROR;
	iodone(bp);
}

ggread(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int	ctlr = minor(dev);

	if (ctlr >= NGG * Ngg_targ_cont)
		return (ENXIO);
	return (physio(ggstrategy, &rggbuf[ctlr], dev, B_READ, ggminphys, uio));
}

ggwrite(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int	ctlr = GG_TARG(minor(dev));

	if (ctlr >= NGG * Ngg_targ_cont)
		return (ENXIO);
	return (physio(ggstrategy, &rggbuf[ctlr], dev, B_WRITE, ggminphys, uio));
}

ggioctl(dev, cmd, op_dcb)
	dev_t				dev;
	register struct hacb_dcb	*op_dcb;
{
	int				ctlr = minor(dev);
	int				cont = GG_CONT(ctlr);
	int				targ = GG_TARG(ctlr);
	register struct ggstat		*stat = &ggstat[ctlr];
	register struct hacb_dcb	*dcb = &stat->s_dcb;
	caddr_t				ump, smp;
	int				s;
	int				error = 0;

	if (stat->s_open == 0)
		return (NODEV);
	while (stat->s_busy) {
		stat->s_want = 1;
		sleep(stat, PRIBIO);
	}
	stat->s_busy = 1;
	bcopy(op_dcb, dcb, sizeof(struct hacb_dcb));
	dcb->dcb_ie = 1;
	if (dcb->dcb_dlen) {
		ump = dcb->dcb_dadr;
		if (useracc(ump, dcb->dcb_dlen,
		    (dcb->dcb_dir == DCB_DIR_IN) ? B_WRITE : B_READ) == NULL) {
			error = EFAULT;
			goto err;
		}
		smp = ggmalloc(dcb->dcb_dlen);
		if (dcb->dcb_dir == DCB_DIR_OUT)
			copyin(ump, smp, dcb->dcb_dlen);
		else
			bzero(smp, dcb->dcb_dlen);
		dcb->dcb_dadr = (caddr_t)svtop(smp);
	}
	s = gssplx(cont);
	stat->s_ready = 1;
	if (ggq[ctlr].gsq_forw == 0)
		gsenq(cont, targ, &ggq[ctlr]);
	sleep(dcb, PRIBIO);
	splx(s);
	op_dcb->dcb_err = dcb->dcb_err;
	op_dcb->dcb_cerr = dcb->dcb_cerr;
	op_dcb->dcb_scsi_status = dcb->dcb_scsi_status;
	if (dcb->dcb_dlen) {
		if (dcb->dcb_dir == DCB_DIR_IN)
			copyout(smp, ump, dcb->dcb_dlen);
		ggmfree(smp);
	}
err:	stat->s_busy = 0;
	if (stat->s_want) {
		stat->s_want = 0;
		wakeup(stat);
	}
	return (error);
}

ggopen(dev)
	dev_t dev;
{
	int			ctlr = minor(dev);
	struct qb_device	*qi;
	struct ggstat		*stat;

	if (!suser())
		return (u.u_error);
	if (ctlr >= NGG * Ngg_targ_cont || (qi = ggdinfo[ctlr]) == 0 ||
	    qi->qi_alive == 0)
		return (ENXIO);
	stat = &ggstat[ctlr];
	if (stat->s_open)
		return (EBUSY);
	stat->s_open = 1;
	return (0);
}

ggclose(dev)
	dev_t dev;
{
	int			ctlr = minor(dev);
	struct qb_device	*qi;

	ggstat[ctlr].s_open = 0;
	return (0);
}

ggissue(qi, stat, dcb, sense)
	register struct qb_device	*qi;
	register struct ggstat		*stat;
	register struct hacb_dcb	*dcb;
{
	int	lun;

	stat->s_ready = 1;
	if (gsfakeint(qi->qi_ctlr, qi->qi_slave, &ggq[qi->qi_unit], 800000) ||
	    dcb->dcb_cerr)
		return (-1);
	if (dcb->dcb_err == 0)
		return (0);
	if (sense && (dcb->dcb_scsi_status&SCSI_STATUS_CHECK)) {
		lun = dcb->dcb_cdb.cdb_0.cdb_0_lun;
		SET_CDB_0(&dcb->dcb_cdb, CMD_SENSE, lun, 0,
		    sizeof(union scsi_sns), 0, 0);
		SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, DCB_NODISC,
		    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
		    svtop(&gg_sns), sizeof(union scsi_sns), 1);
		stat->s_ready = 1;
		gsfakeint(qi->qi_ctlr, qi->qi_slave, &ggq[qi->qi_unit], 800000);
		if (gg_sns.sns_7.sns_7_class == 7)
			return (gg_sns.sns_7.sns_7_key);
	}
	return (-1);
}

ggvers(qi)
	struct qb_device	*qi;
{
	struct ggstat			*stat = &ggstat[qi->qi_unit];
	register struct hacb_dcb	*dcb = &stat->s_dcb;
	char				vers[8];

	bzero(vers, sizeof(vers));     /* clear from last command */
	SET_CDB_0(&dcb->dcb_cdb, CMD_VERSION, 0, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, DCB_NOSW, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
	    svtop(vers), sizeof(vers), 1);
	if (ggissue(qi, stat, dcb, 0)) {
		printf("	Version failed: err=%x/%x",
			dcb->dcb_cerr, dcb->dcb_scsi_status);
		return (-1);
	}
	printf("	HOST ADAPTOR: %8s", vers);
}

gginq(qi, lun)
	struct qb_device	*qi;
{
	register struct ggstat		*stat = &ggstat[qi->qi_unit];
	register struct hacb_dcb	*dcb = &stat->s_dcb;

	SET_CDB_0(&dcb->dcb_cdb, CMD_INQUIRY, lun, 0, sizeof(struct scsi_inq),
	    0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
	    svtop(&gg_inq), sizeof(struct scsi_inq), 1);
	bzero(&gg_inq, sizeof(struct scsi_inq));
	if (ggissue(qi, stat, dcb, 1)) {
	    SET_CDB_0(&dcb->dcb_cdb, CMD_INQUIRY, lun, 0, 
		sizeof(struct scsi_inq), 0, 0);
	    SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, DCB_NODISC,
		sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
		svtop(&gg_inq), sizeof(struct scsi_inq), 1);
	    bzero(&gg_inq, sizeof(struct scsi_inq));
	    if (ggissue(qi, stat, dcb, 1))
		return (-1);
	}
	if (gg_inq.inq_pdtype == INQ_PDT_NOLUN)
		return (-1);
	if (lun)
		printf("\n				");
	printf("	%d:", lun);
	if (gg_inq.inq_pdtype <= INQ_PDT_WORM)
		printf("%6s: ", ascii_pdtype[gg_inq.inq_pdtype]);
	else
		printf("PDT %2x: ", gg_inq.inq_pdtype);
	printf("%8s %10s %4s", gg_inq.inq_vendor, gg_inq.inq_product,
		gg_inq.inq_rev);

	/* start the unit and do a read capacity for read/write interface */
	SET_CDB_0(&dcb->dcb_cdb, CMD_STARTSTOP, lun, 0, 1, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, 0, DCB_NODISC, sizeof(struct cdb_0),
	    0, 0, 0, 0, 0, 0);
	if (ggissue(qi, stat, dcb, 1))
		return (0);
	DELAY(200000);			/* warm up, dont want BUSY status */
	SET_CDB_1(&dcb->dcb_cdb, CMD_RCAPAC, lun, 0, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->s_swcd, DCB_NODISC,
	    sizeof(struct cdb_1), DCB_BW_16, DCB_AM_STD_S_D, 0, 
	    svtop(&gg_rcap), sizeof(gg_rcap), 1);
	if (ggissue(qi, stat, dcb, 1) ||
	    gg_rcap.rcap_bl < 512 || (gg_rcap.rcap_bl % 512) != 0)
		return (0);
	stat->s_bs = gg_rcap.rcap_bl;
	return (0);
}

ggintr(cont)
{
	return (gsintr(cont));
}

ggminphys(bp)
	struct buf	*bp;
{
	bp->b_bcount = MIN(bp->b_bcount, GG_MAXCNT);
}

caddr_t
ggmalloc(len)
{
	struct buf	*bp = (struct buf *)geteblk(len);

	bp->b_actf = ggmembhp;
	ggmembhp = bp;
	return (bp->b_un.b_addr);
}

ggmfree(addr)
	register caddr_t addr;
{
	register struct buf	*bp;

	for (bp = ggmembhp; bp; bp = bp->b_actf)
		if (bp->b_un.b_addr == addr) {
			bfree(bp);
			brelse(bp);
			return (0);
		}
	panic("gg: lost buffer pointer");
}
#endif
