/*
 * VMEBUS: Integrated Solutions GP-SCSI disk driver
 */
#include "gd.h"

#if NGD > 0 || Ngd > 0
#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dk.h"
#include "../h/dkbad.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/scsi.h"
#include "../is68kdev/gsreg.h"
#include "../is68kdev/gsvar.h"
#include "../h/gdio.h"
/*                _________________________________
 * minor(dev):    | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 *                ---------------------------------
 *                 \_cont/ \_targ/ \l/ \partition/
 *                 \_____ctlr____/
 *                 \_______unit______/
 */
#define Ngd_lun_targ		2
#define Ngd_targ_cont		4
#define	GDD_PART(dev)		((dev) & 7)
#define	GDD_UNIT(dev)		((dev) >> 3)
#define	GDU_LUN(unit)		((unit)&1)
#define	GDU_CTLR(unit)		(((unit) >> 1)&0xF)
#define	GDU_TARG(unit)		(GDU_CTLR(unit) & 3)
#define	GDU_CONT(unit)		(GDU_CTLR(unit) >> 2)
#define GDCL_UNIT(ctlr, lun)	(((ctlr) << 1) | (lun))
#define GDCT_CTLR(cont, targ)	(((cont) << 2) | (targ))

#define GD_MAXCNT	0x1FE00
#define	GD_MAXRETRY	3

/*
 * State of controller from last transfer. Since only one transfer can be done
 * at a time per target device, allocate one for each device on each controller.
 */
struct	gdstat       {	       /* pseudoctlr status block */
	struct	{
	    u_char f_nodisc : 1;	/* disallow disconnects */
	    u_char f_swcd   : 1;	/* swap command data */
	    u_char f_swmd   : 1;	/* swap media data */
	    u_char f_noerr  : 1;	/* ignore errors */
	    u_char f_sense  : 1;	/* sense was requested */
	    u_char f_direct : 1;	/* direct polled mode command */
			    : 2;
	    u_char f_lun;		/* current/next lun to access */
	}		gd_statbits;
#define			    gd_nodisc	gd_statbits.f_nodisc
#define			    gd_swcd	gd_statbits.f_swcd
#define			    gd_swmd	gd_statbits.f_swmd
#define			    gd_noerr	gd_statbits.f_noerr
#define			    gd_sense	gd_statbits.f_sense
#define			    gd_direct	gd_statbits.f_direct
#define			    gd_lun	gd_statbits.f_lun
	caddr_t		gd_addr;	/* physical address of transfer */
	u_int		gd_lba;		/* block number */
	int		gd_bleft;	/* bytes left to transfer */
	int		gd_bpart;	/* bytes transferred */
	struct hacb_dcb	gd_dirc;	/* direct command block */
	struct buf	gd_qhd[Ngd_lun_targ]; /* heads of drive queues */
	union scsi_sns	gd_sns;		/* for sense on normal r/w */
	int		gd_sns_vdma;	/* vdma address for sense on normal r/w */
} gdstat[NGD * Ngd_targ_cont];

int	gdprobe(),	gdslave(),	gdattach(),	gdintr();
int	gdstrat(),	gdminphys();
int	gdgetcmd(),	gddoerr();

struct	qb_device	*gddinfo[Ngd];
extern	struct	qb_ctlr *gscinfo[];
extern	u_short		*GSstd[];
extern	char		*ascii_dcb_cerr[], *ascii_sns_7_key[], *ascii_pdtype[];

/* GD/gd controller/unit driver structure */
struct	qb_driver GDdriver =
	{ gdprobe, gdslave, gdattach, GSstd, "gd", gddinfo, "GD", gscinfo };

struct qb_diskst gdst[Ngd];

struct scsi_inq		gd_inq;
int			gd_inq_vdma;
struct scsi_msen	gd_msen;
int			gd_msen_vdma;
struct scsi_msel	gd_msel;
int			gd_msel_vdma;
struct scsi_rcap	gd_rcap;
int			gd_rcap_vdma;

/*
 * gdq is used to tell gs routines that we have commands for it, and to pass
 * pointers to the routines for getting commands and handling completion.
 */
struct gs_queue	gdq[NGD * Ngd_targ_cont];
struct buf	rgdbuf[Ngd];
struct dkbad	gdbad[Ngd];
#define b_lba	b_resid

/* Check that controller exists */
gdprobe(gdaddr, cont)
	ushort	*gdaddr;
{
	return (gsprobe(gdaddr, cont));
}

/* Check that a given unit exists and is really a drive */
gdslave(qi, gdaddr)
	struct qb_device	*qi;
	struct gddevice		*gdaddr;
{
	int		ctlr = GDU_CTLR(qi->qi_unit);
	struct gdstat	*stat = &gdstat[ctlr];

	stat->gd_nodisc = (qi->qi_flags & 1) ? DCB_DISC : DCB_NODISC;
	stat->gd_swcd = (qi->qi_flags & 2) ? DCB_NOSW : DCB_SW;
	stat->gd_swmd = (qi->qi_flags & 4) ? DCB_SW : DCB_NOSW;
	if (gd_inq_vdma == 0) {
		gd_inq_vdma = svqballoc(&gd_inq, sizeof(gd_inq));
		gd_msen_vdma = svqballoc(&gd_msen, sizeof(gd_msen));
		gd_msel_vdma = svqballoc(&gd_msel, sizeof(gd_msel));;
		gd_rcap_vdma = svqballoc(&gd_rcap, sizeof(gd_rcap));;
		stat->gd_sns_vdma = svqballoc(&stat->gd_sns, sizeof(stat->gd_sns));
	}
	gsqinit(&gdq[ctlr], gdgetcmd, gddoerr);
	return gdinq(qi);
}

/* Logically attach a unit to a drive/controler, and set up unit info */
gdattach(qi)
	struct qb_device	*qi;
{
	if (gdgeom(qi) || gdslctmod(qi))
		qi->qi_alive = 0;
}

gdgetcmd(cont, targ, dcb)
	register struct hacb_dcb	*dcb;
{
	int			ctlr = GDCT_CTLR( cont, targ );
	register struct gdstat	*stat = &gdstat[ctlr];
	int			lun, dk;
	struct buf		*bp;

	/* if direct command, copy stuff into dcb and return (1); */
	if (stat->gd_direct) {
		stat->gd_lun = stat->gd_dirc.dcb_cdb.cdb_0.cdb_0_lun;
		bcopy(&stat->gd_dirc, dcb, sizeof(struct hacb_dcb));
		return (1);
	}

	/* find the next lun with a pending command, deq and return if none */
	lun = stat->gd_lun;
	do {
		if (bp = stat->gd_qhd[lun].b_actf)
			break;
	} while ((lun = ((lun + 1) % Ngd_lun_targ)) != stat->gd_lun);
	if (bp == 0) {
		gsdeq(cont, targ, &gdq[ctlr]);
		return (0);
	}
	stat->gd_lun = lun;

	/* save away current data transfer drive info */
	if (stat->gd_bleft == 0) { /* not already working on a buffer */
		stat->gd_lba	= bp->b_lba;
		stat->gd_addr	= (caddr_t )qbaddr(bp);
		stat->gd_bleft	= bp->b_bcount;
	}
	stat->gd_bpart = MIN(stat->gd_bleft, GD_MAXCNT);

	SET_CDB_1(&dcb->dcb_cdb, (bp->b_flags & B_READ)?CMD_XREAD:CMD_XWRITE,
	    lun, 0, stat->gd_lba, (stat->gd_bpart + 511) >> 9, 0, 0);
	SET_DCB(dcb, DCB_IE, (bp->b_flags & B_READ)?DCB_DIR_IN:DCB_DIR_OUT,
	    stat->gd_swmd, stat->gd_nodisc, sizeof(struct cdb_1), DCB_BW_16,
	    DCB_AM_STD_S_D, 0, stat->gd_addr, stat->gd_bpart & ~1, 512);

	dk = gddinfo[GDCL_UNIT(ctlr, lun)]->qi_dk;
	if (dk >= 0) {
		dk_busy |= 1 << dk;
		dk_xfer[dk]++;
		dk_wds[dk] += bp->b_bcount >> 6;
	}
	return (1);
}

/* do upper level interrupt processing */
gddoerr(cont, targ, dcb)
	register struct hacb_dcb	*dcb;
{
	int			ctlr = GDCT_CTLR( cont, targ );
	register struct gdstat	*stat = &gdstat[ctlr];
	register union scsi_sns	*sns = &stat->gd_sns;
	int			lun = stat->gd_lun;
	register struct buf	*bp = stat->gd_qhd[lun].b_actf;
	int			dk = gddinfo[GDCL_UNIT(ctlr, lun)]->qi_dk;

	if (stat->gd_noerr) {
	    stat->gd_direct = 0;
	    if (stat->gd_sense) {
		if (sns->sns_7.sns_7_class == 7 && 
		    sns->sns_7.sns_7_key == SNS_7_KEY_RECOVERED)
	    		stat->gd_dirc.dcb_err = 0;
		return (stat->gd_sense = 0);
	    }
	    stat->gd_dirc.dcb_err = dcb->dcb_err;
	    stat->gd_dirc.dcb_cerr = dcb->dcb_cerr;
	    stat->gd_dirc.dcb_scsi_status = dcb->dcb_scsi_status;
	    if (dcb->dcb_err && (dcb->dcb_scsi_status & SCSI_STATUS_CHECK)) {
		stat->gd_sense = 1;
		SET_CDB_0(&dcb->dcb_cdb, CMD_SENSE, lun, 0,
		    sizeof(union scsi_sns), 0, 0);
		SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
		    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
		    stat->gd_sns_vdma, sizeof(union scsi_sns), 1);
		return (1);
	    }
	    return (0);
	}

	/* if call was a sense request; finish off processing.  */
	if (stat->gd_sense) {
		if (dcb->dcb_err) {
			printf("sense failed: %s: scsi %b",
				ascii_dcb_cerr[dcb->dcb_cerr],
				dcb->dcb_scsi_status, SCSI_STATUS_BITS);
			if ((stat->gd_sense++ < 3) &&
			    (dcb->dcb_scsi_status &  SCSI_STATUS_CHECK)) {
				printf(": retried\n");
				return (1);
			}
			printf("no more retries");
			bzero(sns, sizeof(union scsi_sns));
			dcb->dcb_err = 0;
		}
		stat->gd_sense = 0;
		if (sns->sns_7.sns_7_class != 7) {
			printf("invalid sense class %d", 
				sns->sns_7.sns_7_class);
			bzero(sns, sizeof(union scsi_sns));
		}
		if (sns->sns_7.sns_7_key)
			printf("%s: code %x",
				ascii_sns_7_key[sns->sns_7.sns_7_key],
				sns->sns_7.sns_7_err);
		if (sns->sns_7.sns_7_valid)
			printf(": bn %d", SCSI_HMML(sns->sns_7.sns_7_info));
		if (sns->sns_7.sns_7_key == SNS_7_KEY_RECOVERED)
			printf(": soft error\n");
		else if (++stat->gd_qhd[lun].b_errcnt < GD_MAXRETRY) {
			printf(": reissued\n");
			return (0);
		} else if (!stat->gd_direct) {
			printf("\n");
			harderr(bp, "gd");
			bp->b_flags |= B_ERROR;
		}
	}

	/* check errors, get sense if available */
	if (dcb->dcb_err) {
		printf("gd%d: ", GDCL_UNIT(ctlr, lun));
		if (dcb->dcb_cerr)
			printf("%s", ascii_dcb_cerr[dcb->dcb_cerr]);
		else if (dcb->dcb_scsi_status & SCSI_STATUS_CHECK) {
			/* start sense request processing */
			stat->gd_sense = 1;
			SET_CDB_0(&dcb->dcb_cdb, CMD_SENSE, lun, 0,
			    sizeof(union scsi_sns), 0, 0);
			SET_DCB(dcb, (stat->gd_direct?DCB_NOIE:DCB_IE),
			    DCB_DIR_IN, stat->gd_swcd, stat->gd_nodisc,
			    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0,
			    stat->gd_sns_vdma, sizeof(union scsi_sns), 1);
			return (1);
		} else
			printf("scsi status %b",
				dcb->dcb_scsi_status, SCSI_STATUS_BITS);
		if (++stat->gd_qhd[lun].b_errcnt < GD_MAXRETRY) {
			printf(": retried\n");
			return (0);
		}
		if (!stat->gd_direct)
			bp->b_flags |= B_ERROR;
		printf("\n");
	}
	stat->gd_qhd[lun].b_errcnt = 0;

	/* return if direct command */
	if (stat->gd_direct)
		return (stat->gd_direct = 0);

	/*
	 * check if more data from previous request; if so, update stat struct.
	 * Leave gd_lun alone so we're next.
	 */
	if (((bp->b_flags & B_ERROR) == 0) &&
	    ((stat->gd_bleft -= stat->gd_bpart) > 0)) {
		stat->gd_addr += stat->gd_bpart;
		stat->gd_lba += (( stat->gd_bpart + 511 ) >> 9 );
		return (0);
	}

	/*
	 * if no more data, or had unrecoverable error, clear dk_busy, reset
	 * errcnt, set b_resid, remove buf from qhead, and mark buffer iodone.
	 * Advance to next gd_lun so others get a chance.
	 */
	dk_busy &= ~(1 << dk);
	bp->b_resid = stat->gd_bleft;
	stat->gd_bleft = 0;
	stat->gd_qhd[lun].b_actf = bp->av_forw;
	iodone(bp);
	stat->gd_lun = (lun + 1) % Ngd_lun_targ;
	return (0);
}

gdioctl(dev, cmd, data)
	dev_t	dev;
	char	*data;
{
	int		unit = GDD_UNIT(minor(dev));
	struct gdstat	*stat = &gdstat[GDU_CTLR(unit)];

	switch (cmd) {
	    case GDIOCDISC:
		stat->gd_nodisc = DCB_DISC;
		return(0);
		break;
	    case GDIOCNODISC:
		stat->gd_nodisc = DCB_NODISC;
		return(0);
		break;
	    case DIOCGGEO:
		if (minor(dev) > Ngd)
			return(ENXIO);
		*(struct qb_diskst *)data = gdst[unit];
		return(0);
		break;
	    default:
		return(EIO);
		break;
	}
}

gdopen(dev)
	dev_t	dev;
{
	int			unit = GDD_UNIT(minor(dev));
	struct qb_device	*qi;

	if (unit >= Ngd || (qi = gddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	return (0);
}

gdstrategy(bp)
	register struct buf	*bp;
{
	struct qb_device	*qi;
	register struct qb_diskst	*st;
	int			unit = GDD_UNIT(minor((bp)->b_dev));
	int			partition = GDD_PART(minor(bp->b_dev));

	if (unit >= Ngd)
		goto bad;
	qi = gddinfo[unit];
	if (qi == 0 || qi->qi_alive == 0)
		goto bad;
	st = &gdst[unit];
	if (bp->b_blkno < 0 || bp->b_blkno + ((bp->b_bcount+511) >> 9) >
	    ST_size[partition].nblocks)
		goto bad;
	badstrat(qi, bp, &gdbad[qi->qi_unit], gdstrat, ST_nspt, ST_ntpc,
		ST_ncpd, ST_size[partition].cyloff);
	return;

bad:	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
}

gdstrat(qi, bp)
	struct qb_device	*qi;
	register struct buf	*bp;
{
	int		partition = GDD_PART(minor(bp->b_dev)), s;
	int		unit = GDD_UNIT(minor(bp->b_dev));
	int		ctlr = GDU_CTLR(unit);
	int		cont = GDU_CONT(unit);
	struct qb_diskst	*st = &gdst[unit];

	bp->b_lba = bp->b_blkno + (ST_size[partition].cyloff * ST_nspc);

	/* put buffer on units queue for gdgetcmd to remove when called */
	s = gssplx(cont);
	disksort(&gdstat[ctlr].gd_qhd[GDU_LUN(unit)], bp);

	/* if we aren't queued for this target, put us on */
	if (gdq[ctlr].gsq_forw == (struct gs_queue *)0)
		gsenq(cont, GDU_TARG(unit), &gdq[ctlr]);
	splx(s);
}

gdread(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int	unit = GDD_UNIT(minor(dev));

	if (unit >= Ngd)
		return (ENXIO);
	return (physio(gdstrategy, &rgdbuf[unit], dev, B_READ, gdminphys, uio));
}

gdwrite(dev, uio)
	dev_t		dev;
	struct uio	*uio;
{
	int	unit = GDD_UNIT(minor(dev));

	if (unit >= Ngd)
		return (ENXIO);
	return (physio(gdstrategy, &rgdbuf[unit], dev, B_WRITE, gdminphys, uio));
}

gdreset(ctlr)
{
}

int gddumpstrat();
extern struct buf dumpbuf;
extern int dumpsize;

gddump(dev)
	dev_t	dev;
{
	register struct buf		*bp = &dumpbuf;
	register struct qb_device	*qi;
	register struct qb_diskst		*st;
	int				num, start, bcount;
	int				unit = GDD_UNIT(minor(dev));
	int				partition = GDD_PART(minor(dev));

	if (unit >= Ngd || (qi = gddinfo[unit]) == 0 || qi->qi_alive == 0)
		return ENXIO;
	num = ctod(dumpsize);		/* memory size in disk blocks */
	start = 0x0;			/* start dumping at physical 0 */
	st = &gdst[unit];
	if (dumplo < 0 || dumplo+num > ST_size[partition].nblocks)
		return EINVAL;
	while (num > 0) {
		bcount = (dbtob(num) > GD_MAXCNT) ? GD_MAXCNT : dbtob(num);
		bp->b_dev = dev;
		bp->b_blkno = dumplo + btodb(start);
		bp->b_lba = bp->b_blkno +
			(ST_size[partition].cyloff * ST_nspc);
		bp->b_bcount = bcount;
		bp->b_un.b_addr = (caddr_t) start;
		bp->b_flags = B_WRITE;
		badstrat(qi, bp, &gdbad[qi->qi_unit], gddumpstrat, ST_nspt,
		    ST_ntpc, ST_ncpd, ST_size[partition].cyloff);
		if (bp->b_flags & B_ERROR || bp->b_resid)
			return EIO;
		start += bcount;
		num -= btodb(bcount);
	}
	return 0;
}

gddumpstrat(qi, bp)
	register struct qb_device	*qi;
	register struct buf		*bp;
{
	register struct gdstat	*stat = &gdstat[qi->qi_ctlr];
	struct hacb_dcb		*dcb = &stat->gd_dirc;

	SET_CDB_1(&dcb->dcb_cdb, CMD_XWRITE, GDU_LUN(qi->qi_unit), 0,
	    bp->b_lba, (bp->b_bcount + 511) >> 9, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_OUT, stat->gd_swmd, DCB_NODISC,
	    sizeof(struct cdb_1), DCB_BW_16, DCB_AM_STD_S_D, 0, bp->b_un.b_addr,
	    bp->b_bcount, 512);
	if (gdissue("WRITE", dcb, qi->qi_unit))
		bp->b_flags |= B_ERROR;
	bp->b_resid = 0;
	bp->b_flags |= B_DONE;
	return 0;
}

gdsize(dev)
	dev_t	dev;
{
	struct qb_device	*qi;
	struct qb_diskst		*st;
	int			unit = GDD_UNIT(minor(dev));

	if (unit >= Ngd || (qi = gddinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	st = &gdst[unit];
	return (ST_size[GDD_PART(minor(dev))].nblocks);
}

gdminphys(bp)
	struct buf	*bp;
{
	bp->b_bcount = MIN(bp->b_bcount, GD_MAXCNT);
}

gdissue(command, dcb, unit)
	register char			*command;
	register struct hacb_dcb	*dcb;
{
	register struct gdstat	*stat = &gdstat[GDU_CTLR(unit)];

	stat->gd_direct = 1;
	if (gsfakeint(GDU_CONT(unit), GDU_TARG(unit),
	    &gdq[GDU_CTLR(unit)], 800000) || dcb->dcb_err) {
		stat->gd_direct = 0;
		if (command)
			printf("%s: command failed\n", command);
		return (-1);
	}
	stat->gd_direct = 0;
	return 0;
}

gdinq(qi)
	struct qb_device  *qi;
{
	int				unit = qi->qi_unit;
	int				ctlr = GDU_CTLR(unit);
	register struct gdstat		*stat = &gdstat[ctlr];
	register struct hacb_dcb	*dcb = &stat->gd_dirc;
	register int			lun = GDU_LUN(unit);

	stat->gd_noerr = 1;
	SET_CDB_0(&dcb->dcb_cdb, CMD_INQUIRY, lun, 0, sizeof(gd_inq), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_inq_vdma,
	    sizeof(gd_inq), 1);
	if (gdissue(0, dcb, unit) && (dcb->dcb_cerr || gdissue(0, dcb, unit))) {
		stat->gd_noerr = 0;
		return (0);
	}
	stat->gd_noerr = 0;
	if (gd_inq.inq_pdtype != INQ_PDT_DISK)
		return (0);

	SET_CDB_0(&dcb->dcb_cdb, CMD_STARTSTOP, lun, 0, 1, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, 0, DCB_NODISC, sizeof(struct cdb_0),
	    0, 0, 0, 0, 0, 0);
	if (gdissue("START", dcb, unit))
		return (0);
	DELAY(200000);			/* warm up, dont want BUSY status */

	SET_CDB_0(&dcb->dcb_cdb, CMD_TESTREADY, lun, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_NO, 0, DCB_NODISC, sizeof(struct cdb_0),
	    0, 0, 0, 0, 0, 0);
	if (gdissue("TESTREADY", dcb, unit))
		return (0);
	return (1);
}

gdgeom(qi)
	struct qb_device  *qi;
{
	int				unit = qi->qi_unit;
	int				ctlr = GDU_CTLR(unit);
	struct qb_diskst			*st = &gdst[unit];
	register struct gdstat		*stat = &gdstat[ctlr];
	register struct hacb_dcb	*dcb = &stat->gd_dirc;
	register int			lun = GDU_LUN(unit);
	register struct pag_desc	*pd;
	register int			ncpd, ntpc, nspt, spd, spc;

	/* get capacity information */
	SET_CDB_1(&dcb->dcb_cdb, CMD_RCAPAC, lun, 0, 0, 0, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_1), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_rcap_vdma,
	    sizeof(gd_rcap), 1);
	if (gdissue("RCAPAC", dcb, unit))
		return (-1);
	if (gd_rcap.rcap_bl != 512) {
		printf("gd%d: bad blocksize %d\n", unit, gd_rcap.rcap_bl);
		return (-1);
	}
	spd = gd_rcap.rcap_lba + 1;	/* sectors per disk */

	/* Get cylinder boundary, by setting PMI */
	SET_CDB_1(&dcb->dcb_cdb, CMD_RCAPAC, lun, 0, 1, 1, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_1), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_rcap_vdma,
	    sizeof(gd_rcap), 1);
	if (gdissue("RCAPAC PMI", dcb, unit))
		return (-1);
	spc = gd_rcap.rcap_lba + 1;	/* sectors per cylinder */

	/* get current value geometry information */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, PD_GEOMETRY<<8,
	    sizeof(gd_msen), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msen_vdma,
	    sizeof(gd_msen), 1);
	if (gdissue("MSENSE GEO", dcb, unit))
		return (-1);
	pd = (struct pag_desc *)(((char *)&gd_msen.msen_bd)+gd_msen.msen_bdl);
	ncpd = SCSI_HML(pd->pd_pg.pg_geo.geo_cyl);
	ntpc =  pd->pd_pg.pg_geo.geo_heads;

	/* get current value format information */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, PD_FORMAT<<8,
	    sizeof(gd_msen), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msen_vdma,
	    sizeof(gd_msen), 1);
	if (gdissue("MSENSE FMT", dcb, unit))
		return (-1);
	pd = (struct pag_desc *)(((char *)&gd_msen.msen_bd)+gd_msen.msen_bdl);
	nspt =  pd->pd_pg.pg_fmt.fmt_spt;

	/* test geometry for consistency */
	if ((nspt * ntpc * ncpd) == 0) {
		printf("gd%d: nonsense geometry (%d x %d x %d)\n",
			unit, nspt, ntpc, ncpd);
		return (-1);
	}

	/* if capacity is less than calculated size, try various adjustments */
	if ((nspt * ntpc * ncpd) != spd) {
	    /* if cylinder size too small, try subtracting alternate sectors */
	    if ((nspt * ntpc) > spc && pd->pd_pg.pg_fmt.fmt_tpz != 0)
		nspt -= pd->pd_pg.pg_fmt.fmt_alt_spz / pd->pd_pg.pg_fmt.fmt_tpz;
	    /* if cylsize still too small, try subtracting alt tracks */
	    if ((nspt * ntpc) > spc && ntpc == pd->pd_pg.pg_fmt.fmt_tpz)
		ntpc -= pd->pd_pg.pg_fmt.fmt_alt_tpz;
	    /* if spc still wrong */
	    if ((nspt * ntpc) != spc) {
		if (spc % ntpc) {	/* if spc not multiple of headcount */
		    if (spc % nspt)	/* and spc not multiple of tracksize */
			spc = nspt * ntpc;
		    else		/* spc multiple of tracksize  */
			ntpc = spc / nspt;
		} else			/* spc multiple of headcount */
		    nspt = spc / ntpc;
	    }
	    /* spc now ok; if still unequal, drop cylinders at end */
	    if ((nspt * ntpc * ncpd) != spd)
		ncpd = spd / spc;
	}

	/* form dive name from first 3 characters of vendor and size */
	printf("        %mM %3s%3m (%d x %d x %d)", nspt * ntpc * ncpd * 512,
	    gd_inq.inq_vendor, nspt * ntpc * ncpd * 512, nspt, ntpc, ncpd);
	diskpart(ST_size, nspt, ntpc, ncpd);
	ST_ncpd = ncpd;
	ST_ntpc = ntpc;
	ST_nspt = nspt;
	ST_nspc = nspt * ntpc;
	if (qi->qi_dk >= 0)
		dk_bps[qi->qi_dk] = (3600/60) * 512 * nspt;
	return (0);
}

gdslctmod(qi)
	struct qb_device  *qi;
{
	int				unit = qi->qi_unit;
	int				ctlr = GDU_CTLR(unit);
	register struct gdstat		*stat = &gdstat[ctlr];
	register struct hacb_dcb	*dcb = &stat->gd_dirc;
	register int			lun = GDU_LUN(unit);
	register struct pag_desc	*pdn, *pdl;
	int				len;

	/* get the current error recovery parameter page in msen */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, PD_ERROR<<8, sizeof(gd_msen),
	    0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msen_vdma,
	    sizeof(gd_msen), 1);
	if (gdissue("MSENSE ERR", dcb, unit))
		return (-1);
	pdn = (struct pag_desc *)(((char *)&gd_msen.msen_bd)+gd_msen.msen_bdl);
	if (pdn->pd_pg.pg_err.err_per)
		return (0);
	len = gd_msen.msen_len + 1;
	gd_msen.msen_len = 0;
	gd_msen.msen_wprot = 0;
	gd_msel = *((struct scsi_msel *)&gd_msen);

	/* get changable status of error recovery parameters page in msen */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSENSE, lun, (0x40 | PD_ERROR)<<8,
	    sizeof(gd_msen), 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_IN, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msen_vdma,
	    sizeof(gd_msen), 1);
	if (gdissue("MSENSE CERR", dcb, unit))
		return (-1);

	pdl = (struct pag_desc *)(((char *)&gd_msel.msel_bd)+gd_msel.msel_bdl);
	if (pdn->pd_pg.pg_err.err_per)
		pdl->pd_pg.pg_err.err_per = 1;

	/* send page to drive */
	SET_CDB_0(&dcb->dcb_cdb, CMD_MSELECT, lun, pdl->pd_save<<16, len, 0, 0);
	SET_DCB(dcb, DCB_NOIE, DCB_DIR_OUT, stat->gd_swcd, DCB_NODISC,
	    sizeof(struct cdb_0), DCB_BW_16, DCB_AM_STD_S_D, 0, gd_msel_vdma,
	    len, 1);
	pdl->pd_save = 0;
	if (gdissue("MSELECT ERR", dcb, unit))
		return (-1);
	return (0);
}

gdintr(cont)
{
	return (gsintr(cont));
}
#endif
