/*
 * VMEBUS: IKON 10084/10089 DR11-W EMULATOR for Integrated Solutions system 
 * COPYRIGHT 1987 Integrated Solutions Inc.
 * All rights reserved
 *
 * $Header: dr.c,v 1.2 87/12/10 14:44:20 root Exp $
 * $Source: /usr/src/sys/is68kdev/RCS/dr.c,v $
 * $Log:	dr.c,v $
 * Revision 1.2  87/12/10  14:44:20  root
 * clean up of include file names
 * 
 * 
 */ 

/*
 * Compile with -DDEBUG for console diagnostic output.
 */

#include "dr.h"
#if	NDR > 0

#include "../machine/pte.h"
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"
#include "../h/ioctl.h"
#include "../is68kdev/qbvar.h"
#include "../is68kdev/drreg.h"


/*
 * The following is a  structure type that corresponds to
 * the 10084/10089 board's register set.
 *
 * Note that some registers appear at 
 * different offsets, depending on whether
 * they are being written or read.
 *
 * The dma address register is 24 bits
 * wide - the low 16 bits are accessed as 
 * a ushort value.
 * The high 8 bits may be accesses as a char at one address 
 * or a as a ushort at the next lower work address.
 */
struct	drdevice	{
	ushort	dr_cstat;     	/* control and status sregister */
	ushort	dr_data;     	/* input and output data register */
	char	dr_addmod;     	/* address modifier for dm */
	char	dr_intvect;     /* interrupt vector */
	ushort	dr_pulse;     	/* pulse command register */
	ushort	dr_xx08;     	/* not used */
	ushort	dr_xx0A;     	/* not used */
	ushort	dr_xx0C;     	/* not used */
	ushort	dr_xx0E;     	/* not used */
	ushort	dr_xx10;     	/* not used */
	ushort	dr_walo;     	/* low dma add register--when written */
	ushort	dr_range;     	/* dma range count */
	ushort	dr_ralo;     	/* low dma add register--when read */
	ushort	dr_xx18;     	/* not used */
	char	dr_xx1A;     	/* not used */
	char	dr_wahi;     	/* high dma add register--when written */
	ushort	dr_xx1C;     	/* not used */
	char	dr_xx1E;      	/* not used */
	char	dr_rahi;     	/* high dma add register--when read */
};

u_short *DRstd[] = {	/* DR controller addresses */
	(u_short *)(&vme_stdio[0xFFFE00]),
	(u_short *)(&vme_stdio[0xFFFF00]),
	0
};
		
/*
 * Local buffer headers.  The entries in this buffer header are filled
 * in by physio() which will call the local drstrategy() at which point,
 * the entries will have the same form as system buffers but will
 * actually point to the data in user space.
 */
struct buf	drbuf[NDR];	/* local buffer header */

/*
 * this structure is used to hold pointers, flags, commands, and status for
 * the dr11 driver.
 */
struct	dr_aux	{
	struct	drdevice *dr_pcurad;	/* physical address of active dr1 */
	struct	buf	 *dr_actf;     	/* these pointers to the drdevice */
	struct	buf	 *dr_actl;
	ushort	dr_flags;          	/* See drreg.h */
	ushort	dr_cmd;			/* placed by ioctl for drstrategy */
	ushort	dr_istat;		/* latest interrupt status */
	ushort	dr_time;     		/* # of tocks until time-out */
	ushort	dr_tock;     		/* # of tocks accumulated */
	ushort	dr_cseq;     		/* current sequence number */
	ushort	dr_lseq;     		/* same name for type and struct */
} dr_aux[ NDR ];

int	drprobe(), drslave(), drattach(), drintr(), drminphys();

struct	qb_ctlr		*drcinfo[ NDR ];
struct	qb_device	*drdinfo[ NDR ];

/*
 * There are no minor devices here.  Each major device is another controller.
 */
#define DRUNIT(dev) (minor(dev))
#define DRADDR(dev) ((struct drdevice *)(drdinfo[DRUNIT(dev)])->qi_mi->qm_addr) 
#define	min(a,b)    ((a)<(b)?(a):(b))
#define	BLOCKSIZE	131068	/* max xfer in bytes */
#define DRADMD 0x3A    	        /* address modifier for dm */
#define DR_ZERO 0x0
#define DR_TICK 600		/* # clock ticks for drwatch calls */
#define DR_TOCK 2		/* default # drwatch calls until timeout */
#define DR_PADD 0x55		/* used as a PADD in various places */
#define PAD_NUM	1		/* used as a PADD in various places */

/*
 * dr_cstat bits when used as control register write
 */
#define RDMA 0x8000     	/* resets the dma end-of-range flag */
#define RATN 0x4000     	/* resets the attention flag */
#define RPER 0x2000     	/* resets the device parity error flag */
#define MCLR 0x1000     	/* master clear board and INIt device   */
#define CYCL 0x0100     	/* forces dma cycle if dma enabled */
#define IENB 0x0040     	/* enables interrupts */
#define FCN3 0x0008     	/* function bit 3 to de (FNCT3 H) */
#define FCN2 0x0004     	/* enables dma and pulses GO to device */
                                /* also asserts ACLO FCN2 H to device   */
#define FCN1 0x0002     	/* function bit 1 to device (FCN1 H)  */
#define GO   0x0001     	/* enables dma and pulses GO to device */

/*
 * dr_cstat bits when used as status register read
 */
#define DMAF 0x8000    		/* indicates dma end-of-range */
#define ATTF 0x4000     	/* indicates attention false-to-true */
#define ATTN 0x2000     	/* current state of ATTENTION H input */
#define PERR 0x1000     	/* set by external parity error */
#define STTA 0x0800     	/* STATUS A H input state */
#define STTB 0x0400     	/* STATUS B H input state */
#define STTC 0x0200     	/* STATUS C H input state */
#define REDY 0x0080     	/* board ready for command (dma not on) */
#define IENF 0x0040     	/* interrupts enabled if on */
#define BERR 0x0020     	/* set if bus error during dma */
#define FC3S 0x0008     	/* state of FCN3 latch */
#define FC2S 0x0004     	/* state of FCN2 latch */
#define FC1S 0x0002     	/* state of FCN1 latch */
#define OKMSK	0xFF80

/*
 * Define pulse command register bits -- which include RDMA,
 * RPER, MCLR, CYCL, FCN2, GO and two additional interrupt mask
 * control bits.
 */

#define SMSK 0x0040     	/* pulses interrupt mask on */
#define RMSK 0X0020     	/* pulses interrupt mask off */

/*
 * DR/dr controller/device driver structure
 */
struct	qb_driver DRdriver = {
	drprobe, drslave, drattach, DRstd, "dr", drdinfo, "DR",drcinfo
};

		

drprobe( draddr )
	register struct drdevice *draddr;
{
	/*
	 * Check for presence of hardware at boot time.
	 */
	draddr->dr_intvect = freevec();
	draddr->dr_addmod = DRADMD; 
	draddr->dr_walo = 0x8000;
	draddr->dr_wahi = 0x00;
	draddr->dr_data = DR_PADD;

	if( draddr->dr_cstat & ATTN ) { 
		draddr->dr_range = PAD_NUM;
		draddr->dr_pulse = IENB | GO;
        	draddr->dr_pulse = CYCL;
	} else {
		draddr->dr_cstat = MCLR;
		draddr->dr_range = 0x00;
		draddr->dr_cstat = IENB | RATN;
		draddr->dr_pulse = CYCL | GO;
	}
        DELAY(100); 
#	ifdef DEBUG
		printf( "drprobe: dr_cstat at return = 0x%x\n",
		    draddr->dr_cstat );
#	endif
	return( (draddr->dr_cstat & REDY )==0 ? 0 : sizeof( struct drdevice ) );
}


drslave( qi, draddr )
	struct	qb_device	*qi;
	struct	drdevice	*draddr;
{
#	ifdef DEBUG
		printf( "drslave: dr_cstat = 0x%x\n", draddr->dr_cstat );
#	endif
	return( (draddr->dr_cstat & REDY)==0 ? 0 : 1 );
}


drattach( qi )
        register struct qb_device *qi;
{
	register struct drdevice  *draddr;
	register struct dr_aux	  *drstat;

	drstat = &dr_aux[ qi->qi_mi->qm_ctlr ];
     	draddr->dr_cstat = MCLR;
#	ifdef DEBUG
		printf( "drattach(%d): DR11 present\n", qi->qi_unit );
#	endif
	draddr = (struct drdevice *)qi->qi_mi->qm_addr;
     	drstat->dr_flags = DR_PRES; 	/* clears all-sets pres */
     	drstat->dr_pcurad = draddr;
	drstat->dr_istat = 0;
	drstat->dr_cmd = 0;
}


drwatch( drstat )
        register struct dr_aux	 *drstat;
{
	register struct drdevice *draddr;
	register int		 ctlr;
	register struct buf	 *bp;
	register struct qb_device *qi;

	/*
	 * drwatch() is the local watchdog.  It is called by the system
	 * clock routine approximately every 10 seconds.
	 */
	ctlr = drstat - dr_aux;
#	ifdef DEBUG
		printf("drwatch(%d): called\n", ctlr ); 
#	endif

	draddr = drstat->dr_pcurad;

	if( drstat->dr_cseq != drstat->dr_lseq ) {
		drstat->dr_tock = 0;
	} else {
		if( drstat->dr_flags & DR_ACTV ) {
			drstat->dr_tock++;
			if( drstat->dr_tock >= drstat->dr_time ) {
#				ifdef DEBUG
				  printf("drwatch(%d): end-of-range timeout\n",
				    ctlr );
#				endif
				/*
				 * Disable interrupts and clear DMA.
				 */
				draddr->dr_pulse = RMSK;
				draddr->dr_pulse = MCLR;
				/*
				 * Some applications will not issue a master
				 * clear after DMA timeout since doing so
				 * sends an INIT H pulse to the external
				 * device which may produce udesirable
				 * side-effects.
				 */
				drstat->dr_flags &= ~DR_ACTV;
				drstat->dr_flags |= DR_TMDM;     
				bp = drstat->dr_actf;
				bp->b_flags |= B_ERROR; /* flag error */
				iodone(bp);
			}
		}
		if( drstat->dr_flags & DR_ATWT ) {
			drstat->dr_tock++;     
			if( drstat->dr_tock >= drstat->dr_time ) {
#				ifdef DEBUG
				    printf("drwatch(%d): attention timeout\n",
				     ctlr );
#				endif
				draddr->dr_pulse = RMSK;
				drstat->dr_flags &= ~DR_ATWT;
				drstat->dr_flags |= DR_TMAT;    
				/*
				 * Wake up drioctl().
				 */
				wakeup( (caddr_t)drstat );
			}
		}
	}
	if( drstat->dr_flags & DR_OPEN ) {
#		ifdef DEBUG
			printf("drwatch(%d): calling timeout while OPEN\n",
			    ctlr );
#		endif
		timeout( drwatch, (caddr_t)drstat, DR_TICK );
	}
	drstat->dr_lseq = drstat->dr_cseq;
}


dropen(dev, flag)
	dev_t			  dev;
	int			  flag;
{
	register int		  unit;
	register struct drdevice  *draddr;
        register struct dr_aux	  *drstat;
	register struct qb_device *qi;
	register struct	qb_ctlr	  *qm;

	unit = DRUNIT( dev );
#	ifdef DEBUG
		printf("dropen(%d)\n", unit );
#	endif
	qi = drdinfo[ unit ];

	if( unit >= Ndr || !qi || !qi->qi_alive )
		return( ENXIO );  	/* dr11 not present */

	qm = qi->qi_mi;
	draddr = (struct drdevice *)qm->qm_addr;
	drstat = &dr_aux[ qm->qm_ctlr ];

	if( drstat->dr_flags & DR_OPEN )
		return( ETXTBSY );  	/* dr11 already open */

	drstat->dr_flags | = DR_OPEN;   /* set to open  to avoid multi open */
	draddr->dr_cstat = DR_ZERO;	/* clr func & intr enable latch */
	draddr->dr_pulse = RDMA | RATN;	/* and e-o-r flag */
	/*
	 * Attempt to establish a local watch dog time.
	 * This call will start it -- it will continue to
	 * call itself as long as the dr11 is open
	 */ 
	drstat->dr_time = DR_TOCK;
	drstat->dr_cseq = 1;
	drstat->dr_lseq = 0;
	timeout( drwatch, (caddr_t)drstat, DR_TICK );
	return( 0 );
}


drclose( dev, flag )
	dev_t			  dev;
	int			  flag;
{
	int		  	  unit, intlevel;
	register struct drdevice  *draddr;
        register struct dr_aux	  *drstat;
	register struct qb_device *qi;
	register struct qb_ctlr	  *qm;

	unit = DRUNIT( dev );
#	ifdef DEBUG
		printf("drclose(%d)\n", unit );
#	endif
	qi = drdinfo[ unit ];
	qm = qi->qi_mi;
	draddr = qm->qm_addr;
	drstat = &dr_aux[ qm->qm_ctlr ];

	if( !(drstat->dr_flags & DR_OPEN) )	/* device not open */
		return;

	drstat->dr_flags &= ~DR_OPEN;     
	intlevel = splx( qm->qm_psl );     /* protect int mask clear */
	draddr->dr_cstat = DR_ZERO;
	splx( intlevel );
	return;
}


drstrategy( bp )
	register struct	buf	  *bp;
{
	register struct qb_device *qi;
        register struct	dr_aux	  *drstat;
	register struct	drdevice  *draddr;
	register int		  unit;
	register unsigned int	  bufadr;
		 ushort 	  status;
		 int		  intlevel;

	/*
	 * This is a single user raw i/o device so it doesn't
	 * make a log of sense to implement a buffer queue.
	 */
	unit = DRUNIT( bp->b_dev );
#	ifdef DEBUG
		printf("drstrategy(%d)\n", unit);
#	endif
	drstat = &dr_aux[ unit ];
	draddr = drstat->dr_pcurad;
	qi = drdinfo[ unit ];
	/*
	 * Validate unit and make sure inactive.
	 */
	if( unit >= Ndr || !qi || !qi->qi_alive ||
	    (drstat->dr_flags & DR_ACTV) )
		goto bad;

	bufadr = qbaddr( bp );
	drstat->dr_actf = bp;
	draddr->dr_walo = (ushort)(bufadr >> 1); /* right shift one bit */
	draddr->dr_wahi = (char)(bufadr >> 17); /* get high byte & shift */
#	ifdef DEBUG 
		printf("drstrategy(%d): xfer cnt = %d\n", unit, bp->b_bcount);
#	endif

	if( bp->b_bcount > BLOCKSIZE ) 
		goto bad; 	/* will be greater than 64K word */
	draddr->dr_range = (ushort)((bp->b_bcount >> 1) - 1);

#	ifdef DEBUG
		printf( "drstrategy(%d): dma %s\n", unit,
		    (bp->b_flags & B_READ) ? "read" : "write" );
		printf( "drstrategy(%d): low adr          = 0x%x\n", unit,
		    draddr->dr_ralo );
		printf( "drstrategy(%d): high adr         = 0x%x\n", unit,
		    (ushort)draddr->dr_rahi );
		printf( "drstrategy(%d): range            = 0x%x\n", unit,
		    draddr->dr_range );
		printf( "drstrategy(%d): address modifier = 0x%x\n", unit,
		    draddr->dr_addmod );
		printf( "drstrategy(%d): interrupt vector = 0x%x\n", unit,
		    draddr->dr_intvect );
#	endif
	/*
	 * Clear dmaf and attf to assure a clean dma start.
	 */ 
	draddr->dr_pulse = RDMA | RATN;

	if( drstat->dr_cmd & DR_DFCN ) {
#		ifdef DEBUG
			printf("drstrategy(%d): deferred function bit write\n",
			    unit );
#		endif
		drstat->dr_cmd &= ~DR_DFCN;
		draddr->dr_cstat = drstat->dr_cmd & DR_FMSK;
	}
	/*
	 * Increment sequence counter for watchdog timer.
	 */
	drstat->dr_cseq++;
	drstat->dr_flags |= DR_ACTV;
	draddr->dr_pulse = IENB | GO;

	/*
	 * Check for software cycle request -- usually by
	 * transmitter in link mode.
	 */
	if( drstat->dr_cmd & DR_PCYL ) {     
#		ifdef DEBUG
			printf("drstrategy(%d): software cycle request\n",
			    unit );
#		endif
		drstat->dr_cmd &= ~DR_PCYL;     /* clear request */
		draddr->dr_pulse = CYCL;     /* use pulse reg again */
	}
	/*
	 * Now check for deferred aclo fnct2 pulse request -- usually to tell
	 * the transmitter (via its attention) that we have enabled dma
	 */
	if( drstat->dr_cmd & DR_DACL ) {     
#		ifdef DEBUG
			printf("drstrategy(%d): sending deferred aclo pulse\n",
			    unit);
#		endif DR_TRCM
		drstat->dr_cmd &= ~DR_DACL;     /* clear request */
		draddr->dr_pulse = FCN2;     /* pulse aclo fcn2 sig */
	}
	/*
	 * Return to caller (physio).  Physio will sleep until awakened
	 * by a call to iodone() in the interrupt handler.
	 */
	return;

bad:  
	bp->b_flags |= B_ERROR;
	iodone(bp);
#	ifdef DEBUG
		printf("drstrategy(%d): error exit\n", unit );
#	endif
}


drwrite( dev, uio )
	dev_t		dev;
	struct uio	*uio;
{
	register int	unit;

	unit = DRUNIT( dev );
#	ifdef DEBUG
		printf("drwrite(%d)\n", unit );
#	endif
	if( unit >= Ndr )
		return( ENXIO );

	return( physio( drstrategy, &drbuf[ unit ], dev, B_WRITE,
	    drminphys, uio) );
}


drread(dev,uio)
	dev_t		dev;
	struct uio	*uio;
{
	register int	unit;

	unit = DRUNIT( dev );
#	ifdef DEBUG
		printf("drread(%d)\n", unit );
#	endif
	if( unit >= Ndr )
		return( ENXIO );
	return( physio( drstrategy, &drbuf[ unit ], dev, B_READ,
	    drminphys, uio) );
}


drminphys( bp )
	struct buf	*bp;
{
	bp->b_bcount = min( bp->b_bcount, BLOCKSIZE );
}


drintr( ctlr )
	register int		  ctlr; 
{
	register struct qb_ctlr  *qm;
	register struct qb_device *qi;
	register struct drdevice *draddr; 
	register struct buf 	 *bp;
        register struct dr_aux	 *drstat;
	register struct buf	 *dp;
	register ushort		  status;
	dev_t			  dev;
        int			  unit;

	qm = drcinfo[ ctlr ];
	drstat = &dr_aux[ ctlr ];
	bp = drstat->dr_actf;
	qi = drdinfo[ unit = DRUNIT( bp->b_dev ) ];
	draddr = (struct drdevice *)qm->qm_addr;
	status = draddr->dr_cstat;
#	ifdef DEBUG
		printf("drintr(%d): dr_cstat= %x\n", ctlr, status );
#	endif
	drstat->dr_istat = status;	/* save interrupt status */
	/*
	 * Make sure that the dr11 is really interrupting -- at least one of
	 * attf or dmaf must be set.
	 * If neither is set this is a spurious interrupt -- a very serious
	 * hardware or software error
	 */ 

	if( !(status & (ATTF | DMAF)) ) {
		bp->b_flags |= B_ERROR;
  printf("drintr(%d):  spurious interrupt (dmaf or attf not set)\n", ctlr );
	}
	if( status & DMAF ) {
		drstat->dr_flags |= DR_DMAX;
#		ifdef DEBUG
			printf("drintr(%d): dma end of range interrupt\n",
			    ctlr );
#		endif

		if( !(drstat->dr_flags & DR_ACTV) ) {
#			ifdef DEBUG
  printf("drintr(%d): EOR interrupt when dma not active\n", ctlr );
#			endif
			bp->b_flags |= B_ERROR;    /* report error */
		} else {
			drstat->dr_flags &= ~DR_ACTV;
			iodone (bp);
		}
		draddr->dr_pulse = RDMA;
	}
	/* Now test for attention interrupt -- it may be set in
	 * addition to the dma end of range interrupt.  If we get
	 * one we will issue a wakeup to the drioctl() routine
	 * which is presumably waiting for one. 
	 */
	if( status & ATTF ) {
		drstat->dr_flags |= DR_ATRX;
#		ifdef DEBUG
			printf("drintr(%d): ATTN interrupt\n", ctlr );
#		endif
		drstat->dr_flags &= ~DR_ATWT;
		draddr->dr_pulse = RATN;
		wakeup( (caddr_t)drstat );
	}
}


drioctl(dev, cmd, drop, flag )
	dev_t			dev;
	int			cmd;
	struct	drop		*drop;
	int			flag;
{

	register int		  unit;
	register struct drdevice  *draddr;
        register struct dr_aux    *drstat;
	register struct	qb_device *qi;
	register struct qb_ctlr   *qm;
	int			  intlevel;

	unit = DRUNIT( dev );
	drstat = &dr_aux[ unit ];
	draddr = drstat->dr_pcurad;
	drstat->dr_cmd = drop->dr_cmd;
	qi = drdinfo[ unit ];
	qm = qi->qi_mi;

#	ifdef DEBUG
		printf("drioctl(%d)\n", unit );
#	endif

	if( cmd !=  DR_IOC ) {
#		ifdef DEBUG
			printf("drioctl(%d): invalid IORW designagtor\n",
			    unit );
#		endif
		return( EINVAL );	/* invalid argument */
	}

#	ifdef DEBUG
		printf("drioctl(%d): dr_cmd = 0x%x\n", unit, drop->dr_cmd );
#	endif

	if( drstat->dr_cmd & DR_STIM ) {    
#		ifdef DEBUG
			printf("drioctl(%d): setting time-out = 0x%x\n",
			    unit, drop->dr_timeout );
#		endif
		drstat->dr_time = drop->dr_timeout;
	}

	if( drstat->dr_cmd & DR_PIOW ) {   
#		ifdef DEBUG
			printf("drioctl(%d): p-i/o write = 0x%x\n",
			    unit, drop->dr_outdata );
#		endif
		draddr->dr_data = drop->dr_outdata;	/* write to p-i/o reg */
	}

	if( drstat->dr_cmd & DR_RATN ) { 
#		ifdef DEBUG
			printf("drioctl(%d): clearing attn flag\n", unit );
#		endif
		draddr->dr_pulse = RATN;  /* use pulse register */
	}

	if( drstat->dr_cmd & DR_RDMA ) { 
#		ifdef DEBUG
			printf("drioctl(%d): clearing dmaf flag\n", unit );
#		endif
		draddr->dr_pulse = RDMA;  /* use pulse register */
	}

	if( drstat->dr_cmd & DR_RPER ) {  
#		ifdef DEBUG
			printf("drioctl(%d): clearing parity error flag\n",
			    unit );
#		endif
		draddr->dr_pulse = RPER;  /* use pulse register */
	}

	if(drstat->dr_cmd & DR_RSET) {  
#		ifdef DEBUG
			printf("drioctl(%d): issuing mclr\n", unit );
#		endif
		draddr->dr_pulse = MCLR;
	}

	if( drstat->dr_cmd & DR_SFCN ) {   
#		ifdef DEBUG
			printf("drioctl(%d): fcn bit set = 0x%x\n", unit,
			    drstat->dr_cmd & DR_FMSK );
#		endif
		draddr->dr_cstat = drstat->dr_cmd & DR_FMSK;
	}

	if( drstat->dr_cmd & DR_STGO ) {   
#		ifdef DEBUG
			printf("drioctl(%d): GO bit set\n", unit );
#		endif
		draddr->dr_cstat = GO;
	}

	if(drstat->dr_cmd & DR_PACL) {  
#		ifdef DEBUG
			printf("drioctl(%d): aclo fnct2 pulse\n", unit );
#		endif
		draddr->dr_pulse = FCN2;  /* use pulse register */
	}

	if( drstat->dr_cmd & DR_WAIT ) {  
#		ifdef DEBUG
			printf("drioctl(%d): waiting for ATTN interrupt\n",
			    unit );
#		endif
		/* The setting of the dr11 interrupt enable latch and
		 * the call to sleep will be protected from dr11
		 * interrupts by raising the processor priority above
		 * that of the dr11.
		 */
		intlevel = splx( qm->qm_psl );
		if( !(drstat->dr_flags & DR_ATRX) ) {  
			drstat->dr_cseq++;
			drstat->dr_flags |= DR_ATWT;	/* set waiting flag  */
			draddr->dr_pulse = IENB;	/* enable interrupts */
			sleep( (caddr_t)drstat, PRIBIO );
		}
		splx( intlevel );
	}

	drop->dr_indata = draddr->dr_data;
	drop->dr_status = draddr->dr_cstat;
	drop->dr_flags = drstat->dr_flags;
	return( 0 );
}
#endif	NDR > 0
