/*
 * QBUS: DH-11/DM-11 driver
 */

/*
 * NOTE: There is a bug with the EMULEX CS01 such that writing bits 0-7 of
 * the SCR register will effect bits 8-15 of that register.  This problem
 * has been partialy masked by changing all byte accesses to the SCR to 
 * word accesses and making an attempt to preserve the TI bit. There is
 * still a race condition, however, between the time the CSR is read to 
 * preserve its sense and when it is writen. The result of this race is that 
 * an interupt may be lost, causing a line to hang.  There is an additional
 * problem with the CS01 in that clearing TI in the transmit interupt routine
 * does not always work the first time.			Chris 5/15/84
 */
#define CS01BUGS /**/

#include "dh.h"
#if	NDH > 0
#include "../machine/pte.h"

#include "param.h"
#include "conf.h"
#include "user.h"
#include "proc.h"
#include "ioctl.h"
#include "tty.h"
#include "map.h"
#include "buf.h"
#include "vm.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/dhreg.h"

#include "bk.h"
#include "clist.h"
#include "file.h"
#include "uio.h"
#include "kernel.h"
#include "syslog.h"

#ifdef	SYSV
#include "../sysv/sys/termio.h"
#endif	SYSV

#define	TTY_NODELAY

/*
 * Definition of the driver for the auto-configuration program. There is one 
 * definition for the dh and one for the dm.
 */
int	dhprobe(), dhslave(), dhattach(), dhrint(), dhxint(), dhtimer();
struct	qb_device *dhinfo[NDH];
u_short	*dhstd[] = {	(u_short *)0x3fe010, (u_short *)0x3fe020,
			(u_short *)0x3fe030, (u_short *)0x3fe040,
			(u_short *)0x3fe050, (u_short *)0x3fe060,
			(u_short *)0x3fe070, (u_short *)0x3fe080, 0 };
struct	qb_driver DHdriver =
	{ dhprobe, dhslave, dhattach, dhstd, "dh", dhinfo, "DH" };

int	dmprobe(), dmslave(), dmattach(), dmintr();
struct	qb_device *dminfo[NDH];
u_short	*dmstd[] = {	(u_short *)0x3ff140, (u_short *)0x3ff148,
			(u_short *)0x3ff150, (u_short *)0x3ff158,
			(u_short *)0x3ff160, (u_short *)0x3ff168,
			(u_short *)0x3ff170, (u_short *)0x3ff178, 0 };
struct	qb_driver DMdriver =
	{ dmprobe, dmslave, dmattach, dmstd, "dm", dminfo, "DM" };

#ifndef	PORTSELECTOR
#define	ISPEED	B9600
#define	IFLAGS	(EVENP|ODDP|ECHO)
#else
#define	ISPEED	B4800
#define	IFLAGS	(EVENP|ODDP)
#endif

#define	FASTTIMER	(hz/30)		/* scan rate with silos on */

/*
 * Local variables for the driver
 */
short	dhsbar[NDH];			/* software copy of last bar */

struct	tty dh_tty[NDH*16];
short	dh_nch[NDH*16];			/* number of characters xmiting */
extern  char dh_buf[NDH*16][CBSIZE];
int	ndh_tty	= 0;
int	dhact;				/* mask of active dh's */
int	dhsilos;			/* mask of dh's with silo in use */
int	dhchars[NDH];			/* recent input count */
int	dhrate[NDH];			/* smoothed input count */
int	dhhighrate = 100;		/* silo on if dhchars > dhhighrate */
int	dhlowrate = 75;			/* silo off if dhrate < dhlowrate */
static	short timerstarted;
int	dhstart(), ttrstrt();

/*
 * Routine for configuration to force a dh to interrupt. Set to transmit at 9600
 * baud, and cause a transmitter interrupt.
 */
dhprobe(dhaddr)
	register struct dhdevice *dhaddr;
{
	extern int cvec;

	dhaddr->un.dhcsr = 4;		/* dont confuse a dz with a dh */
	DELAY(100);
	if ((dhaddr->un.dhcsr & 4) == 0)
		return (0);
	dhaddr->un.dhcsr = DH_TI | DH_TIE;
	DELAY(2000);

#ifdef	CS01BUGS
# define MAXLOOP 10
	{
		register int count = MAXLOOP;

		while ((dhaddr->un.dhcsr & DH_TI) == 0 && count--) {
			dhaddr->un.dhcsr = DH_TI | DH_TIE;
			DELAY(10000);
		}
	}
#endif	CS01BUGS

	dhaddr->un.dhcsr = 0;

#ifdef	CS01BUGS
	{
		register int count = MAXLOOP;

		while ((dhaddr->un.dhcsr & DH_TI) && count--) {
			dhaddr->un.dhcsr = 0;
			DELAY(10000);
		}
	}
#endif	CS01BUGS
	ndh_tty += 16;
	if (cvec && cvec != 0x100)
		cvec -= 4;	/* transmit -> receive */
	clevmax = clev_ttymax;
	clev_tty = MAX(clev, clev_tty);
	return (sizeof (struct dhdevice));
}

/*
 * Routine called to slave a dh.
 */
dhslave(qi)
	struct qb_device *qi;
{
	return (1);
}

/*
 * Routine called to attach a dh.
 */
dhattach(qi)
	struct qb_device *qi;
{
}

/*
 * Configuration routine to cause a dm to interrupt.
 */
dmprobe(dmaddr)
	register struct dmdevice *dmaddr;
{
	dmaddr->dmcsr = DM_IE | DM_DONE;
	DELAY(100);
	dmaddr->dmcsr = 0;
	return (1);
}

dmslave(qi)
	struct qb_device *qi;
{
	return (1);
}

dmattach(qi)
	struct qb_device *qi;
{
}

#ifdef	SYSV
int sv_dhproc();
#endif	SYSV

/*
 * Open a DH11 line. Turn on this dh if this is the first use of it.  Also do 
 * a dmopen to wait for carrier.
 */
dhopen(dev, flag)
	dev_t dev;
{
	register struct tty *tp;
	register int unit, dh;
	register struct dhdevice *dhaddr;
	register struct qb_device *qi;
	int s;

	unit = minor(dev);
	dh = unit >> 4;
	if (unit >= NDH*16 || (qi = dhinfo[dh])== 0 || qi->qi_alive == 0)
		return (ENXIO);
	tp = &dh_tty[unit];
	dhaddr = (struct dhdevice *)qi->qi_mi->qm_addr;
	tp->t_addr = (caddr_t)dhaddr;
	tp->t_oproc = dhstart;
#ifdef	SYSV
	tp->svt_proc = sv_dhproc;
#endif	SYSV
	/*
	 * While setting up state block resets which can clear the state.
	 */
	s = splx(qi->qi_mi->qm_psl);
	if (timerstarted == 0) {
		timerstarted++;
		timeout(dhtimer, (caddr_t) 0, hz);
	}
	if ((dhact&(1<<dh)) == 0) {
		dhaddr->un.dhcsr |= DH_IE;
		dhact |= (1<<dh);
		dhaddr->dhsilo = 0;
	}
	splx(s);
	/*
	 * If this is first open, initialze tty state to default.
	 */
	if ((tp->t_state&TS_ISOPEN) == 0) {
		ttychars(tp);
#ifndef PORTSELECTOR
		if (tp->t_ispeed == 0) {
#endif	PORTSELECTOR
			tp->t_ispeed = ISPEED;
			tp->t_ospeed = ISPEED;
			tp->t_flags = IFLAGS;
#ifndef PORTSELECTOR
		}
#endif	PORTSELECTOR
#ifdef	SYSV
		sv_ttinit(tp);
#endif	SYSV
		dhparam(unit);
	} else if ((tp->t_state&TS_XCLUDE) && 
	    ((u.u_uid!=0) || (u.u_procp->p_flag & SLOGIN)))
		return (EBUSY);

	/*
	 * Wait for carrier, then process line discipline specific open.
	 */
	if (dmopen(dev, flag))
		return (EBUSY);
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_open)(dev, tp, flag));
	else
#endif	SYSV
	return ((*linesw[tp->t_line].l_open)(dev, tp));
}

/*
 * Close a DH11 line, turning off the DM11.
 */
dhclose(dev, flag)
	dev_t dev;
	int flag;
{
	register struct tty *tp;
	register unit;

	unit = minor(dev);
	tp = &dh_tty[unit];
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		(*sv_linesw[tp->svt_line].l_close)(tp);
	else
#endif	SYSV
	(*linesw[tp->t_line].l_close)(tp);
	((struct dhdevice *)(tp->t_addr))->dhbreak &= ~(1<<(unit&017));
	if (tp->t_state&TS_HUPCLS || (tp->t_state&TS_ISOPEN)==0)
		dmctl(unit, DML_OFF, DMSET);
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		sv_ttyclose(tp);
	else
#endif	SYSV
	ttyclose(tp);
}

dhread(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register struct tty *tp = &dh_tty[minor(dev)];

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_read)(tp, uio));
	else
#endif	SYSV
	return ((*linesw[tp->t_line].l_read)(tp, uio));
}

dhwrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register struct tty *tp = &dh_tty[minor(dev)];

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_write)(tp, uio));
	else
#endif	SYSV
	return ((*linesw[tp->t_line].l_write)(tp, uio));
}

/*
 * DH11 receiver interrupt.
 */
dhrint(dh)
	int dh;
{
	register struct tty *tp;
	register c;
	register struct dhdevice *dhaddr;
	register struct tty *tp0;
	register struct qb_device *qi;
	int overrun = 0;

	qi = dhinfo[dh];
	if (qi == 0 || qi->qi_alive == 0)
		return;
	dhaddr = (struct dhdevice *)qi->qi_mi->qm_addr;
	tp0 = &dh_tty[dh<<4];
	/*
	 * Loop fetching characters from the silo for this dh.
	 */
	while ((c = dhaddr->dhrcr) < 0) {
		tp = tp0 + ((c>>8)&0xf);
		dhchars[dh]++;
		if ((tp->t_state&TS_ISOPEN)==0) {
			wakeup((caddr_t)tp);
#ifdef PORTSELECTOR
			if ((tp->t_state&TS_WOPEN) == 0)
#endif
				continue;
		}
#ifdef	SYSV
		if (tp->t_universe == UNIVERSE_SYSV) {
			if (c & DH_PE) {
				if (tp->svt_iflag & SV_IGNPAR)
					continue;
			    	if (tp->svt_iflag & SV_PARMRK)
					;		/* ??? */
			}
			if ((c & DH_DO) && overrun == 0) {
				log(LOG_WARNING, "dh%d: silo overflow\n", dh);
				overrun = 1;
			}
			if (c & DH_FE) {
			    	if (tp->svt_iflag & SV_IGNBRK)
					continue;
			    	if (tp->svt_iflag & SV_BRKINT)
					(*sv_linesw[tp->svt_line].l_input)
						(c, tp, L_BREAK);
			    	c = 0;
			}
			if (tp->svt_iflag & SV_ISTRIP)
				c &= 0177;
			else
				c &= 0377;
		} else {
#endif	SYSV
		if (c & DH_PE)
			if ((tp->t_flags&(EVENP|ODDP))==EVENP
			 || (tp->t_flags&(EVENP|ODDP))==ODDP )
				continue;
		if ((c & DH_DO) && overrun == 0) {
			log(LOG_WARNING, "dh%d: silo overflow\n", dh);
			overrun = 1;
		}
		if (c & DH_FE)
			/*
			 * At framing error (break) generate null (in raw mode,
			 * for getty), or interrupt (cooked/cbreak mode)
			 */
			if (tp->t_flags&RAW)
				c = 0;
			else
				c = tp->t_intrc;
#ifdef	SYSV
		}
		if (tp->t_universe == UNIVERSE_SYSV)
			(*sv_linesw[tp->svt_line].l_input)(c, tp, 0);
		else
#endif	SYSV
#if NBK > 0 || Nbk > 0
		if (tp->t_line == NETLDISC) {
			c &= 0177;
			BKINPUT(c, tp);
		} else
#endif
			(*linesw[tp->t_line].l_rint)(c, tp);
	}
}

/*
 * Ioctl for DH11.
 */
dhioctl(dev, cmd, data, flag)
	caddr_t data;
{
	register struct tty *tp;
	register int unit = minor(dev);
	int error;

	tp = &dh_tty[unit];
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV) {
		if ((error = sv_ttiocom(tp, cmd, data)) > 0)
			return error;
		if (cmd == TCSETAF || cmd == TCSETA || cmd == TCSETAW)
			dhparam(unit);
		if (cmd == TCSBRK)
			;		/* ??? */
		return 0;
	}
#endif	SYSV
	error = (*linesw[tp->t_line].l_ioctl)(tp, cmd, data, flag);
	if (error >= 0)
		return (error);
	error = ttioctl(tp, cmd, data, flag);
	if (error >= 0) {
		if (cmd == TIOCSETP || cmd == TIOCSETN ||
		  cmd == TIOCLSET || cmd == TIOCLBIS || cmd == TIOCLBIC)
			dhparam(unit);
		return (error);
	}
	switch (cmd) {
	    case TIOCSBRK:
		((struct dhdevice *)(tp->t_addr))->dhbreak |= 1<<(unit&017);
		break;

	    case TIOCCBRK:
		((struct dhdevice *)(tp->t_addr))->dhbreak &= ~(1<<(unit&017));
		break;

	    case TIOCSDTR:
		dmctl(unit, DML_DTR|DML_RTS, DMBIS);
		break;

	    case TIOCCDTR:
		dmctl(unit, DML_DTR|DML_RTS, DMBIC);
		break;

	    default:
		return (ENOTTY);
	}
	return (0);
}

/*
 * Set parameters from open or stty into the DH hardware registers.
 */
dhparam(unit)
	register int unit;
{
	register struct tty *tp;
	register struct dhdevice *dhaddr;
	register int lpar;
	int s;

	tp = &dh_tty[unit];
	dhaddr = (struct dhdevice *)tp->t_addr;
	/*
	 * Block interrupts so parameters will be set before line interrupts.
	 */
	s = splx(dhinfo[unit>>4]->qi_mi->qm_psl);
#ifdef	CS01BUGS
	dhaddr->un.dhcsr = (dhaddr->un.dhcsr&DH_TI) | (unit&0xf) | DH_IE;
#else	CS01BUGS
	dhaddr->un.dhcsrhl[DH_CSRL] = (unit&0xf) | DH_RIE;
#endif	CS01BUGS
	if ((tp->t_ispeed)==0) {
		tp->t_state |= TS_HUPCLS;
#ifdef	SYSV
		map_flags(tp, TOSV);
#endif	SYSV
		dmctl(unit, DML_OFF, DMSET);
		splx(s);
		return;
	}
	lpar = ((tp->t_ospeed)<<10) | ((tp->t_ispeed)<<6);
#ifdef	SYSV
	if (tp->t_universe == UNIVERSE_SYSV) {
		if ((tp->svt_cflag&SV_CSIZE) == SV_CS8)
			lpar |= BITS8;
		else if ((tp->svt_cflag&SV_CSIZE) == SV_CS7)
			lpar |= BITS7;
		else
			lpar |= BITS6;
		if (tp->svt_cflag&SV_CSTOPB)
			lpar |= TWOSB;
		if (tp->svt_cflag&SV_PARENB) {
			lpar |= PENABLE;
			if (tp->svt_cflag&SV_PARODD)
				lpar |= OPAR;
		}
	} else {
#endif	SYSV
	if ((tp->t_ispeed) == B134)
		lpar |= BITS6|PENABLE|HDUPLX;
	else if (tp->t_flags & (RAW|LITOUT|PASS8))
		lpar |= BITS8;
	else
		lpar |= BITS7|PENABLE;
	if ((tp->t_flags&EVENP) == 0)
		lpar |= OPAR;
	if ((tp->t_ospeed) == B110 || tp->t_state & TS_TWOSB)
		lpar |= TWOSB;
#ifdef	SYSV
	}
#endif	SYSV
	dhaddr->dhlpr = lpar;
	splx(s);
}

#ifdef	SYSV
sv_dhproc(tp, cmd)
register struct tty *tp;
register cmd;
{
	sv_proc(tp, cmd, dhstart, dhparam);
}
#endif	SYSV

/*
 * DH11 transmitter interrupt. Restart each line which used to be active but has
 * terminated transmission since the last interrupt.
 */
dhxint(dh)
	int dh;
{
	register struct tty *tp;
	register struct dhdevice *dhaddr;
	short ttybit, bar, *sbar;
	register struct qb_device *qi;
	register int unit;
	u_short cntr;

	qi = dhinfo[dh];
	dhaddr = (struct dhdevice *)qi->qi_mi->qm_addr;
	if (dhaddr->un.dhcsr & DH_NXM) {
		dhaddr->un.dhcsr |= DH_CNI;
		printf("dh%d: NXM\n", dh);
	}
	sbar = &dhsbar[dh];
	bar = *sbar & ~dhaddr->dhbar;
	unit = dh * 16; 
	ttybit = 1;
#ifdef	CS01BUGS
	while (dhaddr->un.dhcsr & DH_TI)
#endif	CS01BUGS
	dhaddr->un.dhcsr &= ~DH_TI;
	for (; bar; unit++, ttybit <<= 1) {
		if (bar & ttybit) {
			*sbar &= ~ttybit;
			bar &= ~ttybit;
			tp = &dh_tty[unit];
			tp->t_state &= ~TS_BUSY;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			if (tp->t_state&TS_FLUSH)
				tp->t_state &= ~TS_FLUSH;
			else {
#ifdef	CS01BUGS
				dhaddr->un.dhcsr = (dhaddr->un.dhcsr&DH_TI)|
					(unit&0xf)|DH_IE;
#else	CS01BUGS
				dhaddr->un.dhcsrhl[DH_CSRL] = (unit&0xf)|DH_RIE;
#endif	CS01BUGS
				cntr = dhaddr->dhcar - (u_short)dh_buf[unit];
				ndflush(&tp->t_outq, (int)cntr);
			}
#ifdef	SYSV
			map_state(tp, TOSV);
			if (tp->t_universe == UNIVERSE_SYSV) {
				if ((*sv_linesw[tp->svt_line].l_output)(tp))
					dhstart(tp);
			} else {
#endif	SYSV
			if (tp->t_line)
				(*linesw[tp->t_line].l_start)(tp);
			else
				dhstart(tp);
#ifdef	SYSV
			}
#endif	SYSV
		}
	}
}

/*
 * Start (restart) transmission on the given DH11 line.
 */
dhstart(tp)
	register struct tty *tp;
{
	register struct dhdevice *dhaddr;
	register int car, dh, unit, nch;
	short word;
	int s;

	unit = minor(tp->t_dev);
	dh = unit >> 4;
	dhaddr = (struct dhdevice *)tp->t_addr;

	/*
	 * Must hold interrupts in following code to prevent state of the tp 
	 * from changing.
	 */
	s = splx(dhinfo[dh]->qi_mi->qm_psl);
	/*
	 * If it's currently active, or delaying, no need to do anything.
	 */
	if (tp->t_state&(TS_TIMEOUT|TS_BUSY|TS_TTSTOP))
		goto out;
	/*
	 * If there are sleepers, and output has drained below low water mark, 
	 * wake up the sleepers.
	 */
	if (tp->t_outq.c_cc<=TTLOWAT(tp)) {
		if (tp->t_state&TS_ASLEEP) {
			tp->t_state &= ~TS_ASLEEP;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			wakeup((caddr_t)&tp->t_outq);
		}
		if (tp->t_wsel) {
			selwakeup(tp->t_wsel, tp->t_state & TS_WCOLL);
			tp->t_wsel = 0;
			tp->t_state &= ~TS_WCOLL;
		}
	}
	/*
	 * Now restart transmission unless the output queue is empty.
	 */
	if (tp->t_outq.c_cc == 0)
		goto out;
	if (tp->t_flags & (RAW|LITOUT))
		nch = ndqb(&tp->t_outq, 0);
	else {
		nch = ndqb(&tp->t_outq, 0200);
		/*
		 * If first thing on queue is a delay process it.
		 */
		if (nch == 0) {
			nch = getc(&tp->t_outq);
			timeout(ttrstrt, (caddr_t)tp, (nch&0x7f)+6);
			tp->t_state |= TS_TIMEOUT;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			goto out;
		}
	}
	/*
	 * If characters to transmit, restart transmission.
	 */
	if (nch) {
		tp->t_state |= TS_BUSY;
#ifdef	SYSV
		map_state(tp, TOSV);
#endif	SYSV
		car = (int)dh_buf[unit];
		dh_nch[unit] = nch;
		bcopy(tp->t_outq.c_cf, car, nch);
		byterev(car, nch);
#ifdef	CS01BUGS
		dhaddr->un.dhcsr = (dhaddr->un.dhcsr&DH_TI)|
			(unit&0xf)|((car>>12)&0x30)|DH_IE;
#else	CS01BUGS
		dhaddr->un.dhcsrhl[DH_CSRL]=(unit&0xf)|((car>>12)&0x30)|DH_RIE;
#endif	CS01BUGS
		word = 1 << (unit&0xf);
		dhsbar[dh] |= word;
		dhaddr->dhcar = car;
		dhaddr->dhbcr = -nch;
		dhaddr->dhbar |= word;
	}
out:
	splx(s);
}

/*
 * Stop output on a line, e.g. for ^S/^Q or output flush.
 */
dhstop(tp, flag)
	register struct tty *tp;
{
	register struct dhdevice *dhaddr;
	register int unit, s;

	dhaddr = (struct dhdevice *)tp->t_addr;
	/*
	 * Block input/output interrupts while messing with state.
	 */
	s = splx(dhinfo[minor(tp->t_dev)>>4]->qi_mi->qm_psl);
	if (tp->t_state & TS_BUSY) {
		/*
		 * Device is transmitting; stop output by selecting the line and
		 * setting the byte count to -1.  We will clean up later by 
		 * examining the address where the dh stopped.
		 */
		unit = minor(tp->t_dev);
#ifdef	CS01BUGS
		dhaddr->un.dhcsr = (dhaddr->un.dhcsr&DH_TI)|(unit&0xf)|DH_IE;
#else	CS01BUGS
		dhaddr->un.dhcsrhl[DH_CSRL] = (unit&0xf)|DH_RIE;
#endif	CS01BUGS
		if ((tp->t_state&TS_TTSTOP)==0) {
			tp->t_state |= TS_FLUSH;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
		}
		dhaddr->dhbcr = -1;
	}
	splx(s);
}

/*
 * Reset state of driver if UBA reset was necessary.
 * Reset the csrhl[DH_CSRL] and lpr registers on open lines, and
 * restart transmitters.
 */
dhreset()
{
	register int dh, unit;
	register struct tty *tp;
	register struct qb_device *qi;
	int i;

	dh = 0;
	for (dh = 0; dh < NDH; dh++) {
		qi = dhinfo[dh];
		if (qi == 0 || qi->qi_alive == 0)
			continue;
		printf(" dh%d", dh);
		((struct dhdevice *)qi->qi_mi->qm_addr)->un.dhcsr |= DH_IE;
		((struct dhdevice *)qi->qi_mi->qm_addr)->dhsilo = 0;
		unit = dh * 16;
		for (i = 0; i < 16; i++) {
			tp = &dh_tty[unit];
			if (tp->t_state & (TS_ISOPEN|TS_WOPEN)) {
				dhparam(unit);
				dmctl(unit, DML_ON, DMSET);
				tp->t_state &= ~TS_BUSY;
#ifdef	SYSV
				map_state(tp, TOSV);
#endif	SYSV
				dhstart(tp);
			}
			unit++;
		}
	}
	dhsilos = 0;
}

/*int dhtransitions, dhslowtimers, dhfasttimers;		/*DEBUG*/
/*
 * At software clock interrupt time, check status. Empty all the dh silos that 
 * are in use, and decide whether to turn any silos off or on.
 */
dhtimer()
{
	register int dh, s;
	static int timercalls;

	if (dhsilos) {
/*		dhfasttimers++;		/*DEBUG*/
		timercalls++;
		s = spl5();
		for (dh = 0; dh < NDH; dh++)
			if (dhsilos & (1 << dh))
				dhrint(dh);
		splx(s);
	}
	if ((dhsilos == 0) || ((timercalls += FASTTIMER) >= hz)) {
/*		dhslowtimers++;		/*DEBUG*/
		timercalls = 0;
		for (dh = 0; dh < NDH; dh++) {
		    ave(dhrate[dh], dhchars[dh], 8);
		    if ((dhchars[dh] > dhhighrate) &&
		      ((dhsilos & (1 << dh)) == 0)) {
			((struct dhdevice *)(dhinfo[dh]->qi_mi->qm_addr))->dhsilo =
			    (dhchars[dh] > 500? 32 : 16);
			dhsilos |= (1 << dh);
/*			dhtransitions++;		/*DEBUG*/
		    } else if ((dhsilos & (1 << dh)) &&
		      (dhrate[dh] < dhlowrate)) {
			((struct dhdevice *)(dhinfo[dh]->qi_mi->qm_addr))->dhsilo = 0;
			dhsilos &= ~(1 << dh);
		    }
		    dhchars[dh] = 0;
		}
	}
	timeout(dhtimer, (caddr_t) 0, dhsilos? FASTTIMER: hz);
}

/*
 * Turn on the line associated with dh dev.
 */
dmopen(dev, flag)
	dev_t dev;
{
	register struct tty *tp;
	register struct dmdevice *dmaddr;
	register struct qb_device *qi;
	register int unit;
	register int dm;
	int s;

	unit = minor(dev);
	dm = unit >> 4;
	tp = &dh_tty[unit];
	unit &= 0xf;
	if (dm >= NDH || (qi = dminfo[dm]) == 0 || qi->qi_alive == 0) {
		tp->t_state |= TS_CARR_ON;
#ifdef	SYSV
		map_state(tp, TOSV);
#endif	SYSV
		return 0;
	}
	dmaddr = (struct dmdevice *)qi->qi_mi->qm_addr;
	s = splx(qi->qi_mi->qm_psl);
	dmaddr->dmcsr &= ~DM_SE;
	while (dmaddr->dmcsr & DM_BUSY)
		;
	dmaddr->dmcsr = unit;
	dmaddr->dmlstat = DML_ON;
	DELAY(100);
	if (dmaddr->dmlstat&DML_CAR)
#ifdef	SYSV
	    if (u.u_procp->p_universe == UNIVERSE_SYSV)
		(void)(*sv_linesw[tp->svt_line].l_mdmint)(tp, 1);
	    else
#endif	SYSV
		(void)(*linesw[tp->t_line].l_modem)(tp, 1);
	dmaddr->dmcsr = DM_IE|DM_SE;
#ifdef TTY_NODELAY
	if ((flag & FNDELAY) == 0)
#endif
	while ((tp->t_state&TS_CARR_ON)==0) {
	    tp->t_state |= TS_WOPEN;
#ifdef	SYSV
	    map_state(tp, TOSV);
#endif	SYSV
	    sleep((caddr_t)&tp->t_rawq, TTIPRI);
	    if ((tp->t_state&TS_XCLUDE) && (u.u_procp->p_flag&SLOGIN)) {
		splx(s);
		return 1;
	    }
	}
	splx(s);
	return 0;
}

/*
 * Dump control bits into the DM registers.
 */
dmctl(dev, bits, how)
	dev_t dev;
	int bits, how;
{
	register struct qb_device *qi;
	register struct dmdevice *dmaddr;
	register int unit, s;
	int dm;

	unit = minor(dev);
	dm = unit >> 4;
	if ((qi = dminfo[dm]) == 0 || qi->qi_alive == 0)
		return;
	dmaddr = (struct dmdevice *)qi->qi_mi->qm_addr;
	s = splx(dminfo[dm]->qi_mi->qm_psl);
	dmaddr->dmcsr &= ~DM_SE;
	while (dmaddr->dmcsr & DM_BUSY)
		;
	dmaddr->dmcsr = unit & 0xf;
	switch(how) {
	case DMSET:
		dmaddr->dmlstat = bits;
		break;
	case DMBIS:
		dmaddr->dmlstat |= bits;
		break;
	case DMBIC:
		dmaddr->dmlstat &= ~bits;
		break;
	}
	dmaddr->dmcsr = DM_IE|DM_SE;
	splx(s);
}

/*
 * DM11 interrupt; deal with carrier transitions.
 */
dmintr(dm)
	register int dm;
{
	register struct qb_device *qi;
	register struct tty *tp;
	register struct dmdevice *dmaddr;

	qi = dminfo[dm];
	if (qi == 0)
		return;
	dmaddr = (struct dmdevice *)qi->qi_mi->qm_addr;
	if (dmaddr->dmcsr&DM_DONE) {
		if (dmaddr->dmcsr&DM_CF) {
			tp = &dh_tty[(dm<<4)+(dmaddr->dmcsr&0xf)];
			if (dmaddr->dmlstat & DML_CAR)
#ifdef	SYSV
			    if (tp->t_universe == UNIVERSE_SYSV)
				(void)(*sv_linesw[tp->svt_line].l_mdmint)(tp,1);
			    else
#endif	SYSV
				(void)(*linesw[tp->t_line].l_modem)(tp, 1);
	    		else if (tp->t_state&TS_CARR_ON) {
#ifdef	SYSV
			    if (tp->t_universe == UNIVERSE_SYSV) {
				if ((*sv_linesw[tp->svt_line].l_mdmint)(tp, 0)
							== 0)
					dmaddr->dmlstat = 0;
			    } else
#endif	SYSV
			    if ((*linesw[tp->t_line].l_modem)(tp, 0) == 0)
					dmaddr->dmlstat = 0;
			}
		}
		dmaddr->dmcsr = DM_IE|DM_SE;
	}
}
#endif	NDH > 0
