/*
 * VMEBUS: Integrated Solutions {Intelligent} Communications Processor driver
 */
#include "cp.h"
#if	NCP > 0
#include "../h/types.h"
#include "../machine/pte.h"
#include "param.h"
#include "conf.h"
#include "user.h"
#include "proc.h"
#include "ioctl.h"
#include "tty.h"
#include "map.h"
#include "buf.h"
#include "vm.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/cpreg.h"

#include "bk.h"
#include "clist.h"
#include "file.h"
#include "uio.h"
#include "kernel.h"
#include "syslog.h"

#ifdef	SYSV
#include "../sysv/sys/termio.h"
#endif	SYSV

#define	TTY_NODELAY

/* Definition of the driver for the auto-configuration program.  */
u_short	*cpstd[] = {
	 	(u_short *)&vme_stdio[0xFFF520], 
		(u_short *)&vme_stdio[0xFFF560], 
		(u_short *)&vme_stdio[0xFFF580],
		(u_short *)&vme_stdio[0xFFF5A0],
		(u_short *)&vme_stdio[0xFFF5C0],
		(u_short *)&vme_stdio[0xFFF620],
 		0 };
int	cpprobe(), cpslave(), cpattach(), cprint(), cpxint(), cptimer();
struct	qb_device *cpinfo[NCP];
struct	qb_driver CPdriver =
	{ cpprobe, cpslave, cpattach, cpstd, "cp", cpinfo, "CP" };

#define	ISPEED	B9600
#define	IFLAGS	(EVENP|ODDP|ECHO|CRMOD)

#define	MAXLINES	16
struct	tty cp_tty[NCP*MAXLINES];
int	ncp_tty = 0;

/* Local variables for the driver */
int	cp_use_silodma = 1;		/* use silo dma if available */
short	cp_nch[NCP*MAXLINES];		/* number of characters xmiting */
short	cp_nlines[NCP];			/* number of lines per controller */
short	cp_silodma[NCP];		/* controller supports silo dma */
int	cp_silo_max_depth[NCP];		/* largest depth of silo */
int	cp_silo_nch[NCP];		/* number of chars pulled from silo */
int	cp_silo_fill_level = 40;	/* fill level for interrupt */
int	cp_silo_age_time = 11;		/* age time for interrupt */
#define	SILODMA_SIZE	0x1000		/* number of bytes to dma */
short	cp_silo[NCP][SILODMA_SIZE/2];	/* silo */
int	cp_psilo[NCP];			/* VDMA address of silo */

struct	cp_lp {
	short		lp_state;
#define		LP_EXIST	0x01
#define		LP_CENTRON	0x02
#define		LP_OPEN		0x04
#define		LP_ASLP		0x08
#define		LP_TOUT		0x10
#define		LP_NOCHANGE	0x20
	short		lp_canon;
	short		lp_physcol;
	short		lp_logcol;
	short		lp_physline;
	short		lp_maxcol;
	struct buf	*lp_inbuf;
	struct buf	*lp_outbuf;
	char		*lp_outp;
	int		lp_outvdma;
	short		lp_outn;
}	cp_lp[NCP];

#define	LP_PRI		(PZERO+8)
#define	LP_BUFSIZE	(1024)
#define LP_TIMEOUT	(2*60)		/* 2 min timeout, 4 before message */

int	cpstart(), ttrstrt(), lptout();

#ifdef	SYSV
int	sv_cpproc();
#endif	SYSV

/*
 * baud rate conversion	:                           1  1  2  4  9  1  3
 *			          1  1  1  2  3  6  2  8  4  8  6  9  8
 *			    5  7  1  3  5  0  0  0  0  0  0  0  0  .  .
 *			 0  0  5  0  4  0  0  0  0  0  0  0  0  0  2  4 */
char cp_bconv0[16] = {	 0, 0, 0, 1, 2, 3, 4, 4, 5, 6,10, 8, 9,11,12,15, };/**/
char cp_bconv1[16] = {	 0, 0, 0, 1, 2, 3, 3, 4, 5, 6, 6, 8, 9,11,11,12, };/**/
char *cp_bconv;

static int v;
extern unsigned short *battery_clock;

/* Routine for configuration to force a cp to interrupt.  */
cpprobe(cpaddr)
register struct cpdevice *cpaddr;
{
	/* issue a master clear */
	cpaddr->cp_sel = SEL_MC;
	while (cpaddr->cp_sel & SEL_MC)
		DELAY(100);

	/* 
	 * cause interrupt by setting small silo age time and writeing 
	 * something in the silo
	 */
	cpaddr->cp_isr = (ISR_SI << 8) | (v = freevec());
	cpaddr->cp_sel = SEL_SILO;
	cpaddr->cp_pr = 1;		/* silo age time */
	cpaddr->cp_swr = SWR_VDP;	/* simulate reception of character */
	DELAY(10000);
	ncp_tty += MAXLINES;		/* wrong for icp-8 */
	clevmax = clev_ttymax;
	clev_tty = MAX(clev, clev_tty);
	return (sizeof (struct cpdevice));
}

cpslave(qi)
struct qb_device *qi;
{
	return (1);
}

cpattach(qi)
struct qb_device *qi;
{
	register struct cpdevice *cpaddr=(struct cpdevice *)qi->qi_mi->qm_addr;
	register short conf;

	cpaddr->cp_sel = SEL_MC;
	while (cpaddr->cp_sel & SEL_MC)
		DELAY(100);
	conf = cpaddr->cp_sel;
	cp_nlines[qi->qi_ctlr] = conf&SEL_CONF_NLINES;
	cp_lp[qi->qi_ctlr].lp_state = 0;
	cpaddr->cp_sel = SEL_SILO;
	cpaddr->cp_pr = (cp_silo_fill_level << 8) | cp_silo_age_time;
	cpaddr->cp_isr = ((ISR_NI|ISR_SI|ISR_TI|ISR_CI|ISR_PI) << 8) | v;
	printf("	%d terminal lines", cp_nlines[qi->qi_ctlr]);
	if ((conf & SEL_CONF_SILOD) && cp_use_silodma) {
		cp_silodma[qi->qi_ctlr] = 1;
		cp_psilo[qi->qi_ctlr] = 
			    svqballoc(&cp_silo[qi->qi_ctlr][0], SILODMA_SIZE);
		cpaddr->cp_bah = hiword(cp_psilo[qi->qi_ctlr]);
		cpaddr->cp_bal = loword(cp_psilo[qi->qi_ctlr]);
		cpaddr->cp_bc = SILODMA_SIZE-2;
		cpaddr->cp_sr = SSR_SD;
		printf(" with silo dma");
	}
	if (conf & SEL_CONF_BR) {
		printf(",\n					baud table 1");
		cp_bconv = cp_bconv1;
	} else
		cp_bconv = cp_bconv0;
	if (conf & SEL_CONF_LP) {
		cp_lp[qi->qi_ctlr].lp_state |= LP_EXIST;
		if (qi->qi_flags)
			cp_lp[qi->qi_ctlr].lp_maxcol = qi->qi_flags;
		else
			cp_lp[qi->qi_ctlr].lp_maxcol = LP_MAXCOL;
		printf(",\n					%d column ", 
			cp_lp[qi->qi_ctlr].lp_maxcol);
		if (conf & SEL_CONF_LPCEN) {
			cp_lp[qi->qi_ctlr].lp_state |= LP_CENTRON;
			printf("centronics printer");
		} else
			printf("data products printer");
	}
	if (conf & SEL_CONF_QIC2)
		printf(",\n					qic2");
	if (conf & SEL_CONF_CLK && battery_clock == 0) {
		printf(",\n					 battery clock");
		battery_clock = (unsigned short *)cpaddr;
	}
	if (conf & SEL_CONF_DRV)
		printf(",\n					drv11");
	printf(" ");
}

cpopen(dev, flag)
dev_t dev;
int flag;
{
	register struct tty *tp;
	register int unit, cp, line, s;
	register struct qb_device *qi;

	if (ISLP(dev))
		return(lpopen(dev, flag));

	unit = minor(dev);
	cp = CPUNIT(unit);
	line = CPLINE(unit);

	if (cp > NCP || line >= cp_nlines[cp] || (qi = cpinfo[cp])== 0 || 
		qi->qi_alive == 0)
			return (ENXIO);
	tp = &cp_tty[unit];
	tp->t_addr = (caddr_t)qi->qi_mi->qm_addr;
	tp->t_oproc = cpstart;
#ifdef	SYSV
	tp->svt_proc = sv_cpproc;
#endif	SYSV

	if ((tp->t_state&TS_ISOPEN) == 0) {
		ttychars(tp);
		if (tp->t_ispeed == 0) {
			tp->t_ispeed = tp->t_ospeed = ISPEED;
			tp->t_flags = IFLAGS;
		}
#ifdef	SYSV
		sv_ttinit(tp);
#endif	SYSV
		cpparam(unit);
	} else if ((tp->t_state&TS_XCLUDE) && 
	    ((u.u_uid!=0) || (u.u_procp->p_flag&SLOGIN)))
		return (EBUSY);

	/* Wait for carrier, then process line discipline specific open. */
	s = splx(qi->qi_mi->qm_psl);
	if (cpmdm(tp->t_addr, line, 1, 1))
#ifdef	SYSV
	    if (u.u_procp->p_universe == UNIVERSE_SYSV)
		(void)(*sv_linesw[tp->svt_line].l_mdmint)(tp, 1);
	    else
#endif	SYSV
		(void)(*linesw[tp->t_line].l_modem)(tp, 1);

#ifdef TTY_NODELAY
	if ((flag & FNDELAY) == 0)
#endif
	    while ((tp->t_state&TS_CARR_ON)==0) {
		tp->t_state |= TS_WOPEN;
#ifdef	SYSV
		map_state(tp, TOSV);
#endif	SYSV
		sleep((caddr_t)&tp->t_rawq, TTIPRI);
		if ((tp->t_state&TS_XCLUDE) && (u.u_procp->p_flag&SLOGIN)){
			splx(s);
			return (EBUSY);
		}
	    }
	splx(s);

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_open)(dev, tp, flag));
	else
#endif	SYSV
		return ((*linesw[tp->t_line].l_open)(dev, tp));
}

cpclose(dev, flag)
dev_t dev;
int flag;
{
	register struct tty *tp;
	register struct cpdevice *cpaddr;
	register int unit, line;

	if (ISLP(dev)) {
		lpclose(dev, flag);
		return;
	}

	unit = minor(dev);
	line = CPLINE(unit);
	tp = &cp_tty[unit];
	cpaddr = (struct cpdevice *)tp->t_addr;
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		(*sv_linesw[tp->svt_line].l_close)(tp);
	else
#endif	SYSV
		(*linesw[tp->t_line].l_close)(tp);
	cpaddr->cp_brk &= ~(1<<line);
	if (tp->t_state&(TS_HUPCLS|TS_WOPEN) || (tp->t_state&TS_ISOPEN)==0)
		(void) cpmdm(cpaddr, line, 0, 0);
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		sv_ttyclose(tp);
	else
#endif	SYSV
		ttyclose(tp);
}

cpread(dev, uio)
dev_t dev;
struct uio *uio;
{
	register struct tty *tp = &cp_tty[minor(dev)];

	if (ISLP(dev))
		return (EIO);		/* cant read printer */

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_read)(tp, uio));
	else
#endif	SYSV
		return ((*linesw[tp->t_line].l_read)(tp, uio));
}

cpwrite(dev, uio)
dev_t dev;
struct uio *uio;
{
	register struct tty *tp = &cp_tty[minor(dev)];

	if (ISLP(dev))
		return(lpwrite(dev, uio));

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_write)(tp, uio));
	else
#endif	SYSV
		return ((*linesw[tp->t_line].l_write)(tp, uio));
}

cpint(cp)
register int cp;
{
	ushort	isr = ((struct cpdevice *)cpinfo[cp]->qi_mi->qm_addr)->cp_isr;

	switch (isr & 0xFF) {
		case ISR_SI:
			cp_sint(cp);
			break;

		case ISR_TI:
			cp_tint(cp);
			break;

		case ISR_CI:
			cp_cint(cp);
			break;

		case ISR_PI:
			lpint(cp);
			break;

		case ISR_NI:
			printf("cp%d: NXM\n", cp);
	}
}

/* silo service interrupt */
cp_sint(cp)
register int cp;
{
	register struct tty *tp;
	register ushort c;
	register struct cpdevice *cpaddr = (struct cpdevice *)
						(cpinfo[cp]->qi_mi->qm_addr);
	register struct tty *tp0 = &cp_tty[cp<<4];
	register int overrun = 0, nch = 0;
	register u_short *silo = &cp_silo[cp][0];

	if (cp_silodma[cp]) {
		cpaddr->cp_sel = SEL_SILO;
		if ((cpaddr->cp_sr & SSR_SDD) == 0) {
		    printf("cp%d: silo interrupt without SDD\n", cp);
		    return;
		}
	}

	while (1) {
		nch++;
		if (cp_silodma[cp])
			c = *silo++;
		else
			c = cpaddr->cp_swr;
	    	if ((c & SWR_VDP) == 0) {
		    if (cp_silodma[cp]) {
			cpaddr->cp_sel = SEL_SILO;
			cpaddr->cp_bah = hiword(cp_psilo[cp]);
			cpaddr->cp_bal = loword(cp_psilo[cp]);
			cpaddr->cp_bc = SILODMA_SIZE-2;
			cpaddr->cp_sr = SSR_SD;		/* clear SSR_SDD ! */
		    }
		    cp_silo_max_depth[cp] = MAX(cp_silo_max_depth[cp], nch);
		    cp_silo_nch[cp] += nch;
		    return;
		}
		tp = tp0 + ((c&SWR_LN_MASK)>>SWR_LN_SHIFT);
		if ((tp->t_state&TS_ISOPEN) == 0)
			continue;
		if ((c & SWR_DO) && overrun++ == 0)
		    if (c & SWR_PE)
			log(LOG_WARNING, "cp%d: silo overflow\n", cp);
		    else
			log(LOG_WARNING, "cp%d: line %d overflow\n", 
			    cp, (c&SWR_LN_MASK)>>SWR_LN_SHIFT);
#ifdef	SYSV
		if (tp->t_universe == UNIVERSE_SYSV) {
			if (c & SWR_PE) {
			    if (tp->svt_iflag & SV_IGNPAR)
				continue;
			    if (tp->svt_iflag & SV_PARMRK)
				;		/* ??? */
			}
			if (c & SWR_FE) {
			    if (tp->svt_iflag & SV_IGNBRK)
				continue;
			    if (tp->svt_iflag & SV_BRKINT)
				(*sv_linesw[tp->svt_line].l_input)
					(c, tp, L_BREAK);
			    c = 0;
			}
			if (tp->svt_iflag & SV_ISTRIP)
				c &= 0177;
			else
				c &= 0377;
		} else {
#endif	SYSV
			if (c & SWR_PE)
				if ((tp->t_flags&(EVENP|ODDP))==EVENP
				 || (tp->t_flags&(EVENP|ODDP))==ODDP )
					continue;
			if (c & SWR_FE)
				if (tp->t_flags&RAW)
					c = 0;
				else
					c = tp->t_intrc;
#ifdef	SYSV
		}
		if (tp->t_universe == UNIVERSE_SYSV)
			(*sv_linesw[tp->svt_line].l_input)(c, tp, 0);
		else
#endif	SYSV
#if NBK > 0 || Nbk > 0
			if (tp->t_line == NETLDISC) {
				c &= 0177;
				BKINPUT(c, tp);
			} else
#endif
				(*linesw[tp->t_line].l_rint)(c, tp);
	    }
}

/*
 * Ioctl for CP.
 */
cpioctl(dev, cmd, data, flag)
caddr_t data;
{
	register int unit = minor(dev);
	register int line = unit & 0xF;
	register struct tty *tp = &cp_tty[unit];
	register int error;

	if (ISLP(dev))
		return(lpioctl(dev, cmd, data, flag));
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV) {
		if ((error = sv_ttiocom(tp, cmd, data)) > 0)
			return error;
		if (cmd == TCSETAF || cmd == TCSETA || cmd == TCSETAW)
			cpparam(unit);
		if (cmd == TCSBRK)
			;		/* ??? */
		return 0;
	}
#endif	SYSV
	if ((error = (*linesw[tp->t_line].l_ioctl)(tp,cmd,data,flag)) >= 0)
		return (error);
	if ((error = ttioctl(tp, cmd, data, flag)) >= 0) {
		if (cmd == TIOCSETP || cmd == TIOCSETN ||
		    cmd == TIOCLSET || cmd == TIOCLBIS || cmd == TIOCLBIC)
			cpparam(unit);
		return (error);
	}
	switch (cmd) {
	    case TIOCSBRK:
		((struct cpdevice *)(tp->t_addr))->cp_brk |= 1<<line;
		break;

	    case TIOCCBRK:
		((struct cpdevice *)(tp->t_addr))->cp_brk &= ~(1<<line);
		break;

	    case TIOCSDTR:
		cpmdm(tp->t_addr, line, 1, 1);
		break;

	    case TIOCCDTR:
		cpmdm(tp->t_addr, line, 1, 0);
		break;

/* TODO: support these
	    case TIOCMSET:
	    case TIOCMBIS:
	    case TIOCMBIC:
	    case TIOCMGET:
/**/

	    default:
		return (ENOTTY);
	}
	return (0);
}

/*
 * Set parameters from open or stty into the CP hardware registers.
 */
cpparam(unit)
register int unit;
{
	register struct cpdevice *cpaddr;
	register struct tty *tp = &cp_tty[unit];
	register int lpr, line = unit&0xF;
	register int s;

	if (tp->t_ispeed == 0) {
		cpmdm(tp->t_addr, line, 1, 0);
		return;
	}
	lpr = (cp_bconv[tp->t_ispeed]<<4) | cp_bconv[tp->t_ospeed];
#ifdef	SYSV
	if (tp->t_universe == UNIVERSE_SYSV) {
		if ((tp->svt_cflag&SV_CSIZE) == SV_CS8)
			lpr |= PR_BITS8;
		else if ((tp->svt_cflag&SV_CSIZE) == SV_CS7)
			lpr |= PR_BITS7;
		else if ((tp->svt_cflag&SV_CSIZE) == SV_CS6)
			lpr |= PR_BITS6;
		else
			lpr |= PR_BITS5;
		if (tp->svt_cflag&SV_CSTOPB)
			lpr |= PR_TWOSB;
		if (tp->svt_cflag&SV_PARENB) {
			lpr |= PR_PENABLE;
			if (tp->svt_cflag&SV_PARODD)
				lpr |= PR_OPAR;
		}
	} else {
#endif	SYSV
		if ((tp->t_ispeed) == B134)
			lpr |= PR_BITS6|PR_PENABLE|PR_HDUPLX;
		else if (tp->t_flags & (RAW|LITOUT|PASS8))
			lpr |= PR_BITS8;
		else
			lpr |= PR_BITS7|PR_PENABLE;
		if ((tp->t_flags&EVENP) == 0)
			lpr |= PR_OPAR;
		if ((tp->t_ospeed) == B110 || tp->t_state & TS_TWOSB)
			lpr |= PR_TWOSB;
#ifdef	SYSV
	}
#endif	SYSV
	lpr |= PR_XOFF;			/* allways do XOFF quickly */
	cpaddr = (struct cpdevice *)tp->t_addr;
	s = splx(cpinfo[CPUNIT(unit)]->qi_mi->qm_psl);
	cpaddr->cp_sel = line << SEL_SHIFT;
	cpaddr->cp_pr = lpr;
	splx(s);
}

/*
 * CP transmitter interrupt. Restart each line which used to be active but has
 * terminated transmission since the last interrupt.
 */
cp_tint(cp)
int cp;
{
	register struct tty *tp;
	register ushort line;
	register struct cpdevice *cpaddr = 
			(struct cpdevice *)cpinfo[cp]->qi_mi->qm_addr;
	register ushort tcr = cpaddr->cp_tcr;

	for (line = 0, tp = &cp_tty[cp<<4] ; tcr ; line++, tp++) {
		if (tcr & (1 << line)) {
			tcr ^= (1 << line);
			tp->t_state &= ~TS_BUSY;
			if (tp->t_state&TS_FLUSH)
			    tp->t_state &= ~TS_FLUSH;
			else {
			    cpaddr->cp_sel = line << SEL_SHIFT;
			    /* 
			     * flush number asked to transmit - residual not 
			     * transmitted because of XOFF recieved.
			     */
			    ndflush(&tp->t_outq, cp_nch[(cp*MAXLINES)+line] -
				cpaddr->cp_bc);
			}
#ifdef	SYSV
			map_state(tp, TOSV);
			if (tp->t_universe == UNIVERSE_SYSV) {
				if ((*sv_linesw[tp->svt_line].l_output)(tp))
					cpstart(tp);
			}
			else {
#endif	SYSV
			    if (tp->t_line)
				(*linesw[tp->t_line].l_start)(tp);
			    else
				cpstart(tp);
#ifdef	SYSV
			}
#endif	SYSV
		}
	}
}

/*
 * Start (restart) transmission on the given CP line.
 */
cpstart(tp)
register struct tty *tp;
{
	register struct cpdevice *cpaddr;
	register int unit, nch, s;
	register ushort line;
	int stradr;

	/* prevent state of the tp from changing. */
	s = splx(cpinfo[CPUNIT(minor(tp->t_dev))]->qi_mi->qm_psl);

	/* If it's currently active, or delaying, no need to do anything. */
	if (tp->t_state&(TS_TIMEOUT|TS_BUSY|TS_TTSTOP))
		goto out;
	/*
	 * If there are sleepers, and output has drained below low water mark, 
	 * wake up the sleepers.
	 */
	if (tp->t_outq.c_cc<=TTLOWAT(tp)) {
		if (tp->t_state&TS_ASLEEP) {
			tp->t_state &= ~TS_ASLEEP;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			wakeup((caddr_t)&tp->t_outq);
		}
		if (tp->t_wsel) {
			selwakeup(tp->t_wsel, tp->t_state&TS_WCOLL);
			tp->t_wsel = 0;
			tp->t_state &= ~TS_WCOLL;
		}
	}

	/* Now restart transmission unless the output queue is empty. */
	if (tp->t_outq.c_cc == 0)
		goto out;
	if (tp->t_flags & (RAW|LITOUT))
		nch = ndqb(&tp->t_outq, 0);
	else {
		nch = ndqb(&tp->t_outq, 0200);
		/* If first thing on queue is a delay process it. */
		if (nch == 0) {
			nch = getc(&tp->t_outq);
			timeout(ttrstrt, (caddr_t)tp, (nch&0x7f)+6);
			tp->t_state |= TS_TIMEOUT;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			goto out;
		}
	}

	/* If characters to transmit, restart transmission. */
	if (nch) {
		unit = minor(tp->t_dev);
		line = CPLINE(unit);
		cpaddr = (struct cpdevice *)tp->t_addr;
		tp->t_state |= TS_BUSY;
#ifdef	SYSV
		map_state(tp, TOSV);
#endif	SYSV
		cpaddr->cp_sel = line << SEL_SHIFT;
		stradr = CDMA_ADR(tp->t_outq.c_cf);
		cpaddr->cp_bah = hiword(stradr);
		cpaddr->cp_bal = loword(stradr);
		cpaddr->cp_bc = cp_nch[unit] = nch;
		line = 1 << line;
		cpaddr->cp_tcr = line;
	}
out:	splx(s);
}

#ifdef	SYSV
/*
 *	device specific output routine for sysv TTY code
 */
sv_cpproc(tp, cmd)
register struct tty *tp;
register cmd;
{
	sv_proc(tp, cmd, cpstart, cpparam);
}
#endif	SYSV

/*
 * Stop output on a line, e.g. for ^S/^Q or output flush.
 */
cpstop(tp, flag)
register struct tty *tp;
{
	register struct cpdevice *cpaddr = (struct cpdevice *)tp->t_addr;
	register int unit, s;

	/*
	 * Block input/output interrupts while messing with state.
	 */
	s = splx(cpinfo[CPUNIT(minor(tp->t_dev))]->qi_mi->qm_psl);
	if (tp->t_state & TS_BUSY) {
/* TODO: shut down transmit in progress */
		if ((tp->t_state&TS_TTSTOP)==0) {
			tp->t_state |= TS_FLUSH;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
		}
	}
	splx(s);
}

/*
 * Reset state of driver if bus reset was necessary.
 */
cpreset()
{
	register int cp, unit;
	register struct tty *tp;
	register struct qb_device *qi;
	register int i;

	for (cp = 0; cp < NCP; cp++) {
		qi = cpinfo[cp];
		if (qi == 0 || qi->qi_alive == 0)
			continue;
		printf(" CP%d", cp);
		((struct cpdevice *)qi->qi_mi->qm_addr)->cp_sel = SEL_MC;
		tp = &cp_tty[unit = cp * MAXLINES];

		for (i = 0; i < MAXLINES; unit++, i++, tp++) {
			if (tp->t_state & (TS_ISOPEN|TS_WOPEN)) {
				cpparam(unit);
				cpmdm(tp->t_addr, i, 1, 1);
				tp->t_state &= ~TS_BUSY;
#ifdef	SYSV
				map_state(tp, TOSV);
#endif	SYSV
				cpstart(tp);
			}
		}
	}
}

cpmdm(cpaddr, line, enable, assert)
register struct cpdevice *cpaddr;
{
	register ushort ln = 1 << line;
	int s = spltty();

	if (enable)
		cpaddr->cp_ler |= ln;
	else
		cpaddr->cp_ler &= ~ln;
	if (assert)
		cpaddr->cp_acr |= ln;
	else
		cpaddr->cp_acr &= ~ln;

	splx(s);
	return (cpaddr->cp_dcr & ln);
}

/*
 * carrier detect interrupt; deal with carrier transitions.
 */
cp_cint(cp)
register int cp;
{
	register struct tty *tp = &cp_tty[cp<<4];
	register ushort line;
	register ushort dcr = ((struct cpdevice *)
				cpinfo[cp]->qi_mi->qm_addr)->cp_dcr;

	for (line = 0; line < MAXLINES ; line++, tp++) {
	    if (dcr & (1<<line)) {
		if ((tp->t_state & TS_CARR_ON) == 0)
#ifdef	SYSV
		    if (tp->t_universe == UNIVERSE_SYSV)
			(void)(*sv_linesw[tp->svt_line].l_mdmint)(tp, 1);
		    else
#endif	SYSV
			(void)(*linesw[tp->t_line].l_modem)(tp, 1);
	    } else if (tp->t_state&TS_CARR_ON) {
#ifdef	SYSV
		if (tp->t_universe == UNIVERSE_SYSV) {
		    if ((*sv_linesw[tp->svt_line].l_mdmint)(tp, 0) == 0)
			cpmdm(tp->t_addr, line, 1, 0);
		} else
#endif	SYSV
		if ((*linesw[tp->t_line].l_modem)(tp, 0) == 0)
		    cpmdm(tp->t_addr, line, 1, 0);
	    }
	}
}

lpopen(dev, flag)
dev_t dev;
int flag;
{
	register int cp = CPUNIT(minor(dev));
	register struct cp_lp *lp = &cp_lp[cp];
	register struct qb_device *qi = cpinfo[cp];

	if (cp >= NCP || qi == 0 || qi->qi_alive == 0 ||
	    lp->lp_state&LP_OPEN || (lp->lp_state&LP_EXIST) == 0 )
			return (ENXIO);
	lp->lp_state |= LP_OPEN;
	lp->lp_inbuf = geteblk(LP_BUFSIZE);
	lp->lp_outbuf = geteblk(LP_BUFSIZE);
	lp->lp_outp = lp->lp_outbuf->b_un.b_addr;
	lp->lp_outn = 0;
	lp->lp_outvdma = svqballoc(lp->lp_outp, LP_BUFSIZE);
	lp->lp_logcol = lp->lp_physcol = 0;
	lp->lp_physline = 0;
	lp->lp_canon = LPCANON(dev);
	return (0);
}

lpclose(dev, flag)
dev_t dev;
int flag;
{
	register int cp = CPUNIT(minor(dev));
	register struct cp_lp *lp = &cp_lp[cp];

	lpflush(cp);
	brelse(lp->lp_inbuf);
	brelse(lp->lp_outbuf);
	lp->lp_state &= ~LP_OPEN;
}

lpioctl(dev, cmd, data, flag)
caddr_t data;
{
	register int cp = CPUNIT(minor(dev));

	if (cmd == TIOCFLUSH) {
		lpflush(cp);
		return 0;
	} else
		return (ENOTTY);
}

lpwrite(dev, uio)
dev_t dev;
struct uio *uio;
{
	register int cp = CPUNIT(minor(dev));
	register struct cp_lp *lp = &cp_lp[cp];
	register unsigned n;
	register char *p;
	register int error;

	while (n = MIN(LP_BUFSIZE, (unsigned)uio->uio_resid)) {
		p = lp->lp_inbuf->b_un.b_addr;
		error = uiomove(p, (int)n, UIO_WRITE, uio);
		if (error)
			return (error);
		do
			lpcanon(cp, *p++);
		while (--n);
	}
	return (0);
}

lpcanon(cp, c)
register int cp, c;
{
	register struct cp_lp *lp = &cp_lp[cp];
	register int logcol, physcol;

	if (lp->lp_canon&LP_CANON_RAW) {
		lpoutput(cp, c);
		return;
	} 
	if (lp->lp_canon&LP_CANON_CAP) {
		register c2;

		if (c>='a' && c<='z')
			c += 'A'-'a'; 
		else
		    switch (c) {
			case '{':	c2 = '(';	goto esc;

			case '}':	c2 = ')';	goto esc;

			case '`':	c2 = '\'';	goto esc;

			case '|':	c2 = '!';	goto esc;

			case '~':	c2 = '^';

			esc:	lpcanon(cp, c2);
				lp->lp_logcol--;
				c = '-';
		    }
	}
	logcol = lp->lp_logcol;
	physcol = lp->lp_physcol;

	switch(c) {
	    case ' ':
		logcol++;
		break;

	    case '\t':
		logcol = (logcol+8) & ~7;
		break;

	    case '\f':
		if (lp->lp_physline == 0 && physcol == 0)
			break;			/* fall into ... */

	    case '\n':
		lpoutput(cp, c);
		if (c == '\f')
			lp->lp_physline = 0;
		else
			lp->lp_physline++;
		physcol = 0;			/* fall into ... */

	    case '\r':
		logcol = 0;
		break;

	    case '\b':
		if (logcol > 0)
			logcol--;
		break;

	    default:
		if (logcol < physcol) {
			lpoutput(cp, '\r');
			physcol = 0;
		}
		if (logcol < lp->lp_maxcol) {
			while (logcol > physcol) {
				lpoutput(cp, ' ');
				physcol++;
			}
			lpoutput(cp, c);
			physcol++;
		}
		logcol++;
	}
	if (logcol > 1000)			/* ignore long lines  */
		logcol = 1000;
	lp->lp_logcol = logcol;
	lp->lp_physcol = physcol;
}

lpoutput(cp, c)
int cp, c;
{
	register struct cp_lp *lp = &cp_lp[cp];

	while (1)
	    if (lp->lp_outn < LP_BUFSIZE) {
		lp->lp_outn++;
		*lp->lp_outp++ = c;
		return;
	    } else
		lpflush(cp);
}

lpflush(cp)
int cp;
{
	register int s;
	register struct cp_lp *lp = &cp_lp[cp];
	register struct cpdevice *cpaddr = 
				(struct cpdevice *)cpinfo[cp]->qi_mi->qm_addr;

	s = splx(cpinfo[cp]->qi_mi->qm_psl);
	while (lp->lp_outn) {
		cpaddr->cp_sel = SEL_LP;
		lp->lp_outp = lp->lp_outbuf->b_un.b_addr;
		cpaddr->cp_bah = hiword(lp->lp_outvdma);
		cpaddr->cp_bal = loword(lp->lp_outvdma);
		cpaddr->cp_bc = lp->lp_outn;
		lp->lp_state |= LP_ASLP;
		if ((lp->lp_state & LP_TOUT) == 0) {
			lp->lp_state |= LP_TOUT;
			timeout(lptout, (caddr_t)cp, LP_TIMEOUT*hz);
		}
		cpaddr->cp_sr |= LSR_GO;
		sleep((caddr_t)lp, LP_PRI);
	}
	splx(s);
}

lpint(cp)
int cp;
{
	register struct cp_lp *lp = &cp_lp[cp];

	if ((lp->lp_state & (LP_EXIST|LP_ASLP)) == 0) {
		printf("cp%d: stray printer interrupt\n",cp);
		return;
	}
	lp->lp_state &= ~(LP_ASLP|LP_NOCHANGE);
	lp->lp_outn = 0;
	wakeup((caddr_t)lp);
}

lptout(cp)
int cp;
{
	register struct cp_lp *lp = &cp_lp[cp];
	register struct cpdevice *cpaddr;
	register short status;

	if ((lp->lp_state & LP_ASLP) == 0) {	/* no longer waiting */
		lp->lp_state &= ~LP_TOUT;
		return;
	}
	if (lp->lp_state & LP_NOCHANGE) {
		cpaddr  = (struct cpdevice *)cpinfo[cp]->qi_mi->qm_addr;
		cpaddr->cp_sel = SEL_LP;
		status = cpaddr->cp_sr;
		if ((status&LSR_CABLE) == 0)
			log(LOG_WARNING,
				"cp%d: printer not cabled correctly\n", cp);
		else if ((status&LSR_SEL) == 0)
			log(LOG_WARNING,
				"cp%d: printer not online\n", cp);
		else if (status&LSR_PAPE)
			log(LOG_WARNING,
				"cp%d: printer out of paper\n", cp);
		else if ((status&LSR_RDY) == 0)
			log(LOG_WARNING,
				"cp%d: printer not ready\n", cp);
		else
			log(LOG_WARNING,
				"cp%d: printer not responding\n", cp);
		lp->lp_state &= ~LP_NOCHANGE;
	} else
		lp->lp_state |= LP_NOCHANGE;
	timeout(lptout, (caddr_t)cp, LP_TIMEOUT*hz);
}
#endif	NCP > 0
