/*
 * Copyright (c) 1982 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)ufs_machdep.c	6.3 (Berkeley) 6/8/85
 */

#include "pte.h"

#include "param.h"
#include "systm.h"
#include "dir.h"
#include "user.h"
#include "buf.h"
#include "conf.h"
#include "proc.h"
#include "seg.h"
#include "vm.h"
#include "map.h"

int	allocbuf_failed = 0;

/*
 * Machine dependent handling of the buffer cache.
 */

/*
 * Expand or contract the actual memory allocated to a buffer. If no memory is 
 * available, release buffer and take error exit
 */
allocbuf(tp, size)
	register struct buf *tp;
	int size;
{
	register struct buf *bp, *ep;
	int sizealloc, take;
	register char *a;
	struct map *map;
	int osize;

	sizealloc = roundup(size, BUFALLOCSIZE);

	/* Buffer size does not change */
	if (sizealloc == tp->b_bufsize)
		goto out;
#ifdef	QBUS
	map = (tp->b_flags & B_18BIT) ? buffermap18 : buffermap;
#else	QBUS
	map = buffermap;
#endif	QBUS

	/* Buffer size is shrinking, Just put the tail end back in the map */
	if (sizealloc < tp->b_bufsize) {
		rmfree(map, (long)(tp->b_bufsize - sizealloc),
			(long)(tp->b_un.b_addr + sizealloc));
		goto out;
	}

	/*
	 * Buffer is being expanded or created. If being expanded, attempt to 
	 * get contiguous section, otherwise get a new chunk and copy. If no 
	 * space, free up a buffer on the AGE list and try again.
	 */
	do {
		if ((osize = tp->b_bufsize)) {
			a = (char *)rmget(map, (long)(sizealloc-osize),
				(long)(tp->b_un.b_addr + osize));
			if (a == 0) {
				a = (char *)rmalloc(map, (long)sizealloc);
				if (a != 0) {
					bcopy(tp->b_un.b_addr, a, osize);
					rmfree(map, (long)osize, 
						(long)tp->b_un.b_addr);
					tp->b_un.b_addr = a;
				}
			}
		} else {
			a = (char *)rmalloc(map, (long)sizealloc);
			if (a != 0)
				tp->b_un.b_addr = a;
		}
	} while (a == 0 && bfreemem(tp, tp->b_flags & B_QUALIFY));
	if (a == 0) {
		allocbuf_failed++;
		brelse(tp);
		return (0);
	}
out:	tp->b_bufsize = sizealloc;
	tp->b_bcount = size;
	return (1);
}

/*
 * Release space associated with a buffer.
 */
bfree(bp)
	struct buf *bp;
{
	if (bp->b_bufsize) {
#ifdef	QBUS
		rmfree((bp->b_flags & B_18BIT) ? buffermap18 : buffermap,
			(long)bp->b_bufsize, (long)bp->b_un.b_addr);
#else	QBUS
		rmfree(buffermap, (long)bp->b_bufsize, (long)bp->b_un.b_addr);
#endif	QBUS
		bp->b_bufsize = 0;
	}
	bp->b_bcount = 0;
}

/*
 * Attempt to free up buffer space by flushing something in the free list.
 * We start with BQ_AGE because we know that BQ_EMPTY takes no memory.
 */
bfreemem(tp, qual)
	register struct buf *tp;
{
	register struct buf *bp, *dp;
	int s;

loop:
	s = splbio();
	dp = &bfreelist[BQ_AGE];
loop2:
	for (; dp > bfreelist ; dp--)
		if (dp->av_forw != dp)
			break;
	if (dp == bfreelist ) {
		dp->b_flags |= B_WANTED;
		sleep((caddr_t)dp, PRIBIO+1);
		splx(s);
		goto loop;
	}
	for (bp = dp->av_forw; bp != dp ; bp=bp->av_forw)
		if ((bp != tp) && ((bp->b_flags & qual) == qual))
			break;
	if (dp == bp) {
		dp--;
		goto loop2;
	}
	notavail(bp);
	splx(s);
	if (bp->b_flags & B_DELWRI) {
		bawrite(bp);
		goto loop;
	}
	brelvp(bp);
	bp->b_error = 0;
	bp->b_flags &= (B_QUALIFY | B_BUSY);
	bp->b_flags |= B_BUSY | B_INVAL;
	bfree(bp);
	bremhash(bp);
	binshash(bp, &bfreelist[BQ_EMPTY]);
	brelse(bp);
	return(1);
}
