/*
 * System calls for real time processes.
 */

#include "pte.h"

#include "param.h"
#include "systm.h"
#include "user.h"
#include "proc.h"
#include "vm.h"
#include "text.h"

/*
 * Lock the process in core. Bring in all pages and mark the process locked.
 */
plock()
{
	if (textlock())
		if (datalock() == 0)
			tunlock();
}

/*
 * Unlock the process from core and make it swappable / pageable.
 */
punlock()
{
	register struct proc * p = u.u_procp;

	if (((p->p_flag & SPLOCK) == 0) && ((p->p_flag & SXLOCK) == 0))
		return(0);
	if (p->p_flag & SXLOCK)
		tunlock();
	if (p->p_flag & SPLOCK)
		dunlock();
	return(1);
}


textlock()
{
	register struct proc * p = u.u_procp;
	register struct text * xp = p->p_textp;
	register int size;

	if ((xp->x_flag & XPLOCK) == 0){
		size = xp->x_size;
		if (size > lockablemem) {
			u.u_error = EAGAIN;
			return(0);
		}
		lockablemem -= size;
		xlock(xp);
		xp->x_flag |= XPLOCK;
		allpagesin(p, tptov(p, 0), xp->x_size);
		xunlock(xp);
	}
	p->p_flag |= SXLOCK;
	return(1);
}
		
tunlock()
{
	register struct proc * p = u.u_procp;
	register struct text * xp = p->p_textp;

	p->p_flag &= ~SXLOCK;
	if (xp) {
		/*
		 * If no processes sharing this text
		 * are plocked, the text need not be
		 * locked anymore.  Clear XPLOCK bit.
		 */
		for (p = xp->x_caddr; p; p = p->p_xlink)
			if (p->p_flag&SXLOCK)
				return(1);
		xp->x_flag &= ~XPLOCK;
		lockablemem += xp->x_size;
	}
	return(1);
}

datalock()
{
	register struct proc * p = u.u_procp;
	register int size;

	size = p->p_dsize + p->p_ssize;
	if (size > lockablemem) {
		u.u_error = EAGAIN;
		return(0);
	}
	lockablemem -= size;
	p->p_flag |= SPLOCK;

	/* Bring data and stack pages in */
	allpagesin(p, dptov(p, 0), p->p_dsize);
	allpagesin(p, sptov(p, p->p_ssize - 1), p->p_ssize);
	return(1);
}

dunlock()
{
	register struct proc * p = u.u_procp;

	lockablemem += p->p_dsize + p->p_ssize;
	p->p_flag &= ~SPLOCK;
	return(1);
}


/*
 * Bring pages v, v+1 ... v+npf-1 in. If a page is not valid, bring it in.
 */
allpagesin(p, v, npf)
	struct proc * p;
	register unsigned v;
	register int npf;
{
	register struct pte *pte;

	pte = vtopte(p, v);
	while (npf > 0) {
		if (!pte->pg_v)
			pagein(ctob(v), 0);
		pte += CLSIZE;
		v += CLSIZE;
		npf -= CLSIZE;
	}
}

/*
 * Mark the process to be a high priority process.
 */
highpri()
{
	if (suser())
		u.u_procp->p_flag |= SPHIGH;
	else
		u.u_error = EPERM;
}

/*
 * Make the process a normal priority process.
 */
normalpri()
{
	u.u_procp->p_flag &= ~SPHIGH;
}

/*
 * Lock user in core
 */
ulock()
{
	if (suser())
		u.u_procp->p_flag |= SULOCK;
	else
		u.u_error = EPERM;
}
